{*******************************************************}
{*                                                     *}
{*      Pro VCL Extensions Library                     *}
{*      Dialogs Unit                                   *}
{*                                                     *}
{*      Copyright (c) 1996-98 by Dmitry Barabash       *}
{*                                                     *}
{*******************************************************}

unit ProDlgs;

{$I PRO.INC}

interface

uses {$IFDEF WIN32} Windows, {$ELSE} WinProcs, {$ENDIF} SysUtils, Classes,
  Graphics, Controls, Forms, StdCtrls, ExtCtrls, ProCtrls, FileCtrl;

type

{ TProCommonDialog }

  { Dialog language }
  TDialogLanguage = (dlEnglish, dlRussian, dlUkranian);

  TProCommonDialog = class(TComponent)
  private
    { Variables for properties }
    FLanguage : TDialogLanguage;
    FHelpContext : THelpContext;
    FFont : TFont;
    FParentFont : Boolean;
    { Property access methods }
    procedure SetFont(Value : TFont);
  protected
    { Font characteristics of the dialog form. By default }
    { contains default application font.                  }
    property Font : TFont read FFont write SetFont;
    { Set to True to use font of parent form in dialog form or }
    { to False to use Font property                            }
    property ParentFont : Boolean read FParentFont write FParentFont
      default True;
  public
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
  published
    { Specifies dialog language }
    property Language : TDialogLanguage read FLanguage write FLanguage
      default dlEnglish;
    { Specifies help context assigned with the Help button }
    property HelpContext : THelpContext read FHelpContext write FHelpContext
      default 0;
  end;

{ TProSelDirDialog }

  { Options of the direcory dialog:
      ddGetInto          Get into the chosen directory.
      ddAllowEdit        Allow user to type in an edit box the name
                         of a directory.
      ddAllowCreate      Create the directory if it does not exist.
      ddPrompt           Ask the user if the directory should be created. }
  TSelDirDialogOption = (ddGetInto, ddAllowEdit, ddAllowCreate, ddPrompt);
  TSelDirDialogOptions = set of TSelDirDialogOption;

  TProSelDirDialog = class(TProCommonDialog)
  private
    { Variables for properties }
    FOptions : TSelDirDialogOptions;
    FDirectory : string;
  public
    function Execute : Boolean;
  published
    { Defines the directory that appears as the currently selected }
    { directory when Execute function starts and the dialog box    }
    { appears. The name of the directory the user selects becomes  }
    { the value of Directory when the Execute function returns.    }
    property Directory : string read FDirectory write FDirectory;
    { Defines a set of values of the dialog options }
    property Options : TSelDirDialogOptions read FOptions write FOptions
      default [];
    { Enable inherited properties }
    property Font;
    property ParentFont;
  end;

{ TProConfirmDialog }

  TProConfirmDialog = class(TProCommonDialog)
  private
    { Variables for properties }
    FCaption : string;
    FCondition : string;
    FDescription : string;
    FQuestion : string;
    FCancel : Boolean;
  public
    function Execute : Boolean;
    { When Execute function returns Cancel becomes True if user }
    { pressed Cancel button or Esc key                          }
    property Cancel : Boolean read FCancel;
  published
    { Caption of the dialog box }
    property Caption : string read FCaption write FCaption;
    { Condition message that appears when the dialog box appears }
    property Condition : string read FCondition write FCondition;
    { Description message that appears when the dialog box appears }
    property Description : string read FDescription write FDescription;
    { Question that appears when the dialog box appears }
    property Question : string read FQuestion write FQuestion;
    { Enable inherited properties }
    property Font;
    property ParentFont;
  end;

{ TProWarningDialog }

  TProWarningDialog = class(TProCommonDialog)
  private
    { Variables for properties }
    FCaption : string;
    FWarningMessage : string;
    FAlignment : TAlignment;
    FVerticalAlignment : TVerticalAlignment;
  public
    constructor Create(AOwner : TComponent); override;
    function Execute : Boolean;
  published
    { Caption of the dialog box }
    property Caption : string read FCaption write FCaption;
    { Warning message that appears when the dialog box appears }
    property WarningMessage : string read FWarningMessage
      write FWarningMessage;
    { Horozontal alignment of warning message }
    property Alignment : TAlignment read FAlignment write FAlignment
      default taLeftJustify;
    { Vertical alignment of warning message }
    property VerticalAlignment : TVerticalAlignment read FVerticalAlignment
      write FVerticalAlignment default vaCenter;
    { Enable inherited properties }
    property Font;
    property ParentFont;
  end;

{ TProInfoDialog }

  TProInfoDialog = class(TProCommonDialog)
  private
    { Variables for properties }
    FCaption : string;
    FInfoMessage : string;
    FAlignment : TAlignment;
    FVerticalAlignment : TVerticalAlignment;
  public
    constructor Create(AOwner : TComponent); override;
    procedure Execute;
  published
    { Caption of the dialog box }
    property Caption : string read FCaption write FCaption;
    { Information message that appears when the dialog box appears }
    property InfoMessage : string read FInfoMessage write FInfoMessage;
    { Horizontal alignment of warning message }
    property Alignment : TAlignment read FAlignment write FAlignment
      default taLeftJustify;
    { Vertical alignment of warning message }
    property VerticalAlignment : TVerticalAlignment read FVerticalAlignment
      write FVerticalAlignment default vaCenter;
    { Enable inherited properties }
    property Font;
    property ParentFont;
  end;

{ TProStopDialog }

  TProStopDialog = class(TProCommonDialog)
  private
    { Variables for properties }
    FCaption : string;
    FStopMessage : string;
    FAlignment : TAlignment;
    FVerticalAlignment : TVerticalAlignment;
  public
    constructor Create(AOwner : TComponent); override;
    procedure Execute;
  published
    { Caption of the dialog box }
    property Caption : string read FCaption write FCaption;
    { Stop message that appears when the dialog box appears }
    property StopMessage : string read FStopMessage write FStopMessage;
    { Horizontal alignment of warning message }
    property Alignment : TAlignment read FAlignment write FAlignment
      default taLeftJustify;
    { Vertical alignment of warning message }
    property VerticalAlignment : TVerticalAlignment read FVerticalAlignment
      write FVerticalAlignment default vaCenter;
    { Enable inherited properties }
    property Font;
    property ParentFont;
  end;

{ TProPasswordDialog }

  TProPasswordDialog = class(TProCommonDialog)
  private
    { Variables for properties }
    FCaption : string;
    FPassword : string;
  public
    function Execute : Boolean;
    { The password the user enters becomes the value of Password }
    { when the Execute function returns                          }
    property Password : string read FPassword;
  published
    { Caption of the dialog box }
    property Caption : string read FCaption write FCaption;
    { Enable inherited properties }
    property Font;
    property ParentFont;
  end;

{ TProLoginDialog }

  TProLoginDialog = class(TProCommonDialog)
  private
    { Variables for properties }
    FCaption : string;
    FUserName : string;
    FPassword : string;
  public
    function Execute : Boolean;
    { The user name the user enters becomes the value of UserName }
    { when the Execute function returns                           }
    property UserName : string read FUserName;
    { The password the user enters becomes the value of Password }
    { when the Execute function returns                          }
    property Password : string read FPassword;
  published
    { Caption of the dialog box }
    property Caption : string read FCaption write FCaption;
    { Enable inherited properties }
    property Font;
    property ParentFont;
  end;

{ TProImageViewDialog }

  TProImageViewDialog = class(TProCommonDialog)
  private
    { Variables for properties }
    FBorderStyle : TBorderStyle;
    FCaption : string;
    FModal : Boolean;
    FPicture : TPicture;
    { Private variables }
    Form : TForm;
    { Property access methods }
    procedure SetPicture(Value : TPicture);
  public
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
    procedure Execute;
  published
    { Specifies whether the image has a lowered 3D frame drawn around it }
    property BorderStyle : TBorderStyle read FBorderStyle write FBorderStyle
      default bsNone;
    { Caption of the dialog box }
    property Caption : string read FCaption write FCaption;
    { Set to True to make the modal dialog or to False to make }
    { the modeless dialog                                      }
    property Modal : Boolean read FModal write FModal default True;
    { Specifies the image that appears on the dialog box }
    property Picture : TPicture read FPicture write SetPicture;
  end;

{ Dialog functions }

function SelDirDialog(var ADirectory : string;
  ALanguage : TDialogLanguage; AHelpContext : LongInt) : Boolean;
{ Select directory dialog interface routine }

function ConfirmDialog(const ACaption, ACondition, ADescription,
  AQuestion : string; var ACancel : Boolean;
  ALanguage : TDialogLanguage; AHelpContext : LongInt) : Boolean;
{ Confirm dialog interface routine }

function WarningDialog(const ACaption, AWarningMessage : string;
  AAlignment : TAlignment; AVerticalAlignment : TVerticalAlignment;
  ALanguage : TDialogLanguage; AHelpContext : LongInt) : Boolean;
{ Warning dialog interface routine }

procedure InfoDialog(const ACaption, AInfoMessage : string;
  AAlignment : TAlignment; AVerticalAlignment : TVerticalAlignment;
  ALanguage : TDialogLanguage; AHelpContext : LongInt);
{ Information dialog interface routine }

procedure StopDialog(const ACaption, AStopMessage : string;
  AAlignment : TAlignment; AVerticalAlignment : TVerticalAlignment;
  ALanguage : TDialogLanguage; AHelpContext : LongInt);
{ Stop dialog interface routine }

function PasswordDialog(const ACaption : string; var APassword : string;
  ALanguage : TDialogLanguage; AHelpContext : LongInt) : Boolean;
{ Password dialog interface routine }

function LoginDialog(const ACaption : string; var AUserName, APassword : string;
  ALanguage : TDialogLanguage; AHelpContext : LongInt) : Boolean;
{ Login dialog interface routine }

implementation

uses ProConst;

{$R PRODLGS.RES}


{ TProCommonDialog }

constructor TProCommonDialog.Create(AOwner : TComponent);
{ Overrides the constructor to initialize variables }
begin
  inherited Create(AOwner);
  FFont := TFont.Create;
  FParentFont := True;
end; { TProCommonDialog.Create }

destructor TProCommonDialog.Destroy;
{ Overrides the destructor to uninitialize variables }
begin
  FFont.Free;
  inherited Destroy;
end; { TProCommonDialog.Destroy }

procedure TProCommonDialog.SetFont(Value : TFont);
{ Sets the font characteristics of the dialog form }
begin
  FFont.Assign(Value);
end; { TProCommonDialog.SetFont }


{ TCommonDialogForm }

type
  TCommonDialogForm = class(TForm)
  private
    function CreateLabel(Ident : Integer; AFocusControl : TWinControl;
      ALeft, ATop, AWidth, AHeight : Integer) : TProLabel;
    function CreateButton(AKind : TProButtonKind;
      ALeft, ATop : Integer) : TProButton;
  end;

function TCommonDialogForm.CreateLabel(Ident : Integer;
  AFocusControl : TWinControl; ALeft, ATop, AWidth,
  AHeight : Integer) : TProLabel;
{ Creates label }
begin
  Result := TProLabel.Create(Self);
  with Result do
  begin
    Parent := Self;
    AutoSize := False;
    WordWrap := True;           { For confirm, warning and info dialogs }
    SetBounds(ALeft, ATop, AWidth, AHeight);
    if Ident <> 0 then
      Caption := LoadStr(Ident)
    else
      Caption := '';
    FocusControl := AFocusControl;
  end;
end; { TCommonDialogForm.CreateLabel }

function TCommonDialogForm.CreateButton(AKind : TProButtonKind;
  ALeft, ATop : Integer) : TProButton;
{ Creates button }
begin
  Result := TProButton.Create(Self);
  with Result do
  begin
    Parent := Self;
    SetBounds(ALeft, ATop, 100, 30);
    Kind := AKind;
  end;
end; { TCommonDialogForm.CreateButton }


{ TDirectoryForm }

type
  TDirectoryForm = class(TCommonDialogForm)
    DirNameLabel : TProLabel;
    DirLabel : TProLabel;
    DirEdit : TEdit;
    DirListLabel : TProLabel;
    DirListBox : TDirectoryListBox;
    DriveLabel : TProLabel;
    DriveComboBox : TDriveComboBox;
    OKButton : TProButton;
    CancelButton : TProButton;
    HelpButton : TProButton;
    ConfirmDialog : TProConfirmDialog;
    procedure DirListBoxChange(Sender : TObject);
    procedure OKButtonClick(Sender : TObject);
  private
    FGetInto : Boolean;
    FAllowEdit : Boolean;
    FAllowCreate : Boolean;
    FPrompt : Boolean;
    procedure SetAllowEdit(Value : Boolean);
  public
    constructor Create(AOwner : TComponent);
    property GetInto : Boolean read FGetInto write FGetInto;
    property AllowCreate : Boolean read FAllowCreate write FAllowCreate;
    property Prompt : Boolean read FPrompt write FPrompt;
    property AllowEdit : Boolean read FAllowEdit write SetAllowEdit;
  end;

constructor TDirectoryForm.Create(AOwner : TComponent);
{ Overrides the constructor to initialize variables }
begin
  inherited CreateNew(AOwner {$IFDEF CBUILDER_1} , 1 {$ENDIF} );
  BorderStyle := bsDialog;
  Caption := LoadStr(SSelectDirCap);
  ClientWidth := 325;
  ClientHeight := 311;
  Position := poScreenCenter;

  DirLabel := CreateLabel(0, nil, 15, 35, 294, 16);

  DirEdit := TEdit.Create(Self);
  with DirEdit do
  begin
    Parent := Self;
    SetBounds(15, 35, 294, 24);
    Visible := False;
  end;

  DirListBox := TDirectoryListBox.Create(Self);
  with DirListBox do
  begin
    Parent := Self;
    SetBounds(15, 90, 176, 146);
    OnChange := DirListBoxChange;
  end;

  DirLabel.Caption := DirListBox.Directory;
  DirEdit.Text := DirListBox.Directory;

  DriveComboBox := TDriveComboBox.Create(Self);
  with DriveComboBox do
  begin
    Parent := Self;
    SetBounds(15, 270, 176, 22);
    DirList := DirListBox;
  end;

  { Descriptions of controls }
  DirNameLabel := CreateLabel(SDirNameCap, DirEdit, 15, 15, 294, 16);
  DirListLabel := CreateLabel(SDirsCap, DirListBox, 15, 70, 176, 16);
  DriveLabel := CreateLabel(SDrivesCap, DriveComboBox, 15, 250, 176, 16);

  OKButton := CreateButton(pbkOk, 209, 90);
  OKButton.ModalResult := mrNone;
  OKButton.OnClick := OKButtonClick;
  CancelButton := CreateButton(pbkCancel, 209, 125);
  HelpButton := CreateButton(pbkHelp, 209, 160);

  ActiveControl := DirListBox;

  ConfirmDialog := TProConfirmDialog.Create(Self);
  with ConfirmDialog do
  begin
    Caption := LoadStr(SDirConfirmCaption);
    Condition := LoadStr(SDirConfirmCondition);
    Question := LoadStr(SDirConfirmQuestion);
  end;
end; { TDirectoryForm.Create }

procedure TDirectoryForm.DirListBoxChange(Sender : TObject);
{ DirListBox.OnChange event handler }
begin
  DirLabel.Caption := DirListBox.Directory;
  DirEdit.Text := DirListBox.Directory;
end; { TDirectoryForm.DirListBoxChange }

procedure TDirectoryForm.OKButtonClick(Sender : TObject);
{ OKButton.OnClick event handler }
begin
  ConfirmDialog.Description := DirEdit.Text;
  if AllowEdit and AllowCreate and (not DirectoryExists(DirEdit.Text)) and
    (not Prompt or (Prompt and ConfirmDialog.Execute)) then
  begin
    MkDir(DirEdit.Text);
  end;
  if GetInto and DirectoryExists(DirEdit.Text) then
    ChDir(DirEdit.Text);
  if not ConfirmDialog.Cancel then
    ModalResult := mrOk;
end; { TDirectoryForm.OKButtonClick }

procedure TDirectoryForm.SetAllowEdit(Value : Boolean);
{ Sets new value when property is changed }
begin
  if FAllowEdit <> Value then
  begin
    FAllowEdit := Value;
    DirLabel.Visible := not Value;
    DirEdit.Visible := Value;
  end;
end; { TDirectoryForm.SetAllowEdit }


{ TProSelDirDialog }

function TProSelDirDialog.Execute : Boolean;
{ Displays the dialog box }
begin
  with TDirectoryForm.Create(Application) do
  try
    if FDirectory <> '' then
    begin
      DirLabel.Caption := FDirectory;
      DirEdit.Text := FDirectory;
      if DirectoryExists(FDirectory) then
        DirListBox.Directory := FDirectory;
    end;

    case Language of
      dlRussian :
        begin
          Caption := LoadStr(SSelectDirCapRus);
          DirNameLabel.Caption := LoadStr(SDirNameCapRus);
          DirListLabel.Caption := LoadStr(SDirsCapRus);
          DriveLabel.Caption := LoadStr(SDrivesCapRus);
          CancelButton.Kind := pbkCancelRus;
          HelpButton.Kind := pbkHelpRus;
          ConfirmDialog.Caption := LoadStr(SDirConfirmCaptionRus);
          ConfirmDialog.Condition := LoadStr(SDirConfirmConditionRus);
          ConfirmDialog.Question := LoadStr(SDirConfirmQuestionRus);
        end;
      dlUkranian :
        begin
          Caption := LoadStr(SSelectDirCapUkr);
          DirNameLabel.Caption := LoadStr(SDirNameCapUkr);
          DirListLabel.Caption := LoadStr(SDirsCapUkr);
          DriveLabel.Caption := LoadStr(SDrivesCapUkr);
          CancelButton.Kind := pbkCancelUkr;
          HelpButton.Kind := pbkHelpUkr;
          ConfirmDialog.Caption := LoadStr(SDirConfirmCaptionUkr);
          ConfirmDialog.Condition := LoadStr(SDirConfirmConditionUkr);
          ConfirmDialog.Question := LoadStr(SDirConfirmQuestion);
        end;
    end;
    ConfirmDialog.Language := Language;

    HelpContext := Self.HelpContext;
    if HelpContext = 0 then
      HelpButton.Visible := False;

    GetInto := ddGetInto in Options;
    AllowEdit := ddAllowEdit in Options;
    AllowCreate := ddAllowCreate in Options;
    Prompt := ddPrompt in Options;

    if Self.ParentFont and (Self.Owner is TForm) then
      Font.Assign(TForm(Self.Owner).Font)
    else
      Font := Self.Font;

    if PixelsPerInch <> 120 then
      ScaleBy(Screen.PixelsPerInch, 120);

    Result := ShowModal = mrOk;
    if Result then FDirectory := DirEdit.Text;
  finally
    Free;
  end;
end; { TProSelDirDialog.Execute }


{ TConfirmForm }

type
  TConfirmForm = class(TCommonDialogForm)
    Bevel : TBevel;
    Image : TImage;
    ConditionLabel : TProLabel;
    DescriptionLabel : TProLabel;
    QuestionLabel : TProLabel;
    YesButton : TProButton;
    NoButton : TProButton;
    CancelButton : TProButton;
    HelpButton : TProButton;
  public
    constructor Create(AOwner : TComponent);
  end;

constructor TConfirmForm.Create(AOwner : TComponent);
{ Overrides the constructor to initialize variables }
begin
  inherited CreateNew(AOwner {$IFDEF CBUILDER_1} , 1 {$ENDIF} );
  BorderStyle := bsDialog;
  Caption := LoadStr(SConfirmCap);
  ClientWidth := 450;
  ClientHeight := 218;
  Position := poScreenCenter;

  Bevel := TBevel.Create(Self);
  with Bevel do
  begin
    Parent := Self;
    SetBounds(10, 10, 429, 156);
    Shape := bsFrame;
  end;

  Image := TImage.Create(Self);
  with Image do
  begin
    Parent := Self;
    Picture.Bitmap.Handle := LoadBitmap(HInstance, 'QUESTIONICON');
    SetBounds(15, 51, 48, 64);
  end;

  ConditionLabel := CreateLabel(0, nil, 70, 25, 356, 36);
  DescriptionLabel := CreateLabel(0, nil, 110, 72, 316, 36);
  QuestionLabel := CreateLabel(0, nil, 70, 120, 356, 36);

  YesButton := CreateButton(pbkYes, 10, 177);
  NoButton := CreateButton(pbkNo, 120, 177);
  CancelButton := CreateButton(pbkCancel, 230, 177);
  HelpButton := CreateButton(pbkHelp, 340, 177);
end; { TConfirmForm.Create }


{ TProConfirmDialog }

function TProConfirmDialog.Execute : Boolean;
{ Displays the dialog box }
var
  DlgResult : TModalResult;
begin
  with TConfirmForm.Create(Application) do
  try
    ConditionLabel.Caption := FCondition;
    DescriptionLabel.Caption := FDescription;
    QuestionLabel.Caption := FQuestion;

    case Language of
      dlRussian :
        begin
          Caption := LoadStr(SConfirmCapRus);
          YesButton.Kind := pbkYesRus;
          NoButton.Kind := pbkNoRus;
          CancelButton.Kind := pbkCancelRus;
          HelpButton.Kind := pbkHelpRus;
        end;
      dlUkranian :
        begin
          Caption := LoadStr(SConfirmCapUkr);
          YesButton.Kind := pbkYesUkr;
          NoButton.Kind := pbkNoUkr;
          CancelButton.Kind := pbkCancelUkr;
          HelpButton.Kind := pbkHelpUkr;
        end;
    end;
    if FCaption <> '' then Caption := FCaption;

    HelpContext := Self.HelpContext;
    if HelpContext = 0 then
    begin
      HelpButton.Visible := False;
      YesButton.Left := 65;
      NoButton.Left := 175;
      CancelButton.Left := 285;
    end;

    if Self.ParentFont and (Self.Owner is TForm) then
      Font.Assign(TForm(Self.Owner).Font)
    else
      Font := Self.Font;

    if PixelsPerInch <> 120 then
    begin
      ScaleBy(Screen.PixelsPerInch, 120);
      Image.SetBounds(Image.Left, (Bevel.Top + Bevel.Height - 64) shr 1,
        48, 64);
    end;

    DlgResult := ShowModal;
    FCancel := DlgResult = mrCancel;
    Result := DlgResult = mrYes;
  finally
    Free;
  end;
end; { TProConfirmDialog.Execute }


{ TWarningForm }

type
  TWarningForm = class(TCommonDialogForm)
    Bevel : TBevel;
    Image : TImage;
    WarningLabel : TProLabel;
    OKButton : TProButton;
    CancelButton : TProButton;
    HelpButton : TProButton;
  public
    constructor Create(AOwner : TComponent);
  end;

constructor TWarningForm.Create(AOwner : TComponent);
{ Overrides the constructor to initialize variables }
begin
  inherited CreateNew(AOwner {$IFDEF CBUILDER_1} , 1 {$ENDIF} );
  BorderStyle := bsDialog;
  Caption := LoadStr(SWarningCap);
  ClientWidth := 402;
  ClientHeight := 169;
  Position := poScreenCenter;

  Bevel := TBevel.Create(Self);
  with Bevel do
  begin
    Parent := Self;
    SetBounds(10, 10, 381, 106);
    Shape := bsFrame;
  end;

  Image := TImage.Create(Self);
  with Image do
  begin
    Parent := Self;
    Picture.Bitmap.Handle := LoadBitmap(HInstance, 'EXCLAMATIONICON');
    SetBounds(15, 29, 48, 64);
  end;

  WarningLabel := CreateLabel(0, nil, 70, 20, 311, 86);

  OKButton := CreateButton(pbkOK, 36, 128);
  CancelButton := CreateButton(pbkCancel, 151, 128);
  HelpButton := CreateButton(pbkHelp, 266, 128);
end; { TWarningForm.Create }


{ TProWarningDialog }

constructor TProWarningDialog.Create(AOwner : TComponent);
{ Overrides the constructor to initialize variables }
begin
  inherited Create(AOwner);
  FVerticalAlignment := vaCenter;
end; { TProWarningDialog.Create }

function TProWarningDialog.Execute : Boolean;
{ Displays the dialog box }
begin
  with TWarningForm.Create(Application) do
  try
    WarningLabel.Caption := FWarningMessage;
    WarningLabel.Alignment := FAlignment;
    WarningLabel.VerticalAlignment := FVerticalAlignment;

    case Language of
      dlRussian :
        begin
          Caption := LoadStr(SWarningCapRus);
          CancelButton.Kind := pbkCancelRus;
          HelpButton.Kind := pbkHelpRus;
        end;
      dlUkranian :
        begin
          Caption := LoadStr(SWarningCapUkr);
          CancelButton.Kind := pbkCancelUkr;
          HelpButton.Kind := pbkHelpUkr;
        end;
    end;
    if FCaption <> '' then Caption := FCaption;

    HelpContext := Self.HelpContext;
    if HelpContext = 0 then
    begin
      HelpButton.Visible := False;
      OKButton.Left := 93;
      CancelButton.Left := 208;
    end;

    if Self.ParentFont and (Self.Owner is TForm) then
      Font.Assign(TForm(Self.Owner).Font)
    else
      Font := Self.Font;

    if PixelsPerInch <> 120 then
    begin
      ScaleBy(Screen.PixelsPerInch, 120);
      Image.SetBounds(Image.Left, (Bevel.Top + Bevel.Height - 64) shr 1,
        48, 64);
    end;

    Result := ShowModal = mrOk;
  finally
    Free;
  end;
end; { TProWarningDialog.Execute }


{ TInfoForm }

type
  TInfoForm = class(TCommonDialogForm)
    Bevel : TBevel;
    Image : TImage;
    InfoLabel : TProLabel;
    OKButton : TProButton;
    HelpButton : TProButton;
  public
    constructor Create(AOwner : TComponent);
  end;

constructor TInfoForm.Create(AOwner : TComponent);
{ Overrides the constructor to initialize variables }
begin
  inherited CreateNew(AOwner {$IFDEF CBUILDER_1} , 1 {$ENDIF} );
  BorderStyle := bsDialog;
  Caption := LoadStr(SInfoCap);
  ClientWidth := 402;
  ClientHeight := 169;
  Position := poScreenCenter;

  Bevel := TBevel.Create(Self);
  with Bevel do
  begin
    Parent := Self;
    SetBounds(10, 10, 381, 106);
    Shape := bsFrame;
  end;

  Image := TImage.Create(Self);
  with Image do
  begin
    Parent := Self;
    Picture.Bitmap.Handle := LoadBitmap(HInstance, 'INFORMATIONICON');
    SetBounds(15, 29, 48, 64);
  end;

  InfoLabel := CreateLabel(0, nil, 70, 20, 311, 86);

  OKButton := CreateButton(pbkOK, 94, 128);
  HelpButton := CreateButton(pbkHelp, 208, 128);
end; { TInfoForm.Create }


{ TProInfoDialog }

constructor TProInfoDialog.Create(AOwner : TComponent);
{ Overrides the constructor to initialize variables }
begin
  inherited Create(AOwner);
  FVerticalAlignment := vaCenter;
end; { TProInfoDialog.Create }

procedure TProInfoDialog.Execute;
{ Displays the dialog box }
begin
  with TInfoForm.Create(Application) do
  try
    InfoLabel.Caption := FInfoMessage;
    InfoLabel.Alignment := FAlignment;
    InfoLabel.VerticalAlignment := FVerticalAlignment;

    case Language of
      dlRussian :
        begin
          Caption := LoadStr(SInfoCapRus);
          HelpButton.Kind := pbkHelpRus;
        end;
      dlUkranian :
        begin
          Caption := LoadStr(SInfoCapUkr);
          HelpButton.Kind := pbkHelpUkr;
        end;
    end;
    if FCaption <> '' then Caption := FCaption;

    HelpContext := Self.HelpContext;
    if HelpContext = 0 then
    begin
      HelpButton.Visible := False;
      OKButton.Left := 151;
    end;

    if Self.ParentFont and (Self.Owner is TForm) then
      Font.Assign(TForm(Self.Owner).Font)
    else
      Font := Self.Font;

    if PixelsPerInch <> 120 then
    begin
      ScaleBy(Screen.PixelsPerInch, 120);
      Image.SetBounds(Image.Left, (Bevel.Top + Bevel.Height - 64) shr 1,
        48, 64);
    end;

    ShowModal;
  finally
    Free;
  end;
end; { TProInfoDialog.Execute }


{ TStopForm }

type
  TStopForm = class(TCommonDialogForm)
    Bevel : TBevel;
    Image : TImage;
    StopLabel : TProLabel;
    CancelButton : TProButton;
    HelpButton : TProButton;
  public
    constructor Create(AOwner : TComponent);
  end;

constructor TStopForm.Create(AOwner : TComponent);
{ Overrides the constructor to initialize variables }
begin
  inherited CreateNew(AOwner {$IFDEF CBUILDER_1} , 1 {$ENDIF} );
  BorderStyle := bsDialog;
  Caption := LoadStr(SStopCap);
  ClientWidth := 402;
  ClientHeight := 169;
  Position := poScreenCenter;

  Bevel := TBevel.Create(Self);
  with Bevel do
  begin
    Parent := Self;
    SetBounds(10, 10, 381, 106);
    Shape := bsFrame;
  end;

  Image := TImage.Create(Self);
  with Image do
  begin
    Parent := Self;
    Picture.Bitmap.Handle := LoadBitmap(HInstance, 'STOPICON');
    SetBounds(15, 29, 48, 64);
  end;

  StopLabel := CreateLabel(0, nil, 70, 20, 311, 86);

  CancelButton := CreateButton(pbkCancel, 94, 128);
  HelpButton := CreateButton(pbkHelp, 208, 128);
end; { TStopForm.Create }


{ TProStopDialog }

constructor TProStopDialog.Create(AOwner : TComponent);
{ Overrides the constructor to initialize variables }
begin
  inherited Create(AOwner);
  FVerticalAlignment := vaCenter;
end; { TProStopDialog.Create }

procedure TProStopDialog.Execute;
{ Displays the dialog box }
begin
  with TStopForm.Create(Application) do
  try
    StopLabel.Caption := FStopMessage;
    StopLabel.Alignment := FAlignment;
    StopLabel.VerticalAlignment := FVerticalAlignment;

    case Language of
      dlRussian :
        begin
          Caption := LoadStr(SStopCapRus);
          HelpButton.Kind := pbkHelpRus;
          CancelButton.Kind := pbkCancelRus;
        end;
      dlUkranian :
        begin
          Caption := LoadStr(SStopCapUkr);
          HelpButton.Kind := pbkHelpUkr;
          CancelButton.Kind := pbkCancelUkr;
        end;
    end;
    if FCaption <> '' then Caption := FCaption;

    HelpContext := Self.HelpContext;
    if HelpContext = 0 then
    begin
      HelpButton.Visible := False;
      CancelButton.Left := 151;
    end;

    if Self.ParentFont and (Self.Owner is TForm) then
      Font.Assign(TForm(Self.Owner).Font)
    else
      Font := Self.Font;

    if PixelsPerInch <> 120 then
    begin
      ScaleBy(Screen.PixelsPerInch, 120);
      Image.SetBounds(Image.Left, (Bevel.Top + Bevel.Height - 64) shr 1,
        48, 64);
    end;

    ShowModal;
  finally
    Free;
  end;
end; { TProStopDialog.Execute }


{ TPasswordForm }

type
  TPasswordForm = class(TCommonDialogForm)
    PasswordLabel : TProLabel;
    PasswordEdit : TEdit;
    OKButton : TProButton;
    CancelButton : TProButton;
    HelpButton : TProButton;
    procedure EditChange(Sender : TObject);
  public
    constructor Create(AOwner : TComponent);
  end;

constructor TPasswordForm.Create(AOwner : TComponent);
{ Overrides the constructor to initialize variables }
begin
  inherited CreateNew(AOwner {$IFDEF CBUILDER_1} , 1 {$ENDIF} );
  BorderStyle := bsDialog;
  Caption := LoadStr(SPasswordCap);
  ClientWidth := 469;
  ClientHeight := 117;
  Position := poScreenCenter;

  with TBevel.Create(Self) do
  begin
    Parent := Self;
    SetBounds(10, 10, 448, 61);
    Shape := bsFrame;
  end;

  PasswordEdit := TEdit.Create(Self);
  with PasswordEdit do
  begin
    Parent := Self;
    PasswordChar := '*';
    Text := '';
    SetBounds(160, 28, 281, 24);
    OnChange := EditChange;
  end;

  PasswordLabel := CreateLabel(SEnterPasswordCap, PasswordEdit,
    25, 32, 96, 16);

  OKButton := CreateButton(pbkOk, 142, 80);
  OKButton.Enabled := False;
  CancelButton := CreateButton(pbkCancel, 250, 80);
  HelpButton := CreateButton(pbkHelp, 358, 80);
end; { TPasswordForm.Create }

procedure TPasswordForm.EditChange(Sender : TObject);
{ PasswordEdit.OnChange event handler }
begin
  OKButton.Enabled := PasswordEdit.Text <> '';
end; { TPasswordForm.EditChange }


{ TProPasswordDialog }

function TProPasswordDialog.Execute : Boolean;
{ Displays the dialog box }
begin
  with TPasswordForm.Create(Application) do
  try
    case Language of
      dlRussian :
        begin
          Caption := LoadStr(SPasswordCapRus);
          PasswordLabel.Caption := LoadStr(SEnterPasswordCapRus);
          CancelButton.Kind := pbkCancelRus;
          HelpButton.Kind := pbkHelpRus;
        end;
      dlUkranian :
        begin
          Caption := LoadStr(SPasswordCapUkr);
          PasswordLabel.Caption := LoadStr(SEnterPasswordCapUkr);
          CancelButton.Kind := pbkCancelUkr;
          HelpButton.Kind := pbkHelpUkr;
        end;
    end;
    if FCaption <> '' then Caption := FCaption;

    HelpContext := Self.HelpContext;
    if HelpContext = 0 then
    begin
      HelpButton.Visible := False;
      OKButton.Left := 250;
      CancelButton.Left := 358;
    end;

    if Self.ParentFont and (Self.Owner is TForm) then
      Font.Assign(TForm(Self.Owner).Font)
    else
      Font := Self.Font;

    if PixelsPerInch <> 120 then
      ScaleBy(Screen.PixelsPerInch, 120);

    Result := ShowModal = mrOk;

    FPassword := PasswordEdit.Text;
  finally
    Free;
  end;
end; { TProPasswordDialog.Execute }


{ TLoginForm }

type
  TLoginForm = class(TCommonDialogForm)
    PasswordLabel : TProLabel;
    PasswordEdit : TEdit;
    UserNameLabel: TProLabel;
    UserNameEdit: TEdit;
    OKButton : TProButton;
    CancelButton : TProButton;
    HelpButton : TProButton;
    procedure EditChange(Sender : TObject);
  public
    constructor Create(AOwner : TComponent);
  end;

constructor TLoginForm.Create(AOwner : TComponent);
{ Override the constructor to initialize variables }
begin
  inherited CreateNew(AOwner {$IFDEF CBUILDER_1} , 1 {$ENDIF} );
  BorderStyle := bsDialog;
  Caption := LoadStr(SLoginCap);
  ClientWidth := 480;
  ClientHeight := 156;
  Position := poScreenCenter;

  with TBevel.Create(Self) do
  begin
    Parent := Self;
    SetBounds(10, 10, 459, 98);
    Shape := bsFrame;
  end;

  UserNameEdit := TEdit.Create(Self);
  with UserNameEdit do
  begin
    Parent := Self;
    Text := '';
    SetBounds(170, 30, 281, 24);
    OnChange := EditChange;
  end;

  UserNameLabel := CreateLabel(SUserNameCap, UserNameEdit,
    25, 34, 72, 16);

  PasswordEdit := TEdit.Create(Self);
  with PasswordEdit do
  begin
    Parent := Self;
    PasswordChar := '*';
    Text := '';
    SetBounds(170, 65, 281, 24);
    OnChange := EditChange;
  end;

  PasswordLabel := CreateLabel(SPromptPasswordCap, PasswordEdit,
    25, 69, 63, 16);

  OKButton := CreateButton(pbkOk, 151, 118);
  OKButton.Enabled := False;
  CancelButton := CreateButton(pbkCancel, 260, 118);
  HelpButton := CreateButton(pbkHelp, 369, 118);
end; { TLoginForm.Create }

procedure TLoginForm.EditChange(Sender : TObject);
{ PasswordEdit.OnChange event handler }
begin
  OKButton.Enabled := (UserNameEdit.Text <> '') and
    (PasswordEdit.Text <> '');
end; { TLoginForm.EditChange }


{ TProLoginDialog }

function TProLoginDialog.Execute : Boolean;
{ Displays the dialog box }
begin
  with TLoginForm.Create(Application) do
  try
    case Language of
      dlRussian :
        begin
          Caption := LoadStr(SLoginCapRus);
          UserNameLabel.Caption := LoadStr(SUserNameCapRus);
          PasswordLabel.Caption := LoadStr(SPromptPasswordCapRus);
          CancelButton.Kind := pbkCancelRus;
        end;
      dlUkranian :
        begin
          Caption := LoadStr(SLoginCapUkr);
          UserNameLabel.Caption := LoadStr(SUserNameCapUkr);
          PasswordLabel.Caption := LoadStr(SPromptPasswordCapUkr);
          CancelButton.Kind := pbkCancelUkr;
        end;
    end;
    if FCaption <> '' then Caption := FCaption;

    HelpContext := Self.HelpContext;
    if HelpContext = 0 then
    begin
      HelpButton.Visible := False;
      OKButton.Left := 260;
      CancelButton.Left := 369;
    end;

    if Self.ParentFont and (Self.Owner is TForm) then
      Font.Assign(TForm(Self.Owner).Font)
    else
      Font := Self.Font;

    if PixelsPerInch <> 120 then
      ScaleBy(Screen.PixelsPerInch, 120);

    Result := ShowModal = mrOk;

    FUserName := UserNameEdit.Text;;
    FPassword := PasswordEdit.Text;
  finally
    Free;
  end;
end; { TProLoginDialog.Execute }


{ TImageViewForm }

type
  TImageViewForm = class(TCommonDialogForm)
    Panel : TProPanel;
    Image : TImage;
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormDestroy(Sender: TObject);
    procedure PictureChanged(Sender : TObject);
  public
    FormOwner : TComponent;
    constructor Create(AOwner : TComponent);
  end;

constructor TImageViewForm.Create(AOwner : TComponent);
{ Overrides the constructor to initialize variables }
begin
  inherited CreateNew(AOwner {$IFDEF CBUILDER_1} , 1 {$ENDIF} );
  BorderStyle := bsDialog;
  Caption := LoadStr(SImageViewCap);
  Position := poScreenCenter;

  Panel := TProPanel.Create(Self);
  with Panel do
  begin
    Parent := Self;
    BevelOuter := bvNone;
    Align := alClient;
  end;

  Image := TImage.Create(Panel);
  with Image do
  begin
    Parent := Panel;
    Align := alClient;
    Picture.OnChange := PictureChanged;
  end;

  OnClose := FormClose;
  OnDestroy := FormDestroy;
end; { TImageViewForm.Create }

procedure TImageViewForm.FormClose(Sender: TObject; var Action: TCloseAction);
{ OnClose event handler }
begin
  Action := caFree;
end; { TImageViewForm.FormClose }

procedure TImageViewForm.FormDestroy(Sender: TObject);
{ OnDestroy event handler }
begin
  TProImageViewDialog(FormOwner).Form := nil;
end; { TImageViewForm.FormDestroy }

procedure TImageViewForm.PictureChanged(Sender : TObject);
{ Image.OnChange event handler }
begin
  ClientWidth := Image.Picture.Width;
  ClientHeight := Image.Picture.Height;
  if Panel.BevelCtl3D then
  begin
    ClientWidth := ClientWidth + 4;
    ClientHeight := ClientHeight + 4;
  end;
end; { TImageViewForm.PictureChanged }


{ TProImageViewDialog }

constructor TProImageViewDialog.Create(AOwner : TComponent);
{ Overrides the constructor to initialize variables }
begin
  inherited Create(AOwner);
  FModal := True;
  FPicture := TPicture.Create;
end; { TProImageViewDialog.Create }

destructor TProImageViewDialog.Destroy;
{ Overrides the destructor to uninitialize variables }
begin
  FPicture.Free;
  inherited Destroy;
end; { TProImageViewDialog.Destroy }

procedure TProImageViewDialog.SetPicture(Value : TPicture);
{ Sets the picture }
begin
  FPicture.Assign(Value);
end; { TProImageViewDialog.SetPicture }

procedure TProImageViewDialog.Execute;
{ Displays the dialog box }
begin
  if Form <> nil then
    Form.SetFocus
  else
  begin
    Form := TImageViewForm.Create(Application);
    with TImageViewForm(Form) do
    try
      TImageViewForm(Form).FormOwner := Self;

      case Language of
        dlRussian  : Caption := LoadStr(SImageViewCapRus);
        dlUkranian : Caption := LoadStr(SImageViewCapUkr);
      end;
      if FCaption <> '' then Caption := FCaption;

      if FBorderStyle = bsSingle then
      begin
        Panel.BevelOuter := bvLowered;
        Panel.BevelCtl3D := True;
      end;
      Image.Picture := FPicture;

      if FModal then
      begin
        Form.ShowModal;
        Form.Free;
      end
      else
        Form.Show;
    except
      Form.Free;
      raise;
    end;
  end;
end; { TProImageViewDialog.Execute }


{ Dialog functions }

function SelDirDialog(var ADirectory : string;
  ALanguage : TDialogLanguage; AHelpContext : LongInt) : Boolean;
{ Select directory dialog interface routine }
begin
  with TProSelDirDialog.Create(Application) do
  try
    Directory := ADirectory;
    Language := ALanguage;
    HelpContext := AHelpContext;
    Result := Execute;
    if Result then ADirectory := Directory;
  finally
    Free;
  end;
end; { SelDirDialog }

function ConfirmDialog(const ACaption, ACondition, ADescription,
  AQuestion : string; var ACancel : Boolean;
  ALanguage : TDialogLanguage; AHelpContext : LongInt) : Boolean;
{ Confirm dialog interface routine }
begin
  with TProConfirmDialog.Create(Application) do
  try
    Caption := ACaption;
    Condition := ACondition;
    Description := ADescription;
    Question := AQuestion;
    Language := ALanguage;
    HelpContext := AHelpContext;
    Result := Execute;
    ACancel := Cancel;
  finally
    Free;
  end;
end; { ConfirmDialog }

function WarningDialog(const ACaption, AWarningMessage : string;
  AAlignment : TAlignment; AVerticalAlignment : TVerticalAlignment;
  ALanguage : TDialogLanguage; AHelpContext : LongInt) : Boolean;
{ Warning dialog interface routine }
begin
  with TProWarningDialog.Create(Application) do
  try
    Caption := ACaption;
    WarningMessage := AWarningMessage;
    Alignment := AAlignment;
    VerticalAlignment := AVerticalAlignment;
    Language := ALanguage;
    HelpContext := AHelpContext;
    Result := Execute;
  finally
    Free;
  end;
end; { WarningDialog }

procedure InfoDialog(const ACaption, AInfoMessage : string;
  AAlignment : TAlignment; AVerticalAlignment : TVerticalAlignment;
  ALanguage : TDialogLanguage; AHelpContext : LongInt);
{ Information dialog interface routine }
begin
  with TProInfoDialog.Create(Application) do
  try
    Caption := ACaption;
    InfoMessage := AInfoMessage;
    Alignment := AAlignment;
    VerticalAlignment := AVerticalAlignment;
    Language := ALanguage;
    HelpContext := AHelpContext;
    Execute;
  finally
    Free;
  end;
end; { InfoDialog }

procedure StopDialog(const ACaption, AStopMessage : string;
  AAlignment : TAlignment; AVerticalAlignment : TVerticalAlignment;
  ALanguage : TDialogLanguage; AHelpContext : LongInt);
{ Stop dialog interface routine }
begin
  with TProStopDialog.Create(Application) do
  try
    Caption := ACaption;
    StopMessage := AStopMessage;
    Alignment := AAlignment;
    VerticalAlignment := AVerticalAlignment;
    Language := ALanguage;
    HelpContext := AHelpContext;
    Execute;
  finally
    Free;
  end;
end; { StopDialog }

function PasswordDialog(const ACaption : string; var APassword : string;
  ALanguage : TDialogLanguage; AHelpContext : LongInt) : Boolean;
{ Password dialog interface routine }
begin
  with TProPasswordDialog.Create(Application) do
  try
    Caption := ACaption;
    Language := ALanguage;
    HelpContext := AHelpContext;
    Result := Execute;
    if Result then APassword := Password;
  finally
    Free;
  end;
end; { PasswordDialog }

function LoginDialog(const ACaption : string; var AUserName, APassword : string;
  ALanguage : TDialogLanguage; AHelpContext : LongInt) : Boolean;
{ Login dialog interface routine }
begin
  with TProLoginDialog.Create(Application) do
  try
    Caption := ACaption;
    Language := ALanguage;
    HelpContext := AHelpContext;
    Result := Execute;
    if Result then
    begin
      AUserName := UserName;
      APassword := Password;
    end;
  finally
    Free;
  end;
end; { LoginDialog }

end.
