{*******************************************************}
{*                                                     *}
{*      Pro VCL Extensions Library                     *}
{*      Database Controls Unit                         *}
{*                                                     *}
{*      Copyright (c) 1996-98 by Dmitry Barabash       *}
{*                                                     *}
{*******************************************************}

unit ProDBCtl;

{$DEFINE CONTROLS}
{$I PRO.INC}

interface

uses {$IFDEF WIN32} Windows, {$ELSE} WinTypes, WinProcs, {$ENDIF} Messages, 
  SysUtils, Classes, Controls, Graphics, StdCtrls, DB, DBTables, DBCtrls, 
  ProCtrls;

type

{ TProDBText }

  TProDBText = class(TProCustomLabel)
  private
    { Private variables }
    FDataLink : TFieldDataLink;
    { Property access methods }
    function GetField : TField;
    function GetDataSource : TDataSource;
    procedure SetDataSource(Value : TDataSource);
    function GetDataField : string;
    procedure SetDataField(Value : string);
    { Private methods }
    procedure DataChange(Sender : TObject);
  protected
    procedure Notification(AComponent : TComponent;
      Operation : TOperation); override;
  public
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
    { Field object }
    property Field : TField read GetField;
  published
    { Datasource attached to the database table }
    property DataSource : TDataSource read GetDataSource write SetDataSource;
    { Field name in the database table }
    property DataField : string read GetDataField write SetDataField;
    { Enable standard properties }
    property Align;
    property Alignment;
    property Angle;
    property AutoSize;
    property Color;
    property Cursor;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property Layout;
    property OffsetX;
    property OffsetY;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property ShadowColor;
    property ShadowDepth;
    property ShadowDirection;
    property ShadowSize;
    property ShadowStyle;
    property ShowHint;
    property Transparent;
    property VerticalAlignment;
    property Visible;
    property WordWrap;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseEnter;
    property OnMouseLeave;
    property OnMouseMove;
    property OnMouseUp;
  {$IFDEF WIN32}
    property OnStartDrag;
  {$ENDIF}
  end;

{ TProDBListBox }

  TProDBListBox = class(TDBListBox)
  private
    { Variables for properties }
    FAutoHorzScroll : Boolean;
    FHorzScrollExtent : Word;
    { Private variables }
    FMaxItemWidth : Word;
    { Property access methods }
    procedure SetAutoHorzScroll(Value : Boolean);
    procedure SetHorzScrollExtent(Value : Word);
    { Private methods }
    procedure SetHorizontalExtent;
    procedure ResetHorizontalExtent;
  protected
    procedure CreateParams(var Params : TCreateParams); override;
    procedure WndProc(var Message : TMessage); override;
  public
    constructor Create(AOwner : TComponent); override;
  published
    { Set to True to allow the automatic mode of the showing }
    { of the horizontal scroll bar                           }
    property AutoHorzScroll : Boolean read FAutoHorzScroll
      write SetAutoHorzScroll default True;
    { If AutoHorzScroll is True HorzScrollExtent property defines the }
    { addition extent to the automatic determined extent of the       }
    { horizontal scroll bar. If AutoHorzScroll is False this property }
    { completely defines the extent of the horizontal scroll bar.     }
    property HorzScrollExtent : Word read FHorzScrollExtent
      write SetHorzScrollExtent default 0;
  end;

implementation

{ TProDBText }

constructor TProDBText.Create(AOwner : TComponent);
{ Overrides the constructor to initialize variables }
begin
  inherited Create(AOwner);
  ShowAccelChar := False;
  FDataLink := TFieldDataLink.Create;
  FDataLink.OnDataChange := DataChange;
end; { TProDBText.Create }

destructor TProDBText.Destroy;
{ Overrides the destructor to uninitialize variables }
begin
  FDataLink.Free;
  FDataLink := nil;
  inherited Destroy;
end; { TProDBText.Destroy }

procedure TProDBText.Notification(AComponent : TComponent;
  Operation : TOperation);
{ Overrides the inherited method to respond to opRemove notifications
  on the DataSource }
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (FDataLink <> nil) and
     (FDataLink.DataSource = AComponent) then
    FDataLink.DataSource := nil;
end; { TProDBText.Notification }

function TProDBText.GetField : TField;
{ Gets database field }
begin
  Result := FDataLink.Field;
end; { TProDBText.GetField }

function TProDBText.GetDataSource : TDataSource;
{ Gets datasource }
begin
  Result := FDataLink.DataSource;
end; { TProDBText.GetDataSource }

procedure TProDBText.SetDataSource(Value : TDataSource);
{ Sets datasource }
begin
  FDataLink.DataSource := Value;
end; { TProDBText.SetDataSource }

function TProDBText.GetDataField : string;
{ Gets database field name }
begin
  Result := FDataLink.FieldName;
end; { TProDBText.GetDataField }

procedure TProDBText.SetDataField(Value : string);
{ Sets database field name }
begin
  FDataLink.FieldName := Value;
end; { TProDBText.SetDataField }

procedure TProDBText.DataChange(Sender : TObject);
{ Sets caption when data changes }
begin
  if FDataLink.Field = nil then
    Caption := ''
  else
    Caption := FDataLink.Field.DisplayText;
end; { TProDBText.DataChange }


{ TProDBListBox }

constructor TProDBListBox.Create(AOwner : TComponent);
{ Overrides the constructor to initialize variables }
begin
  inherited Create(AOwner);
  FAutoHorzScroll := True;
end; { TProDBListBox.Create }

procedure TProDBListBox.SetAutoHorzScroll(Value : Boolean);
{ Sets the automatic mode of the showing of the horizontal scroll bar
  when property is changed }
begin
  if FAutoHorzScroll <> Value then
  begin
    FAutoHorzScroll := Value;
    ResetHorizontalExtent;
  end;
end; { TProDBListBox.SetAutoHorzScroll }

procedure TProDBListBox.SetHorzScrollExtent(Value : Word);
{ Sets the width of the horizontal scroll bar when property is changed }
begin
  if FHorzScrollExtent <> Value then
  begin
    FHorzScrollExtent := Value;
    ResetHorizontalExtent;
  end;
end; { TProDBListBox.SetHorzScrollExtent }

procedure TProDBListBox.SetHorizontalExtent;
{ Sets the extent of the horizontal scroll bar }
var
  ItemWidth : Word;
begin
  ItemWidth := FHorzScrollExtent;
  if FAutoHorzScroll then Inc(ItemWidth, FMaxItemWidth);
  SendMessage(Handle, LB_SETHORIZONTALEXTENT, ItemWidth, 0);
end; { TProDBListBox.SetHorizontalExtent }

procedure TProDBListBox.ResetHorizontalExtent;
{ Calculates the extent of the horizontal scroll bar as the largest
  width of list box items and reset it }
var
  I : Integer;
  ItemWidth : Word;
begin
  FMaxItemWidth := 0;
  for I := 0 to Items.Count - 1 do
  begin
    ItemWidth := Canvas.TextWidth(Items[I] + 'I');
    if FMaxItemWidth < ItemWidth then
      FMaxItemWidth := ItemWidth;
  end;
  SetHorizontalExtent;
end; { TProDBListBox.ResetHorizontalExtent }

procedure TProDBListBox.CreateParams(var Params : TCreateParams);
{ Initializes the window-creation parameter record passed in the Params
  parameter. Besides it sets WS_HSCROLL Windows style for creation of
  the list box which can have a horizontal scroll bar. }
begin
  inherited CreateParams(Params);
  with Params do
    Style := Style or WS_HSCROLL;
end; { TProDBListBox.CreateParams }

procedure TProDBListBox.WndProc(var Message : TMessage);
{ Provides specific message responses for the list box. Handles messages
  for support the horizontal scroll bar. }
var
  ItemWidth : Word;
begin
  case Message.Msg of
    LB_ADDSTRING, LB_INSERTSTRING :
      begin
        ItemWidth := Canvas.TextWidth(StrPas(PChar(Message.lParam)) + 'I');
        if FMaxItemWidth < ItemWidth then
          FMaxItemWidth := ItemWidth;
        SetHorizontalExtent;
      end;
    LB_DELETESTRING :
      begin
        ItemWidth := Canvas.TextWidth(Items[Message.wParam] + 'I');
        if ItemWidth = FMaxItemWidth then
        begin
          inherited WndProc(Message);
          ResetHorizontalExtent;
          Exit;
        end;
      end;
    LB_RESETCONTENT :
      begin
        FMaxItemWidth := 0;
        SetHorizontalExtent;
      end;
    WM_SETFONT :
      begin
        inherited WndProc(Message);
        Canvas.Font.Assign(Self.Font);
        ResetHorizontalExtent;
        Exit;
      end;
  end;
  inherited WndProc(Message);
end; { TProDBListBox.WndProc }

end.
