{*******************************************************}
{*                                                     *}
{*      Pro VCL Extensions Library                     *}
{*      ProCtrls Unit                                  *}
{*                                                     *}
{*      Copyright (c) 1996-98 by Dmitry Barabash       *}
{*                                                     *}
{*******************************************************}

unit ProCtrls;

{$DEFINE CONTROLS}
{$I PRO.INC}

interface

uses {$IFDEF WIN32} Windows, {$ELSE} WinTypes, WinProcs, {$ENDIF}
  Messages, SysUtils, Classes, Graphics, Controls, Forms, Menus,
  StdCtrls, ExtCtrls, Buttons;

type

{ Common types }

  { Direction type }
  TDirection = (drNone, drUp, drDown, drLeft, drRight, drUpLeft,
    drUpRight, drDownLeft, drDownRight);

  { Vertical alignment }
  TVerticalAlignment = (vaTop, vaCenter, vaBottom);

  { Frame sides }
  TSide = (sdLeft, sdTop, sdRight, sdBottom);
  TSides = set of TSide;

{ TProCustomLabel }

  { Shadow style }
  TShadowStyle = (ssNone, ssCustom, ssRaised, ssRecessed, ssShadow,
    ssThickShadow);

  { Shadow depth }
  TShadowDepth = 0..1280;

  { Shadow size }
  TShadowSize = 0..1280;

{$IFNDEF VCL_3}
  { Vertical placement of the text within the label }
  TTextLayout = (tlTop, tlCenter, tlBottom);
{$ENDIF}

  TProCustomLabel = class(TCustomLabel)
  private
    { Variables for properties }
    FAngle : Integer;
    FOffsetX : Integer;
    FOffsetY : Integer;
    FShadowStyle : TShadowStyle;
    FShadowDirection : TDirection;
    FShadowDepth : TShadowDepth;
    FShadowSize : TShadowSize;
    FShadowColor : TColor;
    FVerticalAlignment : TVerticalAlignment;
    FOnMouseEnter : TNotifyEvent;
    FOnMouseLeave : TNotifyEvent;
    { Private Variables }
    FCosAngle : Double;
    FSinAngle : Double;
    FChangingStyle : Boolean;
    { Property access methods }
    procedure SetAngle(Value : Integer);
    procedure SetOffsetX(Value : Integer);
    procedure SetOffsetY(Value : Integer);
    procedure SetShadowStyle(Value : TShadowStyle);
    procedure SetDirection(Value : TDirection);
    procedure SetShadowDepth(Value : TShadowDepth);
    procedure SetShadowSize(Value : TShadowSize);
    procedure SetColor(Value : TColor);
    procedure SetVerticalAlignment(Value : TVerticalAlignment);
    procedure SetLayout(Value : TTextLayout);
    function GetLayout : TTextLayout;
    { Private methods }
    procedure SetTextAngle(Canvas : TCanvas; AAngle : Integer);
    procedure CMMouseEnter(var Message : TMessage); message CM_MOUSEENTER;
    procedure CMMouseLeave(var Message : TMessage); message CM_MOUSELEAVE;
  protected
    procedure MouseEnter; dynamic;
    procedure MouseLeave; dynamic;
    procedure Paint; override;
    { Rotation angle of the label (only TrueType fonts) }
    property Angle : Integer read FAngle write SetAngle default 0;
    { The offset in pixels from the left of the component rect }
    { to the caption. It can be a negative.                    }
    property OffsetX : Integer read FOffsetX write SetOffsetX default 0;
    { The offset in pixels from the top of the component rect }
    { to the caption. It can be a negative.                    }
    property OffsetY : Integer read FOffsetY write SetOffsetY default 0;
    { Style of the shadow }
    property ShadowStyle : TShadowStyle read FShadowStyle
      write SetShadowStyle default ssNone;
    { Direction of the shadow }
    property ShadowDirection : TDirection read FShadowDirection
      write SeTDirection default drNone;
    { Depth of the shadow }
    property ShadowDepth : TShadowDepth read FShadowDepth
      write SetShadowDepth default 0;
    { Size of the shadow }
    property ShadowSize : TShadowSize read FShadowSize write SetShadowSize
      default 0;
    { Color of the shadow }
    property ShadowColor : TColor read FShadowColor
      write SeTColor default clBtnShadow;
    { Vertical alignment }
    property VerticalAlignment : TVerticalAlignment read FVerticalAlignment
      write SetVerticalAlignment default vaTop;
    { Vertical alignment (for compatibility with TLabel component) }
    property Layout : TTextLayout read GetLayout write SetLayout
      stored False default tlTop;
    { OnMouseEnter occurs whenever the mouse pointer is moved into }
    { the region defined by the component                          }
    property OnMouseEnter : TNotifyEvent read FOnMouseEnter
      write FOnMouseEnter;
    { OnMouseLeave occurs whenever the mouse pointer is moved out of }
    { the region defined by the component                            }
    property OnMouseLeave : TNotifyEvent read FOnMouseLeave
      write FOnMouseLeave;
  public
    constructor Create(AOwner : TComponent); override;
  end;

{ TProLabel }

  TProLabel = class(TProCustomLabel)
  published
    { Enable standard and new properties }
    property Align;
    property Alignment;
    property Angle;
    property AutoSize;
    property Caption;
    property Color;
    property DragCursor;
    property DragMode;
    property Enabled;
    property FocusControl;
    property Font;
    property OffsetX;
    property OffsetY;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShadowStyle;
    property ShadowDirection;
    property ShadowDepth;
    property ShadowSize;
    property ShadowColor;
    property ShowAccelChar;
    property ShowHint;
    property Transparent;
    property VerticalAlignment;
    property Layout;
    property Visible;
    property WordWrap;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseEnter;
    property OnMouseLeave;
    property OnMouseMove;
    property OnMouseUp;
  {$IFDEF WIN32}
    property OnStartDrag;
  {$ENDIF}
  end;

{ TProClockLabel }

  TProClockLabel = class(TProLabel)
  private
    { Variables for properties }
    FFormat : string;
    FOnTimer : TNotifyEvent;
    { Private variables }
    FTimer : TTimer;
    { Property access methods }
    procedure SetFormat(Value : string);
    function GetCaption : string;
    procedure SetEnabled(Value : Boolean);
    function GetEnabled : Boolean;
    procedure SetInterval(Value : Cardinal);
    function GetInterval : Cardinal;
    { Private methods }
    procedure TimerExpired(Sender : TObject);
  public
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
  published
    { Overrides standard Caption property. Use format property }
    { instead of this.                                         }
    property Caption : string read GetCaption;
    { Overrides standard Enabled property. Set to True to allow }
    { the clock updating.                                       }
    property Enabled : Boolean read GetEnabled write SetEnabled
      default True;
    { Defines the TDateTime format string to use as mask into Caption }
    { of the label or usual string to use as Caption                  }
    property Format : string read FFormat write SetFormat;
    { Frequency in ticks or milliseconds that the label showing }
    { the time is updated                                       }
    property Interval : Cardinal read GetInterval write SetInterval
      default 1000;
    { OnTimer occurs when the specified amount of time, deteminated by }
    { the Interval property, has passed                                }
    property OnTimer : TNotifyEvent read FOnTimer write FOnTimer;
  end;

{ TProButton }

const
  { TModalResult values }
  mrDone   = mrAll + 1;
  mrFinish = mrAll + 2;

type
  TProButtonKind = (pbkCustom, pbkOk,
    pbkCancel, pbkCancelRus, pbkCancelUkr,
    pbkHelp,   pbkHelpRus,   pbkHelpUkr,
    pbkYes,    pbkYesRus,    pbkYesUkr,
    pbkNo,     pbkNoRus,     pbkNoUkr,
    pbkClose,  pbkCloseRus,  pbkCloseUkr,
    pbkAbort,  pbkAbortRus,  pbkAbortUkr,
    pbkRetry,  pbkRetryRus,  pbkRetryUkr,
    pbkIgnore, pbkIgnoreRus, pbkIgnoreUkr,
    pbkAll,    pbkAllRus,    pbkAllUkr,
    pbkDone,   pbkDoneRus,   pbkDoneUkr,
    pbkPrev,   pbkPrevRus,   pbkPrevUkr,
    pbkNext,   pbkNextRus,   pbkNextUkr,
    pbkFinish, pbkFinishRus, pbkFinishUkr);

  TProButton = class(TButton)
  private
    { Variables for properties }
    FKind : TProButtonKind;
    { Property access methods }
    procedure SetKind(Value : TProButtonKind);
    function GetKind : TProButtonKind;
    { Private methods }
    function IsCustom : Boolean;
    function IsCustomCaption : Boolean;
  public
    procedure Click; override;
  published
    { Enable standard and new properties }
    property Cancel stored IsCustom;
    property Caption stored IsCustomCaption;
    property Default stored IsCustom;
    { Kind of button }
    property Kind : TProButtonKind read GetKind write SetKind
      default pbkCustom;
  end;

{ TProSpeedButton }

  TProSpeedButton = class(TSpeedButton)
  private
    { Variables for properties }
    FRepeatEnabled : Boolean;
    FRepeatInitInterval : Cardinal;
    FRepeatInterval : Cardinal;
    { Private variables }
    FRepeatTimer : TTimer;
    { Property access methods }
    procedure SetRepeatEnabled(Value : Boolean);
    { Private methods }
    procedure TimerExpired(Sender : TObject);
  protected
    procedure MouseDown(Button : TMouseButton; Shift : TShiftState;
      X, Y : Integer); override;
    procedure MouseUp(Button : TMouseButton; Shift : TShiftState;
      X, Y : Integer); override;
  public
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
  published
    { Set to True to allow repeating OnClick events to occur on }
    { the TProSpeedButton while the button remains pressed      }
    property RepeatEnabled : Boolean read FRepeatEnabled
      write SetRepeatEnabled default False;
    { This measures initial delay in ticks or milliseconds from when }
    { the button is first pressed to when OnClick event first occurs }
    property RepeatInitInterval : Cardinal read FRepeatInitInterval
      write FRepeatInitInterval default 400;
    { This measures to continuing delay between each OnClick event }
    { for the TProSpeedButton                                      }
    property RepeatInterval : Cardinal read FRepeatInterval write FRepeatInterval
      default 100;
  end;

{ TProCustomListBox }

  TProCustomListBox = class(TCustomListBox)
  private
    { Variables for properties }
    FAutoHorzScroll : Boolean;
    FHorzScrollExtent : Cardinal;
    { Private variables }
    FMaxItemWidth : Cardinal;
    { Property access methods }
    procedure SetAutoHorzScroll(Value : Boolean);
    procedure SetHorzScrollExtent(Value : Cardinal);
    { Private methods }
    procedure  SetHorizontalExtent;
    procedure  ResetHorizontalExtent;
  protected
    procedure CreateParams(var Params : TCreateParams); override;
    procedure WndProc(var Message : TMessage); override;
    { Set to True to allow tha automatic mode of showing }
    { of the horizontal scroll bar                       }
    property AutoHorzScroll : Boolean read FAutoHorzScroll
      write SetAutoHorzScroll default True;
    { If AutoHorzScroll is True HorzScrollExtent property defines the }
    { addition extent to the automatic detemined extent of the        }
    { horizontal scroll bar. If AutoHorzScroll is False this property }
    { completly defines the extent of the horizontal scroll bar.      }
    property HorzScrollExtent : Cardinal read FHorzScrollExtent
      write SetHorzScrollExtent default 0;
  public
    constructor Create(AOwner : TComponent); override;
  end;

{ TProListBox }

  TProListBox = class(TProCustomListBox)
  published
    { Enable standard and new properties }
    property Align;
    property AutoHorzScroll;
    property BorderStyle;
    property Color;
    property Columns;
    property Ctl3D;
    property DragCursor;
    property DragMode;
    property Enabled;
    property ExtendedSelect;
    property Font;
    property IntegralHeight;
    property ItemHeight;
    property Items;
    property HorzScrollExtent;
    property MultiSelect;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property Sorted;
    property Style;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnDrawItem;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMeasureItem;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  {$IFDEF WIN32}
    property OnStartDrag;
  {$ENDIF}
  {$IFDEF VCL_3}
    property ImeMode;
    property ImeName;
    property TabWidth;
  {$ENDIF}
  end;

{ TProCustomPanel }

  TProCustomPanel = class(TCustomPanel)
  private
    { Variables for properties }
    FBevelCtl3D : Boolean;
    FBevelSides : TSides;
    FBorderColor : TColor;
    FVerticalAlignment : TVerticalAlignment;
    FOnMouseEnter : TNotifyEvent;
    FOnMouseLeave : TNotifyEvent;
    { Property access methods }
    procedure SetBevelCtl3D(Value : Boolean);
    procedure SetBevelSides(Value : TSides);
    procedure SetBorderColor(Value : TColor);
    procedure SetVerticalAlignment(Value : TVerticalAlignment);
    { Private methods }
    procedure CMMouseEnter(var Message : TMessage); message CM_MOUSEENTER;
    procedure CMMouseLeave(var Message : TMessage); message CM_MOUSELEAVE;
  protected
    procedure MouseEnter; dynamic;
    procedure MouseLeave; dynamic;
    procedure Paint; override;
  {$IFDEF WIN32}
    procedure CreateParams(var Params : TCreateParams); override;
  {$ENDIF}
    procedure AlignControls(AControl : TControl;
      var Rect : TRect); override;
    { Set to True to allow enhanced bevel styles of Windows 95 }
    property BevelCtl3D : Boolean read FBevelCtl3D write SetBevelCtl3D
      default False;
    { Specify which sides of the panel will be framed }
    property BevelSides : TSides read FBevelSides write SetBevelSides
      default [sdLeft, sdTop, sdRight, sdBottom];
    { Color of the border between bevels }
    property BorderColor : TColor read FBorderColor write SetBorderColor
      default clBtnFace;
    { Vertical alignment }
    property VerticalAlignment : TVerticalAlignment read FVerticalAlignment
      write SetVerticalAlignment default vaCenter;
    { OnMouseEnter occurs whenever the mouse pointer is moved into }
    { the region defined by the component                          }
    property OnMouseEnter : TNotifyEvent read FOnMouseEnter
      write FOnMouseEnter;
    { OnMouseLeave occurs whenever the mouse pointer is moved out of }
    { the region defined by the component                            }
    property OnMouseLeave : TNotifyEvent read FOnMouseLeave
      write FOnMouseLeave;
  public
    constructor Create(AOwner : TComponent); override;
  end;

{ TProPanel }

  TProPanel = class(TProCustomPanel)
  published
    { Enable standard properties }
    property Align;
    property Alignment;
    property BevelCtl3D;
    property BevelInner;
    property BevelOuter;
    property BevelSides;
    property BevelWidth;
    property BorderColor;
    property BorderWidth;
    property BorderStyle;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Caption;
    property Color;
    property Ctl3D;
    property Font;
    property Locked;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property VerticalAlignment;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnMouseDown;
    property OnMouseEnter;
    property OnMouseLeave;
    property OnMouseMove;
    property OnMouseUp;
    property OnResize;
  {$IFDEF WIN32}
    property OnStartDrag;
  {$ENDIF}
  {$IFDEF VCL_3}
    property FullRepaint;
  {$ENDIF}
  end;

{ TProBorder }

  TProBorder = class(TGraphicControl)
  private
    { Variables for properties }
    FBevelCtl3D : Boolean;
    FBevelInner : TPanelBevel;
    FBevelOuter : TPanelBevel;
    FBevelSides : TSides;
    FBevelWidth : TBevelWidth;
    FBorderColor : TColor;
    FBorderWidth : TBorderWidth;
    FOnMouseEnter : TNotifyEvent;
    FOnMouseLeave : TNotifyEvent;
    { Property access methods }
    procedure SetBevelCtl3D(Value : Boolean);
    procedure SetBevelInner(Value : TPanelBevel);
    procedure SetBevelOuter(Value : TPanelBevel);
    procedure SetBevelWidth(Value : TBevelWidth);
    procedure SetBevelSides(Value : TSides);
    procedure SetBorderWidth(Value : TBorderWidth);
    procedure SetBorderColor(Value : TColor);
    { Private methods }
    procedure CMMouseEnter(var Message : TMessage); message CM_MOUSEENTER;
    procedure CMMouseLeave(var Message : TMessage); message CM_MOUSELEAVE;
  protected
    procedure MouseEnter; dynamic;
    procedure MouseLeave; dynamic;
    procedure Paint; override;
  public
    constructor Create(AOwner : TComponent); override;
  published
    { Set to True to allow enhanced bevel styles of Windows 95 }
    property BevelCtl3D : Boolean read FBevelCtl3D write SetBevelCtl3D
      default False;
    { Style of the inner bevel }
    property BevelInner : TPanelBevel read FBevelInner
      write SetBevelInner default bvNone;
    { Style of the outer bevel }
    property BevelOuter : TPanelBevel read FBevelOuter
      write SetBevelOuter default bvLowered;
    { Specify which sides of the panel will be framed }
    property BevelSides : TSides read FBevelSides write SetBevelSides
      default [sdLeft, sdTop, sdRight, sdBottom];
    { Width in pixels of both the inner and outer bevels }
    property BevelWidth : TBevelWidth read FBevelWidth
      write SetBevelWidth default 1;
    { Width of the border between bevels }
    { Color of the border between bevels }
    property BorderColor : TColor read FBorderColor write SetBorderColor
      default clBtnFace;
    property BorderWidth: TBorderWidth read FBorderWidth
      write SetBorderWidth default 0;
    { OnMouseEnter occurs whenever the mouse pointer is moved into }
    { the region defined by the component                          }
    property OnMouseEnter : TNotifyEvent read FOnMouseEnter
      write FOnMouseEnter;
    { OnMouseLeave occurs whenever the mouse pointer is moved out of }
    { the region defined by the component                            }
    property OnMouseLeave : TNotifyEvent read FOnMouseLeave
      write FOnMouseLeave;
    { Enable standard properties }
    property Align;
    property ParentShowHint;
    property ShowHint;
    property Visible;
  end;

{ TProToolBar }

  TProToolBar = class(TProPanel)
  public
    constructor Create(AOwner : TComponent); override;
  published
    { Enable standard and new properties with new defaults }
    property Align default alTop;
    property BevelInner default bvRaised;
    property BevelOuter default bvLowered;
    property BevelSides default [sdTop];
  end;

{ Utility routine }

procedure Frame3DSides(Canvas : TCanvas; var Rect : TRect; Sides : TSides;
  TopColor, BottomColor : TColor; Width : Integer);
{ Draws a frame with the specified width and colors of frame sides.
  Sides parameter specifies which sides of frame will be drawn. }

implementation

uses ProConst;

const
  { Horizontal alignment constants for drawing a text }
  TextAlignment : array[TAlignment] of Word = (DT_LEFT, DT_RIGHT, DT_CENTER);

{ Utility routine }

procedure Frame3DSides(Canvas : TCanvas; var Rect : TRect; Sides : TSides;
  TopColor, BottomColor : TColor; Width : Integer);
{ Draws a frame with the specified width and colors of frame sides.
  Sides parameter specifies which sides of frame will be drawn. }

  procedure DoRect;
  { Draws the sides of Rect if necessary }
  begin
    with Canvas, Rect do
    begin
      { Color for top and left sides }
      Pen.Color := TopColor;
      if sdLeft in Sides then
      begin
        MoveTo(Left, Top);
        LineTo(Left, Bottom);
      end;
      if sdTop in Sides then
      begin
        MoveTo(Left, Top);
        LineTo(Right, Top);
      end;
      { Color for bottom and right sides }
      Pen.Color := BottomColor;
      if sdRight in Sides then
      begin
        MoveTo(Right - 1, Top);
        LineTo(Right - 1, Bottom);
      end;
      if sdBottom in Sides then
      begin
        MoveTo(Left, Bottom - 1);
        LineTo(Right, Bottom - 1);
      end;
    end;
  end; { DoRect }

  procedure DoInflateRect;
  { Reduces Rect if necessary }
  begin
    with Rect do
    begin
      if sdLeft in Sides then Inc(Left);
      if sdTop in Sides then Inc(Top);
      if sdRight in Sides then Dec(Right);
      if sdBottom in Sides then Dec(Bottom);
    end;
  end; { DoInflateRect }

begin
  Canvas.Pen.Width := 1;
  while Width > 0 do
  begin
    Dec(Width);
    DoRect;
    DoInflateRect;
  end;
end; { Frame3DSides }


{ TProCustomLabel }

constructor TProCustomLabel.Create(AOwner : TComponent);
{ Overrides the constructor to initialize variables }
begin
  inherited Create(AOwner);
  FShadowColor := clBtnShadow;
end; { TProCustomLabel.Create }

procedure TProCustomLabel.SetAngle(Value : Integer);
{ Sets the angle for the text out when property is changed }
begin
  Value := Value mod 360;
  if Value < 0 then Value := 360 + Value;
  if FAngle <> Value then
  begin
    FAngle := Value;
    FCosAngle := Cos(Value * Pi / 180);
    FSinAngle := Sin(Value * Pi / 180);
    if (Value > 0) and AutoSize then AutoSize := False;
    Invalidate;
  end;
end; { TProCustomLabel.SetAngle }

procedure TProCustomLabel.SetTextAngle(Canvas : TCanvas; AAngle : Integer);
{ Sets the angle for the text out }
var
  FontLogRec: TLogFont;    { Storage area for font information }
begin
  { Get the current font information. We only want to modify the angle }
  GetObject(Canvas.Font.Handle, SizeOf(FontLogRec), Addr(FontLogRec));

  { Modify the angle. "The angle, in tenths of a degrees, between the base
     line of a character and the x-axis." (Windows API Help file.)}
  FontLogRec.lfEscapement := AAngle * 10;
  { Request TrueType precision }
  FontLogRec.lfOutPrecision := OUT_TT_ONLY_PRECIS;

  { Delphi will handle the deallocation of the old font handle }
  Canvas.Font.Handle := CreateFontIndirect(FontLogRec);
end; { TProCustomLabel.SetTextAngle }

procedure TProCustomLabel.SetOffsetX(Value : Integer);
{ Sets the offset by X-axis when property is changed }
begin
  if FOffsetX <> Value then
  begin
    FOffsetX := Value;
    if (Value > 0) and AutoSize then AutoSize := False;
    Invalidate;
  end;
end; { TProCustomLabel.SetOffsetX }

procedure TProCustomLabel.SetOffsetY(Value : Integer);
{ Sets the offset by Y-axis when property is changed }
begin
  if FOffsetY <> Value then
  begin
    FOffsetY := Value;
    if (Value > 0) and AutoSize then AutoSize := False;
    Invalidate;
  end;
end; { TProCustomLabel.SetOffsetY }

procedure TProCustomLabel.SetShadowStyle(Value : TShadowStyle);
{ Sets the style of the shadow when property is changed }
begin
  if FShadowStyle <> Value then
  begin
    FShadowStyle := Value;
    FChangingStyle := True;
    case Value of
      ssNone :
        begin
          ShadowDirection := drNone;
          ShadowDepth := 0;
          ShadowSize := 0;
          ShadowColor := clBtnShadow;
        end;
      ssRaised :
        begin
          ShadowDirection := drUpLeft;
          ShadowDepth := 1;
          ShadowSize := 1;
          ShadowColor := clWhite;
        end;
      ssRecessed :
        begin
          ShadowDirection := drDownRight;
          ShadowDepth := 1;
          ShadowSize := 1;
          ShadowColor := clWhite;
        end;
      ssShadow :
        begin
          ShadowDirection := drDownRight;
          ShadowDepth := 2;
          ShadowSize := 1;
          ShadowColor := clBtnShadow;
        end;
      ssThickShadow :
        begin
          ShadowDirection := drDownRight;
          ShadowDepth := 1;
          ShadowSize := 4;
          ShadowColor := clBtnShadow;
        end;
    end;
    FChangingStyle := False;
  end;
end; { TProCustomLabel.SetShadowStyle }

procedure TProCustomLabel.SetDirection(Value : TDirection);
{ Sets the direction of the shadow when property is changed }
begin
  if FShadowDirection <> Value then
  begin
    FShadowDirection := Value;
    if not FChangingStyle then FShadowStyle := ssCustom;
    Invalidate;
  end;
end; { TProCustomLabel.SetDirection }

procedure TProCustomLabel.SetShadowDepth(Value : TShadowDepth);
{ Sets the depth of the shadow when property is changed }
begin
  if FShadowDepth <> Value then
  begin
    FShadowDepth := Value;
    if not FChangingStyle then FShadowStyle := ssCustom;
    Invalidate;
  end;
end; { TProCustomLabel.SetShadowDepth }

procedure TProCustomLabel.SetShadowSize(Value : TShadowSize);
{ Sets the size of the shadow when property is changed }
begin
  if FShadowSize <> Value then
  begin
    FShadowSize := Value;
    if not FChangingStyle then FShadowStyle := ssCustom;
    Invalidate;
  end;
end; { TProCustomLabel.SetShadowSize }

procedure TProCustomLabel.SetColor(Value : TColor);
{ Sets the color of the shadow when property is changed }
begin
  if FShadowColor <> Value then
  begin
    FShadowColor := Value;
    if not FChangingStyle then FShadowStyle := ssCustom;
    Invalidate;
  end;
end; { TProCustomLabel.SetColor }

procedure TProCustomLabel.SetVerticalAlignment(Value : TVerticalAlignment);
{ Sets the vertical alignment of the label when property is changed }
begin
  if FVerticalAlignment <> Value then
  begin
    FVerticalAlignment := Value;
    Invalidate;
  end;
end; { TProCustomLabel.SetVerticalAlignment }

procedure TProCustomLabel.SetLayout(Value : TTextLayout);
{ Sets the vertical alignment of the label }
begin
  SetVerticalAlignment(TVerticalAlignment(Value));
end; { TProCustomLabel.SetLayout }

function TProCustomLabel.GetLayout : TTextLayout;
{ Gets the vertical alignment of the label }
begin
  Result := TTextLayout(FVerticalAlignment);
end; { TProCustomLabel.SetLayuot }

procedure TProCustomLabel.CMMouseEnter(var Message : TMessage);
{ CM_MOUSEENTER message handler }
begin
  inherited;
  MouseEnter;
end; { TProCustomLabel.CMMouseEnter }

procedure TProCustomLabel.CMMouseLeave(var Message : TMessage);
{ CM_MOUSELEAVE message handler }
begin
  inherited;
  MouseLeave;
end; { TProCustomLabel.CMMouseLeave }

procedure TProCustomLabel.MouseEnter;
{ Calls the event handler attached to the OnMouseEnter event }
begin
  if Assigned(FOnMouseEnter) then FOnMouseEnter(Self);
end; { TProCustomLabel.MouseEnter }

procedure TProCustomLabel.MouseLeave;
{ Calls the event handler attached to the OnMouseLeave event }
begin
  if Assigned(FOnMouseLeave) then FOnMouseLeave(Self);
end; { TProCustomLabel.MouseLeave }

procedure TProCustomLabel.Paint;
{ Draws the label }
type
  { Type for specifying direction of shadow offset }
  TOffsetDirection = (odrX, odrY);
const
  { Factors for specifying direction of shadow offset }
  Offsets : array[TDirection, TOffsetDirection] of -1..1 = (
    (0, 0), (0, -1), (0, 1), (-1, 0), (1, 0),
    (-1, -1), (1, -1), (1, -1), (1, 1));
var
  Flags : Integer;
  Rect, TempRect : TRect;
  WCaption, HCaption, WCos, WSin, HCos, HSin, WRect, HRect, BRect : Integer;
  S : TShadowSize;
{$IFNDEF WIN32}
  Text : array[0..255] of Char;
{$ENDIF}

  procedure DoDrawText;
  { Draw label's caption }
  begin
    with Canvas do
    begin
      if FAngle <> 0 then
        TextOut(Rect.Left, Rect.Top, Caption)
      else
      {$IFDEF WIN32}
        DrawText(Handle, PChar(Caption), -1, Rect, Flags);
      {$ELSE}
        DrawText(Handle, Text, -1, Rect, Flags);
      {$ENDIF}
    end;
  end; { DoDrawText }

begin
  { Set text-drawing flags }
  Flags := DT_EXPANDTABS or TextAlignment[Alignment];
  if WordWrap then Flags := Flags or DT_WORDBREAK;
  if not ShowAccelChar then Flags := Flags or DT_NOPREFIX;
{$IFNDEF WIN32}
  { Get label's caption }
  GetTextBuf(Text, sizeOf(Text));
{$ENDIF}

  with Canvas do
  begin
    { Get the space of the component }
    Rect := ClientRect;
    { Ensure canvas font is set }
    Font := Self.Font;

    { Fill in background }
    if not Transparent then
    begin
      Brush.Color := Self.Color;
      Brush.Style := bsSolid;
      FillRect(Rect);
    end;
    { Don't overwrite background above }
    Brush.Style := bsClear;

    { Calculate the rect for the original text with alignment }
    if Angle <> 0 then
    begin
      { Set an angled font }
      SetTextAngle(Canvas, Angle);

      WCaption := TextWidth(Caption);
      HCaption := TextHeight(Caption);
      WCos := Trunc(WCaption * FCosAngle);
      WSin := Trunc(WCaption * FSinAngle);
      HCos := Trunc(HCaption * FCosAngle);
      HSin := Trunc(HCaption * FSinAngle);

      case FAngle of
        0..89 :
          begin
            WRect := WCos + HSin;
            HRect := WSin + HCos;
          end;
        90..179 :
          begin
            WRect := -WCos + HSin;
            HRect := WSin - HCos;
          end;
        180..269 :
          begin
            WRect := -WCos - HSin;
            HRect := -WSin - HCos;
          end;
        270..359 :
          begin
            WRect := WCos - HSin;
            HRect := -WSin + HCos;
          end;
      end;
      if WRect > Rect.Right then WRect := Rect.Right;
      if HRect > Rect.Bottom then HRect := Rect.Bottom;

      TempRect := Rect;
      Rect := Bounds(Rect.Left, Rect.Top, WRect, HRect);

     { Align the caption within the horizontal space of the component }
      if Alignment = taCenter then
        OffsetRect(Rect, (TempRect.Right - WRect) shr 1, 0)
      else if Alignment = taRightJustify then
        OffsetRect(Rect, TempRect.Right - WRect, 0);

     { Align the caption within the vertical space of the component }
      if FVerticalAlignment = vaCenter then
        OffsetRect(Rect, 0, (TempRect.Bottom - HRect) shr 1)
      else if FVerticalAlignment = vaBottom then
        OffsetRect(Rect, 0, TempRect.Bottom - HRect);

      case FAngle of
        0..89 : OffsetRect(Rect, 0, WSin);
        90..179 : OffsetRect(Rect, -WCos, HRect);
        180..269 : OffsetRect(Rect, WRect, -HCos);
        270..359 : OffsetRect(Rect, -HSin, 0);
      end;
    end
    else if FVerticalAlignment <> vaTop then
    begin
      TempRect := Rect;
    {$IFDEF WIN32}
      HCaption := DrawText(Handle, PChar(Caption), -1, TempRect,
        Flags or DT_CALCRECT);
    {$ELSE}
      HCaption := DrawText(Handle, Text, -1, TempRect,
        Flags or DT_CALCRECT);
    {$ENDIF}
      BRect := Rect.Bottom;
      Rect.Bottom := Rect.Top + HCaption;

      { Align the caption within the vertical space of the component }
      if FVerticalAlignment = vaCenter then
        OffsetRect(Rect, 0, (BRect - Rect.Bottom) shr 1)
      else
        OffsetRect(Rect, 0, BRect - Rect.Bottom);
    end;

    { Move the text within the space of the component }
    OffsetRect(Rect, FOffsetX, FOffsetY);

    { Draw the shadow of the label }
    if FShadowStyle <> ssNone then
    begin
      { Save the rect for the original text }
      TempRect := Rect;
      { Calculate the rect for the shadow }
      OffsetRect(Rect, FShadowDepth * Offsets[FShadowDirection, odrX],
                       FShadowDepth * Offsets[FShadowDirection, odrY]);
      { Set font color for the shadow }
      Font.Color := FShadowColor;
      if not Enabled then Font.Color := clGrayText;
      { Draw the shadow }
      S := FShadowSize;
      while S > 0 do
      begin
        DoDrawText;
        OffsetRect(Rect, Offsets[FShadowDirection, odrX],
                         Offsets[FShadowDirection, odrY]);
        Dec(S);
      end;
      { Restore the rect for the original text }
      Rect := TempRect;
    end;

    { Restore original font color }
    Font.Color := Self.Font.Color;
    if not Enabled then Font.Color := clGrayText;

    { Draw the original text }
    DoDrawText;

    { Restore font to angle 0 }
    if Angle <> 0 then SetTextAngle(Canvas, 0);
  end;
end; { TProCustomLabel.Paint }


{ TProClockLabel }

constructor TProClockLabel.Create(AOwner : TComponent);
{ Overrides the constructor to initialize variables }
begin
  inherited Create(AOwner);
  FFormat := 'c';
  inherited Caption := FormatDateTime(FFormat, Now);
  FTimer := TTimer.Create(Self);
  FTimer.OnTimer := TimerExpired;
end; { TProClockLabel.Create }

destructor TProClockLabel.Destroy;
{ Overrides the destructor to uninitialize variables }
begin
  FTimer.Free;
  inherited Destroy;
end; { TProClockLabel.Destroy }

procedure TProClockLabel.SetFormat(Value : string);
{ Sets the format of the label when property is changed }
begin
  if FFormat <> Value then
  begin
    FFormat := Value;
    inherited Caption := FormatDateTime(Value, Now);
  end;
end; { TProClockLabel.SetFormat }

function TProClockLabel.GetCaption : string;
{ Gets the caption of the label }
begin
  Result := inherited Caption;
end; { TProClockLabel.GetCaption }

procedure TProClockLabel.SetEnabled(Value : Boolean);
{ Sets inherited Enabled property and enable or disable the clock
  updating when property is changed }
begin
  if inherited Enabled <> Value then
  begin
    inherited Enabled := Value;
    FTimer.Enabled := Value;
  end;
end; { TProClockLabel.SetEnabled }

function TProClockLabel.GetEnabled : Boolean;
{ Gets the enable status of the label }
begin
  Result := inherited Enabled;
end; { TProClockLabel.GetEnabled }

procedure TProClockLabel.SetInterval(Value : Cardinal);
{ Sets the interval between timer events when property is changed }
begin
  if FTimer.Interval <> Value then
    FTimer.Interval := Value;
end; { TProClockLabel.SetInterval }

function TProClockLabel.GetInterval : Cardinal;
{ Gets the interval between timer events }
begin
  Result := FTimer.Interval;
end; { TProClockLabel.GetInterval }

procedure TProClockLabel.TimerExpired(Sender : TObject);
{ Inrenal timer event handler. Calls the event handler attached
  to the OnTimer event. }
begin
  inherited Caption := FormatDateTime(FFormat, Now);
  if Assigned(FOnTimer) then FOnTimer(Self);
end; { TProClockLabel.TimerExpired }


{ TProButton }

const
  ButtonCaptions : array[TProButtonKind] of Word = (0, SOKButton,
    SCancelButton, SCancelButtonRus, SCancelButtonUkr,
    SHelpButton,   SHelpButtonRus,   SHelpButtonUkr,
    SYesButton,    SYesButtonRus,    SYesButtonUkr,
    SNoButton,     SNoButtonRus,     SNoButtonUkr,
    SCloseButton,  SCloseButtonRus,  SCloseButtonUkr,
    SAbortButton,  SAbortButtonRus,  SAbortButtonUkr,
    SRetryButton,  SRetryButtonRus,  SRetryButtonUkr,
    SIgnoreButton, SIgnoreButtonRus, SIgnoreButtonUkr,
    SAllButton,    SAllButtonRus,    SAllButtonUkr,
    SDoneButton,   SDoneButtonRus,   SDoneButtonUkr,
    SPrevButton,   SPrevButtonRus,   SPrevButtonUkr,
    SNextButton,   SNextButtonRus,   SNextButtonUkr,
    SFinishButton, SFinishButtonRus, SFinishButtonUkr);
  ButtonModalResults : array[TProButtonKind] of TModalResult = (0, mrOK,
    mrCancel, mrCancel, mrCancel,
    0, 0, 0,
    mrYes, mrYes, mrYes,
    mrNo, mrNo, mrNo,
    0, 0, 0,
    mrAbort, mrAbort, mrAbort,
    mrRetry, mrRetry, mrRetry,
    mrIgnore, mrIgnore, mrIgnore,
    mrAll, mrAll, mrAll,
    mrDone, mrDone, mrDone,
    0, 0, 0,
    0, 0, 0,
    mrFinish, mrFinish, mrFinish);

procedure TProButton.SetKind(Value : TProButtonKind);
{ Set the kind of button when property is changed }
begin
  if FKind <> Value then
  begin
    FKind := Value;
    if Value <> pbkCustom then
    begin
      Default := Value in [pbkOK, pbkYes, pbkYesRus, pbkYesUkr,
        pbkDone, pbkDoneRus, pbkDoneUkr,
        pbkNext, pbkNextRus, pbkNextUkr,
        pbkFinish, pbkFinishRus, pbkFinishUkr];
      Cancel := Value in [pbkCancel, pbkCancelRus, pbkCancelUkr,
        pbkNo, pbkNoRus, pbkNoUkr];

      {  }
      if ((csLoading in ComponentState) and (Caption = '')) or
        (not (csLoading in ComponentState)) then
      begin
        Caption := LoadStr(ButtonCaptions[Value]);
      end;

      ModalResult := ButtonModalResults[Value];
    end;
  end;
end; { TProButton.SetKind }

function TProButton.GetKind : TProButtonKind;
{ Gets the kind of button }
begin
  if FKind <> pbkCustom then
    if ((FKind in [pbkOK,
        pbkYes, pbkYesRus, pbkYesUkr,
        pbkDone, pbkDoneRus, pbkDoneUkr,
        pbkFinish, pbkFinishRus, pbkFinishUkr]) xor Default) or
      ((FKind in [pbkCancel, pbkCancelRus, pbkCancelUkr,
        pbkNo, pbkNoRus, pbkNoUkr]) xor Cancel) or
      (ModalResult <> ButtonModalResults[FKind]) then
      FKind := pbkCustom;
  Result := FKind;
end; { TProButton.GetKind }

procedure TProButton.Click;
{ Simulates a mouse click, as if the user had clicked the button.
  Calls the event handler attached to the OnClick event when Click
  is called any code. If the value of the Kind property is pbkCloseXXX,
  calling Click closes the form. If the value of Kind is pbkHelpXXX,
  calling Click displays the Help screen assigned with the HelpContext
  property to the button. If Kind has any other value, the inherited
  Click method is called. }
var
  Form : TForm;
  Control : TWinControl;
begin
  case FKind of
    pbkClose, pbkCloseRus, pbkCloseUkr :
      begin
        Form := TForm(GetParentForm(Self));
        if Form <> nil then
          Form.Close
        else
          inherited Click;
      end;
    pbkHelp, pbkHelpRus, pbkHelpUkr :
      begin
        Control := Self;
        while (Control <> nil) and (Control.HelpContext = 0) do
          Control := Control.Parent;
        if Control <> nil then
          Application.HelpContext(Control.HelpContext)
        else
          inherited Click;
      end;
    else
      inherited Click;
  end;
end; { TProButton.Click }

function TProButton.IsCustom : Boolean;
{ Returns True if style of label is custom and False otherwise }
begin
  Result := FKind = pbkCustom;
end; { TProButton.IsCustomCaption }

function TProButton.IsCustomCaption : Boolean;
{ Returns True if caption of label is custom and False otherwise }
begin
  Result := CompareStr(Caption, LoadStr(ButtonCaptions[FKind])) <> 0;
end; { TProButton.IsCustomCaption }


{ TProSpeedButton }

constructor TProSpeedButton.Create(AOwner : TComponent);
{ Overrides the constructor to initialize variables }
begin
  inherited Create(AOwner);
  FRepeatInitInterval := 400;
  FRepeatInterval := 100;
end; { TProSpeedButton.Create }

destructor TProSpeedButton.Destroy;
{ Overrides the destructor to uninitialize variables }
begin
  if FRepeatTimer <> nil then
    FRepeatTimer.Free;
  inherited Destroy;
end; { TProSpeedButton.Destroy }

procedure TProSpeedButton.MouseDown(Button : TMouseButton;
  Shift : TShiftState; X, Y : Integer);
{ Calls the event handler attached to the OnMouseDown event. Besides
  it runs the timer. If the timer doesn't exists it creates one. }
begin
  inherited MouseDown(Button, Shift, X, Y);
  if FRepeatEnabled then
  begin
    if FRepeatTimer = nil then
      FRepeatTimer := TTimer.Create(Self);
    FRepeatTimer.OnTimer := TimerExpired;
    FRepeatTimer.Interval := FRepeatInitInterval;
    FRepeatTimer.Enabled := True;
  end;
end; { TProSpeedButton.MouseDown }

procedure TProSpeedButton.MouseUp(Button : TMouseButton;
  Shift : TShiftState; X, Y : Integer);
{ Calls the event handler attached to the OnMouseUp event. Besides
  it stops the timer if one exists. }
begin
  inherited MouseUp(Button, Shift, X, Y);
  if FRepeatTimer <> nil then
    FRepeatTimer.Enabled := False;
end; { TProSpeedButton.MouseUp }

procedure TProSpeedButton.SetRepeatEnabled(Value : Boolean);
{ Sets whether OnTimer events occur on the TProSpeedButton while
  the button remains pressed }
begin
  if FRepeatEnabled <> Value then
  begin
    FRepeatEnabled := Value;
    if not Value and (FRepeatTimer <> nil) then
      FRepeatTimer.Enabled := False;
  end;
end; { TProSpeedButton.SetRepeatEnabled }

procedure TProSpeedButton.TimerExpired(Sender : TObject);
{ Internal timer event handler. Calls Click methods when
  the button is pressed. }
begin
  FRepeatTimer.Interval := FRepeatInterval;
  if (FState = bsDown) and MouseCapture then
  begin
    try
      Click;
    except
      FRepeatTimer.Enabled := False;
      raise;
    end;
  end;
end; { TProSpeedButton.TimerExpired }


{ TProCustomListBox }

constructor TProCustomListBox.Create(AOwner : TComponent);
{ Overrides the constructor to initialize variables }
begin
  inherited Create(AOwner);
  FAutoHorzScroll := True;
end; { TProCustomListBox.Create }

procedure TProCustomListBox.SetAutoHorzScroll(Value : Boolean);
{ Sets the automatic mode of the showing of the horizontal scroll bar
  when property is changed }
begin
  if FAutoHorzScroll <> Value then
  begin
    FAutoHorzScroll := Value;
    ResetHorizontalExtent;
  end;
end; { TProCustomListBox.SetAutoHorzScroll }

procedure TProCustomListBox.SetHorzScrollExtent(Value : Cardinal);
{ Sets the width of the horizontal scroll bar when property is changed }
begin
  if FHorzScrollExtent <> Value then
  begin
    FHorzScrollExtent := Value;
    ResetHorizontalExtent;
  end;
end; { TProCustomListBox.SetHorzScrollExtent }

procedure TProCustomListBox.SetHorizontalExtent;
{ Sets the extent of the horizontal scroll bar }
var
  ItemWidth : Word;
begin
  ItemWidth := FHorzScrollExtent;
  if FAutoHorzScroll then Inc(ItemWidth, FMaxItemWidth);
  SendMessage(Handle, LB_SETHORIZONTALEXTENT, ItemWidth, 0);
end; { TProCustomListBox.SetHorizontalExtent }

procedure TProCustomListBox.ResetHorizontalExtent;
{ Calculates the extent of the horizontal scroll bar as the largest
  width of list box items and reset it }
var
  I : Integer;
  ItemWidth : Word;
begin
  FMaxItemWidth := 0;
  for I := 0 to Items.Count - 1 do
  begin
    ItemWidth := Canvas.TextWidth(Items[I] + 'I');
    if FMaxItemWidth < ItemWidth then
      FMaxItemWidth := ItemWidth;
  end;
  SetHorizontalExtent;
end; { TProCustomListBox.ResetHorizontalExtent }

procedure TProCustomListBox.CreateParams(var Params : TCreateParams);
{ Initializes the window-creation parameter record passed in the Params
  parameter. Besides it sets WS_HSCROLL Windows style for creation of
  the list box which can have a horizontal scroll bar. }
begin
  inherited CreateParams(Params);
  with Params do
    Style := Style or WS_HSCROLL;
end; { TProCustomListBox.CreateParams }

procedure TProCustomListBox.WndProc(var Message : TMessage);
{ Provides specific message responses for the list box. Handles messages
  for support the horizontal scroll bar. }
var
  ItemWidth : Word;
begin
  case Message.Msg of
    LB_ADDSTRING, LB_INSERTSTRING :
      begin
        ItemWidth := Canvas.TextWidth(StrPas(PChar(Message.lParam)) + 'I');
        if FMaxItemWidth < ItemWidth then
          FMaxItemWidth := ItemWidth;
        SetHorizontalExtent;
      end;
    LB_DELETESTRING :
      begin
        ItemWidth := Canvas.TextWidth(Items[Message.wParam] + 'I');
        if ItemWidth = FMaxItemWidth then
        begin
          inherited WndProc(Message);
          ResetHorizontalExtent;
          Exit;
        end;
      end;
    LB_RESETCONTENT :
      begin
        FMaxItemWidth := 0;
        SetHorizontalExtent;
      end;
    WM_SETFONT :
      begin
        inherited WndProc(Message);
        Canvas.Font.Assign(Self.Font);
        ResetHorizontalExtent;
        Exit;
      end;
  end;
  inherited WndProc(Message);
end; { TProCustomListBox.WndProc }


{ TProCustomPanel }

constructor TProCustomPanel.Create(AOwner : TComponent);
{ Overrides the constructor to initialize variables }
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle - [csSetCaption];
  FBevelSides := [sdLeft, sdTop, sdRight, sdBottom];
  FBorderColor := clBtnFace;
  FVerticalAlignment := vaCenter;
end; { TProCustomPanel.Create }

{$IFDEF WIN32}
procedure TProCustomPanel.CreateParams(var Params : TCreateParams);
{ CreateParams is called automatically to initialize the window-creation
  paramemters whenever the window for the panel needs to be created.
  CreateParams overrides the inherited methods to implement the
  BorderStyle property. }
begin
  inherited CreateParams(Params);
  with Params do
    if NewStyleControls and Ctl3D and (BorderStyle = bsSingle) then
    begin
      Style := Style or WS_BORDER;
      ExStyle := ExStyle and not WS_EX_CLIENTEDGE;
    end;
end; { TProCustomPanel. CreateParams}
{$ENDIF}

procedure TProCustomPanel.AlignControls(AControl : TControl; var Rect : TRect);
{ Aligns any controls contained within the panel and specifyes the area
  in which the controls aligned }
begin
  if FBevelCtl3D then BevelWidth := 2;
  inherited AlignControls(AControl, Rect);
end; { TProCustomPanel.AlignControls }

procedure TProCustomPanel.SetBevelCtl3D(Value : Boolean);
{ Sets the 3D style of the bevel when property is changed }
begin
  if FBevelCtl3D <> Value then
  begin
    FBevelCtl3D := Value;
    if FBevelCtl3D then BevelWidth := 2 else BevelWidth := 1;
    Invalidate;
  end;
end; { TProCustomPanel.SetBevelCtl3D }

procedure TProCustomPanel.SetBevelSides(Value : TSides);
{ Sets the sides of the bevels which will be framed when property
  is changed }
begin
  if FBevelSides <> Value then
  begin
    FBevelSides := Value;
    Invalidate;
  end;
end; { TProCustomPanel.SetBevelSides }

procedure TProCustomPanel.SetBorderColor(Value : TColor);
{ Sets the color of the border between bevels when property is changed }
begin
  if FBorderColor <> Value then
  begin
    FBorderColor := Value;
    Invalidate;
  end;
end; { TProCustomPanel.SetBorderColor }

procedure TProCustomPanel.SetVerticalAlignment(Value : TVerticalAlignment);
{ Sets the vertical alignment of the panel's caption }
begin
  if FVerticalAlignment <> Value then
  begin
    FVerticalAlignment := Value;
    Invalidate;
  end;
end; { TProCustomPanel.SetVerticalAlignment }

procedure TProCustomPanel.CMMouseEnter(var Message : TMessage);
{ CM_MOUSEENTER message handler }
begin
  inherited;
  MouseEnter;
end; { TProCustomPanel.CMMouseEnter }

procedure TProCustomPanel.CMMouseLeave(var Message : TMessage);
{ CM_MOUSELEAVE message handler }
begin
  inherited;
  MouseLeave;
end; { TProCustomPanel.CMMouseLeave }

procedure TProCustomPanel.MouseEnter;
{ Calls the event handler attached to the OnMouseEnter event }
begin
  if Assigned(FOnMouseEnter) then FOnMouseEnter(Self);
end; { TProCustomPanel.MouseEnter }

procedure TProCustomPanel.MouseLeave;
{ Calls the event handler attached to the OnMouseLeave event }
begin
  if Assigned(FOnMouseLeave) then FOnMouseLeave(Self);
end; { TProCustomPanel.MouseLeave }

procedure TProCustomPanel.Paint;
{ Draws the panel }
var
  Rect, TempRect : TRect;
  TopColor1, BottomColor1, TopColor2, BottomColor2 : TColor;
  HCaption, BRect : Integer;
{$IFNDEF WIN32}
  Text : array[0..255] of Char;
{$ENDIF}

  procedure AdjustColors(Bevel : TPanelBevel);
  { Adjusts colors for sides of the panel }
  begin
    case Bevel of
      bvLowered :
        begin
          TopColor1 := clBtnShadow;
          BottomColor1 := clBtnHighlight;
          TopColor2 := clBtnText;
        {$IFDEF WIN32}
          BottomColor2 := cl3DLight;
        {$ELSE}
          BottomColor2 := clBtnFace;
        {$ENDIF}
        end;
      bvRaised :
        begin
          TopColor1 := clBtnHighlight;
          BottomColor1 := clBtnShadow;
          if FBevelCtl3D then BottomColor1 := clBtnText;
        {$IFDEF WIN32}
          TopColor2 := cl3DLight;
        {$ELSE}
          TopColor2 := clBtnHighlight;
        {$ENDIF}
          BottomColor2 := clBtnShadow;
        end;
    end;
  end; { AdjustColors }

begin
  { Get the space of the component }
  Rect := ClientRect;

  { Draw the outer bevel }
  if BevelOuter <> bvNone then
  begin
    AdjustColors(BevelOuter);
    if FBevelCtl3D then
    begin
      Frame3DSides(Canvas, Rect, FBevelSides, TopColor1, BottomColor1, 1);
      Frame3DSides(Canvas, Rect, FBevelSides, TopColor2, BottomColor2, 1);
    end
    else
      Frame3DSides(Canvas, Rect, FBevelSides, TopColor1, BottomColor1,
        BevelWidth);
  end;

  { Draw the border }
  Frame3DSides(Canvas, Rect, FBevelSides, FBorderColor, FBorderColor,
    BorderWidth);

  { Draw the inner bevel }
  if BevelInner <> bvNone then
  begin
    AdjustColors(BevelInner);
    if FBevelCtl3D then
    begin
      Frame3DSides(Canvas, Rect, FBevelSides, TopColor1, BottomColor1, 1);
      Frame3DSides(Canvas, Rect, FBevelSides, TopColor2, BottomColor2, 1);
    end
    else
      Frame3DSides(Canvas, Rect, FBevelSides, TopColor1, BottomColor1,
        BevelWidth);
  end;

{$IFNDEF WIN32}
  { Get panel's caption }
  GetTextBuf(Text, sizeOf(Text));
{$ENDIF}

  with Canvas do
  begin
    { Fill in background }
    Brush.Color := Color;
    FillRect(Rect);
    { Don't overwrite background above }
    Brush.Style := bsClear;
    { Ensure canvas font is set }
    Font := Self.Font;
  {$IFNDEF WIN32}
    { Get text of the caption }
    GetTextBuf(Text, SizeOf(Text));
  {$ENDIF}

    { Calculate the rect for the panel's caption }
    if FVerticalAlignment <> vaTop then
    begin
      TempRect := Rect;
    {$IFDEF WIN32}
      HCaption := DrawText(Handle, PChar(Caption), -1, TempRect,
        DT_CALCRECT or DT_EXPANDTABS or DT_WORDBREAK or
        TextAlignment[Alignment]);
    {$ELSE}
      HCaption := DrawText(Handle, Text, -1, TempRect,
        DT_CALCRECT or DT_EXPANDTABS or DT_WORDBREAK or
        TextAlignment[Alignment]);
    {$ENDIF}
      BRect := Rect.Bottom;
      Rect.Bottom := Rect.Top + HCaption;

      { Align the caption within the vertical space of the component }
      if FVerticalAlignment = vaCenter then
        OffsetRect(Rect, 0, (BRect - Rect.Bottom) shr 1)
      else
        OffsetRect(Rect, 0, BRect - Rect.Bottom);
    end;

    { Draw the panel's caption } 
  {$IFDEF WIN32}
    DrawText(Handle, PChar(Caption), -1, Rect, DT_EXPANDTABS or
      DT_WORDBREAK or TextAlignment[Alignment]);
  {$ELSE}
    DrawText(Handle, Text, -1, Rect, DT_EXPANDTABS or
      DT_WORDBREAK or TextAlignment[Alignment]);
  {$ENDIF}
  end;
end; { TProCustomPanel.Paint }


{ TProBorder }

constructor TProBorder.Create(AOwner : TComponent);
{ Overrides the constructor to initialize variables }
begin
  inherited Create(AOwner);
  FBevelOuter := bvLowered;
  FBevelWidth := 1;
  FBevelSides := [sdLeft, sdTop, sdRight, sdBottom];
  FBorderColor := clBtnFace;
  Width := 50;
  Height := 50;
end; { TProBorder.Create }

procedure TProBorder.SetBevelCtl3D(Value : Boolean);
{ Sets the 3D style of the bevel when property is changed }
begin
  if FBevelCtl3D <> Value then
  begin
    FBevelCtl3D := Value;
    if FBevelCtl3D then FBevelWidth := 2 else FBevelWidth := 1;
    Invalidate;
  end;
end; { TProBorder.SetBevelCtl3D }

procedure TProBorder.SetBevelInner(Value : TPanelBevel);
{ Sets the style of the inner bevel when property is changed }
begin
  if FBevelInner <> Value then
  begin
    FBevelInner := Value;
    Invalidate;
  end;
end; { TProBorder.SetBevelInner }

procedure TProBorder.SetBevelOuter(Value : TPanelBevel);
{ Sets the style of the outer bevel when property is changed }
begin
  if FBevelOuter <> Value then
  begin
    FBevelOuter := Value;
    Invalidate;
  end;
end; { TProBorder.SetBevelOuter }

procedure TProBorder.SetBevelSides(Value : TSides);
{ Sets the sides of the bevels which will be framed when property
  is changed }
begin
  if FBevelSides <> Value then
  begin
    FBevelSides := Value;
    Invalidate;
  end;
end; { TProBorder.SetBevelSides }

procedure TProBorder.SetBevelWidth(Value : TBevelWidth);
{ Sets the width in pixels of both the inner and outer bevels
  when property is changed }
begin
  if (FBevelWidth <> Value) and (not FBevelCtl3D) then
  begin
    FBevelWidth := Value;
    Invalidate;
  end;
end; { TProBorder.SetBevelWidth }

procedure TProBorder.SetBorderColor(Value : TColor);
{ Sets the color of the border between bevels when property is changed }
begin
  if FBorderColor <> Value then
  begin
    FBorderColor := Value;
    Invalidate;
  end;
end; { TProBorder.SetBorderColor }

procedure TProBorder.SetBorderWidth(Value : TBorderWidth);
{ Sets the width of the border between bevels when property is changed }
begin
  if FBorderWidth <> Value then
  begin
    FBorderWidth := Value;
    Invalidate;
  end;
end; { TProBorder.SetBorderWidth }

procedure TProBorder.CMMouseEnter(var Message : TMessage);
{ CM_MOUSEENTER message handler }
begin
  inherited;
  MouseEnter;
end; { TProBorder.CMMouseEnter }

procedure TProBorder.CMMouseLeave(var Message : TMessage);
{ CM_MOUSELEAVE message handler }
begin
  inherited;
  MouseLeave;
end; { TProBorder.CMMouseLeave }

procedure TProBorder.MouseEnter;
{ Calls the event handler attached to the OnMouseEnter event }
begin
  if Assigned(FOnMouseEnter) then FOnMouseEnter(Self);
end; { TProBorder.MouseEnter }

procedure TProBorder.MouseLeave;
{ Calls the event handler attached to the OnMouseLeave event }
begin
  if Assigned(FOnMouseLeave) then FOnMouseLeave(Self);
end; { TProBorder.MouseLeave }

procedure TProBorder.Paint;
{ Draws the border }
var
  Rect : TRect;
  TopColor1, BottomColor1, TopColor2, BottomColor2 : TColor;

  procedure AdjustColors(Bevel : TPanelBevel);
  { Adjusts colors for sides of the panel }
  begin
    case Bevel of
      bvLowered :
        begin
          TopColor1 := clBtnShadow;
          BottomColor1 := clBtnHighlight;
          TopColor2 := clBtnText;
        {$IFDEF WIN32}
          BottomColor2 := cl3DLight;
        {$ELSE}
          BottomColor2 := clBtnFace;
        {$ENDIF}
        end;
      bvRaised :
        begin
          TopColor1 := clBtnHighlight;
          BottomColor1 := clBtnShadow;
          if FBevelCtl3D then BottomColor1 := clBtnText;
        {$IFDEF WIN32}
          TopColor2 := cl3DLight;
        {$ELSE}
          TopColor2 := clBtnHighlight;
        {$ENDIF}
          BottomColor2 := clBtnShadow;
        end;
    end;
  end; { AdjustColors }

begin
  { Get the space of the component }
  Rect := ClientRect;

  { Draw the outer bevel }
  if BevelOuter <> bvNone then
  begin
    AdjustColors(BevelOuter);
    if FBevelCtl3D then
    begin
      Frame3DSides(Canvas, Rect, FBevelSides, TopColor1, BottomColor1, 1);
      Frame3DSides(Canvas, Rect, FBevelSides, TopColor2, BottomColor2, 1);
    end
    else
      Frame3DSides(Canvas, Rect, FBevelSides, TopColor1, BottomColor1,
        BevelWidth);
  end;

  { Draw the border }
  Frame3DSides(Canvas, Rect, FBevelSides, FBorderColor, FBorderColor,
    BorderWidth);

  { Draw the inner bevel }
  if BevelInner <> bvNone then
  begin
    AdjustColors(BevelInner);
    if FBevelCtl3D then
    begin
      Frame3DSides(Canvas, Rect, FBevelSides, TopColor1, BottomColor1, 1);
      Frame3DSides(Canvas, Rect, FBevelSides, TopColor2, BottomColor2, 1);
    end
    else
      Frame3DSides(Canvas, Rect, FBevelSides, TopColor1, BottomColor1,
        BevelWidth);
  end;
end; { TProBorder.Paint }


{ TProToolBar }

constructor TProToolBar.Create(AOwner : TComponent);
{ Overrides the constructor to initialize variables }
begin
  inherited Create(AOwner);
  Align := alTop;
  BevelInner := bvRaised;
  BevelOuter := bvLowered;
  BevelSides := [sdTop];
  Height := 31;
end;

end.
