{*******************************************************}
{*                                                     *}
{*      Pro VCL Extensions Library                     *}
{*      Color Grid Unit                                *}
{*                                                     *}
{*      Copyright (c) 1996-98 by Dmitry Barabash       *}
{*                                                     *}
{*******************************************************}

unit ProClGrd;

{$I PRO.INC}

interface

uses Classes, Graphics, Controls, ExtCtrls, ProCtrls,
  {$IFDEF CBUILDER} ColorGr {$ELSE} ColorGrd {$ENDIF};

{ TProColorGrig }

type
  { Grid orientation }
  TColorGridOrientation = (goHorizontal, goVertical);

  TProColorGrid = class(TWinControl)
  private
    { Variables for properties }
    FOrientation : TColorGridOrientation;
    FOnChange : TNotifyEvent;
    { Private variables }
    FColorGrid : TColorGrid;
    FColorPanel : TProPanel;
    FForeColorPanel : TProPanel;
    FBackColorPanel : TProPanel;
    { Property access methods }
    function GetForeColor : TColor;
    function GetBackColor : TColor;
    procedure SetForeIndex(Value : Integer);
    function GetForeIndex : Integer;
    procedure SetBackIndex(Value : Integer);
    function GetBackIndex : Integer;
    procedure SetOrientation(Value : TColorGridOrientation);
    { Private methods }
    procedure AdjustBounds;
    procedure DoChange(Sender : TObject);
  protected
    procedure Change; dynamic;
  public
    constructor Create(AOwner : TComponent); override;
    procedure SetBounds(ALeft, ATop, AWidth, AHeight : Integer); override;
    { Foreground color }
    property ForeColor : TColor read GetForeColor;
    { Background color }
    property BackColor : TColor read GetBackColor;
  published
    { Foreground color index }
    property ForeIndex : Integer read GetForeIndex write SetForeIndex;
    { Background color index }
    property BackIndex : Integer read GetBackIndex write SetBackIndex;
    { Orientation of the grid }
    property Orientation : TColorGridOrientation read FOrientation
      write SetOrientation default goHorizontal;
    { OnChange occurs when the color changes }
    property OnChange : TNotifyEvent read FOnChange write FOnChange;
  end;

implementation

{ TProColorGrid }

constructor TProColorGrid.Create(AOwner : TComponent);
{ Overrides the constructor to initialize variables }
begin
  inherited Create(AOwner);

  FColorGrid := TColorGrid.Create(Self);
  with FColorGrid do
  begin
    Parent := Self;
    ClickEnablesColor := False;
    OnChange := DoChange;
  end;

  FColorPanel := TProPanel.Create(Self);
  with FColorPanel do
  begin
    Parent := Self;
    BevelInner := bvLowered;
    BevelOuter := bvLowered;
  end;

  FBackColorPanel := TProPanel.Create(FColorPanel);
  with FBackColorPanel do
  begin
    Parent := FColorPanel;
    BevelInner := bvRaised;
    Color := FColorGrid.BackgroundColor;
  end;

  FForeColorPanel := TProPanel.Create(FColorPanel);
  with FForeColorPanel do
  begin
    Parent := FColorPanel;
    BevelInner := bvRaised;
    Color := FColorGrid.ForegroundColor;
  end;

  Width := 220;
  Height := 42;
end; { TProColorGrid.Create }

function TProColorGrid.GetForeColor : TColor;
{ Gets the foreground color }
begin
  Result := FColorGrid.ForegroundColor;
end; { TProColorGrid.GetForeColor }

function TProColorGrid.GetBackColor : TColor;
{ Gets the background color }
begin
  Result := FColorGrid.BackgroundColor;
end; { TProColorGrid.GetBackColor }

procedure TProColorGrid.SetForeIndex(Value : Integer);
{ Sets the foreground color index when property is changed }
begin
  FColorGrid.ForegroundIndex := Value;
end; { TProColorGrid.SetForeIndex }

function TProColorGrid.GetForeIndex : Integer;
{ Gets the foreground color index }
begin
  Result := FColorGrid.ForegroundIndex;
end; { TProColorGrid.GetForeIndex }

procedure TProColorGrid.SetBackIndex(Value : Integer);
{ Sets the background color index when property is changed }
begin
  FColorGrid.BackgroundIndex := Value;
end; { TProColorGrid.SetBackIndex }

function TProColorGrid.GetBackIndex : Integer;
{ Gets the background color index }
begin
  Result := FColorGrid.BackgroundIndex;
end; { TProColorGrid.GetBackIndex }

procedure TProColorGrid.SetOrientation(Value : TColorGridOrientation);
{ Sets the grid orientation when property is changed }
begin
  if FOrientation <> Value then
  begin
    FOrientation := Value;
    SetBounds(Left, Top, Height, Width);
  end;
end; { TProColorGrid.SetOrientation }

procedure TProColorGrid.AdjustBounds;
{ Adjusts bounds of panels and color grid }
var
  I : Integer;
begin
  if FOrientation = goHorizontal then
  begin
    I := Height div 8;
    FColorPanel.SetBounds(1, 1, Height - 2, Height - 2);
    FForeColorPanel.SetBounds(I, I, Height div 2, Height div 2);
    FBackColorPanel.SetBounds(I * 3, I * 3, Height div 2, Height div 2);
    FColorGrid.GridOrdering := go8x2;
    FColorGrid.SetBounds(Height, 0, Width - Height, Height);
  end
  else
  begin
    I := Width div 8;
    FColorPanel.SetBounds(1, 1, Width - 2, Width - 2);
    FForeColorPanel.SetBounds(I, I, Width div 2, Width div 2);
    FBackColorPanel.SetBounds(I * 3, I * 3, Width div 2, Width div 2);
    FColorGrid.GridOrdering := go2x8;
    FColorGrid.SetBounds(0, Width, Width, Height - Width);
  end;
end; { TProColorGrid.AdjustBounds }

procedure TProColorGrid.DoChange(Sender : TObject);
{ Changes colors of panels and calls the OnChange event handler }
begin
  FForeColorPanel.Color := FColorGrid.ForegroundColor;
  FBackColorPanel.Color := FColorGrid.BackgroundColor;
  Change;
end; { TProColorGrid.DoChange }

procedure TProColorGrid.Change;
{ Calls the event handler attached to the OnChange event }
begin
  if Assigned(FOnChange) then FOnChange(Self);
end; { TProColorGrid.Change }

procedure TProColorGrid.SetBounds(ALeft, ATop, AWidth, AHeight : Integer);
{ Sets the windowed control's boundary properties all at once and
  adjust bounds of panels and color grid }
begin
  inherited SetBounds(ALeft, ATop, AWidth, AHeight);
  AdjustBounds;
end; { TProColorGrid.SetBounds }

end.
