{*******************************************************}
{*                                                     *}
{*      Pro VCL Extensions Library                     *}
{*      ProCheckListBox Unit                           *}
{*                                                     *}
{*      Copyright (c) 1996-98 by Dmitry Barabash       *}
{*                                                     *}
{*******************************************************}

unit ProChLst;

{$I PRO.INC}

interface


{$IFDEF OWN_CHECKLISTBOX}

uses {$IFDEF WIN32} Windows, {$ELSE} WinTypes, WinProcs, {$ENDIF}
  Messages, SysUtils, Classes, Graphics, Controls,
  StdCtrls, Forms, Menus, ProCtrls;

{ TProCheckListBox }

type
  TStateChangingEvent = procedure(Sender : TObject; Index : Integer;
    NewState : TCheckBoxState; var AllowChange : Boolean) of object;

  TProCheckListBox = class(TProCustomListBox)
  private
    { Variables for properties }
    FAllowGrayed : Boolean;
    FOnChanging : TStateChangingEvent;
    { Private variables }
    FCheckItems : TStringList;
    FSaveCheckItem : string;
    FSaveCheckItemState : LongInt;
    FSaveCheckItemIndex : Integer;
    FSaveItemIndex : Integer;
    CheckedBMP : TBitmap;
    CheckedDisabledBMP : TBitmap;
    PartialCheckedBMP : TBitmap;
    UncheckedBMP : TBitmap;
    UncheckedDisabledBMP : TBitmap;
    GrayedBMP : TBitmap;
    GrayedDisabledBMP : TBitmap;
    { Property access methods }
    procedure SetItemState(Index : Integer; Value : TCheckBoxState);
    function GetItemState(Index : Integer) : TCheckBoxState;
    procedure SetItemPartial(Index : Integer; Value : Boolean);
    function GetItemPartial(Index : Integer) : Boolean;
    procedure SetItemEnabled(Index : Integer; Value : Boolean);
    function GetItemEnabled(Index : Integer) : Boolean;
    { Private methods }
    procedure LoadBitmaps;
    procedure FreeBitmaps;
    procedure InvalidateItem(Index : Integer);
    procedure ResetItemHeight;
    procedure CMFontChanged(var Message : TMessage); message CM_FONTCHANGED;
  protected
    procedure CreateWnd; override;
    procedure WndProc(var Message : TMessage); override;
    procedure MouseDown(Button : TMouseButton; Shift : TShiftState;
      X, Y : Integer); override;
    procedure MouseUp(Button : TMouseButton; Shift : TShiftState;
      X, Y : Integer); override;
    procedure DrawItem(Index: Integer; Rect: TRect;
      State: TOwnerDrawState); override;
  public
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
    { States of items }
    property ItemState[Index : Integer] : TCheckBoxState read GetItemState
      write SetItemState;
    { Partial states of items }
    property ItemPartial[Index : Integer] : Boolean read GetItemPartial
      write SetItemPartial;
    { Set to True to enable specified item or to False to disable }
    property ItemEnabled[Index : Integer] : Boolean read GetItemEnabled
      write SetItemEnabled;
  published
    { Set to True to allow grayed state of items or to False otherwise }
    property AllowGrayed : Boolean read FAllowGrayed write FAllowGrayed
      default False;
    { OnChanging occurs when user changes the state of an item in the list }
    property OnChanging : TStateChangingEvent read FOnChanging
      write FOnChanging;
    { Enable standard properties }
    property Align;
    property BorderStyle;
    property Color;
    property Ctl3D;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property IntegralHeight;
    property ItemHeight;
    property Items;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property Sorted;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  {$IFDEF WIN32}
    property OnStartDrag;
  {$ENDIF}
  end;

implementation

{$R PROCHLST.RES}

{ TProCheckListBox }

constructor TProCheckListBox.Create(AOwner : TComponent);
{ Overrides the constructor to initialize variables }
begin
  inherited Create(AOwner);
  Style := lbOwnerDrawFixed;
  FCheckItems := TStringList.Create;
  LoadBitmaps;
end; { TProCheckListBox.Create }

destructor TProCheckListBox.Destroy;
{ Overrides the destructor to uninitialize variables }
begin
  FreeBitmaps;
  FCheckItems.Free;
  inherited Destroy;
end; { TProCheckListBox.Destroy }

procedure TProCheckListBox.LoadBitmaps;
{ Loads bitmap resources }
begin
  CheckedBMP := TBitmap.Create;
  CheckedBMP.Handle := LoadBitmap(HInstance, 'CHECKED');
  CheckedDisabledBMP := TBitmap.Create;
  CheckedDisabledBMP.Handle := LoadBitmap(HInstance, 'CHECKEDDISABLED');
  PartialCheckedBMP := TBitmap.Create;
  PartialCheckedBMP.Handle := LoadBitmap(HInstance, 'PARTIALCHECKED');
  UncheckedBMP := TBitmap.Create;
  UncheckedBMP.Handle := LoadBitmap(HInstance, 'UNCHECKED');
  UncheckedDisabledBMP := TBitmap.Create;
  UncheckedDisabledBMP.Handle := LoadBitmap(HInstance, 'UNCHECKEDDISABLED');
  GrayedBMP := TBitmap.Create;
  GrayedBMP.Handle := LoadBitmap(HInstance, 'GRAYED');
  GrayedDisabledBMP := TBitmap.Create;
  GrayedDisabledBMP.Handle := LoadBitmap(HInstance, 'GRAYEDDISABLED');
end; { TProCheckListBox.LoadBitmaps }

procedure TProCheckListBox.FreeBitmaps;
{ Destroy bitmaps }
begin
  CheckedBMP.Free;
  CheckedDisabledBMP.Free;
  PartialCheckedBMP.Free;
  UncheckedBMP.Free;
  UncheckedDisabledBMP.Free;
  GrayedBMP.Free;
  GrayedDisabledBMP.Free;
end; { TProCheckListBox.FreeBitmaps }

procedure TProCheckListBox.CreateWnd;
{ Creates a Windows control corresponding to the list box component.
  Besides it resets font size and horizontal scroll extent. }
begin
  inherited CreateWnd;
  ResetItemHeight;
  HorzScrollExtent := ItemHeight + 2;
end; { TProCheckListBox.CreateWnd }

procedure TProCheckListBox.WndProc(var Message : TMessage);
{ Provides specific message responses for the list box }
begin
  inherited WndProc(Message);
  with Message do
    case Msg of
      LB_ADDSTRING :
        begin
        {$IFDEF WIN32}
          FCheckItems.AddObject(PChar(lParam),
            TObject(LongInt(cbUnchecked) or LongInt(Byte(True) shl 16)));
        {$ELSE}
          FCheckItems.AddObject(StrPas(PChar(lParam)),
            TObject(LongInt(cbUnchecked) or LongInt(Byte(True) shl 16)));
        {$ENDIF}
          FSaveCheckItem := '';
        end;
      LB_INSERTSTRING :
        begin
        {$IFDEF WIN32}
          FCheckItems.Insert(wParam, PChar(lParam));
          if (FSaveCheckItem <> '') and
             (FSaveCheckItem = PChar(lParam)) and
        {$ELSE}
          FCheckItems.Insert(wParam, StrPas(PChar(lParam)));
          if (FSaveCheckItem <> '') and
             (FSaveCheckItem = StrPas(PChar(lParam))) and
        {$ENDIF}
             (FSaveCheckItemIndex <> wParam) then
            FCheckItems.Objects[wParam] := TObject(FSaveCheckItemState)
          else
            FCheckItems.Objects[wParam] := TObject(LongInt(cbUnchecked)
              or LongInt(Byte(True) shl 16));
          FSaveCheckItem := '';
        end;
      LB_DELETESTRING :
        begin
          FSaveCheckItemIndex := wParam;
          FSaveCheckItem := FCheckItems[wParam];
          FSaveCheckItemState := LongInt(FCheckItems.Objects[wParam]);
          FCheckItems.Delete(wParam);
        end;
      LB_RESETCONTENT :
        begin
          FCheckItems.Clear;
          FSaveCheckItem := '';
        end;
    end;
end; { TProCheckListBox.WndProc }

procedure TProCheckListBox.MouseDown(Button : TMouseButton;
  Shift : TShiftState; X, Y : Integer);
{ Calls the event handler attached to the OnMouseDown event }
begin
  inherited MouseDown(Button, Shift, X, Y);
  FSaveItemIndex := ItemIndex;
end; { TProCheckListBox.MouseDown }

procedure TProCheckListBox.MouseUp(Button : TMouseButton;
  Shift : TShiftState; X, Y : Integer);
{ Calls the event handler attached to the OnMouseUp event. Besides 
  it changes state of item. }
var
  Rect : TRect;
  NewState : TCheckBoxState;
  AllowChange : Boolean;
begin
  inherited MouseUp(Button, Shift, X, Y);
  if (FSaveItemIndex <> -1) and (FSaveItemIndex = ItemIndex) then
  begin
    Rect := ItemRect(ItemIndex);
    Rect.Right := Rect.Left + ItemHeight + 1;
    if PtInRect(Rect, Point(X, Y)) and ItemEnabled[ItemIndex] then
    begin
      case ItemState[ItemIndex] of
        cbChecked : NewState := cbUnchecked;
        cbUnchecked :
          if FAllowGrayed then NewState := cbGrayed
          else NewState := cbChecked;
        cbGrayed : NewState := cbChecked;
      end;
      AllowChange := True;
      if Assigned(FOnChanging) then
        FOnChanging(Self, ItemIndex, NewState, AllowChange);
      if AllowChange then
        ItemState[ItemIndex] := NewState;
    end;
    FSaveItemIndex := -1;
  end;
end; { TProCheckListBox.MouseUp }

procedure TProCheckListBox.DrawItem(Index : Integer; Rect : TRect;
  State : TOwnerDrawState);
{ List box calls DrawItem for each visible item in its list to draw it }
var
  Bitmap : TBitmap;
{$IFNDEF WIN32}
  Text : array[0..255] of Char;
{$ENDIF}
begin
  with Canvas do
  begin
    { Fill background of item rect }
    FillRect(Rect);

    { Draw state check box }
    case ItemState[Index] of
      cbChecked :
        if ItemEnabled[Index] then
        begin
          if ItemPartial[Index] then Bitmap := PartialCheckedBMP
          else Bitmap := CheckedBMP
        end
        else Bitmap := CheckedDisabledBMP;
      cbUnchecked :
        if ItemEnabled[Index] then Bitmap := UncheckedBMP
        else Bitmap := UncheckedDisabledBMP;
      cbGrayed :
        if ItemEnabled[Index] then Bitmap := GrayedBMP
        else Bitmap := GrayedDisabledBMP;
    end;
    CopyRect(Bounds(
      (Rect.Left + ItemHeight - Bitmap.Width + 2) shr 1,
      (Rect.Top + Rect.Bottom - Bitmap.Height) shr 1,
      Bitmap.Width, Bitmap.Height), Bitmap.Canvas,
      Bounds(0, 0, Bitmap.Width, Bitmap.Height));

    { Draw text of item }
    if not ItemEnabled[Index] then
      Canvas.Font.Color := clGray
    else if not (odSelected in State) then
      Canvas.Font.Color := Font.Color;
    Rect.Left := Rect.Left + ItemHeight + 2;
  {$IFDEF WIN32}
    DrawText(Handle, PChar(Items[Index]), -1, Rect, (DT_SINGLELINE or
      DT_VCENTER or DT_NOPREFIX));
  {$ELSE}
    StrPCopy(Text, Items[Index]);
    DrawText(Handle, Text, -1, Rect, (DT_SINGLELINE or DT_VCENTER or
      DT_NOPREFIX));
  {$ENDIF}
  end;
end; { TProCheckListBox.DrawItem }

procedure TProCheckListBox.ResetItemHeight;
{ Resets font size }
var
  DC : HDC;
  SaveFont : HFont;
  Metrics : TTextMetric;
  NewHeight : Integer;
begin
  DC := GetDC(0);
  try
    SaveFont := SelectObject(DC, Font.Handle);
    GetTextMetrics(DC, Metrics);
    SelectObject(DC, SaveFont);
  finally
    ReleaseDC(0, DC);
  end;
  NewHeight := Metrics.tmHeight + 2;
  if NewHeight < 14 then NewHeight := 14;
  ItemHeight := NewHeight;
end; { TProCheckListBox.ResetItemHeight }

procedure TProCheckListBox.CMFontChanged(var Message : TMessage);
{ CM_FONTCHANGED message handler. Resets font size. }
begin
  inherited;
  ResetItemHeight;
  RecreateWnd;
  HorzScrollExtent := ItemHeight + 2;
end; { TProCheckListBox.CMFontChanged }

procedure TProCheckListBox.InvalidateItem(Index : Integer);
{ Redraws item of list box }
var
  Rect : TRect;
begin
  FSaveCheckItem := '';
  Rect := ItemRect(Index);
  InvalidateRect(Handle, @Rect, True);
end; { TProCheckListBox.InvalidateItem }

procedure TProCheckListBox.SetItemState(Index : Integer;
  Value : TCheckBoxState);
{ Sets item state }
begin
  if (Value = cbGrayed) and not FAllowGrayed then Exit;
  if ItemState[Index] <> Value then
  begin
    FCheckItems.Objects[Index] :=
      TObject(LongInt(FCheckItems.Objects[Index]) and
      $FFFF0000 or LongInt(Value));
    InvalidateItem(Index);
  end;
end; { TProCheckListBox.SetItemState }

function TProCheckListBox.GetItemState(Index : Integer) : TCheckBoxState;
{ Gets item state }
begin
  Result := TCheckBoxState(LongInt(FCheckItems.Objects[Index]) and $0000FFFF);
end; { TProCheckListBox. GetItemState }

procedure TProCheckListBox.SetItemPartial(Index : Integer; Value : Boolean);
{ Sets item partial state }
begin
  if ItemPartial[Index] <> Value then
  begin
    FCheckItems.Objects[Index] :=
      TObject(LongInt(FCheckItems.Objects[Index]) and
      $00FFFFFF or (LongInt(Value) shl 24));
    InvalidateItem(Index);
  end;
end; { TProCheckListBox.SetItemPartial }

function TProCheckListBox.GetItemPartial(Index : Integer) : Boolean;
{ Gets item partial state }
begin
  Result := Boolean(LongInt(FCheckItems.Objects[Index]) shr 24);
end; { TProCheckListBox.GetItemPartial }

procedure TProCheckListBox.SetItemEnabled(Index : Integer; Value : Boolean);
{ Sets to true to enable item }
begin
  if ItemEnabled[Index] <> Value then
  begin
    FCheckItems.Objects[Index] :=
      TObject(LongInt(FCheckItems.Objects[Index]) and
      $FF00FFFF or (LongInt(Value) shl 16));
    InvalidateItem(Index);
  end;
end; { TProCheckListBox.SetItemEnabled }

function TProCheckListBox.GetItemEnabled(Index : Integer) : Boolean;
{ Gets enabled state of item }
begin
  Result := Boolean(LongInt(FCheckItems.Objects[Index]) shl 8 shr 24);
end; { TProCheckListBox.GetItemEnabled }

{$ELSE OWN_CHECKLISTBOX}

{*******************************************************}
{*                                                     *}
{*      Pro VCL Extensions Library                     *}
{*      ProCheckListBox Unit                           *}
{*                                                     *}
{*      Copyright (c) 1996-98 by Dmitry Barabash       *}
{*                                                     *}
{*******************************************************}

uses Windows, Messages, Classes, Graphics, Controls, CheckLst;

{ TProCheckListBox }

type
  TProCheckListBox = class(TCheckListBox)
  private
    { Variables for properties }
    FAutoHorzScroll : Boolean;
    FHorzScrollExtent : Cardinal;
    { Private variables }
    FMaxItemWidth : Cardinal;
    { Property access methods }
    procedure SetAutoHorzScroll(Value : Boolean);
    procedure SetHorzScrollExtent(Value : Cardinal);
    { Private methods }
    procedure SetHorizontalExtent;
    procedure ResetHorizontalExtent;
  protected
    procedure CreateParams(var Params : TCreateParams); override;
    procedure WndProc(var Message : TMessage); override;
  public
    constructor Create(AOwner : TComponent); override;
  published
    { Set to True to allow the automatic mode of the showing }
    { of the horizontal scroll bar                           }
    property AutoHorzScroll : Boolean read FAutoHorzScroll
      write SetAutoHorzScroll default True;
    { If AutoHorzScroll is True HorzScrollExtent property defines the }
    { addition extent to the automatic determined extent of the       }
    { horizontal scroll bar. If AutoHorzScroll is False this property }
    { completely defines the extent of the horizontal scroll bar.     }
    property HorzScrollExtent : Cardinal read FHorzScrollExtent
      write SetHorzScrollExtent default 0;
  end;

implementation

{ TProCheckListBox }

constructor TProCheckListBox.Create(AOwner : TComponent);
{ Overrides the constructor to initialize variables }
begin
  inherited Create(AOwner);
  FAutoHorzScroll := True;
end; { TProCheckListBox.Create }

procedure TProCheckListBox.SetAutoHorzScroll(Value : Boolean);
{ Sets the automatic mode of the showing of the horizontal scroll bar
  when property is changed }
begin
  if FAutoHorzScroll <> Value then
  begin
    FAutoHorzScroll := Value;
    ResetHorizontalExtent;
  end;
end; { TProCheckListBox.SetAutoHorzScroll }

procedure TProCheckListBox.SetHorzScrollExtent(Value : Cardinal);
{ Sets the width of the horizontal scroll bar when property is changed }
begin
  if FHorzScrollExtent <> Value then
  begin
    FHorzScrollExtent := Value;
    SetHorizontalExtent;
  end;
end; { TProCheckListBox.SetHorzScrollExtent }

procedure TProCheckListBox.SetHorizontalExtent;
{ Sets the extent of the horizontal scroll bar }
var
  ItemWidth : Cardinal;
begin
  ItemWidth := FHorzScrollExtent;
  if FAutoHorzScroll then Inc(ItemWidth, FMaxItemWidth);
  SendMessage(Handle, LB_SETHORIZONTALEXTENT, ItemWidth + GetCheckWidth, 0);
end; { TProCheckListBox.SetHorizontalExtent }

procedure TProCheckListBox.ResetHorizontalExtent;
{ Calculates the extent of the horizontal scroll bar as the largest
  width of list box items and resets it }
var
  I : Integer;
  ItemWidth : Cardinal;
begin
  FMaxItemWidth := 0;
  for I := 0 to Items.Count - 1 do
  begin
    ItemWidth := Canvas.TextWidth(Items[I] + 'I');
    if FMaxItemWidth < ItemWidth then
      FMaxItemWidth := ItemWidth;
  end;
  SetHorizontalExtent;
end; { TProCheckListBox.ResetHorizontalExtent }

procedure TProCheckListBox.CreateParams(var Params : TCreateParams);
{ Initializes the window-creation parameter record passed in the Params
  parameter. Besides it sets WS_HSCROLL Windows style for creation of
  the list box which can have a horizontal scroll bar. }
begin
  inherited CreateParams(Params);
  with Params do
    Style := Style or WS_HSCROLL;
end; { TProCheckListBox.CreateParams }

procedure TProCheckListBox.WndProc(var Message : TMessage);
{ Provides specific message responses for the list box. Handles messages
  for support the horizontal scroll bar. }
var
  ItemWidth : Cardinal;
begin
  case Message.Msg of
    LB_ADDSTRING, LB_INSERTSTRING :
      begin
        ItemWidth := Canvas.TextWidth(string(PChar(Message.lParam)) + 'I');
        if FMaxItemWidth < ItemWidth then
        begin
          FMaxItemWidth := ItemWidth;
          SetHorizontalExtent;
        end;
      end;
    LB_DELETESTRING :
      begin
        ItemWidth := Canvas.TextWidth(Items[Message.wParam] + 'I');
        if ItemWidth = FMaxItemWidth then
        begin
          inherited WndProc(Message);
          ResetHorizontalExtent;
          Exit;
        end;
      end;
    LB_RESETCONTENT :
      begin
        FMaxItemWidth := 0;
        SetHorizontalExtent;
      end;
    WM_SETFONT :
      begin
        inherited WndProc(Message);
        Canvas.Font.Assign(Self.Font);
        ResetHorizontalExtent;
        Exit;
      end;
  end;
  inherited WndProc(Message);
end; { TProCheckListBox.WndProc }

{$ENDIF OWN_CHECKLISTBOX}
         
end.
