{*******************************************************}
{*                                                     *}
{*      Pro VCL Extensions Library                     *}
{*      Calculator Dialog Unit                         *}
{*                                                     *}
{*      Copyright (c) 1996-98 by Dmitry Barabash       *}
{*                                                     *}
{*******************************************************}

unit ProCalc;

{$I PRO.INC}

interface

uses WinTypes, SysUtils, Classes, Controls, Forms, Menus, Buttons,
  ProCtrls, ProDlgs;

{ TProCalcDialog }

type
  TProCalcDialog = class(TProCommonDialog)
  private
    { Variables for properties }
    FModal : Boolean;
    FStayOnTop : Boolean;
    { Private variables }
    Form : TForm;
  public
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
    procedure Execute;
  published
    { Set to True to make the modal dialog or to False to make }
    { the modeless dialog                                      }
    property Modal : Boolean read FModal write FModal default True;
    { Set to True to keep this dialog in front of all other }
    { application windows and dialog boxes                  }
    property StayOnTop : Boolean read FStayOnTop write FStayOnTop
      default False;
  end;

{ CalcDialog }

procedure CalcDialog(ALanguage : TDialogLanguage; AHelpContext : LongInt);
{ Calculator dialog interface routine }

implementation

uses ProConst;

const
  { Number of digits in calculator display }
  DisplayDigits = 15;

  { Initialization string }
{$IFDEF WIN32}
  ZeroStr : string = '0';
{$ELSE}
  ZeroStr : string[1] = '0';
{$ENDIF}

type

{ TCalcForm }

  { Calculator state }
  TCalcState = (csFirst, csValid, csError);

  TCalcForm = class(TForm)
    PopupMenu: TPopupMenu;
    StayOnTopMenuItem: TMenuItem;
    HideMenuItem: TMenuItem;
    DisplayPanel: TProPanel;
    ClearButton: TProSpeedButton;
    BackButton: TProSpeedButton;
    PercentButton: TProSpeedButton;
    DivideButton: TProSpeedButton;
    MultiplyButton: TProSpeedButton;
    MinusButton: TProSpeedButton;
    PlusButton: TProSpeedButton;
    EqualButton: TProSpeedButton;
    Button0: TProSpeedButton;
    Button1: TProSpeedButton;
    Button2: TProSpeedButton;
    Button3: TProSpeedButton;
    Button4: TProSpeedButton;
    Button5: TProSpeedButton;
    Button6: TProSpeedButton;
    Button7: TProSpeedButton;
    Button8: TProSpeedButton;
    Button9: TProSpeedButton;
    PointButton: TProSpeedButton;
    NegativeButton: TProSpeedButton;
    procedure FormCreate(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure FormKeyPress(Sender: TObject; var Key: Char);
    procedure ClearButtonClick(Sender: TObject);
    procedure BackButtonClick(Sender: TObject);
    procedure NumberButtonClick(Sender: TObject);
    procedure PointButtonClick(Sender: TObject);
    procedure NegativeButtonClick(Sender: TObject);
    procedure DoCalc(Sender: TObject);
    procedure StayOnTopMenuItemClick(Sender: TObject);
    procedure HideMenuItemClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormDestroy(Sender: TObject);
  private
    FNumber : string;
    procedure SetNumber(Value : string);
  public
    FormOwner : TComponent;
    CalcState : TCalcState;
    Operator : Char;
    Operand : Extended;
    Negative : Boolean;
    ErrorMsg : string;
    property Number : string read FNumber write SetNumber;
  end;

{$R *.DFM}

{ Property access method }

procedure TCalcForm.SetNumber(Value : string);
{ Sets the number when property is changed }
begin
  FNumber := Value;
  if Negative then Value := '-' + Value;
  DisplayPanel.Caption := Value;
end; { TCalcForm.SetNumber }


{ Event handlers }

procedure TCalcForm.FormCreate(Sender: TObject);
{ OnCreate event handler }
begin
  PointButton.Caption := DecimalSeparator;
  ErrorMsg := 'Error';
  ClearButtonClick(nil);
end; { TCalcForm.FormCreate }

procedure TCalcForm.FormClose(Sender: TObject; var Action: TCloseAction);
{ OnClose event handler }
begin
  Action := caFree;
end; { TCalcForm.FormClose }

procedure TCalcForm.FormDestroy(Sender: TObject);
{ OnDestroy event handler }
begin
  TProCalcDialog(FormOwner).Form := nil;
end; { TCalcForm.FormDestroy }

procedure TCalcForm.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
{ OnKeyDown event handler }
begin
  if Key = VK_RETURN then
  begin
    Key := 0;
    DoCalc(EqualButton);
  end;
end; { TCalcForm.FormKeyDown }

procedure TCalcForm.FormKeyPress(Sender: TObject; var Key: Char);
{ OnKeyPress event handler }
begin
  case UpCase(Key) of
    '+' : DoCalc(PlusButton);
    '-' : DoCalc(MinusButton);
    '*' : DoCalc(MultiplyButton);
    '/' : DoCalc(DivideButton);
    '%' : DoCalc(PercentButton);
    '=' : DoCalc(EqualButton);
    '0'..'9' : NumberButtonClick(FindComponent('Button' + Key));
    '_' : NegativeButtonClick(Sender);
    #8  : BackButtonClick(Sender);
    'C' : ClearButtonClick(Sender);
  else
    if Key = DecimalSeparator then PointButton.Click;
  end;
end; { TCalcForm.FormKeyPress }

procedure TCalcForm.ClearButtonClick(Sender: TObject);
begin
  CalcState := csFirst;
  Operand := 0;
  Operator := '=';
  Negative := False;
  Number := ZeroStr;
end; { TCalcForm.ClearButtonClick }

procedure TCalcForm.BackButtonClick(Sender: TObject);
begin
  if CalcState = csError then Exit;
  if CalcState = csValid then
  begin
    Number := Copy(Number, 1, Length(Number) - 1);
    if Number = '' then Number := ZeroStr;
  end;
end; { TCalcForm.BackButtonClick }

procedure TCalcForm.NumberButtonClick(Sender: TObject);
var
  C : Char;
begin
  if CalcState = csError then Exit;

  C := (Sender as TProSpeedButton).Name[Length((Sender as TProSpeedButton).Name)];

  if CalcState = csFirst then
  begin
    CalcState := csValid;
    Negative := False;
    Number := C;
  end
  else if Length(Number) < DisplayDigits then
  begin
    if Number = ZeroStr then
      Number := C
    else
      Number := Number + C;
  end;
end; { TCalcForm.NumberButtonClick }

procedure TCalcForm.PointButtonClick(Sender: TObject);
begin
  if CalcState = csError then Exit;
  if CalcState = csFirst then
  begin
    CalcState := csValid;
    Negative := False;
    Number := ZeroStr;
  end;
  if (Pos(DecimalSeparator, Number) = 0) and (Length(Number) < DisplayDigits) then
    Number := Number + DecimalSeparator;
end; { TCalcForm.PointButtonClick }

procedure TCalcForm.NegativeButtonClick(Sender: TObject);
begin
  if CalcState = csError then Exit;
  Negative := not Negative;
  Number := Number;
end; { TCalcForm.NegativeButtonClick }

procedure TCalcForm.StayOnTopMenuItemClick(Sender: TObject);
begin
  with StayOnTopMenuItem do
  begin
    Checked := not Checked;
    if Checked then FormStyle := fsStayOnTop else FormStyle := fsNormal;
  end;
end; { TCalcForm.StayOnTopMenuItemClick }

procedure TCalcForm.HideMenuItemClick(Sender: TObject);
begin
  Close;
end; { TCalcForm.HideMenuItemClick }


{ Calculation method }

procedure TCalcForm.DoCalc(Sender: TObject);
{ Does calculation }
var
  Value : Extended;
  Key : Char;

  procedure Error;
  { Sets error state }
  begin
    CalcState := csError;
    DisplayPanel.Caption := ErrorMsg;
  end; { Error }

begin
  if CalcState = csError then Exit;

  { First char of button caption }
  Key := (Sender as TProSpeedButton).Caption[1];

  Value := StrToFloat(Number);
  if Negative then Value := -Value;

  if CalcState = csValid then
  begin
    CalcState := csFirst;

    if Key = '%' then
      case Operator of
        '+' : Value := Operand * Value / 100;
        '-' : Value := Operand * Value / 100;
        '*' : Value := Value / 100;
        '/' : Value := Value / 100;
      end;

    case Operator of
      '+' : Value := Operand + Value;
      '-' : Value := Operand - Value;
      '*' : Value := Operand * Value;
      '/' : if Value = 0 then begin Error; Exit; end
            else Value := Operand / Value;
    end;
  end;

  Operand := Value;
  Operator := Key;

  Negative := Value < 0;
  if Negative then Value := -Value;
  Number := LowerCase(FloatToStr(Value));

  if Length(Number) > DisplayDigits then Error;
end; { TCalcForm.DoCalc }


{ TProCalcDialog }

constructor TProCalcDialog.Create(AOwner : TComponent);
{ Overrides the constructor to initialize variables }
begin
  inherited Create(AOwner);
  FModal := True;
end; { TProCalcDialog.Create }

destructor TProCalcDialog.Destroy;
{ Overrides the destructor to uninitialize variables }
begin
  if Form <> nil then Form.Free;
  inherited Destroy;
end; { TProCalcDialog.Destroy }

procedure TProCalcDialog.Execute;
{ Displays the dialog box }
begin
  if Form <> nil then
    Form.SetFocus
  else
  begin
    Form := TCalcForm.Create(Application);
    try
      TCalcForm(Form).FormOwner := Self;

      case Language of
        dlRussian :
          begin
            Form.Caption := LoadStr(SCalcCapRus);
            TCalcForm(Form).ErrorMsg := LoadStr(SCalcErrorRus);
            TCalcForm(Form).StayOnTopMenuItem.Caption := LoadStr(SCalcStayOnTopRus);
            TCalcForm(Form).HideMenuItem.Caption := LoadStr(SCalcCloseRus);
          end;
        dlUkranian :
          begin
            Form.Caption := LoadStr(SCalcCapUkr);
            TCalcForm(Form).ErrorMsg := LoadStr(SCalcErrorUkr);
            TCalcForm(Form).StayOnTopMenuItem.Caption := LoadStr(SCalcStayOnTopUkr);
            TCalcForm(Form).HideMenuItem.Caption := LoadStr(SCalcCloseUkr);
          end;
      end;

      Form.HelpContext := Self.HelpContext;

      TCalcForm(Form).PopupMenu.AutoPopup := not FModal;

      if FModal then
      begin
        Form.ShowModal;
        Form.Free;
      end
      else
      begin
        if FStayOnTop then TCalcForm(Form).StayOnTopMenuItemClick(nil);
        Form.Show;
      end;
    except
      Form.Free;
      raise;
    end;
  end;
end; { TProCalcDialog.Execute }


{ CalcDialog }

procedure CalcDialog(ALanguage : TDialogLanguage; AHelpContext : LongInt);
{ Calculator dialog interface routine }
begin
  with TProCalcDialog.Create(Application) do
  try
    Modal := True;
    Language := ALanguage;
    HelpContext := AHelpContext;
    Execute;
  finally
    Free;
  end;
end; { CalcDialog }

end.
