unit PMGrpFile;
{	Usage of a var PMGFGroup:TPMGFGroup is restricted to the code between the "brackets"
		PMGFGroup:= TPMGFGroup.OpenGrp(FileName);
	and
		PMGFGroup.CloseGrp;
	OpenGrp reserves resources: storage for the object and a file handle.
	The TStringList PMGFGroup.Items contains the names of group items.

	To get access to more information concerning the item with index I, associate to
	it an TPMGFItem object by
		PMGFGroup.OpenItem(I);
	OpenItem(I) is idempotent: more than one call (before a CloseItem(I)) do not matter.

	The TPMGFItem may be dissociated explicitly by
		PMGFGroup.CloseItem(I);
	CloseGrp will implicitly release all TPMGFItems and free all reserved resources.
	Errors are signalled by Exceptions labelled with PMGrpFile/severity
	Freeware, P.J. Veger, Best, The Netherlands}

interface
uses Sysutils, WinTypes, WinProcs, Classes, PMGrps;
type						{describing information as offered to user of this unit}
	TPMGFGroup			= class(TObject)
		public
			GrpFileName	: string[80];  {path name of .grp file}
			GroupName	: string[32];	{name of group}
			CmdShow		: Word;			{group window: sw xxxx constant}
			NormalRect	: TRect;			{group window: normal-window rectangle}
			MinPoint		: TPoint;		{group window: icon-window lower-left corner point}
			LogPixelsX	: Word;			{icon bitmaps: intended hor. resolution}
			LogPixelsY	: Word;			{icon bitmaps: intended vert. resolution}
			BitsPerPixel: Word;			{icon bitmaps: bits-per-pixel}
			Planes		: Word;			{icon bitmaps: count of planes}
			Items			: TStringList; {itemnames, with associated TGrpItem objects}
		private
			F				: file;			{opened in constructor, closed in destructor}
			FSize			: Longint;		{size of file}
			Its			: array[ItemIndexSet] of Word;	{offsets to item info}
			Dir			: array[ItemIndexSet] of Word;	{tagdata: offsets to dirs}
			Hot			: array[ItemIndexSet] of Word;	{tagdata: hotkeys}
			Min			: array[ItemIndexSet] of Boolean;{tagdata: run minimized}
		public
			constructor	Create(FileName: string);
			destructor	Destroy; override;
			procedure	OpenItem(I: ItemIndexSet);
			procedure	CloseItem(I: ItemIndexSet);
			procedure	OpenAllItems;
			procedure	CloseAllItems;
		private
			procedure	GetBlk(Offset: Word; var Buffer; Count: Word);
		end;

	TPMGFItem			= class(TObject)
		public
			CommandLine	: string[144];	{Command Line}
			DefaultDir	: string[80];	{Working Directory}
			ShortCut		: Word;			{Shortcut Key}
			RunMinimized: Boolean;		{Run minimized}
			IconPath		: string[80];	{Icon: path of file containing icon}
			IconIndex	: Word;			{Icon index value: position in its file}
			IconPoint	: TPoint;		{Icon lower-left corner point}
			IconWidth	: Integer;		{Icon width}
			IconHeight	: Integer;		{Icon height}
			IconBPR		: Integer;		{Icon: bytes per row for Word alignment}
			IconBPP		: Word;			{Icon: bits per pixel}
			IconPlanes	: Word;			{Icon: count of planes}
			ANDPlaneSize: Word;			{Icon AND mask: count of bytes}
			ANDPlane		: PChar;			{Icon AND mask: pointer}
			XORPlaneSize: Word;			{Icon XOR mask: count of bytes}
			XORPlane		: PChar;			{Icon XOR mask: pointer}
      private
			constructor	Create(N1, N2: Word);
			destructor	Destroy; override;
		end;

implementation
{$I+}
type						{describing contents of .grp-file}
	PGroupHeader		= ^TGroupHeader;
	TGroupHeader		= record
		cIdentifier 	: array [0..3] of Char;	{validation; must be: 'PMCC'}
		wCheckSum		: Word;	{validation: sum of all Words in the Grp file must be 0}
		cbGroup			: Word;	{grp file: size in bytes, NOT including the TagData}
										{thus: offset from BOF to TagData}
		nCmdShow			: Word;	{group window: sw xxxx constant}
		rcNormal			: TRect;	{group window: normal-window rectangle}
		ptMin				: TPoint;{group window: icon-window lower-left corner point}
		pName				: Word;	{group name: offset from BOF}
		wLogPixelsX		: Word;	{icon bitmaps: intended hor. resolution}
		wLogPixelsY		: Word;	{icon bitmaps: intended vert. resolution}
		wBitsPerPixel	: Byte;	{icon bitmaps: bits-per-pixel}
		wPlanes			: Byte;	{icon bitmaps: count of planes}
		wReserved		: Word;
		cItems			: Word;	{number of items in the rgiItems array}
		rgiItems			: array[ItemIndexSet] of Word {offsets to program item data}
		end;
	PItemData			= ^TItemData;
	TItemData			= record
		pt					: TPoint;{icon: (left, lower) point in group window}
		iIcon				: Word;	{icon index value: position in its file}
		cbHeader			: Word;	{icon resource header: count of bytes}
		cbANDPlane		: Word;	{icon AND mask: count of bytes}
		cbXORPlane		: Word;	{icon XOR mask: count of bytes}
		pHeader			: Word;	{icon resource header: offset from BOF}
		pANDPlane		: Word;	{icon AND mask: offset from BOF}
		pXORPlane		: Word;	{icon XOR mask: offset from BOF}
		pName				: Word;	{item name string: offset from BOF}
		pCommand			: Word;	{command line string: offset from BOF}
		pIconPath		: Word	{path of file containing the icon: offset from BOF}
		end;
	PIconResourceHeader= ^TIconResourceHeader;
	TIconResourceHeader= record
		xHotSpot			: Integer;{should be 0}
		yHotSpot			: Integer;{should be 0}
		cx					: Integer;{icon width}
		cy					: Integer;{icon height}
		cbWidth			: Integer;{bytes per row for Word alignment}
		bPlanes			: Byte;	 {count of planes}
		bBitsPixel		: Byte	 {bits per pixel}
		end;
	PTagData				= ^TTagData;
	TTagData				= record
		wID				: Word;	{case discriminator}
		wItem				: Word;	{index of item the tag data refers to}
		cb					: Word;	{size in bytes of record}
		info				: record
      	case {wID	:} Word of
				$8000: 	();		{first record}
				$8101: 	(rgbString: array [0..144] of Char); {appl program path name}
				$8102:	(rgbShortCut: Word); {hotkey; high: 1=>Shft, 2=>Ctrl,	4=>Alt}
				$8103:	();		{run minimized}
				$ffff:	()			{last record}
			end
		end;

procedure TPMGFGroup.GetBlk(Offset: Word; var Buffer; Count: Word);
begin
	if Count>0 then
		try
			Seek(F, Offset);
			BlockRead(F, Buffer, Count)
		except
			on E: EInOutError do
				raise Exception.CreateFmt(
						'PMGrpFile/3: IO Error number %d while reading file %s'
							, [E.ErrorCode, GrpFileName])
		end
end;

constructor TPMGFGroup.Create(FileName: string);
var FBuf	: PChar;
procedure ValidateGrpFile;
	type Words= array[0..32760] of Word;
	var FW	 : ^Words absolute FBuf;
		CSum, N: Word;
	begin
		if StrLComp(FBuf, 'PMCC', 4)<>0 then
			raise Exception.CreateFmt(
						'PMGrpFile/1: File %s is not a .GRP file', [GrpFileName]);
		CSum:= 0;
		for N:= 0 to Pred(FSize div 2) do Inc(CSum, FW^[N]);
		if CSum<>0 then
			raise Exception.CreateFmt(
						'PMGrpFile/2: File %s has a bad checksum', [GrpFileName]);
	end;
var PGH	: PGroupHeader absolute FBuf;
	I		: ItemIndexSet;
	PID	: PItemData;
	PTD	: PTagData;
	PC		: PChar absolute PTD;
begin
	inherited Create;
	if not FileExists(FileName) then
		raise Exception.CreateFmt(
						'PMGrpFile/1: File %s does not exist', [FileName]);
	GrpFileName:= FileName;
	try
		AssignFile(F, GrpFileName);
		Reset(F, 1)
	except
		on E: EInOutError do
			raise Exception.CreateFmt(
						'PMGrpFile/3: Error %d while opening file %s'
						, [E.ErrorCode, GrpFileName])
	end;
	FSize:= FileSize(F);
	GetMem(FBuf,FSize);
	try
		GetBlk(0, FBuf^, FSize);
		ValidateGrpFile;
		GroupName:= StrPas(FBuf+PGH^.pName);
		CmdShow:= PGH^.nCmdShow;
		NormalRect:= PGH^.rcNormal;
		MinPoint:= PGH^.ptMin;
		LogPixelsX:= PGH^.wLogPixelsX;
		LogPixelsY:= PGH^.wLogPixelsY;
		BitsPerPixel:= PGH^.wBitsPerPixel;
		Planes:= PGH^.wPlanes;
		Items:= TStringList.Create;
		FillChar(Its, SizeOf(Its), 0);
		FillChar(Dir, SizeOf(Dir) ,0);
		Fillchar(Hot, SizeOf(Hot) ,0);
		FillChar(Min, SizeOf(Min) ,False);
		Move(PGH^.rgiItems, Its, PGH^.cItems * SizeOf(Word));
		for I:= 0 to PGH^.cItems-1 do begin
			PID:= PItemData(FBuf+PGH^.rgiItems[I]);
			Items.Add(StrPas(FBuf+PID^.pName))
		end;
		if FSize > PGH^.cbGroup then begin
			PTD:= PTagData(FBuf+PGH^.cbGroup);
			if PTD^.wID<>$8000 then
				raise Exception.CreateFmt(
						'PMGrpFile/2: First tag in file %s is bad', [GrpFileName]);
			Inc(PC,PTD^.cb)	{flush first info};
			while PTD^.wID<>$ffff do begin
				case PTD^.wID of
				$8101: Dir[PTD^.wItem]:= PC+3*SizeOf(Word)-FBuf;
				$8102: Hot[PTD^.wItem]:= PTD^.info.rgbShortCut;
				$8103: Min[PTD^.wItem]:= True
				else raise Exception.CreateFmt(
						'PMGrpFile/2: A Tag in file %s is bad',[GrpFileName]);
				end;
				Inc(PC,PTD^.cb)
			end
		end
	finally
		FreeMem(FBuf,FSize)
	end
end;

destructor TPMGFGroup.Destroy;
var I: Word;
begin
	try
		try
			Close(F)
		except
			on E: EinOutError do
				raise Exception.CreateFmt(
						'PMGrpFile/2: Error %d while closing file %d'
						, [E.ErrorCode, GrpFileName])
		end;
		CloseAllItems;
	finally
		inherited Destroy
	end
end;

procedure TPMGFGroup.OpenAllItems;
var I: Word;
begin for I:= 0 to Items.Count-1 do OpenItem(I) end;

procedure TPMGFGroup.CloseAllItems;
var I: Word;
begin for I:= 0 to Items.Count-1 do CloseItem(I) end;

procedure TPMGFGroup.OpenItem(I: ItemIndexSet);
procedure SetStr(var T: OpenString; Offset: Word);
	var AuxBuf: array [0..255] of char;
		PC		: PChar;
		ToRead: Word;
	begin
		if Offset=0 then T:= ''
		else begin
			if Offset+256<FSize then ToRead:= 256 else ToRead:= FSize-Offset;
			GetBlk(Offset, AuxBuf, ToRead);
			PC:= StrScan(AuxBuf, #0);
			if PC=nil then raise Exception.CreateFmt(
									'PMGrpFile/2: String error in Item %d',[I]);
			if PC-AuxBuf > SizeOf(T) then raise Exception.CreateFmt(
									'PMGrpFile/2: String too large in Item %d',[I]);
			T:= StrPas(AuxBuf);
		end
	end;
var Item	: TPMGFItem;
	ID		: TItemData;
	IRC	: TIconResourceHeader;
	PC		: PChar;
var Pth, Fnm: string[80];
	B		: Byte;
begin
	if I>=Items.Count then
		raise Exception.CreateFmt('PMGrpFile/0: Item index %d too large: there are nly %d items'
											, [I, Items.Count]);
	if Items.Objects[I]=nil then begin
		GetBlk(Its[I], ID, SizeOf(TItemData));
		GetBlk(ID.pHeader, IRC, SizeOf(TIconResourceHeader));
		Item:= TPMGFItem.Create(ID.cbANDPlane, ID.cbXORPlane);
		with Item do begin
			SetStr(CommandLine, ID.pCommand);
			SetStr(DefaultDir, Dir[I]);
			if DefaultDir<>'' then begin {quirk in .prg file strcuture}
				B:= Pos(' ', CommandLine);
				if B=0 then B:= Length(CommandLine);
				Pth:= Copy(CommandLine, 1, B);
				Delete(CommandLine, 1, B);
				Fnm:= ExtractFilename(Pth);
				Pth:= ExtractFilePath(Pth);
				CommandLine:= DefaultDir+Fnm+CommandLine;
				DefaultDir:= Pth
			end;
			ShortCut:= Hot[I];
			RunMinimized:= Min[I];
			SetStr(IconPath, ID.pIconPath);
			IconIndex:= ID.iIcon;
			IconPoint:= ID.pt;
			IconWidth:= IRC.cx;
			IconHeight:= IRC.cy;
			IconBPR:= IRC.cbWidth;
			IconBPP:= IRC.bBitsPixel;
			IconPlanes:= IRC.bPlanes;
			ANDPlaneSize:= ID.cbANDPlane;
			GetBlk(ID.pANDPlane, ANDPlane^, ANDPlaneSize);
			XORPlaneSize:= ID.cbXORPlane;
			GetBlk(ID.pXORPlane, XORPlane^, XORPlaneSize);
		end;
		Items.Objects[I]:= Item;
	end
end;

procedure TPMGFGroup.CloseItem(I: ItemIndexSet);
begin
	if Items.Objects[I] is TPMGFItem then begin
		TPMGFItem(Items.Objects[I]).Destroy;
		Items.Objects[I]:= nil
	end;
end;

constructor TPMGFItem.Create(N1, N2: Word);
begin
	inherited Create;
	if N1>0 then GetMem(ANDPlane, N1) else ANDPlane:= nil;
	if N2>0 then GetMem(XORPlane, N2) else XORPlane:= nil
end;

destructor TPMGFItem.Destroy;
begin
	if ANDPlaneSize>0 then FreeMem(ANDPlane, ANDPlaneSize);
	if XORPlaneSize>0 then FreeMem(XORPlane, XORPlaneSize);
	inherited Destroy
end;

end.
