unit Seredit;

{$I Plot.inc}

{-----------------------------------------------------------------------------
The contents of this file are subject to the Q Public License
("QPL"); you may not use this file except in compliance
with the QPL. You may obtain a copy of the QPL from 
the file QPL.html in this distribution, derived from:

http://www.trolltech.com/products/download/freelicense/license.html

The QPL prohibits development of proprietary software. 
There is a Professional Version of this software available for this. 
Contact sales@chemware.hypermart.net for more information.

Software distributed under the QPL is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either expressed or implied. See the QPL for
the specific language governing rights and limitations under the QPL.

The Original Code is: SerEdit.pas, released 12 September 2000.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@chemware.hypermart.net.

Last Modified: 02/25/2000
Current Version: 2.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind. 

Purpose:
To facilitate user manipluation of Series properties

Known Issues:
-----------------------------------------------------------------------------}

interface

uses
  Classes, SysUtils, TypInfo,
{$IFDEF WINDOWS}
  WinTypes, WinProcs,
  Buttons, Controls, {Dialogs,} Forms, Graphics, StdCtrls,
{$ENDIF}
{$IFDEF WIN32}
  Windows,
  Buttons, Controls, {Dialogs,} Forms, Graphics, StdCtrls,
{$ENDIF}
{$IFDEF LINUX}
  QButtons, QControls, QDialogs, QExtCtrls, QForms, QGraphics, QStdCtrls,
{$ENDIF}

  Misc, NEdit, Plotdefs, Coloredt;

type
  TSeriesProperty = record
    PenColor: TColor;
    PenWidthIndex: Integer;
    PenStyleIndex: Integer;
    BrushColor: TColor;
    BrushStyleIndex: Integer;
    SymbolIndex: Integer;
    SymbolSize: Integer;
    YAxisIndex: Byte;
    DeltaX: Integer;
    DeltaY: Integer;
    XDataIndependent: Boolean;
    ExternalXSeries: Boolean;
    Visible: Boolean;
  end;

  TSeriesEditorForm = class(TForm)
    HelpBitBtn: TBitBtn;
    CancelBitBtn: TBitBtn;
    OKBitBtn: TBitBtn;
    Label1: TLabel;
    Label3: TLabel;
    VisibleCheckBox: TCheckBox;
    SymbolComboBox: TComboBox;
    PenGroupBox: TGroupBox;
    Label2: TLabel;
    Label4: TLabel;
    PenWidthComboBox: TComboBox;
    PenStyleComboBox: TComboBox;
    Label5: TLabel;
    NoComboBox: TComboBox;
    GroupBox1: TGroupBox;
    Label6: TLabel;
    Label7: TLabel;
    Label8: TLabel;
    NoLabel: TLabel;
    NameEdit: TEdit;
    Label10: TLabel;
    YAxisComboBox: TComboBox;
    XDataIndependentCheckBox: TCheckBox;
    BrushGroupBox: TGroupBox;
    Label11: TLabel;
    Label13: TLabel;
    BrushStyleComboBox: TComboBox;
    SymbolSizeNEdit: TNEdit;
    DeltaXNEdit: TNEdit;
    DeltaYNEdit: TNEdit;
    PenColorEdit: TColorEdit;
    BrushColorEdit: TColorEdit;
    ApplyBitBtn: TBitBtn;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure NoComboBoxClick(Sender: TObject);
    procedure PenWidthComboBoxClick(Sender: TObject);
    procedure PenStyleComboBoxClick(Sender: TObject);
    procedure SymbolComboBoxClick(Sender: TObject);
    procedure VisibleCheckBoxClick(Sender: TObject);
    procedure YAxisComboBoxClick(Sender: TObject);
    procedure DeltaXNEditChange(Sender: TObject);
    procedure DeltaYNEditChange(Sender: TObject);
    procedure SymbolSizeNEditChange(Sender: TObject);
    procedure NameEditChange(Sender: TObject);
    procedure XDataIndependentCheckBoxClick(Sender: TObject);
    procedure PenColorEditChange(Sender: TObject);
    procedure BrushColorEditChange(Sender: TObject);
    procedure ApplyBitBtnClick(Sender: TObject);
  private
    pASP: ^TSeriesProperty;
    CurrentIndex: Integer;

  public
    SeriesPropertyList: TList;
    SeriesNames: TStringList;
    ThePlot: TObject;
    
    function AddSeries(AName: String; ASeriesProperty: TSeriesProperty): Integer;
    procedure SelectSeries(Index: Integer);
  end;

var
  SeriesEditorForm: TSeriesEditorForm;

implementation

{$R *.dfm}

uses
  Plot;
  
{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.FormCreate
  Description: standard FormCreate procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the position, populates the PenColor combo, and initializes SeriesNames and SeriesPropertyList
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.FormCreate(Sender: TObject);
var
  i: Integer;
begin
  SetDialogGeometry(Self, ApplyBitBtn, NoLabel.Left);

{Populate the combo boxes:}
  for i := 0 to Ord(High(TBrushStyle)) do
    BrushStyleComboBox.Items.Add(Copy(GetEnumName(TypeInfo(TBrushStyle), i), 3, 99));
  for i := 0 to 20 do
    PenWidthComboBox.Items.Add(IntToStr(i));
  for i := 0 to Ord(High(TPenStyle)) do
    PenStyleComboBox.Items.Add(Copy(GetEnumName(TypeInfo(TPenStyle), i), 3, 99));
  for i := 0 to Ord(High(TSymbol)) do
    SymbolComboBox.Items.Add(Copy(GetEnumName(TypeInfo(TSymbol), i), 3, 99));

{set combo and edit box widths:}
  for i := 0 to Self.ComponentCount - 1 do
    if ((Self.Components[i] is TEdit) or
        (Self.Components[i] is TNEdit) or
        (Self.Components[i] is TColorEdit) or
        (Self.Components[i] is TComboBox)) then
      TControl(Self.Components[i]).Width := 88;
{the exceptions:}
  NoComboBox.Width := 48;
  SymbolSizeNEdit.Width := 48;
  YAxisComboBox.Width := 145;

  SymbolComboBox.ItemIndex := 0;
  SeriesPropertyList := TList.Create;
  SeriesNames := TStringList.Create;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.FormDestroy
  Description: standard FormDestroy procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: frees the SeriesNames and SeriesPropertyList
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.FormDestroy(Sender: TObject);
var
  i: Integer;
begin
  for i := 0 to SeriesPropertyList.Count-1 do
    FreeMem(SeriesPropertyList.Items[i], SizeOf(TSeriesProperty));

  SeriesPropertyList.Free;
  SeriesNames.Free;
end;

{------------------------------------------------------------------------------
     Function: TSeriesEditorForm.AddSeries
  Description: adds a new series to the list
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: stores the Series properties
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesEditorForm.AddSeries(AName: String; ASeriesProperty: TSeriesProperty): Integer;
var
  i: Integer;
  pDestination: Pointer;
  pSource,
  pDest: PChar;
begin
  NoComboBox.Items.Add(IntToStr(SeriesNames.Count));

  SeriesNames.Add(AName);

  GetMem(pDestination, SizeOf(ASeriesProperty));
  pSource := @ASeriesProperty;
{NB: this causes terminal access violations:
  System.Move(pSource, pDestination, SizeOf(TSeriesProperty));}
  pDest := pDestination;
  for i := 1 to SizeOf(ASeriesProperty) do
  begin
    pDest^ := pSource^;
    Inc(pSource);
    Inc(pDest);
  end;

  SeriesPropertyList.Add(pDestination);
  AddSeries := SeriesPropertyList.Count;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.SelectSeries
  Description: responds to selection of a new Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.SelectSeries(Index: Integer);
begin
  pASP := SeriesPropertyList.Items[Index];
  CurrentIndex := Index;

  NameEdit.Text := SeriesNames.Strings[Index];

  PenColorEdit.Color := pASP^.PenColor;
  PenColorEdit.Text := ColorToString(pASP^.PenColor)+ ' ...';
  PenWidthComboBox.ItemIndex := pASP^.PenWidthIndex;
  PenStyleComboBox.ItemIndex := pASP^.PenStyleIndex;

  BrushColorEdit.Color := pASP^.BrushColor;
  BrushColorEdit.Text := ColorToString(pASP^.BrushColor)+ ' ...';
  BrushStyleComboBox.ItemIndex := pASP^.BrushStyleIndex;
  SymbolComboBox.ItemIndex := pASP^.SymbolIndex;
  SymbolSizeNEdit.AsInteger := pASP^.SymbolSize;
{We define YAxisIndex to run from 1 to FAxisList.Count-1}
  YAxisComboBox.ItemIndex := pASP^.YAxisIndex-1;
  DeltaXNEdit.AsInteger := pASP^.DeltaX;
  DeltaYNEdit.AsInteger := pASP^.DeltaY;
  VisibleCheckBox.Checked := pASP^.Visible;
  XDataIndependentCheckBox.Checked := pASP^.XDataIndependent;
  XDataIndependentCheckBox.Visible := pASP^.ExternalXSeries;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.NoComboBoxClick
  Description: responds to user selection of a new Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues: see SelectSeries above
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.NoComboBoxClick(Sender: TObject);
begin
  SelectSeries(NoComboBox.ItemIndex);
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.PenWidthComboBoxClick
  Description: responds to selection of a new Pen Width for the Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.PenWidthComboBoxClick(Sender: TObject);
begin
  pASP^.PenWidthIndex := PenWidthComboBox.ItemIndex;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.PenStyleComboBoxClick
  Description: responds to selection of a new Pen Style for the Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.PenStyleComboBoxClick(Sender: TObject);
begin
  pASP^.PenStyleIndex := PenStyleComboBox.ItemIndex;
  if (PenStyleComboBox.ItemIndex > Ord(psSolid)) then
    PenWidthComboBox.ItemIndex := 1;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.SymbolComboBoxClick
  Description: responds to selection of a new Symbol for the Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.SymbolComboBoxClick(Sender: TObject);
begin
  pASP^.SymbolIndex := SymbolComboBox.ItemIndex;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.VisibleCheckBoxClick
  Description: responds to changes to the Visibility of the Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.VisibleCheckBoxClick(Sender: TObject);
begin
  pASP^.Visible := VisibleCheckBox.Checked;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.YAxisComboBoxClick
  Description: responds to changes in the selected Y Axis (primary or 2nd)
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.YAxisComboBoxClick(Sender: TObject);
begin
{We define YAxisIndex to run from 1 to FAxisList.Count-1}
  pASP^.YAxisIndex := YAxisComboBox.ItemIndex+1;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.DeltaXNEditChange
  Description: responds to changes in DeltaX
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.DeltaXNEditChange(Sender: TObject);
begin
  pASP^.DeltaX := DeltaXNEdit.AsInteger;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.DeltaYNEditChange
  Description: repsonds to changes in DeltaY
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.DeltaYNEditChange(Sender: TObject);
begin
  pASP^.DeltaY := DeltaYNEdit.AsInteger;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.SymbolSizeNEditChange
  Description: repsonds to changes in SymbolSize
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.SymbolSizeNEditChange(Sender: TObject);
begin
  pASP^.SymbolSize := SymbolSizeNEdit.AsInteger;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.NameEditChange
  Description: repsonds to changes in the Series Name
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.NameEditChange(Sender: TObject);
begin
  SeriesNames.Strings[CurrentIndex] := NameEdit.Text;
end;

procedure TSeriesEditorForm.XDataIndependentCheckBoxClick(Sender: TObject);
begin
  pASP^.XDataIndependent := XDataIndependentCheckBox.Checked;
end;

procedure TSeriesEditorForm.PenColorEditChange(Sender: TObject);
begin
  pASP^.PenColor := PenColorEdit.Color;
  pASP^.BrushColor := Misc.GetPalerColor(pASP.PenColor, 70);
end;

procedure TSeriesEditorForm.BrushColorEditChange(Sender: TObject);
begin
  pASP^.BrushColor := BrushColorEdit.Color;
end;

procedure TSeriesEditorForm.ApplyBitBtnClick(Sender: TObject);
begin
  TPlot(ThePlot).ApplySeriesChange(Self);
end;


end.
