unit Datalist;

{$I Plot.inc}

{-----------------------------------------------------------------------------
The contents of this file are subject to the Q Public License
("QPL"); you may not use this file except in compliance
with the QPL. You may obtain a copy of the QPL from 
the file QPL.html in this distribution, derived from:

http://www.trolltech.com/products/download/freelicense/license.html

The QPL prohibits development of proprietary software. 
There is a Professional Version of this software available for this. 
Contact sales@chemware.hypermart.net for more information.

Software distributed under the QPL is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either expressed or implied. See the QPL for
the specific language governing rights and limitations under the QPL.

The Original Code is: PlotList.pas, released 12 September 2000.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@chemware.hypermart.net.

Last Modified: 04/18/2001
Current Version: 2.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind.

Purpose:
This unit contains the TSeriesList sub-component - that manages the data for
ALL Series for TPlot.

Known Issues:
-----------------------------------------------------------------------------}

interface

uses
  Classes, SysUtils,
{$IFDEF NO_MATH}NoMath,{$ELSE}Math,{$ENDIF}
{$IFDEF WINDOWS}
  Controls, Dialogs, Graphics, Windows,
{$ENDIF}
{$IFDEF WIN32}
  Controls, Dialogs, Graphics, Windows,
{$ENDIF}
{$IFDEF LINUX}
  Types,
  QControls, QDialogs, QGraphics,
{$ENDIF}

{$IFDEF FUNCTIONS}
  Parser10,
{$ENDIF}

  Axis, Data, Functons, Parser, Plotdefs, Misc, Titles;

type
{TSeriesList is a TList of Series that frees it's items.
 Use only descendents of TObject:}
  TSeriesList = class(TList)
  private
{The AxisList is created and managed in the Plot unit and TCustomPlot component.
 The specific axes are:
   0 .. X Axis
   1 .. Primary Y Axis
   2 .. Secondary Y Axis
   3 .. Tertiary Y Axis
   4 .. etc.}
    FAxisList: TList;
    FDataChanged: Boolean;
    FIgnoreChanges: Boolean;
    FXAxis: TAxis;
    FYAxis: TAxis;


    FOnStyleChange: TNotifyEvent;
    FOnDataChange: TNotifyEvent;

    LastSavedPoint: Integer;
    PlotBorder: TRect;
    NoPieRows: Integer;
    
    function GetXmin: Single;
    function GetXmax: Single;
    function GetYmin: Single;
    function GetYmax: Single;
    function GetZmin: Single;
    function GetZmax: Single;
    function GetYErrorMin: Single;
    function GetYErrorMax: Single;

    function GetMaxNoPts: Integer;
    function GetMinNoPts: Integer;
    function GetTotalNoPts: Integer;
  protected
    procedure StyleChange(Sender: TObject);
    procedure DataChange(Sender: TObject);
    procedure DoStyleChange; virtual;
    procedure DoDataChange; virtual;

  public
    property DataChanged: Boolean read FDataChanged write FDataChanged stored FALSE;
{Has the data any series changed ?}
    property IgnoreChanges: Boolean read FIgnoreChanges write FIgnoreChanges;
{Shall we ignore Change and DataChange events ?}    
    property TotalNoPts: Integer read GetTotalNoPts stored FALSE;
{The total number of points in all series.}
    property MaxNoPts: Integer read GetMaxNoPts stored FALSE;
{The number of points in the largest series.}
    property MinNoPts: Integer read GetMinNoPts stored FALSE;
{The number of points in the smallest series.}
    property Xmin: Single read GetXmin stored FALSE;
{The minimum X value of ALL Series.}
    property Xmax: Single read GetXmax stored FALSE;
{The maximum X value of ALL Series.}

    property Ymin: Single read GetYmin;
{The minimum Y value of ALL Series connected to the PRIMARY Y Axis.}
    property Ymax: Single read GetYmax;
{The maximum Y value of ALL Series connected to the PRIMARY Y Axis.}
    property Zmin: Single read GetZmin;
{The minimum Z value of ALL Series.}
    property Zmax: Single read GetZmax;
{The maximum Z value of ALL Series.}
    property YErrorMin: Single read GetYErrorMin;
{The minimum Y value of ALL Series plus their error.}
    property YErrorMax: Single read GetYErrorMax;
{The maximum Y value of ALL Series plus their error.}

    property OnStyleChange: TNotifyEvent read FOnStyleChange write FOnStyleChange;
{This notifies the owner (usually TPlot) of a change in style of this series.}
    property OnDataChange: TNotifyEvent read FOnDataChange write FOnDataChange;
{This notifies the owner (usually TPlot) of a change in the data of this series.}
{}
{NOTE: D1 does not allow published properties for TList descendants.}

    Constructor Create(AxisListPtr: TList); virtual;
{Saves the Axes List and initializes LastSavedPoint.}    
    destructor Destroy; override;

    function Add(XSeriesIndex: Integer): Integer; 
{This adds a new, empty series to the list.}
{}
{Like its ancestor function, and its relatives AddInternal and AddExternal,
 Add returns the index of the new item, where the first item in the list has
 an index of 0.}
    function AddExternal(XPointer, YPointer: pSingleArray; NumberOfPoints: Integer): Integer;
{This adds a new, empty series to the list, and sets its data to point to the
 external XPointer, YPointer data.}
{}
{Like its ancestor function, and its relatives AddInternal and Add,
 AddExternal returns the index of the new item, where the first item in the list has
 an index of 0.}
    function AddInternal(XPointer, YPointer: pSingleArray; NumberOfPoints: Integer): Integer;
{This adds a new, empty series to the list, and copies the data from the
 XPointer, YPointer data.}
{}
{Like its ancestor function, and its relatives Add and AddExternal,
 AddInternal returns the index of the new item, where the first item in the list has
 an index of 0.}

    procedure ClearSeries;
{This deletes all Series from the list, freeing the series.}
{}
{Note that the Clear does not exist in the D1-3 version, because TList.Clear
 is a static method and cannot be overridden. This is why we created ClearSeries.}

    function CloneSeries(TheSeries: Integer): Integer;
{This adds a new, empty series to the list, copies the data and properties from
 TheSeries into the new clone, and changes the color and Y Displacement.}
{}
{CloneSeries returns TRUE if successful.}
    procedure DeleteSeries(Index: Integer; Ask: Boolean);
{This deletes TheSeries from the list.}

    function ParseData(TheData: TStringList; TheHelpFile: String): Boolean;
{This parses imported or pasted data and adds it to the SeriesList.}
    function ConvertTextData(ColCount, SeriesCount, FirstLine: Integer;
      Delimiter: String; TheData: TStringList; SeriesInfo: pSeriesInfoArray): Boolean;
{This takes a parsed stringlist converts it to numerical data.}
    function ConvertXYZData(FirstLine: Integer;
          Delimiter: String; InfoGridRows: TStrings; TheData: TStringList): Boolean;
{This takes an UNparsed stringlist with XYZ values and converts it to numerical data.}
    function ConvertBinaryData(ColCount, SeriesCount: Integer;
      TheStream: TMemoryStream; SeriesInfo: pSeriesInfoArray): Boolean;
{This takes a parsed stringlist converts it to numerical data.}

{$IFDEF FUNCTIONS}
    function FunctionSeries: Integer;
{This creates a new series which is a function of the existing series.}
{$ENDIF}

    procedure DataAsHTMLTable(var TheData: TStringList);
{This returns all the data in HTML format as a StringList.}

    procedure GetStream(AsText: Boolean; Delimiter: Char; var TheStream: TMemoryStream);
{This returns all the data as a MemoryStream.}
    procedure GetSubHeaderStream(Delimiter: Char; TheStream: TMemoryStream);
{This returns all the SubHeader data as a MemoryStream.}
    procedure GetBinaryStream(Start, Finish: Integer;
      TheStream: TMemoryStream);
    procedure GetTextStream(Delimiter: Char; Start, Finish: Integer;
      TheStream: TMemoryStream);
    procedure AppendStream(AsText: Boolean; Delimiter: Char; TheStream: TMemoryStreamEx);
{This returns the data collected since LastSavedPoint as a MemoryStream.}
    function LoadFromStream(AStream: TMemoryStream; var AsText: Boolean): Boolean;
{Opens data, parses it, fires the OnHeader event, and runs ConvertTextData,
 or decides to run it through ParseData instead}
    procedure Draw(ACanvas: TCanvas; XYBridge: Integer);
{This draws all the series on the given canvas.}
    procedure DrawError(ACanvas: TCanvas);
{Extended Drawing procedure for series with errorbars.}
    procedure DrawBubble(ACanvas: TCanvas; BubbleSize: Integer);
{Extended Drawing procedure for Bubble plots.}
    procedure DrawMultiple(ACanvas: TCanvas; Multiplicity: Byte; MultiplePen: TPen; MultiJoin1, MultiJoin2: Integer);
{Extended Drawing procedure for linking multiple series.}
    procedure DrawColumns(ACanvas: TCanvas; ColumnGap: TPercent);
{This draws all the series on the given canvas in columns.}
    procedure DrawStack(ACanvas: TCanvas; ColumnGap: TPercent);
{This draws all the series on the given canvas in stacked (summed) columns.}
    procedure DrawNormStack(ACanvas: TCanvas; ColumnGap: TPercent);
{This draws all the series on the given canvas in normalized columns.}
    procedure DrawPie(ACanvas: TCanvas; Border: TBorder; NoRows: Integer);
{This draws all the series on the given canvas as Pie graphs.}
    procedure DrawPolar(ACanvas: TCanvas; PolarRange: Single);
{This draws all the series on the given canvas as a Polar graph.}
    procedure Draw3DWire(ACanvas: TCanvas; ZAxis: TAngleAxis; ZLink: Boolean);
{This draws all the series on the given canvas.}
    procedure Draw3DContour(ACanvas: TCanvas; ZAxis: TAngleAxis; ContourDetail: TContourDetail);
{This draws all the series on the given canvas.}
    procedure DrawContour(ACanvas: TCanvas; ContourDetail: TContourDetail);
{This draws all the series on the given canvas.}
    procedure DrawColorScale(ACanvas: TCanvas; TheMin, Span: Single; TheContourDetail: TContourDetail);

    procedure DrawHistory(ACanvas: TCanvas; HistoryX: Single);
{This draws all the series on the given canvas, in a History mode.}
    procedure DrawHistoryMultiple(ACanvas: TCanvas; Multiplicity: Byte);
{Extended Drawing procedure for linking multiple series in History mode.}
    function GetNearestPoint(
      ThePlotType: TPlotType;
      ColumnGap,
      iX, iY: Integer;
      var TheSeries: Integer;
      var MinDistance: Single;
      var pSeries: TSeries): Integer;
{This returns the Index of the nearest point, and sets TheSeries it belongs to}
  {published}
    function GetSeriesOfZ(ZValue: Single): TSeries;
  end;

implementation

uses
  Plot;

const
  PIE_SIZE = 0.8;

{TSeriesList Constructor and Destructor:---------------------------------------}
{------------------------------------------------------------------------------
  Constructor: TSeriesList.Create
  Description: standard Constructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: saves the Axes List and initializes LastSavedPoint
 Known Issues:
 ------------------------------------------------------------------------------}
Constructor TSeriesList.Create(AxisListPtr: TList);
begin
  inherited Create;
  FAxisList := AxisListPtr;
  FXAxis := TAxis(FAxisList[0]);
  FYAxis := TAxis(FAxisList[1]);
  LastSavedPoint := 0;
end;

{------------------------------------------------------------------------------
   Destructor: TSeriesList.Destroy
  Description: standard Destructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Frees the Axes and events
 Known Issues:
 ------------------------------------------------------------------------------}
destructor TSeriesList.Destroy;
begin
  FOnStyleChange := nil;
  FOnDataChange := nil;
{NOTE: in D1-D3, Clear is static and cannot be overridden, so we had to
 add a ClearSeries for them:}
  ClearSeries;
  inherited Destroy;
end;

{TSeriesList Set procedures ---------------------------------------------------}

{TSeriesList Get functions ----------------------------------------------------}

{------------------------------------------------------------------------------
     Function: TSeriesList.GetXmin
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the value of the XMin Property over ALL Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetXmin: Single;
var
  i: Integer;
  Value: Single;
begin
  Value := 1.e38;
  for i := 0 to Count-1 do
  begin
    if (Value > TSeries(Items[i]).XMin) then
      Value := TSeries(Items[i]).XMin;
  end; {loop over series}
  GetXmin := Value;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetXMax
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the value of the XMax Property over ALL Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetXmax: Single;
var
  i: Integer;
  Value: Single;
begin
  Value := -1.e38;
  for i := 0 to Count-1 do
  begin
    if (Value < TSeries(Items[i]).XMax) then
      Value := TSeries(Items[i]).XMax;
  end; {loop over series}
  GetXmax := Value;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetYMin
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the value of the YMin Property over ALL Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetYMin: Single;
var
  i: Integer;
  Value: Single;
begin
  Value := 1.e38;
  for i := 0 to Count-1 do
  begin
    if (Value > TSeries(Items[i]).YMin) then
      Value := TSeries(Items[i]).YMin;
  end; {loop over series}
  GetYMin := Value;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetYMax
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the value of the YMax Property over ALL Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetYMax: Single;
var
  i: Integer;
  Value: Single;
begin
  Value := -1.e38;
  for i := 0 to Count-1 do
  begin
    if (Value < TSeries(Items[i]).YMax) then
      Value := TSeries(Items[i]).YMax;
  end; {loop over series}
  GetYMax := Value;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetZMin
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the value of the ZMin Property over ALL Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetZMin: Single;
var
  i: Integer;
  Value: Single;
begin
  Value := 1.e38;
  for i := 0 to Count-1 do
  begin
    if (Value > TSeries(Items[i]).ZData) then
      Value := TSeries(Items[i]).ZData;
  end; {loop over series}
  GetZMin := Value;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetZMax
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the value of the ZMax Property over ALL Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetZMax: Single;
var
  i: Integer;
  Value: Single;
begin
  Value := -1.e38;
  for i := 0 to Count-1 do
  begin
    if (Value < TSeries(Items[i]).ZData) then
      Value := TSeries(Items[i]).ZData;
  end; {loop over series}
  GetZMax := Value;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetYErrorMin
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the value of the YErrorMin Property over ALL Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetYErrorMin: Single;
var
  i,
  iSeries: Integer;
  NewValue,
  Value: Single;
  pSeries,
  pErrorSeries: TSeries;
begin
  Value := 1.e38;
  iSeries := 0;
  while (iSeries <= Self.Count-2) do
  begin
    pSeries := TSeries(Items[iSeries]);
    pErrorSeries := TSeries(Items[iSeries+1]);
    for i := 0 to Min(pSeries.NoPts, pErrorSeries.NoPts)-1 do
    begin
      NewValue := pSeries.YData^[i] - pErrorSeries.YData^[i];
      if (Value > NewValue) then
        Value := NewValue;
    end;
    Inc(iSeries, 2)
  end; {loop over series}
  GetYErrorMin := Value;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetYErrorMax
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the value of the YErrorMax Property over ALL Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetYErrorMax: Single;
var
  i,
  iSeries: Integer;
  NewValue,
  Value: Single;
  pSeries,
  pErrorSeries: TSeries;
begin
  Value := 1.e-38;
  iSeries := 0;
  while (iSeries <= Self.Count-2) do
  begin
    pSeries := TSeries(Items[iSeries]);
    pErrorSeries := TSeries(Items[iSeries+1]);
    for i := 0 to Min(pSeries.NoPts, pErrorSeries.NoPts)-1 do
    begin
      NewValue := pSeries.YData^[i] + pErrorSeries.YData^[i];
      if (Value < NewValue) then
        Value := NewValue;
    end;
    Inc(iSeries, 2)
  end; {loop over series}
  GetYErrorMax := Value;
end;

{TSeriesList Memory and list management ---------------------------------------}
{------------------------------------------------------------------------------
     Function: TSeriesList.Add
  Description: Adds a new Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: creates, initializes and adds a new series
 Return Value: the Index of the new Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.Add(XSeriesIndex: Integer): Integer;
var
  Item,
  XSeries: TSeries;
begin
  if (XSeriesIndex > Count-1) then raise
    EComponentError.Create('TSeriesList.Add: invalid X Series Index !');

  if (XSeriesIndex >= 0) then
    XSeries := TSeries(Items[XSeriesIndex])
   else
    XSeries := nil;

  Item := TSeries.Create(Count, FAxisList, XSeries);
  Item.OnStyleChange := Self.StyleChange;
  Item.OnDataChange := Self.DataChange;

  Add := inherited Add(Item);

  DoDataChange;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.AddExternal
  Description: Adds a new Series that is externally maintained
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 01/25/2001 by Mat Ballard
      Purpose: creates, initializes and adds a new series
 Return Value: the Index of the new Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.AddExternal(XPointer, YPointer: pSingleArray; NumberOfPoints: Integer): Integer;
var
  Item: TSeries;
begin
{set to failure in case Item.AddData fails:}
  AddExternal := -1;

  Item := TSeries.Create(Count, FAxisList, nil);
  Item.OnDataChange := Self.DataChange;

  if (Item.PointToData(XPointer, YPointer, NumberOfPoints)) then
  begin
{Success:}
    AddExternal := inherited Add(Item);
  end
  else
  begin
{Failure:}
    Item.Free;
  end;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.AddInternal
  Description: Adds a new Series from external data (copies)
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 01/25/2001 by Mat Ballard
      Purpose: creates, initializes and adds a new series
 Return Value: the Index of the new Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.AddInternal(XPointer, YPointer: pSingleArray; NumberOfPoints: Integer): Integer;
var
  Item: TSeries;
begin
{set to failure in case Item.AddData fails:}
  AddInternal := -1;

  Item := TSeries.Create(Count, FAxisList, nil);
  Item.OnDataChange := Self.DataChange;
  if (Item.AddData(XPointer, YPointer, NumberOfPoints)) then
  begin
{Success:}
    AddInternal := inherited Add(Item);
  end
  else
  begin
{Failure:}
    Item.Free;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.ClearSeries
  Description: frees and deletes all the Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Series management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.ClearSeries;
var
  i: Integer;
  pSeries: TSeries;
begin
  if (Count = 0) then exit;
  
  for i := Count-1 downto 0 do
  begin
    pSeries := TSeries(Items[i]);
    Delete(i);
    pSeries.Free;
  end;
  DoDataChange;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DeleteSeries
  Description: frees and deletes one particular Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Series management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DeleteSeries(Index: Integer; Ask: Boolean);
var
  pSeries: TSeries;
begin
  pSeries := TSeries(Items[Index]);
  if (Ask) then
    Ask := not (MessageDlg(
{$IFDEF LINUX}
    'Delete Series',
{$ENDIF}
    'Do you really want to delete ' + pSeries.Name + ' ?',
    mtWarning, [mbYes,mbNo], 0) = mrYes);

  if (not Ask) then
  begin
    Delete(Index);
    pSeries.Free;
    DoDataChange;
  end;
end;

{$IFDEF FUNCTIONS}
{------------------------------------------------------------------------------
    Procedure: TSeriesList.FunctionSeries
  Description: Creates a new series which is a function of existing series
       Author: Mat Ballard
 Date created: 04/03/2001
Date modified: 04/03/2001 by Mat Ballard
      Purpose: data manipulation
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.FunctionSeries: Integer;
{This creates a new series which is a function of the existing series.}
var
  i, j,
  TheResult: Integer;
  FunctionsForm: TFunctionsForm;
  TheParser: TParser;
  TheExpression: String;
  TheData: array [0..99] of PParserFloat;
begin
  TheResult := -1;
  if (Self.Count > 0) then
  begin
    FunctionsForm := TFunctionsForm.Create(nil);
    for i := 0 to Self.Count-2 do
      FunctionsForm.FunctionMemo.Lines.Add(Format('Series%d', [i]) + ' + ');
    FunctionsForm.FunctionMemo.Lines.Add(Format('Series%d', [Self.Count-1]));
    FunctionsForm.SeriesLabel.Caption := Format('Series%d :=', [Self.Count]);

    if (mrOK = FunctionsForm.ShowModal) then
    begin
      TheResult := Self.Add(0);
      TheParser := TParser.Create(nil);
      TheExpression := '';
{read the equation:}
      for i := 0 to FunctionsForm.FunctionMemo.Lines.Count-1 do
        TheExpression := TheExpression + FunctionsForm.FunctionMemo.Lines[i];
      TheExpression := Misc.CleanString(TheExpression, ' ');

{try to run the Parser:}
      try
{Set up the variables:}
        for j := 0 to Self.Count-2 do
          TheData[j] := TheParser.SetVariable(Format('SERIES%d', [j]), 0);
{Set up equation:}
        TheParser.Expression := TheExpression;
{Loop over all points in Series[0]:}
        for i := 0 to TSeries(Self.Items[0]).NoPts-1 do
        begin
{add the Series variables to the Parser, or reset their values:}
          for j := 0 to Self.Count-2 do
            TheData[j]^ := TSeries(Self.Items[j]).YData^[i];
          TSeries(Self.Items[TheResult]).AddPoint(-1, TheParser.Value, FALSE, TRUE);
        end;
      except
        Self.DeleteSeries(TheResult, FALSE);
        TheResult := -1;
      end;
      TheParser.Free;
    end;
    FunctionsForm.Free;
  end;
  FunctionSeries := TheResult;
end;

{$ENDIF}

{TSeriesList the movie ! ------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TSeriesList.Draw
  Description: standard Drawing procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: draws all the Series on a given canvas
 Known Issues: where do we put the drawing and the "GetNearestPoint" methods ?

    ptXY         needs FXAxis, FYAxis - done in Data
    ptError      needs FXAxis, FYAxis, AND Index - done in Datalist
    ptMultiple   needs FXAxis, FYAxis - done in Data
    ptColumn     needs ColumnGap, Index, FXAxis, FYAxis - done in Datalist
    ptStack      needs ALL Series, ColumnGap, Index, FXAxis, FYAxis - done in Datalist
    ptNormStack  needs ALL Series, ColumnGap, Index, FXAxis, FYAxis - done in Datalist
    ptPie        needs bounding rectangle - done in Data
    ptPolar      unknown as yet
    pt3D         needs ALL Series, FXAxis, FYAxis, AND FZAxis - done in Serlist

    There seems to be no simple solution to this: some methods have to be, or
    are best in this unit; others, according to the principles:
        a. bury as deep as possible;
        b. each series _SHOULD_ know how to draw itself;
    would seem better to be in Data.
 ------------------------------------------------------------------------------}
procedure TSeriesList.Draw(ACanvas: TCanvas; XYBridge: Integer);
var
  i: Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.Draw: ACanvas is nil !');
{$ENDIF}

  for i := 0 to Count-1 do
  begin
    TSeries(Items[i]).Draw(ACanvas, XYBridge);
  end; {loop over series}
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DrawColumns
  Description: standard Drawing procedure for Columns
       Author: Mat Ballard
 Date created: 09/25/2000
Date modified: 09/25/2000 by Mat Ballard
      Purpose: draws all the Series on a given canvas in Columns
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DrawColumns(ACanvas: TCanvas; ColumnGap: TPercent);
var
  i,
  j,
  iX,
  iXp1, {iX plus 1: the next X ordinate;}
  iXStart,
  iXEnd,
  iY: Integer;
  dX: Single;
  pSeries,
  pSeries0: TSeries;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.DrawColumns: ACanvas is nil !');
{$ENDIF}

  if (Count = 0) then
    exit;

{all columns are plotted against the X Axis and Data of the first series,
 otherwise chaos can result.}
  pSeries0 := TSeries(Items[0]);

{loop over every series:}
  for i := 0 to Count-1 do
  begin
    pSeries := TSeries(Items[i]);
    ACanvas.Pen.Assign(pSeries.Pen);
    ACanvas.Brush.Assign(pSeries.Brush);
    if ((pSeries.NoPts > 0) and
        (pSeries.Visible)) then
    begin
      iXp1 := pSeries0.XAxis.FofX(pSeries0.XData^[0]);
  {loop over every point in each series:}
      for j := 0 to pSeries.NoPts-2 do
      begin
        iX := iXp1;
        iXp1 := pSeries0.XAxis.FofX(pSeries0.XData^[j+1]);
        dX := ((100 - ColumnGap) / (100 * Count) * (iXp1-iX));
        iXStart := iX + Round(i*dX);
        iXEnd := iXStart + Round(dX);
        iY := pSeries.YAxis.FofY(pSeries.YData^[j]);
        ACanvas.Rectangle(iXStart, iY, iXEnd, pSeries.YAxis.Bottom);
      end; {for}
    end; {NoPts > 0}
  end; {loop over series}
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DrawStack
  Description: standard Drawing procedure for Stacked Columns
       Author: Mat Ballard
 Date created: 09/25/2000
Date modified: 09/25/2000 by Mat Ballard
      Purpose: draws all the Series on a given canvas in Stacked Columns
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DrawStack(ACanvas: TCanvas; ColumnGap: TPercent);
var
  i,
  j,
  iX,
  iXp1, {iX plus 1: the next X ordinate;}
  iXEnd,
  iY,
  iYNeg,
  iYTop,
  iYNegBottom: Integer;
  NegSum, Sum: Single;
  pSeries0: TSeries;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.DrawColumns: ACanvas is nil !');
{$ENDIF}

  if (Count = 0) then
    exit;

{all columns are plotted against the X and Y Axis and X Data of the first series,
 otherwise chaos can result.}
  pSeries0 := TSeries(Items[0]);
  if (pSeries0.NoPts = 0) then exit;

  iXp1 := pSeries0.XAxis.FofX(pSeries0.XData^[0]);
{loop over every point:}
  for j := 0 to pSeries0.NoPts-2 do
  begin
    iX := iXp1;
    iXp1 := pSeries0.XAxis.FofX(pSeries0.XData^[j+1]);
    iXEnd := iXp1 - Round((ColumnGap) * (iXp1-iX) / 100);
    iYTop := pSeries0.XAxis.MidY;
    iYNegBottom := iYTop;
    Sum := 0;
    NegSum := 0;
  {loop over every series:}
    for i := 0 to Self.Count-1 do
    begin
      if ((TSeries(Items[i]).NoPts > j) and
          (TSeries(Items[i]).Visible)) then
      begin
        ACanvas.Pen.Assign(TSeries(Items[i]).Pen);
        ACanvas.Brush.Assign(TSeries(Items[i]).Brush);
        if (TSeries(Items[i]).YData^[j] >= 0) then
        begin
          Sum := Sum + TSeries(Items[i]).YData^[j];
          iY := iYTop;
          iYTop := pSeries0.YAxis.FofY(Sum);
          ACanvas.Rectangle(iX, iY, iXEnd, iYTop);
        end
        else
        begin
          NegSum := NegSum + TSeries(Items[i]).YData^[j];
          iYNeg := iYNegBottom;
          iYNegBottom := pSeries0.YAxis.FofY(NegSum);
          ACanvas.Rectangle(iX, iYNeg, iXEnd, iYNegBottom);
        end;
      end; {for}
    end; {NoPts > 0}
  end; {loop over series}
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DrawNormStack
  Description: standard Drawing procedure for Normalized Stacked Columns
       Author: Mat Ballard
 Date created: 09/25/2000
Date modified: 09/25/2000 by Mat Ballard
      Purpose: draws all the Series on a given canvas in Normalized (percentage) Stacked Columns
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DrawNormStack(ACanvas: TCanvas; ColumnGap: TPercent);
var
  i,
  j,
  iX,
  iXp1, {iX plus 1: the next X ordinate;}
  iXEnd,
  iY,
  iYNeg,
  iYTop,
  iYNegBottom: Integer;
  Sum, NegSum,
  Total, NegTotal: Single;
  pSeries0: TSeries;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.DrawColumns: ACanvas is nil !');
{$ENDIF}

  if (Count = 0) then
    exit;

{all columns are plotted against the X and Y Axis and X Data of the first series,
 otherwise chaos can result.}
  pSeries0 := TSeries(Items[0]);
  if (pSeries0.NoPts = 0) then exit;

  iXp1 := pSeries0.XAxis.FofX(pSeries0.XData^[0]);
{loop over every point:}
  for j := 0 to pSeries0.NoPts-2 do
  begin
    iX := iXp1;
    iXp1 := pSeries0.XAxis.FofX(pSeries0.XData^[j+1]);
    iXEnd := iXp1 - Round((ColumnGap) * (iXp1-iX) / 100);
    iYTop := pSeries0.XAxis.MidY;
    iYNegBottom := iYTop;
    Sum := 0;
    NegSum := 0;
    Total := 0;
    NegTotal := 0;
{count every series:}
    for i := 0 to Count-1 do
    begin
      if (TSeries(Items[i]).NoPts > j) then
      begin
        if (TSeries(Items[i]).YData^[j] >= 0) then
          Total := Total + TSeries(Items[i]).YData^[j]
         else
          NegTotal := NegTotal + TSeries(Items[i]).YData^[j];
      end; {NoPts > j}
    end; {count every series}
{prepare for conversion of data to percent:}
    Total := Total / 100;
    NegTotal := - NegTotal / 100;

{loop over every series:}
    for i := 0 to Count-1 do
    begin
      if (TSeries(Items[i]).NoPts > j) then
      begin
        ACanvas.Pen.Assign(TSeries(Items[i]).Pen);
        ACanvas.Brush.Assign(TSeries(Items[i]).Brush);
        if (TSeries(Items[i]).YData^[j] >= 0) then
        begin
          Sum := Sum + (TSeries(Items[i]).YData^[j] / Total);
          iY := iYTop;
          iYTop := pSeries0.YAxis.FofY(Sum);
          if (TSeries(Items[i]).Visible) then
            ACanvas.Rectangle(iX, iY, iXEnd, iYTop);
        end
        else
        begin
          NegSum := NegSum + (TSeries(Items[i]).YData^[j] / NegTotal);
          iYNeg := iYNegBottom;
          iYNegBottom := pSeries0.YAxis.FofY(NegSum);
          if (TSeries(Items[i]).Visible) then
            ACanvas.Rectangle(iX, iYNeg, iXEnd, iYNegBottom);
        end;
      end; {for}
    end; {NoPts > 0}
  end; {loop over series}
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DrawPie
  Description: standard Drawing procedure for Normalized Stacked Columns
       Author: Mat Ballard
 Date created: 12/20/2000
Date modified: 12/20/2000 by Mat Ballard
      Purpose: This draws all the series on the given canvas as Pie graphs.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DrawPie(ACanvas: TCanvas; Border: TBorder; NoRows: Integer);
var
  CellWidth,
  CellHeight,
  iSeries,
  iCol,
  jRow,
  NoPieCols,
  PieLeft,
  PieTop,
  PieWidth,
  PieHeight: Integer;
begin
  if ((Count = 0) or (NoRows = 0)) then
    exit;

{remember the number of rows:}
  NoPieRows := NoRows;
  NoPieCols := Trunc(0.99 + Self.Count / NoPieRows);
{remember the border:}
  PlotBorder.Left := Border.Left;
  PlotBorder.Top := Border.Top;
  PlotBorder.Right := Border.Right;
  PlotBorder.Bottom := Border.Bottom;

{each Pie sits in a cell:}
  CellWidth := (PlotBorder.Right - PlotBorder.Left) div NoPieCols;
  CellHeight := (PlotBorder.Bottom - PlotBorder.Top) div NoPieRows;
{... but does not occupy the entire cell:}
  PieWidth := Round(PIE_SIZE * CellWidth);
  PieHeight := Round(PIE_SIZE * CellHeight);
  if (PieHeight > PieWidth) then
    PieHeight := PieWidth;

  iSeries := 0;
  for iCol := 0 to NoPieCols-1 do
  begin
    for jRow := 0 to NoPieRows-1 do
    begin
      if (iSeries >= Count) then break;
{indent the (left, top) a bit:}
      PieLeft := PlotBorder.Left + iCol * CellWidth +
        (CellWidth-PieWidth) div 2;
      PieTop := PlotBorder.Top + jRow * CellHeight +
        (CellHeight-PieHeight) div 2;
{draw it:}
      TSeries(Self.Items[iSeries]).DrawPie(
        ACanvas,
        PieLeft,
        PieTop,
        PieWidth,
        PieHeight);
      Inc(iSeries);
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DrawPolar
  Description: standard Drawing procedure for Normalized Stacked Columns
       Author: Mat Ballard
 Date created: 12/20/2000
Date modified: 12/20/2000 by Mat Ballard
      Purpose: This draws all the series on the given canvas as a Polar graph.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DrawPolar(ACanvas: TCanvas; PolarRange: Single);
var
  i: Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.Draw: ACanvas is nil !');
{$ENDIF}

  if (PolarRange = 0) then
    PolarRange := TWO_PI;
  for i := 0 to Self.Count-1 do
  begin
    TSeries(Items[i]).DrawPolar(ACanvas, PolarRange);
  end; {loop over series}
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.Draw3DWire
  Description: standard Drawing procedure for 3D graphs
       Author: Mat Ballard
 Date created: 01/24/2001
Date modified: 01/24/2001 by Mat Ballard
      Purpose: This draws all the series as a 3D graph
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.Draw3DWire(ACanvas: TCanvas; ZAxis: TAngleAxis; ZLink: Boolean);
var
  iSeries,
  j: Integer;
  Pt00, Pt01, Pt10, Pt11: TPoint;
{These correspond to Point[Series #, Point #]}
  ZShift0,
  ZShift1: TPoint;
  pSeries0,
  pSeries1: TSeries;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.Draw3D: ACanvas is nil !');
  Assert(ZAxis <> nil, 'TSeriesList.Draw3D: ZAxis is nil !');
{$ENDIF}

{Draw each individual series:}
  for iSeries := 0 to Self.Count-1 do
  begin
    ACanvas.Pen.Assign(TSeries(Items[iSeries]).Pen);
    pSeries0 := TSeries(Items[iSeries]);
    Pt10.x := Low(Integer);
    if (pSeries0.NoPts > 0) then
    begin
      ZShift0 := ZAxis.dFofZ(pSeries0.ZData);
      Pt00.x := pSeries0.XAxis.FofX(pSeries0.XData^[0])+ ZShift0.x;
      Pt00.y := pSeries0.YAxis.FofY(pSeries0.YData^[0]) + ZShift0.y;
      if ((ZLink) and (iSeries < Count-1)) then
      begin
        pSeries1 := TSeries(Items[iSeries+1]);
        if (pSeries1.NoPts > 0) then
        begin
          ZShift1 := ZAxis.dFofZ(pSeries1.ZData);
          Pt10.x := pSeries1.XAxis.FofX(pSeries1.XData^[0])+ ZShift1.x;
          Pt10.y := pSeries1.YAxis.FofY(pSeries1.YData^[0]) + ZShift1.y;
        end;
      end;

      for j := 1 to pSeries0.NoPts-1 do
      begin
        Pt01.x := pSeries0.XAxis.FofX(pSeries0.XData^[j]) + ZShift0.x;
        Pt01.y := pSeries0.YAxis.FofY(pSeries0.YData^[j]) + ZShift0.y;
        ACanvas.MoveTo(Pt00.x, Pt00.y);
        ACanvas.LineTo(Pt01.x, Pt01.y);
        if ((ZLink) and (iSeries < Count-1)) then
        begin
{Oh yes it was: Delphi isn't smart enough to see that if we got here,
 then it WAS initialized. Same applies to Pt10.x, Pt10.y.}        
          if (pSeries1.NoPts > 0) then
          begin
            Pt11.x := pSeries1.XAxis.FofX(pSeries1.XData^[j]) + ZShift1.x;
            Pt11.y := pSeries1.YAxis.FofY(pSeries1.YData^[j]) + ZShift1.y;
            ACanvas.MoveTo(Pt10.x, Pt10.y);
            ACanvas.LineTo(Pt00.x, Pt00.y);
{no triangles please, we're British:            
            ACanvas.LineTo(Pt11.x, Pt11.y);}
            Pt10.x := Pt11.x;
            Pt10.y := Pt11.y;
          end;
        end; {not the final series}
        Pt00.x := Pt01.x;
        Pt00.y := Pt01.y;
      end; {loop over points}
      if (Pt10.x <> Low(Integer)) then
      begin
        ACanvas.MoveTo(Pt00.x, Pt00.y);
        ACanvas.LineTo(Pt10.x, Pt10.y);
      end;
    end; {NoPts}
  end; {over every series}
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.Draw3DContour
  Description: standard Drawing procedure for 3D contour graphs
       Author: Mat Ballard
 Date created: 03/06/2001
Date modified: 03/06/2001 by Mat Ballard
      Purpose: This draws all the series as a 3D contour graph
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.Draw3DContour(ACanvas: TCanvas; ZAxis: TAngleAxis; ContourDetail: TContourDetail);
var
  iSeries,
  j: Integer;
  Pt00, Pt01, Pt10, Pt11, PtCentre,
  P1, P2, P3, P4: TPoint;
  Y00, Y01, Y10, Y11, YCentre,
  ZFraction,
  TheMin,
  Span: Single;
{These correspond to Point[Series #, Point #]}
  ZShift0, ZShift1, ZShiftCentre: TPoint;
  pSeries0,
  pSeries1: TSeries;
  iX, iY: Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.Draw3DContour: ACanvas is nil !');
  //Assert(ZAxis <> nil, 'TSeriesList.Draw3DContour: ZAxis is nil !');
{$ENDIF}
  if (ZAxis = nil) then exit;

  TheMin := Self.Ymin;
  Span := Self.YMax - TheMin;
  ACanvas.Brush.Style := bsSolid;
  //ACanvas.Brush.Bitmap := nil;

{Draw each individual series:}
  for iSeries := 0 to Self.Count-1 do
  begin
    pSeries0 := TSeries(Items[iSeries]);
    if (pSeries0.NoPts > 0) then
    begin
      Pt00.x := pSeries0.XAxis.FofX(pSeries0.XData^[0]);
      Y00 := pSeries0.YData^[0];
      Pt00.y := pSeries0.YAxis.FofY(Y00);
      //Z00 := pSeries0.ZData;
      ZShift0 := ZAxis.dFofZ(pSeries0.ZData);
      if (iSeries < Count-1) then
      begin
        pSeries1 := TSeries(Items[iSeries+1]);
        if (pSeries1.NoPts > 0) then
        begin
          Pt10.x := pSeries1.XAxis.FofX(pSeries1.XData^[0]);
          Y10 := pSeries1.YData^[0];
          Pt10.y := pSeries1.YAxis.FofY(Y10);
          //Z10 := pSeries0.ZData;
          ZShift1 := ZAxis.dFofZ(pSeries1.ZData);
        end;
      end;

      for j := 1 to pSeries0.NoPts-1 do
      begin
        Pt01.x := pSeries0.XAxis.FofX(pSeries0.XData^[j]);
        Y01 := pSeries0.YData^[j];
        Pt01.y := pSeries0.YAxis.FofY(Y01);
        //Z01 := pSeries0.ZData;
        if (iSeries < Count-1) then
        begin
          if (pSeries1.NoPts > 0) then
          begin
            Pt11.x := pSeries1.XAxis.FofX(pSeries1.XData^[j]);
            Y11 := pSeries1.YData^[j];
            Pt11.y := pSeries1.YAxis.FofY(Y11);
            //Z01 := pSeries1.ZData;
{Calculate the centrum:}
            PtCentre.x := (Pt00.x + Pt01.x + Pt10.x + Pt11.x) div 4;
            PtCentre.y := (Pt00.y + Pt01.y + Pt10.y + Pt11.y) div 4;
{Oh yes it was: see Delphi comment above}
            YCentre := (Y00 + Y01 + Y10 + Y11) / 4;
            ZShiftCentre := ZAxis.dFofZ((pSeries0.ZData + pSeries1.ZData) /2);

{just how detailed will the plot be ?}
            case ContourDetail of
              cdLow:
                begin
{No triangles:}
                  P1.x := Pt00.x+ZShift0.x;
                  P1.y := Pt00.y+ZShift0.y;
                  P2.x := Pt10.x+ZShift1.x;
                  P2.y := Pt10.y+ZShift1.y;
                  P3.x := Pt11.x+ZShift1.x;
                  P3.y := Pt11.y+ZShift1.y;
                  P4.x := Pt01.x+ZShift0.x;
                  P4.y := Pt01.y+ZShift0.y;
                  ZFraction := (YCentre - TheMin) / Span;
                  ACanvas.Pen.Color := clBlack;
                  ACanvas.Brush.Color := Misc.Rainbow(ZFraction);
                  ACanvas.Polygon([P1, P2, P3, P4]);
                end;

              cdMedium:
                begin
  {Left triangle:}
                  P1.x := Pt00.x+ZShift0.x;
                  P1.y := Pt00.y+ZShift0.y;
                  P2.x := Pt10.x+ZShift1.x;
                  P2.y := Pt10.y+ZShift1.y;
                  P3.x := PtCentre.x+ZShiftCentre.x;
                  P3.y := PtCentre.y+ZShiftCentre.y;
                  ZFraction := ((Y00+Y10+YCentre)/3-TheMin) / Span;
                  ACanvas.Pen.Color := Misc.Rainbow(ZFraction);
                  ACanvas.Brush.Color := ACanvas.Pen.Color;
                  ACanvas.Polygon([P1, P2, P3]);
  {Bottom triangle:}
                  P2.x := Pt01.x+ZShift0.x;
                  P2.y := Pt01.y+ZShift0.y;
                  ZFraction := ((Y00+Y01+YCentre)/3-TheMin) / Span;
                  ACanvas.Pen.Color := Misc.Rainbow(ZFraction);
                  ACanvas.Brush.Color := ACanvas.Pen.Color;
                  ACanvas.Polygon([P1, P2, P3]);
  {Right triangle:}
                  P1.x := Pt11.x+ZShift1.x;
                  P1.y := Pt11.y+ZShift1.y;
                  ZFraction := ((Y01+Y11+YCentre)/3-TheMin) / Span;
                  ACanvas.Pen.Color := Misc.Rainbow(ZFraction);
                  ACanvas.Brush.Color := ACanvas.Pen.Color;
                  ACanvas.Polygon([P1, P2, P3]);
  {Top triangle:}
                  P2.x := Pt10.x+ZShift1.x;
                  P2.y := Pt10.y+ZShift1.y;
                  ZFraction := ((Y10+Y11+YCentre)/3-TheMin) / Span;
                  ACanvas.Pen.Color := Misc.Rainbow(ZFraction);
                  ACanvas.Brush.Color := ACanvas.Pen.Color;
                  ACanvas.Polygon([P1, P2, P3]);
                end;

              cdHigh, cdVHigh:
                begin
                  iX := FYAxis.MidX + 20;
                  iY := FYAxis.Top + 20;
                  ACanvas.TextOut(iX, iY, 'Exercise 1: go to line 1192 of Datalist.pas');
                  Inc(iY, ACanvas.TextHeight('Pp'));
                  ACanvas.TextOut(iX, iY, ' and type in an algorithm to scan across each surface element');
                  Inc(iY, ACanvas.TextHeight('Pp'));
                  ACanvas.TextOut(iX, iY, ' and render each pixel in the correct colour.');
                  Inc(iY, ACanvas.TextHeight('Pp'));
                  ACanvas.TextOut(iX, iY, 'Hint: calculate the intersection (X,Y,Z) of the eye ray trace');
                  Inc(iY, ACanvas.TextHeight('Pp'));
                  ACanvas.TextOut(iX, iY, ' with the plane of the P1 P2 P3 panel.');
                end;
            end;
{update to next points:}
            Pt10.x := Pt11.x;
            Pt10.y := Pt11.y;
            //Z10 := Z11;
          end;
        end; {not the final series}
{update again:}
        Pt00.x := Pt01.x;
        Pt00.y := Pt01.y;
        //Z00 := Z01;
      end; {loop over points}
    end; {NoPts}
  end; {over every series}

{draw the color scale:}
  //DrawColorScale(ACanvas, TheMin, Span, ContourDetail);
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DrawContour
  Description: standard Drawing procedure for 2.5D contour graphs
       Author: Mat Ballard
 Date created: 01/24/2001
Date modified: 01/24/2001 by Mat Ballard
      Purpose: This draws all the series as a 2.5D coloured contour graph
 Known Issues: Note that Y data is plotted as the color, and the Z data
               as screen Y
 ------------------------------------------------------------------------------}
procedure TSeriesList.DrawContour(
  ACanvas: TCanvas; ContourDetail: TContourDetail);
var
  i, iSeries: Integer;
  iCel,
  jCel,
  jMin, jMax: Integer;
  FontHeight, iX, iXp1, iY: Integer;
  ZFraction,
  TheMin,
  Span: Single;
  Pt00, Pt01, Pt10, Pt11, PtCentre: TPoint;
  Z00, Z01, Z10, Z11, ZCentre: Single;
  dZdX, dZdY: Single;
  m0011, m1001,
  b0011, b1001: Single;
  pSeries0,
  pSeries1: TSeries;
  TheText: String;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.DrawContour: ACanvas is nil !');
{$ENDIF}

  TheMin := Self.Ymin;
  Span := Self.YMax - TheMin;
  ACanvas.Brush.Style := bsSolid;
  //ACanvas.Brush.Bitmap := nil;

{Draw each individual series:}
  for iSeries := 0 to Self.Count-1 do
  begin
    pSeries0 := TSeries(Items[iSeries]);
    if (pSeries0.NoPts > 0) then
    begin
{Assign the bottom-left point:}
      Pt00.x := pSeries0.XAxis.FofX(pSeries0.XData^[0]);
{Note: for contours, we plot the Y data as a function of X and Z,
 so that Y determines the color, and Z determines the screen Y value.
 The reason for this is that most data will be in the form of scans (series)
 at repeated times.}
      Pt00.y := pSeries0.YAxis.FofY(pSeries0.ZData);
      Z00 := pSeries0.YData^[0];
      if (iSeries < Count-1) then
      begin
        pSeries1 := TSeries(Items[iSeries+1]);
        if (pSeries1.NoPts > 0) then
        begin
{Assign the top-left point:}
          Pt10.x := pSeries1.XAxis.FofX(pSeries1.XData^[0]);
          Pt10.y := pSeries1.YAxis.FofY(pSeries1.ZData);
          Z10 := pSeries1.YData^[0];
        end;
      end;

      for i := 1 to pSeries0.NoPts-1 do
      begin
{Assign the bottom-right point:}
        Pt01.x := pSeries0.XAxis.FofX(pSeries0.XData^[i]);
        Pt01.y := pSeries0.YAxis.FofY(pSeries0.ZData);
        Z01 := pSeries0.YData^[i];
{Oh yes it was: Delphi ain't that smart.}
        if ((iSeries < Count-1) and (pSeries1.NoPts > 0)) then
        begin
{Assign the top-right point:}
          Pt11.x := pSeries1.XAxis.FofX(pSeries1.XData^[i]);
          Pt11.y := pSeries1.YAxis.FofY(pSeries1.ZData);
          Z11 := pSeries1.YData^[i];

{Calculate the centrum:}
          PtCentre.x := (Pt00.x + Pt01.x + Pt10.x + Pt11.x) div 4;
          PtCentre.y := (Pt00.y + Pt01.y + Pt10.y + Pt11.y) div 4;
{Oh yes it was: see above}
          ZCentre := (Z00 + Z01 + Z10 + Z11) / 4;

{Calculate the two cross-lines: y = mx + b:}
          m0011 := (Pt11.y - Pt00.y) / (Pt11.x - Pt00.x);
          b0011 := Pt00.y - m0011 * Pt00.x;
          m1001 := (Pt10.y - Pt01.y) / (Pt10.x - Pt01.x);
          b1001 := Pt10.y - m1001 * Pt10.x;

{just how detailed will the plot be ?}
          case ContourDetail of
            cdLow:
              begin
{No triangles:}
                ZFraction := (ZCentre - TheMin) / Span;
                ACanvas.Pen.Color := Misc.Rainbow(ZFraction);
                ACanvas.Brush.Color := ACanvas.Pen.Color;
                ACanvas.Polygon([Pt00, Pt10, Pt11, Pt01]);
              end;
              
            cdMedium:
              begin
{Left triangle:}
                ZFraction := ((Z00+Z10+ZCentre)/3-TheMin) / Span;
                ACanvas.Pen.Color := Misc.Rainbow(ZFraction);
                ACanvas.Brush.Color := ACanvas.Pen.Color;
                ACanvas.Polygon([Pt00, Pt10, PtCentre]);
{Bottom triangle:}
                ZFraction := ((Z00+Z01+ZCentre)/3-TheMin) / Span;
                ACanvas.Pen.Color := Misc.Rainbow(ZFraction);
                ACanvas.Brush.Color := ACanvas.Pen.Color;
                ACanvas.Polygon([Pt00, Pt01, PtCentre]);
{Top triangle:}
                ZFraction := ((Z10+Z11+ZCentre)/3-TheMin) / Span;
                ACanvas.Pen.Color := Misc.Rainbow(ZFraction);
                ACanvas.Brush.Color := ACanvas.Pen.Color;
                ACanvas.Polygon([Pt11, Pt10, PtCentre]);
{Right triangle:}
                ZFraction := ((Z01+Z11+ZCentre)/3-TheMin) / Span;
                ACanvas.Pen.Color := Misc.Rainbow(ZFraction);
                ACanvas.Brush.Color := ACanvas.Pen.Color;
                ACanvas.Polygon([Pt11, Pt01, PtCentre]);
              end;

            cdHigh:
              begin
{Left triangle: base partial derivatives on Point00:}
                dZdX := (ZCentre-(Z00+Z10)/2) / (PtCentre.x - (Pt00.x+Pt10.x) div 2);
                dZdY := (Z10-Z00) / (Pt10.y - Pt00.y);
                for iCel := Pt00.x to PtCentre.x do {??? -1}
                begin
                  jMax := Round(m0011 * iCel + b0011);
                  jMin := Round(m1001 * iCel + b1001);
                  for jCel := jMin to jMax do
                  begin
                    ZFraction := (Z00 + dZdX*(iCel-Pt00.x) + dZdY*(jCel-Pt00.y) - TheMin) / Span;
{$IFDEF MSWINDOWS}
                    ACanvas.Pixels[iCel, jCel] := Misc.Rainbow(ZFraction);
{$ENDIF}
{$IFDEF LINUX}
                    ACanvas.Pen.Color := Misc.Rainbow(ZFraction);
                    ACanvas.DrawPoint(iCel, jCel);
{$ENDIF}
                  end;
                end;
{Bottom triangle: base partial derivatives on Point00:}
                dZdX := (Z01-Z00) / (Pt01.x - Pt00.x);
                dZdY := ((Z00+Z01)/2 - ZCentre) / ((Pt00.y+Pt01.y) div 2 - PtCentre.y);
                for jCel := PtCentre.y to Pt00.y do
                begin
                  jMin := Round((jCel - b0011) / m0011);
                  jMax := Round((jCel - b1001) / m1001);
                  for iCel := jMin to jMax do
                  begin
                    ZFraction := (Z00 + dZdX*(iCel-Pt00.x) + dZdY*(jCel-Pt00.y) - TheMin) / Span;
{$IFDEF MSWINDOWS}
                    ACanvas.Pixels[iCel, jCel] := Misc.Rainbow(ZFraction);
{$ENDIF}
{$IFDEF LINUX}
                    ACanvas.Pen.Color := Misc.Rainbow(ZFraction);
                    ACanvas.DrawPoint(iCel, jCel);
{$ENDIF}
                  end;
                end;
{Right triangle: base partial derivatives on Point01:}
                dZdX := (ZCentre-(Z01+Z11)/2) / (PtCentre.x - (Pt01.x+Pt11.x) div 2);
                dZdY := (Z11-Z01) / (Pt11.y - Pt01.y);
                for iCel := PtCentre.x to Pt01.x do
                begin
                  jMin := Round(m0011 * iCel + b0011);
                  jMax := Round(m1001 * iCel + b1001);
                  for jCel := jMin to jMax do
                  begin
                    ZFraction := (Z01 + dZdX*(iCel-Pt01.x) + dZdY*(jCel-Pt01.y) - TheMin) / Span;
{$IFDEF MSWINDOWS}
                    ACanvas.Pixels[iCel, jCel] := Misc.Rainbow(ZFraction);
{$ENDIF}
{$IFDEF LINUX}
                    ACanvas.Pen.Color := Misc.Rainbow(ZFraction);
                    ACanvas.DrawPoint(iCel, jCel);
{$ENDIF}
                  end;
                end;
{Top triangle: base partial derivatives on Point10:}
                dZdX := (Z10-Z11) / (Pt10.x - Pt11.x);
                dZdY := ((Z10+Z11)/2 - ZCentre) / ((Pt10.y+Pt11.y) div 2 - PtCentre.y);
                for jCel := Pt10.Y to PtCentre.y do
                begin
                  jMax := Round((jCel - b0011) / m0011);
                  jMin := Round((jCel - b1001) / m1001);
                  for iCel := jMin to jMax do
                  begin
                    ZFraction := (Z10 + dZdX*(iCel-Pt10.x) + dZdY*(jCel-Pt10.y) - TheMin) / Span;
{$IFDEF MSWINDOWS}
                    ACanvas.Pixels[iCel, jCel] := Misc.Rainbow(ZFraction);
{$ENDIF}
{$IFDEF LINUX}
                    ACanvas.Pen.Color := Misc.Rainbow(ZFraction);
                    ACanvas.DrawPoint(iCel, jCel);
{$ENDIF}
                  end;
                end;
              end;

            cdVHigh:
              begin
{Left triangle: base partial derivatives on Point00:}
                dZdX := (ZCentre-(Z00+Z10)/2) / (PtCentre.x - (Pt00.x+Pt10.x) div 2);
                dZdY := (Z10-Z00) / (Pt10.y - Pt00.y);
                for iCel := Pt00.x to PtCentre.x-1 do
                begin
                  jMax := Round(m0011 * iCel + b0011);
                  jMin := Round(m1001 * iCel + b1001);
                  for jCel := jMin to jMax do
                  begin
                    ZFraction := (Z00 + dZdX*(iCel-Pt00.x) + dZdY*(jCel-Pt00.y) - TheMin) / Span;
                    ACanvas.Pen.Color := Misc.Rainbow(ZFraction);
                    ACanvas.Brush.Color := ACanvas.Pen.Color;
                    ACanvas.FillRect(Rect(iCel, jCel, iCel+1, jCel+1));
                  end;
                end;
{Bottom triangle: base partial derivatives on Point00:}
                dZdX := (Z01-Z00) / (Pt01.x - Pt00.x);
                dZdY := ((Z00+Z01)/2 - ZCentre) / ((Pt00.y+Pt01.y) div 2 - PtCentre.y);
                for jCel := PtCentre.y to Pt00.y do
                begin
                  jMin := Round((jCel - b0011) / m0011);
                  jMax := Round((jCel - b1001) / m1001);
                  for iCel := jMin to jMax do
                  begin
                    ZFraction := (Z00 + dZdX*(iCel-Pt00.x) + dZdY*(jCel-Pt00.y) - TheMin) / Span;
                    ACanvas.Pen.Color := Misc.Rainbow(ZFraction);
                    ACanvas.Brush.Color := ACanvas.Pen.Color;
                    ACanvas.FillRect(Rect(iCel, jCel, iCel+1, jCel+1));
                  end;
                end;
{Right triangle: base partial derivatives on Point01:}
                dZdX := (ZCentre-(Z01+Z11)/2) / (PtCentre.x - (Pt01.x+Pt11.x) div 2);
                dZdY := (Z11-Z01) / (Pt11.y - Pt01.y);
                for iCel := PtCentre.x to Pt01.x do
                begin
                  jMin := Round(m0011 * iCel + b0011);
                  jMax := Round(m1001 * iCel + b1001);
                  for jCel := jMin to jMax do
                  begin
                    ZFraction := (Z01 + dZdX*(iCel-Pt01.x) + dZdY*(jCel-Pt01.y) - TheMin) / Span;
                    ACanvas.Pen.Color := Misc.Rainbow(ZFraction);
                    ACanvas.Brush.Color := ACanvas.Pen.Color;
                    ACanvas.FillRect(Rect(iCel, jCel, iCel+1, jCel+1));
                  end;
                end;
{Top triangle: base partial derivatives on Point10:}
                dZdX := (Z10-Z11) / (Pt10.x - Pt11.x);
                dZdY := ((Z10+Z11)/2 - ZCentre) / ((Pt10.y+Pt11.y) div 2 - PtCentre.y);
                for jCel := Pt10.Y to PtCentre.y do
                begin
                  jMax := Round((jCel - b0011) / m0011);
                  jMin := Round((jCel - b1001) / m1001);
                  for iCel := jMin to jMax do
                  begin
                    ZFraction := (Z10 + dZdX*(iCel-Pt10.x) + dZdY*(jCel-Pt10.y) - TheMin) / Span;
                    ACanvas.Pen.Color := Misc.Rainbow(ZFraction);
                    ACanvas.Brush.Color := ACanvas.Pen.Color;
                    ACanvas.FillRect(Rect(iCel, jCel, iCel+1, jCel+1));
                  end;
                end;
              end;
          end; {case}

{update values:}
          Pt10.x := Pt11.x;
          Pt10.y := Pt11.y;
          Z10 := Z11;
        end; {not the final series}
        Pt00.x := Pt01.x;
        Pt00.y := Pt01.y;
        Z00 := Z01;
      end; {loop over points}
      {ACanvas.MoveTo(Pt00.x, Pt00.y);
      ACanvas.LineTo(Pt10.x, Pt10.y);}
    end; {NoPts}
  end; {over every series}

{draw the color scale:}
  DrawColorScale(ACanvas, TheMin, Span, ContourDetail);
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DrawColorScale
  Description: draws the Color Scale for Contour plots
       Author: Mat Ballard
 Date created: 03/06/2001
Date modified: 03/06/2001 by Mat Ballard
      Purpose: contour plot details
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DrawColorScale(ACanvas: TCanvas; TheMin, Span: Single; TheContourDetail: TContourDetail);
var
  i,
  iX, iXp1, iY,
  FontHeight: Integer;
  TheText: String;
begin
  iX := FXAxis.Right + FXAxis.Width div 50;
  iXp1 := iX + FXAxis.Width div 50;
  if (TheContourDetail = cdVHigh) then
  begin
    for iY := FYAxis.Bottom downto FYAxis.Top do
    begin
      ACanvas.Pen.Color := Misc.Rainbow((FYAxis.Bottom - iY) / (FYAxis.Bottom - FYAxis.Top));
      ACanvas.Brush.Color := ACanvas.Pen.Color;
      ACanvas.FillRect(Rect(iX, iY, iXp1, iY+1));
    end;
  end
  else
  begin {low, medium, high:}
    for iY := FYAxis.Bottom downto FYAxis.Top do
    begin
      ACanvas.Pen.Color := Rainbow((FYAxis.Bottom - iY) / (FYAxis.Bottom - FYAxis.Top));
      ACanvas.MoveTo(iX, iY);
      ACanvas.LineTo(iXp1, iY);
    end;
  end;

{put some labels on it:}
  ACanvas.Font.Assign(FYAxis.Labels.Font);
  ACanvas.Brush.Style := bsClear;
  FontHeight := ACanvas.TextHeight('9') div 2;
  for i := 0 to 4 do
  begin
    iY := FYAxis.Bottom +
      i * (FYAxis.Top - FYAxis.Bottom) div 4 - FontHeight;
    TheText := FYAxis.LabelToStrF(TheMin);
    ACanvas.TextOut(iXp1+2, iY, TheText);
    TheMin := TheMin + Span / 4;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DrawHistory
  Description: standard Drawing procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: draws all the Series on a given canvas IN History MODE
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DrawHistory(ACanvas: TCanvas; HistoryX: Single);
var
  i: Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.DrawHistory: ACanvas is nil !');
{$ENDIF}

  for i := 0 to Count-1 do
  begin
    TSeries(Items[i]).DrawHistory(ACanvas, HistoryX);
  end; {loop over series}
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DrawError
  Description: extended Drawing procedure for linking multiple series
       Author: Mat Ballard
 Date created: 01/21/2001
Date modified: 01/21/2001 by Mat Ballard
      Purpose: links Series in pairs on a given canvas
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DrawError(ACanvas: TCanvas);
var
  i,
  iX,
  iY,
  iSeries: Integer;
  pSeries,
  pErrorSeries: TSeries;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.DrawError: ACanvas is nil !');
{$ENDIF}
  iSeries := 0;
  while (iSeries <= Self.Count-2) do
  begin
    pSeries := TSeries(Items[iSeries]);
    pErrorSeries := TSeries(Items[iSeries+1]);
    if (pSeries.Visible) then
    begin
      ACanvas.Pen.Assign(pSeries.Pen);

      if ((pSeries.Symbol = syNone) or (pSeries.SymbolSize = 0)) then
      begin
{no symbols:}
        for i := 0 to Min(pSeries.NoPts, pErrorSeries.NoPts)-1 do
        begin
{draw the vertical line:}
          ACanvas.MoveTo(
            pSeries.XAxis.FofX(pSeries.XData^[i]) + pSeries.DeltaX,
            pSeries.YAxis.FofY(pSeries.YData^[i] - pErrorSeries.YData^[i]) + pSeries.DeltaY);
          ACanvas.LineTo(
            pSeries.XAxis.FofX(pSeries.XData^[i]) + pSeries.DeltaX,
            pSeries.YAxis.FofY(pSeries.YData^[i] + pErrorSeries.YData^[i]) + pSeries.DeltaY);
{and the horizontal:}
          if (pErrorSeries.ExternalXSeries) then
          begin {no X errors:}
            ACanvas.MoveTo(
              pSeries.XAxis.FofX(pSeries.XData^[i]) - 5 + pSeries.DeltaX,
              pSeries.YAxis.FofY(pSeries.YData^[i]) + pSeries.DeltaY);
            ACanvas.LineTo(
              pSeries.XAxis.FofX(pSeries.XData^[i]) + 5 + pSeries.DeltaX,
              pSeries.YAxis.FofY(pSeries.YData^[i]) + pSeries.DeltaY);
          end
          else
          begin {X errors:}
            ACanvas.MoveTo(
              pSeries.XAxis.FofX(pSeries.XData^[i] - pErrorSeries.XData^[i]) + pSeries.DeltaX,
              pSeries.YAxis.FofY(pSeries.YData^[i]) + pSeries.DeltaY);
            ACanvas.LineTo(
              pSeries.XAxis.FofX(pSeries.XData^[i] + pErrorSeries.XData^[i]) + pSeries.DeltaX,
              pSeries.YAxis.FofY(pSeries.YData^[i]) + pSeries.DeltaY);
          end; {X Errors}
        end; {for}
      end
      else
      begin
{symbols:}
        for i := 0 to Min(pSeries.NoPts, pErrorSeries.NoPts)-1 do
        begin
          iX := pSeries.XAxis.FofX(pSeries.XData^[i]) + pSeries.DeltaX;
          iY := pSeries.YAxis.FofY(pSeries.YData^[i]) + pSeries.DeltaY;
          pSeries.DrawSymbol(ACanvas, iX, iY);
          ACanvas.MoveTo(iX, iY - pSeries.SymbolSize);
          ACanvas.LineTo(iX, pSeries.YAxis.FofY(pSeries.YData^[i] + pErrorSeries.YData^[i]));
          ACanvas.MoveTo(iX, iY + pSeries.SymbolSize);
          ACanvas.LineTo(iX, pSeries.YAxis.FofY(pSeries.YData^[i] - pErrorSeries.YData^[i]));
          if (not pErrorSeries.ExternalXSeries) then
          begin
            ACanvas.MoveTo(iX - pSeries.SymbolSize, iY);
            ACanvas.LineTo(pSeries.XAxis.FofX(pSeries.XData^[i] - pErrorSeries.XData^[i]), iY);
            ACanvas.MoveTo(iX + pSeries.SymbolSize, iY);
            ACanvas.LineTo(pSeries.XAxis.FofX(pSeries.XData^[i] + pErrorSeries.XData^[i]), iY);
          end;
        end; {for}
      end; {if symbols}
    end; {if visible}
    Inc(iSeries, 2);
  end; {while}
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DrawBubble
  Description: extended Drawing procedure for Bubble plots
       Author: Mat Ballard
 Date created: 04/11/2001
Date modified: 04/11/2001 by Mat Ballard
      Purpose: links Series in pairs as Bubble plots
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DrawBubble(ACanvas: TCanvas; BubbleSize: Integer);
var
  i,
  iX,
  iY,
  iSeries,
  YRadius: Integer;
  BubbleMax: Single;
  pSeries,
  pBubbleSeries: TSeries;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.DrawBubble: ACanvas is nil !');
{$ENDIF}
  iSeries := 0;
  while (iSeries <= Self.Count-2) do
  begin
    pSeries := TSeries(Items[iSeries]);
{every odd numbered series contains the Bubble height, and optionally the breadth:}
    pBubbleSeries := TSeries(Items[iSeries+1]);
    BubbleMax := 100 * pBubbleSeries.YMax / (BubbleSize * (FYAxis.Bottom - FYAxis.Top));
    if (pSeries.Visible) then
    begin
      ACanvas.Pen.Assign(pSeries.Pen);
      ACanvas.Brush.Assign(pSeries.Brush);

      for i := 0 to Min(pSeries.NoPts, pBubbleSeries.NoPts)-1 do
      begin
        iX := pSeries.XAxis.FofX(pSeries.XData^[i]) + pSeries.DeltaX;
        iY := pSeries.YAxis.FofY(pSeries.YData^[i]) + pSeries.DeltaY;
        YRadius := Round(pBubbleSeries.YData^[i] / BubbleMax);
        if (YRadius >= 0) then
          ACanvas.Brush.Style := bsSolid
         else
          ACanvas.Brush.Style := bsCross;
        ACanvas.Ellipse(iX - YRadius, iY - YRadius, iX + YRadius, iY + YRadius)
      end; {for}
    end; {if visible}
    Inc(iSeries, 2);
  end; {while}
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DrawMultiple
  Description: extended Drawing procedure for linking multiple series
       Author: Mat Ballard
 Date created: 09/21/2000
Date modified: 09/21/2000 by Mat Ballard
      Purpose: links multiple Series on a given canvas
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DrawMultiple(
  ACanvas: TCanvas;
  Multiplicity: Byte;
  MultiplePen: TPen;
  MultiJoin1, MultiJoin2: Integer);
var
  i,
  j,
  k,
  iSeries,
  NoGroups: Integer;
  pSeries, pSeries2: TSeries;
  DoMultiJoin: Boolean;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.DrawMultiple: ACanvas is nil !');
  Assert(Multiplicity > 1, 'TSeriesList.DrawMultiple: Multiplicity is less than 2 !');
{$ENDIF}

{are there two valid multijoined series ?}
  DoMultiJoin := FALSE;
  if ((0 <= MultiJoin1) and (MultiJoin1 < Self.Count) and
      (0 <= MultiJoin2) and (MultiJoin2 < Self.Count) and
      (MultiJoin1 <> MultiJoin2)) then
  begin
{We don't draw MultiJoined series:}  
    TSeries(Items[MultiJoin1]).Symbol := syNone;
    TSeries(Items[MultiJoin2]).Symbol := syNone;
    DoMultiJoin := TRUE;
  end;

{Draw the normal series lines and symbols:}
  Self.Draw(ACanvas, High(Integer));
{Prepare for the vertical lines:}
  ACanvas.Pen.Assign(MultiplePen);
{And MultiJoin symbols:}  
  if (DoMultiJoin) then
  begin
    ACanvas.Brush.Assign(TSeries(Items[MultiJoin1]).Brush);
  end;

  NoGroups := Count div Multiplicity;
  for i := 0 to MinNoPts-1 do
  begin
    iSeries := -1;
    for j := 1 to NoGroups do
    begin
      Inc(iSeries);
      if (iSeries >= Count) then break;
      pSeries := TSeries(Items[iSeries]);
      ACanvas.MoveTo(
        pSeries.XAxis.FofX(pSeries.XData^[i]),
        pSeries.YAxis.FofY(pSeries.YData^[i]));
      for k := 2 to Multiplicity do
      begin
        Inc(iSeries);
        if (iSeries >= Count) then break;
        pSeries := TSeries(Items[iSeries]);
        ACanvas.LineTo(
          pSeries.XAxis.FofX(pSeries.XData^[i]),
          pSeries.YAxis.FofY(pSeries.YData^[i]));
      end; {Multiplicity}
      if (DoMultiJoin) then
      begin
        pSeries := TSeries(Items[MultiJoin1]);
        pSeries2 := TSeries(Items[MultiJoin2]);
        ACanvas.Rectangle(
          pSeries.XAxis.FofX(pSeries.XData^[i]) - pSeries.SymbolSize,
          pSeries.YAxis.FofY(pSeries.YData^[i]),
          pSeries2.XAxis.FofX(pSeries2.XData^[i]) + pSeries.SymbolSize,
          pSeries2.YAxis.FofY(pSeries2.YData^[i]));
      end;
    end; {NoGroups}
  end; {MinNoPts}
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DrawHistoryMultiple
  Description: extended Drawing procedure for linking multiple series in History mode
       Author: Mat Ballard
 Date created: 09/21/2000
Date modified: 09/21/2000 by Mat Ballard
      Purpose: links multiple Series in History mode on a given canvas
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DrawHistoryMultiple(ACanvas: TCanvas; Multiplicity: Byte);
var
  i,
  j,
  k,
  iSeries,
  NoGroups: Integer;
  pSeries: TSeries;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.DrawHistoryMultiple: ACanvas is nil !');
  Assert(Multiplicity > 1, 'TSeriesList.DrawHistoryMultiple: Multiplicity is less than 2 !');
{$ENDIF}

{we set the pen mode so that a second call to DrawHistory
erases the curve on screen:}
  ACanvas.Pen.Mode := pmNotXOR;
  NoGroups := Count div Multiplicity;
  for i := MinNoPts-1 downto 0 do
  begin
    iSeries := -1;
    for j := 1 to NoGroups do
    begin
      Inc(iSeries);
      if (iSeries >= Count) then break;
      pSeries := TSeries(Items[iSeries]);
      ACanvas.MoveTo(
        pSeries.XAxis.FofX(pSeries.XData^[i]),
        pSeries.YAxis.FofY(pSeries.YData^[i]));
      for k := 2 to Multiplicity do
      begin
        Inc(iSeries);
        if (iSeries >= Count) then break;
        pSeries := TSeries(Items[iSeries]);
        ACanvas.LineTo(
          pSeries.XAxis.FofX(pSeries.XData^[i]),
          pSeries.YAxis.FofY(pSeries.YData^[i]));
      end; {Multiplicity}
    end; {NoGroups}
  end; {MinNoPts}
end;

{TSeriesList editing, both in designer and active modes -----------------------}
{------------------------------------------------------------------------------
     Function: TSeriesList.CloneSeries
  Description: Clones (Adds then Copies) a Series to a new one
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: creates, initializes, adds then copies a Series
 Return Value: the Index of the new Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.CloneSeries(
  TheSeries: Integer): Integer;
var
  pClone,
  pSeries: TSeries;
  TheClone,
  TheXSeries: Integer;
begin
  if ((TheSeries < 0) or (TheSeries > Count-1)) then raise
    ERangeError.CreateFmt(
      'I cannot clone Series #%d because there are only %d Series !',
        [TheSeries, Count]);

{so lets create the clone: the XDataSeries is either nil, or an existing series}
  pSeries := TSeries(Items[TheSeries]);
  TheXSeries := -1;
  if (pSeries.XDataSeries <> nil) then
    TheXSeries := IndexOf(pSeries.XDataSeries);
  TheClone := Add(TheXSeries);
  pClone := TSeries(Items[TheClone]);

{set properties:}
  pClone.YAxisIndex := pSeries.YAxisIndex;
  pClone.DefSize := pSeries.DefSize;
  pClone.DeltaX := pSeries.DeltaX;
{move the cloned series up by 20 pixels:}
  pClone.DeltaY := pSeries.DeltaY - 30;
  pClone.Name := 'Clone of ' + pSeries.Name;
  pClone.OnStyleChange:= pSeries.OnStyleChange;
  pClone.OnDataChange:= pSeries.OnDataChange;
  pClone.Pen.Style := pSeries.Pen.Style;
  pClone.Pen.Width := pSeries.Pen.Width;
  pClone.Symbol:= pSeries.Symbol;
  pClone.SymbolSize:= pSeries.SymbolSize;
  pClone.Visible:= pSeries.Visible;
  {pClone.SeriesType := pSeries.SeriesType;}

  {case pSeries.SeriesType of
    stXY: pClone.AddData(pSeries.XData, pSeries.YData, nil, pSeries.NoPts);
    stXY_Error: pClone.AddData(pSeries.XData, pSeries.YData, pSeries.YErrorData, pSeries.NoPts);
    stXYZ: pClone.AddData(pSeries.XData, pSeries.YData, pSeries.ZData, pSeries.NoPts);
  end;}

  pClone.AddData(pSeries.XData, pSeries.YData, pSeries.NoPts);
  pClone.ResetBounds;
  pClone.GetBounds;

{AddData above fires the OnDataChange event for the series:}
  {DoDataChange;}
  CloneSeries := TheClone;
end;

{TSeriesList editing, both in designer and active modes -----------------------}
{------------------------------------------------------------------------------
    Procedure: TSeriesList.DataAsHTMLTable
  Description: creates a stringlist of the data as a HTML table
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: copying data
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DataAsHTMLTable(var TheData: TStringList);
{This puts the data into a stringlist for saving or copying}
var
  i, j: Integer;
  NoPtsMax: Integer;
  pSeries: TSeries;
  SubHeader: String;
begin
  if (TheData = nil) then exit;

{Determine the maximum number of points in all the series:}
  NoPtsMax := 0;
  for i := 0 to Count-1 do
  begin
    if (NoPtsMax < TSeries(Items[i]).NoPts) then
      NoPtsMax := TSeries(Items[i]).NoPts;
  end;

{loop over all series:}
  SubHeader := #9 + '<tr>';
  for i := 0 to Count-1 do
  begin
    pSeries := TSeries(Items[i]);
{create the sub-headings:}
    if (not pSeries.ExternalXSeries) then
    begin
      SubHeader := SubHeader + '<td>' + pSeries.Name + ' - ' + pSeries.XAxis.Title.Caption + '</td>';
    end;
    SubHeader := SubHeader + '<td>' + pSeries.Name + ' - ' + pSeries.YAxis.Title.Caption + '</td>';
{loop over points in each series:}
    for j := 0 to pSeries.NoPts-1 do
    begin
      if (TheData.Count <= j) then
      begin
{start a new row:}
        TheData.Add(#9+#9+ '<tr>');
      end;
      if (not pSeries.ExternalXSeries) then
      begin
        TheData[j] := TheData[j] + '<td>' + FloatToStr(pSeries.XData^[j]) + '</td>';
      end;
      TheData[j] := TheData[j] + '<td>' + FloatToStr(pSeries.YData^[j]) + '</td>';
    end; {loop over points}
    for j := pSeries.NoPts to NoPtsMax-1 do
    begin
      if (TheData.Count <= j) then
      begin
{start a new row:}
        TheData.Add(#9+#9+ '<tr>');
      end;
      if (not pSeries.ExternalXSeries) then
      begin
        TheData[j] := TheData[j] + '<td></td>';
      end;
      TheData[j] := TheData[j] + '<td></td>';
    end;
  end; {loop over series}
  SubHeader := SubHeader + '</tr>';
  TheData.Insert(0, SubHeader);
  TheData.Insert(0, '<table border=2 cellpadding=2 cellspacing=2>');
  TheData.Add('</table>');
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.AppendStream
  Description: puts the data collected since LastSavedPoint into a stringlist for saving or copying.
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: saving data
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.AppendStream(
  AsText: Boolean;
  Delimiter: Char;
  TheStream: TMemoryStreamEx);
var
  i: Integer;
  NoPtsMax: Integer;
begin
  if (TheStream = nil) then exit;

{Determine the maximum number of points in all the series:}
  NoPtsMax := 0;
  for i := 0 to Count-1 do
  begin
    if (NoPtsMax < TSeries(Items[i]).NoPts) then
      NoPtsMax := TSeries(Items[i]).NoPts;
  end;

  if (AsText) then
    GetTextStream(Delimiter, LastSavedPoint, NoPtsMax, TheStream)
   else
    GetBinaryStream(LastSavedPoint, NoPtsMax, TheStream);

  LastSavedPoint := NoPtsMax-1;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.GetStream
  Description: puts the data into a MemoryStream
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 03/07/2001 by Mat Ballard
      Purpose: for saving or copying
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.GetStream(
  AsText: Boolean;
  Delimiter: Char;
  var TheStream: TMemoryStream);
var
  i,
  NoPtsMax: Integer;
  ALine: String;
  pLine: array [0..1023] of char;
begin
  if (TheStream = nil) then
    TheStream := TMemoryStream.Create;

  Self.GetSubHeaderStream(Delimiter, TheStream);

  if (AsText) then
    StrPCopy(pLine, 'Binary=0' + CRLF)
   else
    StrPCopy(pLine, 'Binary=1' + CRLF);
  TheStream.Write(pLine, StrLen(pLine));

  if (Count = 0) then exit;

{Determine the maximum number of points in all the series:}
  NoPtsMax := 0;
  for i := 0 to Count-1 do
  begin
    if (NoPtsMax < TSeries(Items[i]).NoPts) then
      NoPtsMax := TSeries(Items[i]).NoPts;
  end;

  if (AsText) then
  begin
    ALine := 'ZData:' + Delimiter;
    if (TSeries(Items[0]).XStringData <> nil) then
      if (TSeries(Items[0]).XStringData.Count > 0) then
        ALine := ALine + Delimiter;
    ALine := ALine + FloatToStr(TSeries(Items[0]).ZData);
    for i := 1 to Count-1 do
    begin
      if (not TSeries(Items[i]).ExternalXSeries) then
      begin
        if (TSeries(Items[i]).XStringData <> nil) then
          if (TSeries(Items[i]).XStringData.Count > 0) then
            ALine := ALine + Delimiter;
        ALine := ALine + Delimiter;    
      end;
      ALine := ALine + Delimiter +
        FloatToStr(TSeries(Items[i]).ZData)
    end;
    ALine := ALine + CRLF;
{$IFDEF DELPHI1}
    StrPCopy(pLine, ALine);
    TheStream.Write(pLine, StrLen(pLine));
{$ELSE}
    TheStream.Write(Pointer(ALine)^, Length(ALine));
{$ENDIF}
    GetTextStream(Delimiter, 0, NoPtsMax-1, TheStream)
  end
  else
  begin
    ALine := 'ZData:';
{$IFDEF DELPHI1}
    StrPCopy(pLine, ALine);
    TheStream.Write(pLine, StrLen(pLine));
{$ELSE}
    TheStream.Write(Pointer(ALine)^, Length(ALine));
{$ENDIF}
    for i := 0 to Count-1 do
      TheStream.Write(TSeries(Items[i]).ZData, SizeOf(Single));
    GetBinaryStream(0, NoPtsMax-1, TheStream);
  end;

  FDataChanged := FALSE;
  LastSavedPoint := NoPtsMax-1;
end;


{------------------------------------------------------------------------------
    Procedure: TSeriesList.GetSubHeaderStream
  Description: puts the data sub header onto a stream
       Author: Mat Ballard
 Date created: 08/06/2000
Date modified: 08/06/2000 by Mat Ballard
      Purpose: for saving or copying
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.GetSubHeaderStream(
  Delimiter: Char;
  TheStream: TMemoryStream);
var
  i: Integer;
  pSeries: TSeries;
  SeriesNameLine,
  AxisNameLine,
  DataTypeLine,
  XDataSeriesLine: String;
{$IFDEF DELPHI1}
  pLine: array [0..4095] of char;
{$ENDIF}
begin
  if (TheStream = nil) then exit;

{point at the first series:}
  pSeries := TSeries(Items[0]);
{the first series ALWAYS has both an X data:}
  SeriesNameLine := '';
  AxisNameLine := pSeries.XAxis.Title.Caption;
  DataTypeLine := 'X';
  XDataSeriesLine := '-';
{maybe XTEXT data:}
  if ((pSeries.XStringData <> nil) and
      (pSeries.XStringData.Count > 0)) then
  begin
    SeriesNameLine := SeriesNameLine + Delimiter;
    AxisNameLine := AxisNameLine + Delimiter;
    DataTypeLine := DataTypeLine + Delimiter + 'XTEXT';
    XDataSeriesLine := XDataSeriesLine + Delimiter + '-';
  end;
{and ALWAYS has Y data:}
  SeriesNameLine := SeriesNameLine + Delimiter + pSeries.Name;
  AxisNameLine := AxisNameLine + Delimiter + pSeries.YAxis.Title.Caption;
  DataTypeLine := DataTypeLine + Delimiter + 'Y';
  XDataSeriesLine := XDataSeriesLine + Delimiter + '-';

{loop over all the rest of the series:}
  for i := 1 to Count-1 do
  begin
    pSeries := TSeries(Items[i]);

{create the sub-headings:}
    if (not pSeries.ExternalXSeries) then
    begin
{The X data belongs to this series:}
      SeriesNameLine := SeriesNameLine + Delimiter;
      AxisNameLine := AxisNameLine + Delimiter + pSeries.XAxis.Title.Caption;
      DataTypeLine := DataTypeLine + Delimiter + 'X';
      XDataSeriesLine := XDataSeriesLine + Delimiter + '-';
      if ((pSeries.XStringData <> nil) and
          (pSeries.XStringData.Count > 0)) then
      begin
        SeriesNameLine := SeriesNameLine + Delimiter;
        AxisNameLine := AxisNameLine + Delimiter;
        DataTypeLine := DataTypeLine + Delimiter + 'XTEXT';
        XDataSeriesLine := XDataSeriesLine + Delimiter + '-';
      end;
    end;
{The Y data belongs to this series:}
{put the Series Name and YAxis name above the Y column:}
    SeriesNameLine := SeriesNameLine + Delimiter + pSeries.Name;
    AxisNameLine := AxisNameLine + Delimiter + pSeries.YAxis.Title.Caption;
    DataTypeLine := DataTypeLine + Delimiter + 'Y';
    if (pSeries.ExternalXSeries) then
      XDataSeriesLine := XDataSeriesLine + Delimiter +
        IntToStr(IndexOf(pSeries.XDataSeries))
     else
      XDataSeriesLine := XDataSeriesLine + Delimiter;
  end; {for i}

  SeriesNameLine := SeriesNameLine + CRLF;
  AxisNameLine := AxisNameLine + CRLF;
  DataTypeLine := DataTypeLine + CRLF;
  XDataSeriesLine := XDataSeriesLine + CRLF;

{$IFDEF DELPHI1}
  StrPCopy(pLine, SeriesNameLine);
  TheStream.Write(pLine, StrLen(pLine));
  StrPCopy(pLine, AxisNameLine);
  TheStream.Write(pLine, StrLen(pLine));
  StrPCopy(pLine, DataTypeLine);
  TheStream.Write(pLine, StrLen(pLine));
  StrPCopy(pLine, XDataSeriesLine);
  TheStream.Write(pLine, StrLen(pLine));
{$ELSE}
  TheStream.Write(Pointer(SeriesNameLine)^, Length(SeriesNameLine));
  TheStream.Write(Pointer(AxisNameLine)^, Length(AxisNameLine));
  TheStream.Write(Pointer(DataTypeLine)^, Length(DataTypeLine));
  TheStream.Write(Pointer(XDataSeriesLine)^, Length(XDataSeriesLine));
{$ENDIF}
end;

{------------------------------------------------------------------------------
    procedure: TSeriesList.GetBinaryStream
  Description: gets the data as a binary stream
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management: copying and saving
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.GetBinaryStream(
  Start, Finish: Integer;
  TheStream: TMemoryStream);
var
  i,
  j: Integer;
  pSeries: TSeries;
  pNullData: array [0..16] of char;
  pLine: array [0..255] of char;

  procedure WriteStringData;
  begin
    if (pSeries.XStringData <> nil)  then
      if (pSeries.XStringData.Count > 0)  then
      begin
        if (i < pSeries.XStringData.Count)  then
          StrPCopy(pLine, pSeries.XStringData.Strings[i] + CRLF)
        else
          StrPCopy(pLine, CRLF);
        TheStream.Write(pLine, StrLen(pLine));
      end;
  end;
  
begin
  for i := 0 to SizeOf(Single)-1 do
    pNullData[i] := 'x';
  pNullData[SizeOf(Single)] := Chr(0);

{all the data is written in BINARY format:}
  for i := Start to Finish do
  begin
{loop over all series:}
    for j := 0 to Count-1 do
    begin
      pSeries := TSeries(Items[j]);
      if (i < pSeries.NoPts) then
      begin
        if (not pSeries.ExternalXSeries) then
        begin
          TheStream.Write(pSeries.XData^[i], SizeOf(Single));
          WriteStringData;
        end;
        TheStream.Write(pSeries.YData^[i], SizeOf(Single));
      end
      else
      begin
        if (not pSeries.ExternalXSeries) then
        begin
          TheStream.Write(pNullData, SizeOf(Single));
          WriteStringData;
        end;
        TheStream.Write(pNullData, SizeOf(Single));
      end;
    end; {loop over points}
  end;
end;

{------------------------------------------------------------------------------
    procedure: TSeriesList.GetTextStream
  Description: gets the data as a text stream
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management: copying and saving
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.GetTextStream(
  Delimiter: Char;
  Start, Finish: Integer;
  TheStream: TMemoryStream);
var
  i,
  j: Integer;
  pSeries: TSeries;
  //DoStringXData: Boolean;
  TheLine: String;
{$IFDEF DELPHI1}
  pLine: array [0..255] of char;
{$ENDIF}

  procedure WriteStringData;
  begin
    if (pSeries.XStringData <> nil)  then
      if (pSeries.XStringData.Count > 0)  then
      begin
        if (i < pSeries.XStringData.Count)  then
          TheLine := TheLine + Delimiter + pSeries.XStringData.Strings[i]
        else
          TheLine := TheLine + Delimiter;
      end;
  end;

begin
{all the data is written in text format:}
  for i := Start to Finish do
  begin
{loop over all the remaining series:}
    for j := 0 to Count-1 do
    begin
      pSeries := TSeries(Items[j]);
      if (i < pSeries.NoPts) then
      begin
        if (not pSeries.ExternalXSeries) then
        begin
          TheLine := FloatToStr(pSeries.XData^[i]);
          WriteStringData;
        end;
        TheLine := TheLine + Delimiter + FloatToStr(pSeries.YData^[i]);
      end
      else
      begin
        if (not pSeries.ExternalXSeries) then
        begin
          TheLine := TheLine + Delimiter;
          WriteStringData;
        end;
        TheLine := TheLine + Delimiter;
      end;
    end; {loop over points}
    TheLine := TheLine + CRLF;
{$IFDEF DELPHI1}
    StrPCopy(pLine, TheLine);
    TheStream.Write(pLine, StrLen(pLine));
{$ELSE}
    TheStream.Write(Pointer(TheLine)^, Length(TheLine));
{$ENDIF}
  end;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.LoadFromStream
  Description: Opens data on disk
       Author: Mat Ballard
 Date created: 04/25/2001
Date modified: 04/25/2001 by Mat Ballard
      Purpose: Opens data, parses it, fires the OnHeader event, and runs ConvertTextData,
               or decides to run it through ParseData instead
 Known Issues: Called by TPlot.LoadFromStream
 ------------------------------------------------------------------------------}
function TSeriesList.LoadFromStream(
  AStream: TMemoryStream; var AsText: Boolean): Boolean;
var
  TheResult: Boolean;
  ColCount,
  FileVersion,
  i,
  iColumn,
  InitialSeriesCount,
  LineLength,
  NoFileSeries: Integer;
  TheLine,
  SeriesNameLine,
  AxisNameLine,
  DataTypeLine,
  XDataSeriesLine,
  TheCell: String;
  TheStrings: TStringList;
  SeriesInfo: pSeriesInfoArray;
  SeriesOfCol: pIntegerArray;
  OldIgnoreChanges: Boolean;

  procedure CleanUp;
  begin
    if (SeriesInfo <> nil) then
      FreeMem(SeriesInfo, ColCount * SizeOf(TSeriesInfo));
    SeriesInfo := nil;
    if (SeriesOfCol <> nil) then
      FreeMem(SeriesOfCol, ColCount * SizeOf(Integer));
    SeriesOfCol := nil;
    if (TheStrings <> nil) then
      TheStrings.Free;
    TheStrings := nil;  
  end;

begin
  LoadFromStream := FALSE;
  ColCount := 1;
  SeriesInfo := nil;
  SeriesOfCol := nil;
  TheStrings := nil;

  InitialSeriesCount := Self.Count;
  try
{get the sub-header data:}
    SeriesNameLine := ReadLine(AStream);
    AxisNameLine := ReadLine(AStream);
    DataTypeLine := ReadLine(AStream);
    XDataSeriesLine := ReadLine(AStream);

{find out how many columns there are:}
    for i := 1 to Length(DataTypeLine) do
      if (DataTypeLine[i] = ',') then
        Inc(ColCount);

    if (ColCount < 2) then raise
      EFOpenError.CreateFmt('This file only has %d columns !', [ColCount]);

{allocate memory for the dynamic arrays, which are small:}
    GetMem(SeriesInfo, ColCount * SizeOf(TSeriesInfo));
    GetMem(SeriesOfCol, (ColCount+1) * SizeOf(Integer));
{this allocates more memory than SeriesInfo needs, but so what ?}

{Determine the number of series:}
    NoFileSeries := 0;
    for i := 0 to ColCount-1 do
    begin
      SeriesInfo^[i].XCol := 0;
      SeriesInfo^[i].XTextCol := -1;
    end;
{examine the columns one by one:}
    for iColumn := 1 to ColCount do
    begin
{No new column yet belongs to a Series:}
      SeriesOfCol^[iColumn] := -1;
      TheCell := GetWord(DataTypeLine, ',');
      if (TheCell = 'X') then
      begin
{we've found a X data column to add.}
        SeriesOfCol^[iColumn] := NoFileSeries;
        SeriesInfo^[NoFileSeries].XCol := iColumn;
        GetWord(XDataSeriesLine, ',');
        GetWord(SeriesNameLine, ',');
      end
      else if (TheCell = 'XTEXT') then
      begin
{we've found a X STRING data column to add.}
        SeriesOfCol^[iColumn] := NoFileSeries;
        SeriesInfo^[NoFileSeries].XTextCol := iColumn;
        GetWord(XDataSeriesLine, ',');
        GetWord(SeriesNameLine, ',');
      end
      else if (TheCell = 'Y') then
      begin
{we've found a Y data column to add.}
        SeriesInfo^[NoFileSeries].YCol := iColumn;
{find the X Column that this Y column uses:}
        TheCell := GetWord(XDataSeriesLine, ',');
        if (TheCell = '-') then
        begin
          SeriesInfo^[NoFileSeries].XSeriesIndex := -1;
        end
        else
        begin
          SeriesInfo^[NoFileSeries].XSeriesIndex :=
            StrToInt(TheCell) + InitialSeriesCount;
        end;
{Add a new series:}
        SeriesInfo^[NoFileSeries].Index :=
          Self.Add(SeriesInfo^[NoFileSeries].XSeriesIndex);
        TSeries(Self.Items[SeriesInfo^[NoFileSeries].Index]).Name :=
          GetWord(SeriesNameLine, ',');
        Inc(NoFileSeries);
      end; {found a Y data column}
    end; {for}

{Get the type of data:}
    TheLine := ReadLine(AStream);
    GetWord(TheLine, '=');
{'Binary=X': X=0 => Text, X=1 => Binary:}
    i := StrToInt(TheLine);

{now we try to convert all the data:}
    if (i = 0) then
    begin {text}
      TheStrings := TStringList.Create;
{although not documented, TStrings.LoadFromStream starts from
the CURRENT TStream.Position ! Ain't that nice !}
      TheStrings.LoadFromStream(AStream);
      TheResult := Self.ConvertTextData(ColCount, NoFileSeries, 0, ',', TheStrings, SeriesInfo);
      AsText := TRUE;
    end
    else if (i = 1) then
    begin {binary}
      TheResult := Self.ConvertBinaryData(ColCount, NoFileSeries, AStream, SeriesInfo);
      AsText := FALSE;
    end
    else
    begin
      raise EFOpenError.Create(sLoadFromFile1);
    end;

{new data has not changed:}
    Self.DataChanged := FALSE;
  except
    CleanUp;
    raise;
  end; {try}

  LoadFromStream := TheResult;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetNearestPoint
  Description: gets the nearest series and point to the input point
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: user selection of data on screen
 Return Value: the Index of the nearest point
 Known Issues: 1. this is a long and messy function: while some parts could be done
               lower done in TSeries, other bits can't. i am still not
               completely happy with it.
               2. We "GenerateXXXOutline" here for Columns and Pies, but not for XY types.
 ------------------------------------------------------------------------------}
function TSeriesList.GetNearestPoint(
  ThePlotType: TPlotType;
  ColumnGap,
  iX, iY: Integer;
  var TheSeries: Integer;
  var MinDistance: Single;
  var pSeries: TSeries): Integer;
var
  CellWidth, CellHeight,
  iCol, jRow, NoPieCols,
  iSeries,
  NearestPoint,
  PieLeft, PieTop, PieWidth, PieHeight: Integer;
  Distance, dX, Span, X, Y,
  YSum, YNegSum, YSumOld, YNegSumOld, YTotal, YNegTotal: Single;
  pSeriesi: TSeries;

  function GetColumnIndex: Boolean;
  begin
    GetColumnIndex := FALSE;
{all column plots use Series[0].XData:}
    pSeriesi := TSeries(Self.Items[0]);
    X := FXAxis.XofF(iX);
{bug out if outside span:}
    if (X < pSeriesi.XData^[0]) then exit;
    dX := ((100 - ColumnGap) / 100) * (
      pSeriesi.XData^[pSeriesi.NoPts-1] -
      pSeriesi.XData^[pSeriesi.NoPts-2]);
    if (X > pSeriesi.XData^[pSeriesi.NoPts-1] + dX) then exit;
{find the nearest point:}
    NearestPoint := pSeriesi.GetNearestPointToFX(iX);
    if (NearestPoint <> 0) then
      if (X < pSeriesi.XData^[NearestPoint]) then
        Dec(NearestPoint);
    if (NearestPoint = pSeriesi.NoPts-1) then
      Span := pSeriesi.XData^[NearestPoint] - pSeriesi.XData^[NearestPoint-1]
    else
      Span := pSeriesi.XData^[NearestPoint+1] - pSeriesi.XData^[NearestPoint];
    dX := ((100 - ColumnGap) / 100) * Span;
{was the click in the gap between bars ?}
    if (X > pSeriesi.XData^[NearestPoint] + dX) then
      exit;
    GetColumnIndex := TRUE;
  end;

begin
  Distance := 1.0e38;
  MinDistance := 1.0e38;
  TheSeries := -1;
  GetNearestPoint := -1;
  pSeries := nil;

  if (Self.Count = 0) then exit;

  iSeries := 0;
  case ThePlotType of
    ptXY, ptError, ptMultiple, ptBubble:
      begin
{loop over series: note that ptError skips every second series - the error ones}
        while (iSeries < Count) do
        begin
          pSeriesi := TSeries(Self.Items[iSeries]);
{Find the nearest point IN THIS SERIES:}
          NearestPoint := pSeriesi.GetNearestXYPoint(
            iX, iY,
            0, 0,
            Distance);
{Mirror, Mirror on the wall,
 who is the nearest one of all ?}
          if (Distance < MinDistance) then
          begin
            GetNearestPoint := NearestPoint;
            MinDistance := Distance;
            TheSeries := iSeries;
            pSeries := pSeriesi;
          end;
{Note: we don't pSeries.GenerateXYOutline here, because that would be running
 that method every time the screen got clicked on, which is a bit of a drain.
 However, we do run pSeries.GenerateColumnOutline and pSeries.GeneratePieOutline
 because they are simple assignments.}

{ptError: every second series is just the X and Y error:}
          if ((ThePlotType = ptError) or
              (ThePlotType = ptBubble)) then
            Inc(iSeries, 2)
           else
            Inc(iSeries);
        end; {while over series}
      end;

    ptColumn:
      begin
        if (not GetColumnIndex) then
          exit;
{now home in: which series was it:}
        TheSeries := 0;
        if (Self.Count > 1) then
          TheSeries := Trunc(Self.Count * (X - pSeriesi.XData^[NearestPoint]) / dX);
{we now know which point in  which series:}
        pSeries := TSeries(Self.Items[TheSeries]);
        Y := FYAxis.YofF(iY);
        if (Y > pSeries.YData^[NearestPoint]) then
          exit;
        if ((Y < 0) and (Y < pSeries.YData^[NearestPoint])) then
          exit;
        GetNearestPoint := NearestPoint;
        MinDistance := 0;
        pSeries.GenerateColumnOutline(
          FXAxis.FofX(pSeries.XData^[NearestPoint] + TheSeries * dX / Self.Count),
          FYAxis.FofY(0),
          FXAxis.FofX(pSeries.XData^[NearestPoint] + (TheSeries+1) * dX / Self.Count),
          FYAxis.FofY(pSeries.YData^[NearestPoint]));
      end;

    ptStack, ptNormStack:
      begin
        if (not GetColumnIndex) then
          exit;
        Y := FYAxis.YofF(iY);
{now home in: which series was it:}
        TheSeries := 0;
        YSum := 0;
        YNegSum := 0;
        YTotal := 0;
        YNegTotal := 0;
        if (ThePlotType = ptNormStack) then
{ptStack and ptNormStack are pretty similar expcept for ...}
        begin
          if ((Y < 0) or (Y > 100)) then
            exit;
{count every series:}
          for iSeries := 0 to Count-1 do
          begin
            if (TSeries(Items[iSeries]).YData^[NearestPoint] >= 0) then
              YTotal := YTotal + TSeries(Items[iSeries]).YData^[NearestPoint]
             else
              YNegTotal := YNegTotal + TSeries(Items[iSeries]).YData^[NearestPoint];
          end; {count every series}
{prepare for conversion of data to percent:}
          YTotal := YTotal / 100;
          YNegTotal := - YNegTotal / 100;
        end;
{loop over every series:}
        for iSeries := 0 to Count-1 do
        begin
          pSeries := TSeries(Items[iSeries]);
          if (pSeries.YData^[NearestPoint] >= 0) then
          begin
            YSumOld := YSum;
            if (ThePlotType = ptStack) then
              YSum := YSum + pSeries.YData^[NearestPoint]
             else {ptNormStack}
              YSum := YSum + pSeries.YData^[NearestPoint] / YTotal; 
            if ((YSumOld < Y) and (Y < YSum)) then
            begin {Bingo !}
              GetNearestPoint := NearestPoint;
              MinDistance := 0;
              pSeries.GenerateColumnOutline(
                FXAxis.FofX(pSeries.XData^[NearestPoint]),
                FYAxis.FofY(YSumOld),
                FXAxis.FofX(pSeries.XData^[NearestPoint] + dX),
                FYAxis.FofY(YSum));
              break;
            end;
          end
          else
          begin
            YNegSumOld := YNegSum;
            if (ThePlotType = ptStack) then
              YNegSum := YNegSum + pSeries.YData^[NearestPoint]
             else {ptNormStack}
              YNegSum := YNegSum + pSeries.YData^[NearestPoint] / YNegTotal;
            if ((YNegSum < Y) and (Y < YNegSumOld)) then
            begin {Bingo !}
              GetNearestPoint := NearestPoint;
              MinDistance := 0;
              pSeries.GenerateColumnOutline(
                FXAxis.FofX(pSeries.XData^[NearestPoint]),
                FYAxis.FofY(YNegSumOld),
                FXAxis.FofX(pSeries.XData^[NearestPoint] + dX),
                FYAxis.FofY(YNegSum));
              break;
            end;
          end; {YData >= 0}
        end; {for iSeries}
      end;

    ptPie:
      begin
{each Pie sits in a cell:}
        NoPieCols := Trunc(0.99 + Count / NoPieRows);
        CellWidth := (PlotBorder.Right - PlotBorder.Left) div NoPieCols;
        CellHeight := (PlotBorder.Bottom - PlotBorder.Top) div NoPieRows;
{... but does not occupy the entire cell:}
        PieWidth := Round(PIE_SIZE * CellWidth);
        PieHeight := Round(PIE_SIZE * CellHeight);
        if (PieHeight > PieWidth) then
          PieHeight := PieWidth;

        iSeries := 0;
        for iCol := 0 to NoPieCols-1 do
        begin
          for jRow := 0 to NoPieRows-1 do
          begin
            if (iSeries >= Count) then break;
{indent the (left, top) a bit:}
            PieLeft := PlotBorder.Left + iCol * CellWidth +
              (CellWidth-PieWidth) div 2;
            PieTop := PlotBorder.Top + jRow * CellHeight +
              (CellHeight-PieHeight) div 2;
            pSeries := TSeries(Self.Items[iSeries]);
{Find the nearest point IN THIS SERIES:}
            NearestPoint := pSeries.GetNearestPieSlice(
              iX, iY,
              PieLeft, PieTop, PieWidth, PieHeight,
              Distance);
            if (Distance = 0) then
            begin
              GetNearestPoint := NearestPoint;
              MinDistance := Distance;
              TheSeries := iSeries;
              pSeries.GeneratePieOutline(
                PieLeft,
                PieTop,
                PieWidth,
                PieHeight,
                NearestPoint);
              break;
            end;
            Inc(iSeries);
          end; {jRow}
        end; {iCol}
      end; {ptPie}
    ptPolar:
      begin
      end;
  end; {case}
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetSeriesOfZ
  Description: gets the series with ZData ZValue
       Author: Mat Ballard
 Date created: 04/25/2001
Date modified: 04/25/2001 by Mat Ballard
      Purpose: parsing data from strange files
 Return Value: the guilty series, or nil
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetSeriesOfZ(ZValue: Single): TSeries;
var
  i: Integer;
begin
  GetSeriesOfZ := nil;
  for i := 0 to Self.Count-1 do
  begin
    if (ZValue = TSeries(Self.Items[i]).ZData) then
    begin
      GetSeriesOfZ := TSeries(Self.Items[i]);
      break;
    end;    
  end;       
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DoChange
  Description: event firing proedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: fires the OnDataChange event
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DoStyleChange;
begin
  if Assigned(FOnStyleChange) then OnStyleChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DoDataChange
  Description: event firing proedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: fires the OnDataChange event
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DoDataChange;
begin
  FDataChanged := TRUE;
  if Assigned(FOnDataChange) then OnDataChange(Self);
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetTotalNoPts
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the value of the TotalNoPts Property
 Return Value: Integer
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetTotalNoPts: Integer;
var
  i,
  Sum: Integer;
begin
{loop over all series:}
  Sum := 0;
  for i := 0 to Count-1 do
  begin
    Sum := Sum + TSeries(Items[i]).NoPts;
  end;
  GetTotalNoPts := Sum;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetMaxNoPts
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 09/21/2000
Date modified: 09/21/2000 by Mat Ballard
      Purpose: gets the value of the MaxNoPts Property
 Return Value: Integer
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetMaxNoPts: Integer;
var
  i,
  TheMax: Integer;
begin
  TheMax := 0;
  if (Count > 0) then
  begin
    TheMax := TSeries(Items[0]).NoPts;
    for i := 1 to Count-1 do
    begin
      if (TheMax < TSeries(Items[i]).NoPts) then
        TheMax := TSeries(Items[i]).NoPts;
    end;
  end;
  GetMaxNoPts := TheMax;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetMinNoPts
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 09/21/2000
Date modified: 09/21/2000 by Mat Ballard
      Purpose: gets the value of the MinNoPts Property
 Return Value: Integer
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetMinNoPts: Integer;
var
  i,
  TheMin: Integer;
begin
  TheMin := 0;
  if (Count > 0) then
  begin
    TheMin := TSeries(Items[0]).NoPts;
    for i := 1 to Count-1 do
    begin
      if (TheMin < TSeries(Items[i]).NoPts) then
        TheMin := TSeries(Items[i]).NoPts;
    end;
  end;
  GetMinNoPts := TheMin;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.ParseData
  Description: oversees the importation and pasting of data
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 04/27/2001 by Mat Ballard
      Purpose: runs the ParserForm, and adds the new data as new Axis
 Return Value: TRUE is successful
 Known Issues: moved from TCustomPlot
 ------------------------------------------------------------------------------}
function TSeriesList.ParseData(
  TheData: TStringList;
  TheHelpFile: String): Boolean;
var
  i,
  InitialSeriesCount,
  iColumn,
  jRow,
  NoPastedSeries,
  NoXs, NoYs, NoZs,
  XSeriesCol: Integer;
  Delimiter,
  ZDataLine,
  ZValue: String;
  ParserForm: TParserForm;
  SeriesInfo: pSeriesInfoArray;
  SeriesOfCol: pIntegerArray;
begin
  ParseData := FALSE;
  InitialSeriesCount := Self.Count;

  ParserForm := TParserForm.Create(nil);
  jRow := 0;
  try
    for jRow := 0 to TheData.Count-1 do
    begin
      ParserForm.DataListBox.Items.Add(TheData.Strings[jRow]);
    end;
    jRow := 0;
  except
{the file was to big to place into the listbox:}
    ShowMessageFmt(sParseData1, [jRow-1]);
  end;

  ParserForm.HelpFile := TheHelpFile;

  if (ParserForm.ShowModal = mrOK) then
  begin
    Delimiter := ParserForm.Delimiters[ParserForm.TheDelimiter];
    NoXs := 0;
    NoYs := 0;
    NoZs := 0;
    for iColumn := 1 to ParserForm.InfoGrid.ColCount-1 do
    begin
      if (ParserForm.InfoGrid.Cells[iColumn, X_OR_Y_OR_Z] = 'X') then
        Inc(NoXs)
      else if (ParserForm.InfoGrid.Cells[iColumn, X_OR_Y_OR_Z] = 'Y') then
        Inc(NoYs)
      else if (ParserForm.InfoGrid.Cells[iColumn, X_OR_Y_OR_Z] = 'Z') then
        Inc(NoZs)
    end;
    ZDataLine := ParserForm.InfoGrid.Rows[X_OR_Y_OR_Z].CommaText;
{The data might be in the form of:
    x1,y1,z1
    x2,y2,z2
    ...
 or in the form of columns of series (ie: like we save it):}
    if (NoZs > 0) then
    begin
      if ((NoXs = NoYs) and (NoYs = NoZs)) then
      begin
        ParseData := ConvertXYZData(ParserForm.TheFirstDataLine,
          Delimiter, ParserForm.InfoGrid.Rows[X_OR_Y_OR_Z], TheData);
      end
      else
        ShowMessage('Sorry - your data is too complex for me to understand !');
    end
    else
    begin
{allocate memory for the dynamic arrays, which are small:}
      GetMem(SeriesInfo, ParserForm.InfoGrid.ColCount * SizeOf(TSeriesInfo));
      GetMem(SeriesOfCol, (ParserForm.InfoGrid.ColCount+1) * SizeOf(Integer));
  {this allocates more memory than SeriesInfo needs, but so what ?}
      for i := 0 to ParserForm.InfoGrid.ColCount-1 do
      begin
        SeriesInfo^[i].XCol := 0;
        SeriesInfo^[i].XTextCol := -1;
      end;
      if (NoXs = 0) then
        SeriesInfo^[0].XCol := -2;
  {Grab the line of Z Data, if any:}
  {This is complex: ConvertTextData expects the Z Data line, if present,
   to be String[FirstLine] AND to start with 'ZData' - because this is the
   TPlot file format. However, this may NOT be the case with third party text
   files: the Z Data could come anywhere, or it could even be manually entered
   by the user. Because of these problems, we process the Z Data in this
   routine, then remove any Z references from the Z Data Line.}
      if (ParserForm.TheZDataLine >= 0) then
      begin
        ZDataLine := Uppercase(TheData.Strings[ParserForm.TheZDataLine]);
        iColumn := Pos('ZDATA', ZDataLine);
        if (iColumn > 0) then
        begin
  {remove ZDATA because we process it in this routine:}
          TheData.Strings[ParserForm.TheZDataLine] :=
            Copy(ZDataLine, 1, iColumn-1) +
            Copy(ZDataLine, iColumn+5, 9999);
        end;
      end  
      else
      begin
  {The user may have entered Z Data manually:}    
        ZDataLine := '';
        for iColumn := 1 to ParserForm.InfoGrid.ColCount-1 do
        begin
          if (Length(ParserForm.InfoGrid.Cells[iColumn, Z_DATA_LINE]) > 0) then
          begin
            ZDataLine := ZDataLine +
              ParserForm.InfoGrid.Cells[iColumn, Z_DATA_LINE] + Delimiter;
          end;
        end;
      end;

  {Determine the number of series:}
      NoPastedSeries := 0;
  {examine the columns one by one:}
      for iColumn := 1 to ParserForm.InfoGrid.ColCount-1 do
      begin
        SeriesOfCol^[iColumn] := -1;
        if (ParserForm.InfoGrid.Cells[iColumn, X_OR_Y_OR_Z] = 'X') then
        begin
  {This is a column of X data:}
          SeriesOfCol^[iColumn] := NoPastedSeries;
          SeriesInfo^[NoPastedSeries].XCol := iColumn;
        end
        else if (ParserForm.InfoGrid.Cells[iColumn, X_OR_Y_OR_Z] = 'XTEXT') then
        begin
  {This is a column of X STRING data:}
          SeriesOfCol^[iColumn] := NoPastedSeries;
          SeriesInfo^[NoPastedSeries].XTextCol := iColumn;
        end
        else if (ParserForm.InfoGrid.Cells[iColumn, X_OR_Y_OR_Z] = 'Y') then
        begin
  {we've found a series - this is a column of Y data:}
          SeriesOfCol^[iColumn] := NoPastedSeries;
          SeriesInfo^[NoPastedSeries].YCol := iColumn;
          if (Length(ParserForm.InfoGrid.Cells[iColumn,DEPENDS_ON_X]) > 0) then
          begin
            XSeriesCol := StrToInt(ParserForm.InfoGrid.Cells[iColumn,DEPENDS_ON_X]);
            if (SeriesOfCol^[XSeriesCol] < 0) then
              raise EComponentError.Create('TCustomPlot.ParseData: ' + sParseData2);
            if (SeriesOfCol^[XSeriesCol] = NoPastedSeries) then
            begin
  {This column of Y Data has its own X Data (column):}
              SeriesInfo^[NoPastedSeries].XSeriesIndex := -1;
            end
            else
            begin
  {This column of Y Data uses another Series X Data (column):}
              SeriesInfo^[NoPastedSeries].XSeriesIndex :=
                SeriesOfCol^[XSeriesCol] + InitialSeriesCount;
            end;
          end;
{This Y Column has come before any X columns:}
          if (iColumn = 1) then
            SeriesInfo^[NoPastedSeries].XSeriesIndex := -1;
{There is no X data at all !}
          if ((NoXs = 0) and (iColumn > 1)) then
            SeriesInfo^[NoPastedSeries].XSeriesIndex :=
              InitialSeriesCount;
              
{We add the new series:}
          SeriesInfo^[NoPastedSeries].Index :=
            Self.Add(SeriesInfo^[NoPastedSeries].XSeriesIndex);
  {and set its name:}
          TSeries(Self.Items[SeriesInfo^[NoPastedSeries].Index]).Name :=
            ParserForm.InfoGrid.Cells[iColumn, SERIES_NAMES];
  {and its Z value:}
          if (Length(ZDataLine) > 0) then
          begin
            repeat
              ZValue := GetWord(ZDataLine, Delimiter);
              if IsReal(ZValue) then
              begin
                try
                  TSeries(Self.Items[SeriesInfo^[NoPastedSeries].Index]).ZData :=
                    StrToFloat(ZValue);
                  ZValue := '';
                except
                  On EConvertError do
                    ZValue := '?';
                end;
              end
              else
                ZValue := '?';
            until ((Length(ZValue) = 0) or (Length(ZDataLine) = 0));
          end
          else
          begin
            TSeries(Self.Items[SeriesInfo^[NoPastedSeries].Index]).ZData :=
              SeriesInfo^[NoPastedSeries].Index;
          end;

          Inc(NoPastedSeries);
        end
        else
        begin
          SeriesInfo^[NoPastedSeries].XCol := -1;
        end;
      end;

  {now we try to convert all the data:}
      if (NoPastedSeries > 0) then
        ParseData := ConvertTextData(ParserForm.InfoGrid.ColCount-1, NoPastedSeries, ParserForm.TheFirstDataLine,
          Delimiter, TheData, SeriesInfo);
      FreeMem(SeriesInfo, ParserForm.InfoGrid.ColCount * SizeOf(TSeriesInfo));
      FreeMem(SeriesOfCol, ParserForm.InfoGrid.ColCount * SizeOf(Integer));
    end; {No Z's on us !}
  end; {ShowModal = mrOK}
  ParserForm.Free;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.ConvertBinaryData
  Description: Adds binary data to the new Series
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 04/27/2001 by Mat Ballard
      Purpose: Given a AxisLocationArray, converts the text data to numeric data and adds it to the new Axis
 Return Value: TRUE is successful
 Known Issues: This procedure assumes that TheStream.Position points to the start
               of the binary data.
               Moved from TCustomPlot.
 ------------------------------------------------------------------------------}
function TSeriesList.ConvertBinaryData(
  ColCount,
  SeriesCount: Integer;
  TheStream: TMemoryStream;
  SeriesInfo: pSeriesInfoArray): Boolean;
var
  DataSize,
  i,
  iColumn: Integer;
  MemPosition: Longint;
  NullValue,
  ZValue: Single;
  ptr: Pointer;
  pTheChar: PChar;
  pValues: pSingleArray;
  pLine: array [0..32] of char;
  XTextValue: String;
begin
  //ConvertBinaryData := FALSE;
  GetMem(pValues, ColCount * SizeOf(Single));
  DataSize := SizeOf(Single);

{calculate our "ignore" value:}
  ptr := @NullValue;
  pTheChar := ptr;
  for i := 1 to DataSize do
  begin
    pTheChar^ := 'x';
    Inc(pTheChar);
  end;

{check fir ZData:}
  MemPosition := TheStream.Position;
  TheStream.Read(pLine, 6);
  pLine[6] := Chr(0);
  if (StrComp(pLine, 'ZData:') = 0) then
  begin
    for i := 0 to SeriesCount-1 do
    begin
      TheStream.Read(ZValue, DataSize);
      TSeries(Self.Items[SeriesInfo^[i].Index]).ZData := ZValue;
    end;
  end
  else
    TheStream.Position := MemPosition;

  while (TheStream.Position < TheStream.Size) do
  begin
    for i := 0 to SeriesCount-1 do
    begin
      if (SeriesInfo^[i].XCol > 0) then
        TheStream.Read(SeriesInfo^[i].XValue, DataSize);
      if (SeriesInfo^[i].XTextCol > 0) then
        XTextValue := ReadLine(TheStream);
      TheStream.Read(SeriesInfo^[i].YValue, DataSize);

      if (SeriesInfo^[i].XTextCol > 0) then
        TSeries(Self.Items[SeriesInfo^[i].Index]).AddStringPoint(
          XTextValue,
          SeriesInfo^[i].XValue,
          SeriesInfo^[i].YValue,
          FALSE, FALSE)
      else
        TSeries(Self.Items[SeriesInfo^[i].Index]).AddPoint(
          SeriesInfo^[i].XValue,
          SeriesInfo^[i].YValue,
          FALSE, FALSE);
    end; {for iColumn}
  end; {for lines of data}

  FreeMem(pValues, ColCount * SizeOf(Single));
{for a subsequent SaveClick:}
  ConvertBinaryData := TRUE;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.ConvertTextData
  Description: Adds text data to the new Series
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 04/27/2001 by Mat Ballard
      Purpose: Given a pSeriesInfoArray, converts the text data to numeric data and adds it to the new Axis
 Return Value: TRUE is successful
 Known Issues: moved from TCustomPlot
 ------------------------------------------------------------------------------}
function TSeriesList.ConvertTextData(
  ColCount,
  SeriesCount,
  FirstLine: Integer;
  Delimiter: String;
  TheData: TStringList;
  SeriesInfo: pSeriesInfoArray): Boolean;
var
  i,
  jRow: Integer;
  TheCell,
  TheLine: String;
  TheSortedLine: TStringList;
begin
{Does this contain Z Data ?}
  if (Pos('ZData', TheData.Strings[FirstLine]) > 0) then
  begin
    TheLine := TheData.Strings[0];
    for i := 0 to SeriesCount-1 do
    begin
      if (SeriesInfo^[i].XCol > 0) then
        GetWord(TheLine, Delimiter);
      if (SeriesInfo^[i].XTextCol > 0) then
        GetWord(TheLine, Delimiter);
      TheCell := GetWord(TheLine, Delimiter);
      TSeries(Self.Items[SeriesInfo^[i].Index]).ZData :=
        StrToFloat(TheCell);
    end;
    Inc(FirstLine);
  end;

  TheSortedLine := TStringList.Create;
  for i := 0 to ColCount-1 do
    TheSortedLine.Add('');

  for jRow := FirstLine to TheData.Count-1 do
  begin
    TheLine := TheData.Strings[jRow];

    for i := 0 to ColCount-1 do
    begin
      TheSortedLine.Strings[i] := GetWord(TheLine, Delimiter);
    end;

    for i := 0 to SeriesCount-1 do
    begin
      try
        if (SeriesInfo^[i].XCol = -2) then
          SeriesInfo^[i].XValue := (jRow - FirstLine)
        else if (SeriesInfo^[i].XCol > 0) then
          SeriesInfo^[i].XValue := StrToFloat(TheSortedLine.Strings[SeriesInfo^[i].XCol-1]);
        if (Length(TheSortedLine.Strings[SeriesInfo^[i].YCol-1]) > 0) then
        begin
          SeriesInfo^[i].YValue := StrToFloat(TheSortedLine.Strings[SeriesInfo^[i].YCol-1]);
          if (SeriesInfo^[i].XTextCol > 0) then
            TSeries(Self.Items[SeriesInfo^[i].Index]).AddStringPoint(
              TheSortedLine.Strings[SeriesInfo^[i].XTextCol-1],
              SeriesInfo^[i].XValue,
              SeriesInfo^[i].YValue,
              FALSE, FALSE)
           else
            TSeries(Self.Items[SeriesInfo^[i].Index]).AddPoint(
              SeriesInfo^[i].XValue,
              SeriesInfo^[i].YValue,
              FALSE, FALSE);
        end;      
      except
        SeriesInfo^[i].XValue := -999999;
      end;
    end; {for i}
  end; {for lines of data}
{cleanup:}
  TheSortedLine.Free;

{Make the new Series visible:}
  for i := 0 to SeriesCount-1 do
  begin
    if (SeriesInfo^[i].Index >= 0) then
      TSeries(Self.Items[SeriesInfo^[i].Index]).Visible := TRUE;
  end;

{for a subsequent SaveClick:}
  ConvertTextData := TRUE;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.ConvertXYZData
  Description: Processes XYZ text data
       Author: Mat Ballard
 Date created: 04/19/2001
Date modified: 04/27/2001 by Mat Ballard
      Purpose: data importation
 Return Value: TRUE is successful
 Known Issues: moved from TCustomPlot
 ------------------------------------------------------------------------------}
function TSeriesList.ConvertXYZData(
  FirstLine: Integer;
  Delimiter: String;
  InfoGridRows: TStrings;
  TheData: TStringList): Boolean;
var
  iCol,
  jRow,
  NoXYZs: Integer;
  TheCell, TheLine: String;
  X, Y, Z: Single;
  pSeries: TSeries;
begin
  //ConvertXYZData := FALSE;
  for jRow := FirstLine to TheData.Count-1 do
  begin
    TheLine := TheData[jRow];
    NoXYZs := 0;
    X := -999999;
    Y := -999999;
    Z := -999999;
    for iCol := 1 to InfoGridRows.Count-1 do
    begin
      TheCell := GetWord(TheLine, Delimiter);
      if (InfoGridRows.Strings[iCol] = 'X') then
      begin
        X := StrToFloat(TheCell);
        Inc(NoXYZs);
      end
      else if (InfoGridRows.Strings[iCol] = 'Y') then
      begin
        Y := StrToFloat(TheCell);
        Inc(NoXYZs);
      end
      else if (InfoGridRows.Strings[iCol] = 'Z') then
      begin
        Z := StrToFloat(TheCell);
        Inc(NoXYZs);
      end;
{test for a triple:}
      if (NoXYZs = 3) then
      begin
        NoXYZs := 0;
        pSeries := Self.GetSeriesOfZ(Z);
        if (pSeries = nil) then
        begin
          pSeries := TSeries(Self.Items[Self.Add(-1)]);
          pSeries.ZData := Z;
        end;
        pSeries.AddPoint(X, Y, FALSE, TRUE);
      end;
    end;
  end;
  ConvertXYZData := TRUE;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.StyleChange
  Description: target of all of the TSeries(Items[i]) OnStyleChange events
       Author: Mat Ballard
 Date created: 03/07/2001
Date modified: 03/07/2001 by Mat Ballard
      Purpose: responds to changes in style of the member series
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.StyleChange(
  Sender: TObject);
begin
  if (FIgnoreChanges) then exit;

  DoStyleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DataChange
  Description: target of all of the TSeries(Items[i]) OnStyleChange events
       Author: Mat Ballard
 Date created: 03/07/2001
Date modified: 03/07/2001 by Mat Ballard
      Purpose: responds to changes in data of the member series
 Known Issues: get up to 3 screen re-draws
 ------------------------------------------------------------------------------}
procedure TSeriesList.DataChange(
  Sender: TObject);
begin
  if (IgnoreChanges) then exit;

  DoDataChange;
end;


end.
