unit DBPlot;

{$I Plot.inc}

{-----------------------------------------------------------------------------
The contents of this file are subject to the Q Public License
("QPL"); you may not use this file except in compliance
with the QPL. You may obtain a copy of the QPL from 
the file QPL.html in this distribution, derived from:

http://www.trolltech.com/products/download/freelicense/license.html

The QPL prohibits development of proprietary software. 
There is a Professional Version of this software available for this. 
Contact sales@chemware.hypermart.net for more information.

Software distributed under the QPL is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either expressed or implied. See the QPL for
the specific language governing rights and limitations under the QPL.

The Original Code is: pSeries.pas, released 12 September 2000.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@chemware.hypermart.net.

Last Modified: 02/25/2000
Current Version: 2.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind. 

Purpose:
This unit contains the TDBPlot component - the data-aware version of the
TPlot component

Known Issues:

History:



-----------------------------------------------------------------------------}

interface

uses
  Classes, DB, SysUtils,
{$IFDEF WINDOWS}
  WinTypes, WinProcs,
  DBCtrls, DBTables,
  //Clipbrd, Controls, Dialogs, Extctrls, Forms, Graphics, Menus, Printers, Stdctrls,
{$ENDIF}
{$IFDEF WIN32}
  Windows,
  DBCtrls, DBTables,
  //Clipbrd, Controls, Dialogs, Extctrls, Forms, Graphics, Menus, Printers, Stdctrls,
{$ENDIF}
{$IFDEF LINUX}
  Types, Untranslated,
  DBClient,
  QDBCtrls,
  //QClipbrd, QControls, QDialogs, QExtctrls, QForms, QGraphics,
  //QImgList, QMenus, QStdctrls,
{$ENDIF}

  Misc, Plot, PlotDefs, Data;

type
  TCustomDBPlot = class(TCustomPlot)
  private
    FDataLink: TFieldDataLink;
    FReadOnly: Boolean;
    FUpdating: Boolean;

    function GetDataField: string;
    function GetDataSource: TDataSource;
    function GetReadOnly: Boolean;
    procedure SetDataField(const Value: string);
    procedure SetDataSource(Value: TDataSource);
    procedure SetReadOnly(Value: Boolean);

  protected
    property ReadOnly: Boolean read GetReadOnly write SetReadOnly default False;
{Is the database read-only ?}
    property DataField: string read GetDataField write SetDataField;
{The usual DataField property, which is the FieldName property of the DataLink.}
    property DataSource: TDataSource read GetDataSource write SetDataSource;
{The usual DataSource property, which is the DataSource property of the DataLink.}

    procedure LoadFromDB;
    procedure StoreInDB;

    procedure DataChange(Sender: TObject);
    procedure DoDataChange(Sender: TObject); override;
    procedure EditingChange(Sender: TObject);
    procedure UpdateData(Sender: TObject);

  public
    constructor Create(AOwner: TComponent); override;
{The standard DB constructor, in which the DataLink is created.}
    destructor Destroy; override;
{The standard DB destructor, in which the DataLink is freed.}

    procedure Clear; 
{Saves any changed files (at user request) and then clears the SeriesList.}

{Series manipulation:}
    function AddSeries(XSeriesIndex: Integer): Integer; override;
{If permitted, this adds a new, empty series to the list by calling its ancestor.}
    function AddExternalSeries(XPointer, YPointer: pSingleArray;
      NumberOfPoints: Integer): Integer; override;
{If permitted, this adds a new, empty series to the list by calling its ancestor,
 and sets its data to point to the external XPointer, YPointer data.}
    function AddInternalSeries(XPointer, YPointer: pSingleArray;
      NumberOfPoints: Integer): Integer; override;
{If permitted, this adds a new, empty series to the list by calling its ancestor,
 and copies the data from the XPointer, YPointer data.}
    function CloneSeries(TheSeries: Integer): Integer; override;
{If permitted, this adds a new, empty series to the list by calling its ancestor,
 copies the data and properties from TheSeries into the new clone, and changes the color and Y Displacement.}
    procedure DeleteSeries(Index: Integer); override;
{If permitted, this deletes TheSeries from the list by calling its ancestor.}

{Manipulation of data IN the series:}
    function AddDrawPoint(iSeries: Integer; X, Y: Single): Integer; 
{If permitted, this calls TSeries.AddDrawPoint.}
    function AddPoint(iSeries: Integer; X, Y: Single; FireEvent, AdjustAxes: Boolean): Integer; 
{If permitted, this calls TSeries.AddPoint.}
    function AddStringPoint(iSeries: Integer; XString: String; X, Y: Single; FireEvent, AdjustAxes: Boolean): Integer; 
{If permitted, this calls TSeries.AddStringPoint.}
    function InsertPoint(iSeries: Integer; X, Y: Single): Integer; 
{If permitted, this calls TSeries.InsertPoint.}
    procedure ReplacePoint(iSeries: Integer; N: Integer; NewX, NewY: Single); 
{If permitted, this calls TSeries.ReplacePoint.}
    procedure Compress(iSeries: Integer; CompressRatio: Integer);
{If permitted, this calls TSeries.Compress.}
    procedure Contract(iSeries: Integer; TheStart, TheFinish: Integer);
{If permitted, this calls TSeries.Contract.}
    function DelPoint(iSeries: Integer; X, Y: Single; Confirm: Boolean): Integer;
{If permitted, this calls TSeries.DelPoint.}
    function DelPointNumber(iSeries: Integer; ThePoint: Integer; Confirm: Boolean): Integer;
{If permitted, this calls TSeries.DelPointNumber.}
    function DelData(iSeries: Integer): Boolean;
{If permitted, this calls TSeries.DelData.}
    procedure Smooth(iSeries: Integer; SmoothOrder: Integer);
{If permitted, this calls TSeries.Smooth.}
    procedure Sort(iSeries: Integer);
{If permitted, this calls TSeries.Sort.}
    procedure Differentiate(iSeries: Integer);
{If permitted, this calls TSeries.Differentiate.}
    procedure Integrate(iSeries: Integer);
{If permitted, this calls TSeries.Integrate.}
    procedure MovingAverage(iSeries: Integer; Span: Integer);
{If permitted, this calls TSeries.MovingAverage.}

    procedure UpdateRecord;
{This causes the database to be updated.}
  published

  end;

  TDBPlot = class(TCustomDBPlot)
  protected

  public
    Property AxisList;
{This is a list of all the axes. The 0th is X, the 1st Y.}
    Property Bitmap;
{This returns a Bitmap of the Plot.}
    Property PlotPopUpMenu;
{The public exposure of the popupmenu for PlotMenu's use.}
    Property Series;
{This provided direct access to the series maintained by SeriesList.
 DO NOT access any TSeries methods - just properties !}

{The following property remains protected and so hidden in the DataBase descendant:}
    {Property SeriesList;}
{This is the data in a list of data series.}
    Property ScreenJob;
{This is the job that is in progress in response to user input.}
    Property ZAxis;
{This is the Z Axis. Every nice 3D graph should have an Z Axis.}
{}
{This is fairly experimental at the moment, so use with caution.
 It is also nil with non-3D PlotTypes.}

  published
    property DataField;
{The usual DataField property, which is the FieldName property of the DataLink.}
    property DataSource;
{The usual DataSource property, which is the DataSource property of the DataLink.}
    property ReadOnly;
{Is the database read-only ?}

{All the protected properties of TCustomPlot:}
    Property About;
{Displays the "About" dialog box for this component.}
    Property AxesProperties;
{Displays the "Axis Editor" dialog box for the FAxisList subcomponent.}
    {Property DataProperties;}
{Displays the "Data Editor" dialog box.}
    {Property SeriesProperties;}
{Displays the "Series Editor" dialog box.}
    Property Border;
{Manages the geometry of TCustomPlot: where the axes are, where they can go, etc.}
    Property ClickAndDragDelay;
{The delay (in milliseconds) before a clicked object becomes draggable.}
    Property ColumnGap;
{This is the percentage gap between groups of Columns in ptColumn PlotType.}
    Property ContourDetail;
{This is the detail of the colour interpolation in a contour graph.
     cdLow: each triangular tile has uniform colour; this is pretty grainy
     cdMedium: each pixel in each triangular tile has its own colour: this makes a good bitmap, but a lousy metafile
     cdHigh: each rectangular pixel in each triangular tile has its own colour: this makes a good metafile
 See SeriesList.DrawContour.}

{$IFNDEF DELPHI1}
    Property CreatedBy;
{For a metafile, this is a string that is stored in the metafile.}
    Property Description;
{For a metafile, this is a string that is stored in the metafile.}
{$ENDIF}

    Property DisplayHistory;
{The width of the X Axis when in History mode.}
    Property DisplayMode;
{See TDisplayMode.}
    Property DefaultExtension;
{What is the default extension of Plot files ?}
    Property Editable;
{Are screen objects like axes Editable ?}
    Property FileName;
{This is the filename to which the data is saved, or opened from.}
{}
{If FileName is blank, and an OpenClick or SaveClick occurs, then the standard
 file dialog box appears to let the user pick a name.}
    Property Grid;
{Do we want a grid in XY-type plots ?}
    Property GridStyle;
{What pen style (dash, dot, etc) do we want for Grids ?}
    property HighFont;
{The font for annotation of the Highs and Lows.}

{$IFDEF COMPILER4_UP}
    property Images;
{A publication of the Popup Menu's Images property}
{$ENDIF}

    Property HelpFile;
{When this is set to "", TPlot uses Plot.hlp as its context-sensitive help file.}
{}
{When set to some other file, it uses that instead.}

    Property Instructions;
{This is a hint-like message at the bottom of the graph.
 It used to be Caption - a string - but now has to be a
 stringlist to be fully Delphi-compatible (to avoid API calls).
 Remember Kylix !
 It disappears upon a MouseDown. See Font.}

    property Legend;
{The list of series with their line styles. This is a moveable, on-screen object.}
    Property Movable;
{Are screen objects like axes movable ?}
    Property Multiplicity;
{When the PlotType is ptMultiple, the series are grouped into multiples of Multiplicity.
 Otherwise ignored.}
    property MultiplePen;
{The pen to use for the verticle lines of ptMultiple PlotTypes (eg: High-Low).}
    property MultiJoin;
{Which two series, by number, to join with one rectangular symbol, in a ptMultiple plot.
 Eg: if Multiplicity is 4, and MultiJoin = '2,3', then you have a "Candle" or
 High-Low-Open-Close plot.}    
    property PieRowCount;
{The number of rows of Pie Graphs.}
    property NoSeries;
{The number of Series.}
    property NoYAxes;
{The total number of Y Axes (primary, secondary, tertiary, etc).}
    Property OutlineWidth;
{This is the width of the outline for screen objects like lines: axes and borders.}
    Property PlotType;
{What type of plot is this ?}
    Property PolarRange;
{What does 360 correspond to in a polar graph ? Examples are:
    2 Pi (6.28...),
    360 (degrees),
    60 (minutes),
    24 (hours)
    100 (%)
Get the idea ?    }
    Property PopupOptions;
{If true, then these popup menu items are visible.}
    Property PrintOrientation;
{Shall we print the graph in Landscape or Portrait mode ?}
    Property SaveOptions;
{Shall we save the data as Text or binary ?}
{}
{Shall we also save the Plot properties when we save the data ?
 If we do, then we:
    1. Save the properties in a seperate file;
    2. Look for a properties file to open.}
    Property Title;
{The title of the graph, including its geometry, font and visibility.}
    Property XAxis;
{This is the X Axis. Every nice graph should have an X Axis.}
    Property YAxis;
{This is the Y Axis. Every nice graph should have at least one Y Axis.}
{}
{Each Series must know what Y Axes it is being plotted against:
 Primary (this one) or Secondary.}
    Property ZAngle;
{The angle made by the Z Axis, if any, with the vertical, in a clockwise direction.}
    Property ZLink;
{Should we link 3D series together ?}

{Events:}
    Property OnAfterPaint;
    Property OnAfterDraw;
    Property OnBeforePaint;
    Property OnBeforeDraw;
    Property OnStyleChange;
    Property OnDataChange;

    Property OnFileOpen;
{When a file is opened, the app can be notified of the new file name using this event.}

    Property OnFileClose;
{When a file is closed, the app can be notified of the new file name using this event.}

    Property OnHeader;
{When data is opened, this event passes the header information back to the "user".}

    Property OnHeaderRequest;
{When data is saved or copied, this event allows the user to add a header
 to the data.}

    Property OnHTMLHeaderRequest;
{When data is copied as HTML, this event allows the user to add a header
 to the data.}

    Property OnSelection;
    Property OnDualSelection;

{all the "extra" TPanel properties in D4 (alias "VCL Bloat"):}

{Now all the TPanel properties-------------------------------------------------}
{all the TPanel properties in D1:}
    property Align;    
    {property Alignment;}
    property BevelInner;
    property BevelOuter;
    property BevelWidth;
    property BorderWidth;
    property BorderStyle;
    {property Caption; - replaced by Instructions}
    property Color;
{$IFDEF MSWINDOWS}
    property Ctl3D;
    property DragCursor;
{$ENDIF}
    property DragMode;
    property Enabled;
    property Font;
{This is the font of the hint-like message at the bottom of the graph.}
    property ParentColor;
{$IFDEF MSWINDOWS}
    property Locked;
    property ParentCtl3D;
{$ENDIF}
    property ParentFont;
    property ParentShowHint;
{ Note: D1 to D4 were quite happy for:
        PopupMenu := FPlotPopUpMenu;
  FPlotPopUpMenu was then run by inherited MouseUp.
  However, D5 introduced TControl.WMContextMenu, which ran BEFORE MouseDown.
  This went spastic when it tried to Popup the PopupMenu.

  We have therefore returned to hiding FPlotPopUpMenu, and running it manually.
    property PopupMenu;}
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnResize;

    
{$IFDEF COMPILER2_UP}
{$ENDIF}

{$IFDEF COMPILER3_UP}
  {$IFDEF MSWINDOWS}
    property FullRepaint;
    property OnStartDrag;
  {$ENDIF}
{$ENDIF}

{$IFDEF COMPILER4_UP}
    property Anchors;
    // property AutoSize; - leads to bizzare behaviour
  {$IFDEF MSWINDOWS}
    property BiDiMode;
  {$ENDIF}
    property Constraints;
  {$IFDEF MSWINDOWS}
    property UseDockManager default True;
    property DockSite;
    property DragKind;
    property ParentBiDiMode;
    property OnCanResize;
  {$ENDIF}
    property OnConstrainedResize;
  {$IFDEF MSWINDOWS}
    property OnDockDrop;
    property OnDockOver;
    property OnEndDock;
    property OnGetSiteInfo;
    property OnStartDock;
    property OnUnDock;
  {$ENDIF}
{$ENDIF}

{$IFDEF COMPILER5_UP}
{$ENDIF}
  end;

implementation

{------------------------------------------------------------------------------
  Constructor: DBPlot.Create
  Description: standard DB constructor
       Author: Mat Ballard
 Date created: 03/02/2001
Date modified: 03/02/2001 by Mat Ballard
      Purpose:
 Known Issues:
 ------------------------------------------------------------------------------}
constructor TCustomDBPlot.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FReadOnly := FALSE;
  FDataLink := TFieldDataLink.Create;
  FDataLink.Control := Self;
  FDataLink.OnDataChange := DataChange;
  FDataLink.OnEditingChange := EditingChange;
  FDataLink.OnUpdateData := UpdateData;
  //FDataLink.OnActiveChange := ActiveChange;
end;

{------------------------------------------------------------------------------
   Destructor: DBPlot.Destroy
  Description: standard DB destructor
       Author: Mat Ballard
 Date created: 03/07/2001
Date modified: 03/07/2001 by Mat Ballard
      Purpose: frees the datalink
 Known Issues:
 ------------------------------------------------------------------------------}
destructor TCustomDBPlot.Destroy;
begin
  FDataLink.OnDataChange := nil;
  FDataLink.Free;
  inherited Destroy;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomDBPlot.Clear
  Description: Clears the SeriesList.
       Author: Mat Ballard
 Date created: 05/02/2001
Date modified: 05/02/2001 by Mat Ballard
      Purpose: Series management: wraps TSeriesList.ClearSeries
 Known Issues: it is a fairly brutal way of setting the number of Y Axes
 ------------------------------------------------------------------------------}
procedure TCustomDBPlot.Clear;
begin
  if ((not FUpdating) and ReadOnly) then exit;

  FSeriesList.ClearSeries;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomDBPlot.DataChange
  Description: handles changes in the source DB data
       Author: Mat Ballard
 Date created: 03/07/2001
Date modified: 03/07/2001 by Mat Ballard
      Purpose: loads the data from the database
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomDBPlot.DataChange(Sender: TObject);
begin
{blow away the old data:}
  Self.SeriesList.ClearSeries;
{no new data ? POQ:}
  if FDataLink.Field = nil then exit;

{now the work of loading the data from the DB begins.}
  LoadFromDB;
end;

procedure TCustomDBPlot.DoDataChange(Sender: TObject);
begin
  FDataLink.Edit;
  FDataLink.Modified;
  inherited DoDataChange(Sender);
end;

procedure TCustomDBPlot.EditingChange(Sender: TObject);
begin
  ReadOnly := not FDataLink.Editing;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomDBPlot.UpdateData
  Description: handles changes in the Plot data
       Author: Mat Ballard
 Date created: 03/07/2001
Date modified: 03/07/2001 by Mat Ballard
      Purpose: stores the data in the database
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomDBPlot.UpdateData(Sender: TObject);
begin
{no new data ? POQ:}
  if FDataLink.Field = nil then exit;
{now the work of loading the data from the DB begins.}
  if FDataLink.CanModify then
    StoreInDB;
end;

function TCustomDBPlot.GetDataField: string;
begin
  Result := FDataLink.FieldName;
end;

procedure TCustomDBPlot.SetDataField(const Value: string);
begin
  {if not (csDesigning in ComponentState) then
    ResetMaxLength;}
  FDataLink.FieldName := Value;
end;

function TCustomDBPlot.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

procedure TCustomDBPlot.SetDataSource(Value: TDataSource);
begin
  if not (FDataLink.DataSourceFixed and (csLoading in ComponentState)) then
    FDataLink.DataSource := Value;
  if Value <> nil then Value.FreeNotification(Self);
end;

function TCustomDBPlot.GetReadOnly: Boolean;
begin
  Result := FReadOnly;{FDataLink.ReadOnly;}
end;

procedure TCustomDBPlot.SetReadOnly(Value: Boolean);
begin
  FReadOnly := Value;
end;


{------------------------------------------------------------------------------
     Function: TCustomDBPlot.Add
  Description:
       Author: Mat Ballard
 Date created: 03/02/2001
Date modified: 03/02/2001 by Mat Ballard
      Purpose: Adds a new, empty data Series to the graph
 Return Value: the Index of the new Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomDBPlot.AddSeries(XSeriesIndex: Integer): Integer;
begin
  if ((not FUpdating) and ReadOnly) then
    Result := -1
  else
  begin
    Result := inherited AddSeries(XSeriesIndex);
  end;
end;

{------------------------------------------------------------------------------
     Function: TCustomDBPlot.AddExternal
  Description: wrapper for TSeriesList.AddExternal
       Author: Mat Ballard
 Date created: 03/02/2001
Date modified: 03/02/2001 by Mat Ballard
      Purpose: Adds a new data Series that is maintained elsewhere to the graph
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomDBPlot.AddExternalSeries(
  XPointer,
  YPointer: pSingleArray;
  NumberOfPoints: Integer): Integer;
begin
  if ((not FUpdating) and ReadOnly) then
    Result := -1
  else
  begin
    Result := inherited AddExternalSeries(XPointer, YPointer, NumberOfPoints);
  end;
end;

{------------------------------------------------------------------------------
     Function: TCustomDBPlot.AddInternal
  Description: wrapper for TSeriesList.AddInternal
       Author: Mat Ballard
 Date created: 03/02/2001
Date modified: 03/02/2001 by Mat Ballard
      Purpose: Adds a new data Series from elsewhere to the graph, and saves it internally
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomDBPlot.AddInternalSeries(
  XPointer,
  YPointer: pSingleArray;
  NumberOfPoints: Integer): Integer;
begin
  if ((not FUpdating) and ReadOnly) then
    Result := -1
  else
  begin
    Result := inherited AddInternalSeries(XPointer, YPointer, NumberOfPoints);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomDBPlot.CloneSeries
  Description: wrapper for TSeriesList.CloneSeries
       Author: Mat Ballard
 Date created: 03/02/2001
Date modified: 03/02/2001 by Mat Ballard
      Purpose: Clones the specified Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomDBPlot.CloneSeries(TheSeries: Integer): Integer;
begin
  if ((not FUpdating) and ReadOnly) then
    Result := -1
  else
  begin
    Result := inherited CloneSeries(TheSeries);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomDBPlot.DeleteSeries
  Description: wrapper for TSeriesList.DeleteSeries
       Author: Mat Ballard
 Date created: 03/02/2001
Date modified: 03/02/2001 by Mat Ballard
      Purpose: Deletes the specified Series
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomDBPlot.DeleteSeries(Index: Integer);
begin
  if ((not FUpdating) and ReadOnly) then exit;

  inherited DeleteSeries(Index);
end;

function TCustomDBPlot.AddDrawPoint(iSeries: Integer; X, Y: Single): Integer;
begin
  if ((not FUpdating) and ReadOnly) then
    Result := -1
  else
  begin
    Result := TSeries(FSeriesList.Items[iSeries]).AddDrawPoint(X, Y, Self.Canvas);
  end;
end;

function TCustomDBPlot.AddPoint(iSeries: Integer; X, Y: Single; FireEvent, AdjustAxes: Boolean): Integer;
begin
  if ((not FUpdating) and ReadOnly) then
    Result := -1
  else
  begin
    Result := TSeries(FSeriesList.Items[iSeries]).AddPoint(X, Y, FireEvent, AdjustAxes);
  end;
end;

function TCustomDBPlot.AddStringPoint(iSeries: Integer; XString: String; X, Y: Single; FireEvent, AdjustAxes: Boolean): Integer;
begin
  if ((not FUpdating) and ReadOnly) then
    Result := -1
  else
  begin
    Result := TSeries(FSeriesList.Items[iSeries]).AddStringPoint(XString, X, Y, FireEvent, AdjustAxes);
  end;
end;

function TCustomDBPlot.InsertPoint(iSeries: Integer; X, Y: Single): Integer;
begin
  if ((not FUpdating) and ReadOnly) then
    Result := -1
  else
  begin
    Result := TSeries(FSeriesList.Items[iSeries]).InsertPoint(X, Y);
  end;
end;

procedure TCustomDBPlot.ReplacePoint(iSeries, N: Integer; NewX, NewY: Single);
begin
  if ((not FUpdating) and ReadOnly) then exit;

  TSeries(FSeriesList.Items[iSeries]).ReplacePoint(N, NewX, NewY);
end;

procedure TCustomDBPlot.Compress(iSeries: Integer; CompressRatio: Integer);
begin
  if ((not FUpdating) and ReadOnly) then exit;

  TSeries(FSeriesList.Items[iSeries]).Compress(CompressRatio);
end;

procedure TCustomDBPlot.Contract(iSeries: Integer; TheStart, TheFinish: Integer);
begin
  if ((not FUpdating) and ReadOnly) then exit;

  TSeries(FSeriesList.Items[iSeries]).Contract(TheStart, TheFinish);
end;

function TCustomDBPlot.DelPoint(iSeries: Integer; X, Y: Single; Confirm: Boolean): Integer;
begin
  if ((not FUpdating) and ReadOnly) then
    Result := -1
  else
  begin
    Result := TSeries(FSeriesList.Items[iSeries]).DelPoint(X, Y, Confirm);
  end;
end;

function TCustomDBPlot.DelPointNumber(iSeries: Integer; ThePoint: Integer; Confirm: Boolean): Integer;
begin
  if ((not FUpdating) and ReadOnly) then
    Result := -1
  else
  begin
    Result := TSeries(FSeriesList.Items[iSeries]).DelPointNumber(ThePoint, Confirm);
  end;
end;

function TCustomDBPlot.DelData(iSeries: Integer): Boolean;
begin
  if ((not FUpdating) and ReadOnly) then
    Result := FALSE
  else
  begin
    Result := TSeries(FSeriesList.Items[iSeries]).DelData;
  end;
end;

procedure TCustomDBPlot.Smooth(iSeries: Integer; SmoothOrder: Integer);
begin
  if ((not FUpdating) and ReadOnly) then exit;

  TSeries(FSeriesList.Items[iSeries]).Smooth(SmoothOrder);
end;

procedure TCustomDBPlot.Sort(iSeries: Integer);
begin
  if ((not FUpdating) and ReadOnly) then exit;

  TSeries(FSeriesList.Items[iSeries]).Sort;
end;

procedure TCustomDBPlot.Differentiate(iSeries: Integer);
begin
  if ((not FUpdating) and ReadOnly) then exit;

  TSeries(FSeriesList.Items[iSeries]).Differentiate;
end;

procedure TCustomDBPlot.Integrate(iSeries: Integer);
begin
  if ((not FUpdating) and ReadOnly) then exit;

  TSeries(FSeriesList.Items[iSeries]).Integrate;
end;

procedure TCustomDBPlot.MovingAverage(iSeries: Integer; Span: Integer);
begin
  if ((not FUpdating) and ReadOnly) then exit;

  TSeries(FSeriesList.Items[iSeries]).MovingAverage(Span);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomDBPlot.StoreInDB
  Description: Stores the current plot data in a database
       Author: Mat Ballard
 Date created: 03/02/2001
Date modified: 03/02/2001 by Mat Ballard
      Purpose: database management
 Known Issues:
     Comments: based on a post from: "Bruce Roberts" <ber@attcanada.xnet> on
               Dynamic arrays in databases, Sun Feb 25 2001.
 ------------------------------------------------------------------------------}
procedure TCustomDBPlot.StoreInDB;
var
{$IFDEF MSWINDOWS}
  TheBlobStream: TBlobStream;
{$ENDIF}
{$IFDEF LINUX}
  TheBlobStream: TClientBlobStream;
{$ENDIF}
begin
  if (Self.DataSource.DataSet.CanModify) then
  begin
    TheBlobStream := TClientBlobStream.Create (TBlobField(FDataLink.Field), bmWrite);
    try
      Self.SaveToStream(TMemoryStream(TheBlobStream));
    finally
      TheBlobStream.Free;
    end;
  end;
end;


{------------------------------------------------------------------------------
    Procedure: TCustomDBPlot.LoadFromDB
  Description: loads fresh plot data from a database
       Author: Mat Ballard
 Date created: 03/02/2001
Date modified: 03/02/2001 by Mat Ballard
      Purpose: database management
 Known Issues: we need to create a memorystream because Misc.Readline barfs
               because GetLineLengthFromStream barfs.
     Comments: based on a post from: "Bruce Roberts" <ber@attcanada.xnet> on
               Dynamic arrays in databases, Sun Feb 25 2001.
 ------------------------------------------------------------------------------}
procedure TCustomDBPlot.LoadFromDB;
var
{$IFDEF MSWINDOWS}
  TheBlobStream: TBlobStream;
{$ENDIF}
{$IFDEF LINUX}
  TheBlobStream: TStream;
{$ENDIF}
  TheMemoryStream: TMemoryStream;
begin
  Self.FSeriesList.ClearSeries;
{$IFDEF MSWINDOWS}
  TheBlobStream := TBlobStream.Create (TBlobField(FDataLink.Field), bmRead);
{$ENDIF}
{$IFDEF LINUX}
  TheBlobStream :=
    Self.FDataLink.DataSet.CreateBlobStream(TBlobField(FDataLink.Field), bmRead);
{$ENDIF}
  TheMemoryStream := TMemoryStream.Create;
  try
    if (TheBlobStream.Size > 20) then
    begin
      TheMemoryStream.LoadFromStream(TheBlobStream);
      TheMemoryStream.Seek(0, soFromBeginning);
      Self.LoadFromStream(TheMemoryStream);
    end
     else
      Self.Clear;
  finally
    TheBlobStream.Free;
    TheMemoryStream.Free;
  end;
end;

procedure TCustomDBPlot.UpdateRecord;
begin
  FDatalink.UpdateRecord;
end;

end.
