unit Axisedit;

{$I Plot.inc}

{-----------------------------------------------------------------------------
The contents of this file are subject to the Q Public License
("QPL"); you may not use this file except in compliance
with the QPL. You may obtain a copy of the QPL from 
the file QPL.html in this distribution, derived from:

http://www.trolltech.com/products/download/freelicense/license.html

The QPL prohibits development of proprietary software. 
There is a Professional Version of this software available for this. 
Contact sales@chemware.hypermart.net for more information.

Software distributed under the QPL is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either expressed or implied. See the QPL for
the specific language governing rights and limitations under the QPL.

The Original Code is: AxisEdit.pas, released 12 September 2000.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@chemware.hypermart.net.

Last Modified: 03/11/2001
Current Version: 2.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind. 

Purpose:
To facilitate user manipluation of Axis properties.

Known Issues:
-----------------------------------------------------------------------------}

interface

uses
  Classes, SysUtils, TypInfo,
{$IFDEF WINDOWS}
  WinTypes, WinProcs,
  Buttons, Controls, Forms, Graphics, StdCtrls,
{$ENDIF}
{$IFDEF WIN32}
  Windows,
  Buttons, Controls, Forms, Graphics, StdCtrls,
{$ENDIF}
{$IFDEF LINUX}
  QButtons, QControls, QForms, QGraphics, QStdCtrls,
{$ENDIF}

  Axis, Misc, Titles, Nedit, Coloredt;


type
  TAxisProperty = record
    Visible: Boolean;
    LabelFormat: TLabelFormat;
    LabelDigits: Byte;
    LabelPrecision: Byte;
    LabelDirection: TDirection;
    PenColor: TColor;
    PenWidthIndex: Integer;
    PenStyleIndex: Integer;
    TickSize: Byte;
    TickDirection: TOrientation;
    TickStepSize: Single;
    TickMinors: Byte;
    ScaleMin: Single;
    ScaleMax: Single;
    ScaleIntercept: Single;
    ScaleAuto: Boolean;
    ScaleLog: Boolean;
    ArrowSize: Byte;
    ArrowDirection: TAlignment;
    ZAngle: Word;
    ZLength: Word;
  end;

  TAxisEditorForm = class(TForm)
    HelpBitBtn: TBitBtn;
    CancelBitBtn: TBitBtn;
    OKBitBtn: TBitBtn;
    VisibleCheckBox: TCheckBox;
    PenGroupBox: TGroupBox;
    Label2: TLabel;
    Label4: TLabel;
    PenWidthComboBox: TComboBox;
    PenStyleComboBox: TComboBox;
    Label5: TLabel;
    GroupBox1: TGroupBox;
    Label6: TLabel;
    Label7: TLabel;
    Label9: TLabel;
    DigitsComboBox: TComboBox;
    PrecisionComboBox: TComboBox;
    FormatComboBox: TComboBox;
    GroupBox2: TGroupBox;
    Label10: TLabel;
    Label11: TLabel;
    TickDirectionComboBox: TComboBox;
    StepSizeLabel: TLabel;
    GroupBox3: TGroupBox;
    InterceptLabel: TLabel;
    LogScaleCheckBox: TCheckBox;
    MinLabel: TLabel;
    MaxLabel: TLabel;
    GroupBox4: TGroupBox;
    Label1: TLabel;
    Label3: TLabel;
    ArrowDirectionComboBox: TComboBox;
    TitleEdit: TEdit;
    Label13: TLabel;
    MinorsComboBox: TComboBox;
    NoComboBox: TComboBox;
    AxisNumberLabel: TLabel;
    AutoScaleCheckBox: TCheckBox;
    TickSizeNEdit: TNEdit;
    StepSizeNEdit: TNEdit;
    MinNEdit: TNEdit;
    MaxNEdit: TNEdit;
    InterceptNEdit: TNEdit;
    ArrowSizeNEdit: TNEdit;
    PenColorEdit: TColorEdit;
    ApplyBitBtn: TBitBtn;
    Label8: TLabel;
    LabelDirectionComboBox: TComboBox;
    ZAxisGroupBox: TGroupBox;
    ZAngleLabel: TLabel;
    ZLengthLabel: TLabel;
    ZAngleNEdit: TNEdit;
    ZLengthNEdit: TNEdit;
    procedure FormCreate(Sender: TObject);
    procedure FormatComboBoxClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure PenStyleComboBoxClick(Sender: TObject);
    procedure FormActivate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure NoComboBoxClick(Sender: TObject);
    procedure TitleEditChange(Sender: TObject);
    procedure DigitsComboBoxClick(Sender: TObject);
    procedure PrecisionComboBoxClick(Sender: TObject);
    procedure TickDirectionComboBoxClick(Sender: TObject);
    procedure TickSizeNEditChange(Sender: TObject);
    procedure StepSizeNEditChange(Sender: TObject);
    procedure MinNEditChange(Sender: TObject);
    procedure MaxNEditChange(Sender: TObject);
    procedure InterceptNEditChange(Sender: TObject);
    procedure MinorsComboBoxClick(Sender: TObject);
    procedure ArrowDirectionComboBoxClick(Sender: TObject);
    procedure ArrowSizeNEditChange(Sender: TObject);
    procedure VisibleCheckBoxClick(Sender: TObject);
    procedure AutoScaleCheckBoxClick(Sender: TObject);
    procedure LogScaleCheckBoxClick(Sender: TObject);
    procedure PenWidthComboBoxClick(Sender: TObject);
    procedure PenColorEditChange(Sender: TObject);
    procedure ApplyBitBtnClick(Sender: TObject);
    procedure LabelDirectionComboBoxClick(Sender: TObject);
    procedure ZAngleNEditChange(Sender: TObject);
    procedure ZLengthNEditChange(Sender: TObject);
  private
    pAXP: ^TAxisProperty;
    CurrentIndex: Integer;
  public
    AxisPropertyList: TList;
    AxisNames: TStringList;
    HistoryMode: Boolean;
    ThePlot: TObject;
    
    function AddAxis(AName: String; AnAxisProperty: TAxisProperty): Integer;
    procedure SelectAxis(Index: Integer);
  end;

var
  AxisEditorForm: TAxisEditorForm;

implementation

{$R *.dfm}

uses
  Plot;

{------------------------------------------------------------------------------
    Procedure: TAxisEditorForm.FormCreate
  Description: standard FormCreate procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the position and populates the Color combo
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisEditorForm.FormCreate(Sender: TObject);
var
  i: TColor;
begin
  SetDialogGeometry(Self, ApplyBitBtn, AxisNumberLabel.Left);

{set combo and edit box widths:}
  for i := 0 to Self.ComponentCount - 1 do
    if ((Self.Components[i] is TNEdit) or
        (Self.Components[i] is TColorEdit) or
        (Self.Components[i] is TComboBox)) then
      TControl(Self.Components[i]).Width := 88;
{the exception:}
  NoComboBox.Width := 48;

{Populate the combo boxes:}
  for i := 0 to 18 do
    DigitsComboBox.Items.Add(IntToStr(i));
  for i := 0 to Ord(High(TLabelFormat)) do
    FormatComboBox.Items.Add(Copy(GetEnumName(TypeInfo(TLabelFormat), i), 3, 99));
  for i := 0 to 9 do
    MinorsComboBox.Items.Add(IntToStr(i));
  for i := 0 to 20 do
    PenWidthComboBox.Items.Add(IntToStr(i));
  for i := 0 to Ord(High(TPenStyle)) do
    PenStyleComboBox.Items.Add(Copy(GetEnumName(TypeInfo(TPenStyle), i), 3, 99));
  for i := 0 to 7 do
    PrecisionComboBox.Items.Add(IntToStr(i));
  for i := 0 to Ord(High(TDirection)) do
    LabelDirectionComboBox.Items.Add(Copy(GetEnumName(TypeInfo(TDirection), i), 3, 99));

  AxisPropertyList := TList.Create;
  AxisNames := TStringList.Create;
end;

{------------------------------------------------------------------------------
    Procedure: TAxisEditorForm.FormClose
  Description: standard FormClose procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: checks for valid user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisEditorForm.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
{Check that floating point values are valid:
  Action := caHide;
  try
    if (not AutoScaleCheckBox.Checked) then
    begin
      MinEdit.SetFocus;
      Test := StrToFloat(MinEdit.Text)
      MaxEdit.SetFocus;
      TestMax := StrToFloat(MaxEdit.Text)
      if (TestMax < Test) then
        ERangeError.CreateFmt('The Min (%g) MUST be less than the Max (%g)',
          [Test, TestMax]);
      InterceptEdit.SetFocus;
      Test := StrToFloat(InterceptEdit.Text)
      StepSizeEdit.SetFocus;
      Test := Test + StrToFloat(StepSizeEdit.Text);
  {We do the above and below the squelch the warning messages.
      Tag := Trunc(Test);
    end;
  except
{Prevent closure:
    Action := caNone;
  end;}
end;

{------------------------------------------------------------------------------
    Procedure: TAxisEditorForm.FormatComboBoxClick
  Description: responds to NumberFormat selection
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: loads the Digits combo depending on selection
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisEditorForm.FormatComboBoxClick(Sender: TObject);
var
  i, iMax, OldIndex: Integer;
begin
  pAXP^.LabelFormat := TLabelFormat(FormatComboBox.ItemIndex);
  OldIndex := DigitsComboBox.ItemIndex;
  DigitsComboBox.Clear;
  iMax := 18;
  if ((FormatComboBox.ItemIndex = Ord(ffGeneral)) or
      (FormatComboBox.ItemIndex = Ord(ffExponent))) then
    iMax := 4;
  for i := 0 to iMax do
  begin
    DigitsComboBox.Items.Add(IntToStr(i));
  end;
  if (OldIndex > iMax) then OldIndex := iMax;
  DigitsComboBox.ItemIndex := OldIndex;
end;

{------------------------------------------------------------------------------
    Procedure: TAxisEditorForm.PenStyleComboBoxClick
  Description: responds to user selection
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Width to 1 if neccessary
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisEditorForm.PenStyleComboBoxClick(Sender: TObject);
begin
  pAXP^.PenStyleIndex := PenStyleComboBox.ItemIndex;
  if (PenStyleComboBox.ItemIndex > Ord(psSolid)) then
    PenWidthComboBox.ItemIndex := 1;
end;

{------------------------------------------------------------------------------
    Procedure: TAxisEditorForm.FormActivate
  Description: standard FormActivate procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Visibility of the Max Label
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisEditorForm.FormActivate(Sender: TObject);
begin
  if (MaxNEdit.Enabled = FALSE) then
    MaxLabel.Enabled := FALSE;
  FormatComboBoxClick(Self);
{$IFDEF LINUX}
  HelpBitBtn.Kind := bkCustom;
{$ENDIF}
end;

procedure TAxisEditorForm.FormDestroy(Sender: TObject);
var
  i: Integer;
begin
  for i := 0 to AxisPropertyList.Count-1 do
    FreeMem(AxisPropertyList.Items[i], SizeOf(TAxisProperty));

  AxisPropertyList.Free;
  AxisNames.Free;
end;

{------------------------------------------------------------------------------
    Procedure: TAxisEditorForm.SelectAxis
  Description: responds to selection of a new Axis
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisEditorForm.SelectAxis(Index: Integer);
begin
  pAXP := AxisPropertyList.Items[Index];
  CurrentIndex := Index;

{Load the combo box:}
  ArrowDirectionComboBox.Clear;
  TickDirectionComboBox.Clear;
  if (Index = 0) then {X Axis:}
  begin
    if (HistoryMode) then
      MaxNEdit.Enabled := FALSE;
    ArrowDirectionComboBox.Items.Add('Left');
    ArrowDirectionComboBox.Items.Add('Right');
    ArrowDirectionComboBox.Items.Add('None');
    TickDirectionComboBox.Items.Add('Up');
    TickDirectionComboBox.Items.Add('Down');
  end
  else
  begin
    MaxNEdit.Enabled := TRUE;
    ArrowDirectionComboBox.Items.Add('Down');
    ArrowDirectionComboBox.Items.Add('Up');
    ArrowDirectionComboBox.Items.Add('None');
    TickDirectionComboBox.Items.Add('Left');
    TickDirectionComboBox.Items.Add('Right');
  end;                  

  TitleEdit.Text := AxisNames.Strings[Index];
  FormatComboBox.ItemIndex := Ord(pAXP^.LabelFormat);
  DigitsComboBox.ItemIndex := pAXP^.LabelDigits;
  PrecisionComboBox.ItemIndex := pAXP^.LabelPrecision;
  LabelDirectionComboBox.ItemIndex := Ord(pAXP^.LabelDirection);
  PenColorEdit.SetColour(pAXP^.PenColor);
  PenWidthComboBox.ItemIndex := pAXP^.PenWidthIndex;
  PenStyleComboBox.ItemIndex := pAXP^.PenStyleIndex;
  TickSizeNEdit.AsInteger := pAXP^.TickSize;
  TickDirectionComboBox.ItemIndex := Ord(pAXP^.TickDirection);
  StepSizeNEdit.AsReal := pAXP^.TickStepSize;
  MinorsComboBox.ItemIndex := pAXP^.TickMinors;
  MinNEdit.AsReal := pAXP^.ScaleMin;
  MaxNEdit.AsReal :=  pAXP^.ScaleMax;
  InterceptNEdit.AsReal := pAXP^.ScaleIntercept;
  AutoScaleCheckBox.Checked :=  pAXP^.ScaleAuto;
  LogScaleCheckBox.Checked :=  pAXP^.ScaleLog;
  ArrowSizeNEdit.AsInteger := pAXP^.ArrowSize;
  ArrowDirectionComboBox.ItemIndex := Ord(pAXP^.ArrowDirection);
  if (pAXP^.ZLength > 0) then
  begin
    ZAngleNEdit.AsInteger := pAXP^.ZAngle;
    ZLengthNEdit.AsInteger := pAXP^.ZLength;
    ZAxisGroupBox.Enabled := TRUE;
    ZAngleLabel.Enabled := TRUE;
    ZLengthLabel.Enabled := TRUE;
  end
  else
  begin
    ZAxisGroupBox.Enabled := FALSE;
    ZAngleLabel.Enabled := FALSE;
    ZLengthLabel.Enabled := FALSE;
  end;

  AutoScaleCheckBoxClick(Self);

  VisibleCheckBox.Checked := pAXP^.Visible;
end;

procedure TAxisEditorForm.NoComboBoxClick(Sender: TObject);
var
  Test,
  TestMax: Single;
begin
  try
    if (not AutoScaleCheckBox.Checked) then
    begin
      MinNEdit.SetFocus;
      Test := MinNEdit.AsReal;
      MaxNEdit.SetFocus;
      TestMax := MaxNEdit.AsReal;
      if (TestMax < Test) then
        ERangeError.CreateFmt('The Min (%g) MUST be less than the Max (%g)',
          [Test, TestMax]);
      InterceptNEdit.SetFocus;
      Test := InterceptNEdit.AsReal;
      StepSizeNEdit.SetFocus;
      Test := Test + StepSizeNEdit.AsReal;
{We do the above and below the squelch the warning messages.}
      Tag := Trunc(Test);
    end;
    SelectAxis(NoComboBox.ItemIndex);
  except
  end;
end;

{------------------------------------------------------------------------------
     Function: TAxisEditorForm.AddAxis
  Description: adds a new Axis to the list
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: stores the Axis properties
 Known Issues:
 ------------------------------------------------------------------------------}
function TAxisEditorForm.AddAxis(AName: String; AnAxisProperty: TAxisProperty): Integer;
var
  pDestination: Pointer;
  pDest, pSource: PChar;
  i: Integer;
begin
  NoComboBox.Items.Add(IntToStr(AxisNames.Count));

  AxisNames.Add(AName);

  GetMem(pDestination, SizeOf(AnAxisProperty));
  pSource := @AnAxisProperty;
{NB: this causes terminal access violations:
  System.Move(pSource, pDestination, SizeOf(AnAxisProperty));}

  pDest := pDestination;
  for i := 1 to SizeOf(AnAxisProperty) do
  begin
    pDest^ := pSource^;
    Inc(pSource);
    Inc(pDest);
  end;

  AxisPropertyList.Add(pDestination);
  AddAxis := AxisPropertyList.Count;
end;

procedure TAxisEditorForm.TitleEditChange(Sender: TObject);
begin
  AxisNames.Strings[CurrentIndex] := TitleEdit.Text;
end;

procedure TAxisEditorForm.DigitsComboBoxClick(Sender: TObject);
begin
  pAXP^.LabelDigits := DigitsComboBox.ItemIndex;
end;

procedure TAxisEditorForm.PrecisionComboBoxClick(Sender: TObject);
begin
  pAXP^.LabelPrecision := PrecisionComboBox.ItemIndex;
end;

procedure TAxisEditorForm.TickDirectionComboBoxClick(Sender: TObject);
begin
  pAXP^.TickDirection := TOrientation(TickDirectionComboBox.ItemIndex);
end;

procedure TAxisEditorForm.TickSizeNEditChange(Sender: TObject);
begin
  pAXP^.TickSize := TickSizeNEdit.AsInteger;
end;

procedure TAxisEditorForm.StepSizeNEditChange(Sender: TObject);
begin
  pAXP^.TickStepSize := StepSizeNEdit.AsReal;
end;

procedure TAxisEditorForm.MinNEditChange(Sender: TObject);
begin
  pAXP^.ScaleMin := MinNEdit.AsReal;
end;

procedure TAxisEditorForm.MaxNEditChange(Sender: TObject);
begin
  pAXP^.ScaleMax := MaxNEdit.AsReal;
end;

procedure TAxisEditorForm.InterceptNEditChange(Sender: TObject);
begin
  pAXP^.ScaleIntercept := InterceptNEdit.AsReal;
end;

procedure TAxisEditorForm.MinorsComboBoxClick(Sender: TObject);
begin
  pAXP^.TickMinors := MinorsComboBox.ItemIndex;
end;

procedure TAxisEditorForm.ArrowDirectionComboBoxClick(Sender: TObject);
begin
  pAXP^.ArrowDirection := TAlignment(ArrowDirectionComboBox.ItemIndex);
end;

procedure TAxisEditorForm.ArrowSizeNEditChange(Sender: TObject);
begin
  pAXP^.ArrowSize := ArrowSizeNEdit.AsInteger;
end;

procedure TAxisEditorForm.VisibleCheckBoxClick(Sender: TObject);
begin
  pAXP^.Visible := VisibleCheckBox.Checked;
end;

procedure TAxisEditorForm.AutoScaleCheckBoxClick(Sender: TObject);
begin
  pAXP^.ScaleAuto := AutoScaleCheckBox.Checked;

  StepSizeNEdit.Enabled := not AutoScaleCheckBox.Checked;
  MinNEdit.Enabled := not AutoScaleCheckBox.Checked;
  MaxNEdit.Enabled := not AutoScaleCheckBox.Checked;
  InterceptNEdit.Enabled := not AutoScaleCheckBox.Checked;

  StepSizeLabel.Enabled := not AutoScaleCheckBox.Checked;
  MinLabel.Enabled := not AutoScaleCheckBox.Checked;
  MaxLabel.Enabled := not AutoScaleCheckBox.Checked;
  InterceptLabel.Enabled := not AutoScaleCheckBox.Checked;
end;

procedure TAxisEditorForm.LogScaleCheckBoxClick(Sender: TObject);
begin
  pAXP^.ScaleLog := LogScaleCheckBox.Checked;
end;

procedure TAxisEditorForm.PenWidthComboBoxClick(Sender: TObject);
begin
  pAXP^.PenWidthIndex := PenWidthComboBox.ItemIndex;
end;

procedure TAxisEditorForm.PenColorEditChange(Sender: TObject);
begin
  pAXP^.PenColor := PenColorEdit.Color;
end;

procedure TAxisEditorForm.ApplyBitBtnClick(Sender: TObject);
begin
  TPlot(ThePlot).ApplyAxisChange(Self);
end;


procedure TAxisEditorForm.LabelDirectionComboBoxClick(Sender: TObject);
begin
  pAXP^.LabelDirection := TDirection(LabelDirectionComboBox.ItemIndex);
end;


procedure TAxisEditorForm.ZAngleNEditChange(Sender: TObject);
begin
  pAXP^.ZAngle := ZAngleNEdit.AsInteger;
end;

procedure TAxisEditorForm.ZLengthNEditChange(Sender: TObject);
begin
  pAXP^.ZLength := ZLengthNEdit.AsInteger;
end;

end.
