unit pSeries;

{$I Plot.inc}

{-----------------------------------------------------------------------------
The contents of this file are subject to the Mozilla Public License
Version 1.1 (the "License"); you may not use this file except in compliance
with the License. You may obtain a copy of the License at
http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either expressed or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is: pSeries.pas, released 12 September 2000.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@molsci.csiro.au.

Last Modified: 02/25/2000
Current Version: 1.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind. 

Purpose:
This unit contains the TSeries sub-component - that manages the data for a single series.

Known Issues:
      - This would normally be called Series, but TeeChart already uses that unit name.

History:
 1.01 21 September 2000: add Brush property for columns
-----------------------------------------------------------------------------}

interface

uses
  classes, controls, graphics, SysUtils, Dialogs, Forms, ClipBrd,

{These are the D1 units, aliased to Windows:}
{$IFDEF WIN}
  WinTypes, WinProcs,
{$ENDIF}

  Axis, Displace, PtEdit, Misc;

const
  OUTLINE_DENSITY = 20;
{This is the number of points in a branch of an outline.}

type
  TOnMinMaxChangeEvent = procedure(Sender: TObject; Value: Single) of object;

  THighLow = (hlLow, hlHigh);
  TSetHighLow = set of THighLow;

  TSymbol = (sNone, sDash, sVertDash, sPlus, sCross, sStar, sSquare, sCircle, sUpTriangle, sDownTriangle);
{These are the different symbols that are used to represent the data points.}

  TDataStatus = (dsNone, dsInternal, dsExternal);
{These are the three data states:}
{}
{    dsNone - no data as yet;}
{    dsInternal - there is internal memory;}
{    dsExternal - both the X and Y data are stored elsewhere (not in SciSeriesList).}

{$IFDEF DELPHI1}
  EAccessViolation = class(Exception);
{$ENDIF}

  TSeries = class(TPersistent)
  private
    FAxisList: TList;
    FBrush: TBrush;
    FDataChanged: Boolean;
    FDefSize: Word;
    FDeltaX: Integer;
    FDeltaY: Integer;
    FName: String;
    FNoPts: Integer;
    FPen: TPen;
    FHighCapacity: Integer;
    FHighCount: Integer;
    FHighLow: TSetHighLow;
    FHighs: pIntegerArray;
     FLowCount: Integer;
     FLows: pIntegerArray;
    FSymbol: TSymbol;
    FSymbolSize: Integer;
    FVisible: Boolean;
    FXAxis: TAxis;
    FXMin: Single;
    FXMax: Single;
    FXData: pSingleArray;
    FYAxis: TAxis;
    FYAxisIndex: Byte;
    FYData: pSingleArray;
    Fd2Y_dX2: pSingleArray;
      Size2ndDeriv: Integer;
    FYMin: Single;
    FYMax: Single;

    FOnChange: TNotifyEvent;
    {FOnAddPoint: TNotifyEvent;}
{Currently superceded by direct calls to TAxis.SetMin[Max]FromSeries:}
    {FOnXMinChange: TOnMinMaxChangeEvent;
    FOnXMaxChange: TOnMinMaxChangeEvent;
    FOnYMinChange: TOnMinMaxChangeEvent;
    FOnYMaxChange: TOnMinMaxChangeEvent;}
{may be re-implemented later for other needs.}

    FDependentSeries: TList;
{The list of series that use this series' X-Data.}

    FExternalXSeries: Boolean;
{Is the X data maintained in a different series ?}
    FXDataSeries: TSeries;
{This is the Data Series in which the External X data, if any, is stored.}
    DataStatus: TDataStatus;
{Was this data generated externally ? If it was, then we do not manage it,
 nor manipulate it.}
    MemSize: LongInt;
{The current number of points allocated in memory.}
    Outline: array [0..OUTLINE_DENSITY+1] of TPoint;
{The pointer to the memory block of outline points.
 These points are in screen co-ordinates (pixels).}
{}
{The outline is used for clicking and dragging operations.}
    NoOutlinePts: Integer;
{The number of outline points.}
    FOutlineWidth: Integer;
{This is the width of the outline.}

{The one and only property getting function:}
    function GetXDataRefCount: Word;

{The property-setting routines:}
    procedure SetBrush(Value: TBrush);
    procedure SetDeltaX(Value: Integer);
    procedure SetDeltaY(Value: Integer);
    procedure SetName(Value: String);
    procedure SetPen(Value: TPen);
    procedure SetSymbol(Value: TSymbol);
    procedure SetSymbolSize(Value: Integer);
    procedure SetVisible(Value: Boolean);
    procedure SetYAxisIndex(Value: Byte);

    procedure CheckBounds(ThePointNo: Integer; AdjustAxis: Boolean);
{Check the Min and Max properties against this point.}
    function IncMemSize: Boolean;
{Allocate memory for the data.}

  protected
    procedure StyleChange;
  public
    property DataChanged: Boolean read FDataChanged write FDataChanged;
{Has the data in this series changed ?}
    property ExternalXSeries: Boolean read FExternalXSeries;
{Is the X data maintained in a different series ?}
    property XDataSeries: TSeries read FXDataSeries;
{If the X data is maintained in a different series, this is the series.}
    property NoPts: Integer read FNoPts;
{The number of points in the series.}
    property HighCount: Integer read FHighCount;
{The number of Highs (Peaks)}
    property Highs: pIntegerArray read FHighs;
{This is a list of the Highs (Peaks) in the plot. See Lows.}
    property LowCount: Integer read FLowCount;
{The number of Lows (Troughs)}
    property Lows: pIntegerArray read FLows;
{This is a list of the Lows (Troughs) in the plot. See Highs.}
    property XDataRefCount: Word read GetXDataRefCount;
{This is the number of series that use this series as an X Data holder.}

    property XAxis: TAxis read FXAxis;
{The X Axis to which this series is bound - needed for scaling purposes.}

    property YAxis: TAxis read FYAxis;
{The Y Axis to which this series IS bound - can be any of the Y Axes - needed for scaling purposes.}
    property YAxisIndex: Byte read FYAxisIndex write SetYAxisIndex;
{The Y Axis to which this series IS bound - can be any of the Y Axes - needed for scaling purposes.
 We define YAxisIndex to run from 1 to FAxisList.Count-1:
    1 => The primary Y Axis,
    2 => The secondary Y Axis,
    etc.}

    property XData: pSingleArray read FXData;
{This is the dynamic X data array.
 It can be set by the user, or memory for the data
 can be allocated and managed by this component.}
{}
{The user can access the data points either through the GetPoint / GetXYPoint /
 ReplacePoint methods, or directly by:}
{}
{    ASeries.XData^[i] := NewValue;}
{}
{Note that the POINTER XData is read-only, but that the array elements are
 read/write.}

    property YData: pSingleArray read FYData;
{This is the dynamic Y data array.
 It can be set by the user, or memory for the data
 can be allocated and managed by this component.}
{}
{The user can access the data points either through the GetPoint / GetXYPoint /
 ReplacePoint methods, or directly by:}
{}
{    ASeries.YData^[i] := NewValue;}
{}
{Note that the POINTER YData is read-only, but that the array elements are
 read/write.}

    property d2Y_dX2: pSingleArray read Fd2Y_dX2;
{The array of second derivatives - used in cubic splines.}
    property XMin: Single read FXMin;
{The minimum X value, determined by GetBounds.}
    property XMax: Single read FXMax;
{The maximum X value, determined by GetBounds.}
    property YMin: Single read FYMin;
{The minimum Y value, determined by GetBounds.}
    property YMax: Single read FYMax;
{The maximum Y value, determined by GetBounds.}

    Constructor Create(
      Index: Integer;
      AxisList: TList;
      XDataSeriesValue: TSeries); virtual;
{Each series needs to know a few things:}
{}
{    1. What axes it can relate to;}
{    2. Does it use another (previous) series X Values ?}
    Destructor Destroy; override;

    function AddDrawPoint(X, Y: Single; ACanvas: TCanvas): Integer;
{This adds a single point to the xy data (using AddPoint),
 draws the line segment and new point, and returns the number
 of points: -1 indicates failure.}
    function AddPoint(X, Y: Single; FireEvent, AdjustAxes: Boolean): Integer;
{This adds a single point to the xy data and returns the number of points:
 -1 indicates failure. If no memory has been allocated for the data yet, then
 IncMemSize is called automatically.}

    function AllocateNoPts(Value: LongInt): Boolean;
{Directly allocates memory for a fixed number of points.}
{}
{If AllocateNoPts cannot allocate memory for the requested number of points,
 it allocates what it can and returns FALSE.}

    procedure Contract(ContractRatio: Integer);
{This averages every N points in a row and so reduces the size of the
 data set by a factor of N.}
    procedure CopyToClipBoard;
{Does what it says.}

    procedure Displace;
{Runs the dialog box to set the displacement (DeltaX) of the Series.}

    function AddData(XPointer, YPointer: pSingleArray; NumberOfPoints: Integer): Boolean;
{This adds an entire Internal data set of an X array, a Y array,
 and the new number of points: Success returns TRUE.}
{}
{Internal means that TSeries allocates and manages the memory for this data,
 and makes a copy of the data located at XPointer and YPointer into this
 internally managed memory.}
{}
{It can therefore add, remove or edit any points.}

    function AddDependentSeries(ASeries: TSeries): Boolean;
{This function ADDS a series that depends on this series' X-Data from the list of dependent series.}
    function RemoveDependentSeries(ASeries: TSeries): Boolean;
{This function REMOVES a series that depends on this series' X-Data from the list of dependent series.}
    function AssumeMasterSeries(XPts: Integer; OldMaster: TSeries; AList: TList): Boolean;
{This function makes this series' X-Data the MASTER for the given list of dependent series.}
    function ResetXDataSeries(OldSeries, NewSeries: TSeries): Boolean;

    function DelPoint(X, Y: Single; Confirm: Boolean): Integer;
{This deletes a single point, the closest one, from the xy data.}

    function DelPointNumber(ThePoint: Integer; Confirm: Boolean): Integer;
{This deletes a single point, the Nth one, from the xy data.}

    function DelData: Boolean;
{This deletes an entire data set. It only works on internal data sets.}

    procedure DrawHistory(ACanvas: TCanvas; HistoryX: Single);
{This draws the series on the given canvas, in History mode.
 That is, from the latest point backwards a distance HistoryX}
    procedure Draw(ACanvas: TCanvas);
{This draws the series on the given canvas.}
    procedure DrawSymbol(ACanvas: TCanvas; iX, iY: Integer);
{This draws one of the symbols on the given canvas.}
    procedure Trace(ACanvas: TCanvas);
{This draws the series on the given canvas in an erasable mode.
 The first call draws, the second call erases.}

    procedure EditPoint(ThePointNumber: Integer);
{This runs the Point Editor dialog box.}

    procedure GetBounds;
{Determines the Min and Max properties for the whole series.}
{Data manipulation:}
    procedure ResetBounds;
{Reset the Min and Max properties.}

    function GetNearestPointToX(X: Single): Integer;
{This returns the point that has an X value closest to X.}

    function GetNearestPoint(
      StartPt, EndPt, iX, iY: Integer;
      var NearestiX, NearestiY: Integer;
      var NearestX, NearestY, MinDistance: Single): Integer;
{This returns the Index of the nearest point, and sets its XValue and YValue.
 It is guaranteed to find the nearest point.}

    function GetNearestPointFast(iX, iY: Integer;
      var NearestiX, NearestiY: Integer;
      var NearestX, NearestY, MinDistance: Single): Integer;
{This returns the Index of the nearest point, and sets its XValue and YValue.
 This is much quicker than GetNearestPoint, especially for big data sets,
 but MAY NOT return the closest point.}

    procedure GetPoint(N: Integer; var X, Y: Single);
{This returns the Nth point's X and Y values.}

    function GetXYPoint(N: Integer): TXYPoint;
{This returns the Nth point's X and Y values in a TXYPoint record.}

    function InsertPoint(X, Y: Single): Integer;
{This inserts a single point in the xy data and returns the location of the point:
 -1 indicates failure. The point is inserted at the appropriate X value.}

    function PointToData(XPointer, YPointer: pSingleArray; NumberOfPoints: Integer): Boolean;
{This adds an entire External data set of an X array, a Y array,
 and the new number of points: Success returns TRUE.}
{}
{External means that TSeries does not manage the memory for this data,
 nor can it add, remove or edit any points.}

    procedure ReplacePoint(N: Integer; var NewX, NewY: Single);
{This replaces the Nth point's values with X and Y.}

    procedure Smooth(SmoothOrder: Integer);
{This smooths the xy data using a midpoint method.}

    procedure Sort;
{This sorts the xy data in ascending X order.}

    procedure GenerateOutline(OutlineWidth: Integer);
{This generates an outline from the data. An outline contains
 the screen coordinates for (OUTLINE_DENSITY +1) points.}
{}
{Note that the memory for the Outline is allocated in the constructor and
 freed in the destructor.}
    procedure OutlineSeries(ACanvas: TCanvas);
{This draws (or erases) the Outline on the canvas.}

    procedure MoveSeriesBy(ACanvas: TCanvas; DX, DY: Integer);
{This erases the old Outline from the canvas, then redraws it
 at (DX, DY) from its current position.}

    procedure MoveSeriesTo(ACanvas: TCanvas; X, Y: Integer);
{This erases the old Outline from the canvas, then redraws it
 at the new location (X, Y).}

    procedure LineBestFit(TheLeft, TheRight: Single;
      var NoLSPts: Integer;
      var SumX, SumY, SumXsq, SumXY, SumYsq: Double;
      var Slope, Intercept, Rsq: Single);
{This performs a linear least-squares fit of TheSeries from points Start to Finish,
 and returns the Slope, Intercept and R-Square value.}
{}
{Normally you would initialize NoPts and the Sumxxx variables to zero.}
{}
{However, if you wish to fit over multiple regions (very useful in determining baselines)
 then simply call this function twice in a row with no re-initialization between calls.}

    procedure Differentiate;
{This replaces the series by its differential.}
    procedure Integrate;
{This replaces the series by its integral.}
    function Integral(TheLeft, TheRight: Single): Single;
{This calculates the integral of a series by X co-ordinate.}
    function IntegralByPoint(Start, Finish: Integer): Single;
{This calculates the integral of a series by points.}

    procedure DoSpline(Density: Integer; pSplineSeries: TSeries);
{This calculates the cubic spline interpolation of the data (XSpline, YSpline),
 which resides in another Series.}
    procedure SecondDerivative;
{This calculates the second derivate for a cubic spline interpolation by SplineValue.}
    function SplineValue(X: Single): Single;
{This calculates the cubic spline interpolation of the data at a given point X.}
    procedure ClearSpline;

    function FindHighsLows(Start, Finish, HeightSensitivity: Integer): Integer;
    procedure MovingAverage(Span: Integer);
    function Average(TheLeft, TheRight: Single): Single;
    procedure ClearHighsLows;
    procedure DrawHighs(ACanvas: TCanvas);
    procedure MakeXDataIndependent;

  published
    property Brush: TBrush read FBrush write SetBrush;
{The Brush (color, width, etc) with which the series is drawn on the Canvas.}
    property DeltaX: Integer read FDeltaX write SetDeltaX;
{The displacement of the series on the screen from its X origin.}
    property DeltaY: Integer read FDeltaY write SetDeltaY;
{The displacement of the series on the screen from its Y origin.}
    property DefSize: Word read FDefSize write FDefSize;
{The default memory allocation block size. Allocated memory grows in blocks of
 this number of points.}
    property HighLow: TSetHighLow read FHighLow write FHighLow;
{Do we show any Highs ? any Lows ? Both ? or None ?}
    property Name: String read FName write SetName;
{The name of the data set.}
    property Pen: TPen read FPen write SetPen;
{The Pen (color, width, etc) with which the series is drawn on the Canvas.}
    property Symbol: TSymbol read FSymbol write SetSymbol;
{The symbol (square, circle, etc) with which each data point is drawn.}
    property SymbolSize: Integer read FSymbolSize write SetSymbolSize;
{How big is the Symbol (0 means invisible).}
    property Visible: Boolean read FVisible write SetVisible;
{Is this series visible ?}

    property OnChange: TNotifyEvent read FOnChange write FOnChange;
{This notifies the owner (usually TPlot) of a relevant change in this series.}
    {property OnXMinChange: TOnMinMaxChangeEvent read FOnXMinChange write FOnXMinChange;
    property OnXMaxChange: TOnMinMaxChangeEvent read FOnXMaxChange write FOnXMaxChange;
    property OnYMinChange: TOnMinMaxChangeEvent read FOnYMinChange write FOnYMinChange;
    property OnYMaxChange: TOnMinMaxChangeEvent read FOnYMaxChange write FOnYMaxChange;}

    {property OnAddPoint: TNotifyEvent read FOnAddPoint write FOnAddPoint;}
  end;

  function Compare(Item1, Item2: Pointer): Integer;

implementation

uses
  Plot;

{TSeries Constructor and Destructor:-------------------------------------------}
{------------------------------------------------------------------------------
  Constructor: TSeries.Create
  Description: standard Constructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: creates Pen and initializes many things
 Known Issues:
 ------------------------------------------------------------------------------}
Constructor TSeries.Create(
  Index: Integer;
  AxisList: TList;
  XDataSeriesValue: TSeries);
begin
{First call the ancestor:}
  inherited Create;
{create sub-components:}
  FBrush := TBrush.Create;
  FPen := TPen.Create;
{we insert the default values that cannot be "defaulted":}
  DataStatus := dsNone;
  FDefSize := 256;
  FDeltaX := 0;
  FDeltaY := 0;
  FNoPts := 0;
  FYAxisIndex := 1;
  {FVisible := TRUE;}

{Set axes:}
  FAxisList := AxisList;
  FXAxis := TAxis(AxisList[0]);
  FYAxis := TAxis(AxisList[1]);

  FSymbolSize := 5;

  FDependentSeries := TList.Create;
  FXDataSeries := XDataSeriesValue;
  if (FXDataSeries = nil) then
  begin
    FExternalXSeries := FALSE;
    FXData := nil;
  end
  else
  begin
    FExternalXSeries := TRUE;
    FXData := FXDataSeries.XData;
    FXDataSeries.AddDependentSeries(Self);
  end;

{set names and color:}
  FName := Format('Series %d', [Index]);
  FPen.Color := MyColors[Index mod 16].Value;
{make the brush color paler by 70%:}
  FBrush.Color := Misc.GetPalerColor(FPen.Color, 70);

  FYData := nil;
  MemSize := 0;

  Fd2Y_dX2 := nil;

  FHighs := nil;
  FLows := nil;
  FHighCount := 0;
  FLowCount := 0;
  FHighCapacity := 0;

{allocate memory so as to create X and Y pointers:
  IncMemSize; - not needed: done in AddPoint}
end;

{------------------------------------------------------------------------------
   Destructor: TSeries.Destroy
  Description: standard Destructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Frees Pen and events
 Known Issues: would like a better solution to FXDataRefCount
 ------------------------------------------------------------------------------}
Destructor TSeries.Destroy;
begin
  FOnChange := nil;
  FVisible := FALSE;
  ClearSpline;
  ClearHighsLows;
  FBrush.Free;
  FPen.Free;

  if (FXDataSeries <> nil) then
    FXDataSeries.RemoveDependentSeries(Self);

  DelData;

  FDependentSeries.Free;

{then call ancestor:}
  inherited Destroy;
end;

{Begin Set and Get Functions and Procedures----------------------------------}
{------------------------------------------------------------------------------
    Procedure: TSeries.SetBrush
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 09/21/2000
Date modified: 09/21/2000 by Mat Ballard
      Purpose: sets the Brush Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SetBrush(Value: TBrush);
begin
  FBrush.Assign(Value);
  StyleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetDeltaX
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the DeltaX displacement Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SetDeltaX(Value: Integer);
begin
  if (FDeltaX = Value) then exit;
  FDeltaX := Value;
  StyleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetDeltaY
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the DeltaY displacement Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SetDeltaY(Value: Integer);
begin
  if (FDeltaY = Value) then exit;
  FDeltaY := Value;
  StyleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetName
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Name Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SetName(Value: String);
begin
  if (FName = Value) then exit;
  FName := Value;
  StyleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetPen
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Pen Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SetPen(Value: TPen);
begin
  FPen.Assign(Value);
  StyleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetSymbol
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Symbol Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SetSymbol(Value: TSymbol);
begin
  if (FSymbol = Value) then exit;
  FSymbol := Value;
  StyleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetSymbolSize
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the SymbolSize Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SetSymbolSize(Value: Integer);
begin
  if ((FSymbolSize = Value) or (FSymbolSize < 0)) then exit;
  FSymbolSize := Value;
  StyleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetVisible
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Visible Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SetVisible(Value: Boolean);
begin
{Can't become visible if Axes or Data have not been set:}
  if ((FXAxis = nil) or (FYAxis = nil)) then raise
    EInvalidPointer.CreateFmt('SetVisible: the X and Y Axis pointers are invalid !' +
      CRLF + '(X: %p; Y: %p)', [FXAxis, FYAxis]);

  FVisible := Value;
  if assigned(FOnChange) then OnChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SetYAxisIndex
  Description: property Setting procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the YAxis Property
 Known Issues: We define YAxisIndex to run from 1 to FAxisList.Count-1
 ------------------------------------------------------------------------------}
procedure TSeries.SetYAxisIndex(Value: Byte);
begin
  if ((Value < 1) or
      (Value >= FAxisList.Count)) then raise
    ERangeError.Create('TSeries.SetYAxisIndex: the new Y-Axis Index (%d) must be between 1 and %d !');

  FYAxisIndex := Value;
  FYAxis := TAxis(FAxisList[Value]);
  FYAxis.Visible := TRUE;
end;

{end Set procedures, begin general procedures ---------------------------------}
{------------------------------------------------------------------------------
     Function: TSeries.AllocateNoPts
  Description: allocates memory for data points
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: memory management
 Return Value: TRUE if successful
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.AllocateNoPts(Value: LongInt): Boolean;
var
  Msg: String;
begin
  AllocateNoPts := FALSE;

  if (Value < 0) then
    exit;

  try
{$IFDEF DELPHI1} {Delphi 1 can only allocate 64K chunks locally:}
    if ((Value) * SizeOf(Single) > 65535) then
    begin
      Value := 65535 div SizeOf(Single);
      ShowMessage(Format('Running out of memory - can only allocate %d points',
        [Value]));
      AllocateNoPts := FALSE;
    end;
{$ENDIF}

    if (FExternalXSeries) then
    begin
{we don't allocate memory for X data that is held in a different series:}
      FXData := Self.FXDataSeries.XData
{doesn't hurt, but should not be neccessary.}
    end
    else
{$IFDEF DELPHI1}
    begin
      if (FXData = nil) then
        GetMem(FXData, Value * SizeOf(Single))
       else
        ReAllocMem(FXData, MemSize * SizeOf(Single), Value * SizeOf(Single));
    end;
    if (FYData = nil) then
      GetMem(FYData, Value * SizeOf(Single))
     else
      ReAllocMem(FYData, MemSize * SizeOf(Single), Value * SizeOf(Single));
{$ELSE}
    begin
      ReAllocMem(FXData, Value * SizeOf(Single));
    end;
    ReAllocMem(FYData, Value * SizeOf(Single));
    AllocateNoPts := TRUE;
{$ENDIF}
  except
    Msg := Format('Problem with the %s data series !' + CRLF +
      'Requested No Pts = %d, requested memory = %d bytes',
      [FName, Value, Value * SizeOf(Single)]);
    ShowMessage(Msg);
    raise;
  end;
  MemSize := Value;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.StyleChange
  Description: Fires the OnChange event
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: event handling
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.StyleChange;
begin
  if (Assigned(FOnChange) and Visible) then OnChange(Self);
end;

{Data manipulation Functions and Procedures----------------------------------}
{------------------------------------------------------------------------------
     Function: TSeries.AddData
  Description: adds data from an externally-managed array
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management
 Return Value: TRUE if successful
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.AddData(XPointer, YPointer: pSingleArray; NumberOfPoints: Integer): Boolean;
var
  i: Integer;
begin
{clear any existing data:}
  if (FNoPts > 0) then DelData;

  try
{Allocate memory:}
    AllocateNoPts(NumberOfPoints + FDefSize);

{NB: this causes terminal access violations:
      System.Move(XPointer, FXData, NumberOfPoints * SizeOf(Single));}
    if (not FExternalXSeries) then
    begin
      for i := 0 to NumberOfPoints-1 do
        FXData^[i] := XPointer^[i];
    end;
    for i := 0 to NumberOfPoints-1 do
      FYData^[i] := YPointer^[i];

    DataStatus := dsInternal;
    FNoPts := NumberOfPoints;

{find the new min and max:}
    GetBounds; {which calls ResetBounds}

    StyleChange;
    AddData := TRUE;
  except
    AddData := FALSE;
  end;
end;

{------------------------------------------------------------------------------
     Function: TSeries.MakeXDataIndependent
  Description: This procedure makes an internal copy of external X Data
       Author: Mat Ballard
 Date created: 08/31/2000
Date modified: 08/31/2000 by Mat Ballard
      Purpose: series management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.MakeXDataIndependent;
var
  i: Integer;
  pOldXData: pSingleArray;
  Msg: String;
begin
  if (not FExternalXSeries) then exit;

  pOldXData := FXData;
  try
{$IFDEF DELPHI1}
    GetMem(FXData, MemSize * SizeOf(Single));
{$ELSE}
    ReAllocMem(FXData, MemSize * SizeOf(Single));
{$ENDIF}
{NB: the following generates gross access violations:
    System.Move(pOldXData, FXData, FNoPts * SizeOf(Single));}
    for i := 0 to FNoPts-1 do
      FXData^[i] := pOldXData^[i];
    FXDataSeries.RemoveDependentSeries(Self);
    FExternalXSeries := FALSE;
  except
    Msg := Format('Problem with the %s data series !' + CRLF +
      'Requested No Pts = %d, requested memory = %d bytes',
      [FName, MemSize, MemSize * SizeOf(Single)]);
    ShowMessage(Msg);
    raise;
  end;
end;

{------------------------------------------------------------------------------
     Function: TSeries.AddDependentSeries
  Description: This function ADDS a series that depends on this series' X-Data from the list of dependent series.
       Author: Mat Ballard
 Date created: 08/25/2000
Date modified: 08/25/2000 by Mat Ballard
      Purpose: data management
 Return Value: TRUE if successful
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.AddDependentSeries(ASeries: TSeries): Boolean;
{var
  pASeries: Pointer;}
begin
  if (FDependentSeries.IndexOf(ASeries) < 0) then
  begin
    FDependentSeries.Add(ASeries);
    AddDependentSeries := TRUE;
  end
  else
    AddDependentSeries := FALSE;
end;

{------------------------------------------------------------------------------
     Function: TSeries.RemoveDependentSeries
  Description: This function REMOVES a series that depends on this series' X-Data from the list of dependent series.
       Author: Mat Ballard
 Date created: 08/25/2000
Date modified: 08/25/2000 by Mat Ballard
      Purpose: data management
 Return Value: TRUE if successful
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.RemoveDependentSeries(ASeries: TSeries): Boolean;
{var
  pASeries: Pointer;}
begin
  if (FDependentSeries.IndexOf(ASeries) >= 0) then
  begin
    FDependentSeries.Remove(ASeries);
    RemoveDependentSeries := TRUE;
  end
  else
    RemoveDependentSeries := FALSE;
end;

{------------------------------------------------------------------------------
     Function: TSeries.GetXDataRefCount
  Description: This function returns the number of dependent series
       Author: Mat Ballard
 Date created: 08/25/2000
Date modified: 08/25/2000 by Mat Ballard
      Purpose: data management
 Return Value: Word
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.GetXDataRefCount: Word;
begin
  GetXDataRefCount := FDependentSeries.Count;
end;

{------------------------------------------------------------------------------
     Function: TSeries.AssumeMasterSeries
  Description: This function makes this series' X-Data the MASTER for the given list of dependent series.
       Author: Mat Ballard
 Date created: 08/25/2000
Date modified: 08/25/2000 by Mat Ballard
      Purpose: data management
 Return Value: TRUE if successful
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.AssumeMasterSeries(
  XPts: Integer;
  OldMaster: TSeries;
  AList: TList): Boolean;
var
  i: Integer;
begin
{There are many reasons why this might be a bad idea:}
  if (OldMaster = nil) then raise
    EComponentError.Create(Self.Name +
      ' cannot Assume Mastery from a NIL Series !');

  if (OldMaster <> FXDataSeries) then raise
    EComponentError.Create(Self.Name +
      ' cannot Assume Mastery from ' + FXDataSeries.Name +
      ' because ' + FXDataSeries.Name + ' is NOT its X Data Master !');

  if (XPts <> FNoPts) then raise
    EComponentError.CreateFmt(Self.Name +
      ' (%d points) cannot Assume a Master (X) Series with %d points !',
      [FNoPts, XPts]);

{this last is probably redundant because of test #2:}
  if (FDependentSeries.Count > 0) then raise
    EComponentError.Create(Self.Name +
      ' cannot Assume a Master (X) Series because it already IS a Master Series !');

  for i := 0 to AList.Count-1 do
  begin
    if (AList.Items[i] <> Self) then
    begin
{add these dependent series to our own list:}
      FDependentSeries.Add(AList.Items[i]);
{tell them that this series is now the Master:}
      TSeries(AList.Items[i]).ResetXDataSeries(OldMaster, Self);
    end;
  end;

{the X Data is now internal to this series:}
  FExternalXSeries := FALSE;
  FXDataSeries := nil;
{note that we already KNOW the location of the X Data: FXData !}

  AssumeMasterSeries := TRUE;
end;

{------------------------------------------------------------------------------
     Function: TSeries.ResetXDataSeries
  Description: When a new series Assumes X Data Master status, it has to tell
               all the dependent series
       Author: Mat Ballard
 Date created: 08/25/2000
Date modified: 08/25/2000 by Mat Ballard
      Purpose: data management
 Return Value: TRUE if successful
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.ResetXDataSeries(OldSeries, NewSeries: TSeries): Boolean;
begin
  if (FXDataSeries = OldSeries) then
  begin
    FXDataSeries := NewSeries;
    ResetXDataSeries := TRUE;
  end
  else
    ResetXDataSeries := FALSE;
end;

{------------------------------------------------------------------------------
     Function: TSeries.PointToData
  Description: uses data from an externally-managed array
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management
 Return Value: TRUE if successful
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.PointToData(XPointer, YPointer: pSingleArray; NumberOfPoints: Integer): Boolean;
begin
  PointToData := FALSE;
  if (DataStatus = dsNone) then
  begin
    DataStatus := dsExternal;
    FXData := XPointer;
    FYData := YPointer;
    FNoPts := NumberOfPoints;
    GetBounds; {which calls ResetBounds}
    StyleChange;
    PointToData := TRUE;
  end;
end;

{------------------------------------------------------------------------------
     Function: TSeries.AddDrawPoint
  Description: adds a point then draws it
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management and screen display
 Return Value: the number of data points
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.AddDrawPoint(X, Y: Single; ACanvas: TCanvas): Integer;
var
  iX, iY: Integer;
  TheResult: Integer;
begin
{Add the point; we don't fire any events, but we do adjust axes if required:
 this may trigger a re-draw if Min/Max are exceeded:}
  TheResult := AddPoint(X, Y, FALSE, TRUE);
  AddDrawPoint := TheResult;
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeries.AddDrawPoint: ACanvas is nil !');
{$ENDIF}
  if ((not FVisible) or
      (TheResult < 0)) then exit;

{Draw from last to this point:}
  ACanvas.Pen.Assign(FPen);
  if (FNoPts > 1) then
  begin
    iX := FXAxis.FofX(FXData^[FNoPts-2])+ FDeltaX;
    iY := FYAxis.FofY(FYData^[FNoPts-2]) + FDeltaY;
    ACanvas.MoveTo(iX, iY);
    iX := FXAxis.FofX(FXData^[FNoPts-1]) + FDeltaX;
    iY := FYAxis.FofY(FYData^[FNoPts-1]) + FDeltaY;
    ACanvas.LineTo(iX, iY);
  end
  else
  begin
    iX := FXAxis.FofX(FXData^[FNoPts-1]) + FDeltaX;
    iY := FYAxis.FofY(FYData^[FNoPts-1]) + FDeltaY;
  end;
  if ((FSymbol <> sNone) and (FSymbolSize > 0)) then
    DrawSymbol(ACanvas, iX, iY);
end;

{------------------------------------------------------------------------------
     Function: TSeries.AddPoint
  Description: adds a data point, increasing memory if required
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management
 Return Value: the number of data points
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.AddPoint(X, Y: Single; FireEvent, AdjustAxes: Boolean): Integer;
{This adds a single point (X, Y) to the data and returns the number of points:
 -1 indicates failure. If no memory has been allocated for the data yet, then
 IncMemSize is called automatically.}
{}
{If FireEvent is FALSE, then the Min and Max X and Y values are checked, and the
 OnChange event fired. Otherwise, these two slow steps are skipped.}
begin
  AddPoint := -1;

{$IFDEF DELPHI3_UP}
  Assert(DataStatus <> dsExternal,
    'TSeries.AddPoint: I cannot add data points to the ' + Name + ' series' +
      CRLF + 'because it is externally managed !');
{$ENDIF}

  if (DataStatus = dsNone) then
  begin
    DataStatus := dsInternal;
    if (not IncMemSize) then exit;  {will return false and exit if not enough memory}
    ResetBounds;
  end;

{Check memory available:}
  if (FNoPts >= MemSize-2) then
    if (not IncMemSize) then exit; {will return false and exit if not enough memory}

{If the X data is in another series, then we do not add it:}
  if (FExternalXSeries) then
  begin
{check validity of the External X data:}
    if (FXDataSeries = nil) then raise
      EAccessViolation.Create('AddPoint: I cannot add Y values to the ' + Name +
        ' series because the FXDataSeries is undefined !');
    if (FXDataSeries.NoPts <= FNoPts) then raise
      ERangeError.CreateFmt('AddPoint: the External X data series contains %d points, and I contain %d',
        [FXDataSeries.NoPts, FNoPts]);
    if (FXDataSeries.XData = nil) then  raise
      EAccessViolation.Create('AddPoint: I cannot add Y values to the ' + Name +
        ' series because the FXDataSeries X Data pointer is undefined !');
  end
  else
  begin
{save the X data:}
    FXData^[FNoPts] := X;
  end;

{save the Y data:}
  FYData^[FNoPts] := Y;

{Check the min and max X and Y properties of the series,
 and adjust axes as required:}
  CheckBounds(FNoPts, AdjustAxes);
  if (FireEvent) then
    StyleChange;

  DataChanged := TRUE;
  Inc(FNoPts);
  AddPoint := FNoPts;
end;

{------------------------------------------------------------------------------
     Function: TSeries.DelPoint
  Description: deletes the point nearest to X and Y
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management
 Return Value: the new number of points
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.DelPoint(X, Y: Single; Confirm: Boolean): Integer;
{This deletes a single point, the closest one, from the xy data.}
var
  i, ThePoint: Integer;
  Distance, MinDistance: Single;
begin
  DelPoint := -1;
  if (FNoPts <= 0) then raise
    ERangeError.CreateFmt('DelPoint: this series (%s) contains %d points, so I cannot delete any !', [FName, FNoPts]);

  MinDistance := 3.4e38;
  ThePoint := -1;
  for i := 0 to FNoPts-1 do
  begin
    Distance := Abs(X - FXData^[i]) + Abs(Y - FYData^[i]);
    if (MinDistance > Distance) then
    begin
      ThePoint := i;
    end;
  end;

  if (ThePoint = -1) then
  begin
    exit;
  end;

  DelPoint := DelPointNumber(ThePoint, Confirm);
end;

{------------------------------------------------------------------------------
     Function: TSeries.DelPointNumber
  Description: deletes ThePoint by its index
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management
 Return Value: the new number of points
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.DelPointNumber(ThePoint: Integer; Confirm: Boolean): Integer;
{This deletes a single point, the Nth one, from the xy data.}
{}
{Note: this DOES NOT delete the X Value of externally-maintained X data
 values, so it shifts the upper half of the series one point to the left.}
var
  i: Integer;
begin
  DelPointNumber := -1;

  if (FNoPts <= 0) then raise
    ERangeError.CreateFmt('DelPointNumber: this series (%s) contains %d points, so I cannot delete any !',
      [FName, FNoPts]);
  if ((ThePoint < 0) or (ThePoint >= FNoPts)) then raise
    ERangeError.CreateFmt('DelPointNumber: this series (%s) contains %d points, so I cannot delete point %d !',
      [FName, FNoPts, ThePoint]);
  if (FDependentSeries.Count > 0) then
  begin
    ERangeError.CreateFmt(
      'I cannot delete any points when %d other series use my X Data !',
      [FDependentSeries.Count]);
    exit;
  end;

  if (Confirm) then
  begin
    {if (IDNO = MessageBox(0, PChar(Format('Delete Point %d: (%e.3, %e.3) ?',
      [ThePoint, FXData^[ThePoint], FYData^[ThePoint]])), 'Delete Point', MB_YESNO)) then}
    if (IDNO = MessageDlg(Format('Delete Point %d: (%e.3, %e.3) ?',
        [ThePoint, FXData^[ThePoint], FYData^[ThePoint]]),
        mtConfirmation, [mbYes, mbNo], 0)) then
      exit;
  end;

{we now use the slower method to be more consistent with the
 dynamic array approach:}
  if (not FExternalXSeries) then
  begin
    for i := ThePoint to FNoPts-2 do
    begin
      FXData^[i] := FXData^[i+1];
    end;
  end;

  for i := ThePoint to FNoPts-1 do
  begin
    FYData^[i] := FYData^[i+1];
  end;

  Dec(FNoPts);

  DataChanged := TRUE;
  StyleChange;
  DelPointNumber := FNoPts;
end;

{------------------------------------------------------------------------------
     Function: TSeries.DelData
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: deletes an entire data set. It only works on internal data sets.
 Return Value: TRUE if successful
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.DelData: Boolean;
begin
  DelData := FALSE;
  if (DataStatus = dsNone) then exit;

  if (FDependentSeries.Count > 0) then
  begin
{Merde ! this series is being destroyed, but other series depend on it !
 we therefore need to "pass the buck": pass the X Data that we've created, and
 the list of dependent series to another series:}
    TSeries(FDependentSeries.Items[0]).AssumeMasterSeries(FNoPts, Self, FDependentSeries);
{and now, the X Data is managed by an external series:}
    FExternalXSeries := TRUE;
    FDependentSeries.Clear;
  end;

  if (DataStatus = dsInternal) then
  begin
    if (not FExternalXSeries) then
{$IFDEF DELPHI1}
      FreeMem(FXData, MemSize * SizeOf(Single));
    FreeMem(FYData, MemSize * SizeOf(Single));
{$ELSE}
      ReAllocMem(FXData, 0);
    ReAllocMem(FYData, 0);
{$ENDIF}
    FXData := nil;
    FYData := nil;
  end;

  FExternalXSeries := FALSE;
  FNoPts := 0;
  DataStatus := dsNone;
  MemSize := 0;
  ResetBounds;

  DataChanged := TRUE;
  DelData := TRUE;
  StyleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.ClearHighsLows
  Description: frees the Highs and Lows, and their Counts and Capacities
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Series analysis
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.ClearHighsLows;
begin
  if (FHighs <> nil) then
  begin
    FreeMem(FHighs, FHighCapacity * SizeOf(Integer));
    FHighs := nil;
  end;
  if (FLows <> nil) then
  begin
    FreeMem(FLows, FHighCapacity * SizeOf(Integer));
    FLows := nil;
  end;
  
  FHighLow := [];
  FHighCapacity := 10;
  FHighCount := 0;
  FLowCount := 0;
end;

{------------------------------------------------------------------------------
     Function: TSeries.Average
  Description: calculates the average of a series over a range
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: numerical calculation
 Return Value: the average: single
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.Average(TheLeft, TheRight: Single): Single;
var
  i,
  Start,
  Finish,
  Number: Integer;
  Sum: Single;
begin
  if (TheLeft > TheRight) then
  begin
{swap TheLeft and TheRight}
    Sum := TheLeft;
    TheLeft := TheRight;
    TheRight := Sum;
  end;

{get the TheLeft and TheRight points:}
  Start := GetNearestPointToX(TheLeft);
  Finish := GetNearestPointToX(TheRight);

{adjust TheLeft and TheRight:}
  if (XData^[Start] < TheLeft) then
    Inc(Start);
  if (XData^[Finish] > TheRight) then
    Dec(Finish);

{initialize:}
  Number := 0;
  Sum := 0;
  for i := Start to Finish do
  begin
    Sum := Sum + YData^[i];
    Inc(Number);
  end;

  Average := Sum / Number;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.MovingAverage
  Description: Calculates the movong average
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Smoothing
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.MovingAverage(Span: Integer);
var
  i, j,
  Left, Right: Integer;
  AverageData: pSingleArray;
begin
{allocate memory for arrays:}
  GetMem(AverageData, FNoPts * SizeOf(Single));

  for i := 0 to FNoPts-1 do
  begin
    AverageData^[i] := 0;
    Left := i - Span;
    Right := i + Span;

    if (Left < 0) then
    begin
      Right := 2*i;
      Left := 0;
    end;
    if (Right >= FNoPts) then
    begin
      Left := i - (FNoPts-1 - i);
      Right := FNoPts-1;
    end;

    for j := Left to Right do
    begin
      AverageData^[i] := AverageData^[i] + YData^[j];
    end;
    AverageData^[i] := AverageData^[i] / (1 + Right - Left);
  end;

{NB: the following generates gross access violations:
  System.Move(AverageData, FYData, FNoPts * SizeOf(Single));}
  for i := 0 to FNoPts-1 do
    FYData^[i] := AverageData^[i];

  FreeMem(AverageData, FNoPts * SizeOf(Single));
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.DoSpline
  Description: Does the cubic spline of the data
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Places the cubic spline interpolation into X and Y
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.DoSpline(Density: Integer; pSplineSeries: TSeries);
var
  i,
  j: Integer;
  dX,
  X: Single;
begin
{calculate the ...}
  SecondDerivative;

{Index of the new spline points:}
  for i := 0 to FNoPts-2 do
  begin
    pSplineSeries.AddPoint(XData^[i], YData^[i], FALSE, FALSE);
    dX := (XData^[i+1] - XData^[i]) / (Density+1);
    X := XData^[i];
    for j := 0 to Density-1 do
    begin
      X := X + dX;
      pSplineSeries.AddPoint(X, SplineValue(X), FALSE, FALSE);
    end;
  end;
  pSplineSeries.AddPoint(XData^[FNoPts-1], YData^[FNoPts-1], FALSE, FALSE);
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SplineValue
  Description: Calculates the Y co-ordinate from the cubic spline
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data manipulation
 Known Issues: yet to be done
 ------------------------------------------------------------------------------}
function TSeries.SplineValue(X: Single): Single;
var
  iLeft,
  iRight,
  i: integer;
  dX,
  LeftX,
  RightX: Single;
begin
{in initialize left and right indices:}
  iLeft := 0;
  iRight := FNoPts-1;

{bracket the X value using binary search:}
  while (iRight - iLeft > 1) do
  begin
    i := (iRight+iLeft) div 2;
    if (XData^[i] > X) then
      iRight := i
     else
      iLeft := i;
  end;
{width of bracketing interval is:}
  dX := XData^[iRight] - XData^[iLeft];

{should we chuck a loopy ?}
  if (dX = 0.0) then raise
    ERangeError.CreateFmt('TSeries.SplineValue: bad input data (dX = 0) !' + CRLF+
      'XData[%d] = %g, XData[%d] = %g', [iRight, XData^[iRight], iLeft, XData^[iLeft]]);

{the right and left portions are:}
  RightX := (XData^[iRight]-X) / dX;
  LeftX := (X-XData^[iLeft]) / dX;

{so the cubic spline estimate is:}
  SplineValue := RightX * YData^[iLeft] + LeftX * YData^[iRight] +
    ((IntPower(RightX, 3) - RightX) * Fd2Y_dX2^[iLeft] +
      (IntPower(LeftX, 3) - LeftX) * Fd2Y_dX2^[iRight]) *
        Sqr(dX) / 6.0;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.SecondDerivative
  Description: Does the cubic spline of the data
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Calculates the second derivatives for use by SplineValue
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.SecondDerivative;
var
  i: integer;
  TempVar,
  LeftXFraction: Single;
  UpperTriangle: pSingleArray;
begin
  ClearSpline;

{allocate memory for the second derivatives:}
  Size2ndDeriv := FNoPts * SizeOf(Single);
  GetMem(Fd2Y_dX2, Size2ndDeriv);

  GetMem(UpperTriangle, FNoPts * SizeOf(Single));

{handle the first point: we use "natural" boundary condition of
 zero second derivative:}
  Fd2Y_dX2^[0] := 0;
  UpperTriangle^[0] := 0;

{do the loop over middle points:}
  for i := 1 to FNoPts-2 do begin
    LeftXFraction := (XData^[i] - XData^[i-1]) /
      (XData^[i+1] - XData^[i-1]);
    TempVar := LeftXFraction * Fd2Y_dX2^[i-1] + 2.0;
    Fd2Y_dX2^[i] := (LeftXFraction - 1.0) / TempVar;
    UpperTriangle^[i] := (YData^[i+1] - YData^[i]) / (XData^[i+1] - XData^[i]) -
      (YData^[i] - YData^[i-1]) / (XData^[i] - XData^[i-1]);
    UpperTriangle^[i] := (6.0 * UpperTriangle^[i] / (XData^[i+1] - XData^[i-1]) -
      LeftXFraction * UpperTriangle^[i-1]) / TempVar;
  end;

{handle the last point: we use "natural" boundary condition of
 zero second derivative:}
  Fd2Y_dX2^[FNoPts-1] := 0;

  for i := FNoPts-2 downto 0 do
  begin
    Fd2Y_dX2^[i] := Fd2Y_dX2^[i] * Fd2Y_dX2^[i+1] + UpperTriangle^[i];
  end;
  FreeMem(UpperTriangle, FNoPts * SizeOf(Single));
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.ClearSpline
  Description: frees the second derivative memory
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Spline memory management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.ClearSpline;
begin
  if (Fd2Y_dX2 <> nil) then
  begin
    FreeMem(Fd2Y_dX2, Size2ndDeriv);
    Fd2Y_dX2 := nil;
    Size2ndDeriv := 0;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.Differentiate
  Description: Replaces the Series Y data with its differential
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Data manipulation
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.Differentiate;
var
  i: Integer;
  Differential,
  YOld: Single;
begin
{we save the first data point:}
  YOld := YData^[0];

{now do the first point by difference (1st order):}
  YData^[0] :=
    (YData^[1] - YData^[0]) / (XData^[1] - XData^[0]);

  for i := 1 to FNoPts-2 do
  begin
{we calculate a mid-point (2nd order) differential}
    Differential :=
      (((YData^[i] - YOld) / (XData^[i] - XData^[i-1])) +
       ((YData^[i+1] - YData^[i]) / (XData^[i+1] - XData^[i])))
       / 2;
    YOld := YData^[i];
    YData^[i] := Differential;
  end;

{now do the last point by difference (1st order):}
  YData^[FNoPts-1] :=
    (YData^[FNoPts-1] - YOld) / (XData^[FNoPts-1] - XData^[FNoPts-2]);

{re-scale:}    
  FDeltaX := 0;
  FDeltaY := 0;
  ResetBounds;
  GetBounds;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.Integrate
  Description: Replaces the Series Y data with its integral
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Data manipulation
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.Integrate;
var
  i: Integer;
  Sum,
  YOld: Single;
begin
  Sum := 0;
  YOld := YData^[0];
  for i := 1 to FNoPts-1 do
  begin
    Sum := Sum +
      (YData^[i] + YOld) * (XData^[i] - XData^[i-1]) / 2;
    YOld := YData^[i];
    YData^[i] := Sum;
  end;
{we set the first data point:}
  YData^[0] := 0;

{re-scale:}
  FDeltaX := 0;
  FDeltaY := 0;
  ResetBounds;
  GetBounds;
end;

{------------------------------------------------------------------------------
     Function: TSeries.Integral
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the integral of the Series from Start to Finish
 Return Value: the real area
 Known Issues: see IntegralByPoint below
 ------------------------------------------------------------------------------}
function TSeries.Integral(TheLeft, TheRight: Single): Single;
var
  Start,
  Finish: Integer;
  Sum,
  YEst: Single;
begin
  if (TheLeft > TheRight) then
  begin
{swap TheLeft and TheRight}
    Sum := TheLeft;
    TheLeft := TheRight;
    TheRight := Sum;
  end;

{get the TheLeft and TheRight points:}
  Start := GetNearestPointToX(TheLeft);
  Finish := GetNearestPointToX(TheRight);

{adjust TheLeft and TheRight:}
  if (XData^[Start] < TheLeft) then
    Inc(Start);
  if (XData^[Finish] > TheRight) then
    Dec(Finish);

{Integrate the bulk:}
  Sum := IntegralByPoint(Start, Finish);

{Add the end bits:}
  if ((Start > 0) and
      (XData^[Start] <> TheLeft)) then
  begin
    YEst := YData^[Start-1] +
      (YData^[Start] - YData^[Start-1]) *
      (TheLeft - XData^[Start-1]) / (XData^[Start] - XData^[Start-1]);
    Sum := Sum +
      (XData^[Start] - TheLeft) *
      (YData^[Start] + YEst) / 2;
  end;
  if ((Finish < FNoPts-1) and
      (XData^[Finish] <> TheRight)) then
  begin
    YEst := YData^[Finish] +
      (YData^[Finish+1] - YData^[Finish]) *
      (TheRight - XData^[Finish]) / (XData^[Finish+1] - XData^[Finish]);
    Sum := Sum +
      (TheRight - XData^[Finish]) *
      (YData^[Finish] + YEst) / 2;
  end;

  Integral := Sum;
end;

{------------------------------------------------------------------------------
     Function: TSeries.IntegralByPoint
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the integral of the Series from iStart to iFinish
 Return Value: the real area
 Known Issues: see Integral above
 ------------------------------------------------------------------------------}
function TSeries.IntegralByPoint(Start, Finish: Integer): Single;
var
  i: Integer;
  Sum: Single;
begin
  if (Start > Finish) then
  begin
{swap Start and Finish}
    i := Start;
    Start := Finish;
    Finish := i;
  end;

  Sum := 0;
  for i := Start+1 to Finish do
  begin
    Sum := Sum +
      (YData^[i] + YData^[i-1]) * (XData^[i] - XData^[i-1]) / 2;
  end;
{we set the first data point:}
  IntegralByPoint := Sum;
end;

{------------------------------------------------------------------------------
     Function: TSeries.IncMemSize
  Description: increases the available memory for data
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data and memory management
 Return Value: TRUE if successful
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.IncMemSize: Boolean;
begin
  IncMemSize := AllocateNoPts(MemSize + FDefSize);
end;

{------------------------------------------------------------------------------
     Function: TSeries.InsertPoint
  Description: inserts a data point
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the value of the ??? Property
 Return Value: new number of data points
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.InsertPoint(X, Y: Single): Integer;
{This inserts a single point in the xy data and returns the location of the point:
-1 indicates failure. The point is inserted at the appropriate X value.}
{}
{Note: this DOES NOT insert an X Value into externally-maintained X data
 values, so it shifts the upper half of the series one point to the right.}
var
  i, ThePoint: Integer;
begin
  InsertPoint := -1;

  if ((DataStatus = dsNone) or (FNoPts = 0))then
  begin
{we add the point, firing events and adjusting axes as neccessary:}
    InsertPoint := AddPoint(X, Y, TRUE, TRUE);
    exit;
  end;

{Find out where to insert this point:}
  ThePoint := 0;
  {TheXPointer := FXData;}
  for i := 0 to FNoPts-1 do
  begin
    if (FXData^[i] > X) then
    begin
      ThePoint := i;
    end
    else if (ThePoint > 0) then
    begin
      break;
    end;
    {Inc(TheXPointer);}
  end;

  if (ThePoint = FNoPts-1) then
  begin
{we add the point, firing events and adjusting axes as neccessary:}
    InsertPoint := AddPoint(X, Y, TRUE, TRUE);
    exit;
  end;

{Check memory available:}
  if (FNoPts >= MemSize-2) then
    if (not IncMemSize) then exit; {will return false and exit if not enough memory}

  if (not FExternalXSeries) then
  begin
    for i := FNoPts downto ThePoint+1 do
    begin
      FXData^[i] := FXData^[i-1];
    end;
    FXData^[ThePoint] := X;
  end;
  for i := FNoPts downto ThePoint+1 do
  begin
    FYData^[i] := FYData^[i-1];
  end;
  FYData^[ThePoint] := Y;

  Inc(FNoPts);

  DataChanged := TRUE;
  StyleChange;
  InsertPoint := FNoPts;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.Smooth
  Description: smoothes the data using a modified Savitsky-Golay method 
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data manipulation
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.Smooth(SmoothOrder: Integer);
var
  i, j, K: Integer;
  Start, Finish: Integer;
  IntSum: Integer;
  Sum, SumStart, SumFinish: Single;
  pSmoothData: pSingleArray;
  SCMatrix, pSCMatrix: pInteger; {Longword ?}
{define pSCMatrix(i, j) == pSCMatrix(i + (FNoPts+1) * j)}
  pSCSum: pIntegerArray;    {Longword ?}
  {Msg: String;}

{NOTE: Multidimensional dynamic arrays DON'T WORK !}

  procedure SetSCMatrixPointer(i, j: Integer);
  begin
    if (SmoothOrder < 2) then raise
      ERangeError.CreateFmt('SetSCMatrixPointer: SCMatrix(%d, %d) does not exist !',
        [i, j]);
    pSCMatrix := SCMatrix;
    Inc(pSCMatrix, i + (SmoothOrder+1) * j);
  end;

  {procedure DisplayMatrix;
  var
    ii, jj: Integer;
    DMsg: String;
  begin
  //display the matrix:
    DMsg := Format('Smooth Order = %d, Start = %d, Finish = %d',
                  [SmoothOrder, Start, Finish]) + CRLF;
    DMsg := DMsg + CRLF + 'The smoothing Matrix is:';
    For ii := 0 To SmoothOrder do
    begin
      DMsg := DMsg + CRLF;
      For jj := 0 To SmoothOrder do
      begin
        SetSCMatrixPointer(ii, jj);
        DMsg := DMsg + IntToStr(pSCMatrix^) + ', '
      end;
    end;

    DMsg := DMsg + CRLF + CRLF+ 'The smoothing Sums are:' + #13+#10;
    pSCSum := SCSum;
    For ii := 0 To SmoothOrder do
    begin
      DMsg := DMsg + IntToStr(pSCSum^) + ', ';
      Inc(pSCSum);
    end;

    ShowMessage(DMsg);
  end;}

begin
  if ((SmoothOrder < 2) or (SmoothOrder > 20)) then raise
    ERangeError.CreateFmt('Smooth: the Smoothing Order %d must be in the range: 2...20',
      [SmoothOrder]);

  if (FNoPts <= SmoothOrder+1) then raise
    ERangeError.CreateFmt('Smooth: the Smoothing Order (%d) must be less than the number of points (%d) !',
      [SmoothOrder, FNoPts]);

{allocate memory for arrays:}
  GetMem(pSmoothData, FNoPts * SizeOf(Single));
  GetMem(SCMatrix, (SmoothOrder+1) * (SmoothOrder+1) * SizeOf(Integer));
  GetMem(pSCSum, (SmoothOrder+1) * SizeOf(Integer));

{Zero the matrix:}
  For i := 0 To SmoothOrder do {i <=> Rows}
  begin
    For j := 0 to SmoothOrder do {j <=> Column}
    begin
      SetSCMatrixPointer(i, j);
      pSCMatrix^ := 0;
    end;
  end;

{set the first column and the diagonals to 1:}
  For i := 0 To SmoothOrder do
  begin
    SetSCMatrixPointer(i, 0);
    pSCMatrix^ := 1;
    SetSCMatrixPointer(i, i);
    pSCMatrix^ := 1;
  end;

{Calculate the Smoothing Coefficients:
 now columns 1, 2, ... SmoothOrder:}
  For i := 2 To SmoothOrder do {i <=> Rows}
  begin
    For j := 1 to i-1 do {j <=> Column}
    begin
      SetSCMatrixPointer(i - 1, j - 1);
      IntSum := pSCMatrix^;
      SetSCMatrixPointer(i - 1, j);
      IntSum := IntSum + pSCMatrix^;
      SetSCMatrixPointer(i, j);
      pSCMatrix^ := IntSum;
    end;
  end;
{   For j% = 1 To SmoothOrder%
        For i% = j% To SmoothOrder%
            Sum! = 0
            For K% = 0 To i% - 1
                Sum! = Sum! + SC(K%, j% - 1)
            Next K%
            SC(i%, j%) = Sum!
        Next i%
    Next j%}

{Calculate the sums:}
  For i := 0 To SmoothOrder do {i <=> Rows}
  begin
    pSCSum^[i] := 0;
    For j := 0 To i do {j <=> Columns}
    begin
      SetSCMatrixPointer(i, j);
      pSCSum^[i] := pSCSum^[i] + pSCMatrix^;
    end;
  end;
{    For i% = 0 To SmoothOrder%
        SCSum(i%) = 0
        For j% = 0 To i%
            SCSum(i%) = SCSum(i%) + SC(i%, j%)
        Next j%
    Next i%}

{Calculate the starting and ending points:}
  Start := SmoothOrder div 2;
  Finish := FNoPts - Start;
{    Start% = Int(SmoothOrder% / 2)
    Finish% = Runs.No_Pts - Start%}

  {DisplayMatrix;}

{these first and last points don't change:}
  pSmoothData^[0] := FYData^[0];
  pSmoothData^[FNoPts-1] := FYData^[FNoPts-1];
{    Smooth_Data(0) = Y_Data(0)
    Smooth_Data(Runs.No_Pts) = Y_Data(Runs.No_Pts)}

{Do the messy points in between:}
  For K := 1 To (SmoothOrder - 2) div 2 do
{ For i% = 2 To (SmoothOrder% - 2) Step 2}
  begin
    i := 2*K;
    SumStart := 0;
    SumFinish := 0;
    For j := 0 To i do
    begin
      SetSCMatrixPointer(i, j);
      SumStart := SumStart + FYData^[j] * pSCMatrix^;
{     SumStart& = SumStart& + CLng(Y_Data(j%)) * CLng(SC(i%, j%))}
      SumFinish := SumFinish + FYData^[FNoPts-1-j] * pSCMatrix^;
{     SumFinish& = SumFinish& + CLng(Y_Data(Runs.No_Pts - j%)) * CLng(SC(i%, j%))}
    end;
    pSmoothData^[K] := SumStart / pSCSum^[i];
{   Smooth_Data(i% / 2) = SumStart& / SCSum(i%)}
    pSmoothData^[FNoPts-1-K] := SumFinish / pSCSum^[i];
{   Smooth_Data(Runs.No_Pts - i% / 2) = SumFinish& / SCSum(i%)}
  end;

{    For i% = 2 To (SmoothOrder% - 2) Step 2
        SumStart& = 0
        SumFinish& = 0
        For j% = 0 To i%
            SumStart& = SumStart& + CLng(Y_Data(j%)) * CLng(SC(i%, j%))
            SumFinish& = SumFinish& + CLng(Y_Data(Runs.No_Pts - j%)) * CLng(SC(i%, j%))
        Next j%
        Smooth_Data(i% / 2) = SumStart& / SCSum(i%)
        Smooth_Data(Runs.No_Pts - i% / 2) = SumFinish& / SCSum(i%)
    Next i%}

{loop over the fully-smoothed points:}
  For K := Start To Finish-1 do
  begin
    Sum := 0;
    For j := 0 To SmoothOrder do
    begin
      SetSCMatrixPointer(SmoothOrder, j);
      Sum := Sum + FYData^[K+j-Start] * pSCMatrix^;
{     Sum! = Sum! + Y_Data(K% + j% - Start%) * CSng(SC(SmoothOrder%, j%))}
    end;
    pSmoothData^[K] := Sum / pSCSum^[SmoothOrder];
{   Smooth_Data(K%) = Sum! / SCSum(SmoothOrder%)}
  end;

{finally, update the Y data:}
  For i := 0 To FNoPts-1 do
    FYData^[i] := pSmoothData^[i];
{NB: this causes terminal access violations:
  System.Move(pSmoothData, FYData, FNoPts * SizeOf(Single));}


{$IFDEF DELPHI1}
  FreeMem(pSmoothData, FNoPts * SizeOf(Single));
  FreeMem(SCMatrix, (SmoothOrder+1) * (SmoothOrder+1) * SizeOf(Integer));
  FreeMem(pSCSum, (SmoothOrder+1) * SizeOf(Integer));
{$ELSE}
  FreeMem(pSmoothData);
  FreeMem(SCMatrix);
  FreeMem(pSCSum);
{$ENDIF}

  DataChanged := TRUE;
  StyleChange;
end;

{Sub Smooth (SmoothOrder%, X_Data() As Single, Y_Data() As Single)

'   This function smooths the data using a midpoint method
'   Keywords:
'       smooth
'   Input:
'
'   Modifies:
'       nothing
'   Output:
'       none
'   Returns:
'
'   Called From:
'
'   Calls:
'

Dim i%, j%, K%
Dim Start%, Finish%
Dim SumStart&, SumFinish&
Dim Sum!
Dim Msg$
ReDim Smooth_Data(0 To Runs.Array_Size) As Single

    On Error GoTo Smooth_ErrorHandler

'   declare the matrix of coefficients for smoothing:
    ReDim SC(0 To SmoothOrder%, 0 To SmoothOrder%) As Long
    ReDim SCSum(0 To SmoothOrder%) As Long

'   set the first column to 1:
    For i% = 0 To SmoothOrder%
        SC(i%, 0) = 1
    Next i%

'   Calculate the Smoothing Coefficients:
'   now columns 1, 2, ... SmoothOrder%:
    For j% = 1 To SmoothOrder%
        For i% = j% To SmoothOrder%
            Sum! = 0
            For K% = 0 To i% - 1
                Sum! = Sum! + SC(K%, j% - 1)
            Next K%
            SC(i%, j%) = Sum!
        Next i%
    Next j%

'   Calculate the sums:
    For i% = 0 To SmoothOrder%
        SCSum(i%) = 0
        For j% = 0 To i%
            SCSum(i%) = SCSum(i%) + SC(i%, j%)
        Next j%
    Next i%

'    Msg$ = "Smoothing Matrix:"
'    For i% = 0 To SmoothOrder%
'        Msg$ = Msg$ & LF
'        For j% = 0 To SmoothOrder%
'            Msg$ = Msg$ & Str$(SC(i%, j%)) & ", "
'        Next j%
'    Next i%
'    Msg$ = Msg$ & LF & LF & "Smoothing Sums:"
'    For i% = 0 To SmoothOrder%
'        Msg$ = Msg$ & Str$(SCSum(i%)) & ", "
'    Next i%
'    MsgBox Msg$, MB_OK, "Smoothing"

'   Calculate the starting and ending points:
    Start% = Int(SmoothOrder% / 2)
    Finish% = Runs.No_Pts - Start%

'   Do the smooth; end points are not affected:
    Smooth_Data(0) = Y_Data(0)
    Smooth_Data(Runs.No_Pts) = Y_Data(Runs.No_Pts)
'   Do the messy points in between:
    For i% = 2 To (SmoothOrder% - 2) Step 2
        SumStart& = 0
        SumFinish& = 0
        For j% = 0 To i%
            SumStart& = SumStart& + CLng(Y_Data(j%)) * CLng(SC(i%, j%))
            SumFinish& = SumFinish& + CLng(Y_Data(Runs.No_Pts - j%)) * CLng(SC(i%, j%))
        Next j%
        Smooth_Data(i% / 2) = SumStart& / SCSum(i%)
        Smooth_Data(Runs.No_Pts - i% / 2) = SumFinish& / SCSum(i%)
    Next i%

'   loop over the fully-smoothed points:
    For K% = Start% To Finish%
        Sum! = 0
        For j% = 0 To SmoothOrder%
            Sum! = Sum! + Y_Data(K% + j% - Start%) * CSng(SC(SmoothOrder%, j%))
        Next j%
        Smooth_Data(K%) = Sum! / SCSum(SmoothOrder%)
    Next K%

'   finally, update the RI data:
    For i% = 0 To Runs.No_Pts
        Y_Data(i%) = Smooth_Data(i%)
    Next i%


Smooth_FINISHED:
    Refresh

Exit Sub

Smooth_ErrorHandler:   ' Error handler line label.

    Msg$ = "Panic in " & "Smooth_ErrorHandler !"
    Msg$ = Msg$ & LF & LF & "Error No. " & Str$(Err) & ": " & Error$
    Response% = Message(Msg$, MB_OK + MB_ICONEXCLAMATION, "Error !", NO, H_PANIC)

    Resume Smooth_FINISHED

End Sub
}

{------------------------------------------------------------------------------
    Procedure: TSeries.Sort
  Description: Sorts the data using the HeapSort method
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Data manipulation
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.Sort;
{$IFDEF DELPHI1}
begin
  ShowMessage('Sorting is not supported under Delphi 1.');
end;
{$ELSE}
var
  i: Integer;
  pMem: Pointer;
  pPoint: pXYPoint;
  TheList: TList;
begin
{create and initialize the list of points:}
  TheList := TList.Create;
  TheList.Capacity := FNoPts;
{allocate one big block of memory:}
  GetMem(pMem, FNoPts * SizeOf(TXYPoint));
{point at the beginning:}
  pPoint := pMem;

{loop over all points:}
  for i := 0 to FNoPts-1 do
  begin
    pPoint^.X := XData^[i];
    pPoint^.Y := YData^[i];
    TheList.Add(pPoint);
    Inc(pPoint);
  end;

{do the dirty deed:}
  TheList.Sort(Compare);

{point at the beginning:}
  pPoint := pMem;
{loop over all points to save results:}
  for i := 0 to FNoPts-1 do
  begin
    XData^[i] := pPoint^.X;
    YData^[i] := pPoint^.Y;
    Inc(pPoint);
  end;

  TheList.Free;
  FreeMem(pMem, FNoPts * SizeOf(TXYPoint));
end;
{$ENDIF}

{------------------------------------------------------------------------------
     Function: Compare
  Description: comparison function for sorting
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: compares the X ordinate of point for a TList quicksort
 Return Value: -1, 0 or 1
 Known Issues:
 ------------------------------------------------------------------------------}
function Compare(Item1, Item2: Pointer): Integer;
begin
  if (pXYPoint(Item1)^.X < pXYPoint(Item2)^.X) then
  begin
    Compare := -1;
  end
  else if (pXYPoint(Item1)^.X = pXYPoint(Item2)^.X) then
  begin
    Compare := 0;
  end
  else
  begin
    Compare := 1;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.GetPoint
  Description: returns the Nth (0..NoPts-1) point's X and Y values.
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.GetPoint(N: Integer; var X, Y: Single);
begin
  if ((N < 0) or (N >= FNoPts)) then raise
    ERangeError.CreateFmt('GetPoint: the Point number d is not within the valid range of %0..%d',
      [N, FNoPts]);

  X := FXData^[N];
  Y := FYData^[N];
end;

{------------------------------------------------------------------------------
     Function: TSeries.GetXYPoint
  Description: returns the Nth (0..NoPts-1) point's X and Y values.
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management
 Return Value: XY
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.GetXYPoint(N: Integer): TXYPoint;
{This returns the Nth (0..NoPts-1) point's X and Y values.}
var
  XY: TXYPoint;
begin
  if ((N < 0) or (N >= FNoPts)) then raise
    ERangeError.CreateFmt('GetXYPoint: the Point number %d is not within the valid range of %0..%d',
      [N, FNoPts]);

  XY.X := FXData^[N];
  XY.Y := FYData^[N];
  GetXYPoint := XY;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.Displace
  Description: Runs the "Displace" dialog box
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: user management of Series displacement
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.Displace;
var
  DisplacementForm: TDisplacementForm;
begin
  DisplacementForm := TDisplacementForm.Create(nil);
  with DisplacementForm do
  begin
    SeriesLabel.Caption := FName;

    if (ShowModal = mrOK) then
    begin
      FDeltaX := StrToInt(DeltaXEdit.Text);
      FDeltaY := StrToInt(DeltaYEdit.Text);
    end; {if}
  end;
  DisplacementForm.Free;

  StyleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.EditPoint
  Description: Runs the "EditPoint" dialog box
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: user management of Series displacement
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.EditPoint(ThePointNumber: Integer);
var
  PointEditorForm: TPointEditorForm;
  XNew, YNew: Single;
  TheResult: TModalResult;
begin
  PointEditorForm := TPointEditorForm.Create(nil);
  with PointEditorForm do
  begin
    Init(FXData, FYData, FXAxis, FYAxis);
    PointSlideBar.Max := FNoPts-1;
    PointSlideBar.Position := ThePointNumber;
    FillData(ThePointNumber);
    DetailsLabel.Caption := FName;

    if (FExternalXSeries) then
    begin
      XDataEdit.Enabled := FALSE;
      XScreenEdit.Enabled := FALSE;
    end;

    TheResult := ShowModal;

    if (TheResult <> mrCancel) then
    begin
      {ThePointNumber := PointSlideBar.Position;}
      ThePointNumber := PointSlideBar.Position;
      if (DataGroupBox.Enabled) then
      begin
        XNew := FXAxis.StrToLabel(XDataEdit.Text);
        YNew := FYAxis.StrToLabel(YDataEdit.Text);
      end
      else
      begin {base on screen co-ords:}
        XNew := FXAxis.XofF(StrToInt(XScreenEdit.Text));
        YNew := FYAxis.YofF(StrToInt(YScreenEdit.Text));
      end;
      case TheResult of
        mrOK:
          begin
            ReplacePoint(ThePointNumber, XNew, YNew);
            CheckBounds(ThePointNumber, TRUE);
          end;
        mrYes:
          begin
            AddPoint(XNew, YNew, TRUE, TRUE);
            CheckBounds(FNoPts, TRUE);
          end;
        mrNo:
          DelPointNumber(ThePointNumber, TRUE);
      end;

      DataChanged := TRUE;
    end; {if}
  end;
  PointEditorForm.Free;

  StyleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.ReplacePoint
  Description: Replaces the Nth point with new values
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.ReplacePoint(N: Integer; var NewX, NewY: Single);
begin
  if (DataStatus <> dsInternal) then exit;

  if ((N < 0) or (N >= FNoPts)) then raise
    ERangeError.CreateFmt('GetPoint: the Point number %d is not within the valid range of %0..%d',
      [N, FNoPts]);

  if (not FExternalXSeries) then
    FXData^[N] := NewX;
  FYData^[N] := NewY;

  DataChanged := TRUE;
  StyleChange;
end;

{Odds and sods --------------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TSeries.Contract
  Description: reduces the size of the data set by local averaging
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data manipulation and management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.Contract(ContractRatio: Integer);
var
  i, j, k: Integer;
  XSum, YSum: Single;
begin
  if ((ContractRatio < 2) or (FNoPts div ContractRatio < 10 )) then
  begin
{we used to throw an exception here, but this roots ContractAllSeries}
    ShowMessage(Format('TSeries.Contract: cannot contract %s (%d points) by a Ratio of %d !',
      [FName, FNoPts, ContractRatio]));
    exit;
  end;

  j := 0;
  k := 0;
  XSum := 0;
  YSum := 0;
  for i := 0 to FNoPts-1 do
  begin
    XSum := XSum + FXData^[i];
    YSum := YSum + FYData^[i];
    Inc(j);
    if (j = ContractRatio) then
    begin
      if (not FExternalXSeries) then
        FXData^[k] := XSum / j;
      FYData^[k] := YSum / j;
      j := 0;
      XSum := 0;
      YSum := 0;
      Inc(k);
    end;
  end; {for}
  if (j > 0) then
  begin
    if (not FExternalXSeries) then
      FXData^[k] := XSum / j;
    FYData^[k] := YSum / j;
    Inc(k);
  end;
  FNoPts := k;

  DataChanged := TRUE;
  StyleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.CopyToClipBoard
  Description: Copies this Series to the clipboard as tab-delimited text
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: moving data in and out of the application
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.CopyToClipBoard;
var
{Q: which is more efficient: a String or a TStringList ?}
  TheData: String;
  i: Integer;
begin
  TheData := FName;
  TheData := TheData + CRLF + FXAxis.Title.Caption + #9 + FYAxis.Title.Caption;
  TheData := TheData + CRLF + FXAxis.Title.Units + #9 + FYAxis.Title.Units;
  for i := 0 to FNoPts-1 do
  begin
    TheData := TheData + CRLF + FloatToStr(FXData^[i]) + #9 + FloatToStr(FYData^[i]);
  end;
  ClipBoard.AsText := TheData;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.CheckBounds
  Description: Checks if ThePointNo exceeds the Series Mins and Maxes
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: many: data and screen management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.CheckBounds(ThePointNo: Integer; AdjustAxis: Boolean);
begin
  if (FXMin > FXData^[ThePointNo]) then
  begin
    FXMin := FXData^[ThePointNo];
    if (AdjustAxis) then
      FXAxis.SetMinFromSeries(FXMin);
    {if (assigned(FOnXMinChange) and FVisible) then OnXMinChange(Self, FXMin);}
  end;
  if (FXMax < FXData^[ThePointNo]) then
  begin
    FXMax := FXData^[ThePointNo];
    if (AdjustAxis) then
      FXAxis.SetMaxFromSeries(FXMax);
    {if (assigned(FOnXMaxChange) and FVisible) then OnXMaxChange(Self, FXMax);}
  end;
  if (FYMin > FYData^[ThePointNo]) then
  begin
    FYMin := FYData^[ThePointNo];
    if (AdjustAxis) then
      FYAxis.SetMinFromSeries(FYMin);
    {if (assigned(FOnYMinChange) and FVisible) then OnYMinChange(Self, FYMin);}
  end;
  if (FYMax < FYData^[ThePointNo]) then
  begin
    FYMax := FYData^[ThePointNo];
    if (AdjustAxis) then
      FYAxis.SetMaxFromSeries(FYMax);
    {if (assigned(FOnYMaxChange) and FVisible) then OnYMaxChange(Self, YMax);}
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.GetBounds
  Description: Determines the Mins and Maxes of this Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the XMin, XMax, YMin and YMax  Properties
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.GetBounds;
var
  i: Integer;
begin
  for i := 0 to FNoPts-1 do
  begin
    if (FXMin > FXData^[i]) then FXMin := FXData^[i];
    if (FXMax < FXData^[i]) then FXMax := FXData^[i];
    if (FYMin > FYData^[i]) then FYMin := FYData^[i];
    if (FYMax < FYData^[i]) then FYMax := FYData^[i];
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.ResetBounds
  Description: Resets the Mins and Maxes
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.ResetBounds;
begin
  FXMin := 3.4e38;
  FXMax := -3.4e38;
  FYMin := 3.4e38;
  FYMax := -3.4e38;
end;

{------------------------------------------------------------------------------
     Function: TSeries.GetNearestPointToX
  Description: does what it says
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management
 Return Value: Index of nearest point
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.GetNearestPointToX(X: Single): Integer;
{This uses a binary search method to find the point with an X value closest to X.}
var
  iEst, iLow, iHigh: Integer;
begin
{Is X outside the range of X Values ?}
  if (X >= FXMax) then
  begin
    GetNearestPointToX := FNoPts-1;
    exit;
  end;
  if (X <= FXMin) then
  begin
    GetNearestPointToX := 0;
    exit;
  end;

{The lowest and highest possible points are:}
  iLow := 0;
  iHigh := FNoPts - 1;
{Estimate a starting point:}
  iEst := Round(FNoPts * (X-FXMin)/(FXMax - FXMin));

  while (((iEst-iLow) > 1) and ((iHigh-iEst) > 1)) do
  begin
    if (X < FXData^[iEst]) then
    begin
{The point is lower:}
      iHigh := iEst;
      iEst := (iEst + iLow) div 2;
    end
    else
    begin
{The point is higher:}
      iLow := iEst;
      iEst := (iEst + iHigh) div 2;
    end;
  end;
{find the X values just below and just above:}
  if ((X < FXData^[iLow]) or (X > FXData^[iHigh])) then
  begin
    raise EComponentError.CreateFmt('Failed attempt to find point closest to X = %g'
      + CRLF + 'X Low/Est/High = %g/%g/%g',
        [X, FXData^[iLow], FXData^[iEst], FXData^[iHigh]]);
  end
  else if (X < FXData^[iEst]) then
  begin
{FXData^[iLow] < X < FXData^[iEst]}
    if ((X-FXData^[iLow]) < (FXData^[iEst]-X)) then
    begin
      iEst := iLow;
    end;
  end
  else
  begin
{FXData^[iEst] < X < FXData^[iHigh]}
    if ((FXData^[iEst]-X) > (FXData^[iHigh]-X)) then
    begin
      iEst := iHigh;
    end;
  end;
  GetNearestPointToX := iEst;
end;

{------------------------------------------------------------------------------
     Function: TSeries.GetNearestPoint
  Description: does what it says
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management
 Return Value: Index of nearest point
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.GetNearestPoint(
  StartPt, EndPt, iX, iY: Integer;
  var NearestiX, NearestiY: Integer;
  var NearestX, NearestY, MinDistance: Single): Integer;
{The data may not be sorted, so we check every point:}
var
  Distance: Single;
  i, NearestN: Integer;
  Xi, Yi: Integer;
begin
{adjust for displacement:}
  Dec(iX, FDeltaX);
  Dec(iY, FDeltaY);
{et al:}
  MinDistance := 1.0e38;
  NearestN := 0;
{loop over points in each series:}
  for i := StartPt to EndPt do
  begin
    Xi := FXAxis.FofX(FXData^[i]);
    Yi := FYAxis.FofY(FYData^[i]);
    Distance := Sqr(Int(Xi-iX)) + Sqr(Int(Yi-iY));
    if (MinDistance > Distance) then
    begin
      MinDistance := Distance;
      NearestX := FXData^[i];
      NearestY := FYData^[i];
{adjust for displacement:}
      NearestiX := Xi + FDeltaX;
      NearestiY := Yi + FDeltaY;
      NearestN := i;
    end;
  end; {loop over points}
  MinDistance := Sqrt(MinDistance);
  GetNearestPoint := NearestN;
end;


{------------------------------------------------------------------------------
     Function: TSeries.GetNearestPointFast
  Description: does what it says - a quick and dirty method
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management
 Return Value: Index of nearest point
 Known Issues: will not work on very spiky data
 ------------------------------------------------------------------------------}
function TSeries.GetNearestPointFast(iX, iY: Integer;
  var NearestiX, NearestiY: Integer;
  var NearestX, NearestY, MinDistance: Single): Integer;
var
  X: Single;
  N: Integer;
  StartPt, EndPt: Integer;
begin
  X := FXAxis.XofF(iX);
  N := GetNearestPointToX(X);

  StartPt := N - FNoPts div 20;
  if (StartPt < 0) then StartPt := 0;
  EndPt := N + FNoPts div 20;
  if (EndPt > FNoPts) then EndPt := FNoPts;

  GetNearestPointFast := GetNearestPoint(
    StartPt, EndPt, iX, iY,
    NearestiX, NearestiY,
    NearestX, NearestY, MinDistance);
end;

{TSeries the movie ! ----------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TSeries.Draw
  Description: standard Drawing procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: draws the Series on a given canvas
 Known Issues: caution with all the PERFORMANCE stuff
 ------------------------------------------------------------------------------}
procedure TSeries.Draw(ACanvas: TCanvas);
var
  i: Integer;
  iX, iY: Integer;
{$IFDEF PERFORMANCE}
  FStartTime:          Int64; {TLargeInteger;}
  FFinishTime:         Int64;
  FFrequency:          Int64;
  ElapsedTime: Double;
{$ENDIF}
{$IFDEF POINTERS}
  pX, pY: pSingle;
{$ENDIF}
begin
{$IFDEF PERFORMANCE}
  QueryPerformanceFrequency(FFrequency); {counts per second}
{get the starting time:}
  QueryPerformanceCounter(FStartTime); { LARGE_INTEGER}
{$ENDIF}

{As you can see, we did some performance testing here. Guess what ?
 Delphi was FASTER than direct API calls !

 On a Dell PII-266 laptop, drawing a normal distribution takes:
 No Pts            Time (ms)
 Symbol:           None          Cross           Cross           Cross
 API                No             No             Yes              Yes
 Pointers           No             No              No              Yes
 100                  1.4           4.1            4.5              4.4
 1000                12            39             43               42
 10000              123           400            425              417
 100000            1162          4007           4269             4178

 As well, using pointers instead of dynamic arrays gave roughly a 2% speed
 increase - which just isn't worth it.
 }

{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeries.Draw: ACanvas is nil !');
{$ENDIF}

  if ((not FVisible) or
      (FNoPts = 0)) then exit;

  ACanvas.Pen.Assign(FPen);
  if (ACanvas.Pen.Width > 0) then
  begin
{$IFDEF POINTERS}
    pX := pSingle(FXData);
    pY := pSingle(FYData);
    iX := FXAxis.FofX(pX^)+ FDeltaX;
    iY := FYAxis.FofY(pY^) + FDeltaY;
{$ELSE}
    iX := FXAxis.FofX(FXData^[0])+ FDeltaX;
    iY := FYAxis.FofY(FYData^[0]) + FDeltaY;
{$ENDIF}

{$IFDEF DIRECT_API}
    MoveToEx(ACanvas.Handle, iX, iY, nil);
{$ELSE}
    ACanvas.MoveTo(iX, iY);
{$ENDIF}

    for i := 1 to FNoPts-1 do
    begin
{$IFDEF POINTERS}
      Inc(pX);
      Inc(pY);
      iX := FXAxis.FofX(pX^) + FDeltaX;
      iY := FYAxis.FofY(pY^) + FDeltaY;
{$ELSE}
      iX := FXAxis.FofX(FXData^[i]) + FDeltaX;
      iY := FYAxis.FofY(FYData^[i]) + FDeltaY;
{$ENDIF}

{$IFDEF DIRECT_API}
      LineTo(ACanvas.Handle, iX, iY);
{$ELSE}
      ACanvas.LineTo(iX, iY);
{$ENDIF}
    end; {loop over points}
  end;

  if ((FSymbol <> sNone) and (FSymbolSize > 0)) then
  begin
{$IFDEF POINTERS}
    pX := pSingle(FXData);
    pY := pSingle(FYData);
{$ENDIF}

    for i := 0 to FNoPts-1 do
    begin
{$IFDEF POINTERS}
      iX := FXAxis.FofX(pX^)+ FDeltaX;
      iY := FYAxis.FofY(pY^) + FDeltaY;
      Inc(pX);
      Inc(pY);
{$ELSE}
      iX := FXAxis.FofX(FXData^[i])+ FDeltaX;
      iY := FYAxis.FofY(FYData^[i]) + FDeltaY;
{$ENDIF}
      DrawSymbol(ACanvas, iX, iY);
    end; {loop over points}
  end;

  if (FHighCount > 0) then
    if (FHighLow <> []) then
      DrawHighs(ACanvas);

{$IFDEF PERFORMANCE}
{get the finishing time:}
  QueryPerformanceCounter(FFinishTime); { LARGE_INTEGER}
{take difference and convert to ms:}
  ElapsedTime := 1000 * (FFinishTime - FStartTime) / FFrequency;
  MessageBox(0, PChar(Format('Drawing series 1 takes %g ms', [ElapsedTime])), 'Performance', MB_OK)
{$ENDIF}
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.Trace
  Description: Draws the series in an erasable mode
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: rapidly changing screen display
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.Trace(ACanvas: TCanvas);
var
  i: Integer;
  iX, iY: Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeries.Trace: ACanvas is nil !');
{$ENDIF}

  if ((not FVisible) or
      (FNoPts = 0)) then exit;

  ACanvas.Pen.Assign(FPen);
  ACanvas.Pen.Mode := pmNotXOR;
  iX := FXAxis.FofX(FXData^[0])+ FDeltaX;
  iY := FYAxis.FofY(FYData^[0]) + FDeltaY;
  ACanvas.MoveTo(iX, iY);
  for i := 1 to FNoPts-1 do
  begin
    iX := FXAxis.FofX(FXData^[i]) + FDeltaX;
    iY := FYAxis.FofY(FYData^[i]) + FDeltaY;
    ACanvas.LineTo(iX, iY);
  end; {loop over points}

  if ((FSymbol <> sNone) and (FSymbolSize > 0)) then
  begin
    for i := 0 to FNoPts-1 do
    begin
      iX := FXAxis.FofX(FXData^[i])+ FDeltaX;
      iY := FYAxis.FofY(FYData^[i]) + FDeltaY;
      DrawSymbol(ACanvas, iX, iY);
    end; {loop over points}
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.DrawHighs
  Description: standard Drawing procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: draws the Highs of the Series on a given canvas
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.DrawHighs(ACanvas: TCanvas);
var
  i,
  iX,
  iY: Integer;
  TheValue: String;
  LogRec: TLogFont;
  OldFontHandle, NewFontHandle: hFont;
begin
  ACanvas.Font.Color := ACanvas.Pen.Color;
{Use Windows GDI functions to rotate the font:}
{$IFDEF WIN}
  WinProcs.GetObject(ACanvas.Font.Handle, SizeOf(LogRec), Addr(LogRec));
  LogRec.lfEscapement := 900; {Up}
  LogRec.lfOutPrecision := OUT_DEFAULT_PRECIS;
  NewFontHandle := WinProcs.CreateFontIndirect(LogRec);
{select the new font:}
  OldFontHandle := WinProcs.SelectObject(ACanvas.Handle, NewFontHandle);
{$ELSE}
{What on earth do we do under Linux / QT ?}
{$ENDIF}

{Loop over all Highs:}
  if (hlHigh in FHighLow) then
  begin
    for i := 0 to FHighCount-1 do
    begin
      iX := FXAxis.FofX(XData^[FHighs^[i]]);
      iY := FYAxis.FofY(YData^[FHighs^[i]]);
      ACanvas.MoveTo(iX, iY-2);
      ACanvas.LineTo(iX, iY + ACanvas.Font.Height);
      ACanvas.TextOut(iX + ACanvas.Font.Height div 2,
       iY + ACanvas.Font.Height,
       FXAxis.LabelToStrF(XData^[FHighs^[i]]));
    end;
  end;

{Loop over all Lows:}
  if (hlLow in FHighLow) then
  begin
    for i := 0 to FLowCount-1 do
    begin
      iX := FXAxis.FofX(XData^[FLows^[i]]);
      iY := FYAxis.FofY(YData^[FLows^[i]]);
      ACanvas.MoveTo(iX, iY+2);
      ACanvas.LineTo(iX, iY - ACanvas.Font.Height);
      TheValue := FXAxis.LabelToStrF(XData^[FLows^[i]]);
      ACanvas.TextOut(iX + ACanvas.Font.Height div 2,
       iY - ACanvas.Font.Height + ACanvas.TextWidth(TheValue),
       TheValue);
    end;
  end;

{$IFDEF WIN}
{go back to original font:}
  NewFontHandle := WinProcs.SelectObject(ACanvas.Handle, OldFontHandle);
{and delete the old one:}
  WinProcs.DeleteObject(NewFontHandle);
{$ELSE}
{What on earth do we do under Linux / QT ?}
{$ENDIF}
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.DrawHistory
  Description: standard Drawing procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: draws the Series on a given canvas IN HISTORY MODE
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.DrawHistory(ACanvas: TCanvas; HistoryX: Single);
var
  i: Integer;
  iX, iY: Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeries.DrawHistory: ACanvas is nil !');
{$ENDIF}

  if ((not FVisible) or
      (FNoPts = 0)) then exit;

  ACanvas.Pen.Assign(FPen);
{we set the pen mode so that a second call to DrawHistory
erases the curve on screen:}
  ACanvas.Pen.Mode := pmNotXOR;
  iX := FXAxis.FofX(0) + FDeltaX;
  iY := FYAxis.FofY(FYData^[FNoPts-1]) + FDeltaY;
  ACanvas.MoveTo(iX, iY);
  for i := FNoPts-2 downto 0 do
  begin
    iX := FXAxis.FofX(FXData^[i] - FXData^[FNoPts-1]) + FDeltaX;
{we leave this loop if this is the last point:}
    if (iX < FXAxis.Left) then break;
    iY := FYAxis.FofY(FYData^[i]) + FDeltaY;
    ACanvas.LineTo(iX, iY);
  end; {loop over points}
  if ((FSymbol <> sNone) and (FSymbolSize > 0)) then
  begin
    ACanvas.Brush.Assign(FBrush);
    for i := FNoPts-2 downto 0 do
    begin
      iX := FXAxis.FofX(FXData^[i] - FXData^[FNoPts-1])+ FDeltaX;
{we leave this loop if this is the last point:}
      if (iX < FXAxis.Left) then break;
      iY := FYAxis.FofY(FYData^[i]) + FDeltaY;
      DrawSymbol(ACanvas, iX, iY);
    end; {loop over points}
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.DrawSymbol
  Description: Draws the selected Symbol at each point
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: draws the Symbols of the Series on a given canvas
 Known Issues:
 ------------------------------------------------------------------------------}
{$IFDEF DIRECT_API}
Procedure TSeries.DrawSymbol(ACanvas: TCanvas; iX, iY: Integer);
begin
  case FSymbol of
    sDash:
      begin
        MoveToEx(ACanvas.Handle, iX - FSymbolSize, iY, nil);
        LineTo(ACanvas.Handle, iX + FSymbolSize+1, iY);
      end;
    sVertDash:
      begin
        MoveToEx(ACanvas.Handle, iX, iY - FSymbolSize, nil);
        LineTo(ACanvas.Handle, iX, iY + FSymbolSize+1);
      end;
    sPlus:
      begin
        MoveToEx(ACanvas.Handle, iX - FSymbolSize, iY, nil);
        LineTo(ACanvas.Handle, iX + FSymbolSize+1, iY);
        MoveToEx(ACanvas.Handle, iX, iY - FSymbolSize, nil);
        LineTo(ACanvas.Handle, iX, iY + FSymbolSize+1);
      end;
    sCross:
      begin
        MoveToEx(ACanvas.Handle, iX - FSymbolSize, iY - FSymbolSize, nil);
        LineTo(ACanvas.Handle, iX + FSymbolSize+1, iY + FSymbolSize+1);
        MoveToEx(ACanvas.Handle, iX + FSymbolSize, iY - FSymbolSize, nil);
        LineTo(ACanvas.Handle, iX - FSymbolSize-1, iY + FSymbolSize+1);
      end;
    sStar:
      begin
        MoveToEx(ACanvas.Handle, iX - FSymbolSize, iY, nil);
        LineTo(ACanvas.Handle, iX + FSymbolSize+1, iY);
        MoveToEx(ACanvas.Handle, iX, iY - FSymbolSize, nil);
        LineTo(ACanvas.Handle, iX, iY + FSymbolSize+1);
        MoveToEx(ACanvas.Handle, iX - FSymbolSize, iY - FSymbolSize, nil);
        LineTo(ACanvas.Handle, iX + FSymbolSize+1, iY + FSymbolSize+1);
        MoveToEx(ACanvas.Handle, iX + FSymbolSize, iY - FSymbolSize, nil);
        LineTo(ACanvas.Handle, iX - FSymbolSize-1, iY + FSymbolSize+1);
      end;
    sSquare:
      begin
        Rectangle(ACanvas.Handle, iX - FSymbolSize, iY - FSymbolSize,
          iX + FSymbolSize+1, iY + FSymbolSize+1);
      end;
    sCircle:
      begin
        Ellipse(ACanvas.Handle, iX - FSymbolSize, iY - FSymbolSize,
          iX + FSymbolSize+1, iY + FSymbolSize+1);
      end;
    sUpTriangle:
      begin
        MoveToEx(ACanvas.Handle, iX - FSymbolSize, iY + FSymbolSize+1, nil);
        LineTo(ACanvas.Handle, iX, iY - FSymbolSize);
        LineTo(ACanvas.Handle, iX + FSymbolSize, iY + FSymbolSize+1);
        LineTo(ACanvas.Handle, iX - FSymbolSize, iY + FSymbolSize+1);
      end;
    sDownTriangle:
      begin
        MoveToEx(ACanvas.Handle, iX - FSymbolSize, iY - FSymbolSize, nil);
        LineTo(ACanvas.Handle, iX, iY + FSymbolSize+1);
        LineTo(ACanvas.Handle, iX + FSymbolSize, iY - FSymbolSize);
        LineTo(ACanvas.Handle, iX - FSymbolSize, iY - FSymbolSize);
      end;
  end;
  MoveToEx(ACanvas.Handle, iX, iY, nil);
end;
{$ELSE}
Procedure TSeries.DrawSymbol(ACanvas: TCanvas; iX, iY: Integer);
begin
  case FSymbol of
    sDash:
      begin
        ACanvas.MoveTo(iX - FSymbolSize, iY);
        ACanvas.LineTo(iX + FSymbolSize+1, iY);
      end;
    sVertDash:
      begin
        ACanvas.MoveTo(iX, iY - FSymbolSize);
        ACanvas.LineTo(iX, iY + FSymbolSize+1);
      end;
    sPlus:
      begin
        ACanvas.MoveTo(iX - FSymbolSize, iY);
        ACanvas.LineTo(iX + FSymbolSize+1, iY);
        ACanvas.MoveTo(iX, iY - FSymbolSize);
        ACanvas.LineTo(iX, iY + FSymbolSize+1);
      end;
    sCross:
      begin
        ACanvas.MoveTo(iX - FSymbolSize, iY - FSymbolSize);
        ACanvas.LineTo(iX + FSymbolSize+1, iY + FSymbolSize+1);
        ACanvas.MoveTo(iX + FSymbolSize, iY - FSymbolSize);
        ACanvas.LineTo(iX - FSymbolSize-1, iY + FSymbolSize+1);
      end;
    sStar:
      begin
        ACanvas.MoveTo(iX - FSymbolSize, iY);
        ACanvas.LineTo(iX + FSymbolSize+1, iY);
        ACanvas.MoveTo(iX, iY - FSymbolSize);
        ACanvas.LineTo(iX, iY + FSymbolSize+1);
        ACanvas.MoveTo(iX - FSymbolSize, iY - FSymbolSize);
        ACanvas.LineTo(iX + FSymbolSize+1, iY + FSymbolSize+1);
        ACanvas.MoveTo(iX + FSymbolSize, iY - FSymbolSize);
        ACanvas.LineTo(iX - FSymbolSize-1, iY + FSymbolSize+1);
      end;
    sSquare:
      begin
        ACanvas.Rectangle(iX - FSymbolSize, iY - FSymbolSize,
          iX + FSymbolSize+1, iY + FSymbolSize+1)
      end;
    sCircle:
      begin
        ACanvas.Ellipse(iX - FSymbolSize, iY - FSymbolSize,
          iX + FSymbolSize+1, iY + FSymbolSize+1)
      end;
    sUpTriangle:
      begin
        ACanvas.Polygon([
          Point(iX - FSymbolSize, iY + FSymbolSize+1),
          Point(iX, iY - FSymbolSize),
          Point(iX + FSymbolSize, iY + FSymbolSize+1)]);
        {ACanvas.MoveTo(iX - FSymbolSize, iY + FSymbolSize+1);
        ACanvas.LineTo(iX, iY - FSymbolSize);
        ACanvas.LineTo(iX + FSymbolSize, iY + FSymbolSize+1);
        ACanvas.LineTo(iX - FSymbolSize, iY + FSymbolSize+1);}
      end;
    sDownTriangle:
      begin
        ACanvas.Polygon([
          Point(iX - FSymbolSize, iY - FSymbolSize),
          Point(iX, iY + FSymbolSize+1),
          Point(iX + FSymbolSize, iY - FSymbolSize)]);
        {ACanvas.MoveTo(iX - FSymbolSize, iY - FSymbolSize);
        ACanvas.LineTo(iX, iY + FSymbolSize+1);
        ACanvas.LineTo(iX + FSymbolSize, iY - FSymbolSize);
        ACanvas.LineTo(iX - FSymbolSize, iY - FSymbolSize);}
      end;
  end;
  ACanvas.MoveTo(iX, iY);
end;
{$ENDIF}

{Moving TSeries on the screen -------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TSeries.GenerateOutline
  Description: calculates an outline of the Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Screen display - highlighting a Series
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.GenerateOutline(OutlineWidth: Integer);
var
  i: Integer;
  StepSize: Integer;
begin
  FOutlineWidth := OutlineWidth;

  if (FNoPts > OUTLINE_DENSITY) then
  begin
{initialize:}
    NoOutlinePts := OUTLINE_DENSITY+1; { = 21}
    StepSize := FNoPts div OUTLINE_DENSITY;
{loop over data points:}
    for i := 0 to NoOutlinePts-2 do    {0..19}
    begin
      Outline[i].x := FXAxis.FofX(FXData^[i*StepSize]);
      Outline[i].y := FYAxis.FofY(FYData^[i*StepSize]);
    end;
{do the end point:}
    Outline[OUTLINE_DENSITY].x := FXAxis.FofX(FXData^[FNoPts-1]);
    Outline[OUTLINE_DENSITY].y := FYAxis.FofY(FYData^[FNoPts-1]);
  end
  else
  begin
{not many points:}
    NoOutlinePts := FNoPts;
    for i := 0 to NoOutlinePts-1 do
    begin
      Outline[i].x := FXAxis.FofX(FXData^[i]);
      Outline[i].y := FYAxis.FofY(FYData^[i]);
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.OutlineSeries
  Description: draws an outline of the Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Screen display - highlighting a Series
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.OutlineSeries(ACanvas: TCanvas);
var
  i: Integer;
  {pOutlinePt: pPoint;}
begin
  if (NoOutlinePts = 0) then exit;

  ACanvas.Pen.Color := clLime;
  ACanvas.Pen.Width := FOutlineWidth;
  ACanvas.Pen.Mode := pmNotXOR;
  {ACanvas.Pen.Style := psDash;
  pOutlinePt := Outline;}

  ACanvas.MoveTo(Outline[0].x + FDeltaX, Outline[0].y + FDeltaY);
  for i := 0 to NoOutlinePts-1 do
  begin
    ACanvas.LineTo(Outline[i].x + FDeltaX, Outline[i].y + FDeltaY);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.MoveSeriesBy
  Description: does what it says
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: moves the clicked object outline by (DX, DY) from its current point.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.MoveSeriesBy(ACanvas: TCanvas; DX, DY: Integer);
begin
{erase the old outline:}
  OutlineSeries(ACanvas);
{save the new displacements:}
  Inc(FDeltaX, DX);
  Inc(FDeltaY, DY);

{create the new outline:}
  OutlineSeries(ACanvas);
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.MoveSeriesTo
  Description: does what it says
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: moves the clicked object outline TO (X, Y).
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeries.MoveSeriesTo(ACanvas: TCanvas; X, Y: Integer);
begin
{erase the old outline:}
  OutlineSeries(ACanvas);

{save the new displacements:}
  FDeltaX := X - FXAxis.FofX(FXData^[0]);
  FDeltaY := Y - FYAxis.FofY(FYData^[0]);

{create the new outline:}
  OutlineSeries(ACanvas);
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.LineBestFit
  Description: Does what it says
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: calculates the line of best fit from Start to Finish
 Known Issues: 
 ------------------------------------------------------------------------------}
procedure TSeries.LineBestFit(TheLeft, TheRight: Single;
  var NoLSPts: Integer;
  var SumX, SumY, SumXsq, SumXY, SumYsq: Double;
  var Slope, Intercept, Rsq: Single);
var
  i: Integer;
  Start, Finish: Integer;
  LnX, LnY: Double;
begin
{Determine the starting and ending points:}
  Start := GetNearestPointToX(TheLeft);
  Finish := GetNearestPointToX(TheRight);

  if ((not FXAxis.LogScale) and (not FYAxis.LogScale)) then
  begin
{normal linear fit:}
    for i := Start to Finish do
    begin
      Inc(NoLSPts);
      SumX := SumX + FXData^[i];
      SumY := SumY + FYData^[i];
      SumXsq := SumXsq + Sqr(FXData^[i]);
      SumXY := SumXY + FXData^[i] * FYData^[i];
      SumYsq := SumYsq + Sqr(FYData^[i]);
    end;
  end
  else if ((FXAxis.LogScale) and (not FYAxis.LogScale)) then
  begin
{logarithmic X Axis:}
    for i := Start to Finish do
    begin
      Inc(NoLSPts);
      LnX := Ln(FXData^[i]);
      SumX := SumX + LnX;
      SumY := SumY + FYData^[i];
      SumXsq := SumXsq + Sqr(LnX);
      SumXY := SumXY + LnX * FYData^[i];
      SumYsq := SumYsq + Sqr(FYData^[i]);
    end;
  end
  else if ((not FXAxis.LogScale) and (FYAxis.LogScale)) then
  begin
{logarithmic Y Axis:}
    for i := Start to Finish do
    begin
      Inc(NoLSPts);
      LnY := Ln(FYData^[i]);
      SumX := SumX + FXData^[i];
      SumY := SumY + LnY;
      SumXsq := SumXsq + Sqr(FXData^[i]);
      SumXY := SumXY + FXData^[i] * LnY;
      SumYsq := SumYsq + Sqr(LnY);
    end;
  end
  else if ((FXAxis.LogScale) and (FYAxis.LogScale)) then
  begin
{double logarithmic fit:}
    for i := Start to Finish do
    begin
      Inc(NoLSPts);
      LnX := Ln(FXData^[i]);
      LnY := Ln(FYData^[i]);
      SumX := SumX + LnX;
      SumY := SumY + LnY;
      SumXsq := SumXsq + Sqr(LnX);
      SumXY := SumXY + LnX * LnY;
      SumYsq := SumYsq + Sqr(LnY);
    end;
  end;

{so the slope and intercept are:}
  try
    Slope := (NoLSPts * SumXY - SumX * SumY) /
             (NoLSPts * SumXsq - Sqr(SumX));
    Intercept := (SumY / NoLSPts) - Slope * (SumX / NoLSPts);
    RSQ := Sqr(NoLSPts * SumXY - SumX * SumY) /
           ((NoLSPts * SumXsq - Sqr(SumX)) * (NoLSPts * SumYsq - Sqr(SumY)));
  except
    EMathError.CreateFmt('NoLSPts = %d' + CRLF +
      'SumX = %g' + CRLF +
      'SumY = %g' + CRLF +
      'SumXsq = %g' + CRLF +
      'SumXY = %g' + CRLF +
      'SumYsq = %g.',
      [NoLSPts, SumX, SumY, SumXsq, SumXY, SumYsq]);
  end;
end;

{Sub BestFit (iStart%, iFinish%, X_Data() As Single, Y_Data() As Single, Clear_Regs%, Slope!, Intercept!, RSQ!)
Dim i%
Dim Msg$
Static SumX!
Static SumY!
Static SumXsq!
Static SumXY!
Static SumYsq!
Static No_Pts%

    On Error GoTo BestFit_ErrorHandler

'   we initialise the sums for a least-squares fit:
    If (Clear_Regs% = True) Then
        No_Pts% = 0
        SumX! = 0
        SumY! = 0
        SumXsq! = 0
        SumXY! = 0
        SumYsq! = 0
    End If

    Select Case LogCase()
    Case 0      'neither axis is logged:
'   Do the summation:
        For i% = iStart% To iFinish%
            No_Pts% = No_Pts% + 1
            SumX! = SumX! + X_Data(i%)
            SumY! = SumY! + Y_Data(i%)
            SumXsq! = SumXsq! + X_Data(i%) ^ 2
            SumXY! = SumXY! + X_Data(i%) * Y_Data(i%)
            SumYsq! = SumYsq! + Y_Data(i%) ^ 2
        Next i%
    Case 1      'only the X-axis is logged:
        For i% = iStart% To iFinish%
            No_Pts% = No_Pts% + 1
            SumX! = SumX! + Log(X_Data(i%))
            SumY! = SumY! + Y_Data(i%)
            SumXsq! = SumXsq! + Log(X_Data(i%)) ^ 2
            SumXY! = SumXY! + Log(X_Data(i%)) * Y_Data(i%)
            SumYsq! = SumYsq! + Y_Data(i%) ^ 2
        Next i%
    Case 2      'only the Y-axis is logged:
        For i% = iStart% To iFinish%
            No_Pts% = No_Pts% + 1
            SumX! = SumX! + X_Data(i%)
            SumY! = SumY! + Log(Y_Data(i%))
            SumXsq! = SumXsq! + X_Data(i%) ^ 2
            SumXY! = SumXY! + X_Data(i%) * Log(Y_Data(i%))
            SumYsq! = SumYsq! + Log(Y_Data(i%)) ^ 2
        Next i%
    Case 3      'both axes are logged:
        For i% = iStart% To iFinish%
            No_Pts% = No_Pts% + 1
            SumX! = SumX! + Log(X_Data(i%))
            SumY! = SumY! + Log(Y_Data(i%))
            SumXsq! = SumXsq! + Log(X_Data(i%)) ^ 2
            SumXY! = SumXY! + Log(X_Data(i%)) * Log(Y_Data(i%))
            SumYsq! = SumYsq! + Log(Y_Data(i%)) ^ 2
        Next i%
    Case 4      'X axis is Log10'ed
        For i% = iStart% To iFinish%
            No_Pts% = No_Pts% + 1
            SumX! = SumX! + LOG10_E * Log(X_Data(i%))
            SumY! = SumY! + Y_Data(i%)
            SumXsq! = SumXsq! + (LOG10_E * Log(X_Data(i%))) ^ 2
            SumXY! = SumXY! + LOG10_E * Log(X_Data(i%)) * Y_Data(i%)
            SumYsq! = SumYsq! + Y_Data(i%) ^ 2
        Next i%
    Case 6      'X axis is Log10'ed, Y axis is ln'ed:
        For i% = iStart% To iFinish%
            No_Pts% = No_Pts% + 1
            SumX! = SumX! + LOG10_E * Log(X_Data(i%))
            SumY! = SumY! + Log(Y_Data(i%))
            SumXsq! = SumXsq! + (LOG10_E * Log(X_Data(i%))) ^ 2
            SumXY! = SumXY! + LOG10_E * Log(X_Data(i%)) * Log(Y_Data(i%))
            SumYsq! = SumYsq! + Log(Y_Data(i%)) ^ 2
        Next i%
    Case 8      'Y axis is Log10'ed:
        For i% = iStart% To iFinish%
            No_Pts% = No_Pts% + 1
            SumX! = SumX! + X_Data(i%)
            SumY! = SumY! + LOG10_E * Log(Y_Data(i%))
            SumXsq! = SumXsq! + X_Data(i%) ^ 2
            SumXY! = SumXY! + X_Data(i%) * LOG10_E * Log(Y_Data(i%))
            SumYsq! = SumYsq! + (LOG10_E * Log(Y_Data(i%))) ^ 2
        Next i%
    Case 9      'X axis is ln'ed, Y axis is Log10'ed:
        For i% = iStart% To iFinish%
            No_Pts% = No_Pts% + 1
            SumX! = SumX! + Log(X_Data(i%))
            SumY! = SumY! + LOG10_E * Log(Y_Data(i%))
            SumXsq! = SumXsq! + Log(X_Data(i%)) ^ 2
            SumXY! = SumXY! + Log(X_Data(i%)) * LOG10_E * Log(Y_Data(i%))
            SumYsq! = SumYsq! + (LOG10_E * Log(Y_Data(i%))) ^ 2
        Next i%
    Case 12      'both axes are Log10'ed:
        For i% = iStart% To iFinish%
            No_Pts% = No_Pts% + 1
            SumX! = SumX! + LOG10_E * Log(X_Data(i%))
            SumY! = SumY! + LOG10_E * Log(Y_Data(i%))
            SumXsq! = SumXsq! + (LOG10_E * Log(X_Data(i%))) ^ 2
            SumXY! = SumXY! + LOG10_E * Log(X_Data(i%)) * LOG10_E * Log(Y_Data(i%))
            SumYsq! = SumYsq! + (LOG10_E * Log(Y_Data(i%))) ^ 2
        Next i%
    End Select

'   so the slope and intercept are:
    Slope! = (No_Pts% * SumXY! - SumX! * SumY!) / (No_Pts% * SumXsq! - (SumX! ^ 2))
    Intercept! = (SumY! / No_Pts%) - Slope! * (SumX! / No_Pts%)
    RSQ! = (No_Pts% * SumXY! - SumX! * SumY!) ^ 2 / ((No_Pts% * SumXsq! - (SumX! ^ 2)) * (No_Pts% * SumYsq! - (SumY! ^ 2)))

BestFit_FINISHED:


Exit Sub

BestFit_ErrorHandler:   ' Error handler line label.

    Select Case Err
    Case 5
        Resume Next
    Case Else
        Msg$ = "Panic in " & "BestFit_ErrorHandler !"
        Msg$ = Msg$ & LF & LF & "Error No. " & Str$(Err) & ": " & Error$
        Response% = Message(Msg$, MB_OK + MB_ICONEXCLAMATION, "Error !", NO, H_PANIC)
    End Select

    Resume BestFit_FINISHED


End Sub
}

{------------------------------------------------------------------------------
     Function: TSeries.FindHighsLows
  Description: This function finds all the Highs (and troughs) in a region
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the value of the ??? Property
 Return Value: the number of Highs
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeries.FindHighsLows(Start, Finish, HeightSensitivity: Integer): Integer;
var
  i,
  LastHigh,
  LastLow: Integer;
  Highseek: Boolean;
  Delta: Single;
begin
{this routine finds all the major Highs in a region;
 See "FindAHigh" for a single High finding routine.}

{set the sensitivity:}
  Delta := (HeightSensitivity / 100.0) * (FYMax - FYMin);
  ClearHighsLows;

{initialise variables:}
  LastHigh := Start;
  LastLow := Start;
  Highseek := TRUE;

{allocate memory for results:}
  GetMem(FHighs, FHighCapacity * SizeOf(Integer));
  GetMem(FLows, FHighCapacity * SizeOf(Integer));
{we set the first point to a low}
  Lows^[FLowCount] := LastLow;
  Inc(FLowCount);

  for i := Start to Finish do
  begin
    if (Highseek = TRUE) then
    begin
      if (YData^[i] > YData^[LastHigh]) then
        LastHigh := i;
      if (YData^[i] < (YData^[LastHigh] - Delta)) then
      begin
{The Last High was a real maximum:}
        Highs^[FHighCount] := LastHigh;
        Inc(FHighCount);
        if (FHighCount >= FHighCapacity-2) then
        begin
{add 10 more points:}
{$IFDEF DELPHI1}
          ReAllocMem(FHighs, FHighCapacity * SizeOf(Integer),
            (FHighCapacity+10) * SizeOf(Integer));
          ReAllocMem(FLows, FHighCapacity * SizeOf(Integer),
            (FHighCapacity+10) * SizeOf(Integer));
          Inc(FHighCapacity, 10);
{$ELSE}
          Inc(FHighCapacity, 10);
          ReAllocMem(FHighs, FHighCapacity * SizeOf(Integer));
          ReAllocMem(FLows, FHighCapacity * SizeOf(Integer));
{$ENDIF}
        end;
        Highseek := FALSE;
        LastLow := i;
      end;
    end
    else
    begin
      if (YData^[i] < YData^[LastLow]) then
        LastLow := i;
      if (YData^[i] > (YData^[LastLow] + Delta)) then
      begin
{The Last Low was a real minimum:}
        Lows^[FLowCount] := LastLow;
        Inc(FLowCount);
        Highseek := TRUE;
        LastHigh := i;
      end; {comparison}
    end; {seeking high or low}
  end; {for}
  Lows^[FLowCount] := LastLow;

  FindHighsLows := FHighCount;
end;


end.
