unit Titles;

{$I Plot.inc}

{-----------------------------------------------------------------------------
The contents of this file are subject to the Mozilla Public License
Version 1.1 (the "License"); you may not use this file except in compliance
with the License. You may obtain a copy of the License at
http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either expressed or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is: Titles.pas, released 12 September 2000.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@molsci.csiro.au.

Last Modified: 02/25/2000
Current Version: 1.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind. 

Purpose:
This unit contains many sub-components:
     TRectangle
     TBorder
     TCaption
     TLegend
     TTitle
that manage various screen areas and objects for TPlot.

Known Issues:
      - This would normally be called Series, but TeeChart already uses that unit name.
-----------------------------------------------------------------------------}

interface

uses
  classes, graphics, SysUtils,
{These are the D1 units, aliased to Windows:}
{$IFDEF WIN}
  WinTypes, WinProcs,
{$ENDIF}

{$IFDEF DELPHI1} {Delphi 1 doesn't have a trim function}
  TrimStr,
{$ENDIF}

  Misc;

const
  STRING_DIV_SYMBOL = 3;
{The ratio of the width of the String portion of a Legend to the Symbol portion.}

type
  TDirection = (dHorizontal, dVertical);
{Screen objects are often Horizontal or Vertical.}

  TOrientation = (orRight, orLeft);
{Other screen objects can be Left or Right aligned.}

{TRectangle *******************************************************************}
  TRectangle = class(TPersistent)
{TRectangle is extremely important to the functioning of TSciGraph.}
{}
{Not only is it used for a variety of objects, it is also the base class
 of many on-screen objects: Axes, Titles, Captions, Legends, etc.}
{}
{As well as having the properties that one would expect for a rectangle,
 it also has some properties that are useful to its various descendants:
 Alignment, Name and Visibility to be precise.}
  private
    FAlignment: TAlignment;
    FFireEvents: Boolean;
    FLeft: Integer;
    FTop: Integer;
    FRight: Integer;
    FBottom: Integer;
    FName: String;
    FOwner: TPersistent;
    FTag: Longint;
    FVisible: Boolean;

{Get functions for virtual properties:}
    function GetHeight: Integer;
    function GetWidth: Integer;

{Set procedures:}
    procedure SetAlignment(Value: TAlignment);
    procedure SetRight(Value: Integer);
    procedure SetBottom(Value: Integer);
    procedure SetVisible(Value: Boolean);

{Set procedures for virtual properties:}
    procedure SetHeight(Value: Integer);
    procedure SetWidth(Value: Integer);
    procedure SetDeltaX(Value: Integer);
    procedure SetDeltaY(Value: Integer);

  protected
    FOnChange: TNotifyEvent;
{Events are normally private/published, but D1 does not allow descendants to
 check the assignment of events: FOnChange is inaccessible, and OnChange
 cannot be examined because it is a function.}

    function GetMidX: Integer;
    function GetMidY: Integer;

{Set procedures that we may wish to override later:}
    procedure SetLeft(Value: Integer); virtual;
{This sets the Left screen position property. It also moves the Right by the
 same amount, thereby preserving the Width. Ie: it moves the whole TRectangle.}
    procedure SetMidX(Value: Integer); virtual;
{This sets the MidX screen virtual position property. It thereby moves the Left and Right.}
    procedure SetMidY(Value: Integer); virtual;
{This sets the MidY screen virtual position property. It thereby moves the Top and Bottom.}
    procedure SetTop(Value: Integer); virtual;
{This sets the Top screen position property. It also moves the Bottom by the
 same amount, thereby preserving the Height. Ie: it moves the whole TRectangle.}
    procedure DoHandleChange; virtual;
{All Set methods call this to handle the OnChange logic.}
  public
{virtual write-only properties:}
    property DeltaX: Integer write SetDeltaX;
{This changes the Left and Right by DeltaX, thereby moving the TRectangle.
 It is similar to getting then setting the Left property, but quicker.}
    property DeltaY: Integer write SetDeltaY;
{This changes the Top and Bottom by DeltaY, thereby moving the TRectangle.
 It is similar to getting then setting the Top property, but quicker.}
    property Owner: TPersistent read FOwner;
{This is similar to TComponent.Owner: TComponent, except that:}
{    1. it is a TPersistent;}
{    2. this component is NOT freed automatically when the Owner is freed.}
    property Tag: Longint read FTag write FTag;
{The usual Tag property, as in TComponent.Tag}
{}
{However, DO NOT USE THIS PROPERTY !
 It is used by TPlot to store the object type, and hence control the visible
 behaviour of the TRectangle descendant.}

    Constructor Create(AOwner: TPersistent); virtual;
{The standard constructor, where standard properties are set.}
    Destructor Destroy; override;
{The standard destructor, where the OnChange event is "freed".}

    {procedure Assign(Source: TPersistent); override;}
    procedure AssignTo(Dest: TPersistent); override;
{TRectangle's implementation of the standard Assign(To) method.}
    function ClickedOn(iX, iY: Integer): Boolean;

  published
    property Alignment: TAlignment read FAlignment write SetAlignment;
{Can a rectangle have alignment ? Not really, but its descendants can !}
    property FireEvents: Boolean read FFireEvents write FFireEvents;
{Do we fire events in response to a geometry change ?
 For the TCaption descendants, the answer is no, because they dance around
 the screen with every repaint.}
    property Name: String read FName write FName;
{This is what is displayed when the user is offered a choice.
 Eg: "Move the X-Axis Caption or the Bottom Border ?".}
    property Left: Integer read FLeft write SetLeft;
{This is the usual position property.}
    property Top: Integer read FTop write SetTop;
{This is the usual position property.}
    property Right: Integer read FRight write SetRight;
{This is the usual position property.}
    property Bottom: Integer read FBottom write SetBottom;
{This is the usual position property.}
    Property Visible: Boolean read FVisible write SetVisible;
{Is the Rectangle (or its descendants) visible ?}

{virtual properties:}
    property Height: Integer read GetHeight write SetHeight;
{The standard Height property.}
    property Width: Integer read GetWidth write SetWidth;
{The standard Width property.}
    property MidX: Integer read GetMidX write SetMidX;
{The X midpoint of the TRectangle.}
    property MidY: Integer read GetMidY write SetMidY;
{The Y midpoint of the TRectangle.}

    Property OnChange: TNotifyEvent read FOnChange write FOnChange;
{The standard OnChange event (for geometry).}
  end;

{TBorder **********************************************************************}
  TBorder = class(TRectangle)
{TBorder adds a third point to a TRectangle, hence creating a border.
 The following diagram explains this:}
{}
{  Top, Left
   ----------------------------------------====================
   |                                       |                  +        |
   |                                       |                  +        |
   |                                       |                  +        |
   |                                       |                  +        |
   |                   .MidX,MidY          |<----RightGap---->+        |
   |                                       |                  +        |
   |                                       |                  +        |
   |                                       |Right,Bottom      +    HeightEx
   ----------------------------------------********************        |
   +                   |                   *                  #        |
   +                   |                   *                  #        |
   +              BottomGap                *                  #        |
   +                   |                   *                  #        |
   +                   |                   *                  #        |
   +=======================================*###################
                                                               RightEx,BottomEx
   <---------------------------WidthEx------------------------>
   }
  private
    FRightEx: Integer;
    FBottomEx: Integer;

    procedure SetRightEx(Value: Integer);
    procedure SetBottomEx(Value: Integer);
{Get functions for the new virtual properties:}
    function GetRightGap: Integer;
    function GetBottomGap: Integer;
    function GetHeightEx: Integer;
    function GetWidthEx: Integer;
{Set procedures for the new virtual properties:}
    procedure SetRightGap(Value: Integer);
    procedure SetBottomGap(Value: Integer);
    procedure SetHeightEx(Value: Integer);
    procedure SetWidthEx(Value: Integer);

  protected
    procedure SetLeft(Value: Integer); override;
{Setting the Left also moves the Right and RightEx, thereby preserving the
 Width and RightGap. Ie: it moves the whole TBorder.}
    procedure SetTop(Value: Integer); override;
{Setting the Top also moves the Bottom and BottomEx, thereby preserving the
 Height and BottomGap. Ie: it moves the whole TBorder.}

  public
    Constructor Create(AOwner: TPersistent); override;
{The standard constructor, where standard properties are set.}

  published
    property RightEx: Integer read FRightEx write SetRightEx;
{The extreme right, to the right of Right.}
    property BottomEx: Integer read FBottomEx write SetBottomEx;
{The extreme Bottom, below Bottom.}

{the "virtual" properties:}
    property RightGap: Integer read GetRightGap write SetRightGap;
{The gap (or width) between the Right and RightEx.}
    property BottomGap: Integer read GetBottomGap write SetBottomGap;
{The gap (or height) between the Bottom and BottomEx.}
    property HeightEx: Integer read GetHeightEx write SetHeightEx;
{The total height between the Top and BottomEx.}
    property WidthEx: Integer read GetWidthEx write SetWidthEx;
{The total width between the Left and RightEx.}
  end;

{TCaption *********************************************************************}
  TCaption = class(TRectangle)
{A TCaption inherits the positional behaviour of TRectangle,
 and adds a Caption and a Font.}  
  private
    FCaption: String;
    FFont: TFont;

    FOnCaptionChange: TNotifyEvent;

    procedure SetFont(Value: TFont);

  protected
    procedure CreateName; virtual;
{This sets the name after the Caption is set.}    
    procedure SetCaption(Value: String); virtual;
{This sets the Caption and calls CreateName.}
  public
    constructor Create(AOwner: TPersistent); override;
    destructor Destroy; override;

{The standard constructor, where standard properties are set.}
    {procedure Assign(Source: TPersistent); override;}
    procedure AssignTo(Dest: TPersistent); override;

  published
    property Caption: String read FCaption write SetCaption;
{This is the text that is displayed on the screen.
 Eg: "X-Axis". Setting the Caption also sets the Name:}
{}
{ FName := FCaption + ' Caption';}
    property Font: TFont read FFont write SetFont;
{The font used to display the caption.}

    Property OnCaptionChange: TNotifyEvent read FOnCaptionChange write FOnCaptionChange;
{Has the Caption changed ?}
  end;

{TTitle ***********************************************************************}
  TTitle = class(TCaption)
{This is an extended TCaption that dances around the screen depending on
 Alignment, Orientation and Direction, and Draws itself.}
  private
    FDirection : TDirection;
    FOrientation: TOrientation;
    FEnvelope: TRect;
    FUnits: String;

    FFullCaption: String;

    procedure SetDirection(Value: TDirection);
    procedure SetOrientation(Value: TOrientation);
    procedure SetEnvelope(Value: TRect);
    procedure SetUnits(Value: String);

  protected
    procedure DoGeometry(ACanvas: TCanvas; TheText: String); dynamic;
{This determines where the TTitle is exactly on the screen, depending on the
 Envelope, Direction, Orientation and Alignment.}

    procedure SetCaption(Value: String); override;
{This handles the tricky question of Caption and Units. If Value contains a
 pair of brackets '()', then the contents of these brackets is taken to be the
 Units. If it doesn't, then the FullCaption is constructed from Value and Units.}

  public
    property Envelope: TRect read FEnvelope write SetEnvelope;
{This is the region just outside which the Caption can appear.
 Its exact position will depend on the Alignment, Direction and Orientation}
    property FullCaption: String read FFullCaption;
{This is a read-only property formed from the Caption and the Units.}

    constructor Create(AOwner: TPersistent); override;
{The standard constructor, where standard properties are set.}

    procedure Draw(ACanvas: TCanvas);
    {procedure Assign(Source: TPersistent); override;}
    procedure AssignTo(Dest: TPersistent); override;
{This Draws the TTitle on the given Canvas. It calls DoGeometry, and also
 various API functions to create a vertical font if neccessary.}
  published
    property Units: String read FUnits write SetUnits;
{These are the physical units, eg: mm, mV, seconds, etc, of the Axis.}
    property Direction : TDirection read FDirection write SetDirection;
{Is the Caption drawn Horizontal or Vertical ?}
    property Orientation: TOrientation read FOrientation write SetOrientation;
{Is the caption to the Left or Right of the Axis ?}
  end;

{TLegend **********************************************************************}
  TLegend = class(TCaption)
{This is an extended TCaption that dances around the screen depending on
 Alignment, Orientation and Direction, and Draws itself.}
  private
    FCount: Integer;
    FDirection : TDirection;
    FFontHeight: Integer;
    FStringWidth: Integer;

    function GetSymbolWidth: Integer;
    procedure SetDirection(Value: TDirection);
    procedure SetCount(Value: Integer);
    procedure SetFontHeight(Value: Integer);
    procedure SetStringWidth(Value: Integer);
    procedure DoSize;

  protected

  public
    property Count: Integer read FCount write SetCount;
{How many lines/series names are in the Legend ?}
    property StringWidth: Integer read FStringWidth write SetStringWidth;
{The width of the text portion of the Legend.}
    property SymbolWidth: Integer read GetSymbolWidth;
{The width of the Symbol + Line portion of the Legend.}
{It is 1/3rd of the StringWidth.}

    function GetHit(iX, iY: Integer; var TheRect: TRect): Integer;
{The rectangle of the series name under the point iX, iY.}
  published
    property Direction : TDirection read FDirection write SetDirection;
{Is the Legend drawn Horizontal or Vertical ?}
    property FontHeight: Integer read FFontHeight write SetFontHeight;
  end;

implementation

{We place these two in here to avoid "Circular Unit References".
 They are needed for TLegend.
uses
  SciSer, SerList;}

{Constructor and Destructor:-------------------------------------------------}
{------------------------------------------------------------------------------
  Constructor: TRectangle.Create
  Description: Standard Constructor for TRectangle
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: initializes component and properties
 Known Issues:
 ------------------------------------------------------------------------------}
Constructor TRectangle.Create(AOwner: TPersistent);
begin
{First call the ancestor:
  inherited Create; - TObject.Create does nothing}

  FOwner := AOwner;

{we insert the default values that cannot be "defaulted":}
  FAlignment := taCenter;
  FLeft := 10;
  FTop := 10;
  SetRight(100);
  SetBottom(100);
  FVisible := TRUE;
{global change event handler:}
  FOnChange := nil;
{we do fire events with a geometry change:}
  FireEvents := TRUE;
end;

{------------------------------------------------------------------------------
   Destructor: TRectangle.Destroy
  Description: standard destructor
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: frees the OnChange event
 Known Issues:
 ------------------------------------------------------------------------------}
Destructor TRectangle.Destroy;
begin
  FOnChange := nil;
{then call ancestor:}
  inherited Destroy;
end;

{------------------------------------------------------------------------------
    Procedure: TRectangle.Assign
  Description: standard Assign method
       Author: Mat Ballard
 Date created: 07/06/2000
Date modified: 07/06/2000 by Mat Ballard
      Purpose: implements Assign
 Known Issues:
 ------------------------------------------------------------------------------}
{procedure TRectangle.Assign(Source: TPersistent);
begin
  inherited Assign(Source);
  FLeft := TRectangle(Source).Left;
  FTop := TRectangle(Source).Top;
  FRight := TRectangle(Source).Right;
  FBottom := TRectangle(Source).Bottom;
end;}

{------------------------------------------------------------------------------
    Procedure: TRectangle.AssignTo
  Description: standard AssignTo method
       Author: Mat Ballard
 Date created: 07/06/2000
Date modified: 07/06/2000 by Mat Ballard
      Purpose: implements AssignTo
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.AssignTo(Dest: TPersistent);
begin
{we DON'T call the ancestor, because TPersistent.AssignTo simply throws an
 exception:
  inherited AssignTo(Dest);}
  TRectangle(Dest).Left := FLeft;
  TRectangle(Dest).Top := FTop;
  TRectangle(Dest).Right := FRight;
  TRectangle(Dest).Bottom := FBottom;
end;

{Begin Set and Get Functions and Procedures----------------------------------}
{Get Functions for virtual properties ---------------------------------------}
{------------------------------------------------------------------------------
     Function: TRectangle.GetHeight
  Description: private property Get function
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Gets the Height, which is a virtual property
 Known Issues:
 ------------------------------------------------------------------------------}
function TRectangle.GetHeight: Integer;
begin
  GetHeight := FBottom -  FTop;
end;

{------------------------------------------------------------------------------
     Function: TRectangle.GetWidth
  Description: private property Get function
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Gets the Width, which is a virtual property
 Known Issues:
 ------------------------------------------------------------------------------}
function TRectangle.GetWidth: Integer;
begin
  GetWidth := FRight - FLeft;
end;

{------------------------------------------------------------------------------
     Function: TRectangle.GetMidX
  Description: private property Get function
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Gets the MidX, which is a virtual property
 Known Issues:
 ------------------------------------------------------------------------------}
function TRectangle.GetMidX: Integer;
begin
  GetMidX := (FLeft + FRight) div 2;
end;

{------------------------------------------------------------------------------
     Function: TRectangle.GetMidY
  Description: private property Get function
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Gets the MidY, which is a virtual property
 Known Issues:
 ------------------------------------------------------------------------------}
function TRectangle.GetMidY: Integer;
begin
  GetMidY := (FTop + FBottom) div 2;
end;

{Set Procedures -------------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TRectangle.SetAlignment
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Alignment
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.SetAlignment(Value: TAlignment);
begin
  if (Value = FAlignment) then exit;

  FAlignment := Value;
  DoHandleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TRectangle.SetLeft
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Left, which also moves the Right, thereby preserving the Width
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.SetLeft(Value: Integer);
begin
  if (Value = FLeft) then exit;
  FRight := FRight + (Value - FLeft);
  FLeft := Value;
  DoHandleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TRectangle.SetTop
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Top, which also also moves the Bottom, thereby preserving the Height
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.SetTop(Value: Integer);
begin
  if (Value = FTop) then exit;
  FBottom := FBottom + (Value - FTop);
  FTop := Value;
  DoHandleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TRectangle.SetRight
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Right
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.SetRight(Value: Integer);
begin
  if (Value = FRight) then exit;
  FRight := Value;
  DoHandleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TRectangle.SetBottom
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Bottom
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.SetBottom(Value: Integer);
begin
  if (Value = FBottom) then exit;
  FBottom := Value;
  DoHandleChange;
end;

{Set procedures for virtual properties ---------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TRectangle.SetHeight
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Height, a virtual property, by moving the Bottom
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.SetHeight(Value: Integer);
begin
  if ((Value = 0) or (Value = GetHeight)) then exit;
  Inc(FBottom, Value - (FBottom - FTop));
  DoHandleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TRectangle.SetWidth
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Width, a virtual property, by moving the Right
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.SetWidth(Value: Integer);
begin
  if ((Value = 0) or (Value = GetWidth)) then exit;
  Inc(FRight, Value - (FRight - FLeft));
  DoHandleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TRectangle.SetMidX
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the MidX, a virtual property, by moving the Left and Right
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.SetMidX(Value: Integer);
var
  OldMidX: Integer;
  Change: Integer;
begin
  if (Value = GetMidX) then exit;
  OldMidX := (FRight + FLeft) div 2;
  Change := Value - OldMidX;
  Inc(FLeft, Change);
  Inc(FRight, Change);
  DoHandleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TRectangle.SetMidY
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the MidY, a virtual property, by moving the Top and Bottom
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.SetMidY(Value: Integer);
var
  OldMidY: Integer;
  Change: Integer;
begin
  if (Value = GetMidY) then exit;
  OldMidY := (FTop + FBottom) div 2;
  Change := Value - OldMidY;
  Inc(FTop, Change);
  Inc(FBottom, Change);
  DoHandleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TRectangle.SetDeltaX
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: moves the Rectangle in the X direction, by changing the Left and Right
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.SetDeltaX(Value: Integer);
begin
  if (Value = 0) then exit;
  Inc(FLeft, Value);
  Inc(FRight, Value);
  DoHandleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TRectangle.SetDeltaY
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: moves the Rectangle in the Y direction, by changing the Top and Bottom
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.SetDeltaY(Value: Integer);
begin
  if (Value = 0) then exit;
  Inc(FTop, Value);
  Inc(FBottom, Value);
  DoHandleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TRectangle.SetVisible
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Visibility
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.SetVisible(Value: Boolean);
begin
  if (FVisible = Value) then exit;
  FVisible := Value;
  DoHandleChange;
end;

function TRectangle.ClickedOn(iX, iY: Integer): Boolean;
begin
  if ((FLeft <= iX) and
      (iX <= FRight) and
      (FTop <= iY) and
      (iY <= FBottom) and
      (FVisible)) then
    ClickedOn := TRUE
   else
    ClickedOn := FALSE; 
end;

{------------------------------------------------------------------------------
    Procedure: TRectangle.DoHandleChange
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/27/2000
Date modified: 02/27/2000 by Mat Ballard
      Purpose: all Change Event firing passes through here
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TRectangle.DoHandleChange;
begin
  if (FireEvents and assigned(FOnChange) and FVisible) then OnChange(Self);
end;

{TBorder Constructor and Destructor:-------------------------------------------}
{------------------------------------------------------------------------------
  Constructor: TBorder.Create
  Description: Standard Constructor for TBorder
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: initializes component and properties
 Known Issues:
 ------------------------------------------------------------------------------}
Constructor TBorder.Create(AOwner: TPersistent);
begin
{First call the ancestor:}
  inherited Create(AOwner);

{we insert the default values that cannot be "defaulted":}
  FRightEx := Right + 10;
  FBottomEx := Bottom + 10;
end;

{Begin Get Functions --------------------------------------------------------}
{the "virtual" properties:}
{------------------------------------------------------------------------------
     Function: TBorder.GetRightGap
  Description: private property Get function
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Gets the Right Gap
 Known Issues:
 ------------------------------------------------------------------------------}
function TBorder.GetRightGap: Integer;
begin
  GetRightGap := FRightEx - Right;
end;

{------------------------------------------------------------------------------
     Function: TBorder.GetBottomGap
  Description: private property Get function
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Gets the Bottom Gap
 Known Issues:
 ------------------------------------------------------------------------------}
function TBorder.GetBottomGap: Integer;
begin
  GetBottomGap := FBottomEx - Bottom;
end;

{------------------------------------------------------------------------------
     Function: TBorder.GetHeightEx
  Description: private property Get function
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Gets the Total Height (Height + BottomGap)
 Known Issues:
 ------------------------------------------------------------------------------}
function TBorder.GetHeightEx: Integer;
begin
  GetHeightEx := FBottomEx - Top;
end;

{------------------------------------------------------------------------------
     Function: TBorder.GetWidthEx
  Description: private property Get function
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Gets the Total Width (Width + RightGap)
 Known Issues:
 ------------------------------------------------------------------------------}
function TBorder.GetWidthEx: Integer;
begin
  GetWidthEx := FRightEx - Left;
end;

{Begin Set Procedures -------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TBorder.SetLeft
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Left, which DOES NOT move the Right and the RightEx, unlike TRectangle.SetLeft
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TBorder.SetLeft(Value: Integer);
begin
  if (Value = FLeft) then exit;
  FLeft := Value;
  DoHandleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TBorder.SetTop
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Top, which DOES NOT move the Bottom and BottomEx, unlike TRectangle.SetTop
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TBorder.SetTop(Value: Integer);
begin
  if (Value = FTop) then exit;
  FTop := Value;
  DoHandleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TBorder.SetRightEx
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the RightEx
      Comment: the design philosophy is that changing the RightEx changes the value
               of both Right, AND RightEX. If the user changes the RightEx, then that is about
               making the whole object bigger.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TBorder.SetRightEx(Value: Integer);
var
  Change: Integer;
begin
  if (Value = FRightEx) then exit;
  Change := Value - FRightEx;
  FRightEx := Value;
  Inc(FRight, Change);
  DoHandleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TBorder.SetBottomEx
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the BottomEx
     Comments: See comments for SetRightEx
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TBorder.SetBottomEx(Value: Integer);
var
  Change: Integer;
begin
  if (Value = FBottomEx) then exit;
  Change := Value - FBottomEx;
  FBottomEx := Value;
  Inc(FBottom, Change);
  DoHandleChange;
end;

{Begin Set Procedures for virtual properties --------------------------------}
{------------------------------------------------------------------------------
    Procedure: TBorder.SetRightGap
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Right Gap
      Comment: the design philosophy is that changing the Right Gap changes the value
               of Right, NOT RightEX. If the user changes the gap, then that is about
               re-apportioning the available space (Left -> RightEx) between the Right Gap and
               the Width (Right-Left)
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TBorder.SetRightGap(Value: Integer);
var
  OldRightGap, Change, NewRight: Integer;
begin
  if (Value <= 0) then exit;
  if (Value = GetRightGap) then exit;

  OldRightGap := FRightEx - Right;
  Change := Value - OldRightGap;
  NewRight := Right - Change;

  if (NewRight <= Left) then exit;

  Right := NewRight;
  DoHandleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TBorder.SetBottomGap
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Bottom Gap.
      Comment: See comments for SetRightGap
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TBorder.SetBottomGap(Value: Integer);
var
  OldBottomGap, Change, NewBottom: Integer;
begin
  if (Value <= 0) then exit;
  if (Value = GetBottomGap) then exit;

  OldBottomGap := FBottomEx - Bottom;
  Change := Value - OldBottomGap;
  NewBottom := Bottom - Change;

  if (NewBottom <= Top) then exit;

  Bottom := NewBottom;
  DoHandleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TBorder.SetHeightEx
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the
      Comment: the design philosophy is that changing the Total Height changes the value
               of Bottom AND BottomEX. If the user changes the Total Height, then that is about
               making the Height larger or smaller whilst preserving the BottomGap
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TBorder.SetHeightEx(Value: Integer);
var
  OldHeightEx, Change: Integer;
begin
  if (Value <= GetBottomGap) then exit;
  if (Value = GetHeightEx) then exit;

  OldHeightEx := FBottomEx - Top;
  Change := Value - OldHeightEx;

  Bottom := Bottom + Change;
  Inc(FBottomEx, Change);
  DoHandleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TBorder.SetWidthEx
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the WidthEx
     Comments: See comment about SetHeightEx
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TBorder.SetWidthEx(Value: Integer);
var
  OldWidthEx, Change: Integer;
begin
  if (Value <= GetRightGap) then exit;
  if (Value = GetWidthEx) then exit;

  OldWidthEx := FRightEx - Left;
  Change := Value - OldWidthEx;

  Right := Right + Change;
  Inc(FRightEx, Change);
  DoHandleChange;
end;

{TCaption -------------------------------------------------------------------}
{Constructor and Destructor -------------------------------------------------}
{------------------------------------------------------------------------------
  Constructor: TCaption.Create
  Description: Standard Constructor for TCaption
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: initializes component and properties
 Known Issues:
 ------------------------------------------------------------------------------}
Constructor TCaption.Create(AOwner: TPersistent);
begin
{First call the ancestor:}
  inherited Create(AOwner);

{Create font:}
  FFont := TFont.Create;
  FFont.Name := 'Arial';
  FFont.Size := 8;
end;

destructor TCaption.Destroy; 
begin
  FFont.Free;
end;

{Begin Set Procedures -------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TCaption.SetCaption
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Caption of TCaption
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCaption.SetCaption(Value: String);
begin
  if (Value = FCaption) then exit;
  FCaption := Value;
  CreateName;
  DoHandleChange;
  if assigned(FOnCaptionChange) then OnCaptionChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TCaption.SetFont
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Font
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCaption.SetFont(Value: TFont);
begin
  FFont.Assign(Value);
  DoHandleChange;
end;

{General purpose methods ------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TCaption.CreateName
  Description: protected procedure to set a useful name
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Name, generally in response to a Caption change
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCaption.CreateName;
begin
{eg: Caption: X Axis
     Name: X Axis Caption.}
  Name := FCaption + ' Caption';
end;

{------------------------------------------------------------------------------
    Procedure: TCaption.Assign
  Description: standard Assign method
       Author: Mat Ballard
 Date created: 07/06/2000
Date modified: 07/06/2000 by Mat Ballard
      Purpose: implements Assign
 Known Issues:
 ------------------------------------------------------------------------------}
{procedure TCaption.Assign(Source: TPersistent);
begin
  inherited Assign(Source);
  FCaption := TCaption(Source).Caption;
  FFont.Assign(TCaption(Source).Font);
end;}

{------------------------------------------------------------------------------
    Procedure: TCaption.AssignTo
  Description: standard AssignTo method
       Author: Mat Ballard
 Date created: 07/06/2000
Date modified: 07/06/2000 by Mat Ballard
      Purpose: implements AssignTo
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCaption.AssignTo(Dest: TPersistent);
begin
  inherited AssignTo(Dest);
  TCaption(Dest).Caption := FCaption;
  TCaption(Dest).Font.Assign(FFont);
end;

{TTitle -------------------------------------------------------------------}
{Constructor and Destructor -------------------------------------------------}
{------------------------------------------------------------------------------
  Constructor: TTitle.Create
  Description: Standard Constructor for TTitle
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: initializes component and properties
 Known Issues:
 ------------------------------------------------------------------------------}
Constructor TTitle.Create(AOwner: TPersistent);
begin
{First call the ancestor:}
  inherited Create(AOwner);

  FEnvelope.Left := 0;
  FEnvelope.Right := 100;
  FEnvelope.Top := 90;
  FEnvelope.Bottom := 110;
{we don't fire events with a geometry change:}
  FireEvents := FALSE;
end;

{Get methods -----------------------------------------------------------------}

{Set procedures --------------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TTitle.SetCaption
  Description: protected property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Caption, which is complicated by the presence of Units.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TTitle.SetCaption(Value: String);
var
  NewValue: String;
begin
  if (Pos('(', Value) > 0) then
  begin
{There is a "()" in the value, indicating the presence of Units}
    NewValue := Trim(GetWord(Value, '('));
    FUnits := GetWord(Value, ')');
    if (Length(FUnits) = 0) then
      FFullCaption := NewValue
     else
      FFullCaption := NewValue + ' (' + FUnits + ')';
    inherited SetCaption(NewValue);
  end
  else
  begin
    if (Length(FUnits) = 0) then
      FFullCaption := Value
     else
      FFullCaption := Value + ' (' + FUnits + ')';
    inherited SetCaption(Value);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TTitle.SetDirection
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Direction
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TTitle.SetDirection(Value: TDirection);
begin
  if (Value = FDirection) then exit;
  FDirection := Value;
  DoHandleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TTitle.SetOrientation
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Orientation
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TTitle.SetOrientation(Value: TOrientation);
begin
  if (Value = FOrientation) then exit;
  FOrientation := Value;
  DoHandleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TTitle.SetEnvelope
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Envelope: the screen region around which the Title dances
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TTitle.SetEnvelope(Value: TRect);
begin
  if ((Value.Left = FEnvelope.Left) and
      (Value.Top = FEnvelope.Top) and
      (Value.Right = FEnvelope.Right) and
      (Value.Bottom = FEnvelope.Bottom)) then exit;
  FEnvelope := Value;
  DoHandleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TTitle.SetUnits
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Units (eg: Furlongs / Fortnight ^3), and also the FullCaption
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TTitle.SetUnits(Value: String);
begin
  if (Value = FUnits) then exit;
  FUnits := Value;
  if (Length(FUnits) = 0) then
    FFullCaption := FCaption
   else
    FFullCaption := FCaption + ' (' + FUnits + ')';
  DoHandleChange;
end;

{Drawing ----------------------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TTitle.DoGeometry
  Description: TTitle Geometry manager
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the precise screen position of the TTitle.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TTitle.DoGeometry(ACanvas: TCanvas; TheText: String);
begin
  if (FDirection = dHorizontal) then
  begin
{BUG BUG BUG: if ACanvas is a metafile canvas, then TextHeight and TextWidth
 both return zero in D1!}
    Height := Abs(ACanvas.Font.Height);
    Width := ACanvas.TextWidth(TheText);
{Note how "neat" this is: when D1 returns 0 for these Text dimensions,
 TRectangle rejects them, so Height and Width are unchanged !
 Therefore, when we use them below, we use the previous screen values !}
    if (FOrientation = orLeft) then
    begin
      Top := FEnvelope.Top - Height;
      if (Alignment = taLeftJustify) then
        Left := FEnvelope.Left
       else if (Alignment = taRightJustify) then
        Left := FEnvelope.Right - Width
       else {Alignment = taCenter}
        Left := (FEnvelope.Left + FEnvelope.Right - Width) div 2;
    end
    else {FOrientation = oRight}
    begin
      Top := FEnvelope.Bottom;
      if (Alignment = taLeftJustify) then
        Left := FEnvelope.Left
       else if (Alignment = taRightJustify) then
        Left := FEnvelope.Right - Width
       else {Alignment = taCenter}
        Left := (FEnvelope.Left + FEnvelope.Right - Width) div 2;
    end;
  end
  else {FDirection = dVertical}
  begin
{BUG BUG BUG: if ACanvas is a metafile canvas, then TextHeight and TextWidth
 both return zero in D1!}
    Width := Abs(ACanvas.Font.Height);
    Height := ACanvas.TextWidth(TheText);
    if (FOrientation = orLeft) then
    begin
      Left := FEnvelope.Left - Width;
      if (Alignment = taLeftJustify) then
        Top := FEnvelope.Bottom - Height
       else if (Alignment = taRightJustify) then
        Top := FEnvelope.Top
       else {Alignment = taCenter}
        Top := (FEnvelope.Top + FEnvelope.Bottom - Height) div 2;
    end
    else {FOrientation = oRight}
    begin
      Left := FEnvelope.Right;
      if (Alignment = taLeftJustify) then
        Top := FEnvelope.Bottom - Height
       else if (Alignment = taRightJustify) then
        Top := FEnvelope.Top
       else {Alignment = taCenter}
        Top := (FEnvelope.Top + FEnvelope.Bottom - Height) div 2;
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TTitle.Draw
  Description: public Drawing method
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Draws the Caption, either horizontally or vertically, at the desired position
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TTitle.Draw(ACanvas: TCanvas);
var
  {FontHeight, FontWidth: Integer;
  iX, iY: Integer;}
{$IFDEF WIN}
  LogRec: TLogFont;
  OldFontHandle, NewFontHandle: hFont;
{$ENDIF}  
  TheText: String;
begin
  if (not Visible) then exit;
  if (Length(FCaption) = 0) then exit;
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TTitle.Draw: ACanvas is nil !');
{$ENDIF}

  ACanvas.Font.Assign(FFont);
  TheText := FCaption;
  if (Length(FUnits) > 0) then
    TheText := TheText + ' (' + FUnits + ')';
  DoGeometry(ACanvas, TheText);

{output text to screen:}
  if (FDirection = dHorizontal) then
  begin
    ACanvas.TextOut(Left, Top, TheText);
  end
  else {FDirection = dVertical}
  begin
{$IFDEF WIN}
{Gotta use Windows GDI functions to rotate the font:}
    WinProcs.GetObject(ACanvas.Font.Handle, SizeOf(LogRec), Addr(LogRec));
    LogRec.lfEscapement := 900; {Up}
    LogRec.lfOutPrecision := OUT_DEFAULT_PRECIS;
    NewFontHandle := WinProcs.CreateFontIndirect(LogRec);
{select the new font:}
    OldFontHandle := WinProcs.SelectObject(ACanvas.Handle, NewFontHandle);
{$ELSE}
{What on earth do we do under Linux / QT ?}
{$ENDIF}

{Print the vertical axis Title:}
    ACanvas.TextOut(FLeft, FBottom, TheText);

{$IFDEF WIN}
{go back to original font:}
    NewFontHandle := WinProcs.SelectObject(ACanvas.Handle, OldFontHandle);
{and delete the old one:}
    WinProcs.DeleteObject(NewFontHandle);
{$ELSE}
{What on earth do we do under Linux / QT ?}
{$ENDIF}
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TTitle.Assign
  Description: standard Assign method
       Author: Mat Ballard
 Date created: 07/06/2000
Date modified: 07/06/2000 by Mat Ballard
      Purpose: implements Assign
 Known Issues:
 ------------------------------------------------------------------------------}
{procedure TTitle.Assign(Source: TPersistent);
begin
  inherited Assign(Source);
  FDirection := TTitle(Source).Direction;
  FOrientation := TTitle(Source).Orientation;
  FUnits := TTitle(Source).Units;
end;}

{------------------------------------------------------------------------------
    Procedure: TTitle.AssignTo
  Description: standard AssignTo method
       Author: Mat Ballard
 Date created: 07/06/2000
Date modified: 07/06/2000 by Mat Ballard
      Purpose: implements AssignTo
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TTitle.AssignTo(Dest: TPersistent);
begin
  inherited AssignTo(Dest);
  TTitle(Dest).Direction := FDirection;
  TTitle(Dest).Orientation := FOrientation;
  TTitle(Dest).Units := FUnits;
end;

{TLegend ----------------------------------------------------------------------}
{Get functions ----------------------------------------------------------------}
{------------------------------------------------------------------------------
     Function: TLegend.GetHit
  Description: Interprets mouse click position
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Gets the region of the line of the Legend under the input position
 Known Issues:
 ------------------------------------------------------------------------------}
function TLegend.GetHit(iX, iY: Integer; var TheRect: TRect): Integer;
var
  SeriesWidth: Integer;
  TheHit: Integer;
begin
  if (FDirection = dHorizontal) then
  begin
    SeriesWidth := (2 + STRING_DIV_SYMBOL) * FStringWidth div STRING_DIV_SYMBOL;
    TheHit := (iX - Left) div SeriesWidth;
    TheRect.Left := Left + TheHit * SeriesWidth;
    TheRect.Right := TheRect.Left + SeriesWidth;
    TheRect.Top := Top;
    TheRect.Bottom := Bottom;
  end
  else
  begin {dVertical}
    TheHit := (iY - Top) div FFontHeight;
    TheRect.Left := Left;
    TheRect.Right := Right;
    TheRect.Top := Top + TheHit * FFontHeight;
    TheRect.Bottom := TheRect.Top + FFontHeight;
  end;
  GetHit := TheHit;
end;

function TLegend.GetSymbolWidth: Integer;
begin
  GetSymbolWidth := FStringWidth div STRING_DIV_SYMBOL;
end;

{Set procedures ---------------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TLegend.SetDirection
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Direction
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TLegend.SetDirection(Value: TDirection);
begin
  if (Value = FDirection) then exit;
  FDirection := Value;
  DoSize;
  DoHandleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TLegend.SetCount
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Count - the number of lines, which is used to set the Legend Height
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TLegend.SetCount(Value: Integer);
begin
  if (Value < 0) then exit;
  FCount := Value;
  DoSize;
end;

{------------------------------------------------------------------------------
    Procedure: TLegend.SetFontHeight
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the FontHeight, which is used to set the Legend Height
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TLegend.SetFontHeight(Value: Integer);
begin
  if (Value < 0) then exit;
  FFontHeight := Value;
  DoSize;
end;

{------------------------------------------------------------------------------
    Procedure: TLegend.SetStringWidth
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the String Width - the screen width of the String region of the Legend
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TLegend.SetStringWidth(Value: Integer);
begin
  if (Value < 0) then exit;
  FStringWidth := Value;
  DoSize;
end;

procedure TLegend.DoSize;
begin
  if (FDirection = dHorizontal) then
  begin
    Height := FFontHeight;
{There is one "SymbolWidth" on either side of the string:}
{<---LineWidth---><---StringWidth---><-LineWidth->}
{<-Symbol + Line-><-Name of Series--><--Space---->}
    Width := FCount *
      (2 + STRING_DIV_SYMBOL) * FStringWidth div STRING_DIV_SYMBOL;
  end
  else
  begin
    Height := FCount * FFontHeight;
    Width := (2 + STRING_DIV_SYMBOL) * FStringWidth div STRING_DIV_SYMBOL;
  end;
end;



end.
