unit SerList;

{$I Plot.inc}

{-----------------------------------------------------------------------------
The contents of this file are subject to the Mozilla Public License
Version 1.1 (the "License"); you may not use this file except in compliance
with the License. You may obtain a copy of the License at
http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either expressed or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is: PlotList.pas, released 12 September 2000.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@molsci.csiro.au.

Last Modified: 02/25/2000
Current Version: 1.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind. 

Purpose:
This unit contains the TSeriesList sub-component - that manages the data for
ALL Series for TPlot.

Known Issues:
-----------------------------------------------------------------------------}

interface

uses
  classes, controls, graphics, SysUtils, Dialogs,

  Axis, pSeries, Misc;

type
{TSeriesList is a TList of Series that frees it's items.
 Use only descendents of TObject:}
  TSeriesList = class(TList)
  private
{The AxisList is created and managed in the Plot unit and TCustomPlot component.
 The specific axes are:
   0 .. X Axis
   1 .. Primary Y Axis
   2 .. Secondary Y Axis
   3 .. Tertiary Y Axis
   4 .. etc.}
    FAxisList: TList;
    FXAxis: TAxis;
    FYAxis: TAxis;

    FOnChange: TNotifyEvent;

    LastSavedPoint: Integer;

    function GetXmin: Single;
    function GetXmax: Single;

    function GetDataChanged: Boolean;
    function GetMaxNoPts: Integer;
    function GetMinNoPts: Integer;
    function GetTotalNoPts: Integer;
    procedure SetDataChanged(Value: Boolean);
  protected
    procedure StyleChange; virtual;

  public
    property DataChanged: Boolean read GetDataChanged write SetDataChanged stored FALSE;
{Has the data any series changed ?}
    property TotalNoPts: Integer read GetTotalNoPts stored FALSE;
{The total number of points in all series.}
    property MaxNoPts: Integer read GetMaxNoPts stored FALSE;
{The number of points in the largest series.}
    property MinNoPts: Integer read GetMinNoPts stored FALSE;
{The number of points in the smallest series.}
    property Xmin: Single read GetXmin stored FALSE;
{The minimum X value of ALL Series.}
    property Xmax: Single read GetXmax stored FALSE;
{The maximum X value of ALL Series.}

    {property Ymin: Single read GetYmin;
{The minimum Y value of ALL Series connected to the PRIMARY Y Axis.
    property Ymax: Single read GetYmax;
{The maximum Y value of ALL Series connected to the PRIMARY Y Axis.
    property Y2min: Single read GetY2min;
{The minimum Y value of ALL Series connected to the SECONDARY Y Axis.
    property Y2max: Single read GetY2max;
{The maximum Y value of ALL Series connected to the SECONDARY Y Axis.}

    property OnChange: TNotifyEvent read FOnChange write FOnChange;
{This notifies the owner (usually TPlot) of a relevant change in this series.}
{}
{NOTE: D1 does not allow published properties for TList descendants.}

    Constructor Create(AxisListPtr: TList); virtual;
    destructor Destroy; override;

    function Add(XSeriesIndex: Integer): Integer; 
{This adds a new, empty series to the list.}
{}
{Like its ancestor function, and its relatives AddInternal and AddExternal,
 Add returns the index of the new item, where the first item in the list has
 an index of 0.}
    function AddExternal(XPointer, YPointer: pSingleArray; NumberOfPoints: Integer): Integer;
{This adds a new, empty series to the list, and sets its data to point to the
 external XPointer, YPointer data.}
{}
{Like its ancestor function, and its relatives AddInternal and Add,
 AddExternal returns the index of the new item, where the first item in the list has
 an index of 0.}
    function AddInternal(XPointer, YPointer: pSingleArray; NumberOfPoints: Integer): Integer;
{This adds a new, empty series to the list, and copies the data from the
 XPointer, YPointer data.}
{}
{Like its ancestor function, and its relatives Add and AddExternal,
 AddInternal returns the index of the new item, where the first item in the list has
 an index of 0.}

    procedure ClearSeries;
{This deletes all Series from the list, freeing the series.}
{}
{Note that the Clear does not exist in the D1-3 version, because TList.Clear
 is a static method and cannot be overridden. This is why we created ClearSeries.}

    function CloneSeries(TheSeries: Integer): Integer;
{This adds a new, empty series to the list, copies the data and properties from
 TheSeries into the new clone, and changes the color and Y Displacement.}
{}
{CloneSeries returns TRUE if successful.}
    procedure DeleteSeries(Index: Integer);
{This deletes TheSeries from the list.}
    procedure DataAsHTMLTable(var TheData: TStringList);
{This returns all the data in HTML format as a StringList.}

    procedure GetStream(AsText: Boolean; Delimiter: Char;
      TheStream: TMemoryStream);
{This returns all the data as a MemoryStream.}
    procedure GetSubHeaderStream(Delimiter: Char; TheStream: TMemoryStream);
{This returns all the SubHeader data as a MemoryStream.}
    procedure GetBinaryStream(Start, Finish: Integer;
      TheStream: TMemoryStream);
    procedure GetTextStream(Delimiter: Char; Start, Finish: Integer;
      TheStream: TMemoryStream);
    procedure AppendStream(AsText: Boolean; Delimiter: Char; TheStream: TMemoryStreamEx);
{This returns the data collected since LastSavedPoint as a MemoryStream.}

    procedure Draw(ACanvas: TCanvas);
{This draws all the series on the given canvas.}
    procedure DrawColumns(ACanvas: TCanvas);
{This draws all the series on the given canvas in columns.}
    procedure DrawHistory(ACanvas: TCanvas; HistoryX: Single);
{This draws all the series on the given canvas, in a History mode.}
    procedure DrawMultiple(ACanvas: TCanvas; Multiplicity: Byte);
{Extended Drawing procedure for linking multiple series.}
    procedure DrawHistoryMultiple(ACanvas: TCanvas; Multiplicity: Byte);
{Extended Drawing procedure for linking multiple series in History mode.}
    function GetNearestPoint(iX, iY: Integer;
      var TheSeries, NearestiX, NearestiY: Integer;
      var NearestX, NearestY, MinDistance: Single; var pSeries: TSeries): Integer;
{This returns the Index of the nearest point, and sets TheSeries it belongs to,
 and its NearestX and NearestY.}
  {published}

  end;

implementation

uses
  Plot;

{TSeriesList Constructor and Destructor:---------------------------------------}
{------------------------------------------------------------------------------
  Constructor: TSeriesList.Create
  Description: standard Constructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: creates the Axes and initializes LastSavedPoint
 Known Issues:
 ------------------------------------------------------------------------------}
Constructor TSeriesList.Create(AxisListPtr: TList);
begin
  inherited Create;
  FAxisList := AxisListPtr;
  FXAxis := TAxis(FAxisList[0]);
  FYAxis := TAxis(FAxisList[1]);
  LastSavedPoint := 0;
end;

{------------------------------------------------------------------------------
   Destructor: TSeriesList.Destroy
  Description: standard Destructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Frees the Axes and events
 Known Issues:
 ------------------------------------------------------------------------------}
destructor TSeriesList.Destroy;
begin
  FOnChange := nil;
{NOTE: in D1-D3, Clear is static and cannot be overridden, so we had to
 add a ClearSeries for them:}
  ClearSeries;
  inherited Destroy;
end;

{TSeriesList Set procedures ---------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TSeriesList.SetDataChanged
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the DataChanged Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.SetDataChanged(Value: Boolean);
var
  i: Integer;
begin
  for i := 0 to Count-1 do
  begin
    TSeries(Items[i]).DataChanged := Value;
  end; {loop over series}
end;

{TSeriesList Get functions ----------------------------------------------------}
{------------------------------------------------------------------------------
     Function: TSeriesList.GetDataChanged
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the value of the DataChanged Property
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetDataChanged: Boolean;
var
  i: Integer;
  Changed: Boolean;
begin
  Changed := FALSE;
  for i := 0 to Count-1 do
  begin
    if (TSeries(Items[i]).DataChanged) then
    begin
      Changed := TRUE;
      break;
    end;
  end; {loop over series}
  GetDataChanged := Changed;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetXmin
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the value of the XMin Property over ALL Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetXmin: Single;
var
  i: Integer;
  Value: Single;
begin
  Value := 1.e38;
  for i := 0 to Count-1 do
  begin
    if (Value > TSeries(Items[i]).XMin) then
      Value := TSeries(Items[i]).XMin;
  end; {loop over series}
  GetXmin := Value;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetXMax
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the value of the XMax Property over ALL Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetXmax: Single;
var
  i: Integer;
  Value: Single;
begin
  Value := -1.e38;
  for i := 0 to Count-1 do
  begin
    if (Value < TSeries(Items[i]).XMax) then
      Value := TSeries(Items[i]).XMax;
  end; {loop over series}
  GetXmax := Value;
end;

{TSeriesList Memory and list management ---------------------------------------}
{------------------------------------------------------------------------------
     Function: TSeriesList.Add
  Description: Adds a new Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: creates, initializes and adds a new series
 Return Value: the Index of the new Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.Add(XSeriesIndex: Integer): Integer;
var
  Item,
  XSeries: TSeries;
begin
  if (XSeriesIndex > Count-1) then raise
    EComponentError.Create('TSeriesList.Add: invalid X Series Index !');

  if (XSeriesIndex >= 0) then
    XSeries := TSeries(Items[XSeriesIndex])
   else
    XSeries := nil;

  Item := TSeries.Create(Count, FAxisList, XSeries);

  Add := inherited Add(Item);

  StyleChange;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.AddExternal
  Description: Adds a new Series that is externally maintained
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: creates, initializes and adds a new series
 Return Value: the Index of the new Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.AddExternal(XPointer, YPointer: pSingleArray; NumberOfPoints: Integer): Integer;
var
  Item: TSeries;
begin
{set to failure in case Item.AddData fails:}
  AddExternal := -1;

  Item := TSeries.Create(Count, FAxisList, nil);

  if (Item.PointToData(XPointer, YPointer, NumberOfPoints)) then
  begin
{Success:}
    AddExternal := inherited Add(Item);
    StyleChange;
  end
  else
  begin
{Failure:}
    Item.Free;
  end;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.AddInternal
  Description: Adds a new Series from external data (copies)
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: creates, initializes and adds a new series
 Return Value: the Index of the new Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.AddInternal(XPointer, YPointer: pSingleArray; NumberOfPoints: Integer): Integer;
var
  Item: TSeries;
begin
{set to failure in case Item.AddData fails:}
  AddInternal := -1;

  Item := TSeries.Create(Count, FAxisList, nil);
  if (Item.AddData(XPointer, YPointer, NumberOfPoints)) then
  begin
{Success:}
    AddInternal := inherited Add(Item);
    StyleChange;
  end
  else
  begin
{Failure:}
    Item.Free;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.ClearSeries
  Description: frees and deletes all the Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Series management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.ClearSeries;
var
  i: Integer;
  pSeries: TSeries;
begin
  for i := Count-1 downto 0 do
  begin
    pSeries := TSeries(Items[i]);
    Delete(i);
    pSeries.Free;
  end;
  StyleChange;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DeleteSeries
  Description: frees and deletes one particular Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Series management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DeleteSeries(Index: Integer);
var
  pSeries: TSeries;
begin
  pSeries := TSeries(Items[Index]);
  if (MessageDlg('Are you sure you want to delete ' + pSeries.Name + ' ?',
    mtConfirmation, [mbYes,mbCancel], 0) = mrYes) then
  begin
    Delete(Index);
    pSeries.Free;
    StyleChange;
  end;
end;

{TSeriesList the movie ! ------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TSeriesList.Draw
  Description: standard Drawing procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: draws all the Series on a given canvas
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.Draw(ACanvas: TCanvas);
var
  i: Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.Draw: ACanvas is nil !');
{$ENDIF}

  for i := 0 to Count-1 do
  begin
    TSeries(Items[i]).Draw(ACanvas);
  end; {loop over series}
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DrawColumns
  Description: standard Drawing procedure for Columns
       Author: Mat Ballard
 Date created: 09/25/2000
Date modified: 09/25/2000 by Mat Ballard
      Purpose: draws all the Series on a given canvas in Columns
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DrawColumns(ACanvas: TCanvas);
var
  i,
  j,
  iX,
  iXp1, {iX plus 1: the next X ordinate;}
  iXStart,
  iXEnd,
  iY,
  NoBoxes: Integer;
  dX: Single;
  pSeries,
  pSeries0: TSeries;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.DrawColumns: ACanvas is nil !');
{$ENDIF}

  if (Count = 0) then
    exit
  else if (Count = 1) then
    NoBoxes := 1
  else
{create a gap:}
    NoBoxes := Count + 1;

{all columns are plotted against the X Axis and Data of the first series,
 otherwise chaos can result.}
  pSeries0 := TSeries(Items[0]);

{loop over every series:}
  for i := 0 to Count-1 do
  begin
    pSeries := TSeries(Items[i]);
    ACanvas.Pen.Assign(pSeries.Pen);
    ACanvas.Brush.Assign(pSeries.Brush);
    if (pSeries.NoPts > 0) then
    begin
      iXp1 := pSeries0.XAxis.FofX(pSeries0.XData^[0]);
  {loop over every point in each series:}
      for j := 0 to pSeries.NoPts-2 do
      begin
        iX := iXp1;
        iXp1 := pSeries0.XAxis.FofX(pSeries0.XData^[j+1]);
        dX := (iXp1-iX) / NoBoxes;
        iXStart := iX + Round(i*dX);
        iXEnd := iXStart + Round(dX);
        iY := pSeries.YAxis.FofY(pSeries.YData^[j]);
        ACanvas.Rectangle(iXStart, iY, iXEnd, pSeries.YAxis.Bottom);
      end; {for}
    end; {NoPts > 0}
  end; {loop over series}
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DrawHistory
  Description: standard Drawing procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: draws all the Series on a given canvas IN History MODE
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DrawHistory(ACanvas: TCanvas; HistoryX: Single);
var
  i: Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.DrawHistory: ACanvas is nil !');
{$ENDIF}

  for i := 0 to Count-1 do
  begin
    TSeries(Items[i]).DrawHistory(ACanvas, HistoryX);
  end; {loop over series}
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DrawMultiple
  Description: extended Drawing procedure for linking multiple series
       Author: Mat Ballard
 Date created: 09/21/2000
Date modified: 09/21/2000 by Mat Ballard
      Purpose: links multiple Series on a given canvas
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DrawMultiple(ACanvas: TCanvas; Multiplicity: Byte);
var
  i,
  j,
  k,
  iSeries,
  NoGroups: Integer;
  pSeries: TSeries;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.DrawMultiple: ACanvas is nil !');
  Assert(Multiplicity > 1, 'TSeriesList.DrawMultiple: Multiplicity is less than 2 !');
{$ENDIF}
  NoGroups := Count div Multiplicity;
  for i := 0 to MinNoPts-1 do
  begin
    iSeries := -1;
    for j := 1 to NoGroups do
    begin
      Inc(iSeries);
      if (iSeries >= Count) then break;
      pSeries := TSeries(Items[iSeries]);
      ACanvas.MoveTo(
        pSeries.XAxis.FofX(pSeries.XData^[i]),
        pSeries.YAxis.FofY(pSeries.YData^[i]));
      for k := 2 to Multiplicity do
      begin
        Inc(iSeries);
        if (iSeries >= Count) then break;
        pSeries := TSeries(Items[iSeries]);
        ACanvas.LineTo(
          pSeries.XAxis.FofX(pSeries.XData^[i]),
          pSeries.YAxis.FofY(pSeries.YData^[i]));
      end; {Multiplicity}
    end; {NoGroups}
  end; {MinNoPts}
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.DrawHistoryMultiple
  Description: extended Drawing procedure for linking multiple series in History mode
       Author: Mat Ballard
 Date created: 09/21/2000
Date modified: 09/21/2000 by Mat Ballard
      Purpose: links multiple Series in History mode on a given canvas
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DrawHistoryMultiple(ACanvas: TCanvas; Multiplicity: Byte);
var
  i,
  j,
  k,
  iSeries,
  NoGroups: Integer;
  pSeries: TSeries;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TSeriesList.DrawHistoryMultiple: ACanvas is nil !');
  Assert(Multiplicity > 1, 'TSeriesList.DrawHistoryMultiple: Multiplicity is less than 2 !');
{$ENDIF}

{we set the pen mode so that a second call to DrawHistory
erases the curve on screen:}
  ACanvas.Pen.Mode := pmNotXOR;
  NoGroups := Count div Multiplicity;
  for i := MinNoPts-1 downto 0 do
  begin
    iSeries := -1;
    for j := 1 to NoGroups do
    begin
      Inc(iSeries);
      if (iSeries >= Count) then break;
      pSeries := TSeries(Items[iSeries]);
      ACanvas.MoveTo(
        pSeries.XAxis.FofX(pSeries.XData^[i]),
        pSeries.YAxis.FofY(pSeries.YData^[i]));
      for k := 2 to Multiplicity do
      begin
        Inc(iSeries);
        if (iSeries >= Count) then break;
        pSeries := TSeries(Items[iSeries]);
        ACanvas.LineTo(
          pSeries.XAxis.FofX(pSeries.XData^[i]),
          pSeries.YAxis.FofY(pSeries.YData^[i]));
      end; {Multiplicity}
    end; {NoGroups}
  end; {MinNoPts}
end;

{TSeriesList editing, both in designer and active modes -----------------------}
{------------------------------------------------------------------------------
     Function: TSeriesList.CloneSeries
  Description: Clones (Adds then Copies) a Series to a new one
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: creates, initializes, adds then copies a Series
 Return Value: the Index of the new Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.CloneSeries(
  TheSeries: Integer): Integer;
var
  pClone,
  pSeries: TSeries;
  TheClone,
  TheXSeries: Integer;
begin
  if ((TheSeries < 0) or (TheSeries > Count-1)) then raise
    ERangeError.CreateFmt(
      'I cannot clone Series #%d because there are only %d Series !',
        [TheSeries, Count]);

{so lets create the clone: the XDataSeries is either nil, or an existing series}
  pSeries := TSeries(Items[TheSeries]);
  TheXSeries := -1;
  if (pSeries.XDataSeries <> nil) then
    TheXSeries := IndexOf(pSeries.XDataSeries);
  TheClone := Add(TheXSeries);
  pClone := TSeries(Items[TheClone]);

{set properties:}
  pClone.YAxisIndex := pSeries.YAxisIndex;
  pClone.DefSize := pSeries.DefSize;
  pClone.DeltaX := pSeries.DeltaX;
{move the cloned series up by 20 pixels:}
  pClone.DeltaY := pSeries.DeltaY - 30;
  pClone.Name := 'Clone of ' + pSeries.Name;
  pClone.OnChange:= pSeries.OnChange;
  pClone.Pen.Style := pSeries.Pen.Style;
  pClone.Pen.Width := pSeries.Pen.Width;
  pClone.Symbol:= pSeries.Symbol;
  pClone.SymbolSize:= pSeries.SymbolSize;
  pClone.Visible:= pSeries.Visible;

  pClone.AddData(pSeries.XData, pSeries.YData, pSeries.NoPts);

  pClone.ResetBounds;
  pClone.GetBounds;

{AddData above fires the OnChange event for the series:}
  {StyleChange;}
  CloneSeries := TheClone;
end;

{TSeriesList editing, both in designer and active modes -----------------------}
{------------------------------------------------------------------------------
    Procedure: TSeriesList.DataAsHTMLTable
  Description: creates a stringlist of the data as a HTML table
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: copying data
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.DataAsHTMLTable(var TheData: TStringList);
{This puts the data into a stringlist for saving or copying}
var
  i, j: Integer;
  NoPtsMax: Integer;
  pSeries: TSeries;
  SubHeader: String;
begin
  if (TheData = nil) then exit;

{Determine the maximum number of points in all the series:}
  NoPtsMax := 0;
  for i := 0 to Count-1 do
  begin
    if (NoPtsMax < TSeries(Items[i]).NoPts) then
      NoPtsMax := TSeries(Items[i]).NoPts;
  end;

{loop over all series:}
  SubHeader := #9 + '<tr>';
  for i := 0 to Count-1 do
  begin
    pSeries := TSeries(Items[i]);
{create the sub-headings:}
    if (not pSeries.ExternalXSeries) then
    begin
      SubHeader := SubHeader + '<td>' + pSeries.Name + ' - ' + pSeries.XAxis.Title.Caption + '</td>';
    end;
    SubHeader := SubHeader + '<td>' + pSeries.Name + ' - ' + pSeries.YAxis.Title.Caption + '</td>';
{loop over points in each series:}
    for j := 0 to pSeries.NoPts-1 do
    begin
      if (TheData.Count <= j) then
      begin
{start a new row:}
        TheData.Add(#9+#9+ '<tr>');
      end;
      if (not pSeries.ExternalXSeries) then
      begin
        TheData[j] := TheData[j] + '<td>' + FloatToStr(pSeries.XData^[j]) + '</td>';
      end;
      TheData[j] := TheData[j] + '<td>' + FloatToStr(pSeries.YData^[j]) + '</td>';
    end; {loop over points}
    for j := pSeries.NoPts to NoPtsMax-1 do
    begin
      if (TheData.Count <= j) then
      begin
{start a new row:}
        TheData.Add(#9+#9+ '<tr>');
      end;
      if (not pSeries.ExternalXSeries) then
      begin
        TheData[j] := TheData[j] + '<td></td>';
      end;
      TheData[j] := TheData[j] + '<td></td>';
    end;
  end; {loop over series}
  SubHeader := SubHeader + '</tr>';
  TheData.Insert(0, SubHeader);
  TheData.Insert(0, '<table border=2 cellpadding=2 cellspacing=2>');
  TheData.Add('</table>');
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.AppendStream
  Description: puts the data collected since LastSavedPoint into a stringlist for saving or copying.
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: saving data
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.AppendStream(
  AsText: Boolean;
  Delimiter: Char;
  TheStream: TMemoryStreamEx);
var
  i: Integer;
  NoPtsMax: Integer;
begin
  if (TheStream = nil) then exit;

{Determine the maximum number of points in all the series:}
  NoPtsMax := 0;
  for i := 0 to Count-1 do
  begin
    if (NoPtsMax < TSeries(Items[i]).NoPts) then
      NoPtsMax := TSeries(Items[i]).NoPts;
  end;

  if (AsText) then
    GetTextStream(Delimiter, LastSavedPoint, NoPtsMax, TheStream)
   else
    GetBinaryStream(LastSavedPoint, NoPtsMax, TheStream);

  LastSavedPoint := NoPtsMax-1;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesList.GetStream
  Description: puts the data into a MemoryStream
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: for saving or copying
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.GetStream(
  AsText: Boolean;
  Delimiter: Char;
  TheStream: TMemoryStream);
var
  i,
  NoPtsMax: Integer;
begin
  if (TheStream = nil) then exit;

  
  if (Count = 0) then exit;

{Determine the maximum number of points in all the series:}
  NoPtsMax := 0;
  for i := 0 to Count-1 do
  begin
    if (NoPtsMax < TSeries(Items[i]).NoPts) then
      NoPtsMax := TSeries(Items[i]).NoPts;
  end;

  if (AsText) then
    GetTextStream(Delimiter, 0, NoPtsMax-1, TheStream)
   else
    GetBinaryStream(0, NoPtsMax-1, TheStream);

  LastSavedPoint := NoPtsMax-1;
end;


{------------------------------------------------------------------------------
    Procedure: TSeriesList.GetSubHeaderStream
  Description: puts the data sub header onto a stream
       Author: Mat Ballard
 Date created: 08/06/2000
Date modified: 08/06/2000 by Mat Ballard
      Purpose: for saving or copying
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.GetSubHeaderStream(
  Delimiter: Char;
  TheStream: TMemoryStream);
var
  i: Integer;
  pSeries: TSeries;
  SeriesNameLine,
  AxisNameLine,
  DataTypeLine,
  XDataSeriesLine: String;
{$IFDEF DELPHI1}
  pLine: array [0..4095] of char;
{$ENDIF}
begin
  if (TheStream = nil) then exit;

{point at the first series:}
  pSeries := TSeries(Items[0]);
{the first series ALWAYS has both an X and Y data:}
  SeriesNameLine := Delimiter + pSeries.Name;
  AxisNameLine := pSeries.XAxis.Title.Caption + Delimiter +
          pSeries.YAxis.Title.Caption;
  DataTypeLine := 'X' + Delimiter + 'Y';
  XDataSeriesLine := '-' + Delimiter + '-';

{loop over all the rest of the series:}
  for i := 1 to Count-1 do
  begin
    pSeries := TSeries(Items[i]);

{create the sub-headings:}
    if (pSeries.ExternalXSeries) then
    begin
{The X data is in the column of another series, so we just have a Y Column:}
      SeriesNameLine := SeriesNameLine + Delimiter + pSeries.Name;
      AxisNameLine := AxisNameLine + Delimiter + pSeries.YAxis.Title.Caption;
      DataTypeLine := DataTypeLine + Delimiter + 'Y';
      XDataSeriesLine := XDataSeriesLine + Delimiter +
        IntToStr(IndexOf(pSeries.XDataSeries));
    end
    else
    begin
{The X and Y data belong to this series:}
{put the Series Name above the Y column:}
      SeriesNameLine := SeriesNameLine + Delimiter +
        Delimiter + pSeries.Name;
      AxisNameLine := AxisNameLine + Delimiter + pSeries.XAxis.Title.Caption +
        Delimiter + pSeries.YAxis.Title.Caption;
      DataTypeLine := DataTypeLine + Delimiter + 'X' +
        Delimiter + 'Y';
      XDataSeriesLine := XDataSeriesLine + Delimiter + '-' +
        Delimiter + '-';
    end;
  end;

  SeriesNameLine := SeriesNameLine + CRLF;
  AxisNameLine := AxisNameLine + CRLF;
  DataTypeLine := DataTypeLine + CRLF;
  XDataSeriesLine := XDataSeriesLine + CRLF;

{$IFDEF DELPHI1}
  StrPCopy(pLine, SeriesNameLine);
  TheStream.Write(pLine, StrLen(pLine));
  StrPCopy(pLine, AxisNameLine);
  TheStream.Write(pLine, StrLen(pLine));
  StrPCopy(pLine, DataTypeLine);
  TheStream.Write(pLine, StrLen(pLine));
  StrPCopy(pLine, XDataSeriesLine);
  TheStream.Write(pLine, StrLen(pLine));
{$ELSE}
  TheStream.Write(Pointer(SeriesNameLine)^, Length(SeriesNameLine));
  TheStream.Write(Pointer(AxisNameLine)^, Length(AxisNameLine));
  TheStream.Write(Pointer(DataTypeLine)^, Length(DataTypeLine));
  TheStream.Write(Pointer(XDataSeriesLine)^, Length(XDataSeriesLine));
{$ENDIF}
end;

{------------------------------------------------------------------------------
    procedure: TSeriesList.GetBinaryStream
  Description: gets the data as a binary stream
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management: copying and saving
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.GetBinaryStream(
  Start, Finish: Integer;
  TheStream: TMemoryStream);
var
  i,
  j: Integer;
  pSeries: TSeries;
  pNullData: array [0..255] of char;
begin
  for i := 0 to SizeOf(Single)-1 do
    pNullData[i] := 'x';
  pNullData[SizeOf(Single)] := Chr(0);
{all the data is written in BINARY format:}
  for i := Start to Finish do
  begin
    pSeries := TSeries(Items[0]);
{Note: the very first series HAS TO HAVE its own X series:}
    if (i < pSeries.NoPts) then
    begin
      TheStream.Write(pSeries.XData^[i], SizeOf(Single));
      TheStream.Write(pSeries.YData^[i], SizeOf(Single));
    end
    else
    begin
      TheStream.Write(pNullData, SizeOf(Single));
    end;
{loop over all the remaining series:}
    for j := 1 to Count-1 do
    begin
      pSeries := TSeries(Items[j]);
      if (i < pSeries.NoPts) then
      begin
        if (pSeries.ExternalXSeries) then
        begin
          TheStream.Write(pSeries.YData^[i], SizeOf(Single));
        end
        else
        begin
          TheStream.Write(pSeries.XData^[i], SizeOf(Single));
          TheStream.Write(pSeries.YData^[i], SizeOf(Single));
        end;
      end
      else
      begin
        TheStream.Write(pNullData, SizeOf(Single));
      end;
    end; {loop over points}
  end;
end;

{------------------------------------------------------------------------------
    procedure: TSeriesList.GetTextStream
  Description: gets the data as a text stream
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management: copying and saving
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.GetTextStream(
  Delimiter: Char;
  Start, Finish: Integer;
  TheStream: TMemoryStream);
var
  i,
  j: Integer;
  pSeries: TSeries;
  TheLine: String;
{$IFDEF DELPHI1}
  pLine: array [0..255] of char;
{$ENDIF}
begin
{all the data is written in text format:}
  for i := Start to Finish do
  begin
    pSeries := TSeries(Items[0]);
{Note: the very first series HAS TO HAVE its own X series:}
    if (i < pSeries.NoPts) then
      TheLine := FloatToStr(pSeries.XData^[i]) +
        Delimiter +
        FloatToStr(pSeries.YData^[i])
     else
      TheLine := Delimiter + Delimiter;
{loop over all the remaining series:}
    for j := 1 to Count-1 do
    begin
      pSeries := TSeries(Items[j]);
      if (i < pSeries.NoPts) then
      begin
        if (pSeries.ExternalXSeries) then
          TheLine := TheLine +
            Delimiter +
            FloatToStr(pSeries.YData^[i])
         else
          TheLine := TheLine +
            Delimiter +
            FloatToStr(pSeries.XData^[i]) +
            Delimiter +
            FloatToStr(pSeries.YData^[i]);
      end
      else
        TheLine := Delimiter + Delimiter;
    end; {loop over points}
    TheLine := TheLine + CRLF;
{$IFDEF DELPHI1}
    StrPCopy(pLine, TheLine);
    TheStream.Write(pLine, StrLen(pLine));
{$ELSE}
    TheStream.Write(Pointer(TheLine)^, Length(TheLine));
{$ENDIF}
  end;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetNearestPoint
  Description: gets the nearest series and point to the input point
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: user selection of data on screen
 Return Value: the Index of the nearest point
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetNearestPoint(iX, iY: Integer;
  var TheSeries, NearestiX, NearestiY: Integer;
  var NearestX, NearestY, MinDistance: Single; var pSeries: TSeries): Integer;
var
  Distance: Single;
  NX, NY: Single;
  i, N, NearestN, NiX, NiY: Integer;
begin
  MinDistance := 1.0e38;
  TheSeries := 0;
  NearestN := 0;
  if (Count = 0) then
  begin
    GetNearestPoint := -1;
    pSeries := nil;
    exit;
  end;

{loop over all series:}
  for i := 0 to Count-1 do
  begin
    pSeries := TSeries(Items[i]);
    N := pSeries.GetNearestPoint(0, pSeries.NoPts-1, iX, iY, NiX, NiY, NX, NY, Distance);
    if (MinDistance > Distance) then
    begin
      MinDistance := Distance;
      TheSeries := i;
      NearestiX := NiX;
      NearestiY := NiY;
      NearestX := NX;
      NearestY := NY;
      NearestN := N;
    end;
  end; {loop over series}
  pSeries := TSeries(Items[TheSeries]);
  GetNearestPoint := NearestN;
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.StyleChange
  Description: event firing proedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: fires the OnChange event
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesList.StyleChange;
begin
  if Assigned(FOnChange) then OnChange(Self);
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetTotalNoPts
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the value of the TotalNoPts Property
 Return Value: Integer
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetTotalNoPts: Integer;
var
  i,
  Sum: Integer;
begin
{loop over all series:}
  Sum := 0;
  for i := 0 to Count-1 do
  begin
    Sum := Sum + TSeries(Items[i]).NoPts;
  end;
  GetTotalNoPts := Sum;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetMaxNoPts
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 09/21/2000
Date modified: 09/21/2000 by Mat Ballard
      Purpose: gets the value of the MaxNoPts Property
 Return Value: Integer
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetMaxNoPts: Integer;
var
  i,
  TheMax: Integer;
begin
  TheMax := 0;
  if (Count > 0) then
  begin
    TheMax := TSeries(Items[0]).NoPts;
    for i := 1 to Count-1 do
    begin
      if (TheMax < TSeries(Items[i]).NoPts) then
        TheMax := TSeries(Items[i]).NoPts;
    end;
  end;
  GetMaxNoPts := TheMax;
end;

{------------------------------------------------------------------------------
     Function: TSeriesList.GetMinNoPts
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 09/21/2000
Date modified: 09/21/2000 by Mat Ballard
      Purpose: gets the value of the MinNoPts Property
 Return Value: Integer
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesList.GetMinNoPts: Integer;
var
  i,
  TheMin: Integer;
begin
  TheMin := 0;
  if (Count > 0) then
  begin
    TheMin := TSeries(Items[0]).NoPts;
    for i := 1 to Count-1 do
    begin
      if (TheMin < TSeries(Items[i]).NoPts) then
        TheMin := TSeries(Items[i]).NoPts;
    end;
  end;
  GetMinNoPts := TheMin;
end;

end.
