unit SerEdit;

{$I Plot.inc}

{-----------------------------------------------------------------------------
The contents of this file are subject to the Mozilla Public License
Version 1.1 (the "License"); you may not use this file except in compliance
with the License. You may obtain a copy of the License at
http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either expressed or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is: SerEdit.pas, released 12 September 2000.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@molsci.csiro.au.

Last Modified: 02/25/2000
Current Version: 1.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind. 

Purpose:
To facilitate user manipluation of Series properties

Known Issues:
-----------------------------------------------------------------------------}

interface

uses
  Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, Buttons, ExtCtrls,

{These are the D1 units, aliased to Windows:}
{$IFDEF WIN}
  WinTypes, WinProcs,
{$ENDIF}
  
  Misc, ColorGrd;

type
  TSeriesProperty = record
    PenColor: TColor;
    PenWidthIndex: Integer;
    PenStyleIndex: Integer;
    BrushColor: TColor;
    BrushStyleIndex: Integer;
    SymbolIndex: Integer;
    SymbolSize: Integer;
    YAxisIndex: Byte;
    DeltaX: Integer;
    DeltaY: Integer;
    XDataIndependent: Boolean;
    ExternalXSeries: Boolean;
    Visible: Boolean;
  end;

  TSeriesEditorForm = class(TForm)
{This form is used to edit the properties of the series.}  
    BitBtn1: TBitBtn;
    BitBtn2: TBitBtn;
    BitBtn3: TBitBtn;
    Label1: TLabel;
    Label3: TLabel;
    VisibleCheckBox: TCheckBox;
    SymbolComboBox: TComboBox;
    PenGroupBox: TGroupBox;
    Label2: TLabel;
    Label4: TLabel;
    PenWidthComboBox: TComboBox;
    PenStyleComboBox: TComboBox;
    Label5: TLabel;
    NoComboBox: TComboBox;
    GroupBox1: TGroupBox;
    DeltaXEdit: TEdit;
    DeltaYEdit: TEdit;
    Label6: TLabel;
    Label7: TLabel;
    Label8: TLabel;
    SymbolSizeEdit: TEdit;
    Label9: TLabel;
    NameEdit: TEdit;
    Label10: TLabel;
    YAxisComboBox: TComboBox;
    XDataIndependentCheckBox: TCheckBox;
    BrushGroupBox: TGroupBox;
    Label11: TLabel;
    Label13: TLabel;
    BrushStyleComboBox: TComboBox;
    BrushColorEdit: TEdit;
    PenColorEdit: TEdit;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure NoComboBoxClick(Sender: TObject);
    procedure PenWidthComboBoxClick(Sender: TObject);
    procedure PenStyleComboBoxClick(Sender: TObject);
    procedure SymbolComboBoxClick(Sender: TObject);
    procedure VisibleCheckBoxClick(Sender: TObject);
    procedure YAxisComboBoxClick(Sender: TObject);
    procedure IntEditKeyPress(Sender: TObject; var Key: Char);
    procedure DeltaXEditChange(Sender: TObject);
    procedure DeltaYEditChange(Sender: TObject);
    procedure SymbolSizeEditChange(Sender: TObject);
    procedure NameEditChange(Sender: TObject);
    procedure XDataIndependentCheckBoxClick(Sender: TObject);
    procedure BrushColorEditClick(Sender: TObject);
    procedure BrushColorEditKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
  private
    pASP: ^TSeriesProperty;
    CurrentIndex: Integer;

    procedure RunColorDialog(Sender: TObject);
  public
    SeriesPropertyList: TList;
    SeriesNames: TStringList;

    function AddSeries(AName: String; ASeriesProperty: TSeriesProperty): Integer;
    procedure SelectSeries(Index: Integer);
  end;

var
  SeriesEditorForm: TSeriesEditorForm;

implementation

{$R *.DFM}

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.FormCreate
  Description: standard FormCreate procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the position, populates the PenColor combo, and initializes SeriesNames and SeriesPropertyList
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.FormCreate(Sender: TObject);
begin
  SymbolComboBox.ItemIndex := 0;
  SeriesPropertyList := TList.Create;
  SeriesNames := TStringList.Create;
  Left := 10;
  Top := 10;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.FormDestroy
  Description: standard FormDestroy procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: frees the SeriesNames and SeriesPropertyList
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.FormDestroy(Sender: TObject);
var
  i: Integer;
begin
  for i := 0 to SeriesPropertyList.Count-1 do
    FreeMem(SeriesPropertyList.Items[i], SizeOf(TSeriesProperty));

  SeriesPropertyList.Free;
  SeriesNames.Free;
end;

{------------------------------------------------------------------------------
     Function: TSeriesEditorForm.AddSeries
  Description: adds a new series to the list
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: stores the Series properties
 Known Issues:
 ------------------------------------------------------------------------------}
function TSeriesEditorForm.AddSeries(AName: String; ASeriesProperty: TSeriesProperty): Integer;
var
  i: Integer;
  pDestination: Pointer;
  pSource,
  pDest: PChar;
begin
  NoComboBox.Items.Add(IntToStr(SeriesNames.Count));

  SeriesNames.Add(AName);

  GetMem(pDestination, SizeOf(ASeriesProperty));
  pSource := @ASeriesProperty;
{NB: this causes terminal access violations:
  System.Move(pSource, pDestination, SizeOf(TSeriesProperty));}
  pDest := pDestination;
  for i := 1 to SizeOf(ASeriesProperty) do
  begin
    pDest^ := pSource^;
    Inc(pSource);
    Inc(pDest);
  end;

  SeriesPropertyList.Add(pDestination);
  AddSeries := SeriesPropertyList.Count;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.SelectSeries
  Description: responds to selection of a new Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.SelectSeries(Index: Integer);
begin
  pASP := SeriesPropertyList.Items[Index];
  CurrentIndex := Index;

  NameEdit.Text := SeriesNames.Strings[Index];

  PenColorEdit.Color := pASP^.PenColor;
  PenColorEdit.Text := ColorToString(pASP^.PenColor)+ ' ...';
  PenWidthComboBox.ItemIndex := pASP^.PenWidthIndex;
  PenStyleComboBox.ItemIndex := pASP^.PenStyleIndex;

  BrushColorEdit.Color := pASP^.BrushColor;
  BrushColorEdit.Text := ColorToString(pASP^.BrushColor)+ ' ...';
  BrushStyleComboBox.ItemIndex := pASP^.BrushStyleIndex;
  SymbolComboBox.ItemIndex := pASP^.SymbolIndex;
  SymbolSizeEdit.Text := IntToStr(pASP^.SymbolSize);
{We define YAxisIndex to run from 1 to FAxisList.Count-1}
  YAxisComboBox.ItemIndex := pASP^.YAxisIndex-1;
  DeltaXEdit.Text := IntToStr(pASP^.DeltaX);
  DeltaYEdit.Text := IntToStr(pASP^.DeltaY);
  VisibleCheckBox.Checked := pASP^.Visible;
  XDataIndependentCheckBox.Checked := pASP^.XDataIndependent;
  XDataIndependentCheckBox.Visible := pASP^.ExternalXSeries;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.NoComboBoxClick
  Description: responds to user selection of a new Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues: see SelectSeries above
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.NoComboBoxClick(Sender: TObject);
begin
  SelectSeries(NoComboBox.ItemIndex);
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.PenWidthComboBoxClick
  Description: responds to selection of a new Pen Width for the Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.PenWidthComboBoxClick(Sender: TObject);
begin
  pASP^.PenWidthIndex := PenWidthComboBox.ItemIndex;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.PenStyleComboBoxClick
  Description: responds to selection of a new Pen Style for the Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.PenStyleComboBoxClick(Sender: TObject);
begin
  pASP^.PenStyleIndex := PenStyleComboBox.ItemIndex;
  if (PenStyleComboBox.ItemIndex > Ord(psSolid)) then
    PenWidthComboBox.ItemIndex := 1;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.SymbolComboBoxClick
  Description: responds to selection of a new Symbol for the Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.SymbolComboBoxClick(Sender: TObject);
begin
  pASP^.SymbolIndex := SymbolComboBox.ItemIndex;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.VisibleCheckBoxClick
  Description: responds to changes to the Visibility of the Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.VisibleCheckBoxClick(Sender: TObject);
begin
  pASP^.Visible := VisibleCheckBox.Checked;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.YAxisComboBoxClick
  Description: responds to changes in the selected Y Axis (primary or 2nd)
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.YAxisComboBoxClick(Sender: TObject);
begin
{We define YAxisIndex to run from 1 to FAxisList.Count-1}
  pASP^.YAxisIndex := YAxisComboBox.ItemIndex+1;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.IntEditKeyPress
  Description: allows only integer keys to be pressed
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: restrict user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.IntEditKeyPress(Sender: TObject;
  var Key: Char);
begin
  if not (Key in [#8, '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '-']) then
    Key := Chr(0);
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.DeltaXEditChange
  Description: responds to changes in DeltaX
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.DeltaXEditChange(Sender: TObject);
begin
  try
    if (Length(DeltaXEdit.Text) = 0) then
      pASP^.DeltaX := 0
     else
      pASP^.DeltaX := StrToInt(DeltaXEdit.Text);
  except
    if not (DeltaXEdit.Text = '-') then
      DeltaXEdit.Text := '0';
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.DeltaYEditChange
  Description: repsonds to changes in DeltaY
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.DeltaYEditChange(Sender: TObject);
begin
  try
    if (Length(DeltaYEdit.Text) = 0) then
      pASP^.DeltaY := 0
     else
      pASP^.DeltaY := StrToInt(DeltaYEdit.Text);
  except
    if not (DeltaYEdit.Text = '-') then
      DeltaYEdit.Text := '0';
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.SymbolSizeEditChange
  Description: repsonds to changes in SymbolSize
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.SymbolSizeEditChange(Sender: TObject);
begin
  try
    if (Length(SymbolSizeEdit.Text) = 0) then
      pASP^.SymbolSize := 0
    else
      pASP^.SymbolSize := StrToInt(SymbolSizeEdit.Text);
  except
    SymbolSizeEdit.Text := '8';
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TSeriesEditorForm.NameEditChange
  Description: repsonds to changes in the Series Name
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TSeriesEditorForm.NameEditChange(Sender: TObject);
begin
  SeriesNames.Strings[CurrentIndex] := NameEdit.Text;
end;

procedure TSeriesEditorForm.XDataIndependentCheckBoxClick(Sender: TObject);
begin
  pASP^.XDataIndependent := XDataIndependentCheckBox.Checked;
end;

procedure TSeriesEditorForm.RunColorDialog(Sender: TObject);
var
  ColorDialog: TColorDialog;
begin
  ColorDialog := TColorDialog.Create(Self);
  ColorDialog.Options := [cdFullOpen];
  case TControl(Sender).Tag of
    1: {Pen}
      begin
        ColorDialog.Color := pASP^.PenColor;
        ColorDialog.CustomColors.Add('Current=' + IntToHex(ColorToRGB(pASP^.PenColor), 6));
      end;
    2: {Brush}
      begin
        ColorDialog.Color := pASP^.BrushColor;
        ColorDialog.CustomColors.Add('Current=' + IntToHex(ColorToRGB(pASP^.BrushColor), 6));
      end;
  end;

  if (ColorDialog.Execute) then
  begin
    case TControl(Sender).Tag of
      1: {Pen}
        begin
          pASP^.PenColor := ColorDialog.Color;
          PenColorEdit.Color := ColorDialog.Color;
        end;
      2: {Brush}
        begin
          pASP^.BrushColor := ColorDialog.Color;
          BrushColorEdit.Color := ColorDialog.Color;
        end;
    end;
  end;
  ColorDialog.Free;
end;

procedure TSeriesEditorForm.BrushColorEditClick(Sender: TObject);
begin
  RunColorDialog(Sender);
end;

procedure TSeriesEditorForm.BrushColorEditKeyDown(Sender: TObject;
  var Key: Word; Shift: TShiftState);
begin
  RunColorDialog(Sender);
end;

end.
