unit Plot;

{$I Plot.inc}

{-----------------------------------------------------------------------------
The contents of this file are subject to the Mozilla Public License
Version 1.1 (the "License"); you may not use this file except in compliance
with the License. You may obtain a copy of the License at
http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either expressed or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is: Plot.PAS, released 12 September 2000.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@molsci.csiro.au.

Acknowledgements to:
    Anders Melanders - TGifImage (http://www.melander.dk)
    Uberto Barbini (uberto@usa.net), Edmund H. Hand - TPngImage
      the official PNG site: http://www.cdrom.com/pub/png/
    Atanas Stoyanov (http://www.poboxes.com/astoyanov/) and his marvelous MemProof   

Last Modified: 09/18/2000
Current Version: 1.10

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind.

Known Issues:
    - dependency problems with Borland User Components:
      especially with GifImage and PNG graphics units.
      THE FIX: put GifImage and PngImage into a build-once package of their own,
      GifPng_Rxx.dpk (supplied)
    - the PNG library can provoke 'Runtime Error 216 at 00003C8C' when Delphi 4
      exits, but the finished programs are OK.
    - if you do enable GIFs and/or PNGs in 'Plot.inc', then note that you might
      have to install those libraries and/or tinker with the search path of the
      projects.
    - $IFDEFs: TPlot _IS_ used in a Delphi 1 application, and will be migrated
      to Linux. $IFDEFs are therefore unavoidable.
    - Linux / Kylix support - getting there - wrapped all API calls
      (exception: vertical fonts).
    - Explicit dereferencing (eg: XData^[i]) _IS_ required for compatibility
      with Delphi 1. Besides, I prefer it to distinguish dynamic from static arrays.
    - Since TPlotMenu is useless without TPlot, there is only the one
      registration unit: TPlot_Reg.pas.
    - TPlotMenu DOES NOT WORK under D1 - see PlotMenu.pas for details.
    - Due to the infamous 64K limit, D1 popup menus are NOT context sensitive.
    - D5 uses a different dfm file format, so the demo project 'Normal' is NOT
      in 'Plot100_5.bpg' project group, because once you save it from D5 it
      cannot be recovered for other versions of Delphi.
    - if you work across different versions of Delphi, ALWAYS save any form from
      the lowest version of Delphi (eg: Delphi 1), because otherwise the DFM
      files will contain extra properties that will cause stream read errors in
      lower versions of Delphi.
    - BCB++ support required the elimination of the TSpinEdit component from
      the PtEdit form. There are work-arounds (TChart had similar problems), but
      they are ugly.


TO BE DONE:
    - streaming of component properties sometimes does not work
      eg: try setting Axis widths to different values

Future Development:
    - TPlotAction: rather than having 57 bloody actions, what about 1 action,
      with everything based on Tags, like TPlotMenu ?
    - TPlotToolBar ?
    - Linux version

History:
 1.10 9  Oct 2000:  Add and integrate Column capability with TPlotType type and PlotType properties
                    and TSeriesList.DrawColumns.
                    Consolidate some TCustomPlot.xxxClick methods.
                    ****************************
                    *** Borland BC++ support ***
                    ****************************
 1.02 25 Sept 2000: add and integrate High-Low-Close capability with
                    TSeriesList.DrawMultiple and TSeriesList.DrawHistoryMultiple;
                    Hide menu visibility routine from D1 to avoid 64K problems.
 1.01 21 Sept 2000: fix FontWidth bug in TAxis.Draw
                    add TPlotType type,
                    add PlotType and Multiplicity properties
                    add LabelText property to TAxis (for columns)
      18 Sept 2000: delete PlotList.pas (replaced by SerList.pas);
                    remove GifPng from Plot dpks;
                    re-create saved files.
 1.00 13 Sept 2000: Release version 1.00


-----------------------------------------------------------------------------}



interface

uses
  Classes, Clipbrd, Controls, Dialogs, extctrls, Forms, Graphics,
  menus, Messages, Printers, stdctrls, SysUtils,
{These are the D1 units, aliased to Windows:}
{$IFDEF WIN}
  WinTypes, WinProcs,
{$ENDIF}

  OptnsDlg,
{$IFDEF DELPHI1}
  Metafile,
{$ENDIF}

{$IFDEF GIF}
  GifImage,
{$ENDIF}
{$IFDEF PNG}
  PngImage,
{$ENDIF}
  Axis, AxisEdit, Parser, SerList, SerEdit, Misc, PropEdit, pSeries, Titles, Zoom;

const
  TPLOT_VERSION = 110;

  SUBHEADER = 'Subheader';

  FILE_FORMAT_VERSION = 100;
  MAX_FILE_VERSION = 100;

{$IFDEF DELPHI1}
  DEF_EXTENSION = 'plt';
  PROP_EXTENSION = 'prp';
{$ELSE}
  DEF_EXTENSION = 'plot';
  PROP_EXTENSION = 'props';
{$ENDIF}

  FROM_FLASH_EDIT = 37;
  FREE_FLASH_EDIT = 39;

  TAG_BASE = 1000;
  CAPTION_BASE = 3000;
  HINT_BASE = 4000;

  crScope = 1;
  crX = 2;

  ImageExtensions: array[0..4] of string =
    ('wmf', 'emf', 'bmp', 'gif', 'png');

  PICTURE_TYPES =
    'Metafile (picture)|*.wmf'
    + '|Enhanced Metafile (picture)|*.emf'
    + '|Bitmap|*.bmp'
{$IFDEF GIF}
    + '|Compuserve GIF|*.gif'
{$ENDIF}
{$IFDEF PNG}
    + '|Web Graphic|*.png'
{$ENDIF}
    ;


type
  TOnFileEvent = procedure(Sender: TObject; TheFile: String) of object;
{When a file is opened or closed, the app can be notified of the new file name
 using this event.}

  TOnHeaderEvent = procedure(Sender: TObject; TheStream: TMemoryStream) of object;
{When data is opened, the "user-added" Header (eg: the run date, flow rate,
 comments, etc) is passed back to the user for processing.}

  TOnRequestHeaderEvent = procedure(Sender: TObject; TheStream: TMemoryStream) of object;
{When data is saved or copied, then the "user" can add additional data via the
 Header: eg: the run date, flow rate, comments, etc.}

  TOnRequestHTMLHeaderEvent = procedure(Sender: TObject; Header: TStringList) of object;
{When data is saved or copied, then the "user" can add additional data via the
 Header: eg: the run date, flow rate, comments, etc.}

  TOnSelectionEvent = procedure(Sender: TObject; Sel: TRect) of object;
{When the user clicks and drags over a region, this event is fired.}

  TOnDualSelectionEvent = procedure(Sender: TObject; Sel1, Sel2: TRect) of object;
{When the user clicks and drags over TWO regions, this event is fired.}

  TScreenJob = (sjNone,
                sjDrag,
                sjHide,
                sjZoomIn,
                sjEditAxis,
                sjEditFont,
                sjEditPoint,
                sjEditSeries,
                sjFlashEdit,
                sjCopySeries,
                sjDisplace,
                sjCloneSeries,
                sjDeleteSeries,
                sjPosition,
                sjNearestPoint,
                sjAverage,
                sjContractSeries,
                sjSplineSeries,
                sjHighs,
                sjLows,
                sjMovingAverage,
                sjSmoothSeries,
                sjSortSeries,
                sjDifferentiate,
                sjIntegrate,
                sjIntegral,
                sjLineOfBestFit,
                sjDualLineBestFit1, sjDualLineBestFit2,
                sjSelection,
                sjDualSelection1, sjDualSelection2);
{What particular operation is to be carried out ?}
{}
{Note that a 2-region line of best fit is a 2 stage operation.}

  TObjectType = (soNone,
                 soTitle,
                 soLegend,
                 soResult,
                 soXAxis, soXAxisTitle, soXAxisLabel,
                 soYAxis, soYAxisTitle,soYAxisLabel,
                 soLeftBorder, soTopBorder, soRightBorder, soBottomBorder,
                 soSeries);
{What object on the screen got clicked on ?}
{}
{Note that the soYAxis, soYAxisTitle, and soYAxisLabel are generic: several
 sub-components can share this type.}

  TDisplayMode = (dmNormal, dmNone, dmRun, dmHistory);
{What do we do when new data is added ?}
{}
{    dmNormal: Check (and adjust DisplayMode if neccessary) with the addition of every point.}
{    dmNone: Nothing;}
{    dmRun: Increase the Span of the X Axis by 100% to accomodate the new data,
 and alter (increase) the scale of the Y Axis if needed.}
{    dmHistory: Similar to dmRun, except that the X-Axis runs from (Min-Max)
 through (x-Max) to 0.}
{}
{Note that with dmRun we can expect more data with increasing X values.
 Rather than force a complete screen re-draw every time a data point is
 added, we extend the X Axis by 100%.}
{}
{History mode, dmHistory, deserves a few more words. In this mode TCustomPlot
 displays the data over the last History 's worth of points - older data is not
 displayed. The result is a graph that shows the last few seconds or minutes of
 data in great detail.}

{type definitions for dynamic menu arrays:}
  TMenuArray = array[0..127] of TMenuItem;
  pMenuArray = ^TMenuArray;

{Different possible plot types: these determine how the series are interpreted,
 and so how data are displayed on screen}
  TPlotType = (ptXY, ptMultiple, ptColumn{, pt3D, pt3DSurface, pt3DContour});
{ptXY          - normal XY plot

 ptColumn      - columns of data: the last X value is taken as Xn + (Xn - Xn-1),
                 so the appearance is:
                              (X1, Y1)----(X2, Y1)
                              |                  |
                              |                  (Xn, Yn)----(Xn+dX, Yn)
           (X0, Y0)----(X1, Y0)                  |                     |
           |                  |                  |                     |
           |                  |                  |                     |
           |                  |                  |                     |
           |                  |                  |                     |
           (X0, 0)---------(X1, 0)------------(X2, 0)---------(Xn+dX, 0)
                - when there are N (> 1) series, each gap is divided by (N+1):
                              -------
                              |     |
                              |     |
           -------            |     |
           |     |------      |     |
           |     |     |      |     |------
           |     |     |      |     |     |
           |     |     |      |     |     |
           (X0, 0)---------(X1, 0)------------(X2, 0)

 ptMultiple    - each "point" is composed of a multiple of points from
                 the different series: eg:
  high, low        high, average, low        high, low, open, close;
     X                     X                           H
     |                     |                           |
     |                     |                           O
     |                     O                           |
     |                     |                           |
     |                     |                           |
     |                     |                           |
     |                     |                           C
     |                     |                           |
     X                     X                           L
                 See the Multiplicity property !          

 pt3D          - plot each series in order as succesive slices through the surface}

{the actual menus:}
  TMainMenus = (
    mnuFile,
    mnuEdit,
    mnuView,
    mnuCalc);
  TMainOptions = set of TMainMenus;

  TFileMenus = (
    mnuNew,
    mnuOpen,
    mnuOverlayDiv, {convention: Name the Divs after the following menuitem}
    mnuOverlay,
    mnuClearOverlays,
    mnuSaveDiv,
    mnuSave,
    mnuSaveAs,
    mnuSaveImage,
    mnuPrintDiv,
    mnuPrint);
  TFileOptions = set of TFileMenus;

  TEditMenus = (
    mnuCopy,
    mnuCopyHTML,
    mnuCopySeries,
    mnuPaste,
    mnuDisplaceDiv,
    mnuDisplace,
    mnuResetDisplacement,
    mnuEditSeriesDiv,
    mnuNewSeries,
    mnuCloneSeries,
    mnuEditPoint,
    mnuEditSeries,
    mnuDeleteSeries,
    mnuAxisDiv,
    mnuNewY2Axis,
    mnuEditAxis,
    mnuDeleteY2Axis,
    mnuEditFontDiv,
    mnuEditFont,
    mnuEditPropertiesDiv,
    mnuEditProperties); {19}
  TEditOptions = set of TEditMenus;

  TViewMenus = (
    mnuHide,
    mnuShowAll,
    mnuDisplayModeDiv,
    mnuDisplayMode,
    mnuLegend,
    mnuZoomDiv,
    mnuSetAsNormal,
    mnuNormalView,
    mnuManualZoom,
    mnuZoomIn,
    mnuZoomOut);
  TViewOptions = set of TViewMenus;

  TCalcMenus = (
    mnuPosition,
    mnuNearestPoint,
    mnuCalcAverageDiv,
    mnuCalcAverage,
    mnuContractSeries,
    mnuContractAllSeries,
    mnuCubicSplineSeries,
    mnuHighs,
    mnuMovingAverage,
    mnuSmoothSeries,
    mnuSortSeries,
    mnuCalculusDiv,
    mnuDifferentiate,
    mnuIntegrate,
    mnuIntegral,
    mnuLineOfBestFitDiv,
    mnuLineOfBestFit,
    mnuTwoRegionLineOfBestFit); {18}
  TCalcOptions = set of TCalcMenus;

  TModeMenus = (
    mnuNormal,
    mnuNone,
    mnuRun,
    mnuHistory);

  TSaveOption = (soAsText, soProperties);
  TSaveOptions = set of TSaveOption;

  TPopupOptions = class(TPersistent)
  private
    FMenu: TMainOptions;
    FFile: TFileOptions;
    FEdit: TEditOptions;
    FView: TViewOptions;
    FCalc: TCalcOptions;
  public
    Constructor Create; {$IFDEF DELPHI4_UP}reintroduce;{$ENDIF}
{D1 cannot publish sets with more than 16 members}
{$IFDEF DELPHI1}
  public
{$ELSE}
  published
{$ENDIF}
    property Menu: TMainOptions read FMenu write FMenu;
    property File_: TFileOptions read FFile write FFile;
    property Edit: TEditOptions read FEdit write FEdit;
    property View: TViewOptions read FView write FView;
    property Calc: TCalcOptions read FCalc write FCalc;
  end;

  TCustomPlot = class(TCustomPanel)
{Begin declaration of TCustomPlot = class(TCustomPanel)}
  private
    FAbout: String;
    FAxesProperties: String;
{The AxisList is created and managed in the Plot unit and TCustomPlot component.
 The specific axes are:
   0 .. X Axis
   1 .. Primary Y Axis
   2 .. Secondary Y Axis
   3 .. Tertiary Y Axis
   4 .. etc.}
    FAxisList: TList;
    FBorder: TBorder;
    FInstructions: TStringList;
{$IFNDEF DELPHI1}
    FCreatedBy: String;
    FDescription: String;
{$ENDIF}    
    FDisplayMode: TDisplayMode;
    FDisplayHistory: Single;
    FEditable: Boolean;
    FDefaultExtension: String;

    FFileName,      {D:\Data\Delphi\Plot\Test3.csv}
     {GetFileExtension, {csv}
     {GetFileDriveDir,  {D:\Data\Delphi\Plot}
     {GetFileName,      {Test3.csv}
     {GetFileRoot,      {Test3}
{where are the file types ?}
     OpenDriveDir,      {NB: SavePath == FileDriveDir}
     OverlayDriveDir,   {T:\Projects\}
     ImageDriveDir,     {D:\Data\Images}
     PropsFileName: String;
{What file types ?}
     OpenFilterIndex,
     SaveFilterIndex,
     ImageFilterIndex: Integer;

    FHighFont: TFont;
    FMenuTag: Integer;      {may publish later}
    FMovable: Boolean;
    FMultiplicity: Byte;
    FOutlineWidth: Integer;
    FMultiplePen: TPen;
    FPlotMenu: TMainMenu;
    FPlotType: TPlotType;
    FPopupOptions: TPopupOptions;
    FPrintOrientation: TPrinterOrientation;
    FSaveOptions: TSaveOptions;
     FAsText: Boolean;  {the current, which depends on the file type and is ORed with FSaveOptions}
    FSeriesProperties: String;
    FSeriesList: TSeriesList;
    FXAxis: TAxis;
    FYAxis: TAxis;

    FOnFileOpen: TOnFileEvent;
    FOnFileClose: TOnFileEvent;
    FOnHeader: TOnHeaderEvent;
    FOnHeaderRequest: TOnRequestHeaderEvent;
    FOnHTMLHeaderRequest: TOnRequestHTMLHeaderEvent;
    FOnSelection: TOnSelectionEvent;
    FOnDualSelection: TOnDualSelectionEvent;

{the location of the graph title:}
    FTitle: TTitle;
{The results of things like least-squares fits:}
    FResult: TCaption;
{The Legend of Series}
    FLegend: TLegend;

{the four borders, clickable by the user:}
    LeftBorder: TRectangle;
    TopBorder: TRectangle;
    RightBorder: TRectangle;
    BottomBorder: TRectangle;

{Ignore changes in sub-components during painting:}
    IgnoreChanges: Boolean;

    FScreenJob: TScreenJob;

{The list of (TRectangle descended) objects on screen:}
    ScreenObjectList: TList;

{The timer to start click-and-drag operations:}
    MouseTimer: TTimer;
{the position of the mousedown:}
    MouseStart: TPoint;

{The starting point of click-and-drag operations:
    MouseStart: TPoint; - replaced by Selection1}
{which object(s) were clicked ?}
    ClickedObjectType: TObjectType;
    pClickedObject: Pointer;
    SecondClickedObjectType: TObjectType;
    pSecondClickedObject: Pointer;
{the starting position of the object:}
    ClickedObjectOffset: TPoint;

{the parameters of the line of best fit:}
    Slope, Intercept: Single;
{The rectangular outline of a ScreenObject that is dragged around the screen:}
    Selection: TRectangle;
{The first selection in a Dual Selection operation:}
    Sel1, Sel2: TRect;
    
{Popup menus:}
    FPlotPopUpMenu: TPopupMenu;
    WhichPopUpMenu: TPopupMenu;
    WhichPopUpItems: array[0..1] of TMenuItem;

{the in-place editor:}
    FFlashEdit: TEdit;

{the currently selected (eg: by a mouse-click) series:}
    TheSeries: Integer;
    pSeries: TSeries;
    ThePointNumber: Integer;

{the currently selected (eg: by a mouse-click) Axis:}
    TheAxis: Integer;
    pAxis: TAxis;

{The clipboard format number for HTML:}
    ClipBoardFormatForHTML: Integer; {aka CF_HTML}

{Overlay Management:}
    FirstOverlay: Integer;

    BevelGap: Integer;

    FileExtensions: array[0..3] of String;
   {= (
    FDefaultExtension,
    'csv',
    'txt',
    '*');}

    FileTypes: String;
  {=
    'Plot Files|*.' + FDefaultExtension
    + '|Comma Sep Var Files|*.csv'
    + '|Text Files|*.txt'
    + '|All Files|*.*';}



{Get functions}
    function GetClickAndDragDelay: Integer;
    function GetNoYAxes: Integer;
    function GetSeries(Index: Integer): TSeries;
    function GetSeriesFromUser: Boolean;
    function GetAxisFromUser(StartAxis: Word): Boolean;
    function GetFilterIndex(Ext: String): Integer;
{$IFDEF COMPILER4_UP}
{while TImageList exists from Delphi 2 onwards, TMenu and TPopupMenu
 do not have an Images property until Delphi 4}
    function GetImages: TImageList; {TCustomImageList}
    procedure SetImages(Value: TImageList);
{$ENDIF}

{SetProcedures:}
{The main geometry manager.}
    procedure SetAxisDimensions;
{Sets the behaviour upon adding new data points.}
    procedure SetDisplayModeHistory(HistoryValue: Single; ScalingValue: TDisplayMode);
{$IFNDEF DELPHI1}
    {Puts CreatedBy and Description into the metafile.}
    procedure SetMetafileDescription;
{$ENDIF}

{Responding to mouse events, click & drag:}
    procedure GetTheClickedObject(X, Y: Integer);
    procedure MouseTimeOut(Sender: TObject);
    procedure MoveTheClickedObjectTo(X, Y: Integer);
    procedure OutlineTheClickedObject;
    procedure OutlineTheSelection;
    procedure SetResult(Slope, Intercept, Rsq: Single);
    procedure StretchTheClickedObjectTo(X, Y: Integer);
    procedure SwapEnds;
{These two respond to a user choice between overlaying objects:}
    procedure MoveTheClickedObjectClick(Sender: TObject);
    procedure MoveSecondClickedObjectClick(Sender: TObject);
{Sets the width of the outline for screen objects like lines: axes and borders.}
    procedure CreateFlashEditor;
    procedure FlashEditKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
{Processes key presses from the in-place editor.}
    procedure FlashEditExit(Sender: TObject);
{Deals with exits from the in-place editor.}
    procedure WMUser(var Message: TMessage); message WM_USER;

  protected
{Property Set Procedures:}
    procedure SetDefaultExtension(Value: String);
    procedure SetDisplayMode(Value: TDisplayMode);
    procedure SetHistory(Value: Single);
{Handles file names for saving and opening:}
    procedure SetFileName(Value: String);
    procedure SetNoYAxes(Value: Integer);
    procedure SetPlotType(Value: TPlotType);
    procedure SetOutlineWidth(Value: Integer);
    procedure SetMultiplicity(Value: Byte);
    procedure SetMultiplePen(Value: TPen);
    procedure SetClickAndDragDelay(Value: Integer);
    procedure SetOnSelection(Value: TOnSelectionEvent);
    procedure SetOnDualSelection(Value: TOnDualSelectionEvent);

    procedure StyleChange(Sender: TObject);
{Responds to changes in subcomponents.}


{Copying data:}
    procedure CopyText; virtual;
{Copies the data as tab-delimited text to the Clipboard, with any
 TCustomPlot.Owner added header.}
    procedure CopyHTML(Format: Word); virtual;
{Copies the data as HTML to the Clipboard in CF_HTML format, with any
 TCustomPlot.Owner added header.}
{Copying pictures:}
    procedure CopyBitmap; virtual;
{Does what it says.}
    procedure CopyWMF(Enhanced: Boolean); virtual;
{Does what it says.}

    procedure CreateMenus;
{Creates the two popup menus.}

    procedure DoFileClose(AFileName: String); dynamic;
    procedure DoFileOpen(AFileName: String); dynamic;
    procedure DoHeader(TheStream: TMemoryStream); dynamic;
    procedure DoHeaderRequest(TheStream: TMemoryStream); dynamic;
    procedure DoHTMLHeaderRequest(TheHeader: TStringList); dynamic;
    procedure DoSelection(Sel1: TRect); dynamic;
    procedure DoDualSelection(Sel1, Sel2: TRect); dynamic;

{$IFDEF GIF}
    procedure SaveAsGIF(AFileName: String); virtual;
{$ENDIF}
{Does what it says.}
{}
{If GIF is defined (in Plot.inc) then saving as a GIF is enabled.}

{$IFDEF PNG}
    procedure SaveAsPNG(AFileName: String); virtual;
{$ENDIF}
{Does what it says.}
{}
{If PNG is defined (in Plot.inc) then saving as a PNG is enabled.}

{File manipulation:}
    procedure OpenProperties(AFileName: String);
{Saves the Plot Properties to a file.}
    procedure SaveTheProperties(AFileName: String);
{Saves the Plot Properties to a file.}

{Saving as a picture:}
    procedure SaveAsBitmap(AFileName: String); virtual;
{Does what it says.}
    procedure SaveAsWMF(AFileName: String; Enhanced: Boolean); virtual;
{Does what it says.}

    procedure ParseData(TheData: TStringList);
{This parses imported or pasted data and adds it to the SeriesList.}
    procedure ConvertTextData(ColCount, SeriesCount, FirstLine: Integer;
      Delimiter: String; TheData: TStringList; SeriesInfo: pSeriesInfoArray);
{This takes a parsed stringlist converts it to numerical data.}
    procedure ConvertBinaryData(ColCount, SeriesCount: Integer;
      TheStream: TMemoryStream; SeriesInfo: pSeriesInfoArray);
{This takes a parsed stringlist converts it to numerical data.}
    procedure DblClick; override;
{Some items - ie: the Title Captions, can be edited directly.}
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
{Implements dragging with the mouse.}
    procedure MouseMove(Shift: TShiftState; X,Y: Integer); override;
{Further implements dragging with the mouse.}
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X,Y: Integer); override;
{Implements dragging with the mouse, and right clicking for a popup menu.}
    procedure ProcessClickedObject(pObject: Pointer; TheObjectType: TObjectType); virtual;
{Applies effects of clicking and dragging to the selected object.}

{Paint and Draw:}
    procedure Draw(ACanvas: TCanvas); virtual;
{This draws the graph on a Canvas. The canvas can be:}
{}
{    1. Self.Canvas - ie: on screen;}
{    2. a Bitmap, for copying and saving.}
{    3. a MetafileCanvas, for copying and saving.}
{    4. the printer.}
{}
{It is called by Paint, and also many of the copying and saving routines.}
    procedure DrawLegend(ACanvas: TCanvas);
{This draws the legend - it is called by Draw.}
    procedure Paint; override;
{The normal paint procedure. Most of the work is done in Draw.}
    procedure DrawInstructions;
{Just draws the instructions.}
    procedure Resize; override;
{The normal Resize procedure - it manages screen geometry.}
    procedure ZeroScreenStuff; dynamic;
{This re-initializes all the mouse-selection related variables}

{$IFNDEF DELPHI1}
    procedure DetermineMenuVisibility;
{Sets the Visibility of the items of the internal Popup Menu.}
    procedure SetSeriesVisibility(Value: Boolean);
{Makes series-related menu items visible or otherwise}
    {procedure DetermineMenuEnabledness(TheMenu: TMenu); is public for TPlotMenu}
    procedure SetSeriesEnabledness(TheMenu: TMenu);
{Makes the Series-related items of an external menu Enabled or otherwise}
{$ENDIF}

{Properties that will be published in the non-custom descendant:}
    Property About: string read FAbout write FAbout stored False;
{Displays the "About" dialog box for this component.}
    Property AxesProperties: string read FAxesProperties write FAxesProperties stored False;
{Displays the "Axis Editor" dialog box for the FAxisList subcomponent.}
    Property Border: TBorder read FBorder write FBorder;
{Manages the geometry of TCustomPlot: where the axes are, where they can go, etc.}
    Property Instructions: TStringList read FInstructions write FInstructions;
{This is a message to the user, at the bottom-left of the graph, in the current
 Font. It disappears on a MouseDown event.}

    Property ClickAndDragDelay: Integer read GetClickAndDragDelay write SetClickAndDragDelay;
{The delay (in milliseconds) before a clicked object becomes draggable.}
{$IFNDEF DELPHI1}
    Property CreatedBy: String read FCreatedBy write FCreatedBy;
{A string that is stored in the metafile description.}
    Property Description: String read FDescription write FDescription;
{A string that is stored in the metafile description.}
{$ENDIF}
    Property DisplayHistory: Single read FDisplayHistory write SetHistory;
{The width of the X Axis when in History mode.}
    Property DisplayMode: TDisplayMode read FDisplayMode write SetDisplayMode;
{See TDisplayMode.}
    Property Editable: Boolean read FEditable write FEditable;
{Are screen objects like axes Editable ?}

    Property DefaultExtension: String read FDefaultExtension write SetDefaultExtension;
{What is the default extension of Plot files ?}

    Property FileName: String read FFileName write SetFileName stored FALSE;
{This is the FileName to which the data is saved, or opened from.}
{}
{If FileName is blank, and an OpenClick or SaveClick occurs, then the standard
 file dialog box appears to let the user pick a name.}

    property HighFont: TFont read FHighFont write FHighFont;
{The font for annotation of the Highs and Lows.}
{$IFDEF COMPILER4_UP}
    property Images: TImageList read GetImages write SetImages; {TCustomImageList}
{The images for the popupmenu.}
{$ENDIF}
    property Legend: TLegend read FLegend write FLegend;
{The list of series with their line styles. This is a moveable, on-screen object.}
    property MultiplePen: TPen read FMultiplePen write SetMultiplePen;
{The pen to use for the verticle lines of ptMultiple PlotTypes (eg: High-Low).}
    Property PlotType: TPlotType read FPlotType write SetPlotType;
{What type of plot is this ?}
    Property PopupOptions: TPopupOptions read FPopupOptions write FPopupOptions;
{If true, then these popup menu items are visible.}
    Property Movable: Boolean read FMovable write FMovable;
{Are screen objects like axes movable ?}
    Property Multiplicity: Byte read FMultiplicity write SetMultiplicity;
{When the PlotType is ptMultiple, the series are grouped into multiples of Multiplicity.
 Otherwise ignored.}    
    Property NoYAxes: Integer read GetNoYAxes write SetNoYAxes;
{The total number of Y Axes (primary, secondary, tertiary, etc).}    
    Property PrintOrientation: TPrinterOrientation
      read FPrintOrientation write FPrintOrientation;
{Shall we print the graph in Landscape or Portrait mode ?}
    Property OutlineWidth: Integer read FOutlineWidth write SetOutlineWidth;
{This is the width of the outline for screen objects like lines: axes and borders.}
    Property SaveOptions: TSaveOptions read FSaveOptions write FSaveOptions;
{Shall we save the data as Text or binary ?}
{}
{Shall we also save the Plot properties when we save the data ?
 If we do, then we:
    1. Save the properties in a seperate file;
    2. Look for a properties file to open.}
    Property SeriesProperties: string read FSeriesProperties write FSeriesProperties stored False;
{Displays the "Series Editor" dialog box for the TSeriesList subcomponent.}
    Property Title: TTitle read FTitle write FTitle;
{The title of the graph, including its geometry, font and visibility.}
    Property XAxis: TAxis read FXAxis write FXAxis;
{This is the X Axis. Every nice graph should have an X Axis.}
    Property YAxis: TAxis read FYAxis write FYAxis;
{This is the Y Axis. Every nice graph should have at least one Y Axis.}
{}
{Each Series must know what Y Axes it is being plotted against:
 Primary (this one) or Secondary.}

{Events:}
    Property OnFileOpen: TOnFileEvent read FOnFileOpen write FOnFileOpen;
{When a file is opened, the app can be notified of the new file name using this event.}

    Property OnFileClose: TOnFileEvent read FOnFileClose write FOnFileClose;
{When a file is closed, the app can be notified of the new file name using this event.}

    Property OnHeader: TOnHeaderEvent read FOnHeader write FOnHeader;
{When data is opened, this event passes the header information back to the "user".}

    Property OnHeaderRequest: TOnRequestHeaderEvent read FOnHeaderRequest write FOnHeaderRequest;
{When data is saved or copied, this event allows the user to add a header
 to the data.}

    Property OnHTMLHeaderRequest: TOnRequestHTMLHeaderEvent read FOnHTMLHeaderRequest write FOnHTMLHeaderRequest;
{When data is copied as HTML, this event allows the user to add a header
 to the data.}

    Property OnSelection: TOnSelectionEvent read FOnSelection write SetOnSelection;
{When the user selects a region, then this event is fired.}
{}
{Note that after firing, this event is set to nil, so you have to reset it every usage.}
    Property OnDualSelection: TOnDualSelectionEvent read FOnDualSelection write SetOnDualSelection;
{When the user selects TWO regions, then this event is fired.}
{}
{Note that after firing, this event is set to nil, so you have to reset it every usage.}

  public
    Property PlotPopUpMenu: TPopupMenu read FPlotPopUpMenu;
{The public exposure of the popupmenu for PlotMenu's use.}
    Property Series[Index: Integer]: TSeries read GetSeries {$IFDEF DELPHI2_UP}stored FALSE{$ENDIF}; default;
{This provides direct access to the series maintained by SeriesList.}
    Property SeriesList: TSeriesList read FSeriesList stored FALSE;
{This is the data in a list of data series.}

    Property ScreenJob: TScreenJob read FScreenJob write FScreenJob stored FALSE;

    Constructor Create(AOwner: TComponent); override;
{The usual Constructor, where sub-components are created, properties set, etc.}
    Destructor Destroy; override;
{The usual Destructor, where sub-components are destroyed.}

    function GetFileExtension: String; {csv}
    function GetFileDriveDir: String;  {D:\Data\Delphi\Plot}
    function GetFileRoot: String;      {Test3}

    procedure ShowAbout;
{This displays the components "About" dialog box.}
    procedure ShowSeries;
{This displays the properties of the Series in the SeriesList.}
    procedure ShowAxes;
{This displays the properties of the Series in the SeriesList.}

{Wrappers for the SeriesList functions; the first three also set the
 OnChange event and the Visibility property:}
    function Add(XSeriesIndex: Integer): Integer;
{This adds a new, empty series to the list.}
    function AddExternal(XPointer, YPointer: pSingleArray; NumberOfPoints: Integer): Integer;
{This adds a new, empty series to the list, and sets its data to point to the
 external XPointer, YPointer data.}
    function AddInternal(XPointer, YPointer: pSingleArray; NumberOfPoints: Integer): Integer;
{This adds a new, empty series to the list, and copies the data from the
 XPointer, YPointer data.}
    function CloneSeries(TheSeries: Integer): Integer;
{This adds a new, empty series to the list, copies the data and properties from
 TheSeries into the new clone, and changes the color and Y Displacement.}
    procedure DeleteSeries(Index: Integer);
{This deletes TheSeries from the list.}

    procedure DetermineMenuEnabledness(TheMenu: TMenu); 

    procedure AddSlice(NoPoints: Integer; XYArray: pXYArray);
{Add a slice of readings to the internal series.
 These will become the Nth points in every series.}
{}
{This method is extremely useful for data acquisition.}

    procedure Trace;
{This draws all the Series in an erasable mode without re-drawing the Axes,
 Titles or Legend.}
{}
{More specifically, the first call to Trace draws the Series, the second erases them.}
{}
{This is useful for rapidly changing data such as an oscilloscope. To use it:}
{}
{    1. Set up the Axes, etc.}
{    2. Create each Series you need.}
{    3a. Use AllocateNoPts for each one, or:}
{    3b. Allocate and manage the memory yourself, then use PointToData.}
{    4. Dump you data into each Series via the XData and YData properties.}
{    5. Trace the data.}
{    6. Get more data.}
{    7. Trace the data again to erase the old image.}
{    8. Repeat steps (4)-(7) indefinitely.}
{}
{Note that in step (4), you can either use the XData and YData properties as:}
{}
{    a. pSeries.XData^[i];}
{    b. pX, pY: pSingle; pX := pSeries.XData; pY := pSeries.YData;}

    procedure CopyClick(Sender: TObject);
{This responds to a user selection of "Copy" in the popupmenu,
 and copies the data as text, as a bitmap, and as an enhanced metafile.}
    procedure HideClick(Sender: TObject);
{This hides (.Visible := FALSE) the selected object.}
    procedure PrintClick(Sender: TObject);
{This prints the graph.}
    procedure ShowAllClick(Sender: TObject);
{This shows (.Visible := TRUE) ALL objects.}
    procedure PositionClick(Sender: TObject);
{This reports the Position of the mouse (right) click.}
    procedure NearestPointClick(Sender: TObject);
{This reports the Position of the nearest data series point to the mouse
 (right) click.}
    procedure DeleteSeriesClick(Sender: TObject);
{This deletes the currently selected series.}

    procedure CopySeriesClick(Sender: TObject);
{This copies the selected series to the clipboard in tab-delimited form.}
    procedure CloneSeriesClick(Sender: TObject);
{This creates a new series, and copies the data of the selected series  into it.}
    procedure ModeClick(Sender: TObject);
    {procedure ModeNoneClick(Sender: TObject);
    procedure ModeRunClick(Sender: TObject);
    procedure ModeHistoryClick(Sender: TObject);}
{This sets the Display Mode of the graph.}
    procedure PasteClick(Sender: TObject);
{This pastes (Tab Delimited) data from the clipboard.}
    procedure LineBestFitClick(Sender: TObject);
{This initiates a line of best fit determination.}
    procedure TwoRegionLineBestFitClick(Sender: TObject);
{This initiates a line of best fit determination over two different regions.}
    procedure SmoothSeriesClick(Sender: TObject);
{This smoothes the currently seleccted data series.}
    procedure ContractSeriesClick(Sender: TObject);
{This contracts the currently selected data series by a factor of 2, 3, 4, etc.}
    procedure ContractAllSeriesClick(Sender: TObject);
{This contracts ALL data series by a factor of 2, 3, 4, etc.}

    procedure LegendClick(Sender: TObject);
{This responds to a user selection of "Direction" in the popupmenu.}
    procedure EditAxisClick(Sender: TObject);
{This responds to a user selection of "Axis ..." in the popupmenu,
 runs the Edit Axis Dialog, and assigns any changes to the appropriate axis.}
    procedure EditFontClick(Sender: TObject);
{This responds to a user selection of "Font ..." in the popupmenu,
 runs the FontDialog, and assigns any changes to the appropriate object.}
    procedure EditPointClick(Sender: TObject);
{This responds to a user selection of "Point ..." in the popupmenu,
 runs the Series.PointEdit method, which displays and runs the PointEditor.}
    procedure EditSeriesClick(Sender: TObject);
{This responds to a user selection of "Series ..." in the popupmenu,
 runs the TSeriesList.Edit method, which displays and runs the SeriesEditor.}
    procedure ResetDisplacementClick(Sender: TObject);
{This sets the Displacement properties DeltaX and DeltaY to ZeroScreenStuff.}
    procedure PropertiesClick(Sender: TObject);
{This responds to a user selection of "Properties ..." in the popupmenu,
 runs the PropertiesDialog, and assigns any changes to the appropriate objects.}

    procedure NewClick(Sender: TObject);
{This responds to a user selection of "New" in the File popupmenu,
 and clears all the data and resets the graph.}

    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
{needed by D1 for PlotMenu insertion/removal}

    procedure OpenClick(Sender: TObject);
{This responds to a user selection of "Open ..." in the popupmenu,
 runs the Open Dialog, and calls the OpenAsTextData method.}
    procedure OpenFile(TheFile: String);

    procedure ClearOverlaysClick(Sender: TObject);
{This responds to a user selection of "Clear Overlays" in the popupmenu,
 and removes any Overlays.}
    procedure OverlayClick(Sender: TObject);
{This responds to a user selection of "Overlay Data" in the popupmenu,
 runs the Overlay Dialog, and calls the OpenAsTextData method.}
    procedure SaveImageClick(Sender: TObject);
{This responds to a user selection of "Save Image" in the popupmenu,
 runs the Save Dialog, and calls the SaveAsBitmap, SaveAsGIF or the SaveAsWMF method.}
    procedure SaveClick(Sender: TObject);
{This responds to a user selection of "Save Data" in the popupmenu,
 runs the Save Dialog if FileName is blank, and calls the SaveToFile method.}
    procedure SaveAsClick(Sender: TObject);
{This responds to a user selection of "Save Data As" in the popupmenu,
 runs the Save Dialog, and calls the SaveToFile method.}
    procedure LoadFromFile(AFileName: String); virtual;
{Saves the data as text to a text file, with any TCustomPlot.Owner added header.}
    procedure AppendToFile;
{Appends the data as text to a text file.}
    procedure SaveToFile(AFileName: String; FAsText: Boolean); virtual;
{Saves the data as text to a text file, with any TCustomPlot.Owner added header.}

    procedure SetAsNormalClick(Sender: TObject);
{Defines the current view == Mins and Maxes of axes, as the Normal view.}
    procedure NormalViewClick(Sender: TObject);
{Zooms the screen the screen to the Normal view.}
    procedure ManualZoomClick(Sender: TObject);
{Zooms the screen using a manual dialog box.}
    procedure ZoomOutClick(Sender: TObject);
{Zooms the screen out after a zoom-in operation.}

    procedure MakeDummyData(NoSteps: Integer);
{This procedure is used to generate some dummy data for testing purposes.}

    procedure CopyHTMLClick(Sender: TObject);
{Copies the data as HTML to the Clipboard in CF_TEXT format, with any
 TCustomPlot.Owner added header.}
    procedure DisplaceClick(Sender: TObject);
{This responds to a user selection of "Displace" in the popupmenu, and runs
 the Displacement Form, which moves the selected Series from its origin.}
    procedure DifferentiateClick(Sender: TObject);
{This responds to a user selection of "Differentiate" in the popupmenu,
 and replaces the selected series with its differential.
{}
{(Hint: Clone the series first !)}
    procedure HandleClick(Sender: TObject; TheTag: Integer);
{The externally-exposed event handler for all menu items.}
{}
{It is used by TPlotMenu to forward the click event to TPlot, which then passes
 it onto the appropriate handler method based on the value of TheTag.}
    function GetIndicesFromTag(TheTag: Integer; var i, j: Integer): Boolean;
{Get the i and j indixes for the menu from the Tag.}
    procedure IntegrateClick(Sender: TObject);
{This responds to a user selection of "Integrate" in the popupmenu, and replaces
 the selected series with its integral.}
{}
{(Hint: Clone the series first !)}
    procedure IntegralClick(Sender: TObject);
{This responds to a user selection of "Integral" in the popupmenu,
 and calculates the integral of the selected series over the user-selected
 (by click-and-drag) range.}
    procedure SortClick(Sender: TObject);
{}
    procedure SplineClick(Sender: TObject);
{This responds to a user menu click and performs a cubic spline interpolation
 of  the currently selected data series by calling the Spline method.}
    function Spline(ASeries: Integer): Integer;
{This performs a cubic spline interpolation of ASeries by calling the
 TSeries.Spline method to place the cubic spline into a new data series.}
    procedure ZoomInClick(Sender: TObject);
{}
    procedure HighsClick(Sender: TObject);
{Finds and displays the Highs (peaks) and/or Lows (troughs) of a series.}
    procedure MovingAverageClick(Sender: TObject);
{Calculates and displays the Moving Average of a series.}
    procedure AverageClick(Sender: TObject);
{Calculates the Average of a series over a range.}
    procedure AddAxisClick(Sender: TObject);
{Adds a new axis, based on either the selected or last axis.}    
    procedure DeleteAxis(Index: Integer; Confirm: Boolean);
{Deletes the selected axis.}
    procedure DeleteAxisClick(Sender: TObject);
{Deletes the axis selected by the user.}
    procedure SetPlotMenu(Value: TMainMenu);
{Sets the PlotMenu property.}    
    procedure Clear(Cancellable: Boolean);
{Saves any changed files (at user request) and then clears the SeriesList.}    

  published

  end;
{End of declaration of TCustomPlot.}

  TPlot = class(TCustomPlot)
  published
    Property About;
{Displays the "About" dialog box for this component.}
    Property AxesProperties;
{Displays the "Axis Editor" dialog box for the FAxisList subcomponent.}
    Property Border;
{Manages the geometry of TCustomPlot: where the axes are, where they can go, etc.}
    Property ClickAndDragDelay;
{The delay (in milliseconds) before a clicked object becomes draggable.}

{$IFNDEF DELPHI1}
    Property CreatedBy;
{A string that is stored in the metafile description.}
    Property Description;
{A string that is stored in the metafile description.}
{$ENDIF}

    Property DisplayHistory;
{The width of the X Axis when in History mode.}
    Property DisplayMode;
{See TDisplayMode.}
    Property DefaultExtension;
{What is the default extension of Plot files ?}
    Property Editable;
{Are screen objects like axes Editable ?}
    Property FileName;
{This is the filename to which the data is saved, or opened from.}
{}
{If FileName is blank, and an OpenClick or SaveClick occurs, then the standard
 file dialog box appears to let the user pick a name.}
    property HighFont;
{The font for annotation of the Highs and Lows.}

{$IFDEF COMPILER4_UP}
    property Images;
{A publication of the Popup Menu's Images property}
{$ENDIF}

    Property Instructions;
{This is a hint-like message at the bottom of the graph.
 It used to be Caption - a string - but now has to be a
 stringlist to be fully Delphi-compatible (to avoid API calls).
 Remember Kylix !
 It disappears upon a MouseDown. See Font.}

    property Legend;
{The list of series with their line styles. This is a moveable, on-screen object.}
    Property Movable;
{Are screen objects like axes movable ?}
    Property Multiplicity;
{When the PlotType is ptMultiple, the series are grouped into multiples of Multiplicity.
 Otherwise ignored.}
    property MultiplePen;
{The pen to use for the verticle lines of ptMultiple PlotTypes (eg: High-Low).}

    Property NoYAxes;
{The total number of Y Axes (primary, secondary, tertiary, etc).}
    Property OutlineWidth;
{This is the width of the outline for screen objects like lines: axes and borders.}
    Property PlotType;
{What type of plot is this ?}
    Property PopupOptions;
{If true, then these popup menu items are visible.}
    Property PrintOrientation;
{Shall we print the graph in Landscape or Portrait mode ?}
    Property SaveOptions;
{Shall we save the data as Text or binary ?}
{}    
{Shall we also save the Plot properties when we save the data ?
 If we do, then we:
    1. Save the properties in a seperate file;
    2. Look for a properties file to open.}
    Property SeriesProperties;
{Displays the "Series Editor" dialog box for the TSeriesList subcomponent.}
    Property Title;
{The title of the graph, including its geometry, font and visibility.}
    Property XAxis;
{This is the X Axis. Every nice graph should have an X Axis.}
    Property YAxis;
{This is the Y Axis. Every nice graph should have at least one Y Axis.}
{}
{Each Series must know what Y Axes it is being plotted against:
 Primary (this one) or Secondary.}

{Events:}
    Property OnFileOpen;
{When a file is opened, the app can be notified of the new file name using this event.}

    Property OnFileClose;
{When a file is closed, the app can be notified of the new file name using this event.}

    Property OnHeader;
{When data is opened, this event passes the header information back to the "user".}

    Property OnHeaderRequest;
{When data is saved or copied, this event allows the user to add a header
 to the data.}

    Property OnHTMLHeaderRequest;
{When data is copied as HTML, this event allows the user to add a header
 to the data.}

    Property OnSelection;
    Property OnDualSelection;

{all the "extra" TPanel properties in D4 (alias "VCL Bloat"):}

{all the TPanel properties in D1:}
    property Align;    
    {property Alignment;}
    property BevelInner;
    property BevelOuter;
    property BevelWidth;
    property BorderWidth;
    property BorderStyle;
    {property Caption; - replaced by Instructions}
    property Color;
    property Ctl3D;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
{This is the font of the hint-like message at the bottom of the graph.}
    property Locked;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
{ Note: D1 to D4 were quite happy for:
        PopupMenu := FPlotPopUpMenu;
  FPlotPopUpMenu was then run by inherited MouseUp.
  However, D5 introduced TControl.WMContextMenu, which ran BEFORE MouseDown.
  This went spastic when it tried to Popup the PopupMenu.

  We have therefore returned to hiding FPlotPopUpMenu, and running it manually.
    property PopupMenu;}
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnResize;

    
{$IFDEF COMPILER2_UP}
{$ENDIF}

{$IFDEF COMPILER3_UP}
    property FullRepaint;
    property OnStartDrag;
{$ENDIF}

{$IFDEF COMPILER4_UP}
    property Anchors;
    // property AutoSize; - leads to bizzare behaviour
    property BiDiMode;
    property Constraints;
    property UseDockManager default True;
    property DockSite;
    property DragKind;
    property ParentBiDiMode;
    property OnCanResize;
    property OnConstrainedResize;
    property OnDockDrop;
    property OnDockOver;
    property OnEndDock;
    property OnGetSiteInfo;
    property OnStartDock;
    property OnUnDock;
{$ENDIF}

{$IFDEF COMPILER5_UP}
{$ENDIF}
  end;

{$IFDEF COMPILER3_UP}
resourcestring
  CompanyName = 'Chemware';
  ProductName = 'TCustomPlot';
  Version = 'Version 1.00';
  Copyright = 'Copyright  1999-2000 Chemware';
{$ENDIF}

implementation
{$IFDEF DELPHI1}
{$R strlist.res}
{$R Cursor16.res}
{$ELSE}
{$R strlist32.res}
{$R Cursor32.res}
{$ENDIF}

uses
  PlotMenu;

Constructor TPopupOptions.Create;
begin
  FMenu := [mnuFile,
    mnuEdit,
    mnuView,
    mnuCalc];
  FFile := [mnuNew,
    mnuOpen,
    mnuOverlayDiv, {convention: Name the Divs after the following menuitem}
    mnuOverlay,
    mnuClearOverlays,
    mnuSaveDiv,
    mnuSave,
    mnuSaveAs,
    mnuSaveImage,
    mnuPrintDiv,
    mnuPrint];
  FEdit := [mnuCopy,
    mnuCopyHTML,
    mnuCopySeries,
    mnuPaste,
    mnuDisplaceDiv,
    mnuDisplace,
    mnuResetDisplacement,
    mnuEditSeriesDiv,
    mnuNewSeries,
    mnuCloneSeries,
    mnuEditPoint,
    mnuEditSeries,
    mnuDeleteSeries,
    mnuAxisDiv,
    mnuNewY2Axis,
    mnuEditAxis,
    mnuDeleteY2Axis,
    mnuEditFontDiv,
    mnuEditFont,
    mnuEditPropertiesDiv,
    mnuEditProperties];
  FView := [mnuHide,
    mnuShowAll,
    mnuDisplayModeDiv,
    mnuDisplayMode,
    mnuLegend,
    mnuZoomDiv,
    mnuSetAsNormal,
    mnuNormalView,
    mnuManualZoom,
    mnuZoomIn,
    mnuZoomOut];
  FCalc := [mnuPosition,
    mnuNearestPoint,
    mnuCalcAverageDiv,
    mnuCalcAverage,
    mnuContractSeries,
    mnuContractAllSeries,
    mnuCubicSplineSeries,
    mnuHighs,
    mnuMovingAverage,
    mnuSmoothSeries,
    mnuSortSeries,
    mnuCalculusDiv,
    mnuDifferentiate,
    mnuIntegrate,
    mnuIntegral,
    mnuLineOfBestFitDiv,
    mnuLineOfBestFit,
    mnuTwoRegionLineOfBestFit]; 
end;

{TCustomPlot methods --------------------------------------------------------------}
procedure TCustomPlot.ShowAbout;
var
  Msg: string;
begin
  Msg := 'TPlot version ' + FloatToStrF(TPLOT_VERSION / 100, ffFixed, 5, 2) + #10;
  Msg := Msg + 'Copyright  2000 Mat Ballard' + #10;
  Msg := Msg + 'e-mail: mat.ballard@molsci.csiro.au' + #10;
  Msg := Msg + 'Build date: 11 Oct 2000';
  ShowMessage(Msg);
end;

procedure TCustomPlot.ShowSeries;
begin
  EditSeriesClick(Self);
end;

procedure TCustomPlot.ShowAxes;
begin
  EditAxisClick(Self);
end;

{Constructor and Destructor:-------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.Create
  Description: class constructor
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: initialize variables and create sub-components
 Known Issues:
 ------------------------------------------------------------------------------}
Constructor TCustomPlot.Create(
  AOwner:TComponent);
begin
{First call the ancestor:}
  inherited Create(AOwner);

{This is a typical size:}
  Height := 300;
  Width := 400;
{Users can edit some screen text:}
  FEditable := TRUE;
{Users can move screen objects:}
  FMovable := TRUE;

{High-Low is the default Multiple plot:}  
  FMultiPlicity := 2;

{No mouse timer as yet:}
  MouseTimer := TTimer.Create(Self);
  MouseTimer.Enabled := FALSE;
  MouseTimer.OnTimer := MouseTimeOut;

{no series has been selected yet:}
  pSeries := nil;
  TheSeries := -1;
  ClickedObjectType := soNone;
{Initialize the Instruction font:}
  Font.Name := 'Arial';
  Font.Size := 8;
  Font.Color := clRed;
  FDisplayHistory := 30;
  FirstOverlay := -1;
  FPlotMenu := nil;
  FPrintOrientation := poLandscape;
{ZeroScreenStuff everything to do with mice:}
  FScreenJob := sjNone;
  FOnSelection := nil;
  FOnDualSelection := nil;
  FSaveOptions := [soProperties];

  DefaultExtension := DEF_EXTENSION;
  FileExtensions[1] := 'csv';
  FileExtensions[2] := 'txt';
  FileExtensions[3] := '*';

  FPlotPopUpMenu := TPopupMenu.Create(Self);
{ see note above at 'property PopupMenu':
  PopUpMenu := FPlotPopUpMenu;}

  OpenFilterIndex := 1;
  SaveFilterIndex := 1;
  ImageFilterIndex := 1;

  BevelGap := 1;

  FInstructions := TStringList.Create;
{Take the hint:}
  FInstructions.Add('Click and Drag to move, Double-Click to edit,');
  FInstructions.Add('Right-Click to Act, and Shift-Click and Drag to Zoom In !');

  FFlashEdit := nil;
  
{create the list of all axes:}
  FAxisList := TList.Create;
{create all the Axes:}
  FXAxis := TAxis.Create(Self);
  FAxisList.Add(FXAxis);
  FYAxis := TAxis.Create(Self);
  FAxisList.Add(FYAxis);

{... and initialize the X-Axis:}
  FXAxis.AxisType := atPrimary;
  FXAxis.TickDirection := orRight;
  FXAxis.Title.Units := 's';
  FXAxis.Tag := Ord(soXAxis);
  FXAxis.Labels.Tag := Ord(soXAxisLabel);
  FXAxis.Title.Tag := Ord(soXAxisTitle);

{... and initialize the Y-Axis:}
  FYAxis.AxisType := atPrimary;
  FYAxis.Direction := dVertical;
  FYAxis.TickDirection := orLeft;
  FYAxis.Title.Orientation := orLeft;
  FYAxis.Title.Units := 'mV';
  FYAxis.Title.Caption := 'Y-Axis';
  FYAxis.Tag := Ord(soYAxis);
  FYAxis.Labels.Tag := Ord(soYAxisLabel);
  FYAxis.Title.Tag := Ord(soYAxisTitle);

{initialize the axis intercepts:}
  FXAxis.Intercept := 0;
  FYAxis.Intercept := 0;

{create the border of the graph:}
  FBorder := TBorder.Create(Self);
{set the graph dimensions:}
  FBorder.Left := 70;
  FBorder.Top := 40;
  FBorder.RightEx := Width;
  FBorder.BottomEx := Height;
  FBorder.RightGap := 50;
  FBorder.BottomGap := 80;

{create the user-clickable edges of the graph (for clicking and dragging):}
  LeftBorder := TRectangle.Create(Self);
  TopBorder := TRectangle.Create(Self);
  RightBorder := TRectangle.Create(Self);
  BottomBorder := TRectangle.Create(Self);
  LeftBorder.Name := 'Left Border';
  LeftBorder.Tag := Ord(soLeftBorder);
  TopBorder.Name := 'Top Border';
  TopBorder.Tag := Ord(soTopBorder);
  RightBorder.Name := 'Right Border';
  RightBorder.Tag := Ord(soRightBorder);
  BottomBorder.Name := 'Bottom Border';
  BottomBorder.Tag := Ord(soBottomBorder);

{create the graph title:}
  FTitle := TTitle.Create(Self);
{... and initialize it:}
  FTitle.Caption := ClassName;
  FTitle.Name := 'Plot Title';
  FTitle.Orientation := orLeft;
  FTitle.Font.Size := 12;
  FTitle.Tag := Ord(soTitle);

{The Result of a least-squares fit:}
  FResult := TCaption.Create(Self);
  FResult.Font.Size := 8;
{invisibilize it:}
  FResult.Top := -100;
  FResult.Tag := Ord(soResult);

{create the graph legend:}
  FLegend := TLegend.Create(Self);
{... and initialize it:}
  FLegend.Name := 'Legend';
  FLegend.Font.Size := 6;
  FLegend.Tag := Ord(soLegend);

{the font for annotation of the highs and lows:}
  FHighFont := TFont.Create;
  FHighFont.Name := 'Arial';
  FHighFont.Size := 6;

  FMultiplePen := TPen.Create;

{create the list of objects on the screen:}
  ScreenObjectList := TList.Create;
{load the list of screen objects:}
  ScreenObjectList.Add(Selection);
  ScreenObjectList.Add(FTitle);
  ScreenObjectList.Add(FLegend);
  ScreenObjectList.Add(FResult);
  ScreenObjectList.Add(FXAxis);
  ScreenObjectList.Add(FXAxis.Title);
  ScreenObjectList.Add(FXAxis.Labels);
  ScreenObjectList.Add(FYAxis);
  ScreenObjectList.Add(FYAxis.Title);
  ScreenObjectList.Add(FYAxis.Labels);
  ScreenObjectList.Add(LeftBorder);
  ScreenObjectList.Add(TopBorder);
  ScreenObjectList.Add(RightBorder);
  ScreenObjectList.Add(BottomBorder);

{create the object that is clicked and dragged:}
  Selection := TRectangle.Create(Self);

{create the list of data series:}
  FSeriesList := TSeriesList.Create(FAxisList);
{Do things with the series list:}

  SetPlotType(ptXY);

{set the user-clickable border thicknesses:}
  SetOutlineWidth(10);
{this fires SetAxisDimensions.}

  FPopupOptions := TPopupOptions.Create;
  CreateMenus;

{Set all the OnChanges:}
  FXAxis.OnChange := StyleChange;
  FYAxis.OnChange := StyleChange;
  FBorder.OnChange := StyleChange;
  FTitle.OnChange := StyleChange;
  FSeriesList.OnChange := StyleChange;
  FLegend.OnChange := StyleChange;

{load some cursors:}
{$IFDEF WIN}
  Screen.Cursors[crScope] := WinProcs.LoadCursor(HInstance, 'crScope');
  Screen.Cursors[crX] := WinProcs.LoadCursor(HInstance, 'crX');
{$ELSE}
{$ENDIF}
end;

{functions called by constructor ----------------------------------------------}

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.MakeDummyData
  Description: procedure to add two series, with sine wave data
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: 1. test TCustomPlot
               2. display capabilities in design mode.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.MakeDummyData(NoSteps: Integer);
var
  i: Integer;
  Amplitude,
  MidY,
  Phase,
  StepSize,
  X,
  Y: Single;
begin
{clean up first:}
  FSeriesList.ClearSeries;
{add two series, the second depending on the first:}
  FSeriesList.Add(-1);
  TSeries(FSeriesList.Items[0]).OnChange := StyleChange;
{this second series uses the X data of the first series:}
  FSeriesList.Add(0);
  TSeries(FSeriesList.Items[1]).OnChange := StyleChange;

{seed it:}
  Randomize;
{set the phase for trig functions:}
  Phase := Random;
{initialize: calculate the step size:}
  StepSize := (FXAxis.Max - FXAxis.Min) / NoSteps;
{... and the start:}
  X := FXaxis.Min;
{... and the size:}
  Amplitude := (FYAxis.Max - FYAxis.Min) / 2;
  MidY := (FYAxis.Max + FYAxis.Min) / 2;
  for i := 0 to NoSteps do
  begin
    Y := MidY + Amplitude * Sin(X + Phase) + Random;
{Don't fire any events, and don't adjust axes:}
    TSeries(FSeriesList.Items[0]).AddPoint(X, Y, FALSE, FALSE);
    Y := MidY + Amplitude * Cos(X + Phase) + Random;
    TSeries(FSeriesList.Items[1]).AddPoint(0, Y, FALSE, FALSE);
    X := X + StepSize;
  end;
  {TSeries(FSeriesList.Items[0]).GetBounds;}
  {TSeries(FSeriesList.Items[1]).GetBounds;}
  TSeries(FSeriesList.Items[0]).Visible := TRUE;
  TSeries(FSeriesList.Items[1]).Visible := TRUE;
  ZoomOutClick(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.CreateMenus
  Description: creates popup menus that are accessible by right-click
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 04/20/2000 by Mat Ballard
      Purpose: modularize user-interface code
 Known Issues: this was a bitch to get right !
 ------------------------------------------------------------------------------}
procedure TCustomPlot.CreateMenus;
var
  i, j,
  Index: Word;
{$IFDEF COMPILER4_UP}
  ImageIndex: Word;
{$ENDIF}
{This following is just a dummy matrix of menu items used to create sub-menus,
 then removed and the menuitems freed.}
  TempMenu: array [0..Ord(mnuCalc)] of array [0..0] of TMenuItem;
  TempMenuItem: TMenuItem;
  lpBuffer: array [0..255] of Char;
  StrResCaption,
  StrResName: String;
const
  NoMenusItems: array [0..Ord(mnuCalc)] of Integer =
    (Ord(mnuPrint),
     Ord(mnuEditProperties),
     Ord(mnuZoomOut),
     Ord(mnuTwoRegionLineOfBestFit));
begin
{create the sub-menus:}
  Index := 0;
{$IFDEF COMPILER4_UP}
  ImageIndex := 0;
{$ENDIF}
  for i := Ord(mnuFile) to Ord(mnuCalc) do
  begin
    Inc(Index);
{we create a temporary menu array to add the submenu, and then later remove it:}
    TempMenu[i][0] := TMenuItem.Create(Self);
{load the caption from resource:}
{$IFDEF WIN}
    WinProcs.LoadString(HINSTANCE, CAPTION_BASE + Index, lpBuffer, 256);
{$ELSE}
{$ENDIF}
    StrResCaption := StrPas(lpBuffer);
    StrResName := CleanString(StrResCaption, '&');
    StrResName := CleanString(StrResName, '.');
    StrResName := CleanString(StrResName, ' ');
    StrResName := StrResName + 'SubMenu';
    TempMenuItem := NewSubMenu(
      StrResCaption,
      0,
      StrResName,
      TempMenu[i]);
    TempMenuItem.Tag := Index + TAG_BASE;
    FPlotPopUpMenu.Items.Add(TempMenuItem);
  end;
  
{create the menus in each sub-menu:}
  for i := Ord(mnuFile) to Ord(mnuCalc) do
  begin
    for j := 0 to NoMenusItems[i] do
    begin
      Inc(Index);
{load the caption from resource:}
{$IFDEF WIN}
      WinProcs.LoadString(HINSTANCE, CAPTION_BASE + Index, lpBuffer, 256);
{$ELSE}
{$ENDIF}
      StrResCaption := StrPas(lpBuffer);
      TempMenuItem := TMenuItem.Create(Self);
      TempMenuItem.Caption := StrResCaption;
      TempMenuItem.Tag := Index + TAG_BASE;
      if (StrResCaption <> '-') then
      begin
{$IFDEF COMPILER4_UP}
        TempMenuItem.ImageIndex := ImageIndex;
        Inc(ImageIndex);
{$ENDIF}

{load the hint from resource}
{$IFDEF WIN}
        if (WinProcs.LoadString(HINSTANCE, HINT_BASE + Index, lpBuffer, 256) > 0) then
{$ELSE}
{$ENDIF}
        begin
          StrResCaption := StrPas(lpBuffer);
          TempMenuItem.Hint := StrResCaption;
        end;
      end; {not a line}
{add the TempMenuItem to the popup:}
      FPlotPopUpMenu.Items[i].Add(TempMenuItem);
    end; {j over menu items}
{remove the temporary menu array used to create the submenu:}
    if (FPlotPopUpMenu.Items[i].Items[0].Tag = 0) then
    begin
      FPlotPopUpMenu.Items[i].Remove(TempMenu[i][0]);
{then free it:}
      TempMenu[i][0].Free;
    end;
  end; {i over submenus}

{now set all the OnClick event handlers. what a bitch !}
  FPlotPopUpMenu.Items[Ord(mnuFile)].Items[Ord(mnuNew)].OnClick := NewClick;
  FPlotPopUpMenu.Items[Ord(mnuFile)].Items[Ord(mnuOpen)].OnClick := OpenClick;
  FPlotPopUpMenu.Items[Ord(mnuFile)].Items[Ord(mnuOverlayDiv)].OnClick := nil;
  FPlotPopUpMenu.Items[Ord(mnuFile)].Items[Ord(mnuOverlay)].OnClick := OverlayClick;
  FPlotPopUpMenu.Items[Ord(mnuFile)].Items[Ord(mnuClearOverlays)].OnClick := ClearOverlaysClick;
  FPlotPopUpMenu.Items[Ord(mnuFile)].Items[Ord(mnuSaveDiv)].OnClick := nil;
  FPlotPopUpMenu.Items[Ord(mnuFile)].Items[Ord(mnuSave)].OnClick := SaveClick;
  FPlotPopUpMenu.Items[Ord(mnuFile)].Items[Ord(mnuSaveAs)].OnClick := SaveAsClick;
  FPlotPopUpMenu.Items[Ord(mnuFile)].Items[Ord(mnuSaveImage)].OnClick := SaveImageClick;
  FPlotPopUpMenu.Items[Ord(mnuFile)].Items[Ord(mnuPrintDiv)].OnClick := nil;
  FPlotPopUpMenu.Items[Ord(mnuFile)].Items[Ord(mnuPrint)].OnClick := PrintClick;

  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuCopy)].OnClick := CopyClick;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuCopyHTML)].OnClick := CopyHTMLClick;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuCopySeries)].OnClick := CopySeriesClick;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuPaste)].OnClick := PasteClick;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuDisplaceDiv)].OnClick := nil;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuDisplace)].OnClick := DisplaceClick;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuResetDisplacement)].OnClick := ResetDisplacementClick;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuEditSeriesDiv)].OnClick := nil;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuCloneSeries)].OnClick := CloneSeriesClick;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuEditPoint)].OnClick := EditPointClick;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuEditSeries)].OnClick := EditSeriesClick;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuDeleteSeries)].OnClick := DeleteSeriesClick;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuAxisDiv)].OnClick := nil;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuNewY2Axis)].OnClick := AddAxisClick;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuEditAxis)].OnClick := EditAxisClick;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuDeleteY2Axis)].OnClick := DeleteAxisClick;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuEditFontDiv)].OnClick := nil;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuEditFont)].OnClick := EditFontClick;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuEditPropertiesDiv)].OnClick := nil;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuEditProperties)].OnClick := PropertiesClick;

  FPlotPopUpMenu.Items[Ord(mnuView)].Items[Ord(mnuHide)].OnClick := HideClick;
  FPlotPopUpMenu.Items[Ord(mnuView)].Items[Ord(mnuShowAll)].OnClick := ShowAllClick;
  FPlotPopUpMenu.Items[Ord(mnuView)].Items[Ord(mnuZoomDiv)].OnClick := nil;
  FPlotPopUpMenu.Items[Ord(mnuView)].Items[Ord(mnuSetAsNormal)].OnClick := SetAsNormalClick;
  FPlotPopUpMenu.Items[Ord(mnuView)].Items[Ord(mnuNormalView)].OnClick := NormalViewClick;
  FPlotPopUpMenu.Items[Ord(mnuView)].Items[Ord(mnuNormalView)].Enabled := FALSE;
  FPlotPopUpMenu.Items[Ord(mnuView)].Items[Ord(mnuManualZoom)].OnClick := ManualZoomClick;
  FPlotPopUpMenu.Items[Ord(mnuView)].Items[Ord(mnuZoomIn)].OnClick := ZoomInClick;
  FPlotPopUpMenu.Items[Ord(mnuView)].Items[Ord(mnuZoomOut)].OnClick := ZoomOutClick;

  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuPosition)].OnClick := PositionClick;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuNearestPoint)].OnClick := NearestPointClick;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuCalcAverageDiv)].OnClick := nil;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuCalcAverage)].OnClick := AverageClick;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuContractSeries)].OnClick := ContractSeriesClick;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuContractAllSeries)].OnClick := ContractAllSeriesClick;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuHighs)].OnClick := HighsClick;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuMovingAverage)].OnClick := MovingAverageClick;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuCubicSplineSeries)].OnClick := SplineClick;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuSmoothSeries)].OnClick := SmoothSeriesClick;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuSortSeries)].OnClick := SortClick;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuCalculusDiv)].OnClick := nil;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuDifferentiate)].OnClick := DifferentiateClick;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuIntegrate)].OnClick := IntegrateClick;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuIntegral)].OnClick := IntegralClick;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuLineOfBestFitDiv)].OnClick := nil;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuLineOfBestFit)].OnClick := LineBestFitClick;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuTwoRegionLineOfBestFit)].OnClick := TwoRegionLineBestFitClick;

  FPlotPopUpMenu.Items[Ord(mnuView)].Items[Ord(mnuDisplayMode)].OnClick := ModeClick;

  FPlotPopUpMenu.Items[Ord(mnuView)].Items[Ord(mnuLegend)].OnClick := LegendClick;

  WhichPopUpMenu := TPopUpMenu.Create(Self);
  for i := 0 to 1 do begin
    WhichPopUpItems[i] := TMenuItem.Create(Self);
    WhichPopUpItems[i].Tag := i;
    WhichPopUpMenu.Items.Add(WhichPopUpItems[i]);
  end;
{Note: we set the FInstructions just before we display this popup:}
  {WhichPopUpItems[0].Caption := 'ClickedObjectType';}
  {WhichPopUpItems[1].Caption := 'SecondClickedObjectType';}
  WhichPopUpItems[0].OnClick := MoveTheClickedObjectClick;
  WhichPopUpItems[1].OnClick := MoveSecondClickedObjectClick;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.Destroy
  Description: standard destructor
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: free sub-components
 Known Issues:
 ------------------------------------------------------------------------------}
Destructor TCustomPlot.Destroy;
var
  i: Integer;
begin
{nil out the events:}
  FOnFileOpen := nil;
  FOnFileClose := nil;
  FOnHeader := nil;
  FOnHeaderRequest := nil;
  FOnHTMLHeaderRequest := nil;
  FOnSelection := nil;
  FOnDualSelection := nil;

{Clear the SeriesList, thereby provoking a file save if required:}
  Clear(FALSE);

{Free all the axes:}
  for i := FAxisList.Count-1 downto 0 do
  begin
    TAxis(FAxisList.Items[i]).Free;
  end;

  {FOnChange := nil;}

{Put your de-allocation, etc, here:}
  FInstructions.Free;
  FResult.Free;
  FLegend.Free;
  FTitle.Free;
  FSeriesList.Free;
  FAxisList.Free;
  FBorder.Free;
  LeftBorder.Free;;
  TopBorder.Free;;
  RightBorder.Free;;
  BottomBorder.Free;;
  ScreenObjectList.Free;
  Selection.Free;
  FHighFont.Free;
  FMultiplePen.Free;
  MouseTimer.Free;

  FPopupOptions.Free;
  FPlotPopUpMenu.Free;
  WhichPopUpMenu.Free;


{then call ancestor:}
  inherited Destroy;
end;

{End Constructor and Destructor: ----------------------------------------------}

{Get functions ----------------------------------------------------------------}
{------------------------------------------------------------------------------
     Function: TCustomPlot.GetSeries
  Description: private property Get function
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: interface to Series property, which is the default property
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.GetSeries(
  Index: Integer): TSeries;
begin
  if ((Index < 0) or (Index >= FSeriesList.Count)) then raise
    ERangeError.CreateFmt('There is no series %d: valid indices are from 0 to %d',
      [Index, FSeriesList.Count-1]);
  GetSeries := TSeries(FSeriesList.Items[Index]);
end;

{Set procedures ---------------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetAxisDimensions
  Description: geometry manager
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets up the border, axes and Title position
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetAxisDimensions;
var
  i: Integer;
  TheRect: TRect;
  pThisYAxis: TAxis;
begin
  IgnoreChanges := TRUE;

{do the simple bits, where the border sets the axis lengths:}
  FXAxis.Left := FBorder.Left;
  FXAxis.Right := FBorder.Right;
  for i := 1 to FAxisList.Count-1 do
  begin
    TAxis(FAxisList[i]).Top := FBorder.Top;
    TAxis(FAxisList[i]).Bottom := FBorder.Bottom;
  end;

{Limit the X Axis intercept:}
  if (FXAxis.AutoScale) then
  begin
    if ((FYAxis.Min <= 0) and (0 <= FYAxis.Max)) then
      FXAxis.Intercept := 0
     else
      FXAxis.Intercept := FYAxis.Min;
  end
  else
  begin
    if (FXAxis.Intercept < FYAxis.Min) then
      FXAxis.Intercept := FYAxis.Min;
    if (FXAxis.Intercept > FYAxis.Max) then
      FXAxis.Intercept := FYAxis.Max;
  end;

{Limit the X Axis intercept:}
  if (FYAxis.AutoScale) then
  begin
    if ((FXAxis.Min <= 0) and (0 <= FXAxis.Max)) then
      FYAxis.Intercept := 0
     else
      FYAxis.Intercept := FXAxis.Min;
  end
  else
  begin
    if (FYAxis.Intercept < FXAxis.Min) then
      FYAxis.Intercept := FXAxis.Min;
    if (FYAxis.Intercept > FXAxis.Max) then
      FYAxis.Intercept := FXAxis.Max;
  end;

{Limit the secondary Y Axes intercepts:}
  for i := 2 to FAxisList.Count-1 do
  begin
    pThisYAxis := TAxis(FAxisList[i]);
    if (not pThisYAxis.AutoScale) then
    begin
      pThisYAxis.Intercept := FXAxis.Max +
        (i-2) * Width;
    end;

    if (pThisYAxis.AxisType = atTertiary) then
    begin
      if (pThisYAxis.Intercept < FXAxis.XofF(1)) then
        pThisYAxis.Intercept := FXAxis.XofF(1);
      if (pThisYAxis.Intercept > FXAxis.XofF(Width-2)) then
        pThisYAxis.Intercept := FXAxis.XofF(Width-2);
    end
    else
    begin
{ant Secondary axis is limited to the borders:}
      if (pThisYAxis.Intercept < FXAxis.Min) then
        pThisYAxis.Intercept := FXAxis.Min;
      if (pThisYAxis.Intercept > FXAxis.Max) then
        pThisYAxis.Intercept := FXAxis.Max;
    end;
  end;

{Set the screen positions based on the Intercepts:}                            
  FXAxis.MidY := FYAxis.FofY(FXAxis.Intercept);
  for i := 1 to FAxisList.Count-1 do
  begin
    pThisYAxis := TAxis(FAxisList[i]);
    pThisYAxis.MidX := FXAxis.FofX(pThisYAxis.Intercept);
  end;
  
{do the borders for click-and-drag purposes:}
  LeftBorder.Top := FBorder.Top;
  LeftBorder.Bottom := FBorder.Bottom;
  LeftBorder.MidX := FBorder.Left;

  RightBorder.Top := FBorder.Top;
  RightBorder.Bottom := FBorder.Bottom;
  RightBorder.MidX := FBorder.Right;

  TopBorder.Left := FBorder.Left;
  TopBorder.Right := FBorder.Right;
  TopBorder.MidY := FBorder.Top;

  BottomBorder.Left := FBorder.Left;
  BottomBorder.Right := FBorder.Right;
  BottomBorder.MidY := FBorder.Bottom;

{set up the title envelope:}
  TheRect.Left := FBorder.Left;
  TheRect.Right := FBorder.Right;
  TheRect.Top := BevelGap + FTitle.Height;
  TheRect.Bottom := Height - BevelGap - FTitle.Height;
  FTitle.Envelope := TheRect;

  IgnoreChanges := FALSE;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetClickAndDragDelay
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the ClickAndDragDelay Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetClickAndDragDelay(Value: Integer);
begin
  MouseTimer.Interval := Value;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetDefaultExtension
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the DefaultExtension for TPlot files
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetDefaultExtension(
  Value: String);
begin
  FDefaultExtension := Value;
  FileExtensions[0] := Value;
  FileTypes :=
    'Plot Files|*.' + Value
    + '|Comma Sep Var Files|*.csv'
    + '|Text Files|*.txt'
    + '|All Files|*.*';
  end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetDisplayMode
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the DisplayMode property, which is how
               graphs are updated when more data is Added
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetDisplayMode(
  Value: TDisplayMode);
begin
  if (FDisplayMode = Value) then exit;

  SetDisplayModeHistory(FDisplayHistory, Value);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetDisplayModeHistory
  Description: adjusts axes and sets DisplayMode and History properties
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: DisplayMode and History must be set and the graph updated simultaneously.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetDisplayModeHistory(
  HistoryValue: Single;
  ScalingValue: TDisplayMode);
begin
  if (ScalingValue = dmHistory) then
  begin
{we are changing to History from normal behaviour,
 or we are in History mode:}
    FXAxis.Min := -HistoryValue;
    FXAxis.Max := 0;
    FYAxis.Intercept := -HistoryValue;
  end
  else if (FDisplayMode = dmHistory) then
  begin
{We are changing from History to normal behaviour.
 We therefore need to reset the X Axis dimensions:}
    if (ScalingValue = dmRun) then
      FXAxis.Max := 1.5 * FSeriesList.Xmax
    else
      FXAxis.Max := FSeriesList.Xmax;
    FXAxis.Min := FSeriesList.Xmin;
    FYAxis.Intercept := FXAxis.Min;
  end;
  FDisplayHistory := HistoryValue;
  FDisplayMode := ScalingValue;
  Refresh;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetFileName
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets and parses the FileName property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetFileName(
  Value: String);
begin
  if (FFileName = Value) then exit;

  FFileName := Value;
  if (Length(FFileName) > 0) then
  begin
    PropsFileName := GetFileDriveDir;
    PropsFileName := PropsFileName + GetFileRoot + '.';
    PropsFileName := PropsFileName + PROP_EXTENSION;
  end
   else
    PropsFileName := '';
end;

{------------------------------------------------------------------------------
    Functions: TCustomPlot.GetFileXXX
  Description: gets various file parameters from the name
       Author: Mat Ballard
 Date created: 08/10/2000
Date modified: 08/10/2000 by Mat Ballard
      Purpose: file management
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.GetFileExtension: String; {csv}
var
  FileExtension: String;
begin
  FileExtension := LowerCase(ExtractFileExt(FFileName));
  if (Pos('.', FileExtension) = 1) then
    FileExtension := Copy(FileExtension, 2, Length(FileExtension));
  GetFileExtension := FileExtension;
end;

function TCustomPlot.GetFileDriveDir: String;  {D:\Data\Delphi\Plot}
var
  FileDriveDir: String;
begin
  FileDriveDir := ExtractFilePath(FFileName);
  if (Length(FileDriveDir) = 0) then
    FileDriveDir := GetCurrentDir;
  GetFileDriveDir := FileDriveDir;
end;

function TCustomPlot.GetFileRoot: String;      {Test3}
var
  Ext,
  FileRoot: String;
  i: Integer;
begin
  Ext := GetFileExtension;
  if (Length(Ext) > 0) then
  begin
    FileRoot := ExtractFileName(FFileName);
    i := Pos(Ext, FileRoot);
    FileRoot := Copy(FileRoot, 1, i-2);
  end
  else
    FileRoot := FFileName;
  GetFileRoot := FileRoot;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.GetFilterIndex
  Description: gets the file filter index from the extension
       Author: Mat Ballard
 Date created: 08/10/2000
Date modified: 08/10/2000 by Mat Ballard
      Purpose: file management
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.GetFilterIndex(
  Ext: String): Integer;
var
  i: Integer;
begin
{the default filterindex is actually '*'}
  GetFilterIndex := 4;
  for i := 0 to 3 do
  begin
    if (LowerCase(Ext) = FileExtensions[i]) then
    begin
      GetFilterIndex := i+1;
      break;
    end;
  end;
end;
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetHistory
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the History property: which is how far back
               a History graph goes
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetHistory(
  Value: Single);
begin
  if (FDisplayHistory = Value) then exit;

  SetDisplayModeHistory(Value, FDisplayMode);
end;

{$IFDEF COMPILER4_UP}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetImages
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 10/09/1999
Date modified: 10/09/2000 by Mat Ballard
      Purpose: sets the Images property
 Known Issues: TMenu.Images is of type TCustomImageList, which lurks in unit
               imglist; however, BC++'s DCLSTD35 contains an imglist, and so
               we get a namespace collision.
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetImages(Value: TImageList);
begin
  FPlotPopUpMenu.Images := Value;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.GetImages
  Description: private property Set procedure
       Author: Mat Ballard
 Date created: 10/09/1999
Date modified: 10/09/2000 by Mat Ballard
      Purpose: sets the Images property
 Known Issues: TMenu.Images is of type TCustomImageList, which lurks in unit
               imglist; however, BC++'s DCLSTD35 contains an imglist, and so
               we get a namespace collision.
 ------------------------------------------------------------------------------}
function TCustomPlot.GetImages: TImageList;
begin
  GetImages := TImageList(FPlotPopUpMenu.Images);
end;
{$ENDIF}

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetMetafileDescription
  Description: sets the CreatedBy and Description properties
               if they are not yet set
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: fully utilise enhanced metafile capabilities to
               put keywords into WMF
 Known Issues:
 ------------------------------------------------------------------------------}
{$IFNDEF DELPHI1}
procedure TCustomPlot.SetMetafileDescription;
var
  i: Integer;
begin
  if (Length(FCreatedBy) = 0) then
    FCreatedBy := 'TCustomPlot by Chemware';

  if (Length(FDescription) = 0) then
  begin
    FDescription := FTitle.Caption;
    for i := 0 to FSeriesList.Count-1 do
    begin
      FDescription := FDescription + ', ' + TSeries(FSeriesList.Items[i]).Name;
    end;
  end;
end;
{$ENDIF}

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetMultiplicity
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Multiplicity property
 Known Issues: see also: PlotTpe property
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetMultiplicity(Value: Byte);
begin
  FMultiplicity := Value;
  if (FPlotType = ptMultiple) then Refresh;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetMultiplePen
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the Multiplicity property
 Known Issues: see also: PlotTpe property
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetMultiplePen(Value: TPen);
begin
  FMultiplePen.Assign(Value);
  if (FPlotType = ptMultiple) then Refresh;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetOutlineWidth
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the OutlineWidth property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetOutlineWidth(
  Value: Integer);
begin
  if (FOutlineWidth = Value) then exit;

{Set border widths:}
  FOutlineWidth := Value;
  LeftBorder.Height := FOutlineWidth;
  LeftBorder.Width := FOutlineWidth;
  TopBorder.Height := FOutlineWidth;
  TopBorder.Width := FOutlineWidth;
  RightBorder.Height := FOutlineWidth;
  RightBorder.Width := FOutlineWidth;
  BottomBorder.Height := FOutlineWidth;
  BottomBorder.Width := FOutlineWidth;

{Set axis widths:}
  FXAxis.Height := FOutlineWidth;
  FYAxis.Width := FOutlineWidth;

  SetAxisDimensions;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetOnSelection
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the OnSelection event
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetOnSelection(Value: TOnSelectionEvent);
begin
  FOnSelection := Value;
  ScreenJob := sjSelection;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetOnDualSelection
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the OnDualSelection event
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetOnDualSelection(Value: TOnDualSelectionEvent);
begin
  FOnDualSelection := Value;
  ScreenJob := sjDualSelection1;
end;


{The painting/drawing methods -----------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.Paint
  Description: painting the graph
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: paints the background, border, then draws the graph: NOT called by graphics and printer.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.Paint;
{const
  Alignments: array[TAlignment] of Longint = (DT_LEFT, DT_RIGHT, DT_CENTER);}
var
  Rect: TRect;
  TopColor,
  BottomColor: TColor;

  procedure AdjustColors(Bevel: TPanelBevel);
  begin
    TopColor := clBtnHighlight;
    if Bevel = bvLowered then TopColor := clBtnShadow;
    BottomColor := clBtnShadow;
    if Bevel = bvLowered then BottomColor := clBtnHighlight;
  end;

begin
{$IFDEF DELPHI3_UP}
  Assert(Canvas <> nil, 'TCustomPlot.Draw: ACanvas is nil !');
{$ENDIF}

  Canvas.Pen.Mode := pmCopy;
  Canvas.Pen.Style := psSolid;

  BevelGap := 0;
  Rect := GetClientRect;
  if BevelOuter <> bvNone then
  begin
    Inc(BevelGap);
    AdjustColors(BevelOuter);
    Frame3D(Canvas, Rect, TopColor, BottomColor, BevelWidth);
  end;

  Frame3D(Canvas, Rect, Color, Color, BorderWidth);

  if BevelInner <> bvNone then
  begin
    Inc(BevelGap);
    AdjustColors(BevelInner);
    Frame3D(Canvas, Rect, TopColor, BottomColor, BevelWidth);
  end;
  BevelGap := BevelGap * BevelWidth;

  with Canvas do
  begin
    Brush.Color := Color;
    FillRect(Rect);
    Brush.Style := bsClear;
  end;

  Draw(Canvas);

  with Canvas do
  begin
    Brush.Color := Color;
    Brush.Style := bsClear;
  end;
{The Instructions are usually an instruction to the user.
 As such, it does not need to be copied or printed,
 so it is placed here, rather than in the "Draw" method:}
  DrawInstructions;

{outline all the screen objects: test of geometry logic for debugging:
 just add a curly bracket here to put it back in:
  if (csDesigning in ComponentState) then
  begin
    Canvas.Pen.Color := clYellow;
    for i := 0 to ScreenObjectList.Count-1 do
    begin
      Canvas.Rectangle(
        TRectangle(ScreenObjectList.Items[i]).Left,
        TRectangle(ScreenObjectList.Items[i]).Top,
        TRectangle(ScreenObjectList.Items[i]).Right,
        TRectangle(ScreenObjectList.Items[i]).Bottom);
    end;
    for i := 0 to FSeriesList.Count-1 do
    begin
      pSeries := TSeries(FSeriesList.Items[i]);
      pSeries.GenerateOutline(FOutlineWidth);
      pSeries.OutlineSeries(Canvas);
    end;
  end;}
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DrawInstructions
  Description: draws the instructions
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: tell the user what to do
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DrawInstructions;
var
  FontHeight,
  iX,
  iY,
  i: Integer;
begin
{The Instructions are usually an instruction to the user.
 As such, it does not need to be copied or printed,
 so it is placed here, rather than in the "Draw" method:}
  if (FInstructions.Count > 0) then
  begin
    Canvas.Font.Assign(Font);
    FontHeight := Canvas.TextHeight('Wp');
{Adjust the Position appropriately:}
    iX := BevelGap + 5;
    iY := Height - BevelGap - FontHeight;
{how many lines ?}
    for i := FInstructions.Count-1 downto 0 do
    begin
{Output the text:}
      Canvas.TextOut(iX, iY, FInstructions[i]);
      Dec(iY, FontHeight);
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.Draw
  Description: painting the graph
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: draws the graph on a canvas: graphics and printers call this procedure directly
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.Draw(
  ACanvas: TCanvas);
var
  FontHeight,
  FontWidth,
  i,
  iX,
  iY: Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TCustomPlot.Draw: ACanvas is nil !');
{$ENDIF}

  IgnoreChanges := TRUE;

  FTitle.Draw(ACanvas);

  if (FResult.Visible) and (Length(FResult.Caption) > 0) then
  begin
    ACanvas.Font.Assign(FResult.Font);
    FontHeight := Abs(ACanvas.Font.Height);
    FontWidth := ACanvas.TextWidth(FResult.Caption);
{calculate the caption dimensions:}
    {FResult.Top := Selection.Top;
    FResult.Left := Selection.Left;}
    FResult.Right := FResult.Left + FontWidth;
    FResult.Bottom := FResult.Top + FontHeight;
{output text to screen:}
    ACanvas.TextOut(FResult.Left, FResult.Top, FResult.Caption);
{now draw the line itself:}
{Y = Intercept + Slope * X  <=> X = (Y - Intercept) / Slope}
    ACanvas.Pen.Style := psDot;
    ACanvas.Pen.Color := Font.Color;
    iX := FXAxis.FofX(XAxis.Min);
    iY := FYAxis.FofY(Intercept + Slope * XAxis.Min);
    if (iY < Border.Top) then
    begin
      iY := Border.Top;
      iX := FXAxis.FofX((FYAxis.YofF(iY) - Intercept)/Slope);
    end
    else if (iY > Border.Bottom) then
    begin
      iY := Border.Bottom;
      iX := FXAxis.FofX((FYAxis.YofF(iY) - Intercept)/Slope);
    end;
    ACanvas.MoveTo(iX, iY);
    iX := FXAxis.FofX(XAxis.Max);
    iY := FYAxis.FofY(Intercept + Slope * XAxis.Max);
    if (iY < Border.Top) then
    begin
      iY := Border.Top;
      iX := FXAxis.FofX((FYAxis.YofF(iY) - Intercept)/Slope);
    end
    else if (iY > Border.Bottom) then
    begin
      iY := Border.Bottom;
      iX := FXAxis.FofX((FYAxis.YofF(iY) - Intercept)/Slope);
    end;
    Canvas.LineTo(iX, iY);
  end; {Result Visible}

  for i := 0 to FAxisList.Count-1 do
    TAxis(FAxisList[i]).Draw(ACanvas);

  DrawLegend(ACanvas);

  ACanvas.Font.Assign(FHighFont);
  case FPlotType of
    ptXY:
      begin
        if (FDisplayMode < dmHistory) then
          FSeriesList.Draw(ACanvas)
         else
          FSeriesList.DrawHistory(ACanvas, FDisplayHistory);
      end;
    ptMultiple:
      begin
        if (FDisplayMode < dmHistory) then
        begin
          FSeriesList.Draw(ACanvas);
          ACanvas.Pen.Assign(FMultiplePen);
          FSeriesList.DrawMultiple(ACanvas, FMultiplicity);
        end
        else
        begin
          FSeriesList.DrawHistory(ACanvas, FDisplayHistory);
          ACanvas.Pen.Assign(FMultiplePen);
          FSeriesList.DrawHistoryMultiple(ACanvas, FMultiplicity);
        end;
      end;
    ptColumn:
      begin
        if (FDisplayMode < dmHistory) then
          FSeriesList.DrawColumns(ACanvas);
         {else
          FSeriesList.DrawHistory(ACanvas, FDisplayHistory);}
      end;
  end;

  IgnoreChanges := FALSE;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DrawLegend
  Description: painting the graph
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: draws the Legend, a list of series, and their lineshapes
     Comments: we draw the Legend here, rather than in TLegend, because we need access to the Series Names, Symbols and Pens. 
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DrawLegend(
  ACanvas: TCanvas);
var
  Chars,
  i,
  iX,
  iMaxChars,
  MaxChars,
  LineWidth,
  LineY,
  TextY: Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TCustomPlot.DrawLegend: ACanvas is nil !');
{$ENDIF}

  if (not FLegend.Visible) then exit;
  if (FSeriesList.Count = 0) then exit;

  FLegend.Count := FSeriesList.Count;

  ACanvas.Font.Assign(FLegend.Font);

  MaxChars := 0;
  iMaxChars := -1;
  for i := 0 to FSeriesList.Count-1 do
  begin
    Chars := Length(TSeries(FSeriesList.Items[i]).Name);
    if (MaxChars < Chars) then
    begin
      MaxChars := Chars;
      iMaxChars := i;
    end;
  end;
  if (iMaxChars < 0) then exit;

{Allow for symbols and lines:}
  FLegend.StringWidth := ACanvas.TextWidth(
    TSeries(FSeriesList.Items[iMaxChars]).Name);
  LineWidth := FLegend.StringWidth div 3;
  FLegend.FontHeight := ACanvas.TextHeight('Ap');

  LineY := FLegend.Top + FLegend.FontHeight div 2;
  TextY := FLegend.Top;

  if (FLegend.Direction = dVertical) then
  begin
    for i := 0 to FSeriesList.Count-1 do
    begin
      ACanvas.Pen.Assign(TSeries(FSeriesList.Items[i]).Pen);
      TSeries(FSeriesList.Items[i]).DrawSymbol(ACanvas, FLegend.Left, LineY);
      ACanvas.LineTo(FLegend.Left + LineWidth, LineY);
  {output text to screen:}
      ACanvas.Font.Color := ACanvas.Pen.Color;
      ACanvas.TextOut(FLegend.Left + FLegend.SymbolWidth, TextY,
        TSeries(FSeriesList.Items[i]).Name);
      Inc(LineY, FLegend.FontHeight);
      Inc(TextY, FLegend.FontHeight);
    end;
  end
  else {Horizontal}
  begin
{Note: in Horizontal mode, the size of each series name is:}
{<---LineWidth---><---StringWidth---><-LineWidth->}
{<-Symbol + Line-><-Name of Series--><--Space---->}
    LineY := FLegend.Top + FLegend.FontHeight div 2;
    TextY := FLegend.Top;
    iX := FLegend.Left;
    for i := 0 to FSeriesList.Count-1 do
    begin
      ACanvas.Pen.Assign(TSeries(FSeriesList.Items[i]).Pen);
      TSeries(FSeriesList.Items[i]).DrawSymbol(ACanvas, iX, LineY);
      Inc(iX, LineWidth);
      ACanvas.LineTo(iX-1, LineY);
  {output text to screen:}
      ACanvas.Font.Color := ACanvas.Pen.Color;
      ACanvas.TextOut(iX, TextY,
        TSeries(FSeriesList.Items[i]).Name);
      iX := iX + FLegend.StringWidth + LineWidth;
    end; {for}
    FLegend.Width := iX - FLegend.Left;
    FLegend.Height := FLegend.FontHeight;
  end; {if}
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.Trace
  Description: This traces all series: useful for Oscilloscopes
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Draws all Series in erasable mode
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.Trace;
var
  i: Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(Canvas <> nil, 'TCustomPlot.Trace: Canvas is nil !');
{$ENDIF}

  for i := 0 to FSeriesList.Count-1 do
  begin
    TSeries(FSeriesList.Items[i]).Trace(Canvas);
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.Resize
  Description: overrides ancestor's ReSize
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: responds to a resize of the Plot
 Known Issues:
 ------------------------------------------------------------------------------}

procedure TCustomPlot.Resize;
begin
  FBorder.RightEx := Width;
  FBorder.BottomEx := Height;
  SetAxisDimensions;
  inherited Resize;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.StyleChange
  Description: target of all of the sub-component OnChange events
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: responds to changes in sub-components
 Known Issues: get up to 3 screen re-draws
 ------------------------------------------------------------------------------}
procedure TCustomPlot.StyleChange(
  Sender: TObject);
begin
  if (IgnoreChanges) then exit;

  SetAxisDimensions;
  Refresh;
end;

{Mousey stuff -----------------------------------------------------------------}

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DblClick
  Description: overrides ancestor's DblClick
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: activates in-place editing of titles
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DblClick;
var
  i: Integer;
  TheRect: TRect;
  TheTitle: TTitle;
  TheRight: Integer;
begin
{get rid of the mouse moving timer:}
  MouseTimer.Enabled := FALSE;

  if (FEditable) then
  begin
    if ((ClickedObjectType = soTitle) or
        (ClickedObjectType = soXAxisTitle) or
        (ClickedObjectType = soYAxisTitle)) then
    begin
      FScreenJob := sjFlashEdit;
      TheTitle := TTitle(pClickedObject);
{create the in-place editor:}
      CreateFlashEditor;
{... and initialize it:}
      FFlashEdit.Text := TheTitle.FullCaption;
      FFlashEdit.Height := TheTitle.Height + 10;
      FFlashEdit.Width  := 2 * TheTitle.Width;
      if (FFlashEdit.Height > FFlashEdit.Width) then
      begin
{height > width, so it is a vertical caption:}
        i := FFlashEdit.Height;
        FFlashEdit.Height := FFlashEdit.Width;
        FFlashEdit.Width := i;
      end;
      FFlashEdit.Top := TheTitle.Top;
{Have to check that the edit box is on-screen:}
      TheRight := TheTitle.Left + FFlashEdit.Width;
      if (TheRight > Width) then
        FFlashEdit.Left := TheTitle.Right - FFlashEdit.Width
      else
        FFlashEdit.Left := TheTitle.Left;

      FFlashEdit.Tag := Ord(ClickedObjectType);
      FFlashEdit.Font.Assign(TheTitle.Font);

      FFlashEdit.Visible := TRUE;
      FFlashEdit.SetFocus;
    end {Title or axis caption}
    else if (ClickedObjectType = soLegend) then
    begin
      FScreenJob := sjFlashEdit;
{create the in-place editor:}
      CreateFlashEditor;
      TheSeries := FLegend.GetHit(Selection.Left, Selection.Top, TheRect);
      FFlashEdit.Height := FLegend.FontHeight + 2;
      FFlashEdit.Width  := FLegend.SymbolWidth + FLegend.StringWidth;
      FFlashEdit.Left := TheRect.Left;
      FFlashEdit.Top := TheRect.Top - 1;
      FFlashEdit.Tag := Ord(soLegend);
{we use HelpContext because MouseDown is called after DblClick, which nukes TheSeries:}
      FFlashEdit.HelpContext := TheSeries;
      FFlashEdit.Font.Assign(FLegend.Font);
      FFlashEdit.Text := TSeries(FSeriesList[TheSeries]).Name;
      FFlashEdit.Visible := TRUE;
      FFlashEdit.SetFocus;
    end; {Legend}
  end; {editable}

  inherited DblClick;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.CreateFlashEditor
  Description: Creates the FlashEdit in-place editor
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: caption management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.CreateFlashEditor;
{create the in-place editor:}
begin
  FFlashEdit := TEdit.Create(Self);
  FFlashEdit.Parent := Self;
  FFlashEdit.OnKeyDown := FlashEditKeyDown;
  FFlashEdit.OnExit := FlashEditExit;
  FFlashEdit.Hint := 'Type in the new caption, then press <-Enter, or Escape to cancel.';
  FFlashEdit.ShowHint := TRUE;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.FlashEditKeyDown
  Description: KeyDown event handler of FFlashEdit in-place editor
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Cancel the FFlashEditor if Esc pressed, or save the changed Title.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.FlashEditKeyDown(
  Sender: TObject;
  var Key: Word;
  Shift: TShiftState);
begin
  if (Key = VK_ESCAPE) then
  begin
    FFlashEdit.Visible := FALSE;
    Key := 0;
  end;

  if (Key = VK_RETURN) then
  begin
{this will throw an exception if Tag is not a valid TObjectType:}
    case TObjectType(FFlashEdit.Tag) of
      soTitle: FTitle.Caption := FFlashEdit.Text;
      {soXAxis, soYAxis}
      soXAxisTitle: FXAxis.Title.Caption := FFlashEdit.Text;
      soYAxisTitle: TTitle(pClickedObject).Caption := FFlashEdit.Text;
      {soXAxisLabel, soYAxisLabel, soYAxis2Label,
      soLeftBorder, soTopBorder, soRightBorder, soBottomBorder}
{we use HelpContext because MouseDown is called after DblClick, which nukes TheSeries:}
      soLegend: TSeries(FSeriesList[FFlashEdit.HelpContext]).Name := FFlashEdit.Text;
    end;
    FFlashEdit.Visible := FALSE;
    Key := 0;
    Refresh;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.FlashEditExit
  Description: Exit event handler of FFlashEdit in-place editor
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: hide the FFlashEditor
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.FlashEditExit(
  Sender: TObject);
var
  wParam,
  lParam: Longint;
begin
  FFlashEdit.Visible := FALSE;
  FFlashEdit.Text := '';
  ZeroScreenStuff;
  wParam := FROM_FLASH_EDIT;
  lParam := FREE_FLASH_EDIT;
  PostMessage(
    Self.Handle, {handle of destination window}
    WM_USER,	 {message to post}
    wParam,	 {first message parameter}
    lParam); 	 {second message parameter}
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.WMUser
  Description: Message handler for FlashEdit freeing
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Free the FlashEditor after use
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.WMUser(var Message: TMessage);
begin
  if (Message.WParam = FROM_FLASH_EDIT) then
    if (Message.LParam = FREE_FLASH_EDIT) then
    begin
      FFlashEdit.Free;
      FFlashEdit := nil;
    end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.MouseDown
  Description: MouseDown event handler
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 08/31/2000 by Mat Ballard
      Purpose: The start of all mouse routines
 Known Issues: MouseDown gets called AFTER DblClick !
      Changes: GetTheClickedObject now moved to within the 'if (FScreenJob = sjNone) then'
 ------------------------------------------------------------------------------}
procedure TCustomPlot.MouseDown(
  Button: TMouseButton;
  Shift: TShiftState;
  X,
  Y: Integer);
begin
{Nuke any Caption:}
  FInstructions.Clear;

{record the beginning:}
  MouseStart.x := X;
  MouseStart.y := Y;
{set the moving object co-ordinates:}
  Selection.Left := X;
  Selection.Top := Y;
  Selection.Height := 1;
  Selection.Width := 1;

{  if (FScreenJob <> sjFlashEdit) then
    FFlashEdit.Visible := FALSE;}

{if no ScreenJob has been set yet, it could be several things:}
  if (FScreenJob = sjNone) then
  begin
{no job yet}
{what got clicked ?}
    GetTheClickedObject(X, Y);
    if (Button = mbLeft) then
    begin
{left click:}
      if (ssShift in Shift) then
      begin
{We want to zoom in:}
        FScreenJob := sjZoomIn;
      end
      else if ((ClickedObjectType <> soNone) and
               (FMovable)) then
      begin
{left clicks can lead to click and drag:}
        MouseTimer.Enabled := TRUE;
      end;
    end; {left button}
    {NOTE: if it is the right button, then the popup menu will be displayed
     at the end of the MouseUp}
  end; {if sjNone}

  case FScreenJob of
    {sjNone: already done}
    {sjDrag: set by MouseTimeOut}
    sjHide: HideClick(Self);
    sjZoomIn:
      begin
        Screen.Cursor := crSize;
        OutlineTheClickedObject;
      end;
    {sjEditAxis:
    sjEditFont:
    sjEditPoint:
    sjEditAxis:
    sjCopAxis: ;
    sjDisplace:   ;
    sjCloneAxis: ;
    sjDeleteAxis: all done by popupmenu or option}
    sjPosition: PositionClick(Self);
    sjNearestPoint: NearestPointClick(Self);
    sjAverage,
    {sjContractAxis: ;
    sjSplineAxis: ;
    sjHighs,
    sjLows,
    sjMovingAverage,
    sjSmoothAxis:   ;
    sjSortAxis: ;
    sjDifferentiate:   ;
    sjIntegrate: all done by popupmenu or option}
    sjIntegral,
    sjLineOfBestFit,
    sjDualLineBestFit1,
    sjDualLineBestFit2,
    sjSelection,
    sjDualSelection1,
    sjDualSelection2:
      begin
        Screen.Cursor := crSize;
        OutlineTheSelection;
      end;
  end;

  inherited MouseDown(Button, Shift, X, Y);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.MouseMove
  Description: MouseMove event handler
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Moves the dashed outline around the screen; how it moves depends on the object
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.MouseMove(
  Shift: TShiftState;
  X,
  Y: Integer);
var
  Gap,
  NewLeft,
  NewTop: Integer;
  Ptr: Pointer;
begin
  MouseTimer.Enabled := FALSE;

  if (ssLeft in Shift) then
  begin
    case FScreenJob of
      {sjNone:}
      sjDrag:
        case ClickedObjectType of
          soTitle:
            begin
              if (X < (FBorder.Left + FBorder.MidX) div 2) then
                NewLeft := FBorder.Left
              else if (X > (FBorder.Right + FBorder.MidX) div 2) then
                NewLeft := FBorder.Right - Selection.Width
              else
                NewLeft := FBorder.MidX - Selection.Width div 2;
              if (Y > FXAxis.MidY) then
                NewTop := Height - BevelGap - FTitle.Height
              else
                NewTop := BevelGap;
              MoveTheClickedObjectTo(NewLeft, NewTop);
            end;
          soXAxis, soTopBorder, soBottomBorder:
            begin
              MoveTheClickedObjectTo(
                TRectangle(pClickedObject).Left,
                Y - ClickedObjectOffset.y);
            end;
          soYAxis, soLeftBorder, soRightBorder:
            begin
              MoveTheClickedObjectTo(
                X - ClickedObjectOffset.x,
                TRectangle(pClickedObject).Top);
            end;
          soXAxisTitle:
            begin
              if (X < (FBorder.Left + FBorder.MidX) div 2) then
                NewLeft := FBorder.Left
              else if (X > (FBorder.Right + FBorder.MidX) div 2) then
                NewLeft := FBorder.Right - Selection.Width
              else
                NewLeft := FBorder.MidX - Selection.Width div 2;
              Gap := Abs(FXAxis.Title.MidY - FXAxis.MidY);
              if (Y > FXAxis.MidY) then
                NewTop := FXAxis.MidY + Gap - FXAxis.Title.Height div 2
              else
                NewTop := FXAxis.MidY - Gap - FXAxis.Title.Height div 2;
              MoveTheClickedObjectTo(NewLeft, NewTop);
            end;
          soYAxisTitle:
            begin
{Which Y Axis owns this Title ?}
              Ptr := TRectangle(pClickedObject).Owner;
              Gap := Abs(TRectangle(pClickedObject).MidX - TAxis(Ptr).MidX);
              if (X < TAxis(Ptr).MidX) then
                NewLeft := TAxis(Ptr).MidX - Gap - TRectangle(pClickedObject).Width div 2
              else
                NewLeft := TAxis(Ptr).MidX + Gap - TRectangle(pClickedObject).Width div 2;
              if (Y < (FBorder.Top + FBorder.MidY) div 2) then
                NewTop := FBorder.Top
              else if (Y > (FBorder.Bottom + FBorder.MidY) div 2) then
                NewTop := FBorder.Bottom - Selection.Height
              else
                NewTop := FBorder.MidY - Selection.Height div 2;
              MoveTheClickedObjectTo(NewLeft, NewTop);
            end;
          soXAxisLabel:
            begin
              Gap := Abs(FXAxis.Labels.MidY - FXAxis.MidY);
              if (Y < FXAxis.MidY) then
                MoveTheClickedObjectTo(Selection.Left, FXAxis.MidY - Gap -
                  FXAxis.Labels.Height div 2)
              else
                MoveTheClickedObjectTo(Selection.Left, FXAxis.MidY + Gap -
                  FXAxis.Labels.Height div 2);
            end;
          soYAxisLabel:
            begin
{Which Y Axis owns this Title ?}
              Ptr := TRectangle(pClickedObject).Owner;
              Gap := Abs(TRectangle(pClickedObject).MidX - TAxis(Ptr).MidX);
              if (X < TAxis(Ptr).MidX) then
                MoveTheClickedObjectTo(
                  TAxis(Ptr).MidX - Gap - TRectangle(pClickedObject).Width div 2,
                  Selection.Top)
              else
                MoveTheClickedObjectTo(
                  TAxis(Ptr).MidX + Gap - TRectangle(pClickedObject).Width div 2,
                  Selection.Top);
            end;
          soLegend, soResult:
            begin {both of these can move freely:}
              MoveTheClickedObjectTo(
                X - ClickedObjectOffset.x,
                Y - ClickedObjectOffset.y);
            end;
          soSeries:
            begin
              pSeries.MoveSeriesBy(Canvas, X-Selection.Left, Y-Selection.Top);
              Selection.Left := X;
              Selection.Top := Y;
            end;
        end; {end case sjDrag}
      {sjHide:}
      sjZoomIn:
        StretchTheClickedObjectTo(X, Y);
      {sjEditAxis: ;
      sjEditFont: ;
      sjEditPoint: ;
      sjEditAxis: ;
      sjCopAxis: ;
      sjDisplace:   ;
      sjCloneAxis: ;
      sjDeleteAxis: all done by popupmenu or option}
      {sjPosition: already done, or by popupmenu}
      {sjNearestPoint: already done, or by popupmenu}
      sjAverage,
      {sjContractAxis: ;
      sjSplineAxis: ;
      sjHighs,
      sjLows,
      sjMovingAverage,
      sjSmoothAxis:   ;
      sjSortAxis: ;
      sjDifferentiate:   ;
      sjIntegrate: all done by popupmenu or option}
      sjIntegral,
      sjLineOfBestFit,
      sjDualLineBestFit1,
      sjDualLineBestFit2,
      sjSelection,
      sjDualSelection1,
      sjDualSelection2:
        StretchTheClickedObjectTo(X, Y);
    end;
  end; {if (ssLeft in Shift)}

  inherited MouseMove(Shift, X, Y);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.MouseUp
  Description: MouseUp event handler
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Reacts to the user finishing an action with the mouse
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.MouseUp(
  Button: TMouseButton;
  Shift: TShiftState;
  X,
  Y: Integer);
var
  i: Integer;
  Point: TPoint;
  pTheYAxis: TAxis;
{Variables used in least-squares fitting:}
  NoLeastSquarePts: Integer;
  SumX, SumY, SumXsq, SumXY, SumYsq: Double;
  Rsq: Single;
{  Slope, Intercept: Single; - are globals to allow drawing of line}

  procedure InitializeFit;
  begin
{Initialize the fit parameters:}
    NoLeastSquarePts := 0;
    SumX := 0;
    SumY := 0;
    SumXsq := 0;
    SumXY := 0;
    SumYsq := 0;
  end;

begin
  MouseTimer.Enabled := FALSE;

  if (Button = mbLeft) then
  begin
    case FScreenJob of
      {sjNone:}
      sjDrag:
        begin
          OutlineTheClickedObject;
          if (SecondClickedObjectType = soNone) then
          begin
            MoveTheClickedObjectClick(Self);
          end
          else
          begin
            Point.x := X;
            Point.y := Y;
            Point := ClientToScreen(Point);
            if (pClickedObject <> nil) then
              WhichPopUpItems[0].Caption := 'Move the ' +
                TRectangle(pClickedObject).Name;
            if (pSecondClickedObject <> nil) then
              WhichPopUpItems[1].Caption := 'Move the ' +
                TRectangle(pSecondClickedObject).Name;
            WhichPopUpMenu.Popup(Point.x, Point.y);
          end;
        end;
      {sjHide:}
      sjZoomIn:
        begin
          OutlineTheClickedObject;
          SwapEnds;
          FXAxis.Min := FXAxis.XofF(Selection.Left);
          FXAxis.Max := FXAxis.XofF(Selection.Right);
          for i := 1 to FAxisList.Count-1 do
          begin
            pTheYAxis := TAxis(FAxisList[i]);
            pTheYAxis.AutoScale := TRUE;
            pTheYAxis.Min := pTheYAxis.YofF(Selection.Bottom);
            pTheYAxis.Max := pTheYAxis.YofF(Selection.Top);
          end;
          ZeroScreenStuff;
        end;
      {sjEditAxis: ;
      sjEditFont: ;
      sjEditPoint: ;
      sjEditAxis: ;
      sjCopAxis: ;
      sjDisplace:   ;
      sjCloneAxis: ;
      sjDeleteAxis: ;}
      {sjPosition: already done, or by popupmenu}
      {sjNearestPoint: already done, or by popupmenu}
      sjAverage:
        begin
          OutlineTheSelection;
          SwapEnds;
          AverageClick(Self);
        end;
      {sjContractAxis: ;
      sjSplineAxis: ;
      sjHighs,
      sjLows,
      sjMovingAverage,
      sjSmoothAxis:   ;
      sjSortAxis: ;
      sjDifferentiate:   ;
      sjIntegrate:        ;}
      sjIntegral:
        begin
          OutlineTheSelection;
          SwapEnds;
          IntegralClick(Self);
        end;
      sjLineOfBestFit:
        begin
          OutlineTheSelection;
          SwapEnds;
          InitializeFit;
          pSeries.LineBestFit(XAxis.XofF(Selection.Left), XAxis.XofF(Selection.Right),
            NoLeastSquarePts,
            SumX, SumY, SumXsq, SumXY, SumYsq,
            Slope, Intercept, Rsq);
          SetResult(Slope, Intercept, Rsq);
          ZeroScreenStuff;
        end;
      sjDualLineBestFit1:
        begin
          OutlineTheSelection;
          SwapEnds;
          Sel1.Left := Selection.Left;
          Sel1.Top := Selection.Top;
          Sel1.Right := Selection.Right;
          Sel1.Bottom := Selection.Bottom;
          ScreenJob := sjDualLineBestFit2;
          FInstructions.Clear;
          FInstructions.Add('Click and Drag over the SECOND region to fit');
          Refresh;
        end;
      sjDualLineBestFit2:
        begin
          OutlineTheSelection;
          SwapEnds;
          InitializeFit;
          pSeries.LineBestFit(XAxis.XofF(Sel1.Left), XAxis.XofF(Sel1.Right),
            NoLeastSquarePts,
            SumX, SumY, SumXsq, SumXY, SumYsq,
            Slope, Intercept, Rsq);
          pSeries.LineBestFit(XAxis.XofF(Selection.Left), XAxis.XofF(Selection.Right),
            NoLeastSquarePts,
            SumX, SumY, SumXsq, SumXY, SumYsq,
            Slope, Intercept, Rsq);
          SetResult(Slope, Intercept, Rsq);
          ZeroScreenStuff;
        end;
      sjSelection:
        begin
          OutlineTheSelection;
          SwapEnds;
          Sel1.Left := Selection.Left;
          Sel1.Top := Selection.Top;
          Sel1.Right := Selection.Right;
          Sel1.Bottom := Selection.Bottom;
          DoSelection(Sel1);
        end;
      sjDualSelection1:
        begin
          OutlineTheSelection;
          SwapEnds;
          Sel1.Left := Selection.Left;
          Sel1.Top := Selection.Top;
          Sel1.Right := Selection.Right;
          Sel1.Bottom := Selection.Bottom;
          ScreenJob := sjDualLineBestFit2;
          FInstructions.Clear;
          FInstructions.Add('Click and Drag over the SECOND region');
          Refresh;
        end;
      sjDualSelection2:
        begin
          OutlineTheSelection;
          SwapEnds;
          Sel2.Left := Selection.Left;
          Sel2.Top := Selection.Top;
          Sel2.Right := Selection.Right;
          Sel2.Bottom := Selection.Bottom;
          DoDualSelection(Sel1, Sel2);
        end;
    end; {end case}
    Refresh;
  end {end Left Button}
  else
  begin {Right Button}
{$IFNDEF DELPHI1}
{what does that do to menu visibility ?}
    DetermineMenuVisibility;
{$ENDIF}
{we no longer let the ancestor run the popup:
 see note above on 'property PopupMenu':}
    Point.x := X;
    Point.y := Y;
    Point := ClientToScreen(Point);
    FPlotPopUpMenu.Popup(Point.x, Point.y);
  end; {end Right Button}

{inherited runs the popup if neccessary:}
  inherited MouseUp(Button, Shift, X, Y);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DoSelection
  Description: Fires the Selection event
       Author: Mat Ballard
 Date created: 09/07/2000
Date modified: 09/07/2000 by Mat Ballard
      Purpose:
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DoSelection(Sel1: TRect);
begin
  if Assigned(FOnSelection) then
    OnSelection(Self, Sel1);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DoDualSelection
  Description: Fires the DualSelection event
       Author: Mat Ballard
 Date created: 09/07/2000
Date modified: 09/07/2000 by Mat Ballard
      Purpose:
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DoDualSelection(Sel1, Sel2: TRect);
begin
  if Assigned(FOnDualSelection) then
    OnDualSelection(Self, Sel1, Sel2);
end;

{$IFNDEF DELPHI1}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DetermineMenuVisibility
  Description: Sets the visibility of Axis-related menus
       Author: Mat Ballard
 Date created: 04/17/2000
Date modified: 04/17/2000 by Mat Ballard
      Purpose: menu management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DetermineMenuVisibility;
var
  i: Integer;
  SeriesVisibility: Boolean;
begin
{$IFDEF SHOWALLMENUS}
  exit;
{$ENDIF}

  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuPaste)].Enabled :=
    (ClipBoard.HasFormat(CF_TEXT));

  FPlotPopUpMenu.Items[Ord(mnuView)].Items[Ord(mnuHide)].Visible := TRUE;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuEditAxis)].Visible := FALSE;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuEditFont)].Visible := FALSE;
  FPlotPopUpMenu.Items[Ord(mnuView)].Items[Ord(mnuLegend)].Visible := FALSE;
  SeriesVisibility := FALSE;

  FPlotPopUpMenu.Items[Ord(mnuFile)].Items[Ord(mnuClearOverlays)].Visible :=
    (FirstOverlay >= 0);

  case ClickedObjectType of
    soTitle,
    soXAxisTitle, soYAxisTitle,
    soXAxisLabel, soYAxisLabel,
    soResult:
      begin
        FPlotPopUpMenu.Items[Ord(mnuView)].Items[Ord(mnuHide)].Visible := TRUE;
        FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuEditFont)].Visible := TRUE;
      end;
    soLegend:
      begin
        FPlotPopUpMenu.Items[Ord(mnuView)].Items[Ord(mnuHide)].Visible := TRUE;
        FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuEditFont)].Visible := TRUE;
        if (FSeriesList.Count > 1) then
        begin
          FPlotPopUpMenu.Items[Ord(mnuView)].Items[Ord(mnuLegend)].Visible := TRUE;
        end;
      end;
    soXAxis, soYAxis:
      begin
        FPlotPopUpMenu.Items[Ord(mnuView)].Items[Ord(mnuHide)].Visible := TRUE;
        FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuEditAxis)].Visible := TRUE;
      end;
    soSeries:
      begin
        pSeries.OutlineSeries(Canvas);
        FPlotPopUpMenu.Items[Ord(mnuView)].Items[Ord(mnuHide)].Visible := TRUE;
        SeriesVisibility := TRUE;
      end;
  else
    {soLeftBorder, soTopBorder, soRightBorder, soBottomBorder:}
    begin
      FPlotPopUpMenu.Items[Ord(mnuView)].Items[Ord(mnuHide)].Visible := FALSE;
    end;
  end;
  SetSeriesVisibility(SeriesVisibility);

  for i := 0 to Ord(High(TMainMenus)) do
    FPlotPopUpMenu.Items[i].Visible :=
      FPlotPopUpMenu.Items[i].Visible and
        (TMainMenus(i) in FPopupOptions.Menu);
  for i := 0 to Ord(High(TFileMenus)) do
    FPlotPopUpMenu.Items[Ord(mnuFile)].Items[i].Visible :=
      FPlotPopUpMenu.Items[Ord(mnuFile)].Items[i].Visible and
        (TFileMenus(i) in FPopupOptions.File_);
  for i := 0 to Ord(High(TEditMenus)) do
    FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[i].Visible :=
      FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[i].Visible and
        (TEditMenus(i) in FPopupOptions.Edit);
  for i := 0 to Ord(High(TViewMenus)) do
    FPlotPopUpMenu.Items[Ord(mnuView)].Items[i].Visible :=
      FPlotPopUpMenu.Items[Ord(mnuView)].Items[i].Visible and
        (TViewMenus(i) in FPopupOptions.View);
  for i := 0 to Ord(High(TCalcMenus)) do
    FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[i].Visible :=
      FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[i].Visible and
        (TCalcMenus(i) in FPopupOptions.Calc);

  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuEditFontDiv)].Visible :=
    FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuEditFont)].Visible;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuDisplaceDiv)].Visible :=
      FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuDisplace)].Visible;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuEditSeriesDiv)].Visible :=
    FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuEditSeries)].Visible;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuCalcAverageDiv)].Visible :=
    FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuCalcAverage)].Visible;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuLineOfBestFitDiv)].Visible := 
    FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuLineOfBestFit)].Visible;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetSeriesVisibility
  Description: Sets the visibility of Axis-related menus
       Author: Mat Ballard
 Date created: 04/17/2000
Date modified: 04/17/2000 by Mat Ballard
      Purpose: menu management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetSeriesVisibility(Value: Boolean);
begin
{Can't do anything to Axis if there is no data:}
  Value := Value and (FSeriesList.TotalNoPts > 0);

{The following are independent of whether or not a Series has been selected:}
  FPlotPopUpMenu.Items[Ord(mnuFile)].Items[Ord(mnuSave)].Visible :=
    (FSeriesList.TotalNoPts > 0) and FSeriesList.DataChanged;
  FPlotPopUpMenu.Items[Ord(mnuFile)].Items[Ord(mnuSaveAs)].Visible :=
    (FSeriesList.TotalNoPts > 0);

  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuDeleteY2Axis)].Visible :=
    (FAxisList.Count > 2);
    
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuNearestPoint)].Visible :=

    (FSeriesList.TotalNoPts > 0);
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuContractAllSeries)].Visible :=
    (FSeriesList.TotalNoPts > 20);

{The following DO DEPEND on whether or not a Series has been selected:}
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuCopyHTML)].Visible := Value;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuCopySeries)].Visible := Value;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuDisplace)].Visible := Value;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuResetDisplacement)].Visible := Value;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuCloneSeries)].Visible := Value;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuEditPoint)].Visible := Value;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuDeleteSeries)].Visible := Value;
  FPlotPopUpMenu.Items[Ord(mnuEdit)].Items[Ord(mnuEditSeries)].Visible := Value;

  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuCalcAverage)].Visible := Value;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuContractSeries)].Visible :=
    Value and
    (FSeriesList.TotalNoPts > 20) and
    (pSeries <> nil) and
    (not ((pSeries.ExternalXSeries) or (pSeries.XDataRefCount > 0)));
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuCubicSplineSeries)].Visible := Value;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuHighs)].Visible := Value;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuMovingAverage)].Visible := Value;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuSmoothSeries)].Visible := Value;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuSortSeries)].Visible := Value;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuCalculusDiv)].Visible := Value;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuDifferentiate)].Visible := Value;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuIntegrate)].Visible := Value;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuIntegral)].Visible := Value;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuLineOfBestFit)].Visible := Value;
  FPlotPopUpMenu.Items[Ord(mnuCalc)].Items[Ord(mnuTwoRegionLineOfBestFit)].Visible := Value;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DetermineMenuEnabledness
  Description: Sets the Enabledness of Axis-related menus
       Author: Mat Ballard
 Date created: 04/17/2000
Date modified: 04/17/2000 by Mat Ballard
      Purpose: menu management
 Known Issues: called from TPlotMenu.HandleClunk
               Because we have added a "Reopen" sub-submenu, we have to
               kludge and add "1" to the index of all mnuFile menuitems.
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DetermineMenuEnabledness(TheMenu: TMenu);
begin
  TheMenu.Items[Ord(mnuEdit)].Items[Ord(mnuPaste)].Enabled :=
    (ClipBoard.HasFormat(CF_TEXT));

  SetSeriesEnabledness(TheMenu);

{Note Reopen kludge: "1+"}
  TheMenu.Items[Ord(mnuFile)].Items[1+Ord(mnuClearOverlays)].Enabled :=
    (FirstOverlay >= 0);

  TheMenu.Items[Ord(mnuEdit)].Items[Ord(mnuDeleteY2Axis)].Enabled :=
    (FAxisList.Count > 2);
  TheMenu.Items[Ord(mnuEdit)].Items[Ord(mnuEditFont)].Enabled := TRUE;

  TheMenu.Items[Ord(mnuView)].Items[Ord(mnuHide)].Enabled := TRUE;
  TheMenu.Items[Ord(mnuView)].Items[Ord(mnuLegend)].Enabled :=
    (FSeriesList.Count > 1);
  TheMenu.Items[Ord(mnuView)].Items[Ord(mnuNormalView)].Enabled :=
    FPlotPopUpMenu.Items[Ord(mnuView)].Items[Ord(mnuNormalView)].Enabled;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetSeriesEnabledness
  Description: Sets the Enabledness of Axis-related menus
       Author: Mat Ballard
 Date created: 04/17/2000
Date modified: 04/17/2000 by Mat Ballard
      Purpose: menu management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetSeriesEnabledness(TheMenu: TMenu);
begin
{Note Reopen kludge: "1+"}
  TheMenu.Items[Ord(mnuFile)].Items[1+Ord(mnuSave)].Enabled :=
    (FSeriesList.TotalNoPts > 0) and FSeriesList.DataChanged;
  TheMenu.Items[Ord(mnuFile)].Items[1+Ord(mnuSaveAs)].Enabled :=
    (FSeriesList.TotalNoPts > 0);

  TheMenu.Items[Ord(mnuCalc)].Items[Ord(mnuNearestPoint)].Enabled :=
    (FSeriesList.TotalNoPts > 0);
  TheMenu.Items[Ord(mnuCalc)].Items[Ord(mnuContractAllSeries)].Enabled :=
    (FSeriesList.TotalNoPts > 20);

  TheMenu.Items[Ord(mnuEdit)].Items[Ord(mnuCopyHTML)].Enabled :=
    (FSeriesList.TotalNoPts > 0);
  TheMenu.Items[Ord(mnuEdit)].Items[Ord(mnuCopySeries)].Enabled :=
    (FSeriesList.TotalNoPts > 0);
  TheMenu.Items[Ord(mnuEdit)].Items[Ord(mnuDisplaceDiv)].Enabled :=
    (FSeriesList.TotalNoPts > 0);
  TheMenu.Items[Ord(mnuEdit)].Items[Ord(mnuDisplace)].Enabled :=
    (FSeriesList.TotalNoPts > 0);
  TheMenu.Items[Ord(mnuEdit)].Items[Ord(mnuResetDisplacement)].Enabled :=
    (FSeriesList.TotalNoPts > 0);
  TheMenu.Items[Ord(mnuEdit)].Items[Ord(mnuEditSeriesDiv)].Enabled :=
    (FSeriesList.TotalNoPts > 0);
  TheMenu.Items[Ord(mnuEdit)].Items[Ord(mnuCloneSeries)].Enabled :=
    (FSeriesList.TotalNoPts > 0);
  TheMenu.Items[Ord(mnuEdit)].Items[Ord(mnuEditPoint)].Enabled :=
    (FSeriesList.TotalNoPts > 0);
  TheMenu.Items[Ord(mnuEdit)].Items[Ord(mnuDeleteSeries)].Enabled :=
    (FSeriesList.TotalNoPts > 0);
  TheMenu.Items[Ord(mnuEdit)].Items[Ord(mnuEditSeries)].Enabled :=
    (FSeriesList.TotalNoPts > 0);

  TheMenu.Items[Ord(mnuCalc)].Items[Ord(mnuCalcAverageDiv)].Enabled :=
    (FSeriesList.TotalNoPts > 0);
  TheMenu.Items[Ord(mnuCalc)].Items[Ord(mnuCalcAverage)].Enabled :=
    (FSeriesList.TotalNoPts > 0);
  TheMenu.Items[Ord(mnuCalc)].Items[Ord(mnuContractSeries)].Enabled :=
     (FSeriesList.TotalNoPts > 20);
  TheMenu.Items[Ord(mnuCalc)].Items[Ord(mnuCubicSplineSeries)].Enabled :=
    (FSeriesList.TotalNoPts > 0);
  TheMenu.Items[Ord(mnuCalc)].Items[Ord(mnuHighs)].Enabled :=
    (FSeriesList.TotalNoPts > 0);
  TheMenu.Items[Ord(mnuCalc)].Items[Ord(mnuMovingAverage)].Enabled :=
    (FSeriesList.TotalNoPts > 0);
  TheMenu.Items[Ord(mnuCalc)].Items[Ord(mnuSmoothSeries)].Enabled :=
    (FSeriesList.TotalNoPts > 0);
  TheMenu.Items[Ord(mnuCalc)].Items[Ord(mnuSortSeries)].Enabled :=
    (FSeriesList.TotalNoPts > 0);
  TheMenu.Items[Ord(mnuCalc)].Items[Ord(mnuCalculusDiv)].Enabled :=
    (FSeriesList.TotalNoPts > 0);
  TheMenu.Items[Ord(mnuCalc)].Items[Ord(mnuDifferentiate)].Enabled :=
    (FSeriesList.TotalNoPts > 0);
  TheMenu.Items[Ord(mnuCalc)].Items[Ord(mnuIntegrate)].Enabled :=
    (FSeriesList.TotalNoPts > 0);
  TheMenu.Items[Ord(mnuCalc)].Items[Ord(mnuIntegral)].Enabled :=
    (FSeriesList.TotalNoPts > 0);
  TheMenu.Items[Ord(mnuCalc)].Items[Ord(mnuLineOfBestFitDiv)].Enabled :=
    (FSeriesList.TotalNoPts > 0);
  TheMenu.Items[Ord(mnuCalc)].Items[Ord(mnuLineOfBestFit)].Enabled :=
    (FSeriesList.TotalNoPts > 0);
  TheMenu.Items[Ord(mnuCalc)].Items[Ord(mnuTwoRegionLineOfBestFit)].Enabled :=
    (FSeriesList.TotalNoPts > 0);
end;
{$ENDIF}

{------------------------------------------------------------------------------
    Procedure: SwapEnds
  Description: Swaps the selection's Left-Right and Top-Bottom if needed
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: manageing region selections
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SwapEnds;
var
  iX: Integer;
begin
{swap Left and Right:}
  if (Selection.Left > Selection.Right) then
  begin
    iX := Selection.Left;
    Selection.Left := Selection.Right;
    Selection.Right := iX;
  end;

{swap Top and Bottom:}
  if (Selection.Top > Selection.Bottom) then
  begin
    iX := Selection.Top;
    Selection.Top := Selection.Bottom;
    Selection.Bottom := iX;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: SetResult
  Description: Performs calculations on Screen Data
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Finishes off Line of Best Fit determinations
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetResult(
  Slope,
  Intercept,
  Rsq: Single);
begin
  if (pSeries.XAxis.LogScale) then
    FResult.Caption := 'Ln(' + pSeries.XAxis.Title.Caption + ')  =  '
   else
    FResult.Caption := pSeries.XAxis.Title.Caption + '  =  ';
  FResult.Caption := FResult.Caption + FloatToStrF(Intercept, ffGeneral, 5, 3) + ' + ';
  if (pSeries.XAxis.LogScale) then
    FResult.Caption := FResult.Caption + FloatToStrF(Slope, ffGeneral, 5, 3) +
      '  Ln(' + pSeries.YAxis.Title.Caption + '),  R-Square = '
   else
    FResult.Caption := FResult.Caption + FloatToStrF(Slope, ffGeneral, 5, 3) +
      '  ' + pSeries.YAxis.Title.Caption + ',  R-Square = ';
  FResult.Caption := FResult.Caption + FloatToStrF(Rsq, ffGeneral, 5, 3);
  ClipBoard.AsText := FResult.Caption;
  FResult.Font.Color := pSeries.Pen.Color;
  FResult.Left := Selection.Left;
  FResult.Top := Selection.Top;
  FResult.Visible := TRUE;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.MouseTimeOut
  Description: responds to the mouse button being held down
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Frees the timer, identifies the clicked object, outlines it, and prepares to move it
 Known Issues:
 ------------------------------------------------------------------------------}

procedure TCustomPlot.MouseTimeOut(
  Sender: TObject);
{If this is fired, then the user has held the mouse still for one second
 on a screen object: this is a cue for a move.}
begin
  MouseTimer.Enabled := FALSE;

  if (ClickedObjectType = soNone) then exit;

  FScreenJob := sjDrag;
{$IFDEF COMPILER3_UP}
  Screen.Cursor := crHandPoint;
{$ENDIF}
  if (ClickedObjectType = soSeries) then
  begin
    pSeries.OutlineSeries(Canvas);
  end
  else
  begin
{This is fascinating: the following call attempts to assign all the TAxis
 properties of pClickedObject to Selection - which of course pukes:
    Selection.Assign(TRectangle(pClickedObject));
 So we have to assign position manually:}
    if (pClickedObject <> nil) then
    begin
      Selection.Left := TRectangle(pClickedObject).Left;
      Selection.Top := TRectangle(pClickedObject).Top;
      Selection.Width := TRectangle(pClickedObject).Width;
      Selection.Height := TRectangle(pClickedObject).Height;
    end;

    OutlineTheClickedObject;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.GetTheClickedObject
  Description: identifies the object(s) that was clicked on
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets ClickedObjectType and SecondClickedObjectType, and TheSeries if it was a Axis.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.GetTheClickedObject(
  X,
  Y: Integer);
var
  i: Integer;
  NearestX, NearestY, MinDistance: Single;
  NearestiX, NearestiY: Integer;
begin
  pClickedObject := nil;
  pSecondClickedObject := nil;
  
  for i := 1 to ScreenObjectList.Count-1 do
  begin
{identify the clicked-on object:}
    if (TRectangle(ScreenObjectList.Items[i]).ClickedOn(X, Y)) then
    begin
      if (pClickedObject = nil) then
      begin
        pClickedObject := ScreenObjectList.Items[i];
        ClickedObjectType := TObjectType(TRectangle(pClickedObject).Tag);
        ClickedObjectOffset.x := X - TRectangle(pClickedObject).Left;
        ClickedObjectOffset.y := Y - TRectangle(pClickedObject).Top;
        if ((ClickedObjectType = soXAxis) or
            (ClickedObjectType = soYAxis)) then
        begin
          pAxis := TAxis(pClickedObject);
          TheAxis := FAxisList.IndexOf(pAxis);
        end;
      end
      else
      begin
{there are two objects under the mouse: usually an axis and a border:}
        pSecondClickedObject := ScreenObjectList.Items[i];
        SecondClickedObjectType := TObjectType(TRectangle(pSecondClickedObject).Tag);
        exit;
      end;
    end; {clicked on object i}
  end; {for}

  if (ClickedObjectType = soNone) then
  begin
{was it a Series ?}
    ThePointNumber := FSeriesList.GetNearestPoint(Selection.Left, Selection.Top,
      TheSeries, NearestiX, NearestiY, NearestX, NearestY, MinDistance, pSeries);
{give it a wide hit range:}
    if (MinDistance < (FOutlineWidth)) then
    begin
      ClickedObjectType := soSeries;
      pSeries.GenerateOutline(FOutlineWidth);
    end;
  end; {ClickedObjectType = soNone}
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ProcessClickedObject
  Description: Adjusts the geometry of a moved (clicked and dragged) object
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets appropriate property(ies) of the object that has been manipulated on screen
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ProcessClickedObject(
  pObject: Pointer;
  TheObjectType: TObjectType);
begin
  case TheObjectType of
    soTitle:
      begin
        if (Selection.Left = FBorder.Left) then
          FTitle.Alignment := taLeftJustify
         else if (Selection.Right = FBorder.Right) then
          FTitle.Alignment := taRightJustify
         else
          FTitle.Alignment := taCenter;
        if (Selection.Top < FBorder.MidY) then
          FTitle.Orientation := orLeft
         else
          FTitle.Orientation := orRight;
      end;

    soLegend:
      begin
        FLegend.Left := Selection.Left;
        FLegend.Top := Selection.Top;
      end;

    soResult:
      begin
        FResult.Left := Selection.Left;
        FResult.Top := Selection.Top;
      end;

    soXAxis:
      begin
        FXAxis.Intercept := FYAxis.YofF(Selection.MidY);
      end;

    soXAxisTitle:
      begin
        if (Selection.Left = FBorder.Left) then
          FXAxis.Title.Alignment := taLeftJustify
         else if (Selection.Right = FBorder.Right) then
          FXAxis.Title.Alignment := taRightJustify
         else
          FXAxis.Title.Alignment := taCenter;
        if (Selection.Top < FXAxis.MidY) then
          FXAxis.Title.Orientation := orLeft
         else
          FXAxis.Title.Orientation := orRight;
      end;
    soXAxisLabel:
      begin
        if (Selection.Top < FXAxis.MidY) then
          FXAxis.TickDirection := orLeft
         else
          FXAxis.TickDirection := orRight;
      end;

    soYAxis:
      begin
        TAxis(pObject).Intercept := FXAxis.XofF(Selection.MidX);
        Resize;
      end;
    soYAxisTitle:
      begin
        if (Selection.Left < TAxis(TTitle(pObject).Owner).MidX) then
          TTitle(pObject).Orientation := orLeft
         else
          TTitle(pObject).Orientation := orRight;
        if (Selection.Top = FBorder.Top) then
          TTitle(pObject).Alignment := taRightJustify
         else if (Selection.Bottom = FBorder.Bottom) then
          TTitle(pObject).Alignment := taLeftJustify
         else
          TTitle(pObject).Alignment := taCenter;
      end;
    soYAxisLabel:
      begin
        if (Selection.Left < TAxis(TAxisLabel(pObject).Owner).MidX) then
          TAxis(TAxisLabel(pObject).Owner).TickDirection := orLeft
         else
          TAxis(TAxisLabel(pObject).Owner).TickDirection := orRight;
      end;

    soLeftBorder: FBorder.Left := Selection.MidX;
    soRightBorder: FBorder.Right := Selection.MidX;
    soTopBorder: FBorder.Top := Selection.MidY;
    soBottomBorder: FBorder.Bottom := Selection.MidY;

    soSeries: { already moved by direct change of DeltaX and DeltaY properties}
      begin

      end;
  end; {case TheObjectType}
  ZeroScreenStuff;
  Refresh;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.OutlineTheClickedObject
  Description: Outlines The Clicked Object
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: gives the user a guide to what they are manipulating with the mouse
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.OutlineTheClickedObject;
begin
  if (ClickedObjectType = soSeries) then
    pSeries.OutlineSeries(Canvas)
   else
    OutlineTheSelection;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.OutlineTheSelection
  Description: Outlines The Moving Object
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: gives the user a guide to what they are moving with the mouse
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.OutlineTheSelection;
begin
  Canvas.Pen.Color := clBlack;
  Canvas.Pen.Mode := pmNotXOR;
  Canvas.Pen.Style := psDash;
  Canvas.Rectangle(Selection.Left, Selection.Top,
                   Selection.Right, Selection.Bottom);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.MoveTheClickedObjectTo
  Description: This moves the clicked object outline TO (X, Y)
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: gives the user a guide to what they are moving with the mouse
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.MoveTheClickedObjectTo(
  X,
  Y: Integer);
begin
  if ((Selection.Left = X) and (Selection.Top = Y)) then exit;

{erase the old outline:}
  OutlineTheClickedObject;

{re-initialize the Selection:}
  Selection.Left := X;
  Selection.Top := Y;

{create the new outline:}
  OutlineTheClickedObject;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.StretchTheClickedObjectTo
  Description: Stretch The Clicked Object To
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: gives the user a guide to what they are selecting with the mouse
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.StretchTheClickedObjectTo(
  X,
  Y: Integer);
{This moves the far (right, bottom) point of the Stretched object outline TO (X, Y).}
begin
  if ((Selection.Right = X) and (Selection.Bottom = Y)) then
    exit;

{erase the old outline:}
  OutlineTheSelection;

{re-initialize the Selection:}
  Selection.Right := X;
  Selection.Bottom := Y;

{create the new outline:}
  OutlineTheSelection;
end;

{------------------------------------------------------------------------------
    Procedure: MoveTheClickedObjectClick
  Description: deals with the end of a screen operation at MouseUp
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Processes the first Clicked Object
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.MoveTheClickedObjectClick(
  Sender: TObject);
begin
  ProcessClickedObject(pClickedObject, ClickedObjectType);
end;

{------------------------------------------------------------------------------
    Procedure: MoveSecondClickedObjectClick
  Description: deals with the end of a screen operation at MouseUp
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Processes the SECOND Clicked Object
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.MoveSecondClickedObjectClick(
  Sender: TObject);
begin
  ProcessClickedObject(pSecondClickedObject, SecondClickedObjectType);
end;

{General public methods -----------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.AddSlice
  Description: Add data to the graph
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: adds data to all Axis simultaneously, avoiding re-drawing the data 
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.AddSlice(
  NoPoints: Integer;
  XYArray: pXYArray);
var
  i: Integer;
begin
  if ((NoPoints > 0) and (NoPoints <= FSeriesList.Count)) then
  begin
    if (FDisplayMode <> dmHistory) then
    begin
      for i := 0 to NoPoints-1 do
      begin
        TSeries(FSeriesList.Items[i]).AddDrawPoint(XYArray^[i].X, XYArray^[i].Y, Canvas);
      end;
    end
    else
    begin
      case FPlotType of
        ptXY:
          begin
{erase the old curve:}
            FSeriesList.DrawHistory(Canvas, FDisplayHistory);
            for i := 0 to NoPoints-1 do
            begin
              TSeries(FSeriesList.Items[i]).AddPoint(XYArray^[i].X, XYArray^[i].Y, FALSE, FALSE);
            end;
{draw the new one:}
            FSeriesList.DrawHistory(Canvas, FDisplayHistory);
          end;
        ptMultiple:
          begin
{erase the old curve:}
            FSeriesList.DrawHistory(Canvas, FDisplayHistory);
            Canvas.Pen.Assign(FMultiplePen);
            FSeriesList.DrawHistoryMultiple(Canvas, FMultiplicity);
            for i := 0 to NoPoints-1 do
            begin
              TSeries(FSeriesList.Items[i]).AddPoint(XYArray^[i].X, XYArray^[i].Y, FALSE, FALSE);
            end;
{draw the new one:}
            FSeriesList.DrawHistory(Canvas, FDisplayHistory);
            Canvas.Pen.Assign(FMultiplePen);
            FSeriesList.DrawHistoryMultiple(Canvas, FMultiplicity);
          end;
        ptColumn:
          begin
          end;
      end;
    end;
  end
  else
    EMathError.CreateFmt('AddSlice: you must add between 1 and %d points !', [FSeriesList.Count]);
end;

{------------------------------------------------------------------------------
     Function: TCustomPlot.Add
  Description: wrapper for TSeriesList.Add
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Adds a new, empty data Series to the graph
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.Add(
  XSeriesIndex: Integer): Integer;
begin
  Add := FSeriesList.Add(XSeriesIndex);
  TSeries(FSeriesList.Items[FSeriesList.Count-1]).OnChange := StyleChange;
  {Cannot make Series visible yet because memory may not be allocated:}
  TSeries(FSeriesList.Items[FSeriesList.Count-1]).Visible := TRUE;
end;

{------------------------------------------------------------------------------
     Function: TCustomPlot.AddExternal
  Description: wrapper for TSeriesList.AddExternal
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Adds a new data Series that is maintained elsewhere to the graph
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.AddExternal(
  XPointer,
  YPointer: pSingleArray;
  NumberOfPoints: Integer): Integer;
begin
  AddExternal := FSeriesList.AddExternal(XPointer, YPointer, NumberOfPoints);
  TSeries(FSeriesList.Items[FSeriesList.Count-1]).OnChange := StyleChange;
  TSeries(FSeriesList.Items[FSeriesList.Count-1]).Visible := TRUE;
end;

{------------------------------------------------------------------------------
     Function: TCustomPlot.AddInternal
  Description: wrapper for TSeriesList.AddInternal
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Adds a new data Series from elsewhere to the graph, and asves it internally
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.AddInternal(
  XPointer,
  YPointer: pSingleArray;
  NumberOfPoints: Integer): Integer;
begin
  AddInternal := FSeriesList.AddInternal(XPointer, YPointer, NumberOfPoints);
  TSeries(FSeriesList.Items[FSeriesList.Count-1]).OnChange := StyleChange;
  TSeries(FSeriesList.Items[FSeriesList.Count-1]).Visible := TRUE;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.CloneSeries
  Description: wrapper for TSeriesList.CloneSeries
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Clones the specified Series
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.CloneSeries(
  TheSeries: Integer): Integer;
begin
  CloneSeries := FSeriesList.CloneSeries(TheSeries);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DeleteSeries
  Description: wrapper for TSeriesList.DeleteSeries
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Deletes the specified Series
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DeleteSeries(
  Index: Integer);
begin
  FSeriesList.DeleteSeries(Index);
end;

{Responding to user (right) click menus -------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.CopyClick
  Description: The public copying method
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: copies the graph to the clipboard, in all formats simultaneously
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.CopyClick(
  Sender: TObject);
begin
  ClipBoardFormatForHTML := RegisterClipboardFormat('HTML');
  ClipBoard.Open;
  try
{copy all three formats to the clipboard at once:}
    CopyText;
    CopyHTML(ClipBoardFormatForHTML);
    CopyBitmap;
    CopyWMF(TRUE);
  finally
    ClipBoard.Close;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.HideClick
  Description: Hides part of the graph
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: makes the selected object invisible
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.HideClick(
  Sender: TObject);
begin
  if (Sender is TPlotMenu) then
  begin
    FInstructions.Clear;
    FInstructions.Add('Click on the object you want to hide');
    ScreenJob := sjHide;
    Refresh;
    exit;
  end;

  case ClickedObjectType of
    soTitle: FTitle.Visible := FALSE;
    soXAxis: FXAxis.Visible := FALSE;
    soYAxis: TAxis(pClickedObject).Visible := FALSE;
    soXAxisTitle: FXAxis.Title.Visible := FALSE;
    soYAxisTitle: TTitle(pClickedObject).Visible := FALSE;
    soXAxisLabel: FXAxis.Labels.Visible := FALSE;
    soYAxisLabel: TAxisLabel(pClickedObject).Visible := FALSE;
    soLegend: FLegend.Visible := FALSE;
    soResult: FResult.Visible := FALSE;
    soSeries: pSeries.Visible := FALSE;
  end;
  Refresh;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.PrintClick
  Description: The public printing method
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Print the graph
 Known Issues: We kludge this one a bit by creating a metafile, then playing it
               on the printer canvas. It would be nicer to draw directly on the
               printer canvas.
 ------------------------------------------------------------------------------}
procedure TCustomPlot.PrintClick(
  Sender: TObject);
var
  i: Integer;
  Copies: Integer;
  PrintDialog: TPrintDialog;
  PrintBorder: TRect;
  HorzSizeMM, VertSizeMM: Integer;
{for the metafile:}
  AMetafile: TMetafile;
  AMetafileCanvas: TMetafileCanvas;
begin
  Printer.Orientation := FPrintOrientation;
  PrintDialog := TPrintDialog.Create(Self);
  PrintDialog.Options := [poPrintToFile, poWarning];
  if (PrintDialog.Execute) then
  begin
    if (PrintDialog.Copies > 1) then
      Copies := PrintDialog.Copies
     else
      Copies := 1;
    Printer.Title := Application.ExeName + ' - ' + FTitle.Caption;

{$IFDEF WIN}
    HorzSizeMM := WinProcs.GetDeviceCaps(Printer.Handle, HORZSIZE);
    VertSizeMM := WinProcs.GetDeviceCaps(Printer.Handle, VERTSIZE);
{$ENDIF}

{Set the margins to 25 mm:}
    PrintBorder.Left := 25 * (Printer.PageWidth div HorzSizeMM);
    PrintBorder.Top := 25 * (Printer.PageHeight div VertSizeMM);
    PrintBorder.Right := Printer.PageWidth - PrintBorder.Left;
    PrintBorder.Bottom := Printer.PageHeight - PrintBorder.Top;

    AMetafile := TMetafile.Create;
  {$IFNDEF DELPHI1}
    AMetafile.Enhanced := TRUE;
  {$ENDIF}

  {NOTE: you _MUST_ set the height and width before doing anything !}
    AMetafile.Height := Height; {PrintBorder.Bottom - PrintBorder.Top;}
    AMetafile.Width := Width; {PrintBorder.Right - PrintBorder.Left;}

  {$IFDEF DELPHI1} 
  {create the metafile canvas to draw on:}
    AMetafileCanvas :=
      TMetafileCanvas.Create(AMetafile, 0);
  {$ELSE}
    SetMetafileDescription;
  {create the metafile canvas to draw on:}
    AMetafileCanvas :=
      TMetafileCanvas.CreateWithComment(AMetafile, 0, FCreatedBy, FDescription);
  {$ENDIF}

 {draw the graph on the metafile:}
    Draw(AMetafileCanvas);
    AMetafileCanvas.Free;

{note: the D4 TPrinter has a Copies property, but not all printers support it,
 so do printing the slow way:}
    Printer.BeginDoc;
    for i := 1 to Copies do
    begin
      if (i > 1) then
        Printer.NewPage;
      Printer.Canvas.StretchDraw(PrintBorder, AMetafile);
    end;

    AMetafile.Free;
    Printer.EndDoc;
    Refresh;
  end; {PrintDialog}
  PrintDialog.Free;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ShowAllClick
  Description: Shows/reveals all screen objects
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: makes everything visible.
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ShowAllClick(
  Sender: TObject);
var
  i: Integer;
begin
  for i := 0 to FAxisList.Count-1 do
  begin
    TAxis(FAxisList[i]).Visible := TRUE;
    TAxis(FAxisList[i]).Title.Visible := TRUE;
    TAxis(FAxisList[i]).Labels.Visible := TRUE;
  end;

  for i := 0 to FSeriesList.Count-1 do
    TSeries(FSeriesList[i]).Visible := TRUE;

  FLegend.Visible := TRUE;

  FResult.Visible := TRUE;

  Refresh;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.PositionClick
  Description: Where the hell are we ?
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Displays (and copies) the current mouse click position, in USER units
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.PositionClick(
  Sender: TObject);
var
  Msg: String;
begin
  if (Sender is TPlotMenu) then
{came via }
  begin
    ScreenJob := sjPosition;
    Screen.Cursor := crScope;
    FInstructions.Clear;
    FInstructions.Add('Click on the position you want the details of');
    Refresh;
  end
  else
  begin
    Msg := Format('The screen co-ordinates are (%d, %d),',
      [Selection.Left, Selection.Top]) + CRLF +
        'and the Position is (' +
          FXAxis.LabelToStrF(FXAxis.XofF(Selection.Left));
    if (Length(FXAxis.Title.Units) > 0) then
      Msg := Msg + ' ' + FXAxis.Title.Units;
    Msg := Msg + ', ' +
      FYAxis.LabelToStrF(FYAxis.YofF(Selection.Top));
    if (Length(FYAxis.Title.Units) > 0) then
      Msg := Msg + ' ' + FYAxis.Title.Units;
    Msg := Msg + ').';
    ShowMessage(Msg);
    ClipBoard.AsText := Msg;
    ZeroScreenStuff;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.NearestPointClick
  Description: Where the hell is it ?
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: finds the nearest point of the nearest Axis
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.NearestPointClick(
  Sender: TObject);
var
  NearestiX,
  NearestiY: Integer;
  NearestX,
  NearestY,
  MinDistance: Single;
  Msg: String;
begin
  if (Sender is TPlotMenu) then
{came via }
  begin
    ScreenJob := sjNearestPoint;
    Screen.Cursor := crX;
    FInstructions.Clear;
    FInstructions.Add('Click near the point you want');
    Refresh;
  end
  else
  begin
    ThePointNumber := FSeriesList.GetNearestPoint(Selection.Left, Selection.Top,
      TheSeries, NearestiX, NearestiY, NearestX, NearestY, MinDistance, pSeries);
    Msg := 'The nearest point is #' + IntToStr(ThePointNumber) + ' in ' +
      pSeries.Name + CRLF + 'at (' +
        FXAxis.LabelToStrF(NearestX);
    if (Length(FXAxis.Title.Units) > 0) then
      Msg := Msg + ' ' + FXAxis.Title.Units;
    Msg := Msg + ', ' +
      FYAxis.LabelToStrF(NearestY);
    if (Length(FYAxis.Title.Units) > 0) then
      Msg := Msg + ' ' + FYAxis.Title.Units;
    Msg := Msg + ').';
    with Canvas do
    begin
      Pen.Color := clRed;
      Pen.Mode := pmNotXOR;
      Pen.Style := psSolid;
      Pen.Width := 2;
      Ellipse(NearestiX-10, NearestiY-10, NearestiX+10, NearestiY+10);
    end;

    ShowMessage(Msg);
    ClipBoard.AsText := Msg;

    with Canvas do
    begin
      {Pen.Color := clBlack;
      Pen.Mode := pmNotXOR;
      Pen.Style := psSolid;}
      Ellipse(NearestiX-10, NearestiY-10, NearestiX+10, NearestiY+10);
    end;
    ZeroScreenStuff;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DeleteSeriesClick
  Description: wrapper for TSeriesList.DeleteSeries
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Deletes the selected Series
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DeleteSeriesClick(
  Sender: TObject);
begin
  if (GetSeriesFromUser) then
  begin
    FSeriesList.DeleteSeries(TheSeries);
    Refresh;
  end;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.CopySeriesClick
  Description: wrapper for TSeriesList.CopySeries
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Copies the selected Series (as text)
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.CopySeriesClick(
  Sender: TObject);
begin
  if (GetSeriesFromUser) then
    pSeries.CopyToClipBoard;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.CloneSeriesClick
  Description: wrapper for TSeriesList.CloneSeries
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Clones the selected Series
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.CloneSeriesClick(
  Sender: TObject);
begin
  if (GetSeriesFromUser) then
  begin
    FSeriesList.CloneSeries(TheSeries);
    Refresh;
  end;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ModeClick
  Description: Changes how the graph appears and reacts to new data
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the DisplayMode property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ModeClick(
  Sender: TObject);
var
  Index: Integer;
  DisplayHistoryStr: String;
  OptionsDlg: TOptionsDlg;
begin
  OptionsDlg := TOptionsDlg.Create(nil);
  with OptionsDlg do
  begin
    FormTitle := 'Display Mode ?';
    Question := 'How do you want to display the data when a new point is added?';
    OptionList.Add('Normal - expand the Axes if necessary');
    OptionList.Add('None - do nothing');
    OptionList.Add('Run - if neccessary, expand the Y-Axis but double the X-Axis');
    OptionList.Add('History - only show the most recent data');
    Index := Execute - 1;
  end;
  OptionsDlg.Free;

  if (Index >= 0) then
  begin
    if (Index = Ord(dmHistory)) then
    begin
      DisplayHistoryStr := FloatToStr(FDisplayHistory);
      if (InputQuery('History Range', '', DisplayHistoryStr)) then
      FDisplayHistory := StrToFloat(DisplayHistoryStr);
    end;
    SetDisplayMode(TDisplayMode(Index));
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ParseData
  Description: oversees the importation and pasting of data
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: runs the ParserForm, and adds the new data as new Axis
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ParseData(
  TheData: TStringList);
var
  InitialSeriesCount,
  iColumn,
  jRow,
  NoSeries,
  XSeriesCol: Integer;
  Delimiter: String;
  ParserForm: TParserForm;
  SeriesInfo: pSeriesInfoArray;
  SeriesOfCol: pIntegerArray;
begin
  InitialSeriesCount := FSeriesList.Count;
  ParserForm := TParserForm.Create(Self);
  jRow := 0;
  try
    for jRow := 0 to TheData.Count-1 do
    begin
      ParserForm.DataListBox.Items.Add(TheData.Strings[jRow]);
    end;
    jRow := 0;
  except
{the file was to big to place into the listbox:}
    ShowMessage('Minor problem: you can only look at '  +
      IntToStr(jRow-1) + ' lines of data !');
  end;

  if (ParserForm.ShowModal = mrOK) then
  begin
    Delimiter := ParserForm.Delimiters[ParserForm.TheDelimiter];
{allocate memory for the dynamic arrays, which are small:}
    GetMem(SeriesInfo, ParserForm.InfoGrid.ColCount * SizeOf(TSeriesInfo));
    GetMem(SeriesOfCol, (ParserForm.InfoGrid.ColCount+1) * SizeOf(Integer));
{this allocates more memory than SeriesInfo needs, but so what ?}

{Determine the number of series:}
    NoSeries := 0;
{examine the columns one by one:}
    for iColumn := 1 to ParserForm.InfoGrid.ColCount-1 do
    begin
      SeriesOfCol^[iColumn] := -1;
      if (ParserForm.InfoGrid.Cells[iColumn, X_OR_Y] = 'X') then
      begin
        SeriesOfCol^[iColumn] := NoSeries;
        SeriesInfo^[NoSeries].XCol := iColumn;
      end
      else if (ParserForm.InfoGrid.Cells[iColumn, X_OR_Y] = 'Y') then
      begin
{we've found a series:}
        SeriesOfCol^[iColumn] := NoSeries;
        SeriesInfo^[NoSeries].YCol := iColumn;
        if (Length(ParserForm.InfoGrid.Cells[iColumn,DEPENDS_ON_X]) > 0) then
        begin
          XSeriesCol := StrToInt(ParserForm.InfoGrid.Cells[iColumn,DEPENDS_ON_X]);
          if (SeriesOfCol^[XSeriesCol] < 0) then
            raise EComponentError.Create('TCustomPlot.ParseData: I am befuddled by the XSeriesCol !');
          if (SeriesOfCol^[XSeriesCol] = NoSeries) then
          begin
            SeriesInfo^[NoSeries].XSeriesIndex := -1;
          end
          else
          begin
            SeriesInfo^[NoSeries].XSeriesIndex :=
              SeriesOfCol^[XSeriesCol] + InitialSeriesCount;
          end;
        end;
        SeriesInfo^[NoSeries].Index :=
          FSeriesList.Add(SeriesInfo^[NoSeries].XSeriesIndex);
        TSeries(FSeriesList.Items[SeriesInfo^[NoSeries].Index]).Name :=
          ParserForm.InfoGrid.Cells[iColumn, SERIES_NAMES];

        Inc(NoSeries);
      end;
    end;

{now we try to convert all the data:}
    if (NoSeries > 0) then
      ConvertTextData(ParserForm.InfoGrid.ColCount-1, NoSeries, ParserForm.TheFirstDataLine,
        Delimiter, TheData, SeriesInfo);
    FreeMem(SeriesInfo, ParserForm.InfoGrid.ColCount * SizeOf(TSeriesInfo));
    FreeMem(SeriesOfCol, ParserForm.InfoGrid.ColCount * SizeOf(Integer));
  end; {ShowModal = mrOK}
  ParserForm.Free;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ConvertBinaryData
  Description: Adds binary data to the new Series
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Given a AxisLocationArray, converts the text data to numeric data and adds it to the new Axis
 Known Issues: This procedure assumes that TheStream.Position points to the start
               of the binary data.
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ConvertBinaryData(
  ColCount,
  SeriesCount: Integer;
  TheStream: TMemoryStream;
  SeriesInfo: pSeriesInfoArray);
var
  DataSize,
  i,
  iColumn: Integer;
  NullValue: Single;
  ptr: Pointer;
  pTheChar: PChar;
  pValues: pSingleArray;
begin
  GetMem(pValues, ColCount * SizeOf(Single));
  DataSize := SizeOf(Single);

{calculate our "ignore" value:}
  ptr := @NullValue;
  pTheChar := ptr;
  for i := 1 to DataSize do
  begin
    pTheChar^ := 'x';
    Inc(pTheChar);
  end;

  while (TheStream.Position < TheStream.Size) do
  begin
    for iColumn := 0 to ColCount-1 do
      TheStream.Read(pValues^[iColumn], DataSize);

    for i := 0 to SeriesCount-1 do
    begin
      if (SeriesInfo^[i].XCol > 0) then
        SeriesInfo^[i].XValue := pValues^[SeriesInfo^[i].XCol-1];
      SeriesInfo^[i].YValue := pValues^[SeriesInfo^[i].YCol-1];

      TSeries(FSeriesList.Items[SeriesInfo^[i].Index]).AddPoint(
        SeriesInfo^[i].XValue,
        SeriesInfo^[i].YValue,
        FALSE, FALSE);
    end; {for iColumn}
  end; {for lines of data}

{Make the new Series visible:}
  for i := 0 to SeriesCount-1 do
  begin
    if (SeriesInfo^[i].Index >= 0) then
      TSeries(FSeriesList.Items[SeriesInfo^[i].Index]).Visible := TRUE;
  end;

  FreeMem(pValues, ColCount * SizeOf(Single));
{for a subsequent SaveClick:}
  FAsText := FALSE;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ConvertTextData
  Description: Adds text data to the new Series
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Given a pSeriesInfoArray, converts the text data to numeric data and adds it to the new Axis
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ConvertTextData(
  ColCount,
  SeriesCount,
  FirstLine: Integer;
  Delimiter: String;
  TheData: TStringList;
  SeriesInfo: pSeriesInfoArray);
var
  i,
  jRow: Integer;
  TheCell,
  TheLine: String;
  pValues: pSingleArray;
begin
  GetMem(pValues, ColCount * SizeOf(Single));
  for jRow := FirstLine to TheData.Count-1 do
  begin
    TheLine := TheData.Strings[jRow];

    for i := 0 to ColCount-1 do
    begin
      TheCell := GetWord(TheLine, Delimiter);
      try
        pValues^[i] := StrToFloat(TheCell);
      finally
      end;
    end;

    for i := 0 to SeriesCount-1 do
    begin
      if (SeriesInfo^[i].XCol > 0) then
        SeriesInfo^[i].XValue := pValues^[SeriesInfo^[i].XCol-1];
      SeriesInfo^[i].YValue := pValues^[SeriesInfo^[i].YCol-1];

      TSeries(FSeriesList.Items[SeriesInfo^[i].Index]).AddPoint(
        SeriesInfo^[i].XValue,
        SeriesInfo^[i].YValue,
        FALSE, FALSE);
    end; {for i}
  end; {for lines of data}

{Make the new Series visible:}
  for i := 0 to SeriesCount-1 do
  begin
    if (SeriesInfo^[i].Index >= 0) then
      TSeries(FSeriesList.Items[SeriesInfo^[i].Index]).Visible := TRUE;
  end;

  FreeMem(pValues, ColCount * SizeOf(Single));
{for a subsequent SaveClick:}
  FAsText := TRUE;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.PasteSeriesClick
  Description: Pastes data from the Clipboard
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Collects the data from the Clipboard and runs the ParseData method on it.
 Known Issues: limited to 32 K under D1. Can be fixed with messy memory management.
 ------------------------------------------------------------------------------}
procedure TCustomPlot.PasteClick(
  Sender: TObject);
var
  TheData: TStringList;
{$IFDEF DELPHI1}
  LongStr: PChar;
{$ENDIF}
begin
  TheData := TStringList.Create;
{$IFDEF DELPHI1}
  GetMem(LongStr, 32767);
  Clipboard.GetTextBuf(LongStr, 32767);
  TheData.SetText(LongStr);
  FreeMem(LongStr, 32767);
{$ELSE}
  TheData.Text := Clipboard.AsText;
{$ENDIF}
  ParseData(TheData);
  TheData.Free;

  ZoomOutClick(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.LineBestFitClick
  Description: Initiates a Line of Best Fit.
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Sets the Instructions and the ScreenJob
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.LineBestFitClick(
  Sender: TObject);
begin
  if (GetSeriesFromUser) then
  begin
    FInstructions.Clear;
    FInstructions.Add('Click and Drag over the region to fit.');
    ScreenJob := sjLineOfBestFit;
    FResult.Visible := False;
    Refresh;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.TwoRegionLineBestFitClick
  Description: Initiates a Two Region Line of Best Fit.
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Sets the FInstructions and the ScreenJob
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.TwoRegionLineBestFitClick(Sender: TObject);
begin
  if (GetSeriesFromUser) then
  begin
    FInstructions.Clear;
    FInstructions.Add('Click and Drag over the FIRST region to fit.');
    ScreenJob := sjDualLineBestFit1;
    FResult.Visible := False;
    Refresh;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SmoothSeriesClick
  Description: Smoothes the selected data Series
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Obtains the Smoothing Order then runs the selected Series' Smooth method
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SmoothSeriesClick(
  Sender: TObject);
var
  SmoothOrder: Integer;
  SmoothStr: String;
begin
  if (GetSeriesFromUser) then
  begin
    SmoothStr := '10';
    if (InputQuery('Smoothing ' + pSeries.Name,
                   'Enter the Smoothing Order (2..20)',
                   SmoothStr)) then
    begin
      try
        SmoothOrder := StrToInt(SmoothStr);
        pSeries.Smooth(SmoothOrder);
        Refresh;
      except
        ShowMessage('Smoothing Failed !');
      end;
    end;
  end;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SortClick
  Description: sorts the selected data
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management and manipulation
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SortClick(Sender: TObject);
begin
  if (GetSeriesFromUser) then
  begin
    pSeries.Sort;
    Refresh;
  end;
  ZeroScreenStuff;
end;

procedure TCustomPlot.SplineClick(Sender: TObject);
begin
  if (GetSeriesFromUser) then
  begin
    Self.Spline(TheSeries);
    Refresh;
  end;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
     Function: TCustomPlot.Spline
  Description: wrapper for TSeriesList.Spline
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Adds a new, empty data Series to the graph
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.Spline(ASeries: Integer): Integer;
var
  Density: Word;
  pSplineSeries: TSeries;
  TheString: String;
begin
  Spline := -1;
{if it isn't already ...}
  pSeries := TSeries(FSeriesList.Items[ASeries]);

  TheString := '1';
  if (InputQuery('Cubic Spline',
    'Please enter the number of divisions',
      TheString)) then
  begin
    Density := StrToInt(TheString);

    ASeries := FSeriesList.Add(-1);
    pSplineSeries := TSeries(FSeriesList.Items[ASeries]);
    pSplineSeries.AllocateNoPts(pSeries.NoPts * (Density + 1));

    pSeries.DoSpline(Density, pSplineSeries);

    pSplineSeries.Name := 'Cubic Spline of ' + pSeries.Name;
    pSplineSeries.Pen.Style := psDot;
    pSplineSeries.OnChange := StyleChange;
    pSplineSeries.Visible := TRUE;
    Spline := ASeries;
{should we call pSeries.ClearSpline ?}
    Refresh;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ContractSeriesClick
  Description: Reduces the number of data points of the selected Series
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Obtains the Smoothing Order then runs the selected Series' Contract method
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ContractSeriesClick(
  Sender: TObject);
var
  ContractRatio: Integer;
  ContractStr: String;
begin
  if (GetSeriesFromUser) then
  begin
    if (pSeries.XDataRefCount > 0) then raise
      EComponentError.CreateFmt(
        'Cannot Contract %s !' + CRLF + '%d other series depend on it !',
        [pSeries.Name, pSeries.XDataRefCount]);

    if (pSeries.ExternalXSeries) then raise
      EComponentError.CreateFmt(
        'Cannot Contract %s !' + CRLF + 'It depends on the X Data in %s !',
        [pSeries.Name, pSeries.XDataSeries.Name]);

    ContractStr := '10';
    if (InputQuery('Contracting ' + pSeries.Name,
                   'Enter the Contraction Ratio (2..20)',
                   ContractStr)) then
    begin
      try
        ContractRatio := StrToInt(ContractStr);
        pSeries.Contract(ContractRatio);
      except
        ShowMessage('Contraction Failed !');
      end;
    end;
    Refresh;
  end;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ContractAllSeriesClick
  Description: Reduces the number of data points
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Obtains the Smoothing Order then runs ALL Series' Contract method
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ContractAllSeriesClick(
  Sender: TObject);
var
  ContractRatio: Integer;
  ContractStr: String;
  i: Integer;
begin
  ContractStr := '10';
  if (InputQuery('Contracting ALL Series',
                 'Enter the Contracting Ratio (2..20)',
                 ContractStr)) then
  begin
    try
      ContractRatio := StrToInt(ContractStr);
      for i := 0 to FSeriesList.Count-1 do
      begin
        TSeries(FSeriesList.Items[i]).Contract(ContractRatio);
      end;
    except
      ShowMessage('Contraction Failed !');
    end;
  end;
  Refresh;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.LegendClick
  Description: Sets the Legend Direction
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose:
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.LegendClick(
  Sender: TObject);
var
  Index: Integer;
  OptionsDlg: TOptionsDlg;
begin
  OptionsDlg := TOptionsDlg.Create(nil);
  with OptionsDlg do
  begin
    FormTitle := 'Legend direction ?';
    Question := 'How do you want to display the Legend ?';
    OptionList.Add('Horizontally');
    OptionList.Add('Vertically');
    Index := Execute - 1;
  end;
  OptionsDlg.Free;

  if (Index >= 0) then
  begin
    FLegend.Direction := TDirection(Index);
  end;
  ZeroScreenStuff;
  Refresh;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.EditAxisClick
  Description: Runs the Axis Editor of the selected Axis
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 06/28/2000 by Mat Ballard
      Purpose:
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.EditAxisClick(
  Sender: TObject);
var
  i: Integer;
  AxisEditor: TAxisEditorForm;
  AXP: TAxisProperty;
  pAXP: ^TAxisProperty;
begin
  if (GetAxisFromUser(0)) then
  begin
    AxisEditor := TAxisEditorForm.Create(nil);

    if (FDisplayMode = dmHistory) then
      AxisEditor.HistoryMode := TRUE;

{Iterate over all axes:}
    for i := 0 to FAxisList.Count-1 do
    begin
      pAxis := TAxis(FAxisList.Items[i]);
      AXP.LabelFormat := pAxis.Labels.NumberFormat;
      AXP.LabelDigits := pAxis.Labels.Digits;
      AXP.LabelPrecision := pAxis.Labels.Precision;
      AXP.PenColor := IndexOfColorValue(pAxis.Pen.Color);
      AXP.PenWidthIndex := pAxis.Pen.Width;
      AXP.PenStyleIndex := Ord(pAxis.Pen.Style);
      AXP.TickSize := pAxis.TickSize;
      AXP.TickDirection := pAxis.TickDirection;
      AXP.TickStepSize := pAxis.StepSize;
      AXP.TickMinors := pAxis.TickMinor;
      AXP.ScaleMin := pAxis.Min;
      AXP.ScaleMax := pAxis.Max;
      AXP.ScaleIntercept := pAxis.Intercept;
      AXP.ScaleAuto := pAxis.AutoScale;
      AXP.ScaleLog := pAxis.LogScale;
      AXP.ArrowSize := pAxis.ArrowSize;
      AXP.ArrowDirection := pAxis.Alignment;
      AXP.Visible := pAxis.Visible;
      AxisEditor.AddAxis(pAxis.Title.Caption, AXP);
    end;

    if (TheAxis >= 0) then
      AxisEditor.NoComboBox.ItemIndex := TheAxis;
    AxisEditor.SelectAxis(TheAxis);

    if (AxisEditor.ShowModal = mrOK) then
    begin
      for i := 0 to FAxisList.Count-1 do
      begin
        pAXP := AxisEditor.AxisPropertyList.Items[i];
        pAxis := TAxis(FAxisList.Items[i]);
        pAxis.Title.Caption := AxisEditor.AxisNames.Strings[i];
        pAxis.Labels.NumberFormat := pAXP^.LabelFormat;
        pAxis.Labels.Digits := pAXP^.LabelDigits;
        pAxis.Labels.Precision := pAXP^.LabelPrecision;
        pAxis.Pen.Color := MyColors[pAXP^.PenColor].Value;
        pAxis.Pen.Width := pAXP^.PenWidthIndex;
        pAxis.Pen.Style := TPenStyle(pAXP^.PenStyleIndex);
        pAxis.TickSize := pAXP^.TickSize;
        pAxis.TickDirection := pAXP^.TickDirection;
        pAxis.TickMinor := pAXP^.TickMinors;
        pAxis.AutoScale := pAXP^.ScaleAuto;
        pAxis.StepSize := pAXP^.TickStepSize;
        pAxis.Min := pAXP^.ScaleMin;
        pAxis.Max := pAXP^.ScaleMax;
        pAxis.Intercept := pAXP^.ScaleIntercept;
        pAxis.LogScale := pAXP^.ScaleLog;
        pAxis.ArrowSize := pAXP^.ArrowSize;
        pAxis.Alignment := pAXP^.ArrowDirection;
        pAxis.Visible := pAXP^.Visible;
      end; {for}
      StyleChange(Self);
    end; {mrOK}
    AxisEditor.Free;

    ZeroScreenStuff;
    Refresh;
  end; {GetAxisFrom User}
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.EditFontClick
  Description: Edits the font of the selected Axis
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs the Font common Dialog box, and applies the results to the selected object
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.EditFontClick(
  Sender: TObject);
var
  TheFont: TFont;
  FontDialog: TFontDialog;
  OptionsDlg: TOptionsDlg;
  TheResult: Integer;
begin
{has the user already selected an object ?}
  if ((ClickedObjectType = soNone) or
      (ClickedObjectType = soXAxis) or
      (ClickedObjectType = soYAxis) or
      (ClickedObjectType = soLeftBorder) or
      (ClickedObjectType = soTopBorder) or
      (ClickedObjectType = soRightBorder) or
      (ClickedObjectType = soBottomBorder)) then
  begin
{get the user to select an object:}
    OptionsDlg := TOptionsDlg.Create(nil);
    with OptionsDlg do
    begin
      FormTitle := 'Edit which Font ?';
      Question := 'Which Font you want to edit ?';
      OptionList.Add('Plot Title');
      OptionList.Add('X-Axis Title');
      OptionList.Add('Y-Axis Title');
      OptionList.Add('Secondary Y-Axis Title');
      OptionList.Add('X-Axis Labels');
      OptionList.Add('Y-Axis Labels');
      OptionList.Add('Secondary Y-Axis Labels');
      OptionList.Add('Legend');
      OptionList.Add('Result');

      TheResult := Execute;
      case TheResult of
        1: ClickedObjectType := soTitle;
        2: ClickedObjectType := soXAxisTitle;
        3: ClickedObjectType := soYAxisTitle;
        5: ClickedObjectType := soXAxisLabel;
        6: ClickedObjectType := soYAxisLabel;
        8: ClickedObjectType := soLegend;
        9: ClickedObjectType := soResult;      
      end;
    end;
    OptionsDlg.Free;
  end; {if object selected}

  FontDialog := TFontDialog.Create(Self);

{assign TheFont:}
  case ClickedObjectType of
    soTitle: TheFont := FTitle.Font;
    soXAxisTitle: TheFont := FXAxis.Title.Font;
    soYAxisTitle: TheFont := TTitle(pClickedObject).Font;
    soXAxisLabel: TheFont := FXAxis.Labels.Font;
    soYAxisLabel: TheFont := TAxisLabel(pClickedObject).Font;
    soLegend: TheFont := FLegend.Font;
    soResult: TheFont := FResult.Font;
  else
    FontDialog.Free;
{don't like bugging out here, but is most elegant solution:}
    exit;
  end;
  FontDialog.Font.Assign(TheFont);

  if (FontDialog.Execute) then
  begin
    TheFont.Assign(FontDialog.Font);
    Refresh;
  end;

  ZeroScreenStuff;
  FontDialog.Free;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.PropertiesClick
  Description: Edits the other properties of the Plot
       Author: Mat Ballard
 Date created: 10/10/2000
Date modified: 10/10/2000 by Mat Ballard
      Purpose: Runs the Properties Dialog box, and applies the results to the selected objects
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.PropertiesClick(
  Sender: TObject);
var
  PlotPropertyEditor: TPlotPropertyEditorForm;
begin
  PlotPropertyEditor := TPlotPropertyEditorForm.Create(nil);

  PlotPropertyEditor.PlotTypeComboBox.ItemIndex := Ord(FPlotType);
  PlotPropertyEditor.MultiplicityEdit.Text := IntToStr(FMultiplicity);
  PlotPropertyEditor.BackColorComboBox.ItemIndex := IndexOfColorValue(Color);
  PlotPropertyEditor.PenColorComboBox.ItemIndex := IndexOfColorValue(FMultiplePen.Color);
  PlotPropertyEditor.PenWidthComboBox.ItemIndex := FMultiplePen.Width;
  PlotPropertyEditor.PenStyleComboBox.ItemIndex := Ord(FMultiplePen.Style);
  PlotPropertyEditor.CreatedByEdit.Text := FCreatedBy;
  PlotPropertyEditor.DescriptionEdit.Text := FDescription;

  if (PlotPropertyEditor.ShowModal = mrOK) then
  begin
    StyleChange(Self);
  end; {mrOK}
  PlotPropertyEditor.Free;

  ZeroScreenStuff;
  Refresh;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.EditPointClick
  Description: Runs the Point Editor of the selected data point
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose:
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.EditPointClick(
  Sender: TObject);
begin
  if (GetSeriesFromUser) then
  begin
    pSeries.EditPoint(ThePointNumber);
    Refresh;
  end;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.GetSeriesFromUser
  Description: Gets the user to select (if not already done so) a Axis
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: user interface management
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.GetSeriesFromUser: Boolean;
var
  OptionsDlg: TOptionsDlg;
  i: Integer;
begin
{has the user already selected an object ?}
  if (ClickedObjectType <> soSeries) then
  begin
    ThePointNumber := 0;
    if (FSeriesList.Count = 1) then
    begin
{there is only one Axis:}
      ClickedObjectType := soSeries;
      TheSeries := 0;
      pSeries := TSeries(FSeriesList.Items[0]);
    end;

    if (ClickedObjectType <> soSeries) then
  {still no Series selected:}
    begin
  {get the user to select an object:}
      OptionsDlg := TOptionsDlg.Create(nil);
      OptionsDlg.FormTitle := 'Which Series ?';
      OptionsDlg.Question := 'Which Series you want to work on ?';
      for i := 0 to FSeriesList.Count-1 do
      begin
        OptionsDlg.OptionList.Add(TSeries(FSeriesList.Items[i]).Name);
      end;
      TheSeries := OptionsDlg.Execute - 1;
      if (TheSeries >= 0) then
      begin
        ClickedObjectType := soSeries;
        pSeries := TSeries(FSeriesList.Items[TheSeries]);
      end;
      OptionsDlg.Free;
    end; {if object selected}
  end; {if clicked object is Series}
  if (TheSeries >= 0) then
    GetSeriesFromUser := TRUE
   else
    GetSeriesFromUser := FALSE;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.GetAxisFromUser
  Description: Gets the user to select (if not already done so) a Axis
       Author: Mat Ballard
 Date created: 06/25/1999
Date modified: 06/25/2000 by Mat Ballard
      Purpose: user interface management
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.GetAxisFromUser(StartAxis: Word): Boolean;
var
  OptionsDlg: TOptionsDlg;
  i,
  TheResult: Integer;
begin
{has the user already selected an object ?}
{see GetTheClickedObject}
  if ((TheAxis >= StartAxis) and (pAxis <> nil)) then
  begin
    GetAxisFromUser := TRUE;
    exit;
  end;

  if (StartAxis = FAxisList.Count - 1) then
  begin
{there is only one Axis that it could be:}
    ClickedObjectType := soYAxis;
{NB: the Y Axes are numbered 1, 2..N:}
    TheAxis := StartAxis;
    pAxis := TAxis(FAxisList.Items[TheAxis]);
    GetAxisFromUser := TRUE;
    exit;
  end;

{still no Axis selected:}
{get the user to select an object:}
  OptionsDlg := TOptionsDlg.Create(nil);
  with OptionsDlg do
  begin
    FormTitle := 'Which Axis ?';
    Question := 'Which Axis you want to work on ?';
    for i := StartAxis to FAxisList.Count-1 do
    begin
      OptionList.Add(TAxis(FAxisList.Items[i]).Name);
    end;

    TheResult := Execute;
    if (TheResult > 0) then
    begin
      TheAxis := StartAxis + TheResult - 1; {Execute = -1, 1,2,3}
      if (TheAxis = 0) then
        ClickedObjectType := soXAxis;
      if (TheAxis > 0) then
        ClickedObjectType := soYAxis;
      pAxis := TAxis(FAxisList.Items[TheAxis]);
    end;
  end;
  OptionsDlg.Free;

  if (TheAxis >= 0) then
    GetAxisFromUser := TRUE
   else
    GetAxisFromUser := FALSE;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.EditSeriesClick
  Description: Runs the Series Editor of the selected data Series
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose:
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.EditSeriesClick(
  Sender: TObject);
var
  i: Integer;
  SeriesEditor: TSeriesEditorForm;
  ASP: TSeriesProperty;
  pASP: ^TSeriesProperty;
  pSeries: TSeries;
begin
  if (GetSeriesFromUser) then
  begin
    SeriesEditor := TSeriesEditorForm.Create(nil);

    with SeriesEditor do
    begin
  {Load the Y Axis Combo Box:}
      for i := 1 to FAxisList.Count-1 do
      begin
        YAxisComboBox.Items.Add(TAxis(FAxisList.Items[i]).Title.Caption);
      end;

  {Iterate over all series:}
      for i := 0 to FSeriesList.Count-1 do
      begin
        pSeries := TSeries(FSeriesList.Items[i]);
  {returns 0..MY_COLORS_MAX}
        ASP.PenColor := pSeries.Pen.Color;
        ASP.PenWidthIndex := pSeries.Pen.Width;
        ASP.PenStyleIndex := Ord(pSeries.Pen.Style);
        ASP.BrushColor := pSeries.Brush.Color;
        ASP.BrushStyleIndex := Ord(pSeries.Brush.Style);
        ASP.SymbolIndex := Ord(pSeries.Symbol);
        ASP.SymbolSize := pSeries.SymbolSize;
        ASP.YAxisIndex := pSeries.YAxisIndex;
        ASP.DeltaX := pSeries.DeltaX;
        ASP.DeltaY := pSeries.DeltaY;
        ASP.XDataIndependent := not pSeries.ExternalXSeries;
        ASP.ExternalXSeries := pSeries.ExternalXSeries;
        ASP.Visible := pSeries.Visible;
        AddSeries(pSeries.Name, ASP);
      end;
      NoComboBox.ItemIndex := TheSeries;
      SelectSeries(TheSeries);

      if (ShowModal = mrOK) then
      begin
        for i := 0 to FSeriesList.Count-1 do
        begin
          pASP := SeriesPropertyList.Items[i];
          pSeries := TSeries(FSeriesList.Items[i]);
          pSeries.Pen.Color := pASP^.PenColor;
          pSeries.Pen.Width := pASP^.PenWidthIndex;
          pSeries.Pen.Style := TPenStyle(pASP^.PenStyleIndex);
          pSeries.Brush.Color := ASP.BrushColor;
          pSeries.Brush.Style := TBrushStyle(ASP.BrushStyleIndex);
          pSeries.Symbol := TSymbol(pASP^.SymbolIndex);
          pSeries.SymbolSize := pASP^.SymbolSize;
          pSeries.YAxisIndex := pASP^.YAxisIndex;
          pSeries.DeltaX := pASP^.DeltaX;
          pSeries.DeltaY := pASP^.DeltaY;
          pSeries.Visible := pASP^.Visible;
          pSeries.Name := SeriesNames.Strings[i];
          if ((pASP^.XDataIndependent) and (pSeries.ExternalXSeries)) then
          begin
{this series did depend on X Data in another series,
 but user wants to make it independent:}
            pSeries.MakeXDataIndependent;
          end;
        end; {for}
        StyleChange(Self);
      end; {mrOK}
    end; {with SeriesEditor}
    SeriesEditor.Free;

    Refresh;
  end;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ResetDisplacementClick
  Description: Puts the selected Axis back where it came from
       Author: Mat Ballard
 Date created: 12/1/1999
Date modified: 02/25/2000 by Mat Ballard
      Purpose: sets the DeltaX and DeltaY properties of the selected Axis to ZeroScreenStuff
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ResetDisplacementClick(
  Sender: TObject);
begin
  if (GetSeriesFromUser) then
  begin
    pSeries.DeltaX := 0;
    pSeries.DeltaY := 0;
    Refresh;
  end;
  ZeroScreenStuff;
end;

{File manipulation ------------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.NewDataClick
  Description: Creates a new, blank Graph
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: data management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.NewClick(
  Sender: TObject);
begin
  Self.Clear(TRUE);
end;

{------------------------------------------------------------------------------
    Procedure: TPlot.Notification
  Description: needed for D1
       Author: Mat Ballard
 Date created: 09/07/2000
Date modified: 09/07/2000 by Mat Ballard
      Purpose:
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  {if (Operation = opRemove) and (AComponent = Plot) then
  begin
    Plot := nil;
  end;}
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.OpenClick
  Description: Opens a file on disk
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs the File Open common dialog then the LoadFromFile method
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.OpenClick(
  Sender: TObject);
var
  OpenDialog: TOpenDialog;
begin
{We have to display a File Open Dialog:}
  OpenDialog := TOpenDialog.Create(Self);
  OpenDialog.Title := 'Open';
  OpenDialog.Filter := FileTypes;
  OpenDialog.Options := [ofOverwritePrompt];
  if (Length(FFileName) = 0) then
  begin
    FileName := '*.' + FDefaultExtension;
  end;

  OpenFilterIndex := GetFilterIndex(GetFileExtension);
  OpenDialog.FilterIndex := OpenFilterIndex;

  OpenDialog.FileName := '*.' + GetFileExtension;

  if (Length(OpenDriveDir) > 0) then
    OpenDialog.InitialDir := ExtractFileName(OpenDriveDir)
   else
    OpenDialog.InitialDir := GetFileDriveDir;

  if (OpenDialog.Execute) then
  begin
    OpenFile(OpenDialog.FileName);
    OpenFilterIndex := OpenDialog.FilterIndex;
  end;
  OpenDialog.Free;

  ZoomOutClick(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.OpenFile
  Description: Opens a file on disk
       Author: Mat Ballard
 Date created: 08/12/2000
Date modified: 08/12/2000 by Mat Ballard
      Purpose: Called from TPlotMenu.HandleFileClick
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.OpenFile(
  TheFile: String);
begin
  if (FileExists(TheFile)) then
  begin
{Delete any existing Series:}
    Clear(FALSE);
    FileName := TheFile;
    OpenDriveDir := ExtractFilePath(FFileName);
{Finally, Open it:}
    LoadFromFile(FFileName);

    DoFileOpen(FFileName);
  end;

  ZoomOutClick(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DoFileOpen
  Description: Fires the OnFileOpen event
       Author: Mat Ballard
 Date created: 09/07/2000
Date modified: 09/07/2000 by Mat Ballard
      Purpose:
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DoFileOpen(AFileName: String);
begin
  if Assigned(FOnFileOpen) then
    OnFileOpen(Self, AFileName);
end;


{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ClearOverlaysClick
  Description: Gets rid of the overlaid data
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs the FSeriesList.Delete method for each overlaid Axis
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ClearOverlaysClick(
  Sender: TObject);
var
  i: Integer;
begin
  if (FirstOverlay < 0) then raise
    EComponentError.Create('There are no Overlays to Clear !');

  for i := FSeriesList.Count-1 downto FirstOverlay do
  begin
    TSeries(FSeriesList.Items[i]).Free;
    FSeriesList.Delete(i);
  end;
  FirstOverlay := -1;
  Refresh;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.OverlayDataClick
  Description: Overlays data
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs the Open common dialog, then LoadFromFile the selected files
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.OverlayClick(
  Sender: TObject);
var
  i: Integer;
  OverlayDialog: TOpenDialog;
begin
{We have to display a File Overlay Dialog:}
  OverlayDialog := TOpenDialog.Create(Self);
  OverlayDialog.Title := 'Overlay Data As';
  OverlayDialog.Filter := FileTypes;
  OverlayDialog.Options :=
    [ofFileMustExist, ofPathMustExist, ofAllowMultiSelect];

  OverlayDialog.FileName := '*.' + FDefaultExtension;

  OverlayDialog.FilterIndex := OpenFilterIndex;

  if (Length(OverlayDriveDir) > 0) then
    OverlayDialog.InitialDir := ExtractFileName(OpenDriveDir)
   else
    OverlayDialog.InitialDir := GetFileDriveDir;

  if (OverlayDialog.Execute) then
  begin
    FirstOverlay := FSeriesList.Count;
    for i := 0 to OverlayDialog.Files.Count - 1 do
    begin
      OverlayDriveDir := ExtractFileName(OverlayDialog.Files.Strings[i]);
{Finally, Overlay it:}
      Self.LoadFromFile(OverlayDialog.Files.Strings[i]);
    end;
  end;
  OverlayDialog.Free;

  ZoomOutClick(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SaveImageClick
  Description: saves the current plot as an image
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: responds to "Save Image" menu selection
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SaveImageClick(Sender: TObject);
var
  SaveImageDialog: TSaveDialog;
  Extension, ImageName: String;
begin
{We have to display a File Save Dialog:}
  SaveImageDialog := TSaveDialog.Create(Self);
  SaveImageDialog.Title := 'Save Image As';
  SaveImageDialog.Filter := PICTURE_TYPES;
  SaveImageDialog.Options := [ofOverwritePrompt];

  if (Length(ImageDriveDir) = 0) then
    if (Length(FFileName) > 0) then
      ImageDriveDir := GetFileDriveDir;

  SaveImageDialog.FilterIndex := ImageFilterIndex;
{which starts off at zero, then  may change}

  if (Length(FFileName) > 0) then
  begin
    SaveImageDialog.FileName :=
      GetFileRoot + '.' + ImageExtensions[ImageFilterIndex-1];
  end
  else
  begin
    SaveImageDialog.FileName :=
      '*.' + ImageExtensions[ImageFilterIndex-1];
  end;

  SaveImageDialog.InitialDir := ImageDriveDir;

  if (SaveImageDialog.Execute) then
  begin
    ImageName := SaveImageDialog.FileName;
    ImageDriveDir := ExtractFilePath(ImageName);

    Extension := LowerCase(ExtractFileExt(ImageName));
    if (Length(Extension) = 0) then
    begin
      ImageName := ImageName + '.' +
        ImageExtensions[SaveImageDialog.FilterIndex-1];
    end;

{Finally, save it:}
{We base this on the extension, rather than FilterIndex:}
    if (Extension = '.wmf') then
    begin
      ImageFilterIndex := 1;
      SaveAsWMF(ImageName, FALSE);
    end
    else if (Extension = '.emf') then
    begin
      ImageFilterIndex := 2;
      SaveAsWMF(ImageName, TRUE);
    end
{$IFDEF GIF}
    else if (Extension = '.gif') then
    begin
      ImageFilterIndex := 4;
      SaveAsGIF(ImageName);
    end
{$ENDIF}
{$IFDEF PNG}
    else if (Extension = '.png') then
    begin
      ImageFilterIndex := 5;
      SaveAsPNG(ImageName);
    end
{$ENDIF}
    else if (Extension = '.bmp') then
    begin
      ImageFilterIndex := 3;
      SaveAsBitMap(ImageName);
    end;
  end;
  SaveImageDialog.Free;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SaveClick
  Description: Saves the graph
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs SaveToFile or SaveAsClick
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SaveClick(Sender: TObject);
begin
  if (Length(FFileName) > 0) then
    SaveToFile(FFileName, FAsText)
   else
    SaveAsClick(Sender);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SaveAsClick
  Description: Saves the data to disk
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs the Save common dialog box then SaveToFile
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SaveAsClick(Sender: TObject);
var
  Ext: String;
  SaveDialog: TSaveDialog;
begin
{We have to display a File Save Dialog:}
  SaveDialog := TSaveDialog.Create(Self);
  SaveDialog.Title := 'Save Data As';
  SaveDialog.Filter := FileTypes;
  SaveDialog.Options := [ofOverwritePrompt];
  if (Length(FFileName) = 0) then
  begin
    FileName := '*.' + FDefaultExtension;
  end;

  SaveFilterIndex := GetFilterIndex(GetFileExtension);
  SaveDialog.FilterIndex := SaveFilterIndex;

  SaveDialog.InitialDir := GetFileDriveDir;
  SaveDialog.FileName := ExtractFileName(FFileName);

  if (SaveDialog.Execute) then
  begin
    FileName := SaveDialog.FileName;
    Ext := GetFileExtension;
    if (Length(Ext) = 0) then
    begin
      Ext := FileExtensions[SaveDialog.FilterIndex];
      FileName := FileName + '.' + Ext;
    end;

{Double-whammy problem: save with 'plot'extension, but different filter,
 should save in that (text) format.
 Save with other extension, but FilterIndex=0 (plot type), then extension
 should override, so also save in text.}

    if ((Ext = FileExtensions[0]) and (SaveDialog.FilterIndex = 1)) then
    begin {.plot}
      SaveFilterIndex := 1;
      FAsText := FALSE or (soAsText in FSaveOptions);
    end {.plot}
    else if ((Ext = FileExtensions[0]) and (SaveDialog.FilterIndex > 1)) then
    begin {'plot' extension, text type}
      SaveFilterIndex := 1;
      FAsText := TRUE;
    end {'plot' extension, text type}
    else if (Ext = FileExtensions[1]) then
    begin {.csv}
      SaveFilterIndex := 2;
      FAsText := TRUE;
    end {.csv}
    else if (Ext = FileExtensions[2]) then
    begin
      SaveFilterIndex := 3;
      FAsText := TRUE;
    end {.txt}
    else
    begin
      SaveFilterIndex := 4;
      FAsText := TRUE;
    end; {.*}

{Finally, save it:}
    Self.SaveToFile(FFileName, FAsText);
    SaveFilterIndex := SaveDialog.FilterIndex;
  end;
  SaveDialog.Free;
end;

{Saving data to disk --------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.LoadFromFile
  Description: Opens data on disk
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Opens data, parses it, fires the OnHeader event, and runs ConvertTextData,
               or decides to run it through ParseData instead
 Known Issues: Can be called by either OpenClick or OverlayClick
 ------------------------------------------------------------------------------}
procedure TCustomPlot.LoadFromFile(
  AFileName: String);
var
  ColCount,
  FileVersion,
  i,
  iColumn,
  InitialSeriesCount,
  LineLength,
  NoSeries: Integer;
  TheLine,
  SeriesNameLine,
  AxisNameLine,
  DataTypeLine,
  XDataSeriesLine,
  TheCell: String;
  TheStream: TMemoryStream;
  TheStrings: TStringList;
  SeriesInfo: pSeriesInfoArray;
  SeriesOfCol: pIntegerArray;

  procedure CleanUp;
  begin
    if (SeriesInfo <> nil) then
      FreeMem(SeriesInfo, ColCount * SizeOf(TSeriesInfo));
    SeriesInfo := nil;
    if (SeriesOfCol <> nil) then
      FreeMem(SeriesOfCol, ColCount * SizeOf(Integer));
    SeriesOfCol := nil;
    if (TheStream <> nil) then
      TheStream.Free;
    TheStream := nil;
    if (TheStrings <> nil) then
      TheStrings.Free;
    TheStrings := nil;
  end;

  function ReadLine: String;
  var
    pLine: array [0..1023] of char;
  begin
    LineLength := GetLineLengthFromStream(TheStream);
{get the line of text:}
{$IFDEF DELPHI1}
    TheStream.Read(pLine, LineLength);
    Result := StrPas(pLine);
{$ELSE}
    SetString(Result, PChar(nil), LineLength);
    TheStream.Read(Pointer(Result)^, LineLength);
{$ENDIF}
{get the CRLF:}
    TheStream.Read(pLine, 2);
  end;

  function IsPlotFile: Boolean;
  begin
    IsPlotFile := FALSE;
{Line the first:}
    TheLine := ReadLine;
    if (Pos('TPlot', TheLine) = 0) then exit;

{Line the second:}
    TheLine := ReadLine;
    if (Pos('FileFormat', TheLine) = 0) then exit;
    GetWord(TheLine, '=');
    FileVersion := StrToInt(TheLine);
    if (FileVersion > MAX_FILE_VERSION) then exit;

    FTitle.Caption := ReadLine;

{Now comes the developer-defined header:}
    DoHeader(TheStream);

    TheLine := ReadLine;
    if (TheLine <> SUBHEADER) then
    begin
{either a stuffed file, or the developer has done a naughty
 and overrun his own header; we therefore try to find it from the beginning,
 then reset the srteam position:}
      TheStream.Position := 0;
      if (not (FindStringInStream(SUBHEADER, TheStream))) then exit;
      TheLine := ReadLine;
    end;
    IsPlotFile := TRUE;
  end;

begin
  ColCount := 1;
  SeriesInfo := nil;
  SeriesOfCol := nil;
  TheStrings := nil;
  TheStream := nil;

  InitialSeriesCount := FSeriesList.Count;
  try
    TheStream := TMemoryStream.Create;

{open it:}
    TheStream.LoadFromFile(AFileName);

    if (IsPlotFile) then
    begin
{get the sub-header data:}
      SeriesNameLine := ReadLine;
      AxisNameLine := ReadLine;
      DataTypeLine := ReadLine;
      XDataSeriesLine := ReadLine;

{find out how many columns there are:}
      for i := 1 to Length(DataTypeLine) do
        if (DataTypeLine[i] = ',') then
          Inc(ColCount);

      if (ColCount < 2) then raise
        EFOpenError.CreateFmt(AFileName + ' only has %d columns !', [ColCount]);

{allocate memory for the dynamic arrays, which are small:}
      GetMem(SeriesInfo, ColCount * SizeOf(TSeriesInfo));
      GetMem(SeriesOfCol, (ColCount+1) * SizeOf(Integer));
{this allocates more memory than SeriesInfo needs, but so what ?}

{Determine the number of series:}
      NoSeries := 0;
      for i := 0 to ColCount-1 do
        SeriesInfo^[i].XCol := 0;
{examine the columns one by one:}
      for iColumn := 1 to ColCount do
      begin
{No new column yet belongs to a Series:}
        SeriesOfCol^[iColumn] := -1;
        TheCell := GetWord(DataTypeLine, ',');
        if (TheCell = 'X') then
        begin
{we've found a X data column to add.}
          SeriesOfCol^[iColumn] := NoSeries;
          SeriesInfo^[NoSeries].XCol := iColumn;
          GetWord(XDataSeriesLine, ',');
          GetWord(SeriesNameLine, ',');
        end
        else if (TheCell = 'Y') then
        begin
{we've found a Y data column to add.}
          SeriesInfo^[NoSeries].YCol := iColumn;
{find the X Column that this Y column uses:}
          TheCell := GetWord(XDataSeriesLine, ',');
          if (TheCell = '-') then
          begin
            SeriesInfo^[NoSeries].XSeriesIndex := -1;
          end
          else
          begin
            SeriesInfo^[NoSeries].XSeriesIndex :=
              StrToInt(TheCell) + InitialSeriesCount;
          end;
          SeriesInfo^[NoSeries].Index :=
            FSeriesList.Add(SeriesInfo^[NoSeries].XSeriesIndex);
          TSeries(FSeriesList.Items[SeriesInfo^[NoSeries].Index]).Name :=
            GetWord(SeriesNameLine, ',');
          Inc(NoSeries);
        end; {found a Y data column}
      end; {for}

{Get the type of data:}
      TheLine := ReadLine;
      GetWord(TheLine, '=');
{'Binary=X': X=0 => Text, X=1 => Binary:}
      i := StrToInt(TheLine);

{now we try to convert all the data:}
      if (i = 0) then
      begin {text}
        TheStrings := TStringList.Create;
{although not documented, TStrings.LoadFromStream starts from
 the CURRENT TStream.Position ! Ain't that nice !}
        TheStrings.LoadFromStream(TheStream);
        ConvertTextData(ColCount, NoSeries, 0, ',', TheStrings, SeriesInfo);
      end
      else if (i = 1) then
      begin {binary}
        ConvertBinaryData(ColCount, NoSeries, TheStream, SeriesInfo);
      end
      else
      begin
        raise EFOpenError.Create(AFileName + ' has no binary marker !');
      end;

{new data has not changed:}
      FSeriesList.DataChanged := FALSE;
    end
    else
    begin
{maybe it's just a text file:}
      TheStream.Seek(0, soFromBeginning);
      TheStrings := TStringList.Create;
      TheStrings.LoadFromStream(TheStream);
      ParseData(TheStrings);
    end; {IsPlotFile}
  except
    CleanUp;
    raise;
  end; {try}

  CleanUp;
  OpenProperties('');
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DoHeader
  Description: Informs the user of a user-defined file header
       Author: Mat Ballard
 Date created: 09/07/2000
Date modified: 09/07/2000 by Mat Ballard
      Purpose: Fires the OnHeader event
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DoHeader(TheStream: TMemoryStream);
begin
  if assigned(FOnHeader) then
    OnHeader(Self, TheStream);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SaveToFile
  Description: Saves the data as text
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 08/06/2000 by Mat Ballard
      Purpose: Fires the OnHeaderRequest event, then the GetData of SeriesList
 Known Issues:
      Comment: Note that we now use a TStream.
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SaveToFile(
  AFileName: String;
  FAsText: Boolean);
var
  TheStream: TMemoryStream;
  pLine: array [0..1023] of char;
begin
  TheStream := TMemoryStream.Create;

{D1 does not like Pointer(TheLine)^:
  TheLine := 'TPlot=' + IntToStr(TPLOT_VERSION) + CRLF;
  TheStream.Write(Pointer(TheLine)^, Length(TheLine));
so:}
  StrPCopy(pLine, 'TPlot=' + IntToStr(TPLOT_VERSION) + CRLF);
  TheStream.Write(pLine, StrLen(pLine));
  StrPCopy(pLine, 'FileFormat=' + IntToStr(FILE_FORMAT_VERSION) + CRLF);
  TheStream.Write(pLine, StrLen(pLine));

  StrPCopy(pLine, FTitle.Caption + CRLF);
  TheStream.Write(pLine, StrLen(pLine));

  DoHeaderRequest(TheStream);

  StrPCopy(pLine, SUBHEADER + CRLF);
  TheStream.Write(pLine, StrLen(pLine));

  FSeriesList.GetSubHeaderStream(',', TheStream);

  if (FAsText) then
    StrPCopy(pLine, 'Binary=0' + CRLF)
   else
    StrPCopy(pLine, 'Binary=1' + CRLF);
  TheStream.Write(pLine, StrLen(pLine));

{create the data in binary or text format:}
  FSeriesList.GetStream(FAsText, ',', TheStream);

{determine the file name:}
  if (Length(AFileName) > 0) then
    SetFileName(AFileName);

{save it:}
  TheStream.SaveToFile(FFileName);

  TheStream.Free;

  if (soProperties in FSaveOptions) then
    SaveTheProperties('');
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DoHeaderRequest
  Description: Asks the user for a header
       Author: Mat Ballard
 Date created: 09/07/2000
Date modified: 09/07/2000 by Mat Ballard
      Purpose: Fires the OnHeaderRequest event
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DoHeaderRequest(TheStream: TMemoryStream);
begin
  if assigned(FOnHeaderRequest) then
    OnHeaderRequest(Self, TheStream);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.OpenProperties
  Description: Opens the properties of this instance of TPlot
       Author: Mat Ballard
 Date created: 08/03/2000
Date modified: 08/03/2000 by Mat Ballard
      Purpose: Saves the appearance of the Plot
 Known Issues:
      Comment: Note that if AFileName is blank, we use a name generated from
               the FileName property (in SetFileName).
 ------------------------------------------------------------------------------}
procedure TCustomPlot.OpenProperties(AFileName: String);
var
  FileStream: TFileStream;
begin
  if (Length(AFileName) > 0) then
    PropsFileName := AFileName;

  if (FileExists(PropsFileName)) then
  begin
    FileStream := TFileStream.Create(PropsFileName, fmOpenRead + fmShareDenyWrite);
    try
      FileStream.ReadComponent(Self);
      {The following dont work; TAxis descends from TPersistent !
      for i := 2 to FAxisList.Count-1 do
      begin
        TAxis(FAxisList[i]).ReadComponent(TAxis(FAxisList[i]));
      end;}
    finally
      FileStream.Free;
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SaveTheProperties
  Description: Saves the properties of this instance of TPlot
       Author: Mat Ballard
 Date created: 08/03/2000
Date modified: 08/03/2000 by Mat Ballard
      Purpose: Saves the appearance of the Plot
 Known Issues:
      Comment: Note that if AFileName is blank, we use a name generated from
               the FileName property (in SetFileName).
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SaveTheProperties(AFileName: String);
var
  FileStream: TFileStream;
begin
  if (Length(AFileName) > 0) then
    PropsFileName := AFileName;

  FileStream := TFileStream.Create(PropsFileName, fmCreate or fmShareExclusive);
  FileStream.WriteComponent(Self);
  {The following dont work; TAxis descends from TPersistent !
  for i := 2 to FAxisList.Count-1 do
  begin
    TAxis(FAxisList[i]).WriteComponent(TAxis(FAxisList[i]));
  end;}
  FileStream.Free;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.AppendToFile
  Description: Appends the data to FileName
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs the GetData method of SeriesList, then the Append method of the new TFileList
 Known Issues: Needs work on GetData and testing
 ------------------------------------------------------------------------------}
procedure TCustomPlot.AppendToFile;
var
  TheStream: TMemoryStreamEx;
begin
  if (FileExists(FFileName)) then
  begin
{create the FileList, an extension of TStringList:}
    TheStream := TMemoryStreamEx.Create;

{create the data in text format:}
    FSeriesList.AppendStream((soAsText in FSaveOptions), ',', TheStream);
{save it:}
    TheStream.AppendToFile(FFileName);

    TheStream.Free;
  end
  else
  begin
    EComponentError.Create('TCustomPlot.AppendToFile: ' + FFileName + ' does not exist !');
  end;
end;


{Saving as a picture --------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SaveAsBitmap
  Description: Saves a picture of the graph as a bitmap
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs Draw method over Bitmap then saves Bitmap
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SaveAsBitmap(
  AFileName: String);
var
  Rect: TRect;
  ABitMap: Graphics.TBitMap;
begin
  Rect := GetClientRect;
  ABitMap := Graphics.TBitMap.Create;
  ABitMap.Height := Rect.Bottom - Rect.Top;
  ABitMap.Width := Rect.Right - Rect.Left;

  Draw(ABitMap.Canvas);

  ABitMap.SaveToFile(AFileName);
  ABitMap.Free;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SaveAsWMF
  Description: Saves a picture of the graph as a Windows Metafile
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs Draw method over Metafile Canvas then saves Metafile
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SaveAsWMF(
  AFileName: String;
  Enhanced: Boolean);
var
  AMetafile: TMetafile;
  AMetafileCanvas: TMetafileCanvas;
begin
  AMetafile := TMetafile.Create;
{$IFDEF COMPILER3_UP}
  AMetafile.Enhanced := Enhanced;
{$ENDIF}

{NOTE: you _MUST_ set the height and width before doing anything !}
  AMetafile.Height := Height;
  AMetafile.Width := Width;

{$IFDEF DELPHI1}
{create the metafile canvas to draw on:}
  AMetafileCanvas :=
    TMetafileCanvas.Create(AMetafile, 0);
{$ELSE}
  SetMetafileDescription;
{create the metafile canvas to draw on:}
  AMetafileCanvas :=
    TMetafileCanvas.CreateWithComment(AMetafile, 0, FCreatedBy, FDescription);
{$ENDIF}

{draw the graph on the metafile:}
  Draw(AMetafileCanvas);
  AMetafileCanvas.Free;

  AMetafile.SaveToFile(AFileName);
  AMetafile.Free;
end;

{$IFDEF GIF}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SaveAsGIF
  Description: Saves a picture of the graph as a GIF file
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs Draw method over ABitmap of AGifImage then saves AGifImage
 Known Issues: 1. Requires Anders Melander's TGifImage
               2. Package dependency problems means that it is easier to let end users add this functionality 
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SaveAsGIF(
  AFileName: String);
var
  Rect: TRect;
  ABitMap: Graphics.TBitMap;
  AGifImage: TGifImage;
begin
  ABitMap := Graphics.TBitMap.Create;
  AGifImage := TGifImage.Create;
  with  AGifImage do
  try
    try
      Rect := GetClientRect;
      ABitMap.Height := Rect.Bottom - Rect.Top;
      ABitMap.Width := Rect.Right - Rect.Left;

      Draw(ABitMap.Canvas);
      Assign(ABitMap);
    finally
      ABitMap.Free;
    end ;
    SaveToFile(AFileName);
  finally
    Free;
  end ;
end;
{$ENDIF}

{$IFDEF PNG}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SaveAsPNG
  Description: Saves a picture of the graph as a PNG file
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs Draw method over ABitmap of APngImage then saves APngImage
 Known Issues: 1. Requires TPngImage of Uberto Barbini and Edmund H. Hand
               2. Package dependency problems means that it is easier to let end users add this functionality
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SaveAsPng(
  AFileName: String);
var
  Rect: TRect;
  ABitMap: Graphics.TBitMap;
  APngImage: TPngImage;
begin
  ABitMap := Graphics.TBitMap.Create;
  APngImage := TPngImage.Create;
  with APngImage do
  try
    try
      Rect := GetClientRect;
      ABitMap.Height := Rect.Bottom - Rect.Top;
      ABitMap.Width := Rect.Right - Rect.Left;
      Self.Draw(ABitMap.Canvas);
      APngImage.CopyFromBmp(ABitMap);
    finally
      ABitMap.Free;
    end;
    SaveToFile(AFileName);
  finally
    Free;
  end;
end;
{$ENDIF}

{Copying data to the clipboard ----------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.CopyHTML
  Description: Copies data as HTML to Clipboard
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Creates a Header, fires the OnHTMLHeaderRequest event, then runs
               the DataAsHTMLTable method of SeriesList
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.CopyHTML(Format: Word);
var
  i: Integer;
  Size,
  LineLength: LongInt;
  TextHandle: THandle;
  pText,
  TextPtr: PChar;
  TheData,
  TheHeader: TStringList;
{$IFDEF DELPHI1}
  pLine: array [0..1023] of char;
{$ENDIF}
begin
  TheHeader := TStringList.Create;
  TheHeader.Add('<html>');
  TheHeader.Add('<head>');
  TheHeader.Add('<title>' + FTitle.Caption + '</title>');
  TheHeader.Add('</head>');
  TheHeader.Add('<body bgcolor="white">');
  TheHeader.Add('<h1>' + FTitle.Caption + '</h1>');
  TheHeader.Add('<p>');

  DoHTMLHeaderRequest(TheHeader);

  TheData := TStringList.Create;

{create the data in text format:}
  FSeriesList.DataAsHTMLTable(TheData);
{insert the header:}
  for i := 0 to TheHeader.Count-1 do
    TheData.Insert(0, TheHeader[i]);
{Calculate the size:}
  Size := 8;
  for i := 0 to TheData.Count-1 do
    Inc(Size, Length(TheData[i])+2);
{save it:}
  TextHandle := GlobalAlloc(GMEM_MOVEABLE, Size);
  TextPtr := GlobalLock(TextHandle);

  pText := TextPtr;
  for i := 0 to TheData.Count - 1 do
  begin
    LineLength := Length(TheData[i]);
    if LineLength <> 0 then
    begin
{$IFDEF DELPHI1}
      StrPCopy(pLine, TheData[i]);
      System.Move(pLine, pText^, LineLength);
{$ELSE}
{for some unknown reason, this Move works !}
      System.Move(Pointer(TheData[i])^, pText^, LineLength);
{$ENDIF}
      Inc(pText, LineLength);
    end;
    pText^ := #13;
    Inc(pText);
    pText^ := #10;
    Inc(pText);
  end;       

  ClipBoard.SetAsHandle(Format, TextHandle);
  GlobalUnlock(TextHandle);

  TheHeader.Free;
  TheData.Free;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DoHTMLHeaderRequest
  Description: Asks the user for their HTML header data
       Author: Mat Ballard
 Date created: 09/07/2000
Date modified: 09/07/2000 by Mat Ballard
      Purpose: fires the OnHTMLHeaderRequest event
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DoHTMLHeaderRequest(TheHeader: TStringList);
begin
  if assigned(FOnHTMLHeaderRequest) then
    OnHTMLHeaderRequest(Self, TheHeader);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.CopyText
  Description: Copies data as tab-delimited text to Clipboard
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Creates a Header, fires the OnHeaderRequest event, then runs the GetData method of SeriesList
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.CopyText;
var
  TextHandle: THandle;
  TextPtr: Pointer;
  TheStream: TMemoryStream;
  pLine: array [0..1023] of char;
begin
  StrPCopy(pLine, FTitle.Caption + CRLF);
  TheStream := TMemoryStream.Create;
  TheStream.Write(pLine, StrLen(pLine));

  DoHeaderRequest(TheStream);

{create the data in TEXT format:}
  FSeriesList.GetSubHeaderStream(#9, TheStream);
  FSeriesList.GetStream(TRUE, #9, TheStream);

{save it:}
  TextHandle := GlobalAlloc(GMEM_MOVEABLE, TheStream.Size+1);
  TextPtr := GlobalLock(TextHandle);
{for some unknown reason, this Move works !}
  System.Move(TheStream.Memory^, TextPtr^, TheStream.Size);
  ClipBoard.SetAsHandle(CF_TEXT, TextHandle);
  GlobalUnlock(TextHandle);

  TheStream.Free;
end;

{Copying picture to the clipboard -------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TCustomPlot.CopyBitmap
  Description: Copies a picture of the graph as a bitmap to the Clipboard
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs Draw method over Bitmap then copies Bitmap
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.CopyBitmap;
var
  Rect: TRect;
  ABitMap: Graphics.TBitMap;
  MyFormat : Word;
  AData : THandle;
  APalette: HPALETTE;
begin
  Rect := GetClientRect;
  ABitMap := Graphics.TBitMap.Create;
  ABitMap.Height := Rect.Bottom - Rect.Top;
  ABitMap.Width := Rect.Right - Rect.Left;

  Draw(ABitMap.Canvas);

  ABitMap.SaveToClipboardFormat(MyFormat, AData, APalette);
  ClipBoard.SetAsHandle(MyFormat,AData);

  ABitMap.Free;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.CopyWMF
  Description: Copies a picture of the graph as a Windows Metafile to the Clipboard
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs Draw method over AMetafileCanvas then copies Metafile
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.CopyWMF(Enhanced: Boolean);
var
  AMetafile: TMetafile;
  AMetafileCanvas: TMetafileCanvas;
  MyFormat : Word;
  AData : THandle;
  APalette: HPALETTE;
begin
  AMetafile := TMetafile.Create;
{$IFDEF COMPILER3_UP}
  AMetafile.Enhanced := Enhanced;
{$ENDIF}

{NOTE: you _MUST_ set the height and width before doing anything !}
  AMetafile.Height := Height;
  AMetafile.Width := Width;

{$IFDEF DELPHI1}
{create the metafile canvas to draw on:}
  AMetafileCanvas :=
    TMetafileCanvas.Create(AMetafile, 0);
{$ELSE}
  SetMetafileDescription;
{create the metafile canvas to draw on:}
  AMetafileCanvas :=
    TMetafileCanvas.CreateWithComment(AMetafile, 0, FCreatedBy, FDescription);
{$ENDIF}

{draw the graph on the metafile:}
  Draw(AMetafileCanvas);
  AMetafileCanvas.Free;

  AMetafile.SaveToClipboardFormat(MyFormat, AData, APalette);
  ClipBoard.SetAsHandle(MyFormat,AData);

  AMetafile.Free;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetAsNormalClick
  Description: Defines the current view == Mins and Maxes of axes, as the Normal view.
       Author: Mat Ballard
 Date created: 08/12/2000
Date modified: 08/12/2000 by Mat Ballard
      Purpose: Zoom == Axis Min/Max management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetAsNormalClick(Sender: TObject);
var
  i: Integer;
  pTheAxis: TAxis;
begin
  for i := 0 to FAxisList.Count-1 do
  begin
    pTheAxis := TAxis(FAxisList.Items[i]);
    pTheAxis.ZoomIntercept := pTheAxis.Intercept;
    pTheAxis.ZoomMin := pTheAxis.Min;
    pTheAxis.ZoomMax := pTheAxis.Max;
  end;
  FPlotPopUpMenu.Items[Ord(mnuView)].Items[Ord(mnuNormalView)].Enabled := TRUE;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.NormalViewClick
  Description: Zooms to the Normal view
       Author: Mat Ballard
 Date created: 08/12/2000
Date modified: 08/12/2000 by Mat Ballard
      Purpose: Zoom == Axis Min/Max management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.NormalViewClick(Sender: TObject);
var
  i: Integer;
  pTheAxis: TAxis;
begin
  for i := 0 to FAxisList.Count-1 do
  begin
    pTheAxis := TAxis(FAxisList.Items[i]);
    if (pTheAxis.ZoomMin < pTheAxis.ZoomMax) then
    begin
      pTheAxis.Min := pTheAxis.ZoomMin;
      pTheAxis.Max := pTheAxis.ZoomMax;
      pTheAxis.Intercept := pTheAxis.ZoomIntercept;
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ManualZoomClick
  Description: Manually Zooms In
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Runs the ZoomForm and adjusts Axes accordingly
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ManualZoomClick(Sender: TObject);
var
  ZoomForm: TZoomForm;
  i,
  PixelYMin,
  PixelYMax: Integer;
  pTheYAxis: TAxis;
begin
  ZoomForm := TZoomForm.Create(Self);
  with ZoomForm do
  begin
    XMinEdit.Text :=
      FXAxis.LabelToStrF(FXAxis.Min);
    XMaxEdit.Text :=
      FXAxis.LabelToStrF(FXAxis.Max);
    YMinEdit.Text :=
      FYAxis.LabelToStrF(FYAxis.Min);
    YMaxEdit.Text :=
      FYAxis.LabelToStrF(FYAxis.Max);
    if (ShowModal = mrOK) then
    begin
      FXAxis.Min := FXAxis.StrToLabel(XMinEdit.Text);
      FXAxis.Max := FXAxis.StrToLabel(XMaxEdit.Text);
      FYAxis.Min := FYAxis.StrToLabel(YMinEdit.Text);
      FYAxis.Max := FYAxis.StrToLabel(YMaxEdit.Text);
      PixelYMin := FYAxis.FofY(FYAxis.Min);
      PixelYMax := FYAxis.FofY(FYAxis.Max);
      for i := 2 to FAxisList.Count-1 do
      begin
        pTheYAxis := TAxis(FAxisList[i]);
        pTheYAxis.Min := pTheYAxis.YofF(PixelYMin);
        pTheYAxis.Max := pTheYAxis.YofF(PixelYMax);
      end;
    end;
  end;
  ZoomForm.Free;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ZoomOutClick
  Description: Zooms out after Zooming In
       Author: Mat Ballard
 Date created: 02/25/2000
Date modified: 02/25/2000 by Mat Ballard
      Purpose: Resets the axes Min and Max values to those of the Axis
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ZoomOutClick(Sender: TObject);
var
  i: Integer;
  pSeries: TSeries;
begin
  FXAxis.Min := FSeriesList.Xmin;
  FXAxis.Max := FSeriesList.Xmax;
  FYAxis.Intercept := FXAxis.Min;
{  for i := 1 to FAxisList.Count-1 do
  begin
    TAxis(FAxisList[i]).Visible := FALSE;
  end;}

  for i := 0 to FSeriesList.Count-1 do
  begin
    pSeries := FSeriesList[i];
{NB: YAxisIndex now runs from 1 .. FAxisList.Count-1:
 0 means the X Axis !}
    TAxis(FAxisList[pSeries.YAxisIndex]).Min :=
      pSeries.YMin;
    TAxis(FAxisList[pSeries.YAxisIndex]).Max :=
      pSeries.YMax;
  end;
  FXAxis.Intercept := FYAxis.Min;

  Refresh;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.CopyHTMLClick
  Description: copys the data as HTML
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: ... in CF_TEXT format
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.CopyHTMLClick(Sender: TObject);
begin
  CopyHTML(CF_TEXT);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.HandleClick
  Description: fires the OnClick event of the menuitem with Tag Tag
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: allows the TPlotMenu component to function
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.HandleClick(Sender: TObject; TheTag: Integer);
var
  i, j: Integer;
begin
{needed to handle Mode and Direction:}
  FMenuTag := TheTag;
  if (GetIndicesFromTag(TheTag, i, j)) then
{$IFDEF COMPILER2_UP}
    if Assigned(FPlotPopUpMenu.Items[i].Items[j].OnClick) then
{$ENDIF}
      FPlotPopUpMenu.Items[i].Items[j].OnClick(Sender);
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.GetIndicesFromTag
  Description: Gets the i and j indices from the Tag
       Author: Mat Ballard
 Date created: 05/25/2000
Date modified: 05/25/2000 by Mat Ballard
      Purpose: interfacing with external components
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.GetIndicesFromTag(TheTag: Integer; var i, j: Integer): Boolean;
var
  MenuIndex: Integer;
begin
  if (TheTag <= TAG_BASE) then raise
    ERangeError.CreateFmt('GetIndicesFromTag: invalid Tag %d', [Tag]);

  GetIndicesFromTag := FALSE;

  MenuIndex := TheTag - TAG_BASE - Ord(mnuCalc) - 2;
  if ((MenuIndex >= 0) and (MenuIndex <= Ord(mnuPrint))) then
  begin
    i := 0;
    j := MenuIndex;
    GetIndicesFromTag := TRUE;
  end
  else
  begin
    MenuIndex := MenuIndex - Ord(mnuPrint) - 1;
    if ((MenuIndex >= 0) and (MenuIndex <= Ord(mnuEditProperties))) then
    begin
      i := 1;
      j := MenuIndex;
      GetIndicesFromTag := TRUE;
    end
    else
    begin
      MenuIndex := MenuIndex - Ord(mnuEditProperties) - 1;
      if ((MenuIndex >= 0) and (MenuIndex <= Ord(mnuZoomOut))) then
      begin
        i := 2;
        j := MenuIndex;
        GetIndicesFromTag := TRUE;
      end
      else
      begin
        MenuIndex := MenuIndex - Ord(mnuZoomOut) - 1;
        if ((MenuIndex >= 0) and (MenuIndex <= Ord(mnuTwoRegionLineOfBestFit))) then
        begin
          i := 3;
          j := MenuIndex;
          GetIndicesFromTag := TRUE;
        end;
      end;
    end;
  end;
{$IFDEF DELPHI3_UP}
  Assert(FPlotPopUpMenu.Items[i].Items[j].Tag = TheTag,
    'TCustomPlot.GetIndicesFromTag: Tags do not match !');
{$ENDIF}
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DisplaceClick
  Description: runs the "Displacement" dialog box
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: responds to "Displace" menu selection
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DisplaceClick(Sender: TObject);
begin
  if (GetSeriesFromUser) then
  begin
    pSeries.Displace;
    Refresh;
  end;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ZoomInClick
  Description: Zooms in using the mouse
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: responds to "Zoom In" menu selection
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ZoomInClick(Sender: TObject);
begin
  FInstructions.Clear;
  FInstructions.Add('Click and drag over the region you want to zoom in on');
  FScreenJob := sjZoomIn;
  Refresh;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DifferentiateClick
  Description: Replaces the selected Axis with its differential
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: responds to "Differential" menu selection
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DifferentiateClick(Sender: TObject);
begin
  if (GetSeriesFromUser) then
  begin
    pSeries.Differentiate;
    ZoomOutClick(Self);
  end;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.IntegrateClick
  Description: Replaces the selected Axis with its differential
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: responds to "Integrate" menu selection
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.IntegrateClick(Sender: TObject);
begin
  if (GetSeriesFromUser) then
  begin
    pSeries.Integrate;
    ZoomOutClick(Self);
  end;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.IntegralClick
  Description: calculates the integral of the selected Axis over a selected range
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: responds to "Integral" menu selection
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.IntegralClick(Sender: TObject);
var
  Msg: String;
  Sum,
  TheLeft,
  TheRight: Single;
begin
  if (GetSeriesFromUser) then
  begin
    if (FScreenJob = sjIntegral) then
    begin
      TheLeft := FXAxis.XofF(Selection.Left);
      TheRight := FXAxis.XofF(Selection.Right);
      Sum := pSeries.Integral(TheLeft, TheRight);
      Msg := 'The integral of ' + pSeries.Name + ' from' + CRLF;
      Msg := Msg + FXAxis.LabelToStrF(TheLeft) + ' to ' +
        FXAxis.LabelToStrF(TheRight);
      if (Length(FXAxis.Title.Units) > 0) then
        Msg := Msg + ' ' + FXAxis.Title.Units;
      Msg := Msg + ' is' + CRLF +
        Format('%g', [Sum]);
      if ((Length(FXAxis.Title.Units) > 0) and
          (Length(pSeries.YAxis.Title.Units) > 0)) then
      begin
        Msg := Msg + ' (' + pSeries.YAxis.Title.Units + '.' +
          FXAxis.Title.Units + ')';
      end
      else if (Length(FXAxis.Title.Units) > 0) then
      begin
        Msg := Msg + ' (' + FXAxis.Title.Units + ')';
      end
      else if (Length(pSeries.YAxis.Title.Units) > 0) then
      begin
        Msg := Msg + ' (' + pSeries.YAxis.Title.Units + ')';
      end;
      ShowMessage(Msg);
      ClipBoard.AsText := Msg;
      ZeroScreenStuff;
    end
    else
    begin
      FScreenJob := sjIntegral;
      FInstructions.Clear;
      FInstructions.Add('Click and Drag over the (X-) region to calculate the Integral');
      Refresh;
    end;
  end;
end;

{------------------------------------------------------------------------------
     Function: TCustomPlot.GetClickAndDragDelay
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: gets the value of the ClickAndDragDelay Property
 Return value: Integer
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.GetClickAndDragDelay: Integer;
begin
  GetClickAndDragDelay := MouseTimer.Interval;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.ZeroScreenStuff
  Description: cleans up after a menu event handler - XXXClick
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: mouse management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.ZeroScreenStuff;
begin
  FScreenJob := sjNone;
  ClickedObjectType := soNone;
  pClickedObject := nil;
  SecondClickedObjectType := soNone;
  pSecondClickedObject := nil;
  TheSeries := -1;
  pSeries := nil;
  TheAxis := -1;
  pAxis := nil;
  FOnSelection := nil;
  FOnDualSelection := nil;
  Screen.Cursor := crDefault;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.HighsClick
  Description: Calculates the Highs (peaks) and/or Lows (troughs) of a series
       Author: Mat Ballard
 Date created: 09/25/2000
Date modified: 09/25/2000 by Mat Ballard
      Purpose: data processing
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.HighsClick(Sender: TObject);
var
  OptionsDlg: TOptionsDlg;
begin
  if (GetSeriesFromUser) then
  begin
    OptionsDlg := TOptionsDlg.Create(nil);
    with OptionsDlg do
    begin
      FormTitle := 'Highs and Lows';
      Question := 'What do you want to do ?';
      OptionList.Add('Hide Highs and Lows');
      OptionList.Add('Show Highs');
      OptionList.Add('Show Lows');
      OptionList.Add('Show Highs and Lows');
      case Execute of
        1: pSeries.ClearHighsLows;
        2:
          begin
            pSeries.FindHighsLows(0, pSeries.NoPts, 5);
            pSeries.HighLow := pSeries.HighLow + [hlHigh];
          end;
        3:
          begin
            pSeries.FindHighsLows(0, pSeries.NoPts, 5);
            pSeries.HighLow := pSeries.HighLow + [hlLow];
          end;
        4:
          begin
            pSeries.FindHighsLows(0, pSeries.NoPts, 5);
            pSeries.HighLow := pSeries.HighLow + [hlLow, hlHigh];
          end;
      end;
    end;
    OptionsDlg.Free;

    Refresh;
  end;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.MovingAverageClick
  Description: Calculates and displays the Moving Average of a series
       Author: Mat Ballard
 Date created: 09/25/2000
Date modified: 09/25/2000 by Mat Ballard
      Purpose: data smoothing
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.MovingAverageClick(Sender: TObject);
var
  Span: Integer;
  SpanStr: String;
begin
  if (GetSeriesFromUser) then
  begin
    SpanStr := '10';
    if (InputQuery('Calculation of the Moving Average',
                   'Enter the number of points to average over',
                   SpanStr)) then
    begin
      try
        Span := StrToInt(SpanStr);
        pSeries.MovingAverage(Span);
      except
        ShowMessage('Moving Average Failed !');
      end;
    end;
    Refresh;
  end;
  ZeroScreenStuff;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.AverageClick
  Description: Calculates the Average of a series over a range
       Author: Mat Ballard
 Date created: 09/25/2000
Date modified: 09/25/2000 by Mat Ballard
      Purpose: data processing
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.AverageClick(Sender: TObject);
var
  Msg: String;
  Sum,
  TheLeft,
  TheRight: Single;
begin
  if (GetSeriesFromUser) then
  begin
    if (FScreenJob = sjAverage) then
    begin
      TheLeft := FXAxis.XofF(Selection.Left);
      TheRight := FXAxis.XofF(Selection.Right);
      Sum := pSeries.Average(TheLeft, TheRight);
      Msg := 'The Average of ' + pSeries.Name + ' from' + CRLF;
      Msg := Msg + FXAxis.LabelToStrF(TheLeft) + ' to ' +
        FXAxis.LabelToStrF(TheRight);
      if (Length(FXAxis.Title.Units) > 0) then
        Msg := Msg + ' ' + FXAxis.Title.Units;
      Msg := Msg + ' is' + CRLF +
        Format('%g', [Sum]);
      if (Length(pSeries.YAxis.Title.Units) > 0) then
      begin
        Msg := Msg + ' (' + pSeries.YAxis.Title.Units + ')';
      end;
      ShowMessage(Msg);
      ClipBoard.AsText := Msg;
      ZeroScreenStuff;
    end
    else
    begin
      FScreenJob := sjAverage;
      FInstructions.Clear;
      FInstructions.Add('Click and Drag over the (X-) region to calculate the Average');
      Refresh;
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.AddAxisClick
  Description: Adds a new axis, based on either the selected or last axis.
       Author: Mat Ballard
 Date created: 06/12/2000
Date modified: 06/12/2000 by Mat Ballard
      Purpose: axis management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.AddAxisClick(Sender: TObject);
var
  TheTemplate: Integer;
  NewYAxis: TAxis;
begin
{has the user already selected an object ?}
{see GetTheClickedObject and ZeroScreenStuff}
  TheTemplate := TheAxis;
  if (TheTemplate < 0) then
    TheTemplate := FAxisList.Count - 1;

  NewYAxis := TAxis.Create(Self);
  FAxisList.Add(NewYAxis);
  NewYAxis.Assign(TAxis(FAxisList[TheTemplate]));
  NewYAxis.OnChange := StyleChange;
{Move the new axis to the right:}
  if (FAxisList.Count = 3) then
  begin
{this new one is a secondary axis:}
    NewYAxis.AxisType := atSecondary;
    NewYAxis.Intercept := FXAxis.Max;
    NewYAxis.TickDirection := orRight;
    NewYAxis.Title.Orientation := orRight;
  end
  else
  begin
    NewYAxis.AxisType := atTertiary;
{place the new axis half-way between the last and the right hand side:}
    NewYAxis.Intercept := FXAxis.XofF((NewYAxis.MidX + Width) div 2);
  end;
{if this over-runs the panel width, then wrap around:}
  if (NewYAxis.Left > Width) then
    NewYAxis.Left := NewYAxis.Left - Width;
{... and rename it:}
  NewYAxis.Name := 'Copy of ' + NewYAxis.Name;
  NewYAxis.Title.Caption := 'Copy of ' + NewYAxis.Title.Caption;
  NewYAxis.Tag := Ord(soYAxis);
  NewYAxis.Labels.Tag := Ord(soYAxisLabel);
  NewYAxis.Title.Tag := Ord(soYAxisTitle);

{add this New Y Axis to the ScreenObjectList:}
  ScreenObjectList.Insert(Ord(soLeftBorder), NewYAxis.Labels);
  ScreenObjectList.Insert(Ord(soLeftBorder), NewYAxis.Title);
  ScreenObjectList.Insert(Ord(soLeftBorder), NewYAxis);
{note that we insert: we do this in order to make Y Axes be found before
 borders and series.}

{we need to call re-size to force a SetAxisDimensions:}
  Resize;
  Refresh;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DeleteAxisClick
  Description: Deletes an axis
       Author: Mat Ballard
 Date created: 06/12/2000
Date modified: 06/12/2000 by Mat Ballard
      Purpose: axis management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DeleteAxisClick(Sender: TObject);
begin
  if (GetAxisFromUser(2)) then
    DeleteAxis(TheAxis, TRUE);
end;

{------------------------------------------------------------------------------
     Function: TCustomPlot.DeleteAxis
  Description: Deletes the selected axis
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: axis management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DeleteAxis(Index: Integer; Confirm: Boolean);
var
  pAxisToDelete: TAxis;
begin
  if ((Index <= 1) or
      (Index >= FAxisList.Count)) then raise
    ERangeError.CreateFmt('Cannot delete Axis #%d; Secondary Axes are numbered from 2 to %d',
      [Index, FAxisList.Count-1]);

  if (Confirm) then
  begin
    Confirm := (IDNO = MessageBox(0,
      'Are you sure you want to delete',
      'Delete Secondary Axis',
      MB_YESNO + MB_ICONQUESTION));
  end;

  if (not Confirm) then
  begin
    pAxisToDelete := TAxis(FAxisList[Index]);
{remove this Y Axis from the ScreenObjectList:}
    ScreenObjectList.Delete(ScreenObjectList.IndexOf(pAxisToDelete));
    ScreenObjectList.Delete(ScreenObjectList.IndexOf(pAxisToDelete.Title));
    ScreenObjectList.Delete(ScreenObjectList.IndexOf(pAxisToDelete.Labels));
{remove this Y Axis from the AxisList:}
    FAxisList.Delete(Index);
{and remove it:}
    pAxisToDelete.Free;
  end;
end;

{------------------------------------------------------------------------------
     Function: TCustomPlot.GetNoYAxes
  Description: standard property Get function
       Author: Mat Ballard
 Date created: 06/25/2000
Date modified: 06/25/2000 by Mat Ballard
      Purpose: gets the number of Y Axes
 Return Value: integer
 Known Issues:
 ------------------------------------------------------------------------------}
function TCustomPlot.GetNoYAxes: Integer;
begin
  GetNoYAxes := FAxisList.Count-1;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetNoYAxes
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the number of Y Axes
 Known Issues: it is a fairly brutal way of setting the number of Y Axes
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetNoYAxes(Value: Integer);
var
  i: Integer;
begin
  if (Value < 1) then exit;

  if (Value = FAxisList.Count-1) then exit;

  if (Value > 15) then raise
    EComponentError.CreateFmt(
      'You must be joking ! I only support 15 Y Axes - not %d !',
      [Value]);

  if (Value < FAxisList.Count-1) then
  begin
{we need to delete some axes:}
    for i := FAxisList.Count-1 downto Value+1 do
    begin
      DeleteAxis(i, FALSE);
    end;
  end
  else
  begin
{add some axes:}
    for i := FAxisList.Count-1 to Value-1 do
    begin
      AddAxisClick(Self);
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetPlotType
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 09/20/2000
Date modified: 09/20/2000 by Mat Ballard
      Purpose: sets the PlotType property
 Known Issues: under development
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetPlotType(Value: TPlotType);
begin
  FPlotType := Value;
{display dummy data while designing in Delphi:}
  if (csDesigning in ComponentState) then
  begin
    case FPlotType of
      ptXY: MakeDummyData(100);
      ptMultiple: MakeDummyData(20);
      ptColumn: MakeDummyData(10);
    end;
  end;
  Refresh;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.SetPlotMenu
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 09/25/2000
Date modified: 09/25/2000 by Mat Ballard
      Purpose: sets the PlotMenu property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.SetPlotMenu(Value: TMainMenu);
begin
  FPlotMenu := Value;
end;

{------------------------------------------------------------------------------
    Procedure: TCustomPlot.Clear
  Description: Saves any changed files (at user request) and then clears the SeriesList.
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Series management: wraps TSeriesList.ClearSeries
 Known Issues: it is a fairly brutal way of setting the number of Y Axes
 ------------------------------------------------------------------------------}
procedure TCustomPlot.Clear(Cancellable: Boolean);
var
  TheResult: Integer;
  uType: WORD;
  TheMessage: String;
{$IFDEF DELPHI1}
  pMessage: array [0..255] of char;
{$ENDIF}
begin
  if (csDesigning in ComponentState) then exit;

  if (FSeriesList.Count > 0) then
  begin
    if (FSeriesList.DataChanged) then
    begin
      if (Cancellable) then
        uType := MB_YESNOCANCEL	+ MB_ICONQUESTION
       else
        uType := MB_YESNO + MB_ICONQUESTION;
      if (Length(FFileName) > 0) then
        TheMessage := ExtractFileName(FFileName)
       else
        TheMessage := 'this file';
       TheMessage := 'Save ' + TheMessage + ' before closing it ?';

{NB: MessageDlg provokes an access violation in D5:}
{$IFDEF DELPHI1}
      StrPCopy(pMessage, TheMessage);
      TheResult := MessageBox(0, pMessage, 'File has Changed', uType);
{$ELSE}
      TheResult := MessageBox(0, PChar(TheMessage), 'File has Changed', uType);
{$ENDIF}
      case TheResult of
        IDYES: SaveClick(Self);
        IDNO: ;
        IDCANCEL: Exit;
      end;
    end;

    FSeriesList.ClearSeries;
    if ((FPlotMenu <> nil) and (Length(FFileName) > 0)) then
      TPlotMenu(FPlotMenu).AddHistory(FFileName);
    DoFileClose(FFileName);
    if assigned(FOnFileClose) then OnFileClose(Self, FFileName);
  end;
  FileName := '';
end;


{------------------------------------------------------------------------------
    Procedure: TCustomPlot.DoFileClose
  Description: informs the user of a file close
       Author: Mat Ballard
 Date created: 09/07/2000
Date modified: 09/07/2000 by Mat Ballard
      Purpose: Fires the OnFileClose event
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TCustomPlot.DoFileClose(AFileName: String);
begin
  if assigned(FOnFileClose) then
    OnFileClose(Self, AFileName);
end;

end.

