unit PtEdit;

{-----------------------------------------------------------------------------
The contents of this file are subject to the Mozilla Public License
Version 1.1 (the "License"); you may not use this file except in compliance
with the License. You may obtain a copy of the License at
http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either expressed or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is: PtEdit.pas, released 12 September 2000.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@molsci.csiro.au.

Last Modified: 02/25/2000
Current Version: 1.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind. 

Purpose:
To facilitate user manipluation of Point values.

Known Issues:
-----------------------------------------------------------------------------}

interface

uses
  Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, Buttons,

{These are the D1 units, aliased to Windows:}
{$IFDEF WIN}
  WinTypes, WinProcs,
{$ENDIF}
{$IFDEF WIN32}
  ComCtrls,
{$ENDIF}

  Axis, Misc, SlideBar, ShapeEx, ExtCtrls;

type
  TPointEditorForm = class(TForm)
    HelpBitBtn: TBitBtn;
    CancelBitBtn: TBitBtn;
    OKBitBtn: TBitBtn;
    DeleteBitBtn: TBitBtn;
    AddBitBtn: TBitBtn;
    DataGroupBox: TGroupBox;
    Label1: TLabel;
    Label2: TLabel;
    XDataEdit: TEdit;
    YDataEdit: TEdit;
    ScreenGroupBox: TGroupBox;
    Label3: TLabel;
    Label4: TLabel;
    XScreenEdit: TEdit;
    YScreenEdit: TEdit;
    DetailsLabel: TLabel;
    Label5: TLabel;
    PointSlideBar: TSlideBar;
    PointEdit: TEdit;
    Panel1: TPanel;
    UpShapeEx: TShapeEx;
    DownShapeEx: TShapeEx;

    procedure IntEditKeyPress(Sender: TObject; var Key: Char);
    procedure FloatEditKeyPress(Sender: TObject; var Key: Char);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormCreate(Sender: TObject);
    procedure CancelBitBtnClick(Sender: TObject);
    procedure PointSlideBarChange(Sender: TObject);
    procedure UpShapeExClick(Sender: TObject);
    procedure DownShapeExClick(Sender: TObject);
    procedure UpShapeExDblClick(Sender: TObject);
    procedure DownShapeExDblClick(Sender: TObject);
    procedure PointEditChange(Sender: TObject);

  private
    XData, YData: pSingleArray;
    XAxis, YAxis: TAxis;

    procedure GetPoint(N: Integer; var X, Y: Single);

  public
    procedure Init(FXData, FYData: pSingleArray; FXAxis, FYAxisCurrent: TAxis);
    procedure FillData(ThePoint: Integer);
  end;

var
  PointEditorForm: TPointEditorForm;

implementation

{$R *.DFM}

uses
  pSeries;

{------------------------------------------------------------------------------
    Procedure: TPointEditorForm.FormCreate
  Description: standard FormCreate procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the position
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TPointEditorForm.FormCreate(Sender: TObject);
begin
  Left := 10;
  Top := 10;
end;

{------------------------------------------------------------------------------
    Procedure: TPointEditorForm.Init
  Description: initializes pointers to data and axes
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 05/04/2000 by Mat Ballard
      Purpose: user management of data
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TPointEditorForm.Init(FXData, FYData: pSingleArray;
  FXAxis, FYAxisCurrent: TAxis);
begin
  XData := FXData;
  YData := FYData;
  XAxis := FXAxis;
  YAxis := FYAxisCurrent;
end;

{------------------------------------------------------------------------------
    Procedure: TPointEditorForm.FormClose
  Description: standard FormClose procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: checks for valid user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TPointEditorForm.FormClose(Sender: TObject;
  var Action: TCloseAction);
var
  Test: Single;
begin
{Check that floating point values are valid:}
  Action := caHide;
  if (DataGroupBox.Enabled = TRUE) then
  begin
    try
      Test := StrToFloat(XDataEdit.Text);
      Test := Test + StrToFloat(YDataEdit.Text);
{We do the above and below the squelch the warning messages.}
      Tag := Trunc(Test);
    except
{Prevent closure:}
      Action := caNone;
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TPointEditorForm.IntEditKeyPress
  Description: allows only integer keys to be pressed
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: restrict user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TPointEditorForm.IntEditKeyPress(Sender: TObject;
  var Key: Char);
begin
  if not (Key in [#8, '0', '1', '2', '3', '4', '5', '6', '7', '8', '9']) then
    Key := Chr(0);
  DataGroupBox.Enabled := FALSE;
end;

{------------------------------------------------------------------------------
    Procedure: TPointEditorForm.FloatEditKeyPress
  Description: allows only "real" keys to be pressed
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: restrict user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TPointEditorForm.FloatEditKeyPress(Sender: TObject;
  var Key: Char);
begin
  if not (Key in [#8, '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '-', '.', 'e']) then
    Key := Chr(0);
  ScreenGroupBox.Enabled := FALSE;
end;

{------------------------------------------------------------------------------
    Procedure: TPointEditorForm.FillData
  Description: Gets the point data
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Synchronization of the two controls
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TPointEditorForm.FillData(ThePoint: Integer);
var
  X, Y: Single;
begin
  GetPoint(ThePoint, X, Y);

  XDataEdit.Text := XAxis.LabelToStrF(X);
  YDataEdit.Text := YAxis.LabelToStrF(Y);

  XScreenEdit.Text := IntToStr(XAxis.FofX(X));
  YScreenEdit.Text := IntToStr(YAxis.FofY(Y));
end;

{------------------------------------------------------------------------------
    Procedure: TSeries.GetPoint
  Description: returns the Nth (0..NoPts-1) point's X and Y values.
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: data management
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TPointEditorForm.GetPoint(N: Integer; var X, Y: Single);
begin
  if ((N < 0) or (N > PointSlideBar.Position)) then raise
    ERangeError.CreateFmt('GetPoint: the Point number %d is not within the valid range of 0 .. %d',
      [N, PointSlideBar.Position]);

  X := XData^[N];
  Y := YData^[N];
end;

{------------------------------------------------------------------------------
    Procedure: TPointEditorForm.CancelBitBtnClick
  Description: Standard Cancel button event handler
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Disables the DataGroupBox, and hence prevents checking of edit boxes for validity
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TPointEditorForm.CancelBitBtnClick(Sender: TObject);
begin
  DataGroupBox.Enabled := FALSE;
end;

{------------------------------------------------------------------------------
    Procedure: TPointEditorForm.PointSlideBarChange
  Description: Sets the value of PointEdit
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Synchronization of the two controls
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TPointEditorForm.PointSlideBarChange(Sender: TObject);
begin
  PointEdit.Text := IntToStr(PointSlideBar.Position);
  FillData(PointSlideBar.Position);
end;

procedure TPointEditorForm.UpShapeExClick(Sender: TObject);
begin
  PointSlideBar.Position := PointSlideBar.Position + 1;
end;

procedure TPointEditorForm.DownShapeExClick(Sender: TObject);
begin
  PointSlideBar.Position := PointSlideBar.Position - 1;
end;

procedure TPointEditorForm.UpShapeExDblClick(Sender: TObject);
begin
  PointSlideBar.Position := PointSlideBar.Position + 10;
end;

procedure TPointEditorForm.DownShapeExDblClick(Sender: TObject);
begin
  PointSlideBar.Position := PointSlideBar.Position - 10;
end;

procedure TPointEditorForm.PointEditChange(Sender: TObject);
begin
  try
    PointSlideBar.Position := StrToInt(PointEdit.Text);
  finally
  end;  
end;

end.
