unit Parser;

{$I Plot.inc}

{-----------------------------------------------------------------------------
The contents of this file are subject to the Mozilla Public License
Version 1.1 (the "License"); you may not use this file except in compliance
with the License. You may obtain a copy of the License at
http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either expressed or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is: Parser.pas, released 12 September 2000.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@molsci.csiro.au.

Last Modified: 02/25/2000
Current Version: 1.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind. 

Purpose:
To allow users to paste complex data into TPlot.

Known Issues:
-----------------------------------------------------------------------------}

interface

uses
  Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, Buttons, Grids,

{These are the D1 units, aliased to Windows:}
{$IFDEF WIN}
  WinTypes, WinProcs,
{$ENDIF}

  Misc;

const
  COLUMN_NOS = 0;
  SERIES_NAMES = 1;
  FIRST_LINE_OF_DATA = 2;
  X_OR_Y = 3;
  DEPENDS_ON_X = 4;

type
  TColumnType = (ctIgnore, ctX, ctY);

  TSeriesInfo = Record   {The series are 0, 1, 2}
    Index: Integer;          {What is the Index of this series of data in the SeriesList ?}
    XCol: Integer;           {in which column is this series' X data ?}
    YCol: Integer;           {in which column is this series' Y data ?}
    XSeriesIndex: Integer;   {what is the index of the series that contains the X Data in the SeriesList ?}
    XValue: Single;          {the X Value, after a string conversion}
    YValue: Single;          {the Y Value, after a string conversion}
  end;

{A TSeriesInfoArray is never declared directly - instead, it is used as a
 template for a dynamic array}
  TSeriesInfoArray = array[0..1023] of TSeriesInfo;
  pSeriesInfoArray = ^TSeriesInfoArray;

  TDelimiter = (dlNone, dlTab, dlComma, dlSpace, dlColon, dlSemiColon,
                dlLineFeed, dlOwn);

  TParserForm = class(TForm)
    DataListBox: TListBox;
    Label1: TLabel;
    DelimiterComboBox: TComboBox;
    LineEdit: TEdit;
    GroupBox1: TGroupBox;
    SeriesNamesButton: TButton;
    FirstLineOfDataButton: TButton;
    BitBtn1: TBitBtn;
    BitBtn2: TBitBtn;
    OKBitBtn: TBitBtn;
    InfoGrid: TStringGrid;
    PickXDataComboBox: TComboBox;
    procedure FormCreate(Sender: TObject);
    procedure DataListBoxClick(Sender: TObject);
    procedure FirstLineOfDataButtonClick(Sender: TObject);
    procedure SeriesNamesButtonClick(Sender: TObject);
    procedure InfoGridClick(Sender: TObject);
    procedure PickXDataComboBoxClick(Sender: TObject);
  private
    procedure CheckDelimiter;
  public
    Delimiters: array[TDelimiter] of String;
    DelimiterNames: array[TDelimiter] of String;
    TheDelimiter: TDelimiter;
    TheFirstDataLine: Integer;
    TheSeriesNamesLine: Integer;
    TheSeriesUnitsLine: Integer;
    NumberOfSeries: Integer;
    TheCol, TheRow: Integer;
  end;

var
  ParserForm: TParserForm;

implementation

{$R *.DFM}

{------------------------------------------------------------------------------
    Procedure: TAxisEditorForm.FormCreate
  Description: standard FormCreate procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the position, the Delimiters and their names, and putsw headings on the stringgrid
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TParserForm.FormCreate(Sender: TObject);
var
  iDelimiter: TDelimiter;
begin
  Left := 10;
  Top := 10;

  Delimiters[dlNone] := '';
  DelimiterNames[dlNone] := 'None';
{Tabs come from spreadsheet / table data from the clipboard:}
  Delimiters[dlTab] := #9;
  DelimiterNames[dlTab] := 'Tab   ->';
{Commas come from disk data:}
  Delimiters[dlComma] := ',';
  DelimiterNames[dlComma] := 'Comma  ,';
  Delimiters[dlSpace] := ' ';
  DelimiterNames[dlSpace] := 'Space   ';
  Delimiters[dlColon] := ';';
  DelimiterNames[dlColon] := 'Colon  ;';
  Delimiters[dlSemicolon] := #58;
  DelimiterNames[dlSemicolon] := 'Semicolon  :';
  Delimiters[dlLinefeed] := #10;
  DelimiterNames[dlLinefeed] := 'Line Feed   ';
  Delimiters[dlOwn] := 'Type your own';
  DelimiterNames[dlOwn] := 'Type your own';
  for iDelimiter := dlNone to dlOwn do
  begin
    DelimiterComboBox.Items.Add(DelimiterNames[iDelimiter]);
  end;

  with InfoGrid do
  begin
    Cells[0, COLUMN_NOS] := 'Column Number:';
    Cells[0, SERIES_NAMES] := 'Series Names:';
    Cells[0, FIRST_LINE_OF_DATA] := '1st Line of Data:';
    Cells[0, X_OR_Y] := 'X or Y data ?';
    Cells[0, DEPENDS_ON_X] := 'X Data in Column #:';
    DefaultColWidth := 80;
    ColWidths[0] := 130;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TParserForm.DataListBoxClick
  Description: responds to user selection of a line
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: places the selected line into the Edit box, and checks it for Delimiters
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TParserForm.DataListBoxClick(Sender: TObject);
begin
  LineEdit.Text := DataListBox.Items[DataListBox.ItemIndex];
  CheckDelimiter;
end;

{------------------------------------------------------------------------------
    Procedure: TParserForm.CheckDelimiter
  Description: determines the Delimiter in a line
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets Delimiter
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TParserForm.CheckDelimiter;
var
  iDelimiter: TDelimiter;
begin
  TheDelimiter := dlNone;
  for iDelimiter := dlTab to dlLineFeed do
  begin
    if (Pos(Delimiters[iDelimiter], LineEdit.Text) > 0) then
    begin
      TheDelimiter := iDelimiter;
      break;
    end;
  end;
  DelimiterComboBox.ItemIndex := Ord(TheDelimiter);
end;

{------------------------------------------------------------------------------
    Procedure: TParserForm.FirstLineOfDataButtonClick
  Description: responds to the selection of the first line of data
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: fills the InfoGrid cells with the parsed line
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TParserForm.FirstLineOfDataButtonClick(Sender: TObject);
var
  TheLine: String;
  iColumn: Integer;
  {AvailWidth, UsedWidth: Integer;}
begin
  TheFirstDataLine := DataListBox.ItemIndex;
  TheLine := DataListBox.Items[DataListBox.ItemIndex];
  iColumn := 1;
  NumberOfSeries := 0;
  {InfoGrid.ColCount := 3;}
  while (Length(TheLine) > 0) do
  begin
    if (InfoGrid.ColCount <= iColumn) then
      InfoGrid.ColCount := iColumn+1;
    InfoGrid.Cells[iColumn, FIRST_LINE_OF_DATA] :=
      GetWord(TheLine, Delimiters[TheDelimiter]);
    Inc(iColumn);
    Inc(NumberOfSeries);
  end;

  for iColumn := 1 to InfoGrid.ColCount-1 do
  begin
    InfoGrid.Cells[iColumn, COLUMN_NOS] := IntToStr(iColumn);
  end;

  {AvailWidth := InfoGrid.Width -
    InfoGrid.ColWidths[0];
  UsedWidth := (InfoGrid.ColCount -1)*
    InfoGrid.DefaultColWidth;
  if (UsedWidth < AvailWidth) then
  begin
    InfoGrid.DefaultColWidth := AvailWidth div
      (InfoGrid.ColCount -1);
  end;}

  InfoGrid.Cells[1, X_OR_Y] := 'X';
  for iColumn := 2 to InfoGrid.ColCount-1 do
  begin
    InfoGrid.Cells[iColumn, X_OR_Y] := 'Y';
  end;

  InfoGrid.Cells[1, DEPENDS_ON_X] := '';
  for iColumn := 2 to InfoGrid.ColCount-1 do
  begin
    InfoGrid.Cells[iColumn, DEPENDS_ON_X] := '1';
  end;

  OKBitBtn.Tag := OKBitBtn.Tag or 2;
  if (OKBitBtn.Tag = 3) then
    OKBitBtn.Enabled := TRUE;
end;

{------------------------------------------------------------------------------
    Procedure: TParserForm.SeriesNamesButtonClick
  Description: responds to the selection of the Series names
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: fills the InfoGrid Series Names row with the parsed line
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TParserForm.SeriesNamesButtonClick(Sender: TObject);
var
  TheLine: String;
  iColumn: Integer;
begin
  TheSeriesNamesLine := DataListBox.ItemIndex;
  TheLine := DataListBox.Items[DataListBox.ItemIndex];
  iColumn := 1;
  while (Length(TheLine) > 0) do
  begin
    if (InfoGrid.ColCount <= iColumn) then
      InfoGrid.ColCount := iColumn+1;
    InfoGrid.Cells[iColumn, SERIES_NAMES] :=
      GetWord(TheLine, Delimiters[TheDelimiter]);
    Inc(iColumn);
  end;
  InfoGrid.ColCount := iColumn;

  for iColumn := 1 to InfoGrid.ColCount-1 do
  begin
    InfoGrid.Cells[iColumn, COLUMN_NOS] := IntToStr(iColumn);
  end;
  
  OKBitBtn.Tag := OKBitBtn.Tag or 1;
  if (OKBitBtn.Tag = 3) then
    OKBitBtn.Enabled := TRUE;
end;

{------------------------------------------------------------------------------
    Procedure: TParserForm.InfoGridClick
  Description: responds to the selection of a cell in the StringGrid
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the InfoGrid options, contents of some cells, and PickXDataComboBox position and visibility
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TParserForm.InfoGridClick(Sender: TObject);
var
  Rect: TRect;
  iColumn: Integer;
begin
  with InfoGrid do
  begin
    if ((Row = SERIES_NAMES) and (Col > 0)) then
    begin
      Options := [goFixedVertLine,goFixedHorzLine,goVertLine,goHorzLine,
        goDrawFocusSelected,goColSizing, goEditing];
    end
    else
    begin
      Options := [goFixedVertLine,goFixedHorzLine,goVertLine,goHorzLine,
        goDrawFocusSelected,goColSizing];
    end;

    if ((Row = X_OR_Y) and (Col > 2)) then
    begin
      if (Cells[Col, Row] = 'X') then
      begin
        Cells[Col, Row] := 'Y';
        Cells[Col, Row + 1] := '1';
      end
      else if (Cells[Col, Row] = 'Y') then
      begin
        Cells[Col, Row] := 'ignore';
        Cells[Col, Row + 1] := '';
      end
      else
      begin
        Cells[Col, Row] := 'X';
        Cells[Col, Row + 1] := '';
      end;
    end;

    if ((Row = DEPENDS_ON_X) and (Col > 2)) then
    begin
      TheCol := Col;
      TheRow := Row;
      {Rect := TRect(CellRect(Col, Row));}
      Rect := CellRect(TheCol, TheRow);
      PickXDataComboBox.Left :=
        InfoGrid.Left + Rect.Left;
      PickXDataComboBox.Top :=
        InfoGrid.Top + Rect.Top;
      PickXDataComboBox.Width := Rect.Right - Rect.Left;
      PickXDataComboBox.Height := Rect.Bottom - Rect.Top;
      PickXDataComboBox.Clear;
      for iColumn := 1 to ColCount-1 do
      begin
        if (Cells[iColumn, X_OR_Y] = 'X') then
          PickXDataComboBox.Items.Add(IntToStr(iColumn));
      end;
      PickXDataComboBox.Visible := TRUE;
    end;
  end;
end;

{------------------------------------------------------------------------------
    Procedure: TParserForm.PickXDataComboBoxClick
  Description: responds to the selection of a particular X Data column
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the XData column
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TParserForm.PickXDataComboBoxClick(Sender: TObject);
begin
  with PickXDataComboBox do
  begin
    Visible := FALSE;
    if (ItemIndex >= 0) then
      InfoGrid.Cells[TheCol, TheRow] := Items[ItemIndex];
  end;
end;

end.
