unit Axis;

{$I Plot.inc}

{-----------------------------------------------------------------------------
The contents of this file are subject to the Mozilla Public License
Version 1.1 (the "License"); you may not use this file except in compliance
with the License. You may obtain a copy of the License at
http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either expressed or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is: Axis.pas, released 12 September 2000.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@molsci.csiro.au.

Last Modified: 09/21/2000
Current Version: 1.01

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind. 

Purpose:
To implement an Axis component for use by the main TPlot graphing component.

Known Issues:

History:
 1.01 21 September 2000: fix FontWidth bug in TAxis.Draw
        add LabelText property to TAxis (for columns)
-----------------------------------------------------------------------------}

interface

uses
  Classes, Graphics, SysUtils, controls,

{$IFNDEF NO_MATH}
  Math,
{$ENDIF}

{These are the D1 units, aliased to Windows:}
{$IFDEF WIN}
  WinTypes, WinProcs,
{$ENDIF}

  Titles, Misc;

{const}

type
  TAxisType = (atPrimary, atSecondary, atTertiary);

  TLabelFormat = (lfGeneral, lfExponent, lfFixed, lfNumber, lfCurrency, lfPercent,
                  lfSeconds, lfMinutes, lfHours, lfDays, lfShortTime, lfShortDate);
{lfGeneral ... lfCurrency are just TFloatFormat.}
{}
{We then add Percentage, then the rest are so that we can display times in various formats.}

{  TOnPositionChangeEvent = procedure(
    Sender: TObject;
    bIntercept: Boolean; did the Intercept change ? or the screen position ?
    var TheIntercept: Single;
    var ThePosition: Integer) of object;}

{Begin TAxisLabel declarations ------------------------------------------------}
  TAxisLabel = class(TCaption)
  private
    FDigits: Byte;
    FPrecision: Byte;
    FNumberFormat: TLabelFormat;

    {FOnChange: TNotifyEvent;}

    procedure SetDigits(Value: Byte);
    procedure SetPrecision(Value: Byte);
    procedure SetNumberFormat(Value: TLabelFormat);

  protected

  public
    Constructor Create(AOwner: TPersistent); override;
{The standard constructor, where standard properties are set.}
    Destructor Destroy; override;
{The standard destructor, where the OnChange event is "freed".}

    {procedure Assign(Source: TPersistent); override;}
    procedure AssignTo(Dest: TPersistent); override;

  published
    Property Digits: Byte read FDigits write SetDigits;
{This (and Precision) control the numeric format of the Axis Labels.
 See the Borland documentation on FloatToStrF for the precise meaning of
 this property, or simply experiment in the IDE designer.}
    Property Precision: Byte read FPrecision write SetPrecision;
{This (and Digits) control the numeric format of the Axis Labels.
 See the Borland documentation on FloatToStrF for the precise meaning of
 this property, or simply experiment in the IDE designer.}
    Property NumberFormat: TLabelFormat read FNumberFormat write SetNumberFormat;
{This property controls how the numbers of the Axis labels are displayed.}

  end;

{Begin TAxis declarations ---------------------------------------------------}
  TAxis = class(TRectangle)
  private
    FArrowSize: Byte;
    FAutoScale: Boolean;
    FAxisType: TAxisType;
    FDirection: TDirection;
    FIntercept: Single;
    FLabels: TAxisLabel;
    FLabelText: TStringList;
    FLogScale: Boolean;
    FLogSpan: Single;
    FMin: Single;
    FMax: Single;
    FPen: TPen;
    FStepSize: Single;
    FStepStart: Single;
    FSpan: Single;
    FTickMinor: Byte;
    FTickSign: Integer;
    FTickSize: Byte;
    FTickDirection: TOrientation;
    FTickNum: Byte;
    FTitle: TTitle;
    FZoomIntercept: Single;
    FZoomMin: Single;
    FZoomMax: Single;

    procedure ReScale;
    procedure SetupHorizontalEnvelope;
    procedure SetupVerticalEnvelope(StrWidth: Integer);

  protected
{Set procedures:}
    procedure SetArrowSize(Value: Byte);
    procedure SetAutoScale(Value: Boolean);
    procedure SetDirection(Value: TDirection);
    procedure SetIntercept(Value: Single);
    procedure SetLabelText(Value: TStringList);
    procedure SetLogScale(Value: Boolean);
    procedure SetMin(Value: Single);
    procedure SetMax(Value: Single);
    procedure SetPen(Value: TPen);
    procedure SetStepSize(Value: Single);
    procedure SetTickMinor(Value: Byte);
    {procedure SetTickNum(Value: Byte);}
    procedure SetTickSize(Value: Byte);
    procedure SetOrientation(Value: TOrientation);

    procedure StyleChange(Sender: TObject); virtual;
    procedure TitleChange(Sender: TObject); virtual;

  public
    Property AxisType: TAxisType read FAxisType write FAxisType;
{What sort of axis is this ?}
    Property ZoomIntercept: Single read FZoomIntercept write FZoomIntercept;
{The (old) ZOOMED OUT Intercept in data co-ordinates.}
    Property ZoomMin: Single read FZoomMin write FZoomMin;
{The (old) ZOOMED OUT minimum, Left or Bottom of the Axis, in data co-ordinates.}
    Property ZoomMax: Single read FZoomMax write FZoomMax;
{The (old) ZOOMED OUT maximum, Right or Top of the Axis, in data co-ordinates.}

    Constructor Create(AOwner: TPersistent); {$IFDEF DELPHI4_UP}reintroduce;{$ENDIF} {squelch the error message}
{The standard constructor, where sub-components are created, and standard
 properties are set.}


    Destructor Destroy; override;
{The standard destructor, where sub-components and the OnChange event is "freed".}

    procedure DeSci(ExtNumber: Extended; var Mantissa: Extended; var Exponent: Integer);
{This method breaks a number down into its mantissa and exponent.
 Eg: 0.00579 has a mantissa of 5.79, and an exponent of -3.}
    procedure Draw(ACanvas: TCanvas);
{This draws the Axis on the given Canvas.}
    function LabelToStrF(Value: Single): String;
{This method converts a number to a string, given the current Labels' NumberFormat.}
    function StrToLabel(Value: String): Single;
{This method converts a string to a number, given the current Labels' NumberFormat.}
    function FofX(X: Single): Integer;
{This converts an X data value to a screen X co-ordinate.}
    function FofY(Y: Single): Integer;
{This converts a Y data value to a screen Y co-ordinate.}
    function XofF(F: Integer): Single;
{This converts a screen X co-ordinate to a X data value.}
    function YofF(F: Integer): Single;
{This converts a screen Y co-ordinate to a Y data value.}
    procedure SetMinFromSeries(Value: Single);
{This sets the Min property of the Axis. It is used exclusively by TSeries.}
    procedure SetMaxFromSeries(Value: Single);
{This sets the Max property of the Axis. It is used exclusively by TSeries.
 Exactly how it affects the Axis depends on TPlot.DisplayMode.}
    {procedure Assign(Source: TPersistent); override;}
    procedure AssignTo(Dest: TPersistent); override;

  published
    Property ArrowSize: Byte read FArrowSize write SetArrowSize;
{This is the size (in pixels) of the arrowhead on the Axis.}
    Property AutoScale: Boolean read FAutoScale write SetAutoScale;
{Do we use the StepSize property or does TPlot work them out ?}
    Property Title: TTitle read FTitle write FTitle;
{The Title on and of the Axis. Note that the Title can be clicked and dragged
 around the Axis.}
    Property Direction: TDirection read FDirection write SetDirection;
{Is the Axis Horizontal (X) or Vertical (Y or Y2).}
    Property Intercept: Single read FIntercept write SetIntercept;
{The intercept of this Axis on the complementary Axis.}
    Property Labels: TAxisLabel read FLabels write FLabels;
{The numerals on the Axis.}
    property LabelText: TStringList read FLabelText write SetLabelText;
{The user-defined text on the axes, instead of Labels.}    
    Property LogScale: Boolean read FLogScale write SetLogScale;
{Is this Axis on a logarithmic scale ?}
    Property Min: Single read FMin write SetMin;
{The minimum, Left or Bottom of the Axis, in data co-ordinates.}
    Property Max: Single read FMax write SetMax;
{The maximum, Right or Top of the Axis, in data co-ordinates.}
    Property Pen: TPen read FPen write SetPen;
{The Pen that the Axis is drawn with.}
    Property StepSize: Single read FStepSize write SetStepSize;
{The interval between tick (and labels) on the Axis.}
{}
{If the axis is a Log Scale, then this is the multiple, not the interval !}
    Property TickMinor: Byte read FTickMinor write SetTickMinor;
{Sets the number of minor ticks between labels.}
    Property TickSize: Byte read FTickSize write SetTickSize;
{The Length of the Ticks, in screen pixels.}
    Property TickDirection: TOrientation read FTickDirection write SetOrientation;
{Are the Ticks to the left or right of the Axis ?}
    {Property TickNum: Byte read FTickNum write SetTickNum;}
{The approximate number of ticks: TPlot recalculates the number of ticks
 depending on the StepSize.}

  end;

implementation

uses
  Plot;

{TAxislabel methods ---------------------------------------------------------}
{Constructor and Destructor:-------------------------------------------------}
{------------------------------------------------------------------------------
  Constructor: TAxisLabel.Create
  Description: standard Constructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Precision and Digits Properties
 Known Issues:
 ------------------------------------------------------------------------------}
Constructor TAxisLabel.Create(AOwner: TPersistent);
begin
{First call the ancestor:}
  inherited Create(AOwner);

{Put your own initialisation (memory allocation, etc) here:}

{we insert the default values that cannot be "defaulted":}
  FPrecision := 3;
  FDigits := 1;
end;

{------------------------------------------------------------------------------
  Destructor: TAxisLabel.Destroy
  Description: standard Destructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: frees the OnChange event
 Known Issues:
 ------------------------------------------------------------------------------}
Destructor TAxisLabel.Destroy;
begin
  FOnChange := nil;
{Put your de-allocation, etc, here:}

{then call ancestor:}
 inherited Destroy;
end;

{End Constructor and Destructor:---------------------------------------------}

{------------------------------------------------------------------------------
    Procedure: TAxisLabel.Assign
  Description: standard Assign method
       Author: Mat Ballard
 Date created: 07/06/2000
Date modified: 07/06/2000 by Mat Ballard
      Purpose: implements Assign
 Known Issues:
 ------------------------------------------------------------------------------}
{procedure TAxisLabel.Assign(Source: TPersistent);
begin
  inherited Assign(Source);
  FDigits := TAxisLabel(Source).Digits;
  FNumberFormat := TAxisLabel(Source).NumberFormat;
  FPrecision := TAxisLabel(Source).Precision;
end;}

{------------------------------------------------------------------------------
    Procedure: TAxisLabel.Assign
  Description: standard Assign method
       Author: Mat Ballard
 Date created: 07/06/2000
Date modified: 07/06/2000 by Mat Ballard
      Purpose: implements Assign
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisLabel.AssignTo(Dest: TPersistent);
begin
  inherited AssignTo(Dest);
  TAxisLabel(Dest).Digits := FDigits;
  TAxisLabel(Dest).NumberFormat := FNumberFormat;
  TAxisLabel(Dest).Precision := FPrecision;
end;

{Begin Set Procedures --------------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TAxisLabel.SetDigits
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Digits Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisLabel.SetDigits(Value: Byte);
begin
  if (FDigits = Value) then exit;

  if (FDigits > 18) then exit;

  case FNumberFormat of
    lfGeneral:  if (FDigits > 4) then exit;
    lfExponent: if (FDigits > 4) then exit;
  end;
  FDigits := Value;

  if Assigned(FOnChange) then OnChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TAxisLabel.SetPrecision
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Precision Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisLabel.SetPrecision(Value: Byte);
begin
  if (FPrecision = Value) then exit;
  if (FPrecision > 7) then exit;
  FPrecision := Value;

  if Assigned(FOnChange) then OnChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TAxisLabel.SetNumberFormat
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the NumberFormat Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisLabel.SetNumberFormat(Value: TLabelFormat);
begin
  if (FNumberFormat = Value) then exit;
  FNumberFormat := Value;
  case FNumberFormat of
    lfGeneral:  if (FDigits > 4) then FDigits := 4;
    lfExponent: if (FDigits > 4) then FDigits := 4;
  end;

  if Assigned(FOnChange) then OnChange(Self);
end;

{TAxis methods --------------------------------------------------------------}
{Constructor and Destructor:-------------------------------------------------}
{------------------------------------------------------------------------------
  Constructor: TAxis.Create
  Description: standard Constructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: creates the subcomponents and sets various Properties
 Known Issues:
 ------------------------------------------------------------------------------}
Constructor TAxis.Create(AOwner: TPersistent);
begin
{First call the ancestor:}
  inherited Create(AOwner);

{Create Pen:}
  FPen := TPen.Create;
  FPen.Color := clRed;

  FLabels := TAxisLabel.Create(Self);
  FLabels.OnChange := StyleChange;
  FLabelText := TStringList.Create;

{create the Title geometry manager:}
  FTitle := TTitle.Create(Self);
  FTitle.OnChange := StyleChange;
  FTitle.OnCaptionChange := TitleChange;
  FTitle.Caption := 'X-Axis';
  FTitle.Font.Size := 10;

  FArrowSize := 10;
  FAutoScale := TRUE;
  SetDirection(dHorizontal);
  FMin := 0;
  FMax := 10;
  FTickSize := 10;
  FTickNum := 5;
  Alignment := taRightJustify;
  Visible := TRUE;
  ReScale;
end;

{------------------------------------------------------------------------------
  Destructor: TAxis.Destroy
  Description: standard Destructor
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: frees the subcomponents and the OnChange event
 Known Issues:
 ------------------------------------------------------------------------------}
Destructor TAxis.Destroy;
begin
  FOnChange := nil;
{Put your de-allocation, etc, here:}
  FLabels.Free;
  FPen.Free;
  FTitle.Free;
  FLabelText.Free;
  
{then call ancestor:}
  inherited Destroy;
end;
{End Constructor and Destructor:---------------------------------------------}

{------------------------------------------------------------------------------
    Procedure: TAxis.TitleChange
  Description: sets the Name and Label's Name
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: responds to a change in the Title
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.TitleChange(Sender: TObject);
begin
  if (Pos('xis', FTitle.Caption) > 0) then
  begin
    Name := FTitle.Caption;
    FLabels.Name := FTitle.Caption + ' Labels';
  end
  else
  begin
{Stick Axis in in the names:}
    Name := FTitle.Caption + ' Axis';
    FLabels.Name := FTitle.Caption + ' Axis Labels';
  end;
end;

{Begin normal Set Procedures -------------------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TAxis.SetArrowSize
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the ArrowSize Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetArrowSize(Value: Byte);
begin
  if (Value = FArrowSize) then exit;

  FArrowSize := Value;
  StyleChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetAutoScale
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the AutoScale Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetAutoScale(Value: Boolean);
begin
  if (Value = FAutoScale) then exit;

  FAutoScale := Value;
  StyleChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetDirection
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Direction Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetDirection(Value: TDirection);
begin
  if (Value = FDirection) then exit;

  FDirection := Value;
  FTitle.Direction := Value;
{TTitle.SetDirection usually fires the OnChange:}
  if ((not FTitle.Visible) and
      assigned(FOnChange) and
      Visible) then OnChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetIntercept
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Intercept virtual Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetIntercept(Value: Single);
begin
  if (FIntercept = Value) then exit;
  FIntercept := Value;
  StyleChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetLogScale
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the LogScale Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetLogScale(Value: Boolean);
begin
  if (Value = FLogScale) then exit;
  if (Value = TRUE) then
  begin {we are going to a log scale:}
    if (FMin <= 0) then exit;
    if (FMax <= 0) then exit;
  end;

  FLogScale := Value;
  ReScale;
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetMin
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Min Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetMin(Value: Single);
begin
  if (Value = FMin) then exit;
  if (Value >= FMax) then exit;
  if ((Value <= 0) and (FLogScale)) then exit;

  FMin := Value;
  ReScale;
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetMax
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Max Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetMax(Value: Single);
begin
  if (Value = FMax) then exit;
  if (Value <= FMin) then exit;

  FMax := Value;
  ReScale;
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetMinFromSeries
  Description: property Setting procedure for calling by a Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Min Property when new data is added to a Series
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetMinFromSeries(Value: Single);
begin
  if (Value >= FMin) then exit;
  if ((Value <= 0) and (FLogScale)) then exit;

  FMin := Value;
  Rescale;
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetMaxFromSeries
  Description: property Setting procedure for calling by a Series
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Max Property when new data is added to a Series
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetMaxFromSeries(Value: Single);
begin
  if (Value <= FMax) then exit;

  FMax := Value;
  if ((TPlot(Owner).DisplayMode = dmRun) and
      (FDirection = dHorizontal)) then
  begin
{We are in a "run", and so we can expect more data with increasing X values.
 Rather than force a complete screen re-draw every time a data point is
 added, we extend the X Axis by 100%:}
    FMax := 2.0 * FMax;
  end;
  Rescale;
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetPen
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Pen Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetPen(Value: TPen);
begin
  FPen.Assign(Value);
  {FFont.Color := FPen.Color;
  FLabels.Font.Color := FPen.Color;}
  StyleChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetOrientation
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Orientation Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetOrientation(Value: TOrientation);
begin
  {if (Value = FTickDirection) then exit;}

  FTickDirection := Value;
  if (FTickDirection = orRight) then
    FTickSign := 1
   else
    FTickSign := -1;
{check the names of the titles and labels}
  {TitleChange(Self);}
  StyleChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetStepSize
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the StepSize (distance between ticks) Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetStepSize(Value: Single);
begin
  if (FAutoScale) then exit;
  if (Value = FStepSize) then exit;

  FStepSize := Value;
  StyleChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetTickMinor
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the TickMinor (number of minor ticks) Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetTickMinor(Value: Byte);
begin
  if (Value = FTickMinor) then exit;
{limit the number of minors:}  
  if (Value > 9) then
    Value := 9;

  FTickMinor := Value;
  StyleChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetTickNum
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the TickNum Property
 Known Issues:
 ------------------------------------------------------------------------------
procedure TAxis.SetTickNum(Value: Byte);
begin
  if (Value = FTickNum) then exit;

  FTickNum := Value;
  ReScale;
end;}

{------------------------------------------------------------------------------
    Procedure: TAxis.SetTickSize
  Description: standard property Set procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the TickSize Property
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetTickSize(Value: Byte);
begin
  if (Value = FTickSize) then exit;

  FTickSize := Value;
  ReScale;
end;

{Various other Functions and Procedures--------------------------------------}
{------------------------------------------------------------------------------
    Procedure: TAxis.StyleChange
  Description: event firing proedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: fires the OnChange event
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.StyleChange(Sender: TObject);
begin
  if (assigned(FOnChange) and Visible) then OnChange(Sender);
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.ReScale
  Description: geometry manager
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: determines the ticks and labels
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.ReScale;
{This method determines the Axis geometry (StepStart and StepSize).}
var
  RoughStepSize: Single;
  Mantissa: Extended;
  Exponent: Integer;
begin
  if (not FAutoScale) then
  begin
    FStepStart := FMin;
    FSpan := FMax - FMin;
    exit;
  end;

  if (FLogScale) then
  begin
    FLogSpan := Log10(FMax / FMin);
    DeSci(FMin, Mantissa, Exponent);
{work out a starting point, 1 x 10^Exponent:}
    FStepStart := IntPower(10.0, Exponent);

    if (not FAutoScale) then
    begin
      if (FLogSpan >= 2) then
      begin
{many decades of data:}
        if (not FAutoScale) then
          FStepSize := 10;
      end
      else
      begin
        RoughStepSize := FLogSpan / (FTickNum+1);
        RoughStepSize := Power(10.0, RoughStepSize);
        if (RoughStepSize > 1.5) then
        begin
{get the Mantissa and Exponent:}
          DeSci(RoughStepSize, Mantissa, Exponent);
          FStepSize := Round(Mantissa) * IntPower(10.0, Exponent);
        end
        else
        begin
          FStepSize := RoughStepSize;
        end;
{$IFDEF DELPHI3_UP}
        Assert(FStepSize > 1.0,
          'TAxis.ReScale Error: The calculated StepSize on a Log scale is ' +
            FloatToStr(FStepSize));
{$ENDIF}
      end; {how big is FLogSpan ?}
    end; {not AutoScale}
    while (FStepStart < FMin) do
{go to next multiple of FStepSize:}
      FStepStart := FStepSize * FStepStart;
  end
  else
  begin {normal linear scale:}
    FSpan := FMax - FMin;
    if ((FAutoScale) or (FStepSize <= 0)) then
    begin
      RoughStepSize := FSpan / (FTickNum+1);
{get the Mantissa and Exponent:}
      DeSci(RoughStepSize, Mantissa, Exponent);
      FStepSize := Round(Mantissa) * IntPower(10.0, Exponent);
      {FTickNum := Trunc(FSpan / FStepSize);}
    end;
    FStepStart := FStepSize * Int((FMin / FStepSize) + 0.999);
    while (FStepStart < FMin) do
{increase FStepStart by FStepSize:}
      FStepStart := FStepSize + FStepStart;
  end;

  StyleChange(Self);
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.Draw
  Description: standard Drawing procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: draws the Axis on a given canvas
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.Draw(ACanvas: TCanvas);
{Comments:
 This method is quite complex, in a tedious way.
 It has to account for the following variations:
    1. Visible or not;
    2. Arrows visible or not;
    3. Axis direction (Horizontal or vertical);
    4. Tick (and Label and Title) direction);
    5. Title Alignment Direction and Orientation;
    6. Tick, Label and Title visibility.
 An added complication is that we must generate a vertical font for the Title
 of vertical axes. Note that this only works with TrueType fonts - NOT fonts
 that are purely screen or printer.}
var
  i,
  iX,
  iY,
  FontHeight,
  FontWidth,
  iFontWidth,
  FontDescent,
  MinorTickSize: Integer;
  MinorStepSize,
  MinorStepStart,
  YValue,
  XValue: Single;
  TheText: String;

  function GetNextXValue(XValue: Single): Single;
  begin
    if (FLogScale) then
      GetNextXValue := XValue * FStepSize
     else
      GetNextXValue := XValue + FStepSize;
  end;

  function GetNextMinorXValue(XValue: Single): Single;
  begin
    if (FLogScale) then
      GetNextMinorXValue := XValue * MinorStepSize
     else
      GetNextMinorXValue := XValue + MinorStepSize;
  end;

begin
{the most common reason for exit:}
  if (not Visible) then exit;
{$IFDEF DELPHI3_UP}
  Assert(ACanvas <> nil, 'TAxis.Draw: ACanvas is nil !');
{$ENDIF}

  FontWidth := 1;
  ACanvas.Pen.Assign(FPen);
  if (FDirection = dHorizontal) then
  begin
{Draw the axis:}
    ACanvas.MoveTo(Left, MidY);
    ACanvas.LineTo(Right, MidY);
{Draw the arrows on the axis:}
    if (FArrowSize > 0) then
    begin
      if (Alignment = taLeftJustify) then
      begin
        ACanvas.MoveTo(Left+FArrowSize, Top);
        ACanvas.LineTo(Left, MidY);
        ACanvas.LineTo(Left+FArrowSize, Bottom);
      end;
      if (Alignment = taRightJustify) then
      begin
        ACanvas.LineTo(Right-FArrowSize, Top);
        ACanvas.MoveTo(Right, MidY);
        ACanvas.LineTo(Right-FArrowSize, Bottom);
      end; {taCenter therefore means no arrows !}
    end;

{Major Ticks on the axis:}
    if (FTickSize > 0) then
    begin
      iY := MidY;
      XValue := FStepStart;
      while (XValue < FMax) do
      begin
        iX := FofX(XValue);
        ACanvas.MoveTo(iX, iY);
        ACanvas.LineTo(iX, iY + FTickSign*FTickSize);
        XValue := GetNextXValue(XValue);
      end;
    end;

{Minor Ticks on the axis:}
    if ((FTickSize > 1) and
        (FTickMinor > 0)) then
    begin
{find out where the minors start:}
      MinorStepSize := FStepSize / (FTickMinor+1);
      MinorStepStart := FStepStart;
      MinorTickSize := FTickSign * FTickSize div 2;
      while ((MinorStepStart - MinorStepSize) >= FMin) do
        MinorStepStart := MinorStepStart - MinorStepSize;
      iY := MidY;
      XValue := MinorStepStart;
      while (XValue < FMax) do
      begin
        iX := FofX(XValue);
        ACanvas.MoveTo(iX, iY);
        ACanvas.LineTo(iX, iY + MinorTickSize);
        XValue := GetNextMinorXValue(XValue);
      end;
    end;

{Set text output orientation.}
{Labels on the axis:}
    if (FLabels.Visible) then
    begin
      ACanvas.Font.Assign(FLabels.Font);
      FontHeight := ACanvas.TextHeight('9');
      iY := MidY + FTickSign*FTickSize;
      if (FTickDirection = orLeft) then
        Dec(iY, FontHeight);
      iX := 0;
      XValue := FStepStart;
      i := 0;
      while (XValue < FMax) do
      begin
        if (FLabelText.Count > 0) then
          if (i < FLabelText.Count) then
            TheText := LabelText.Strings[i]
           else
            break {if there are not enough LabelText strings}
        else
          TheText := LabelToStrF(XValue);
        iFontWidth := ACanvas.TextWidth(TheText);
        if (iFontWidth > FontWidth) then
          FontWidth := iFontWidth;
        iX := FofX(XValue);
        ACanvas.TextOut(iX - iFontWidth div 2, iY, TheText);
        XValue := GetNextXValue(XValue);
        Inc(i);
      end;

{record the position of the labels for use by TPlot:}
      FLabels.Left := FofX(FStepStart) - FontWidth div 2;
      FLabels.Top := iY;
      FLabels.Bottom := iY + FontHeight;
      FLabels.Right := iX + FontWidth div 2;
    end;

    SetupHorizontalEnvelope;
{Print the axis Title:}
    FTitle.Draw(ACanvas);
  end

{Draw the Vertical axis:}
  else
  begin
    ACanvas.MoveTo(MidX, Bottom);
    ACanvas.LineTo(MidX, Top);
{Draw the arrows on the axis:}
    if (FArrowSize > 0) then
    begin
      ACanvas.LineTo(Left, Top+FArrowSize);
      ACanvas.MoveTo(MidX, Top);
      ACanvas.LineTo(Right, Top+FArrowSize);
    end;

{Ticks on the axis:}
    if (FTickSize > 0) then
    begin
      iX := MidX;
      YValue := FStepStart;
      while (YValue < FMax) do
      begin
        iY := FofY(YValue);
        ACanvas.MoveTo(iX, iY);
        ACanvas.LineTo(iX + FTickSign*FTickSize, iY);
        YValue := GetNextXValue(YValue);
      end;
    end;

{Minor Ticks on the axis:}
    if ((FTickSize > 1) and
        (FTickMinor > 0)) then
    begin
{find out where the minors start:}
      MinorStepSize := FStepSize / (FTickMinor + 1);
      MinorStepStart := FStepStart;
      MinorTickSize := FTickSign * FTickSize div 2;
      while ((MinorStepStart - MinorStepSize) >= FMin) do
        MinorStepStart := MinorStepStart - MinorStepSize;
      iX := MidX;
      YValue := MinorStepStart;
      while (YValue < FMax) do
      begin
        iY := FofY(YValue);
        ACanvas.MoveTo(iX, iY);
        ACanvas.LineTo(iX + MinorTickSize, iY);
        YValue := GetNextMinorXValue(YValue);
      end;
    end;

    FontWidth := 1; {see below}
{Labels on the axis:}
    if (FLabels.Visible) then
    begin
      ACanvas.Font.Assign(FLabels.Font);
      FontWidth := ACanvas.TextWidth('9');
      FontHeight := ACanvas.TextHeight('9');
{We could call GetOutlineTextMetrics to get
 the Descent (gap between baseline and bottom of a font), but:}
      FontDescent := FontHeight div 5;
      iX := MidX + FTickSign*(FTickSize + FontWidth div 5);
      iY := 0; {see below}
      YValue := FStepStart;
      i := 0;
      while (YValue < FMax) do
      begin
        if (FLabelText.Count > 0) then
          if (i < FLabelText.Count) then
            TheText := LabelText.Strings[i]
           else
            break {if there are not enough LabelText strings}
        else
          TheText := LabelToStrF(YValue);
        iY := FofY(YValue) - FontHeight + FontDescent;
        iFontWidth := ACanvas.TextWidth(TheText);
{remember which label is widest:}
        if (FontWidth < iFontWidth) then
          FontWidth := iFontWidth;
        if (FTickDirection = orRight) then
          ACanvas.TextOut(iX, iY, TheText)
         else
          ACanvas.TextOut(iX - iFontWidth, iY, TheText);
        YValue := GetNextXValue(YValue);
      end;

      FLabels.Bottom := FofY(FStepStart);
{record the position of the labels for use by TPlot:}
      FLabels.Top := iY - Abs(ACanvas.Font.Height);
      if (FTickDirection = orRight) then
      begin
        FLabels.Left := iX;
        FLabels.Right := iX + FontWidth;
      end
      else
      begin
        FLabels.Left := iX - FontWidth;
        FLabels.Right := iX;
      end;
    end; {Labels Visible}

    SetupVerticalEnvelope(FontWidth);
{Print the axis Title:}
    FTitle.Draw(ACanvas);
  end; {Horizontal or Vertical}
end;

{------------------------------------------------------------------------------
     Function: TAxis.FofX
  Description: standard X transform
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: returns the pixel position on screen as a function of the real data ordinate X
 Known Issues:
 ------------------------------------------------------------------------------}
function TAxis.FofX(X: Single): Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(FDirection = dHorizontal, 'A vertical Axis cannot return F(X) !');
{$ENDIF}

  if (FLogScale) then
    FofX := Round(Left + Width * ((Log10(X / FMin)) / FLogSpan))
   else
    FofX := Round(Left + Width * ((X - FMin) / (FSpan)));
end;

{------------------------------------------------------------------------------
     Function: TAxis.FofY
  Description: standard Y transform
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: returns the pixel position on screen as a function of the real data co-ordinate Y
 Known Issues:
 ------------------------------------------------------------------------------}
function TAxis.FofY(Y: Single): Integer;
begin
{$IFDEF DELPHI3_UP}
  Assert(FDirection = dVertical, 'A Horizontal Axis cannot return F(Y) !');
{$ENDIF}

  if (FLogScale) then
    FofY := Round(Bottom - Height * ((Log10(Y / FMin)) / FLogSpan))
   else
    FofY := Round(Bottom - Height * ((Y - FMin) / (FSpan)));
end;

{------------------------------------------------------------------------------
     Function: TAxis.XofF
  Description: inverse X transform
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: returns the real data ordinate X as a function of the pixel position on screen
 Known Issues:
 ------------------------------------------------------------------------------}
function TAxis.XofF(F: Integer): Single;
{this function returns the real data ordinate X
 as a function of the pixel position F on screen:}
begin
{$IFDEF DELPHI3_UP}
  Assert(FDirection = dHorizontal, 'A Vertical Axis cannot return F(X) !');
{$ENDIF}

  if (FLogScale) then
    XofF := FMin * Power(10.0, (FLogSpan * (F-Left) / Width))
   else
    XofF := FSpan * ((F-Left) / Width) + FMin;
end;

{------------------------------------------------------------------------------
     Function: TAxis.YofF
  Description: inverse Y transform
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: returns the real data ordinate Y as a function of the pixel position on screen
 Known Issues:
 ------------------------------------------------------------------------------}
function TAxis.YofF(F: Integer): Single;
{this function returns the real data ordinate X
 as a function of the pixel position F on screen:}
begin
{$IFDEF DELPHI3_UP}
  Assert(FDirection = dVertical, 'A Horizontal Axis cannot return F(Y) !');
{$ENDIF}

  if (FLogScale) then
    YofF := FMin * Power(10.0, (FLogSpan * (Bottom-F) / Height))
   else
    YofF := FSpan * ((Bottom-F) / Height) + FMin;
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.DeSci
  Description: breaks a number up into its Mantissa and Exponent
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: Tick and Label scaling
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.DeSci(ExtNumber: Extended; var Mantissa: Extended; var Exponent: Integer);
var
  TheLog: Extended;
  TheSign: Extended;
begin
  TheSign := 1;
  if (ExtNumber < 0) then
  begin
    TheSign := -1;
    ExtNumber := -ExtNumber;
  end;

  TheLog := Log10(ExtNumber);
  Exponent := Floor(TheLog);
  Mantissa := TheLog - Exponent;
  Mantissa := Power(10.0, Mantissa);
  if (TheSign < 0) then Mantissa := -Mantissa;
end;

{------------------------------------------------------------------------------
     Function: TAxis.StrToLabel
  Description: converts a string to a number, depending on the NumberFormat
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: user IO
 Known Issues:
 ------------------------------------------------------------------------------}
function TAxis.StrToLabel(Value: String): Single;
begin
  case (FLabels.NumberFormat) of
    lfGeneral .. lfCurrency:
      StrToLabel := StrToFloat(Value);
    lfPercent:
      StrToLabel := StrToFloat(Value) / 100;
    lfSeconds:
      StrToLabel := StrToFloat(Value);
    lfMinutes:
      StrToLabel := 60 * StrToFloat(Value);
    lfHours:
      StrToLabel := 3600 * StrToFloat(Value);
    lfDays:
      StrToLabel := 86400 * StrToFloat(Value);
    lfShortTime:
      StrToLabel := StrToDateTime(Value);
    lfShortDate:
      StrToLabel := StrToDateTime(Value);
    else
      StrToLabel := 0.0;
  end;
end;

{------------------------------------------------------------------------------
     Function: TAxis.LabelToStrF
  Description: converts a number to a string, depending on the NumberFormat
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: user IO
 Known Issues:
 ------------------------------------------------------------------------------}
function TAxis.LabelToStrF(Value: Single): String;
var
  TheText: String;
  TheDateTime: TDateTime;
begin
  case (FLabels.NumberFormat) of
    lfGeneral .. lfCurrency:
      TheText := FloatToStrF(Value, TFloatFormat(FLabels.NumberFormat),
        FLabels.Precision, FLabels.Digits);
    lfPercent:
      TheText := FloatToStrF(100 * Value, TFloatFormat(FLabels.NumberFormat),
        FLabels.Precision, FLabels.Digits);
    lfSeconds:
      TheText := FloatToStrF(Round(Value), ffGeneral,
        FLabels.Precision, FLabels.Digits);
    lfMinutes:
      TheText := FloatToStrF(Round(Value / 60), ffGeneral,
        FLabels.Precision, FLabels.Digits);
    lfHours:
      TheText := FloatToStrF(Round(Value / 3600), ffGeneral,
        FLabels.Precision, FLabels.Digits);
    lfDays:
      TheText := FloatToStrF(Round(Value / 86400), ffGeneral,
        FLabels.Precision, FLabels.Digits);
    lfShortTime:
      begin
        TheDateTime := Value;
        TheText := FormatDateTime('t', TheDateTime);
      end;
    lfShortDate:
      begin
        TheDateTime := Value;
        TheText := FormatDateTime('ddddd', TheDateTime);
      end;
  end;

  LabelToStrF := TheText;
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetupHorizontalEnvelope
  Description: sets up the Horizontal (X Axis) envelope around which the Title dances
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manages the appearance of the Axis
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetupHorizontalEnvelope;
var
  TheRect: TRect;
begin
  TheRect.Left := Left;
  TheRect.Right := Right;
  if (FTickDirection = orLeft) then
  begin
    TheRect.Top := MidY - FTickSize;
    TheRect.Bottom := MidY + 1;
    if (FLabels.Visible) then
      TheRect.Top := TheRect.Top - Abs(FLabels.Font.Height);
  end
  else  {oRight}
  begin
    TheRect.Top := MidY - 1;
    TheRect.Bottom := MidY + FTickSize;
    if (FLabels.Visible) then
      TheRect.Bottom := TheRect.Bottom + Abs(FLabels.Font.Height);
  end; {FTickDirection}
  FTitle.Envelope := TheRect;
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.SetupVerticalEnvelope
  Description: sets up the Vertical (Y Axis) envelope around which the Title dances
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manages the appearance of the Axis
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.SetupVerticalEnvelope(StrWidth: Integer);
var
  TheRect: TRect;
begin
  TheRect.Top := Top;
  TheRect.Bottom := Bottom;
  if (FTickDirection = orLeft) then
  begin
    TheRect.Left := MidX - FTickSize - Abs(FLabels.Font.Height) div 2;
    TheRect.Right := MidX + 1;
    if (FLabels.Visible) then
      TheRect.Left := TheRect.Left - StrWidth;
  end
  else {oRight}
  begin
    TheRect.Left := MidX - 1;
    TheRect.Right := MidX + FTickSize;
    if (FLabels.Visible) then
      TheRect.Right := TheRect.Right + StrWidth;
  end; {FTickDirection}
  FTitle.Envelope := TheRect;
end;

{------------------------------------------------------------------------------
    Procedure: TAxis.Assign
  Description: standard Assign method
       Author: Mat Ballard
 Date created: 07/06/2000
Date modified: 07/06/2000 by Mat Ballard
      Purpose: implements Assign
 Known Issues:
 ------------------------------------------------------------------------------}
{procedure TAxis.Assign(Source: TPersistent);
begin
  inherited Assign(Source);
  FArrowSize := TAxis(Source).ArrowSize;
  FDirection := TAxis(Source).Direction;
  FIntercept := TAxis(Source).Intercept;
  FLogscale := TAxis(Source).Logscale;
  FMax := TAxis(Source).Max;
  FMin := TAxis(Source).Min;
  FStepSize := TAxis(Source).StepSize;
  FTickDirection := TAxis(Source).TickDirection;
  FTickNum := TAxis(Source).TickNum;
  FTickSize := TAxis(Source).TickSize;

  FLabels.Assign(TAxis(Source).Labels);
  FPen.Assign(TAxis(Source).Pen);
  FTitle.Assign(TAxis(Source).Title);
end;}

{------------------------------------------------------------------------------
    Procedure: TAxis.AssignTo
  Description: standard AssignTo method
       Author: Mat Ballard
 Date created: 07/06/2000
Date modified: 07/06/2000 by Mat Ballard
      Purpose: implements AssignTo
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxis.AssignTo(Dest: TPersistent);
begin
  inherited AssignTo(Dest);
  TAxis(Dest).ArrowSize := FArrowSize;
  TAxis(Dest).Direction := FDirection;
  {TAxis(Dest).Intercept := FIntercept;}
  TAxis(Dest).Logscale := FLogscale;
  TAxis(Dest).Max := FMax;
  TAxis(Dest).Min := FMin;
  TAxis(Dest).StepSize := FStepSize;
  TAxis(Dest).TickDirection := FTickDirection;
  TAxis(Dest).TickSize := FTickSize;

  TAxis(Dest).Labels.Assign(FLabels);
  TAxis(Dest).Pen.Assign(FPen);
  TAxis(Dest).Title.Assign(FTitle);
end;

procedure TAxis.SetLabelText(Value: TStringList);
begin
  FLabelText.Assign(Value);
  StyleChange(Self);
end;

end.
