unit AxisEdit;

{$I Plot.inc}

{-----------------------------------------------------------------------------
The contents of this file are subject to the Mozilla Public License
Version 1.1 (the "License"); you may not use this file except in compliance
with the License. You may obtain a copy of the License at
http://www.mozilla.org/MPL/MPL-1.1.html

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either expressed or implied. See the License for
the specific language governing rights and limitations under the License.

The Original Code is: AxisEdit.pas, released 12 September 2000.

The Initial Developer of the Original Code is Mat Ballard.
Portions created by Mat Ballard are Copyright (C) 1999 Mat Ballard.
Portions created by Microsoft are Copyright (C) 1998, 1999 Microsoft Corp.
All Rights Reserved.

Contributor(s): Mat Ballard                 e-mail: mat.ballard@molsci.csiro.au.

Last Modified: 08/11/2000
Current Version: 1.00

You may retrieve the latest version of this file from:

        http://Chemware.hypermart.net/

This work was created with the Project JEDI VCL guidelines:

        http://www.delphi-jedi.org/Jedi:VCLVCL

in mind. 

Purpose:
To facilitate user manipluation of Axis properties.

Known Issues:
-----------------------------------------------------------------------------}

interface

uses
  Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, Buttons, ExtCtrls,

{These are the D1 units, aliased to Windows:}
{$IFDEF WIN}
  WinTypes, WinProcs,
{$ENDIF}

  Axis, Misc, Titles;

type
  TAxisProperty = record
    Visible: Boolean;
    LabelFormat:TLabelFormat;
    LabelDigits: Byte;
    LabelPrecision: Byte;
    PenColor: TColor;
    PenWidthIndex: Integer;
    PenStyleIndex: Integer;
    TickSize: Byte;
    TickDirection: TOrientation;
    TickStepSize: Single;
    TickMinors: Byte;
    ScaleMin: Single;
    ScaleMax: Single;
    ScaleIntercept: Single;
    ScaleAuto: Boolean;
    ScaleLog: Boolean;
    ArrowSize: Byte;
    ArrowDirection: TAlignment;
  end;

  TAxisEditorForm = class(TForm)
    HelpBitBtn: TBitBtn;
    CancelBitBtn: TBitBtn;
    OKBitBtn: TBitBtn;
    VisibleCheckBox: TCheckBox;
    PenGroupBox: TGroupBox;
    PenColorComboBox: TComboBox;
    Label2: TLabel;
    Label4: TLabel;
    PenWidthComboBox: TComboBox;
    PenStyleComboBox: TComboBox;
    Label5: TLabel;
    GroupBox1: TGroupBox;
    Label6: TLabel;
    Label7: TLabel;
    Label9: TLabel;
    DigitsComboBox: TComboBox;
    PrecisionComboBox: TComboBox;
    FormatComboBox: TComboBox;
    GroupBox2: TGroupBox;
    Label10: TLabel;
    Label11: TLabel;
    TickSizeEdit: TEdit;
    TickDirectionComboBox: TComboBox;
    StepSizeLabel: TLabel;
    StepSizeEdit: TEdit;
    GroupBox3: TGroupBox;
    InterceptLabel: TLabel;
    InterceptEdit: TEdit;
    LogScaleCheckBox: TCheckBox;
    MinLabel: TLabel;
    MaxLabel: TLabel;
    MinEdit: TEdit;
    MaxEdit: TEdit;
    GroupBox4: TGroupBox;
    Label1: TLabel;
    Label3: TLabel;
    ArrowSizeEdit: TEdit;
    ArrowDirectionComboBox: TComboBox;
    TitleEdit: TEdit;
    Label13: TLabel;
    MinorsComboBox: TComboBox;
    NoComboBox: TComboBox;
    Label14: TLabel;
    AutoScaleCheckBox: TCheckBox;
    procedure FormCreate(Sender: TObject);
    procedure IntEditKeyPress(Sender: TObject; var Key: Char);
    procedure FloatEditKeyPress(Sender: TObject; var Key: Char);
    procedure FormatComboBoxClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure PenStyleComboBoxClick(Sender: TObject);
    procedure FormActivate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure NoComboBoxClick(Sender: TObject);
    procedure TitleEditChange(Sender: TObject);
    procedure DigitsComboBoxClick(Sender: TObject);
    procedure PrecisionComboBoxClick(Sender: TObject);
    procedure TickDirectionComboBoxClick(Sender: TObject);
    procedure TickSizeEditChange(Sender: TObject);
    procedure StepSizeEditChange(Sender: TObject);
    procedure MinEditChange(Sender: TObject);
    procedure MaxEditChange(Sender: TObject);
    procedure InterceptEditChange(Sender: TObject);
    procedure MinorsComboBoxClick(Sender: TObject);
    procedure ArrowDirectionComboBoxClick(Sender: TObject);
    procedure ArrowSizeEditChange(Sender: TObject);
    procedure VisibleCheckBoxClick(Sender: TObject);
    procedure AutoScaleCheckBoxClick(Sender: TObject);
    procedure LogScaleCheckBoxClick(Sender: TObject);
    procedure PenColorComboBoxClick(Sender: TObject);
    procedure PenWidthComboBoxClick(Sender: TObject);
  private
    pAXP: ^TAxisProperty;
    CurrentIndex: Integer;
  public
    AxisPropertyList: TList;
    AxisNames: TStringList;
    HistoryMode: Boolean;

    function AddAxis(AName: String; AnAxisProperty: TAxisProperty): Integer;
    procedure SelectAxis(Index: Integer);
  end;

var
  AxisEditorForm: TAxisEditorForm;

implementation

{$R *.DFM}

{------------------------------------------------------------------------------
    Procedure: TAxisEditorForm.FormCreate
  Description: standard FormCreate procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the position and populates the Color combo
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisEditorForm.FormCreate(Sender: TObject);
var
  i: TColor;
begin
  for i := 0 to MY_COLORS_MAX do
  begin
    PenColorComboBox.Items.Add(MyColors[i].Name);
  end;

  AxisPropertyList := TList.Create;
  AxisNames := TStringList.Create;

  Left := 10;
  Top := 10;
end;

{------------------------------------------------------------------------------
    Procedure: TAxisEditorForm.FormClose
  Description: standard FormClose procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: checks for valid user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisEditorForm.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
{Check that floating point values are valid:
  Action := caHide;
  try
    if (not AutoScaleCheckBox.Checked) then
    begin
      MinEdit.SetFocus;
      Test := StrToFloat(MinEdit.Text)
      MaxEdit.SetFocus;
      TestMax := StrToFloat(MaxEdit.Text)
      if (TestMax < Test) then
        ERangeError.CreateFmt('The Min (%g) MUST be less than the Max (%g)',
          [Test, TestMax]);
      InterceptEdit.SetFocus;
      Test := StrToFloat(InterceptEdit.Text)
      StepSizeEdit.SetFocus;
      Test := Test + StrToFloat(StepSizeEdit.Text);
  {We do the above and below the squelch the warning messages.
      Tag := Trunc(Test);
    end;
  except
{Prevent closure:
    Action := caNone;
  end;}
end;

{------------------------------------------------------------------------------
    Procedure: TAxisEditorForm.IntEditKeyPress
  Description: allows only integer keys to be pressed
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: restrict user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisEditorForm.IntEditKeyPress(Sender: TObject;
  var Key: Char);
begin
  if not (Key in [#8, '0', '1', '2', '3', '4', '5', '6', '7', '8', '9']) then
    Key := Chr(0);
end;

{------------------------------------------------------------------------------
    Procedure: TAxisEditorForm.FloatEditKeyPress
  Description: allows only "real" keys to be pressed
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: restrict user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisEditorForm.FloatEditKeyPress(Sender: TObject;
  var Key: Char);
begin
  if not (Key in [#8, '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '-', '.', 'e']) then
    Key := Chr(0);
end;

{------------------------------------------------------------------------------
    Procedure: TAxisEditorForm.FormatComboBoxClick
  Description: responds to NumberFormat selection
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: loads the Digits combo depending on selection
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisEditorForm.FormatComboBoxClick(Sender: TObject);
var
  i, iMax, OldIndex: Integer;
begin
  OldIndex := DigitsComboBox.ItemIndex;
  DigitsComboBox.Clear;
  iMax := 18;
  if ((FormatComboBox.ItemIndex = Ord(ffGeneral)) or
      (FormatComboBox.ItemIndex = Ord(ffExponent))) then
    iMax := 4;
  for i := 0 to iMax do
  begin
    DigitsComboBox.Items.Add(IntToStr(i));
  end;
  if (OldIndex > iMax) then OldIndex := iMax;
  DigitsComboBox.ItemIndex := OldIndex;
end;

{------------------------------------------------------------------------------
    Procedure: TAxisEditorForm.PenStyleComboBoxClick
  Description: responds to user selection
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Width to 1 if neccessary
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisEditorForm.PenStyleComboBoxClick(Sender: TObject);
begin
  pAXP^.PenStyleIndex := PenStyleComboBox.ItemIndex;
  if (PenStyleComboBox.ItemIndex > Ord(psSolid)) then
    PenWidthComboBox.ItemIndex := 1;
end;

{------------------------------------------------------------------------------
    Procedure: TAxisEditorForm.FormActivate
  Description: standard FormActivate procedure
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: sets the Visibility of the Max Label
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisEditorForm.FormActivate(Sender: TObject);
begin
  if (MaxEdit.Enabled = FALSE) then
    MaxLabel.Enabled := FALSE;
end;

procedure TAxisEditorForm.FormDestroy(Sender: TObject);
var
  i: Integer;
begin
  for i := 0 to AxisPropertyList.Count-1 do
    FreeMem(AxisPropertyList.Items[i], SizeOf(TAxisProperty));

  AxisPropertyList.Free;
  AxisNames.Free;
end;

{------------------------------------------------------------------------------
    Procedure: TAxisEditorForm.SelectAxis
  Description: responds to selection of a new Axis
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: manage user input
 Known Issues:
 ------------------------------------------------------------------------------}
procedure TAxisEditorForm.SelectAxis(Index: Integer);
begin
  pAXP := AxisPropertyList.Items[Index];
  CurrentIndex := Index;

{Load the combo box:}
  ArrowDirectionComboBox.Clear;
  TickDirectionComboBox.Clear;
  if (Index = 0) then {X Axis:}
  begin
    if (HistoryMode) then
      MaxEdit.Enabled := FALSE;
    ArrowDirectionComboBox.Items.Add('Left');
    ArrowDirectionComboBox.Items.Add('Right');
    ArrowDirectionComboBox.Items.Add('None');
    TickDirectionComboBox.Items.Add('Up');
    TickDirectionComboBox.Items.Add('Down');
  end
  else
  begin
    MaxEdit.Enabled := TRUE;
    ArrowDirectionComboBox.Items.Add('Down');
    ArrowDirectionComboBox.Items.Add('Up');
    ArrowDirectionComboBox.Items.Add('None');
    TickDirectionComboBox.Items.Add('Left');
    TickDirectionComboBox.Items.Add('Right');
  end;                  

  TitleEdit.Text := AxisNames.Strings[Index];
  FormatComboBox.ItemIndex := Ord(pAXP^.LabelFormat);
  DigitsComboBox.ItemIndex := pAXP^.LabelDigits;
  PrecisionComboBox.ItemIndex := pAXP^.LabelPrecision;
  PenColorComboBox.ItemIndex := pAXP^.PenColor;
  PenWidthComboBox.ItemIndex := pAXP^.PenWidthIndex;
  PenStyleComboBox.ItemIndex := pAXP^.PenStyleIndex;
  TickSizeEdit.Text := IntToStr(pAXP^.TickSize);
  TickDirectionComboBox.ItemIndex := Ord(pAXP^.TickDirection);
  StepSizeEdit.Text := FloatToStrF(pAXP^.TickStepSize,
    ffGeneral, pAXP^.LabelPrecision, pAXP^.LabelDigits);
  MinorsComboBox.ItemIndex := pAXP^.TickMinors;
  MinEdit.Text := FloatToStrF(pAXP^.ScaleMin,
    ffGeneral, pAXP^.LabelPrecision, pAXP^.LabelDigits);
  MaxEdit.Text :=  FloatToStrF(pAXP^.ScaleMax,
    ffGeneral, pAXP^.LabelPrecision, pAXP^.LabelDigits);
  InterceptEdit.Text := FloatToStrF(pAXP^.ScaleIntercept,
    ffGeneral, pAXP^.LabelPrecision, pAXP^.LabelDigits);
  AutoScaleCheckBox.Checked :=  pAXP^.ScaleAuto;
  LogScaleCheckBox.Checked :=  pAXP^.ScaleLog;
  ArrowSizeEdit.Text := IntToStr(pAXP^.ArrowSize);
  ArrowDirectionComboBox.ItemIndex := Ord(pAXP^.ArrowDirection);

  AutoScaleCheckBoxClick(Self);

  VisibleCheckBox.Checked := pAXP^.Visible;
end;

procedure TAxisEditorForm.NoComboBoxClick(Sender: TObject);
var
  Test,
  TestMax: Single;
begin
  try
    if (not AutoScaleCheckBox.Checked) then
    begin
      MinEdit.SetFocus;
      Test := StrToFloat(MinEdit.Text);
      MaxEdit.SetFocus;
      TestMax := StrToFloat(MaxEdit.Text);
      if (TestMax < Test) then
        ERangeError.CreateFmt('The Min (%g) MUST be less than the Max (%g)',
          [Test, TestMax]);
      InterceptEdit.SetFocus;
      Test := StrToFloat(InterceptEdit.Text);
      StepSizeEdit.SetFocus;
      Test := Test + StrToFloat(StepSizeEdit.Text);
{We do the above and below the squelch the warning messages.}
      Tag := Trunc(Test);
    end;
    SelectAxis(NoComboBox.ItemIndex);
  except
  end;
end;

{------------------------------------------------------------------------------
     Function: TAxisEditorForm.AddAxis
  Description: adds a new Axis to the list
       Author: Mat Ballard
 Date created: 04/25/2000
Date modified: 04/25/2000 by Mat Ballard
      Purpose: stores the Axis properties
 Known Issues:
 ------------------------------------------------------------------------------}
function TAxisEditorForm.AddAxis(AName: String; AnAxisProperty: TAxisProperty): Integer;
var
  pDestination: Pointer;
  pDest, pSource: PChar;
  i: Integer;
begin
  NoComboBox.Items.Add(IntToStr(AxisNames.Count));

  AxisNames.Add(AName);

  GetMem(pDestination, SizeOf(AnAxisProperty));
  pSource := @AnAxisProperty;
{NB: this causes terminal access violations:
  System.Move(pSource, pDestination, SizeOf(AnAxisProperty));}

  pDest := pDestination;
  for i := 1 to SizeOf(AnAxisProperty) do
  begin
    pDest^ := pSource^;
    Inc(pSource);
    Inc(pDest);
  end;

  AxisPropertyList.Add(pDestination);
  AddAxis := AxisPropertyList.Count;
end;

procedure TAxisEditorForm.TitleEditChange(Sender: TObject);
begin
  AxisNames.Strings[CurrentIndex] := TitleEdit.Text;
end;

procedure TAxisEditorForm.DigitsComboBoxClick(Sender: TObject);
begin
  pAXP^.LabelDigits := DigitsComboBox.ItemIndex;
end;

procedure TAxisEditorForm.PrecisionComboBoxClick(Sender: TObject);
begin
  pAXP^.LabelPrecision := PrecisionComboBox.ItemIndex;
end;

procedure TAxisEditorForm.TickDirectionComboBoxClick(Sender: TObject);
begin
  pAXP^.TickDirection := TOrientation(TickDirectionComboBox.ItemIndex);
end;

procedure TAxisEditorForm.TickSizeEditChange(Sender: TObject);
begin
  try
    if (Length(TickSizeEdit.Text) = 0) then
      pAXP^.TickSize := 0
    else
      pAXP^.TickSize := StrToInt(TickSizeEdit.Text);
  finally
  end;
end;

procedure TAxisEditorForm.StepSizeEditChange(Sender: TObject);
begin
  try
    if (Length(StepSizeEdit.Text) = 0) then
      pAXP^.TickStepSize := 0
    else
      pAXP^.TickStepSize := StrToFloat(StepSizeEdit.Text);
  finally
  end;
end;

procedure TAxisEditorForm.MinEditChange(Sender: TObject);
begin
  try
    if (Length(MinEdit.Text) = 0) then
      pAXP^.ScaleMin := 0
    else
      pAXP^.ScaleMin := StrToFloat(MinEdit.Text);
  finally
  end;
end;

procedure TAxisEditorForm.MaxEditChange(Sender: TObject);
begin
  try
    if (Length(MaxEdit.Text) = 0) then
      pAXP^.ScaleMax := 0
    else
      pAXP^.ScaleMax := StrToFloat(MaxEdit.Text);
  finally
  end;
end;

procedure TAxisEditorForm.InterceptEditChange(Sender: TObject);
begin
  try
    if (Length(InterceptEdit.Text) = 0) then
      pAXP^.ScaleIntercept := 0
    else
      pAXP^.ScaleIntercept := StrToFloat(InterceptEdit.Text);
  finally
  end;
end;

procedure TAxisEditorForm.MinorsComboBoxClick(Sender: TObject);
begin
  pAXP^.TickMinors := MinorsComboBox.ItemIndex;
end;

procedure TAxisEditorForm.ArrowDirectionComboBoxClick(Sender: TObject);
begin
  pAXP^.ArrowDirection := TAlignment(ArrowDirectionComboBox.ItemIndex);
end;

procedure TAxisEditorForm.ArrowSizeEditChange(Sender: TObject);
begin
  try
    if (Length(ArrowSizeEdit.Text) = 0) then
      pAXP^.ArrowSize := 0
    else
      pAXP^.ArrowSize := StrToInt(ArrowSizeEdit.Text);
  finally
  end;
end;

procedure TAxisEditorForm.VisibleCheckBoxClick(Sender: TObject);
begin
  pAXP^.Visible := VisibleCheckBox.Checked;
end;

procedure TAxisEditorForm.AutoScaleCheckBoxClick(Sender: TObject);
begin
  pAXP^.ScaleAuto := AutoScaleCheckBox.Checked;

  StepSizeEdit.Enabled := not AutoScaleCheckBox.Checked;
  MinEdit.Enabled := not AutoScaleCheckBox.Checked;
  MaxEdit.Enabled := not AutoScaleCheckBox.Checked;
  InterceptEdit.Enabled := not AutoScaleCheckBox.Checked;

  StepSizeLabel.Enabled := not AutoScaleCheckBox.Checked;
  MinLabel.Enabled := not AutoScaleCheckBox.Checked;
  MaxLabel.Enabled := not AutoScaleCheckBox.Checked;
  InterceptLabel.Enabled := not AutoScaleCheckBox.Checked;
end;

procedure TAxisEditorForm.LogScaleCheckBoxClick(Sender: TObject);
begin
  pAXP^.ScaleLog := LogScaleCheckBox.Checked;
end;

procedure TAxisEditorForm.PenColorComboBoxClick(Sender: TObject);
begin
  pAXP^.PenColor := PenColorComboBox.ItemIndex;
end;

procedure TAxisEditorForm.PenWidthComboBoxClick(Sender: TObject);
begin
  pAXP^.PenWidthIndex := PenWidthComboBox.ItemIndex;
end;

end.
