{ ##
  @FILE                     PJAbout.pas
  @COMMENTS                 About Dialog Box component source file
  @PROJECT_NAME             About Dialog Box Component
  @PROJECT_DESC             Component that displays an about dialog box
  @OTHER_NAMES              + Original unit name was AboutBox.pas
                            + Changed to PJAbout.pas at v3.2
  @AUTHOR                   Peter Johnson, LLANARTH, Ceredigion, Wales, UK
  @EMAIL                    peter.johnson@openlink.org
  @WEBSITE                  http://www.delphidabbler.com/
  @COPYRIGHT                 Peter D Johnson, 1998-2003.
  @LEGAL_NOTICE             This component and source code is placed in the
                            public domain. It may be freely copied and
                            circulated on a not for profit basis providing that
                            the code is unmodified and this notice and
                            information about the author and his copyright
                            remains attached to the source code.
  @DEPENDENCIES             + PJSoft VerInfo unit v1 for Delphi 1.
                            + PJSoft PJVersionInfo unit v3.0 for Delphi 2-6.
  @HISTORY(
    @REVISION(
      @VERSION              1.0
      @DATE                 16/02/1998
      @COMMENTS             Original version - 16 bit only.
    )
    @REVISION(
      @VERSION              2.0
      @DATE                 25/04/1998
      @COMMENTS             + Added VersionInfo property to link to any
                              TPJVersionInfo component on form to allow text in
                              dialog box to come from file version info instead
                              of ProgramName, Version, Copyright and Notes
                              properties.
                            + Replaced Caption property with a Title property
                              which has same effect but has name in common with
                              other VCL dialog box components.
                            + Added button height property to control height of
                              Done button.
    )
    @REVISION(
      @VERSION              2.1
      @DATE                 01/08/1998
      @COMMENTS             Added HelpContext property to allow help to be
                            displayed when F1 is pressed if property has a value
                            > 0.
    )
    @REVISION(
      @VERSION              2.2
      @DATE                 01/08/1998
      @COMMENTS             Made appearance of glyphs on buttons depend on state
                            of system global variable MsgDlgGlyphs - if this
                            value is false it overrides ButtonGlyph property and
                            glyphs do not appear on buttons - if the value is
                            true the appearance of glyphs depends on the
                            ButtonGlyph property.
    )
    @REVISION(
      @VERSION              3.0
      @DATE                 06/12/1998
      @COMMENTS             + Made compatible with Delphi 1 & 2.
                            + Made appearance of glyphs on buttons depend on
                              state of new property AutoDetectGlyphs. If this is
                              true (default) then whether buttons are displayed
                              depends on whether 16 or 32 bit Delphi is being
                              used. With 16 bit Delphi, appearance of glyphs
                              depends on MsgDlgGlyphs typed constant (which is
                              same default behaviour as v2.2). With 32 bit
                              Delphi no glyphs will appear when AutoDetectGlyphs
                              is true, which conforms with behaviour of
                              MessageDlgXXX procedures.
    )
    @REVISION(
      @VERSION              3.0.1
      @DATE                 09/04/1999
      @COMMENTS             Changed installation palette to "PJ Stuff" from
                            "Own".
    )
    @REVISION(
      @VERSION              3.1
      @DATE                 27/11/1999
      @COMMENTS             + Made compatible with Delphi 1, 2, 3 & 4.
                            + Fixed potential bug where UseButtonGlyphProperty
                              variable in Execute method was not being set under
                              some conditions.
                            + Replaced reference to VerInfo unit with reference
                              to VInfo unit when compiled under 32 bit Delphi
                              (VerInfo name clashes with an existing unit in
                              Delphi v3 and v4).
                            + Moved declaration of TAboutDlg form class from
                              implementation to interface section to allow to
                              compile under Delphi 4.
    )
    @REVISION(
      @VERSION              3.2
      @DATE                 09/03/2002
      @COMMENTS             + Replaced reference to VInfo unit with reference to
                              PJVersionInfo when compiled under 32 bit Delphi.
                            + Changed form name to TPJAboutBoxForm.
                            + Added Notification method override to remove
                              reference to version information component if that
                              component is deleted.
                            + Added new ButtonWidth property.
                            + Made default button height and width properties
                              different when compiled under 16 and 32 bit
                              Delphi. These value reflect the default button
                              sizes used by 16 and 32 bit Delphi.
                            + Changed dialog box font size to 8 when compiled
                              under 32 bit Delphi. This font size now conforms
                              to Windows 95 standards.
                            + Changed component palette from 'PJ Stuff' to
                              'PJSoft'.
    )
    @REVISION(
      @VERSION              3.2.1
      @DATE                 10/07/2003
      @COMMENTS             Changed component palette from 'PJSoft' to
                            'DelphiDabbler'.
    )
  )
}


unit PJAbout;


interface


uses
  { Delphi }
  Forms, ExtCtrls, StdCtrls, Buttons, Classes, Controls,
  { PJSoft components }
  {$IFDEF WIN32}
    PJVersionInfo;
  {$ELSE}
    VerInfo;
  {$ENDIF}


{$R *.DFM}


type

  {
  TPJAboutBoxForm:
    Form class that defines the about box component's dialog box.
  }
  TPJAboutBoxForm = class(TForm)
    Bevel1: TBevel;
    Bevel2: TBevel;
    Bevel3: TBevel;
    ProgramLbl: TLabel;
    VersionLbl: TLabel;
    CopyrightLbl: TLabel;
    NotesLbl: TLabel;
    IconImg: TImage;
    DoneBtn: TBitBtn;
  end;

  {
  TPJAboutBtnPlacing:
    Determines horizontal placement of OK button in about dialog.
  }
  TPJAboutBtnPlacing = (abpLeft, abpCentre, abpRight);

  {
  TPJAboutBtnKinds:
    Determines caption of about dialog's OK button.
  }
  TPJAboutBtnKinds = (abkOK, abkDone, abkClose, abkCancel);

  {
  TPJAboutBtnGlyphs:
    Determines the glyph displayed on about dialog's OK button.
  }
  TPJAboutBtnGlyphs = (abgOK, abgCancel, abgIgnore, abgClose, abgNone);

  {
  TPJAboutBoxDlg:
    The about box component class.
  }
  TPJAboutBoxDlg = class(TComponent)
  private
    {Property storage}
    fTitle: string;
    fProgramName: string;
    fVersion: string;
    fCopyright: string;
    fNotes: string;
    fButtonPlacing: TPJAboutBtnPlacing;
    fButtonKind: TPJAboutBtnKinds;
    fButtonGlyph: TPJAboutBtnGlyphs;
    fAutoDetectGlyphs: Boolean;
    fButtonHeight: Integer;
    fButtonWidth: Integer;
    fCentreDlg: Boolean;
    fDlgLeft: Integer;
    fDlgTop: Integer;
    fVersionInfo: TPJVersionInfo;
    fHelpContext: THelpContext;
  protected
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
      {Check to see if any linked version info component is being deleted and
      clear reference to it if so}
  public
    constructor Create(AOwner: TComponent); override;
      {Component constructor}
    procedure Execute;
      {Execute method displays the dialog box}
  published
    property Title: string
      read fTitle write fTitle;
      {The text appearing in the dialog box title bar - default is "About"}
    property ProgramName: string
      read fProgramName write fProgramName;
      {The name of the program - appears on 1st line of about box - ignored if
      the VersionInfo property is assigned}
    property Version: string
      read fVersion write fVersion;
      {The program version - appears on 2nd line of about box - ignored if the
      VersionInfo property is assigned}
    property Copyright: string
      read fCopyright write fCopyright;
      {The program copyright message - appears on 3rd line of about box -
      ignored if the VersionInfo property is assigned}
    property Notes: string
      read fNotes write fNotes;
      {Notes about program - text wraps up to 3 lines - last entry in about box
      - ignored if the VersionInfo property is assigned}
    property ButtonPlacing: TPJAboutBtnPlacing
      read fButtonPlacing write fButtonPlacing default abpCentre;
      {Where to place "Done" button - left, centre or right of bottom of box}
    property ButtonKind: TPJAboutBtnKinds
      read fButtonKind write fButtonKind default abkOK;
      {Kind of button - determines text that appears: 'OK', 'Cancel', 'Done' or
      'Close'}
    property ButtonGlyph: TPJAboutBtnGlyphs
      read fButtonGlyph write fButtonGlyph default abgOK;
      {Glyph (if any) that appears on button. If AutoDetectGylphs property is
      true then whether this property has an effect depends on system settings}
    property AutoDetectGlyphs: Boolean
      read fAutoDetectGlyphs write fAutoDetectGlyphs default True;
      {Determine whether to display glyphs according to ButtonGlyph property
      (when this property is false) or whether to display glyphs according to
      system settings. In Delphi 1 this depends on state of MsgDlgGlyphs typed
      constant (if it's true then glyphs are displayed per ButtonGlyphs property
      and not otherwise) while in Delphi 2 glyphs aren't displayed}
    property ButtonHeight: Integer
      read fButtonHeight write fButtonHeight
      {$IFDEF WIN32}
        default 25;
      {$ELSE}
        default 33;
      {$ENDIF}
      {The height of the button}
    property ButtonWidth: Integer
      read fButtonWidth write fButtonWidth
      {$IFDEF WIN32}
        default 75;
      {$ELSE}
        default 89;
      {$ENDIF}
      {The width of the button}
    property DlgLeft: Integer
      read fDlgLeft write fDlgLeft stored True;
      {Position of left side of about box on screen - ignored if CentreDlg is
      true}
    property DlgTop: Integer
      read fDlgTop write fDlgTop stored True;
      {Position of top side of about box on screen - ignored if CentreDlg is
      true}
    property CentreDlg: Boolean
      read fCentreDlg write fCentreDlg stored True default True;
      {About box is centred on screen when this property is true, and positioned
      according to DlgLeft and DlgTop when false}
    property VersionInfo: TPJVersionInfo
      read fVersionInfo write fVersionInfo;
      {Link to a TPJVersionInfo component. If this property is not nil then the
      ProductName, ProductVersion, LegalCopyright & Comments properties of the
      control are used instead of the ProgramName, Version, Copyright and Notes
      properties of this component}
    property HelpContext: THelpContext
      read fHelpContext write fHelpContext;
      {Sets help context for dialog box - if this is non-zero pressing F1 when
      dialog box is displayed causes help topic with given context number in
      application's help file to be displayed}
  end;


procedure Register;
  {Register this component}


implementation


uses
  { Delphi }
  {$IFDEF WIN32}
    Windows;
  {$ELSE}
    WinTypes, WinProcs, Dialogs;
  {$ENDIF}


{ Component registration routine }

procedure Register;
  {Register this component}
begin
  RegisterComponents('DelphiDabbler', [TPJAboutBoxDlg]);
end;


{ TPJAboutBoxDlg }

constructor TPJAboutBoxDlg.Create(AOwner: TComponent);
  {Component constructor}
begin
  inherited Create(AOwner);
  {Set default values for properties}
  fButtonPlacing := abpCentre;      {place button at centre of box}
  fButtonKind := abkOK;             {button kind is "OK"}
  fButtonGlyph := abgOK;            {use OK glyph}
  {$IFDEF WIN32}
    fButtonHeight := 25;            {default button height - 32 bit}
    fButtonWidth := 75;             {default button width - 32 bit}
  {$ELSE}
    fButtonHeight := 33;            {default button height - 16 bit}
    fButtonWidth := 89;             {default button width - 16 bit}
  {$ENDIF}
  fTitle := 'About';                {default caption is "About"}
  fCentreDlg := True;               {ensures dialog box is centred}
  fAutoDetectGlyphs := True;        {default property value}
end;

procedure TPJAboutBoxDlg.Execute;
  {Execute method displays the dialog box}
var
  Box: TPJAboutBoxForm;              {the instance of the dialog box}
  UseButtonGlyphProperty: Boolean;   {flag true if to use ButtonGlyph property}
begin
  {Create the dialog box instance}
  Box := TPJAboutBoxForm.Create(Application);
  try
    {$IFDEF WIN32}
      Box.Font.Size := 8;
    {$ENDIF}
    {Set caption}
    Box.Caption := fTitle;
    {Decide whether to use version info or to use properties for dialog's text}
    if fVersionInfo <> nil then
    begin
      {Get displayed text from linked TPJVersionInfo component properties}
      Box.ProgramLbl.Caption := fVersionInfo.ProductName;
      Box.VersionLbl.Caption := fVersionInfo.ProductVersion;
      Box.CopyrightLbl.Caption := fVersionInfo.LegalCopyright;
      Box.NotesLbl.Caption := fVersionInfo.Comments;
    end
    else
    begin
      {No linked TPJVersionInfo: get displayed text from string properties}
      {set program name, if none specified use application title}
      if ProgramName = '' then
        Box.ProgramLbl.Caption := Application.Title
      else
        Box.ProgramLbl.Caption := ProgramName;
      {set version, copyright and notes}
      Box.VersionLbl.Caption := Version;
      Box.CopyrightLbl.Caption := Copyright;
      Box.NotesLbl.Caption := Notes;
    end;
    {Set icon to application icon}
    Box.IconImg.Picture.Graphic := Application.Icon;
    {Configure done button}
    {set button size}
    Box.DoneBtn.Height := fButtonHeight;
    Box.DoneBtn.Width := fButtonWidth;
    {place button horizontally}
    case ButtonPlacing of
      abpLeft:
        Box.DoneBtn.Left := Box.Bevel1.Left;
      abpRight:
        Box.DoneBtn.Left := Box.Bevel1.Width + Box.Bevel1.Left
          - Box.DoneBtn.Width;
      abpCentre:
        Box.DoneBtn.Left := (Box.ClientWidth - Box.DoneBtn.Width) div 2;
    end;
    {decide whether to use button glyph property, depending on value of
    AutoDetectGlyphs property and whether using 16 bit or 32 bit Delphi}
    if fAutoDetectGlyphs then
    begin
      {$IFDEF WIN32}
        UseButtonGlyphProperty := False;
      {$ELSE}
        UseButtonGlyphProperty := MsgDlgGlyphs;
      {$ENDIF}
    end
    else
      UseButtonGlyphProperty := True;
    {set button's glyph if required}
    if UseButtonGlyphProperty then
      {use ButtonGlyph property: load bitmap resource included by TBitBtn}
      case ButtonGlyph of
        abgOK:
          Box.DoneBtn.Glyph.Handle := LoadBitmap(HInstance, 'BBOK');
        abgCancel:
          Box.DoneBtn.Glyph.Handle := LoadBitmap(HInstance, 'BBCANCEL');
        abgIgnore:
          Box.DoneBtn.Glyph.Handle := LoadBitmap(HInstance, 'BBIGNORE');
        abgClose:
          Box.DoneBtn.Glyph.Handle := LoadBitmap(HInstance, 'BBCLOSE');
        abgNone:
          Box.DoneBtn.Glyph := nil;
      end
    else
      {ignore ButtonGlyphs property: don't use glyphs}
      Box.DoneBtn.Glyph := nil;
    {set correct button text according to button kind property}
    case ButtonKind of
      abkOK: Box.DoneBtn.Caption := 'OK';
      abkDone: Box.DoneBtn.Caption := 'Done';
      abkClose: Box.DoneBtn.Caption := 'Close';
      abkCancel: Box.DoneBtn.Caption := 'Cancel';
    end;
    {adjust dialog box height according to button height}
    Box.ClientHeight := 166 + fButtonHeight;
    {Position dialog on screen per positioning properties}
    if fCentreDlg then
    begin
      {centre on screen - ignore DlgLeft & DlgTop}
      Box.Left := (Screen.Width - Box.Width) div 2;
      Box.Top := (Screen.Height - Box.Height) div 2;
    end
    else
    begin
      {position per DlgLeft and DlgTop and adjust to keep on screen}
      if fDlgLeft + Box.Width > Screen.Width then
        Box.Left := Screen.Width - Box.Width
      else
        Box.Left := fDlgLeft;
      if fDlgTop + Box.Height > Screen.Height then
        Box.Top := Screen.Height - Box.Height
      else
        Box.Top := fDlgTop;
    end;
    {Set required help context for about box: displays help on pressing F1 key}
    Box.HelpContext := fHelpContext;
    {Show the about box}
    Box.ShowModal;
  finally
    {Free the instance of the about box}
    Box.Free;
  end;
end;

procedure TPJAboutBoxDlg.Notification(AComponent: TComponent;
  Operation: TOperation);
  {Check to see if any linked version info component is being deleted and clear
  reference to it if so}
begin
  if (Operation = opRemove) and (AComponent = fVersionInfo) then
    fVersionInfo := nil;
end;

end.