{ TParser - component for parsing and evaluating mathematical expressions
  Renate Schaaf (schaaf@math.usu.edu), 1993
  Alin Flaider (aflaidar@datalog.ro), 1996

  Version 9: Stefan Hoffmeister
    1996-1997 (those who have time are the lucky ones...)


  Some code cleanup, especially assigning meaningful (English) names to
  variables, functions and procedures (the original code seems to have
  been in German...)

  Fixes bug: 2^500 would not have been evaluated in V8
  Fixes bug: x^(real number) would have caused syntax error
  Fixes bug: crash if access to Value when no Expression has been parsed
  [I am not sure whether originally I introduced all these bugs myself...]

  Reduce stack usage by approx 4.2 kb
  speed up things a wee bit in _parsing_ (streamlining)
  [according to Alin Flaider the 32bit version evaluataion was sped up by 2%-3%]
  reduce pointer usage -> less memory in parsing, faster parsing
  add two new operators: "MOD" and "DIV", performing their respective PASCAL operations
  WARNING: these operators _implicitly_ perform a trunc() on their operands!

  Rework the function parsing; now it is possible to simply add new functions by overriding
  Create and saying something like

      FunctionOne.AddObject('RND', TObject(@MyProcedure));
      FunctionTwo.AddObject('MAX', TObject(@MyProcedure));

  Also two new functions have been added:


  MyProcedure must be a far procedure and declared as procedure MyProcedure(xxx: POperation)
  (that is the type TMathProcedure); as arguments use
     xxx^.arg1^ and xxx^.arg2^; store the result of the operation in xxx^.dest^

  Of course it also is possible to add functions on the fly making it possible (in theory)
  to link a couple of parsers: the first parser defines a function used by the second
  parser...

  Added error checking for variable names and function names;
  these must be valid Object Pascal identifiers and they MUST NOT contain
  the letter combination "MOD" and "DIV" - this would confuse the parser...

  Added exceptions for better flow control; made memory deallocation in case of an
  exception / internal error dead safe...

  General code cleanup (removal of dead code, "dead" variables (reassigned) etc)
}
unit Parser9;

{$IFDEF VER90}
  {$H+,S-} { long strings, no stack-checking}
{$ENDIF}

{.$DEFINE DEBUG} { by default make it lean and efficient }

{$IFNDEF DEBUG}
  {$D-} {$L-} {$Q-} {$R-} {$S-}
{$ENDIF}


{$I+} { I/O checking ON }

interface

uses
  SysUtils,
  Classes,

  P9Build;

type
  { we COULD use Forms and the TExceptionEvent therein,
    but that would give us all the VCL overhead.
    Consequentially we just redeclare an appropriate event }
  TParserExceptionEvent = procedure (Sender: TObject; E: Exception) of object;

  { functions that are added to the engine MUST have this declaration }
  { make sure that the procedure is declared far !!! }
  TFuncPrototype = procedure( AnOp: Poperation); {far;}

  TParser = class(TComponent)
  private
    FExpression : string;
    FParserError : boolean;
    FNumberOperators: integer;

    FA,FB,FC,FD,FE,
    FX,FY,FT: Float;

    FVariables: TStringList;

    FStartOperationList: POperation;

    FOnParserError : TParserExceptionEvent;

    function CalcValue: Float;
    procedure SetExpression(const AnExpression: string);
    procedure SetVar(const VarName: string; Value: Float);
  protected
    { lists of available functions, see .Create for example use }
    FunctionOne : TStringList;     { functions with ONE argument, e.g. exp() }
    FunctionTwo : TStringList;     { functions with TWO arguments, e.g. max(,) }
  public
    constructor Create(AParent: TComponent); override;
    destructor Destroy; override;

    { returns the string with the blanks inside removed }
    class function RemoveBlanks(const s: string): string;

    function ParseExpression(const AnExpression: string): boolean;
    procedure FreeExpression;

    { The PFloat returned points to the place in memory where the variable
      actually sits; to speed up assignment you can use
      PFloat is declared P9UILD.PAS }
    function SetVariable(VarName: string; Value: Float): PFloat;
    function GetVariable(const VarName: string): Float;

	   procedure AddFunctionOneParam(const AFunctionName: string; Func: TFuncPrototype);
	   procedure AddFunctionTwoParam(const AFunctionName: string; Func: TFuncPrototype);

    property ParserError: boolean read FParserError;
    property LinkedOperationList: POperation read FStartOperationList;

    property Variable[const VarName: string]: Float read GetVariable write SetVar;
  published
    { predefined variables - could be left out }
    property A: Float read FA write FA;
    property B: Float read FB write FB;
    property C: Float read FC write FC;
    property D: Float read FD write FD;
    property E: Float read FE write FE;
    property T: Float read FT write FT;
    property X: Float read FX write FX;
    property Y: Float read FY write FY;

    property Value: Float read CalcValue;

    { setting Expression automatically parses it - warning: exceptions may be raised }
    property Expression: string read FExpression write SetExpression;

    property OnParserError: TParserExceptionEvent read FOnParserError write FOnParserError;
  end;

procedure Register;

implementation

{ Make sure we get the right component icon for each version of Delphi }
{$IFDEF Win32}
   {$R PARSER9.D32}
{$ELSE}
   {$R PARSER9.D16}
{$ENDIF}


procedure Register;
begin
  RegisterComponents('Samples', [TParser]);
end;

{
****************************************************************
* These are the calculating procedures; add here               *
****************************************************************


Naming convention for functions:

  Name of built-in function, prepended with an underscore.
  Example:

    ln --> _ln

Passed arguments / results:

  If the function takes any arguments - i.e. if it has been added to
  either the FunctionOne or the FunctionTwo list:

  - First  argument --> arg1^
  - Second argument --> arg2^

  The result of the operation must ALWAYS be put into

     dest^


 Note: These are POINTERS to floats.
}




procedure _nothing(AnOp: POperation); far;
begin
end;

procedure _Add(AnOp: POperation); far;
begin
  with AnOp^ do
    dest^:= arg1^ + arg2^;
end;

procedure _Subtract(AnOp: POperation); far;
begin
  with AnOp^ do
    dest^ := arg1^ - arg2^;
end;

procedure _Multiply(AnOp: POperation); far;
begin
  with AnOp^ do
    dest^ := arg1^ * arg2^;
end;

procedure _RealDivide(AnOp: POperation); far;
begin
  with AnOp^ do
    dest^ := arg1^ / arg2^;
end;

procedure _Modulo(AnOp: POperation); far;
var
  T1, T2 : longint;
begin
  with AnOp^ do
  begin
    T1 := MaxLongint;
    T2 := MaxLongint;

    if Abs(arg1^) > MaxLongint then
      inc(T1); { force an integer overflow }
    T1 := trunc(arg1^);

    if Abs(arg2^) > MaxLongint then
      inc(T2); { force an integer overflow }
    T2 := trunc(arg2^);

    dest^ := T1 mod T2;
  end;
end;

procedure _IntDiv(AnOp: POperation); far;
var
  T1, T2 : longint;
begin
  with AnOp^ do
  begin
    T1 := MaxLongint;
    T2 := MaxLongint;

    if Abs(arg1^) > MaxLongint then
      inc(T1); { force an integer overflow }
    T1 := trunc(arg1^);

    if Abs(arg2^) > MaxLongint then
      inc(T2); { force an integer overflow }
    T2 := trunc(arg2^);

    dest^ := T1 div T2;
  end;
end;

procedure _Negate(AnOp: POperation); far;
begin
  with AnOp^ do
    dest^ := -arg1^;
end;

procedure _IntPower(AnOp: POperation); far;
var
  n,i:longint;
begin

  with AnOp^ do
  begin
    n:=trunc(abs(arg2^))-1;

    case n of
      -1: dest^:=1;
       0: dest^:=arg1^;
    else
      dest^:=arg1^;
      for i:=1 to n do
        dest^:=dest^*arg1^;
    end;

    if arg2^<0 then
      dest^:=1/dest^;

  end;
end;

procedure _square(AnOp: POperation); far;
begin
  with AnOp^ do
    dest^:= sqr(arg1^);
end;

procedure _third(AnOp: POperation); far;
begin
  with AnOp^ do
    dest^:= arg1^ * arg1^ * arg1^;
end;

procedure _forth(AnOp: POperation); far;
begin
  with AnOp^ do
    dest^ := sqr(sqr(arg1^));
end;

procedure _realpower(AnOp: POperation); far;
begin;
  with AnOp^ do
    dest^:= exp(arg2^*ln(arg1^));
end;

procedure _cos(AnOp: POperation); far;
begin
  with AnOp^ do
    dest^ := cos(arg1^);
end;

procedure _sin(AnOp: POperation); far;
begin
  with AnOp^ do
    dest^ := sin(arg1^);
end;

procedure _exp(AnOp: POperation); far;
begin
  with AnOp^ do
    dest^ := exp(arg1^);
end;

procedure _ln(AnOp: POperation); far;
begin
  with AnOp^ do
    dest^ := ln(arg1^);
end;

procedure _sqrt(AnOp: POperation); far;
begin
  with AnOp^ do
    dest^ := sqrt(arg1^);
end;

procedure _arctan(AnOp: POperation); far;
begin
  with AnOp^ do
    dest^ := arctan(arg1^);
end;

procedure _abs(AnOp: POperation); far;
begin
  with AnOp^ do
    dest^ := abs(arg1^);
end;

procedure _min(AnOp: POperation); far;
begin
  with AnOp^ do
    if arg1^ < arg2^ then
      dest^ := arg1^
    else
      dest^ := arg2^;
end;

procedure _max(AnOp: POperation); far;
begin
  with AnOp^ do
    if arg1^ < arg2^ then
      dest^ := arg2^
    else
      dest^ := arg1^;
end;

procedure _heaviside(AnOp: POperation); far;
begin
  with AnOp^ do
    if arg1^ < 0 then
      dest^ := 0
    else
      dest^ := 1;
end;

procedure _sign(AnOp: POperation); far;
begin
  with AnOp^ do
    if arg1^ < 0 then
      dest^ := -1
    else
      if arg1^ > 0 then
        dest^ := 1.0
      else
        dest^ := 0.0;
end;

procedure _zero(AnOp: POperation); far;
begin
  with AnOp^ do
    if arg1^ = 0.0 then
      dest^ := 0.0
    else
      dest^ := 1.0;
end;

procedure _phase(AnOp: POperation); far;
var
  a:Float;
begin
  with AnOp^ do
  begin
    a := arg1 ^/ (2/pi);
    dest^ := (2*pi) * (a-round(a));
  end;
end;

procedure _rnd(AnOp: POperation); far;
begin
  with AnOp^ do
    dest^ := Random * round(arg1^);
end;

procedure _arg(AnOp: POperation); far;
begin
  with AnOp^ do
    if arg1^ < 0 then
      dest^ := arctan(arg2^/arg1^)+Pi
    else
      if arg1^>0 then
        dest^ := arctan(arg2^/arg1^)
      else
        if arg2^ > 0 then
          dest^ := Pi/2
        else
          dest^:= -Pi/2;
end;

procedure _cosh(AnOp: POperation); far;
begin
  with AnOp^ do
    dest^ := (exp(arg1^)+exp(-arg1^))/2;
end;

procedure _sinh(AnOp: POperation); far;
begin
  with AnOp^ do
    dest^:= (exp(arg1^)-exp(-arg1^))/2;
end;

procedure _radius(AnOp: POperation); far;
begin
  with AnOp^ do
    dest^ := sqrt(sqr(arg1^)+sqr(arg2^));
end;


procedure _tangens(AnOp: POperation); far;
begin
  with AnOp^ do
    dest^ := sin(arg1^) / cos(arg1^);
end;


{TParser}

function TParser.ParseExpression(const AnExpression: string):boolean;
var
  OperationLoop: POperation;
begin
  if AnExpression <> '' then
  begin
    Result := false;

    FExpression := AnExpression;

    try
      ParseFunction( AnExpression,

                     FVariables,

                     FunctionOne,
                     FunctionTwo,

                     FStartOperationList,
                     FNumberOperators,
                     Result);

      FParserError := Result;

    except
      on E:Exception do
      begin
        FParserError := true;

        if Assigned(FOnParserError) then
        begin
          FOnParserError(Self, E);
          exit;
        end
        else
          raise;
      end;
    end;

    Result := not Result;

    OperationLoop := FStartOperationList;
    while OperationLoop <> nil do
    begin
      with OperationLoop^ do
      begin
        case Token of

          variab,
          constant,
          brack:         Operation :=_nothing;

          minus:         Operation :=_negate;

          sum:           Operation :=_add;
          diff:          Operation :=_subtract;
          prod:          Operation :=_multiply;
          divis:         Operation :=_RealDivide;
          modulo:        Operation :=_Modulo;
          intdiv:        Operation :=_IntDiv;

          intpower:      Operation :=_intpower;
          realpower:     Operation :=_RealPower;

          square:        Operation :=_square;
          third:         Operation :=_third;
          fourth:         Operation :=_forth;

          FuncOneVar, FuncTwoVar:    { job has been done in build already !};
        end; {case}

        OperationLoop := NextOperation;
      end; {with OperationLoop^}

    end; {while OperationLoop<>nil}
  end;
end;

constructor TParser.Create(AParent: TComponent);
begin
  inherited Create(AParent);

  FVariables := TStringList.Create;
  with FVariables do
  begin
    Duplicates := dupIgnore;
    Sorted := true;
    AddObject( 'A', TObject(@FA));
    AddObject( 'B', TObject(@FB));
    AddObject( 'C', TObject(@FC));
    AddObject( 'D', TObject(@FD));
    AddObject( 'E', TObject(@FE));
    AddObject( 'X', TObject(@FX));
    AddObject( 'Y', TObject(@FY));
    AddObject( 'T', TObject(@FT));
  end;

  FunctionOne := TStringList.Create;

  FunctionOne.AddObject('TAN', TObject(@_tangens));
  FunctionOne.AddObject('COS', TObject(@_cos));
  FunctionOne.AddObject('SIN', TObject(@_sin));
  FunctionOne.AddObject('SINH', TObject(@_sinh));
  FunctionOne.AddObject('COSH', TObject(@_cosh));
  FunctionOne.AddObject('ARCTAN', TObject(@_arctan));

  FunctionOne.AddObject('EXP', TObject(@_exp));
  FunctionOne.AddObject('LN', TObject(@_ln));

  FunctionOne.AddObject('SQRT', TObject(@_sqrt));
  FunctionOne.AddObject('ABS', TObject(@_abs));
  FunctionOne.AddObject('HEAV', TObject(@_heaviside));
  FunctionOne.AddObject('SIGN', TObject(@_sign));
  FunctionOne.AddObject('ZERO', TObject(@_zero));
  FunctionOne.AddObject('PH', TObject(@_phase));
  FunctionOne.AddObject('RND', TObject(@_rnd));

  FunctionTwo := TStringList.Create;
  FunctionTwo.AddObject('MAX', TObject(@_max));
  FunctionTwo.AddObject('MIN', TObject(@_min));
end;

destructor TParser.Destroy;
var
  i: integer;
begin
  FreeExpression;
  FStartOperationList := nil;

  with FVariables do
  begin
    for i := 0 to pred(count) do
      if not (Strings[i][1] in ['A','B','C','D','E','T','X','Y']) then
        if PFloat(Objects[i]) <> nil then
          dispose( PFloat(Objects[i]) );
    Free;
  end;

  FunctionOne.Free;
  FunctionTwo.Free;

  inherited Destroy;
end;

class function TParser.RemoveBlanks(const s: string): string;
{deletes all blanks in s}
var
  i : integer;
begin
  Result := s;

  i := pos(' ', Result);
  while i > 0 do
  begin
    delete(Result, i, 1);
    i := pos(' ' ,Result);
  end;
end;



procedure TParser.SetVar(const VarName: string; Value: Float);
begin
  SetVariable(VarName, Value);
end;

function TParser.SetVariable(VarName: string; Value: Float): PFloat;
var
  i: integer;
  v: PFloat;
begin
  { is the variable name a valid identifier? }
  if not IsValidIdent(VarName) then
    raise EBadName.Create(VarName);

  VarName := UpperCase(VarName);

  { check whether the variable contains any of the operators (DIV and MOD)
    this would confuse the parser... }
  for i := low(MathOperators) to high(MathOperators) do
  begin
    if pos(MathOperators[i], Varname) <> 0 then
      raise EBadName.Create(VarName);
  end;

  with FVariables do
    if Find(VarName, i) then
    begin
      Result := PFloat(Objects[i]);
      Result^ := Value
    end
    else
    begin
      new(v);
      v^ := Value;
      AddObject(VarName, TObject(v));
      Result := v;
    end
end;

function TParser.GetVariable(const VarName: string): Float;
var
  i: integer;
begin
  with FVariables do
    if Find(VarName, i) then
      Result := PFloat(objects[i])^
    else
      Result := 0.0;
end;

procedure TParser.AddFunctionOneParam(const AFunctionName: string; Func: TFuncPrototype);
begin
  if IsValidIdent(AFunctionName) then
    FunctionOne.AddObject(AFunctionName, TObject(@Func))
  else
    raise EBadName.Create(AFunctionName);
end;

procedure TParser.AddFunctionTwoParam(const AFunctionName: string; Func: TFuncPrototype);
begin
  if IsValidIdent(AFunctionName) then
    FunctionTwo.AddObject(AFunctionName, TObject(@Func))
  else
    raise EBadName.Create(AFunctionName);
end;

procedure TParser.FreeExpression;
var
  LastOP,
  NextOP : POperation;
begin
  LastOP := FStartOperationList;

  while LastOP <> nil do
  begin
    NextOP := LastOP^.NextOperation;

    while NextOP <> nil do
      with NextOP^ do
      begin
        if Arg1 = lastop^.Arg1 then Arg1:=nil;
        if Arg2 = lastop^.Arg1 then Arg2:=nil;
        if Dest = lastop^.Arg1 then Dest:=nil;
        if Arg1 = lastop^.Arg2 then Arg1:=nil;
        if Arg2 = lastop^.Arg2 then Arg2:=nil;
        if Dest = lastop^.Arg2 then Dest:=nil;
        if Arg1 = lastop^.Dest then Arg1:=nil;
        if Arg2 = lastop^.Dest then Arg2:=nil;
        if Dest = lastop^.Dest then Dest:=nil;

        NextOP :=NextOperation;
      end;

    with LastOP^ do
    begin
      if FVariables.IndexOfObject( TObject(Arg1)) <> -1 then Arg1:=nil;
      if FVariables.IndexOfObject( TObject(Arg2)) <> -1 then Arg2:=nil;
      if FVariables.IndexOfObject( TObject(Dest)) <> -1 then Dest:=nil;

      if (Dest<>nil) and (Dest<>Arg2) and (Dest<>Arg1) then
         dispose(Dest);

      if (Arg2<>nil) and (Arg2<>Arg1) then
         dispose(Arg2);

      if (Arg1<>nil) then
         dispose(Arg1);
    end;

    NextOP:=LastOP^.NextOperation;
    dispose(LastOP);
    LastOP:=NextOP;
  end;

end;

procedure TParser.SetExpression(const AnExpression: string);
begin
  if AnExpression <> '' then
  begin
    FreeExpression;
    FStartOperationList := nil;

    ParseExpression(AnExpression); { this implies FExpression := AnExpression }
  end;
end;


function TParser.CalcValue: Float;
var
  LastOP: POperation;
begin
  if FStartOperationList <> nil then
  begin
    LastOP := FStartOperationList;

    while LastOP^.NextOperation <> nil do
    begin
      with LastOP^ do
      begin
        Operation(LastOP);
        LastOP := NextOperation;
      end;
    end;
    LastOP^.Operation(LastOP);

    Result := LastOP^.Dest^;
  end
  else
    Result := 0;
end;

end.
