{ TParser - component for parsing and evaluating mathematical expressions
  32 bit version
  Renate Schaaf (schaaf@math.usu.edu), 1993
  Alin Flaider (aflaidar@datalog.ro), 1996
}
unit Parser8;
{$H+,S-}
interface

uses Classes, p8build;

type

TParser = class(TComponent)
 private
  fstring:string;
  pA,pB,pC,pD,pE, pX,pY,pT: double;
  fVars: TStringList;
  numop:integer;
  fop:operationpointer;
  function CalcValue: double;
  procedure SetExpression( s: string);
 public
  ParseError: boolean;
  constructor Create( AParent: TComponent); override;
  destructor Destroy; override;
  procedure ParseExpression( s: string; var error:boolean);
  procedure FreeExpression;
  procedure SetVar( VarName: string; Value: double);
  function GetVar( VarName: string): double;
 published
  property A: double read pA write pA;
  property B: double read pB write pB;
  property C: double read pC write pC;
  property D: double read pD write pD;
  property E: double read pE write pE;
  property X: double read pX write pX;
  property Y: double read pY write pY;
  property t: double read pt write pT;
  property Value: double read CalcValue;
  property Expression: string read fstring write SetExpression;
end;

procedure Register;

implementation
uses Sysutils;

procedure Register;
begin
  RegisterComponents('Samples', [TParser]);
end;

var lastop:operationpointer;

procedure mynothing;
begin
end;

procedure mysum;
begin
  lastop^.dest^:=lastop^.arg1^+lastop^.arg2^;
end;

procedure mydiff;
begin
  with lastop^ do
     dest^:=arg1^-arg2^;
end;

procedure myprod;
begin
  with lastop^ do
     dest^:=arg1^*arg2^;
end;

procedure mydivis;
begin
  with lastop^ do
     dest^:=arg1^/arg2^;
end;

procedure myminus;
begin
  with lastop^ do
     dest^:=-arg1^;
end;

procedure myintpower;
var n,i:longint;
begin
  with lastop^ do
  begin
    n:=trunc(abs(arg2^))-1;
    case n of
    -1: dest^:=1;
     0: dest^:=arg1^;
    else
    begin
      dest^:=arg1^;
      for i:=1 to n do
       dest^:=dest^*arg1^;
    end;
   end;
  if arg2^<0 then dest^:=1/dest^;
 end;
end;

procedure mysquare;
begin
  with lastop^ do
    dest^:=sqr(arg1^);
end;

procedure mythird;
begin
  with lastop^ do
    dest^:=arg1^*arg1^*arg1^;
end;

procedure myforth;
begin
  with lastop^ do
    dest^:=sqr(sqr(arg1^));
end;

procedure myrealpower;
begin;
  with lastop^ do
    dest^:=exp(arg2^*ln(arg1^));
end;

procedure mycos;
begin
  with lastop^ do
    dest^:=cos(arg1^);
end;

procedure mysin;
begin
  with lastop^ do
    dest^:=sin(arg1^);
end;

procedure myexp;
begin
  with lastop^ do
    dest^:=exp(arg1^);
end;

procedure myln;
begin
  with lastop^ do
    dest^:=ln(arg1^);
end;

procedure mysqrt;
begin
  with lastop^ do
    dest^:=sqrt(arg1^);
end;

procedure myarctan;
begin
  with lastop^ do
    dest^:=arctan(arg1^);
end;

procedure myabs;
begin
  with lastop^ do
    dest^:=abs(arg1^);
end;

procedure mymin;
begin
  with lastop^ do
    if arg1^<arg2^ then dest^:=arg1^ else dest^:=arg2^;
end;

procedure mymax;
begin
  with lastop^ do
    if arg1^<arg2^ then dest^:=arg2^ else dest^:=arg1^;
end;

procedure myheavi;
begin
  with lastop^ do
    if arg1^<0 then dest^:=0 else dest^:=1;
end;

procedure mysign;
begin
  with lastop^ do
    if arg1^<0 then dest^:= -1 else
      if arg1^>0 then dest^:=1
                 else dest^:= 0.0
end;

procedure myzero;
begin
  with lastop^ do
    if arg1^= 0.0 then dest^:= 0.0 else dest^:=1.0;
end;

procedure myphase;
var a:double;
begin
  with lastop^ do
  begin
    a:=arg1^/2/pi;
    dest^:=2*pi*(a-round(a));
  end;
end;

procedure myrand;
var j:word;
begin
  with lastop^ do
  begin
  j:=round(arg2^);
  if j=randomresult then dest^:=1 else dest^:=0;
  end;
end;

procedure myarg;
begin
  with lastop^ do
  if arg1^<0 then dest^:=arctan(arg2^/arg1^)+Pi else
  if arg1^>0 then dest^:=arctan(arg2^/arg1^) else if arg2^>0
  then dest^:=Pi/2 else dest^:=-Pi/2;
end;

procedure mycosh;
begin
  with lastop^ do
    dest^:=(exp(arg1^)+exp(-arg1^))/2;
end;

procedure mysinh;
begin
  with lastop^ do
    dest^:=(exp(arg1^)-exp(-arg1^))/2;
end;

procedure myradius;
begin
  with lastop^ do
    dest^:=sqrt(sqr(arg1^)+sqr(arg2^));
end;

procedure myrandrand;
begin
  with lastop^ do
  dest^:=arg1^+arg2^*contrandresult;
end;

{TParser}

procedure TParser.ParseExpression( s: string; var error:boolean);
var lop:operationpointer;
begin
    fstring:=s;
    parsefunction(s,fop,fVars,numop,error);
    lop:=fop;
    while lop<>nil do
    begin
      with lop^ do
      begin
        case termsorttype(opnum) of
          variab,
          constant,
          brack:    op:=mynothing;
          minus:    op:=myminus;
          sum:      op:=mysum;
          diff:     op:=mydiff;
          prod:     op:=myprod;
          divis:    op:=mydivis;
          intpower: op:=myintpower;
          realpower:op:=myrealpower;
          cosine:   op:=mycos;
          sine:     op:=mysin;
          expo:     op:=myexp;
          logar:    op:=myln;
          sqroot:   op:=mysqrt;
          arctang:  op:=myarctan;
          square:   op:=mysquare;
          third:    op:=mythird;
          forth:    op:=myforth;
          abso:     op:=myabs;
          maxim:    op:=mymax;
          minim:    op:=mymin;
          heavi:    op:=myheavi;
          phase:    op:=myphase;
          randfunc: op:=myrand;
          argu:     op:=myarg;
          hypersine:op:=mysinh;
          hypercosine:op:=mycosh;
          radius:   op:=myradius;
          randrand: op:=myrandrand;
          sign:     op:=mysign;
          zero:     op:=myzero;

        end; {case}
      end; {with lop^}
      lop:=lop^.next
    end; {while lop<>nil}

end;

constructor TParser.Create( AParent: TComponent);
begin
   inherited Create(AParent);
   fVars := TStringList.Create;
   with fVars do begin
      Duplicates := dupIgnore;
      Sorted := true;
      AddObject( 'A', @pA);
      AddObject( 'B', @pB);
      AddObject( 'C', @pC);
      AddObject( 'D', @pD);
      AddObject( 'E', @pE);
      AddObject( 'X', @pX);
      AddObject( 'Y', @pY);
      AddObject( 'T', @pT);
   end;
end;

destructor TParser.Destroy;
var i: integer;
begin
   FreeExpression;
   with fVars do begin
     for i := 0 to pred(count) do
       if not (Strings[i][1] in ['A','B','C','D','E','T','X','Y']) then
         dispose( rpointer(Objects[i]));
     Free;
   end;
   inherited Destroy
end;

procedure TParser.SetVar( VarName: string; Value: double);
var s: string[100];
    i: integer;
    v: rpointer;
begin
   if not (VarName[1] in ['A'..'Z','a'..'z']) then exit;
   s := uppercase(VarName);
   with fVars do if Find( VarName, i) then
     rpointer(Objects[i])^ := Value
   else begin
     new(v);
     v^ := Value;
     AddObject( VarName, TObject(v));
   end
end;

function TParser.GetVar( VarName: string): double;
var i: integer;
begin
   with fVars do if Find( VarName, i) then result := rpointer(objects[i])^
   else result := 0.0 
end;

procedure TParser.FreeExpression;
var lastop,nextop:operationpointer;
begin
  lastop:=fop;
  while lastop<>nil do
  begin
    nextop:=lastop^.next;
    while nextop<>nil do
    begin
          if nextop^.arg1 = lastop^.arg1 then nextop^.arg1:=nil;
          if nextop^.arg2 = lastop^.arg1 then nextop^.arg2:=nil;
          if nextop^.dest = lastop^.arg1 then nextop^.dest:=nil;
          if nextop^.arg1 = lastop^.arg2 then nextop^.arg1:=nil;
          if nextop^.arg2 = lastop^.arg2 then nextop^.arg2:=nil;
          if nextop^.dest = lastop^.arg2 then nextop^.dest:=nil;
          if nextop^.arg1 = lastop^.dest then nextop^.arg1:=nil;
          if nextop^.arg2 = lastop^.dest then nextop^.arg2:=nil;
          if nextop^.dest = lastop^.dest then nextop^.dest:=nil;
          nextop:=nextop^.next;
    end;
    with lastop^ do
    begin
      if fVars.IndexOfObject( TObject(arg1)) <> -1 then arg1:=nil;
      if fVars.IndexOfObject( TObject(arg2)) <> -1 then arg2:=nil;
      if fVars.IndexOfObject( TObject(dest)) <> -1 then dest:=nil;
      {
      if (arg1=@px) or (arg1=@py) or (arg1=@pt) or (arg1=@pa) or
      (arg1=@pb) or (arg1=@pc) or (arg1=@pd) or (arg1=@pe) then arg1:=nil;
      if (arg2=@px) or (arg2=@py) or (arg2=@pt) or (arg2=@pa) or
      (arg2=@pb) or (arg2=@pc) or (arg2=@pd) or (arg2=@pe) then arg2:=nil;
      if (dest=@px) or (dest=@py) or (dest=@pt) or (dest=@pa) or
      (dest=@pb) or (dest=@pc) or (dest=@pd) or (dest=@pe) then dest:=nil;
      }
      if arg1<>nil then dispose(arg1);
      if (arg2<>nil) and (arg2<>arg1) then dispose(arg2);
      if (dest<>nil) and (dest<>arg2) and (dest<>arg1) then dispose(dest);
    end;
    nextop:=lastop^.next;
    dispose(lastop);
    lastop:=nextop;
  end;

end;

procedure Tparser.SetExpression( s: string);
begin
   FreeExpression;
   fstring := s;
   fop := nil;
   ParseExpression( fstring, ParseError);
end;


function TParser.CalcValue: double;
begin
    lastop:=fop;
    while lastop^.next<>nil do
    begin
      lastop^.op;
      lastop:=lastop^.next;
    end;
    lastop^.op;
    Result :=lastop^.dest^;
end;

end.
