{ This unit handles all the MapIcon stuff.. }

unit OHMStuff;

interface

uses
    Graphics, Classes, Sysutils;
type
    TMapIconList = class(TList)
    protected
    FSig:Longint;
    public
    procedure Clear; virtual;
    constructor Create; virtual;
    destructor Destroy; override;
    procedure CreateMasks(trcolor:TColor); virtual;
    function  IconIndex(ID:Longint):Integer; virtual;
    property  Signature:Longint read FSig write FSig;
    procedure LoadFromFile(const filename:String); virtual;
    procedure SaveToFile(const filename:String); virtual;
    end;

    TCounter = class(TObject)
    protected
    FHighest:Longint;
    function GetNewId:Longint;
    public
    property NewID:Longint read GetNewID;
    property Highest:Longint read FHighest write FHighest;
    end;

    TBlockDefList = class(TList)
    protected
    FSig:Longint;
    public
    procedure Clear; virtual;
    constructor Create; virtual;
    destructor Destroy; override;
    property  Signature:Longint read FSig write FSig;
    procedure LoadFromFile(const filename:String); virtual;
    procedure SaveToFile(const filename:String); virtual;
    end;

    TBlockDefinition = class(TPersistent)
    protected
    FBlockData:array [0..9,0..9] of Integer;
    FXSize,FYSize:Byte;
    FName:String[30];
    procedure SetName(nom:String);
    function  GetName:string;
    public
    property  Name:string read GetName write SetName;
    property  XSize:Byte read FXSize write FXSize;
    property  YSize:Byte read FYSize write FYSize;
    function  GetIconAt(x,y:Integer):Integer;
    procedure SetIconAt(x,y,tile:Integer);
    procedure WriteData(Writer:TWriter); virtual;
    procedure ReadData (Reader:TReader); virtual;
    end;

    TMapicon = class(TPersistent)
    private
    FIconID:LongInt; {Icon's unique ID number}
    Fmask,Fbm:TBitMap;
    FName:String[30];
    FAttributes:LongInt;
    FAlternate:Longint; {Some other icon's unique ID number.}
    protected
    function GetName: string;
    procedure SetName(name:String);
    procedure ReplaceTransColor(trColor:TColor);
    public
    constructor Create; virtual;
    destructor Destroy; override;
    property Image:TBitMap read Fbm write Fbm;
    property Imagemask:TBitmap read FMask write FMask;
    property Name:String read GetName write SetName;
    property ID:LongInt read FIconId write FIconId;
    property Alternate:Longint read FAlternate write FAlternate;
    property Attributes:Longint read FAttributes write FAttributes;

    procedure CreateMask(trColor:TColor); virtual;
    procedure WriteData(Writer:TWriter); virtual;
    procedure ReadData(Reader:TReader); virtual;
    procedure ReadBitmap(const filename:String); virtual;
    end;

    TOHMap = class(TPersistent)
    private
    FXDim:Integer;
    FYDim:Integer;
    FZDim:Integer;
    FMapData:TMemoryStream;
    procedure SetXDim(xd:Integer);
    procedure SetYDim(yd:Integer);
    procedure SetZDim(zd:Integer);
    public
    constructor Create(x,y,z:Integer); virtual;
    destructor  Destroy; override;
    procedure   SetDimensions(x,y,z:Integer); virtual;
    procedure   NewMapData(nm:TMemoryStream); virtual;
    procedure   SavetoFile(const filename:String); virtual;
    procedure   LoadFromFile(const filename: String); virtual;
    function    IconAt(x,y,z:Integer):Byte; virtual;
    procedure   SetIconAt(x,y,z:Integer;icon:Byte); virtual;
    procedure   Clear; virtual;

    property XDim:Integer read FXDim write SetXdim;
    property YDim:Integer read FYDim write SetYdim;
    property ZDim:Integer read FZDim write SetZdim;
    end;

implementation

procedure TBlockDefinition.SetName(nom:String);
begin
     if nom<>FName then
        Fname:=nom;
end;

function TBlockDefinition.GetName:string;
begin
     result:=FName;
end;

function TBlockDefinition.GetIconAt(x,y:Integer):Integer;
begin
     if (x<10) and (y<10) then
        result:=FBlockData[x,y]
     else
         result:=0;
end;

procedure TBlockDefinition.SetIconAt(x,y,tile:Integer);
begin
     if (x<10) and (y<10) then
        FBlockData[x,y]:=tile;
end;

procedure TBlockDefinition.WriteData(Writer:TWriter);
var
x,y:Integer;
begin
     with writer do begin
          WriteListBegin;
          WriteString(FName);
          WriteInteger(FXSize);
          WriteInteger(FYSize);
          for x:=0 to 9 do
              for y:=0 to 9 do
                  WriteInteger(GetIconAt(x,y));
          WriteListEnd;
     end;
end;

procedure TBlockDefinition.ReadData(Reader:TReader);
var
x,y:Integer;
begin
     with reader do begin
          ReadListBegin;
          FName:=ReadString;
          FXsize:=ReadInteger;
          FYSize:=ReadInteger;
          for x:=0 to 9 do
              for y:=0 to 9 do
                  SetIconAt(x,y,ReadInteger);
          ReadListEnd;
     end;
end;


constructor TBlockDefList.create;
begin
     inherited Create;
     FSig:=$b1b1;
end;

destructor TBlockDefList.destroy;
begin
     clear;
     inherited Destroy;
end;

procedure TBlockDefList.clear;
var
a:Integer;
begin
     {This routine deallocates all resources inside this here list}
     if count>0 then
        for a:=0 to count-1 do
            TObject(items[a]).free;
     inherited Clear;
end;

procedure TBlockDefList.loadfromfile(const filename:string);
var
   i:Integer;
   Reader:Treader;
   Stream:TFileStream;
   obj:TBlockDefinition;
begin
     stream:=TFileStream.create(filename,fmOpenRead);
     try
        reader:=TReader.create(stream,$ff);
        try
           with reader do begin
                try
                   ReadSignature;
                   if ReadInteger<>FSig then
                      Raise EReadError.Create('Incorrect file signature. File is not of correct type.');
                except
                      Raise EReadError.Create('Not a valid file.');
                end;
                ReadListBegin;
                while not EndofList do begin
                    obj:=TBlockDefinition.create;
                    try
                       obj.ReadData(reader);
                    except
                          obj.free;
                          raise;
                    end;
                    self.add(obj);
                end;
                ReadListEnd;
           end;
        finally
               reader.free;
        end;
     finally
            stream.free;
     end;
end;

procedure TBlockDefList.savetofile(const filename:String);
var
   Stream:TFileStream;
   Writer:TWriter;
   i:Integer;
begin
     stream:=TFileStream.create(filename,fmCreate or fmOpenWrite);
     try
        writer:=TWriter.create(stream,$ff);
        try
           with writer do begin
                try
                   WriteSignature;
                   WriteInteger(FSig);
                   WriteListBegin;
                   for i:=0 to self.count-1 do
                       TBlockDefinition(self.items[i]).writedata(writer);
                   WriteListEnd;
                except
                      raise;
                end;
           end;
        finally
               writer.free;
        end;
     finally
            stream.free;
     end;
end;


function TCounter.GetNewId:Longint;
begin
     FHighest:=FHighest+1;
     result:=FHighest;
end;

destructor TMapIcon.Destroy;
begin
     Fbm.free;
     if Assigned(Fmask) then
        FMask.free;
     inherited Destroy;
end;

constructor TMapIcon.Create;
begin
     inherited Create;
     Fbm:=Tbitmap.create;
     FAlternate:=-1;
end;

procedure TMapIcon.CreateMask(trColor:TColor);
var
testbm1,testbm2: TBitmap;
trColorInv: TColor;
begin
     if Assigned(Fmask) then FMask.Free;

     FMask:=TBitmap.create;
     FMask.width:=Fbm.width;
     FMask.height:=Fbm.height;

     testbm1 := TBitmap.Create;
     testbm1.width := 1;
     testbm1.height:=1;
     testbm2 := TBitmap.Create;
     testbm2.width := 1;
     testbm2.height:=1;
     testbm1.Canvas.Pixels[0,0]:=trColor;
     testbm2.Canvas.CopyMode:=cmSrcInvert;
     testbm2.Canvas.Draw(0,0,testbm1);
     trColorInv:=testbm2.Canvas.Pixels[0,0];
     testbm1.free;
     testbm2.free;
     with FMask.Canvas do begin
          Brush.Color:= trColorInv;
          BrushCopy( Rect(0,0,FMask.Width,FMask.Height),Fbm,
          Rect(0,0,Fbm.Width,Fbm.Height),trColor);
          CopyMode:=cmSrcInvert;
          Draw(0,0,Fbm);
     end;
     ReplaceTransColor(trColor);
end;

procedure TMapIcon.ReplaceTransColor(trColor: TColor);
begin
     with Fbm.Canvas do begin
          CopyMode:=cmSrcCopy;
          Brush.Color:= clBlack;
          BrushCopy( Rect(0,0,Fbm.Width,Fbm.Height),Fbm,
               Rect(0,0,Fbm.Width,Fbm.Height),trColor);
     end;
end;

procedure TMapIcon.WriteData(Writer:TWriter);
var
   mstream:TMemoryStream;
   size:Longint;
begin
     mstream:=TMemoryStream.create;
     try
        fbm.savetostream(mstream);
        size:=mstream.size;
        with writer do begin
          WriteListBegin;
          WriteString(FName);
          WriteInteger(FAttributes);
          WriteInteger(FAlternate);
          WriteInteger(FIconID);
          WriteInteger(size);
          Write(mstream.Memory^, size);
          WriteListEnd;
        end;
     finally
            mstream.free;
     end;

end;

procedure TMapIcon.ReadData(Reader:TReader);
var
   mstream:TMemoryStream;
   size:Longint;
begin
     mstream:=TMemoryStream.create;
     try
     with reader do begin
          ReadListBegin;
          Fname         :=ReadString;
          FAttributes   :=ReadInteger;
          FAlternate    :=ReadInteger;
          FIconId       :=ReadInteger;
          size          :=ReadInteger;
          mstream.setsize(size);
          Read(mstream.Memory^, size);
          fbm.loadfromstream(mstream);
          ReadListEnd;
     end;
     finally
            mstream.free;
     end;
end;

procedure TMapIcon.ReadBitmap(const filename:String);
begin
     Fbm.Free;
     fbm:=TBitmap.Create;
     Fbm.loadfromfile(filename);
end;

function TMapIcon.GetName:String;
begin
     result:=Fname;
end;

procedure TMapIcon.SetName(name:String);
begin
     Fname:=name;
end;

constructor TMapIconList.Create;
begin
     inherited Create;
     FSig:=$f00d;
end;

procedure TMapIconList.CreateMasks(trColor:TColor);
var
a:Integer;
begin
     for a:=0 to count-1 do
         TMapIcon(items[a]).CreateMask(trColor);
end;
{
 This function returns the index number of a specified icon by
 it's ID number.
}
function TMapIconList.IconIndex(ID:Longint):Integer;
var
a:Integer;
begin
     for a:=0 to count-1 do begin
         if TMapIcon(items[a]).id=id then begin
            result:=a;
            exit;
         end;
     end;
     result:=-1;
end;

destructor TMapIconList.destroy;
begin
     clear;
     inherited Destroy;
end;

procedure TMapIconList.clear;
var
a:Integer;
begin
     {This routine deallocates all resources inside this here list}
     if count>0 then
        for a:=0 to count-1 do
            TObject(items[a]).free;
     inherited Clear;
end;

procedure TMapIconList.loadfromfile(const filename:string);
var
   i:Integer;
   Reader:Treader;
   Stream:TFileStream;
   obj:TMapIcon;
begin
     stream:=TFileStream.create(filename,fmOpenRead);
     try
        reader:=TReader.create(stream,$ff);
        try
           with reader do begin
                try
                   ReadSignature;
                   if ReadInteger<>FSig then
                      Raise EReadError.Create('Incorrect file signature. File is not of correct type.');
                except
                      Raise EReadError.Create('Not a valid file.');
                end;
                ReadListBegin;
                while not EndofList do begin
                    obj:=TMapIcon.create;
                    try
                       TMapIcon(obj).ReadData(reader);
                    except
                          obj.free;
                          raise;
                    end;
                    self.add(obj);
                end;
                ReadListEnd;
           end;
        finally
               reader.free;
        end;
     finally
            stream.free;
     end;
end;

procedure TMapIconList.savetofile(const filename:String);
var
   Stream:TFileStream;
   Writer:TWriter;
   i:Integer;
begin
     stream:=TFileStream.create(filename,fmCreate or fmOpenWrite);
     try
        writer:=TWriter.create(stream,$ff);
        try
           with writer do begin
                try
                   WriteSignature;
                   WriteInteger(FSig);
                   WriteListBegin;
                   for i:=0 to self.count-1 do
                       TMapIcon(self.items[i]).writedata(writer);
                   WriteListEnd;
                except
                      raise;
                end;
           end;
        finally
               writer.free;
        end;
     finally
            stream.free;
     end;
end;

constructor TOHmap.Create(x,y,z:Integer);
begin
     try
        inherited Create;
        FXDim:=x;
        FYDim:=y;
        FZDim:=z;
        FMapData:=TMemoryStream.create;
        FMapData.SetSize(x*y*z);
     except
           raise EOutOfMemory.create('Not enough memory to allocate map.');
     end;
end;

destructor TOHMap.Destroy;
begin
     FMapData.free;
     inherited Destroy;
end;

procedure TOHMap.Clear;
var
x,y,z:Integer;
begin
     for x:=0 to xdim-1 do
         for y:=0 to ydim-1 do
             for z:=0 to zdim-1 do
                 SetIconAt(x,y,z,0);
end;

procedure TOHMap.SetXDim(xd:Integer);
begin
     FXdim:=xd;
     FMapData.Setsize(Fxdim*Fydim*FZdim);
end;

procedure TOHMap.SetYDim(yd:Integer);
begin
     FYDim:=yd;
     FMapData.Setsize(Fxdim*Fydim*FZdim);
end;

procedure TOHMap.SetZDim(zd:Integer);
begin
     FZDim:=zd;
     FMapData.Setsize(Fxdim*Fydim*FZdim);
end;

procedure TOHMap.SetDimensions(x,y,z:Integer);
begin
     FXdim:=x; FYDim:=y ; FZDim:=z;
     FMapData.SetSize(FXdim*FYDim*FZDim);
end;

procedure TOHMap.Savetofile(const filename:String);
var
stream:TFileStream;
writer:TWriter;
begin
     try
        stream:=TFileStream.Create(filename,fmCreate or fmOpenWrite);
        try
           writer:=TWriter.create(stream,$ff);
           try
              with writer do begin
                WriteSignature;
                WriteListBegin;
                WriteInteger(FXdim*FYdim*FZdim);
                Write(FMapData.memory^,FXdim*FYDim*FZdim);
                WriteInteger(FXDim);
                WriteInteger(FYDim);
                WriteInteger(FZDim);
                WriteListEnd;
              end;
           finally
                  writer.free;
           end;
        finally
            stream.free;
        end;
     except
           Raise EFCreateError.Create('Unable to save map file.');
     end;
end;

procedure TOHMap.Loadfromfile(const filename:String);
var
reader:TReader;
stream:TFileStream;
mapsize:Longint;
begin
     try
        stream:=TFileStream.create(filename,fmOpenRead);
        try
           reader:=TReader.create(stream,$ff);
           try
              with reader do begin
                ReadSignature;
                ReadListBegin;
                mapsize:=ReadInteger;
                FMapdata.setsize(mapsize);
                Read(FMapdata.memory^,mapsize);
                FXDim:=ReadInteger;
                FYDim:=ReadInteger;
                FZDim:=ReadInteger;
                ReadListEnd;
              end;
           finally
               reader.free;
           end;
        finally
               stream.free;
        end;
     except
           Raise EFOpenError.Create('Unable to load map file.');
     end;
end;

procedure TOHMap.NewMapData(nm:TMemoryStream);
begin
     if nm<>nil then begin
        FMapData:=nm;
     end;
end;

function TOHMap.IconAt(x,y,z:Integer):Byte;
var
ct:Integer;
begin
     FMapData.Position:=x+(y*xdim)+(z*(xdim*ydim));
     ct:=FMapData.Read(result,1);
end;

procedure TOHMap.SetIconAt(x,y,z:Integer;icon:Byte);
var
ct:Integer;
begin
     FMapData.Position:=x+(y*xdim)+(z*(xdim*ydim));
     ct:=FMapData.Write(icon,1);
end;

end.


