{$A+,B-,D-,F-,G+,I-,K+,L-,N+,P+,Q-,R-,S-,T-,V-,W-,X+,Y-}
{ $IFDEF DEBUG_ALL}
  {$D+,L+,Y+}
{ $ENDIF}
UNIT Objects7;

{ Objects7 unit V1.0 Copyright by Fink Guy

  Purpose       : Implement the Objects-Unit from BP 7.0 under Delphi

  This unit is freeware.

  USE AT YOUR OWN RISK! No garantees expressed or implied.

  The autor is not responsible for any damages resulting from the use
  of the routines in this unit.

  You are free to distribute this unit in an unmodified form and without any
  charges other than those necessary for the needed media and the copy costs.
  Please do not distribute modified versions.

  THIS UNIT IS NOT PUBLIC DOMAIN!!! }

INTERFACE

{$R OBJECTS7.RES}

USES
  SysUtils,
  ExtObjects,
  Classes;

CONST
  stReg_InvalidRegCode          = -1;
  stReg_NILClass                = -2;
  stReg_InvalidClass            = -3;
  stReg_CodeAlreadyUsed         = -4;

  stRead_RegBaseEmpty           = -5;
  stRead_UnregisteredObject     = -6;
  stRead_InvalidObject          = -7;

  stWrite_RegBaseEmpty          = -8;
  stWrite_UnregisteredObject    = -9;
  stWrite_InvalidObject         = -10;
  stWrite_InvalidParentCode     = -11;

  stCasting_NotExpectedType     = -12;

  stLastError                   = -12;

  co_NILItem                    = -1;
  co_StackUnderflow             = -2;
  co_StackOverflow              = -3;
  co_QueueEmpty                 = -2;
  co_QueueOverflow              = -3;

  coLastError                   = -3;

TYPE
  EObjectStream = CLASS(EStreamError)
  PRIVATE
    FErrorCode  : INTEGER;
  PUBLIC
    CONSTRUCTOR CreateRes(Ident: Integer);
    CONSTRUCTOR CreateResFmt(Ident: Integer; CONST Args: ARRAY OF CONST);
    PROPERTY ErrorCode  : INTEGER READ FErrorCode;
  END;

  ECollectionError      = CLASS(EListError)
  PRIVATE
    FErrorCode  : INTEGER;
  PUBLIC
    CONSTRUCTOR CreateRes(Ident: Integer);
    CONSTRUCTOR CreateResFmt(Ident: Integer; CONST Args: ARRAY OF CONST);
    PROPERTY ErrorCode  : INTEGER READ FErrorCode;
  END;

  TObjectStream = CLASS;

  TStreamableObject = CLASS(TExtendedObject)
  PUBLIC
    CLASS FUNCTION GetRegistrationCode : CARDINAL; VIRTUAL; ABSTRACT;
    CONSTRUCTOR Load(S : TObjectStream; RCode : CARDINAL); VIRTUAL; ABSTRACT;
    PROCEDURE Store(S : TObjectStream); VIRTUAL; ABSTRACT;
  END;

  TStreamClass = CLASS OF TStreamableObject;

{ TObjectStream }

  TObjectStream = CLASS(TExtendedObject)
  PRIVATE
    FStream     : TStream;
    FCloseFile  : BOOLEAN;
  PROTECTED
    FUNCTION GetPosition : LONGINT;
    PROCEDURE SetPosition(Pos : LONGINT);
    FUNCTION GetSize : LONGINT;
    FUNCTION GetHandle : INTEGER;
    FUNCTION GetMemory : POINTER;
  PUBLIC
    CONSTRUCTOR Create;
    CONSTRUCTOR InitHandleStream(AHandle : INTEGER);
    CONSTRUCTOR InitFileStream(CONST FileName : STRING; Mode : Word);

    PROCEDURE ClearData; OVERRIDE;
    FUNCTION DupData2(Destination       : TExtendedObject) : BOOLEAN; OVERRIDE;
    FUNCTION DuplicateStream(Destination: TExtendedObject) : BOOLEAN; VIRTUAL;

    FUNCTION Get : TStreamableObject;
    PROCEDURE Put(P : TStreamableObject);
    FUNCTION ReadParentCode(P : TStreamClass) : CARDINAL;
    PROCEDURE WriteParentCode(P : TStreamClass);
    PROCEDURE Flush; VIRTUAL;
    PROCEDURE Truncate; VIRTUAL;

    FUNCTION ReadStr : PSTRING;
    FUNCTION StrRead : PCHAR;

    PROCEDURE WriteStr(P : PSTRING);
    PROCEDURE StrWrite(P : PCHAR);

    FUNCTION Read(VAR Buffer; Count : LONGINT): LONGINT;
    FUNCTION Write(CONST Buffer; Count : LONGINT): LONGINT;
    FUNCTION Seek(Offset : LONGINT; Origin : Word): LONGINT;
    FUNCTION CopyFrom(Source : TStream; Count : LONGINT): LONGINT;
    PROPERTY Position : LONGINT READ GetPosition WRITE SetPosition;
    PROPERTY Size : LONGINT READ GetSize;
    PROPERTY Stream :TStream READ FStream;

    PROPERTY Handle : INTEGER READ GetHandle;
    PROPERTY CloseFile : BOOLEAN READ FCloseFile WRITE FCloseFile;

    PROCEDURE LoadFromStream(Stream : TStream);
    PROCEDURE LoadFromFile(CONST FileName : STRING);
    PROCEDURE SaveToStream(Stream : TStream);
    PROCEDURE SaveToFile(CONST FileName : STRING);
    PROCEDURE SetSize(Size : LONGINT);
    PROCEDURE Clear;
    PROPERTY Memory : POINTER READ GetMemory;
  END;

{ TCollection object }

  TCollection = CLASS(TStreamableObject)
  PRIVATE
    FList       : TList;
  PROTECTED
    PROCEDURE CheckList;
    FUNCTION Get(Index: INTEGER): POINTER;
    PROCEDURE Put(Index: INTEGER; Item: POINTER);
  PUBLIC
    CONSTRUCTOR Create(ALimit : INTEGER);
    PROCEDURE ClearData; OVERRIDE;
    FUNCTION DupData2(Destination       : TExtendedObject) : BOOLEAN; OVERRIDE;

    CLASS FUNCTION GetRegistrationCode : CARDINAL; OVERRIDE;
    CONSTRUCTOR Load(S : TObjectStream; RCode : CARDINAL); OVERRIDE;
    PROCEDURE Store(S : TObjectStream); OVERRIDE;
    FUNCTION GetItem(S : TObjectStream): POINTER; VIRTUAL;
    PROCEDURE PutItem(S : TObjectStream; Item: POINTER); VIRTUAL;

    FUNCTION DuplicateItem(Item : POINTER) : POINTER; VIRTUAL;

    FUNCTION InsertItem(Item : POINTER) : POINTER; VIRTUAL;
    PROCEDURE Insert(Item : POINTER); VIRTUAL;
    PROCEDURE AtInsert(Index : INTEGER; Item: POINTER);
    PROCEDURE AtPut(Index : INTEGER; Item: POINTER);
    PROCEDURE AtReplace(Index: INTEGER; Item: POINTER);

    FUNCTION At(Index : INTEGER): POINTER;
    FUNCTION IndexOf(Item : POINTER): INTEGER; VIRTUAL;

    PROCEDURE AtDelete(Index : INTEGER);
    PROCEDURE Delete(Item : POINTER);
    PROCEDURE DeleteAll;

    PROCEDURE FreeItem(Item : POINTER); VIRTUAL;
    PROCEDURE AtFree(Index : INTEGER);
    PROCEDURE Remove(Item : POINTER);
    PROCEDURE FreeAll;

    FUNCTION FirstThat(Test : POINTER): POINTER;
    PROCEDURE ForEach(Action : POINTER);
    FUNCTION LastThat(Test : POINTER): POINTER;

    PROCEDURE Pack;
    PROCEDURE SetLimit(ALimit : INTEGER);

    PROPERTY List  : TList READ FList;
    PROPERTY Items[INDEX : INTEGER]: POINTER READ Get WRITE Put; DEFAULT;
{ Use TCollection as a Stack e.g. LIFO }
    PROCEDURE Push(Item : POINTER);
    FUNCTION Pop : POINTER;
    FUNCTION PeekTOS : POINTER;                 { TOS = Top of Stack }
    PROCEDURE ReplaceTOS(Item : POINTER);
    PROCEDURE SwapTOS;
    PROCEDURE PopFree;
{ Use TCollection as a Queue e.g. FIFO }
    PROCEDURE Write(Item : POINTER);
    FUNCTION Read : POINTER;
    FUNCTION Peek : POINTER;
    PROCEDURE Drop;
  END;

{ TSortedCollection object }

  TSortedCollection = CLASS(TCollection)
  PRIVATE
    FDuplicates,
    FNotSorted  : BOOLEAN;
  PROTECTED
    PROCEDURE AllowDuplicates(D : BOOLEAN);
    FUNCTION IsSorted : BOOLEAN;
    PROCEDURE SetSorted(S : BOOLEAN);
    PROCEDURE Sort;
  PUBLIC
    CLASS FUNCTION GetRegistrationCode : CARDINAL; OVERRIDE;
    CONSTRUCTOR Load(S : TObjectStream; RCode : CARDINAL); OVERRIDE;
    PROCEDURE Store(S : TObjectStream); OVERRIDE;

    FUNCTION IndexOf(Item : POINTER): INTEGER; OVERRIDE;
    FUNCTION FirstIndexOf(Item : POINTER): INTEGER;
    FUNCTION LastIndexOf(Item : POINTER): INTEGER;
    PROCEDURE Insert(Item : POINTER); OVERRIDE;
    FUNCTION Compare(Key1, Key2 : POINTER): INTEGER; VIRTUAL; ABSTRACT;
    FUNCTION KeyOf(Item : POINTER) : POINTER; VIRTUAL;
    FUNCTION Search(Key : POINTER; VAR Index: INTEGER) : BOOLEAN; VIRTUAL;
    PROPERTY Duplicates : BOOLEAN READ FDuplicates WRITE AllowDuplicates;
    PROPERTY Sorted : BOOLEAN READ IsSorted WRITE SetSorted;
  END;

CONST
{ Change these values if you add, remove or change the order of virtual methods in
  TSortedCollection or one of its anchestors !!! Count all virtual methods from
  TExtendedObject to TSortedCollection without the overridden methods.

  I've done this because the internal assembler is not able to produce the offset
  into the VMT for a virtual method, like the compiler does. The internal assembler
  ALWAYS generates static adresses, even for a virtual method !!! THANK YOU BORLAND.

  This is very important for the Search method, or your program will crash !!! }

  CompareIndex  = 13;
  KeyOfIndex    = 14;
  CompareOffset = (CompareIndex-1)*4;
  KeyOfOffset   = (KeyOfIndex-1)*4;

TYPE

{ TStringCollection object }

  TStringCollection = CLASS(TSortedCollection)
  PRIVATE
    FNotCaseSensitive,
    FAnsiCharSet        : BOOLEAN;
  PROTECTED
    FUNCTION IsCaseSensitive : BOOLEAN;
    PROCEDURE SetCaseSensitive(CS : BOOLEAN);
    PROCEDURE SetAnsiCharSet(ACS : BOOLEAN);
  PUBLIC
    CLASS FUNCTION GetRegistrationCode : CARDINAL; OVERRIDE;
    CONSTRUCTOR Load(S : TObjectStream; RCode : CARDINAL); OVERRIDE;
    PROCEDURE Store(S : TObjectStream); OVERRIDE;

    FUNCTION DuplicateItem(Item : POINTER) : POINTER; OVERRIDE;

    FUNCTION Compare(Key1, Key2 : POINTER) : INTEGER; OVERRIDE;
    PROCEDURE FreeItem(Item : POINTER); OVERRIDE;
    FUNCTION GetItem(S : TObjectStream): POINTER; OVERRIDE;
    PROCEDURE PutItem(S : TObjectStream; Item: POINTER); OVERRIDE;
    PROPERTY CaseSensitive : BOOLEAN READ IsCaseSensitive WRITE SetCaseSensitive;
    PROPERTY AnsiCharSet : BOOLEAN READ FAnsiCharSet WRITE SetAnsiCharSet;
  END;

{ TStrCollection object }

  TStrCollection = CLASS(TStringCollection)
  PUBLIC
    CLASS FUNCTION GetRegistrationCode : CARDINAL; OVERRIDE;
    CONSTRUCTOR Load(S : TObjectStream; RCode : CARDINAL); OVERRIDE;
    PROCEDURE Store(S : TObjectStream); OVERRIDE;

    FUNCTION DuplicateItem(Item : POINTER) : POINTER; OVERRIDE;

    FUNCTION Compare(Key1, Key2 : POINTER) : INTEGER; OVERRIDE;
    PROCEDURE FreeItem(Item : POINTER); OVERRIDE;
    FUNCTION GetItem(S : TObjectStream) : POINTER; OVERRIDE;
    PROCEDURE PutItem(S : TObjectStream; Item : POINTER); OVERRIDE;
  END;


PROCEDURE RegisterNewObject(    CType   : TStreamClass);
PROCEDURE RegisterObject(       CType   : TStreamClass;
                                RCode   : CARDINAL);

PROCEDURE RegisterObjects;

IMPLEMENTATION

USES
  WinProcs,
  Consts,
  SysTools;

{ TStream type registration routines }

{$L OBJECTS7.OBJ}

TYPE
  TStreamRec = RECORD
    ObjType     : CARDINAL;
    VmtLink     : TStreamClass;
  END;
  TStreamRecArray       = ARRAY[1..1] OF TStreamRec;
  PStreamRecArray       = ^TStreamRecArray;

CONST
  RecordIncrement       = 8;
  ArrayIncrement        = RecordIncrement * SIZEOF(TStreamRec);
  NrOfStRecs            : WORD = 0;
  MaxNrOfStRecs         : WORD = 0;
  StreamRecords         : PStreamRecArray = NIL;

TYPE
  StringCompareFunc = FUNCTION(CONST S1,S2 : STRING) : INTEGER;
  StrCompareFunc    = FUNCTION(S1,S2 : PCHAR) : INTEGER;

CONST
  StringCompare : ARRAY[FALSE..TRUE,FALSE..TRUE] OF StringCompareFunc =
   ((CompareStr, AnsiCompareStr),(CompareText, AnsiCompareText));
  StrCompare : ARRAY[FALSE..TRUE,FALSE..TRUE] OF StrCompareFunc =
   ((StrComp, lStrCmp),(StrIComp, lStrCmpi));

CONST
  stFirstErrorMsg       = 60000;
  coFirstErrorMsg       = 60032;
  coStackUnderflow      = 60048;
  coStackOverflow       = 60049;
  coQueueEmpty          = 60050;
  coQueueOverflow       = 60051;

CONSTRUCTOR EObjectStream.CreateRes(Ident: Integer);

BEGIN
  FErrorCode := Ident;
  IF (Ident < 0) AND (Ident >= stLastError) THEN
    INHERITED CreateRes(stFirstErrorMsg + ABS(Ident))
  ELSE
    INHERITED CreateResFmt(stFirstErrorMsg,[Ident]);
END;

CONSTRUCTOR EObjectStream.CreateResFmt(Ident: Integer; CONST Args: ARRAY OF CONST);

BEGIN
  FErrorCode := Ident;
  IF (Ident < 0) AND (Ident >= stLastError) THEN
    INHERITED CreateResFmt(stFirstErrorMsg + ABS(Ident), Args)
  ELSE
    INHERITED CreateResFmt(stFirstErrorMsg,[Ident]);
END;

CONSTRUCTOR ECollectionError.CreateRes(Ident: Integer);

BEGIN
  FErrorCode := Ident;
  IF (Ident < 0) AND (Ident >= coLastError) THEN
    INHERITED CreateRes(coFirstErrorMsg + ABS(Ident))
  ELSE
    INHERITED CreateResFmt(coFirstErrorMsg,[Ident]);
END;

CONSTRUCTOR ECollectionError.CreateResFmt(Ident: Integer; CONST Args: ARRAY OF CONST);

BEGIN
  FErrorCode := Ident;
  IF (Ident < 0) AND (Ident >= coLastError) THEN
    INHERITED CreateResFmt(coFirstErrorMsg + ABS(Ident), Args)
  ELSE
    INHERITED CreateResFmt(coFirstErrorMsg,[Ident]);
END;

PROCEDURE RegisterNewObject(    CType   : TStreamClass);

BEGIN
  RegisterObject(CType, CType.GetRegistrationCode);
END;

PROCEDURE RegisterObject(       CType   : TStreamClass;
                                RCode   : CARDINAL);

VAR
  CurrentSize : WORD;
  i     : CARDINAL;

BEGIN
  IF    (CType  = NIL) THEN
    RAISE EObjectStream.CreateResFmt(stReg_NILClass,[RCode]);
  IF    (RCode  = 0) THEN
    RAISE EObjectStream.CreateResFmt(stReg_InvalidRegCode,[CType.ClassName]);
  IF NOT(CType.InheritsFrom(TStreamableObject)) THEN
    RAISE EObjectStream.CreateResFmt(stReg_InvalidClass,[CType.ClassName,RCode]);
  IF NrOfStRecs > 0 THEN
  BEGIN
    i   := 1;
    REPEAT
      WITH StreamRecords^[i] DO
      BEGIN
        IF (ObjType = RCode) THEN
        BEGIN
           IF (VmtLink = CType) THEN
             Exit       { Object is registered correctly }
           ELSE
             RAISE EObjectStream.CreateResFmt(stReg_CodeAlreadyUsed,[CType.ClassName,VmtLink.ClassName,RCode]);
        END;
      END;
      INC(I);
    UNTIL i > NrOfStRecs;
  END
  ELSE
  BEGIN
    StreamRecords       := AllocMem(ArrayIncrement);
    MaxNrOfStRecs       := RecordIncrement;
  END;
  IF MaxNrOfStRecs = NrOfStRecs THEN
  BEGIN
    CurrentSize         := MaxNrOfStRecs * SIZEOF(TStreamRec);
    StreamRecords       := ReAllocMem(  StreamRecords,
                                        CurrentSize,
                                        CurrentSize + ArrayIncrement);
    INC(MaxNrOfStRecs,RecordIncrement);
  END;
  INC(NrOfStRecs);
  WITH StreamRecords^[NrOfStRecs] DO
  BEGIN
    ObjType     := RCode;
    VmtLink     := CType;
  END;
END;

CONSTRUCTOR TObjectStream.Create;

BEGIN
  FStream       := TMemoryStream.Create;
END;

CONSTRUCTOR TObjectStream.InitHandleStream(AHandle: INTEGER);

BEGIN
  FStream       := THandleStream.Create(AHandle);
END;

CONSTRUCTOR TObjectStream.InitFileStream(CONST FileName: STRING; Mode: Word);

BEGIN
  FStream       := TFileStream.Create(FileName,Mode);
END;

PROCEDURE TObjectStream.ClearData;

BEGIN
  IF FCloseFile AND FStream.InheritsFrom(THandleStream) THEN
    FileClose(THandleStream(FStream).Handle);
  FStream.Free;
  INHERITED ClearData;
END;

FUNCTION TObjectStream.DupData2(Destination     : TExtendedObject) : BOOLEAN;

BEGIN
  Result        := INHERITED DupData2(Destination);
  IF Result AND (FStream <> NIL) THEN
    Result      := DuplicateStream(Destination);
END;

FUNCTION TObjectStream.DuplicateStream(Destination: TExtendedObject) : BOOLEAN;

VAR
  OldStream     : TStream;
  Hnd           : INTEGER;
  Pos           : LONGINT;

BEGIN
  Result        := TRUE;
  WITH TObjectStream(Destination) DO
  BEGIN
    OldStream   := FStream;
    FStream     := NIL;
    WITH OldStream DO
    BEGIN
      IF InheritsFrom(THandleStream) THEN
      BEGIN
        Hnd     := Handle;
        ASM
                MOV     AH, $45                 {Duplicate file handle}
                MOV     BX, Hnd
                CALL    DOS3Call
                JNC     @@1
                MOV     InOutRes, AX
                XOR     AX, AX
@@1:            MOV     Hnd, AX
        END;
        IF Hnd = 0 THEN
          Result := FALSE
        ELSE
        BEGIN
          FStream       := THandleStream.Create(Hnd);
          FCloseFile    := TRUE;                { This is important if FStream is a THandlestream }
        END;                                    { or the file will not be closed !}
      END
      ELSE IF InheritsFrom(TMemoryStream) THEN
      BEGIN
        Pos             := Position;
        FStream         := TMemoryStream.Create;
        TMemoryStream(FStream).LoadFromStream(OldStream);
        Position        := Pos;
        FStream.Position:= Pos;
      END;
    END;
  END;
END;

FUNCTION TObjectStream.ReadParentCode(P : TStreamClass) : CARDINAL;

VAR
  Parent: TClass;

BEGIN
  Parent        := P.ClassParent;
  IF Parent.InheritsFrom(TStreamableObject) THEN
    FStream.Read(Result, SIZEOF(Result))
  ELSE
    RAISE EObjectStream.CreateResFmt(stRead_InvalidObject,[Parent.ClassName]);
END;

PROCEDURE TObjectStream.WriteParentCode(P : TStreamClass);

VAR
  Parent: TClass;
  Code  : CARDINAL;

BEGIN
  Parent        := P;
  REPEAT
    Parent      := Parent.ClassParent;
    IF Parent.InheritsFrom(TStreamableObject) THEN
    BEGIN
      Code      := TStreamClass(Parent).GetRegistrationCode;
      IF Code = 0 THEN
        RAISE EObjectStream.CreateResFmt(stWrite_InvalidParentCode,[Parent.ClassName]);
    END
    ELSE
      RAISE EObjectStream.CreateResFmt(stWrite_InvalidObject,[Parent.ClassName]);
  UNTIL Code > 0;
  FStream.Write(Code, SIZEOF(Code));
END;

PROCEDURE TObjectStream.Flush;

BEGIN
  IF FStream.InheritsFrom(THandleStream) THEN
    FileFlush(THandleStream(FStream).Handle)
END;

PROCEDURE TObjectStream.Truncate;

BEGIN
  IF FStream.InheritsFrom(THandleStream) THEN
    FileTruncate(THandleStream(FStream).Handle)
END;

FUNCTION TObjectStream.Get: TStreamableObject;

VAR
  OType : CARDINAL;
  i     : WORD;

BEGIN
  IF NrOfStRecs = 0 THEN
    RAISE EObjectStream.CreateRes(stRead_RegBaseEmpty);
  FStream.Read(OType, SIZEOF(OType));
  i := 1;
  REPEAT
    IF StreamRecords^[i].ObjType = OType THEN
      Break;
    INC(I);
  UNTIL i > NrOfStRecs;
  IF i > NrOfStRecs THEN
      RAISE EObjectStream.CreateResFmt(stRead_UnregisteredObject,[OType]);
  Result        := StreamRecords^[i].VmtLink.Load(Self, OType);
END;

PROCEDURE TObjectStream.Put(P : TStreamableObject);

VAR
  OType,
  i     : WORD;

BEGIN
  IF NrOfStRecs = 0 THEN
    RAISE EObjectStream.CreateRes(stWrite_RegBaseEmpty);
  i := 1;
  REPEAT
    IF StreamRecords^[i].VmtLink = P.ClassType THEN
      Break;
    INC(I);
  UNTIL i > NrOfStRecs;
  IF i > NrOfStRecs THEN
    RAISE EObjectStream.CreateResFmt(stWrite_UnregisteredObject,[P.ClassName]);
  WITH StreamRecords^[i] DO
  BEGIN
    FStream.Write(ObjType, SIZEOF(ObjType));
    P.Store(Self);
  END;
END;

FUNCTION TObjectStream.ReadStr: PSTRING;

VAR
  L: Byte;

BEGIN
  Result := NIL;
  FStream.Read(L, SIZEOF(L));
  IF L > 0 THEN
  BEGIN
    GetMem(Result, L + 1);
    Result^[0] := CHAR(L);
    FStream.Read(Result^[1], L);
  END;
END;

PROCEDURE TObjectStream.WriteStr(P: PSTRING);

BEGIN
  IF P <> NIL THEN
    FStream.Write(P^, Length(P^) + 1)
  ELSE
    FStream.Write(EmptyStr, 1);
END;

FUNCTION TObjectStream.StrRead: PCHAR;

VAR
  L : WORD;

BEGIN
  Result := NIL;
  FStream.Read(L, SizeOf(L));
  IF L = 0 THEN
    StrRead := NIL
  ELSE
  BEGIN
    Result:= StrAlloc(L + 1);
    FStream.Read(Result[0], L);
    Result[L] := #0;
  END;
END;

PROCEDURE TObjectStream.StrWrite(P: PCHAR);

VAR
  L: Word;

BEGIN
  IF P = NIL THEN
    L := 0
  ELSE
    L := StrLen(P);
  FStream.Write(L, SizeOf(Word));
  IF P <> NIL THEN
    FStream.Write(P[0], L);
END;

FUNCTION TObjectStream.Read(VAR Buffer; Count: LONGINT): LONGINT;

BEGIN
  Result        := FStream.Read(Buffer,Count);
END;

FUNCTION TObjectStream.Write(CONST Buffer; Count: LONGINT): LONGINT;

BEGIN
  Result        := FStream.Write(Buffer,Count);
END;

FUNCTION TObjectStream.Seek(Offset: LONGINT; Origin: Word): LONGINT;

BEGIN
  Result        := FStream.Seek(Offset,Origin);
END;

FUNCTION TObjectStream.CopyFrom(Source: TStream; Count: LONGINT): LONGINT;

BEGIN
  Result        := FStream.CopyFrom(Source,Count);
END;

FUNCTION TObjectStream.GetPosition: LONGINT;

BEGIN
  Result        := FStream.Position;
END;

PROCEDURE TObjectStream.SetPosition(Pos: LONGINT);

BEGIN
  FStream.Position := Pos;
END;

FUNCTION TObjectStream.GetSize: LONGINT;

BEGIN
  Result        := FStream.Size;
END;

FUNCTION TObjectStream.GetHandle: INTEGER;

BEGIN
  IF FStream IS THandleStream THEN
    Result      := THandleStream(FStream).Handle
  ELSE
    RAISE EObjectStream.CreateResFmt(stCasting_NotExpectedType,[FStream.ClassName,THandleStream.ClassName]);
END;

PROCEDURE TObjectStream.LoadFromStream(Stream: TStream);

BEGIN
  IF FStream IS TMemoryStream THEN
    TMemoryStream(FStream).LoadFromStream(Stream)
  ELSE
    RAISE EObjectStream.CreateResFmt(stCasting_NotExpectedType,[FStream.ClassName,TMemoryStream.ClassName]);
END;

PROCEDURE TObjectStream.LoadFromFile(CONST FileName: STRING);

BEGIN
  IF FStream IS TMemoryStream THEN
    TMemoryStream(FStream).LoadFromStream(Stream)
  ELSE
    RAISE EObjectStream.CreateResFmt(stCasting_NotExpectedType,[FStream.ClassName,TMemoryStream.ClassName]);
END;

PROCEDURE TObjectStream.SaveToStream(Stream: TStream);

BEGIN
  IF FStream IS TMemoryStream THEN
    TMemoryStream(FStream).SaveToStream(Stream)
  ELSE
    RAISE EObjectStream.CreateResFmt(stCasting_NotExpectedType,[FStream.ClassName,TMemoryStream.ClassName]);
END;

PROCEDURE TObjectStream.SaveToFile(CONST FileName: STRING);

BEGIN
  IF FStream IS TMemoryStream THEN
    TMemoryStream(FStream).SaveToFile(FileName)
  ELSE
    RAISE EObjectStream.CreateResFmt(stCasting_NotExpectedType,[FStream.ClassName,TMemoryStream.ClassName]);
END;

PROCEDURE TObjectStream.SetSize(Size: LONGINT);

BEGIN
  IF FStream IS TMemoryStream THEN
    TMemoryStream(FStream).SetSize(Size)
  ELSE
    RAISE EObjectStream.CreateResFmt(stCasting_NotExpectedType,[FStream.ClassName,TMemoryStream.ClassName]);
END;

PROCEDURE TObjectStream.Clear;

BEGIN
  IF FStream IS TMemoryStream THEN
    TMemoryStream(FStream).Clear
  ELSE
    RAISE EObjectStream.CreateResFmt(stCasting_NotExpectedType,[FStream.ClassName,TMemoryStream.ClassName]);
END;

FUNCTION TObjectStream.GetMemory : POINTER;

BEGIN
  IF FStream IS TMemoryStream THEN
    Result      := TMemoryStream(FStream).Memory
  ELSE
    RAISE EObjectStream.CreateResFmt(stCasting_NotExpectedType,[FStream.ClassName,TMemoryStream.ClassName]);
END;

{ TCollection }

PROCEDURE TCollection.CheckList;

BEGIN
  IF Flist = NIL THEN
    RAISE EListError.Create(LoadStr(SListIndexError))
END;

FUNCTION TCollection.Get(Index: INTEGER): POINTER;

BEGIN
  CheckList;
  Result        := FList[Index];
END;

PROCEDURE TCollection.Put(Index: INTEGER; Item: POINTER);

BEGIN
  IF Item = NIL THEN
    RAISE ECollectionError.CreateResFmt(co_NILItem,[Self.ClassName]);
  CheckList;
  FList[Index]  := InsertItem(Item);
END;

CONSTRUCTOR TCollection.Create(ALimit : INTEGER);

BEGIN
  FList         := TList.Create;
  FList.Capacity:= ALimit;
END;

PROCEDURE TCollection.ClearData;

BEGIN
  FreeAll;
  FList.Free;
  INHERITED ClearData;
END;

FUNCTION TCollection.DupData2(Destination       : TExtendedObject) : BOOLEAN;

VAR
  OldList       : TList;
  I             : INTEGER;

BEGIN
  Result        := INHERITED DupData2(Destination);
  IF Result AND (FList <> NIL) THEN
  BEGIN
    WITH TCOllection(Destination) DO
    BEGIN
      OldList   := FList;
      FList     := NIL;
      FList     := TList.Create;
      WITH OldList DO
      BEGIN
        FList.Capacity  := Capacity;
        FOR i := 0 TO Count-1 DO
          FList.Add(DuplicateItem(List^[I]));
      END;
    END;
  END;
END;

CLASS FUNCTION TCollection.GetRegistrationCode : CARDINAL;

BEGIN
  Result := 150;
END;

CONSTRUCTOR TCollection.Load(S: TObjectStream; RCode : CARDINAL);

VAR
  I,
  Count,
  Capacity,
  Delta : INTEGER;

BEGIN
  S.Read(Count, SizeOf(Count));
  S.Read(Capacity, SizeOf(Capacity));
  IF RCode <> 150 THEN                  { RCode may be 50 or 0 }
    S.Read(Delta, SizeOf(Delta));       { The old TCollectionObject had this field also. }
  ClearData;                            { Destroy and clear everything. }
  Create(Capacity);
  FOR I := 1 TO Count DO
    FList.Add(GetItem(S));
END;

PROCEDURE TCollection.Store(S: TObjectStream);

VAR
  I,
  Count,
  Capacity      : INTEGER;

BEGIN
  IF Assigned(FList) THEN
  BEGIN
    Count       := FList.Count;
    Capacity    := FList.Capacity;
  END ELSE
  BEGIN
    Count       := 0;
    Capacity    := 0;
  END;
  S.Write(Count, SizeOf(Count));
  S.Write(Capacity, SizeOf(Capacity));
  FOR I := 0 TO Count-1 DO
    PutItem(S,FList.List^[I])
END;

FUNCTION TCollection.GetItem(S: TObjectStream): POINTER;

BEGIN
  GetItem := S.Get;
END;

PROCEDURE TCollection.PutItem(S: TObjectStream; Item: POINTER);

BEGIN
  S.Put(Item);
END;

FUNCTION TCollection.DuplicateItem(Item : POINTER) : POINTER;

BEGIN
  Result        := NIL;
  IF TObject(Item).InheritsFrom(TExtendedObject) THEN
    Result      := TExtendedObject(Item).Duplicate(TRUE);
END;

FUNCTION TCollection.InsertItem(Item: POINTER) : POINTER;

BEGIN
  InsertItem    := Item;
END;

PROCEDURE TCollection.Insert(Item: POINTER);

BEGIN
  IF Item = NIL THEN
    RAISE ECollectionError.CreateResFmt(co_NILItem,[Self.ClassName]);
  IF Flist = NIL THEN
    FList       := TList.Create;
  FList.Add(InsertItem(Item));
END;

PROCEDURE TCollection.AtInsert(Index: INTEGER; Item: POINTER);

BEGIN
  IF Item = NIL THEN
    RAISE ECollectionError.CreateResFmt(co_NILItem,[Self.ClassName]);
  CheckList;
  FList.Insert(Index, InsertItem(Item));
END;

PROCEDURE TCollection.AtPut(Index: INTEGER; Item: POINTER);

BEGIN
  IF Item = NIL THEN
    RAISE ECollectionError.CreateResFmt(co_NILItem,[Self.ClassName]);
  CheckList;
  FList.Items[Index] := InsertItem(Item);
END;

PROCEDURE TCollection.AtReplace(Index: INTEGER; Item: POINTER);

BEGIN
  IF Item = NIL THEN
    RAISE ECollectionError.CreateResFmt(co_NILItem,[Self.ClassName]);
  CheckList;
  WITH FList DO
  BEGIN
    FreeItem(Items[Index]);
    Items[Index] := InsertItem(Item);
  END;
END;

FUNCTION TCollection.IndexOf(Item: POINTER): INTEGER;

BEGIN
  CheckList;
  Result        := FList.IndexOf(Item);
END;

FUNCTION TCollection.At(Index: INTEGER): POINTER;

BEGIN
  CheckList;
  Result := FList.Items[Index];
END;

PROCEDURE TCollection.AtDelete(Index: INTEGER);

BEGIN
  CheckList;
  FList.Delete(Index);
END;

PROCEDURE TCollection.Delete(Item: POINTER);

BEGIN
  CheckList;
  FList.Remove(Item);
END;

PROCEDURE TCollection.Remove(Item: POINTER);

BEGIN
  CheckList;
  FList.Remove(Item);
  FreeItem(Item);
END;

PROCEDURE TCollection.DeleteAll;

BEGIN
  IF Flist <> NIL THEN
    FList.Count := 0;
END;

PROCEDURE TCollection.FreeItem(Item: POINTER);

BEGIN
  TObject(Item).Free;
END;

PROCEDURE TCollection.AtFree(Index: INTEGER);

BEGIN
  CheckList;
  WITH FList DO
  BEGIN
    FreeItem(Items[Index]);
    Delete(Index);
  END;
END;

PROCEDURE TCollection.FreeAll;

VAR
  I: INTEGER;

BEGIN
  IF Flist <> NIL THEN
  BEGIN
    WITH FList DO
    BEGIN
      FOR I := 0 TO Count - 1 DO
        FreeItem(List^[I]);
      Count := 0;
    END;
  END;
END;

FUNCTION TCollection.FirstThat(Test: POINTER): POINTER; EXTERNAL;

PROCEDURE TCollection.ForEach(Action: POINTER); EXTERNAL;

FUNCTION TCollection.LastThat(Test: POINTER): POINTER; EXTERNAL;

PROCEDURE TCollection.Pack;

BEGIN
  IF Flist <> NIL THEN
    FList.Pack;
END;

PROCEDURE TCollection.SetLimit(ALimit: INTEGER);

BEGIN
  IF FList = NIL THEN
    FList       := TList.Create;
  FList.Capacity        := ALimit;
END;

PROCEDURE TCollection.Push(Item : POINTER);

BEGIN
  IF Flist = NIL THEN
    FList       := TList.Create
  ELSE IF FList.Count = MaxListSize THEN
    RAISE ECollectionError.CreateResFmt(co_StackOverflow,[LoadStr(coStackOverflow),Self.ClassName]);
  FList.Add(InsertItem(Item));
END;

FUNCTION TCollection.Pop : POINTER;

BEGIN
  IF (FList = NIL) OR (FList.Count = 0) THEN
    RAISE ECollectionError.CreateResFmt(co_StackUnderflow,[LoadStr(coStackUnderflow),Self.ClassName]);
  WITH FList DO
  BEGIN
    Result := Items[Count-1];
    Delete(Count-1);
  END;
END;

PROCEDURE TCollection.PopFree;

BEGIN
  IF (FList = NIL) OR (FList.Count = 0) THEN
    RAISE ECollectionError.CreateResFmt(co_StackUnderflow,[LoadStr(coStackUnderflow),Self.ClassName]);
  WITH FList DO
  BEGIN
    FreeItem(Items[Count-1]);
    Delete(Count-1);
  END;
END;

FUNCTION TCollection.PeekTOS : POINTER;

BEGIN
  IF (FList = NIL) OR (FList.Count = 0) THEN
    Result := NIL
  ELSE
    Result := FList.Items[Flist.Count-1];
END;

PROCEDURE TCollection.ReplaceTOS(Item : POINTER);

BEGIN
  IF (FList = NIL) OR (FList.Count = 0) THEN
    RAISE ECollectionError.CreateResFmt(co_StackUnderflow,[LoadStr(coStackUnderflow),Self.ClassName]);
  WITH FList DO
  BEGIN
    FreeItem(Items[Count-1]);
    Delete(Count-1);
    Add(InsertItem(Item));
  END;
END;

PROCEDURE TCollection.SwapTOS;

BEGIN
  IF (FList = NIL) OR (FList.Count <= 1) THEN
    RAISE ECollectionError.CreateResFmt(co_StackUnderflow,[LoadStr(coStackUnderflow),Self.ClassName]);
  WITH FList DO
    Exchange(Count-1,Count-2);
END;

PROCEDURE TCollection.Write(Item : POINTER);

BEGIN
  IF Flist = NIL THEN
    FList       := TList.Create
  ELSE IF FList.Count = MaxListSize THEN
    RAISE ECollectionError.CreateResFmt(co_QueueOverflow,[LoadStr(coQueueOverflow),Self.ClassName]);
  FList.Add(InsertItem(Item));
END;

FUNCTION TCollection.Read : POINTER;

BEGIN
  IF (FList = NIL) OR (FList.Count = 0) THEN
    RAISE ECollectionError.CreateResFmt(co_QueueEmpty,[LoadStr(coQueueEmpty),Self.ClassName]);
  WITH FList DO
  BEGIN
    Result := Items[0];
    Delete(0);
  END;
END;

FUNCTION TCollection.Peek : POINTER;

BEGIN
  IF (FList = NIL) OR (FList.Count = 0) THEN
    Result := NIL
  ELSE
    Result := FList.Items[0];
END;

PROCEDURE TCollection.Drop;

BEGIN
  IF (FList = NIL) OR (FList.Count = 0) THEN
    RAISE ECollectionError.CreateResFmt(co_QueueEmpty,[LoadStr(coQueueEmpty),Self.ClassName]);
  WITH FList DO
  BEGIN
    FreeItem(Items[0]);
    Delete(0);
  END;
END;

{ TSortedCollection }

CLASS FUNCTION TSortedCollection.GetRegistrationCode : CARDINAL;

BEGIN
  Result := 250;
END;

CONSTRUCTOR TSortedCollection.Load(S: TObjectStream; RCode : CARDINAL);

BEGIN
  IF RCode <> 250 THEN          { Stream has been written with old method }
    INHERITED Load(S, 0)        { so load TCollection with RCode = 0 }
  ELSE
    INHERITED Load(S, S.ReadParentCode(TSortedCollection));
  S.Read(FDuplicates, SizeOf(FDuplicates)*2);
END;

PROCEDURE TSortedCollection.Store(S: TObjectStream);

BEGIN
  S.WriteParentCode(TSortedCollection); { Write parent code to enable automatic version control }
  INHERITED Store(S);
  S.Write(FDuplicates, SizeOf(FDuplicates)*2);
END;

PROCEDURE TSortedCollection.AllowDuplicates(D : BOOLEAN);

VAR
  OD    : BOOLEAN;

BEGIN
  OD            := FDuplicates;
  FDuplicates   := D;
  IF NOT(D) AND (OD <> D) THEN
    Sort;
END;

FUNCTION TSortedCollection.IsSorted : BOOLEAN;

BEGIN
  IsSorted      := NOT(FNotSorted);
END;

PROCEDURE TSortedCollection.SetSorted(S : BOOLEAN);

BEGIN
  FNotSorted    := NOT(S);
  Sort;
END;

PROCEDURE TSortedCollection.Sort;

VAR
  C,
  I,
  Index : INTEGER;
  Item  : POINTER;

BEGIN
  IF NOT(FNotSorted) THEN
  BEGIN
    C   := FList.Count;
    FList.Count := 1;
    FOR I       := 1 TO C-1 DO
    BEGIN
      Item      := FList.List^[i];
      IF NOT Search(KeyOf(Item), Index) OR
         FDuplicates THEN
      BEGIN
        FList.Insert(Index, Item);
      END
      ELSE
        FreeItem(Item);
    END;
  END;
END;

FUNCTION TSortedCollection.IndexOf(Item : POINTER): INTEGER;

VAR
  I : INTEGER;

BEGIN
  Result := -1;
  IF Search(KeyOf(Item), I) THEN
  BEGIN
    IF FDuplicates THEN
    BEGIN
      WITH FList DO
      BEGIN
        WHILE (I < Count) AND
              (Item <> List^[I]) DO
          Inc(I);
        IF I < Count THEN
          Result := I;
      END;
    END;
  END;
END;

FUNCTION TSortedCollection.FirstIndexOf(Item : POINTER): INTEGER;

VAR
  I : INTEGER;

BEGIN
  Result := -1;
  IF Search(KeyOf(Item), I) THEN
      Result := I
END;

FUNCTION TSortedCollection.LastIndexOf(Item : POINTER): INTEGER;

VAR
  I : INTEGER;

BEGIN
  Result := -1;
  IF Search(KeyOf(Item), I) THEN
  BEGIN
    WITH FList DO
    BEGIN
      IF FDuplicates AND
         (I < (Count-1)) THEN
      BEGIN
        REPEAT
          INC(I)
        UNTIL (I = (Count-1)) OR
              (Compare(KeyOf(Item), KeyOf(List^[I])) <> 0);
        DEC(I, ORD(Compare(KeyOf(Item), KeyOf(List^[I])) <> 0));
      END;
    END;
    Result := I;
  END;
END;

PROCEDURE TSortedCollection.Insert(Item: POINTER);

VAR
  I: INTEGER;

BEGIN
  IF FNotSorted THEN
    INHERITED Insert(Item)
  ELSE IF NOT Search(KeyOf(Item), I) OR
          FDuplicates THEN
    AtInsert(I, Item);
END;

FUNCTION TSortedCollection.KeyOf(Item: POINTER): POINTER;

BEGIN
  KeyOf := Item;
END;

FUNCTION TSortedCollection.Search(Key: POINTER; VAR Index: INTEGER): BOOLEAN; ASSEMBLER;

VAR
  DataSeg       : WORD;

ASM
        MOV     [DataSeg], DS
        LES     DI, Self
        LDS     SI, [ES:DI+TSortedCollection.FList]
        XOR     DX, DX                  { DX = Index }
        MOV     AX, DS
        OR      AX, SI
        JZ      @@1                     { FList = NIL }
        XOR     AX, AX                  { Result = FALSE }
        MOV     CX, [SI+8]              { CX = Count }
        JCXZ    @@1
        LDS     SI, [SI+4]              { TList.List }
        XOR     BX, BX
        MOV     AL,[ES:DI+TSortedCollection.FNotSorted]
        OR      AL, AL
        JZ      @@5                     { FList is sorted }
@@6:    CALL    @@10
        OR      AX, AX                  { Test Result }
        JZ      @@2                     { Ok, found }
        INC     DX                      { INC(Index) }
        ADD     BX, 4                   { Next Pointer }
        LOOP    @@6                     { End of linear search }
@@5:    DEC     CX                      { CX := Count - 1 }
@@4:    CMP     DX, CX
        JG      @@1
        MOV     AX, DX
        ADD     AX, CX
        SHR     AX, 1
        PUSH    AX
        SHL     AX, 2                   { Scale AX }
        MOV     BX, AX
        CALL    @@10
        OR      AX, AX                  { Test Result }
        JS      @@7                     { if CompareKey < 0 }
        JNZ     @@9                     { if CompareKey > 0 }
        POP     CX                      { was PUSH AX }
        CMP     [BYTE PTR ES:DI+TSortedCollection.FDuplicates],0        { Found }
        JNE     @@8                     { FDuplicates = TRUE }
        MOV     DX, CX
@@8:    DEC     CX
        XOR     AX, AX
        INC     AX                      { Result := TRUE }
        JMP     @@4
@@7:    POP     DX                      { was PUSH AX }
        INC     DX
        XOR     AX, AX                  { Result := FALSE }
        JMP     @@4
@@9:    POP     CX                      { was PUSH AX }
        DEC     CX
        XOR     AX, AX                  { Result := FALSE }
        JMP     @@4
@@2:    INC     AX                      { Result = TRUE }
@@1:    LDS     SI, Index
        MOV     [SI], DX
        MOV     DS, [DataSeg]
        LEAVE
        RETF    10

{ This is the compare key subroutine }
{ On Entry ES:DI points to self, DS:SI points to FList.List, BX has the offset into List }
@@10:   PUSH    DS                      { Save used registers }
        PUSH    SI
        PUSH    BX
        PUSH    DX
        PUSH    CX
        PUSH    WORD PTR [SI+BX+2]      { Push Item }
        PUSH    WORD PTR [SI+BX]
        LDS     SI, [Key]
        PUSH    DS                      { Push Self }
        PUSH    SI
        PUSH    ES                      { Push Self }
        PUSH    DI
        LES     DI, [ES:DI]             { Load Pointer to method table }
        MOV     DS, [DataSeg]           { Restore correct data segment }
        CALL    DWORD PTR [ES:DI+KeyOfOffset]
        PUSH    DX                      { Push Result }
        PUSH    AX
        LES     DI, [Self]
        PUSH    ES                      { Push Self }
        PUSH    DI
        LES     DI, [ES:DI]             { Load Pointer to method table }
        CALL    DWORD PTR [ES:DI+CompareOffset]
        POP     CX                      { Restore used registers }
        POP     DX
        POP     BX
        POP     SI
        POP     DS
        LES     DI, [Self]
        RETN
END;

{ TStringCollection }

FUNCTION TStringCollection.DuplicateItem(Item   : POINTER) : POINTER;

BEGIN
  Result        := NewStr(PSTRING(Item)^);
END;

CLASS FUNCTION TStringCollection.GetRegistrationCode : CARDINAL;

BEGIN
  Result := 151;
END;

CONSTRUCTOR TStringCollection.Load(S: TObjectStream; RCode : CARDINAL);

BEGIN
  IF RCode <> 151 THEN          { Stream has been written with old method }
    INHERITED Load(S, 0)        { so load TSortedCollection with RCode = 0 }
  ELSE                          { FNotCaseSensitive and FAnsiCharSet have been }
  BEGIN                         { initialized to FALSE from Create }
    INHERITED Load(S, S.ReadParentCode(TStringCollection));
    S.Read(FNotCaseSensitive, 2*SizeOf(FNotCaseSensitive));
  END;
END;

PROCEDURE TStringCollection.Store(S: TObjectStream);

BEGIN
  S.WriteParentCode(TStringCollection); { Write parent code to enable automatic version control }
  INHERITED Store(S);
  S.Write(FNotCaseSensitive, 2*SizeOf(FNotCaseSensitive));
END;

FUNCTION TStringCollection.IsCaseSensitive: BOOLEAN;

BEGIN
  Result := NOT(FNotCaseSensitive);
END;

PROCEDURE TStringCollection.SetCaseSensitive(CS : BOOLEAN);

VAR
  OCS   : BOOLEAN;

BEGIN
  OCS                   := FNotCaseSensitive;
  FNotCaseSensitive     := NOT(CS);
  IF OCS = CS THEN
    Sort;
END;

PROCEDURE TStringCollection.SetAnsiCharSet(ACS : BOOLEAN);

VAR
  OACS  : BOOLEAN;

BEGIN
  OACS          := FAnsiCharSet;
  FAnsiCharSet  := ACS;
  IF OACS <> ACS THEN
    Sort;
END;

FUNCTION TStringCollection.Compare(Key1, Key2: POINTER): INTEGER;

BEGIN
  Result := StringCompare[FNotCaseSensitive, FAnsiCharSet](PSTRING(Key1)^,PSTRING(Key2)^);
END;

PROCEDURE TStringCollection.FreeItem(Item: POINTER);

BEGIN
  DisposeStr(Item);
END;

FUNCTION TStringCollection.GetItem(S: TObjectStream): POINTER;

BEGIN
  Result := S.ReadStr;
END;

PROCEDURE TStringCollection.PutItem(S: TObjectStream; Item: POINTER);

BEGIN
  S.WriteStr(Item);
END;

{ TStrCollection }

FUNCTION TStrCollection.DuplicateItem(Item      : POINTER) : POINTER;

BEGIN
  Result        := StrNew(Item);
END;

CLASS FUNCTION TStrCollection.GetRegistrationCode : CARDINAL;

BEGIN
  Result := 169;
END;

CONSTRUCTOR TStrCollection.Load(S: TObjectStream; RCode : CARDINAL);

BEGIN
  IF RCode <> 169 THEN          { Stream has been written with old method }
    INHERITED Load(S, 0)        { so load TStringCollection with RCode = 0 }
  ELSE
    INHERITED Load(S, S.ReadParentCode(TStrCollection));
END;

PROCEDURE TStrCollection.Store(S: TObjectStream);

BEGIN
  S.WriteParentCode(TStrCollection);    { Write parent code to enable automatic version control }
  INHERITED Store(S);
END;

FUNCTION TStrCollection.Compare(Key1, Key2: POINTER): INTEGER;

BEGIN
  Result        := StrCompare[FNotCaseSensitive, FAnsiCharSet](Key1,Key2);
END;

PROCEDURE TStrCollection.FreeItem(Item: POINTER);

BEGIN
  StrDispose(Item);
END;

FUNCTION TStrCollection.GetItem(S: TObjectStream): POINTER;

BEGIN
  Result := S.StrRead;
END;

PROCEDURE TStrCollection.PutItem(S: TObjectStream; Item: POINTER);

BEGIN
  S.StrWrite(Item);
END;

{ Objects registration PROCEDURE }

PROCEDURE RegisterObjects;
BEGIN
  RegisterNewObject(TCollection);
  RegisterNewObject(TStringCollection);
  RegisterNewObject(TStrCollection);
  RegisterObject(TCollection,50);
  RegisterObject(TStringCollection,51);
  RegisterObject(TStrCollection,69);
END;

END.
