unit Mytimer;

{stripped down version of ttimer that doesn't require the vcl overhead}

interface

uses WinTypes, WinProcs, Messages, Sysutils;

type timerproc = procedure;

type
  TMyTimer = class
  private
    FEnabled: Boolean;
    FReserved: Byte;
    FInterval: Word;
    FWindowHandle: HWND;
    FTimerProc:timerproc;
    procedure WndProc(var Msg: TMessage);
  public
    constructor Create(interval:word;
                timerproc:timerproc);
    destructor Destroy; override;
  end;


implementation

{$IFNDEF WIN32}
{ Utility routines }

type
   TWndMethod = procedure(var Message: TMessage) of object;

{ Object instance management }

const
  InstanceCount = 91;

  type
  PObjectInstance = ^TObjectInstance;
  TObjectInstance = packed record
    Code: Byte;
    Offset: Integer;
    case Integer of
      0: (Next: PObjectInstance);
      1: (Method: TWndMethod);
  end;

type
  PInstanceBlock = ^TInstanceBlock;
  TInstanceBlock = packed record
    Next: Word;
    Code: array[1..6] of Byte;
    WndProcPtr: Pointer;
    Instances: array[0..InstanceCount] of TObjectInstance;
  end;

var
  InstBlockList: Word;
  InstFreeList: PObjectInstance;

{ Standard window procedure }
{ In    ES:BX = Address of method pointer }
{ Out   DX:AX = Result }

function StdWndProc(Window: HWND; Message: Word; WParam: Word;
  LParam: Longint): Longint; export; assembler;
asm
        XOR     AX,AX
        PUSH    AX
        PUSH    AX
        PUSH    LParam.Word[2]
        PUSH    LParam.Word[0]
        PUSH    WParam
        PUSH    Message
        MOV     AX,SP
        PUSH    SS
        PUSH    AX
        PUSH    ES:[BX].Word[6]
        PUSH    ES:[BX].Word[4]
        CALL    ES:[BX].Pointer
        ADD     SP,8
        POP     AX
        POP     DX
end;

function MakeObjectInstance(Method: TWndMethod): Pointer;
const
  BlockCode: array[1..6] of Byte = (
    $8C, $CB,  { MOV BX,CS }
    $8E, $C3,  { MOV ES,BX }
    $5B,       { POP BX }
    $EA);      { JMP FAR StdWndProc }
var
  Block: PInstanceBlock;
  Instance: PObjectInstance;
begin
  if InstFreeList = nil then
  begin
    Block := GlobalLock(GlobalAlloc(HeapAllocFlags, SizeOf(TInstanceBlock)));
    Block^.Next := InstBlockList;
    Move(BlockCode, Block^.Code, SizeOf(BlockCode));
    Block^.WndProcPtr := @StdWndProc;
    Instance := @Block^.Instances;
    repeat
      Instance^.Code := $E8;  { CALL NEAR PTR Offset }
      Instance^.Offset := (2 - 3) - PtrRec(Instance).Ofs;
      Instance^.Next := InstFreeList;
      InstFreeList := Instance;
      Inc(PtrRec(Instance).Ofs, SizeOf(TObjectInstance));
    until PtrRec(Instance).Ofs = SizeOf(TInstanceBlock);
    InstBlockList := PtrRec(Block).Seg;
    ChangeSelector(PtrRec(Block).Seg, PtrRec(Block).Seg);
  end;
  Result := InstFreeList;
  PtrRec(Instance).Ofs := PtrRec(InstFreeList).Ofs;
  PtrRec(Instance).Seg := AllocCSToDSAlias(PtrRec(InstFreeList).Seg);
  InstFreeList := Instance^.Next;
  Instance^.Method := Method;
  FreeSelector(PtrRec(Instance).Seg);
end;

{ Free an object instance }

procedure FreeObjectInstance(ObjectInstance: Pointer);
var
  Instance: PObjectInstance;
begin
  if ObjectInstance <> nil then
  begin
    PtrRec(Instance).Ofs := PtrRec(ObjectInstance).Ofs;
    PtrRec(Instance).Seg := AllocCSToDSAlias(PtrRec(ObjectInstance).Seg);
    Instance^.Next := InstFreeList;
    FreeSelector(PtrRec(Instance).Seg);
    InstFreeList := ObjectInstance;
  end;
end;

const
  UtilWindowClass: TWndClass = (
    style: 0;
    lpfnWndProc: @DefWindowProc;
    cbClsExtra: 0;
    cbWndExtra: 0;
    hInstance: 0;
    hIcon: 0;
    hCursor: 0;
    hbrBackground: 0;
    lpszMenuName: nil;
    lpszClassName: 'TPUtilWindow');

function AllocateHWnd(Method: TWndMethod): HWND;
var
  TempClass: TWndClass;
begin
  UtilWindowClass.hInstance := HInstance;
  if not GetClassInfo(HInstance, UtilWindowClass.lpszClassName, TempClass) then
    WinProcs.RegisterClass(UtilWindowClass);
  Result := CreateWindow(UtilWindowClass.lpszClassName, '', 0,
    0, 0, 0, 0, 0, 0, HInstance, nil);
  SetWindowLong(Result, GWL_WNDPROC, Longint(MakeObjectInstance(Method)));
end;

procedure DeallocateHWnd(Wnd: HWND);
var
  Instance: Pointer;
begin
  Instance := Pointer(GetWindowLong(Wnd, GWL_WNDPROC));
  DestroyWindow(Wnd);
  FreeObjectInstance(Instance);
end;

{$ELSE}

type
  TWndMethod = procedure(var Message: TMessage) of object;

const
  InstanceCount = 313;

{ Object instance management }

type
  PObjectInstance = ^TObjectInstance;
  TObjectInstance = packed record
    Code: Byte;
    Offset: Integer;
    case Integer of
      0: (Next: PObjectInstance);
      1: (Method: TWndMethod);
  end;

type
  PInstanceBlock = ^TInstanceBlock;
  TInstanceBlock = packed record
    Next: PInstanceBlock;
    Code: array[1..2] of Byte;
    WndProcPtr: Pointer;
    Instances: array[0..InstanceCount] of TObjectInstance;
  end;

var
  InstBlockList: PInstanceBlock;
  InstFreeList: PObjectInstance;

{ Standard window procedure }
{ In    ECX = Address of method pointer }
{ Out   EAX = Result }

function StdWndProc(Window: HWND; Message, WParam: Longint;
  LParam: Longint): Longint; stdcall; assembler;
asm
        XOR     EAX,EAX
        PUSH    EAX
        PUSH    LParam
        PUSH    WParam
        PUSH    Message
        MOV     EDX,ESP
        MOV     EAX,[ECX].Longint[4]
        CALL    [ECX].Pointer
        ADD     ESP,12
        POP     EAX
end;

{ Allocate an object instance }

function CalcJmpOffset(Src, Dest: Pointer): Longint;
begin
  Result := Longint(Dest) - (Longint(Src) + 5);
end;

function MakeObjectInstance(Method: TWndMethod): Pointer;
const
  BlockCode: array[1..2] of Byte = (
    $59,       { POP ECX }
    $E9);      { JMP StdWndProc }
  PageSize = 4096;
var
  Block: PInstanceBlock;
  Instance: PObjectInstance;
begin
  if InstFreeList = nil then
  begin
    Block := VirtualAlloc(nil, PageSize, MEM_COMMIT, PAGE_EXECUTE_READWRITE);
    Block^.Next := InstBlockList;
    Move(BlockCode, Block^.Code, SizeOf(BlockCode));
    Block^.WndProcPtr := Pointer(CalcJmpOffset(@Block^.Code[2], @StdWndProc));
    Instance := @Block^.Instances;
    repeat
      Instance^.Code := $E8;  { CALL NEAR PTR Offset }
      Instance^.Offset := CalcJmpOffset(Instance, @Block^.Code);
      Instance^.Next := InstFreeList;
      InstFreeList := Instance;
      Inc(Longint(Instance), SizeOf(TObjectInstance));
    until Longint(Instance) - Longint(Block) >= SizeOf(TInstanceBlock);
    InstBlockList := Block;
  end;
  Result := InstFreeList;
  Instance := InstFreeList;
  InstFreeList := Instance^.Next;
  Instance^.Method := Method;
end;

{ Free an object instance }

procedure FreeObjectInstance(ObjectInstance: Pointer);
begin
  if ObjectInstance <> nil then
  begin
    PObjectInstance(ObjectInstance)^.Next := InstFreeList;
    InstFreeList := ObjectInstance;
  end;
end;

var
  UtilWindowClass: TWndClass = (
    style: 0;
    lpfnWndProc: @DefWindowProc;
    cbClsExtra: 0;
    cbWndExtra: 0;
    hInstance: 0;
    hIcon: 0;
    hCursor: 0;
    hbrBackground: 0;
    lpszMenuName: nil;
    lpszClassName: 'TPUtilWindow');


function AllocateHWnd(Method: TWndMethod): HWND;
var
  TempClass: TWndClass;
  ClassRegistered: Boolean;
begin
  UtilWindowClass.hInstance := HInstance;
  ClassRegistered := GetClassInfo(HInstance, UtilWindowClass.lpszClassName,
    TempClass);
  if not ClassRegistered or (TempClass.lpfnWndProc <> @DefWindowProc) then
  begin
    if ClassRegistered then
      {Windows.}UnregisterClass(UtilWindowClass.lpszClassName, HInstance);
      {Windows.}RegisterClass(UtilWindowClass);
  end;
  Result := CreateWindow(UtilWindowClass.lpszClassName, '', 0,
    0, 0, 0, 0, 0, 0, HInstance, nil);
  SetWindowLong(Result, GWL_WNDPROC, Longint(MakeObjectInstance(Method)));
end;

procedure DeallocateHWnd(Wnd: HWND);
var
  Instance: Pointer;
begin
  Instance := Pointer(GetWindowLong(Wnd, GWL_WNDPROC));
  DestroyWindow(Wnd);
  FreeObjectInstance(Instance);
end;

{$ENDIF}

{ TMyTimer }

constructor TMyTimer.Create;
begin
  FTimerProc := TimerProc;
  FInterval := Interval;
  FWindowHandle := AllocateHWnd(WndProc);
  if SetTimer(FWindowHandle, 1, FInterval, nil) = 0 then
    messagebox(0,'window error','testing',mb_ok);
end;

destructor TMyTimer.Destroy;
begin
  KillTimer(FWindowHandle, 1);
  DeallocateHWnd(FWindowHandle);
  inherited Destroy;
end;

procedure TMyTimer.WndProc(var Msg: TMessage);
begin
  with Msg do
    if Msg = WM_TIMER then
      FTimerProc
    else
      Result := DefWindowProc(FWindowHandle, Msg, wParam, lParam);
end;

end.
