unit MSComm;

{ TMSComm VCL component version history
  -----------------------------------

  12/6/95 Version 1.10, Additions and fixes by Rhett R. Rodewald
                        (srrodewa@eclat.uccs.edu)
  7/24/95 Version 1.00, FREEWARE by Jeff Atwood

  General information
  -------------------

  This is a drop-in replacement for the MSCOMM control available in VB 3.0
  professional! I modified it with the goal of making the control work like that
  one, since I used it all the time.. but that was pre-Delphi. :)

  There are no known bugs. This control is freely distributable. Any comments,
  rants, raves, or other horticultural delights can be E-Mailed to me at
  JAtwood159@AOL.COM. Especially let me know if you find a bug or add a new
  nifty feature!

  Version 1.10
  ------------
  Fixed Write.  Removed Application.ProcessMessages as this will cause pending
  writes to be processed in REVERSE order.  Added timeout to wait for buffer to
  clear before write.  No longer fails if transmitting a block longer than the
  buffer.  General Cleanup.  Code much smaller in most places.  Added OnError
  handler.  (Note that the last error to occur will still be reported by
  GetError.)  Added new methods to clear the TX and RX buffers.
}

interface

uses Messages, WinTypes, WinProcs, Classes, Forms, SysUtils;

{ These are the enumerated types supported by the TMSComm control }

type
  TBaudRate    = (br110, br300, br600, br1200, br2400, br4800, br9600, br14400,
                  br19200, br38400, br56000, br128000, br256000);
  TParityBits  = (pbNone, pbOdd, pbEven, pbMark, pbSpace);
  TDataBits    = (dbFour, dbFive, dbSix, dbSeven, dbEight);
  TStopBits    = (sbOne, sbOnePointFive, sbTwo);
  TCommEvent   = (ceRxChar, ceRxFlag, ceTxEmpty, ceCts, ceDsr, cdRlsd, ceBreak,
                  ceErr, ceRing, cePErr, ceCtss, ceDummy, ceRlsds );
  TFlowControl = (fcNone, fcRTSCTS, fcXONXOFF);
  TCommEvents  = set of TCommEvent;

type

  { These are the events for the TComm object }

  TNotifyCommEventEvent   = procedure(Sender    : TObject;
                                      CommEvent : TCommEvents) of object;

  TNotifyReceiveEvent     = procedure(Sender    : TObject;
                                      Count     : Word       ) of object;

  TNotifyTransmitLowEvent = procedure(Sender    : TObject;
                                      Count     : Word       ) of object;

  TNotifyErrorEvent       = procedure(Sender    : TObject;
                                      CommErr   : Word;
                                      ErrStr    : String     ) of object;

  { This is the TMSComm object }

  TMSComm = class(TComponent)
  private
    FVersion       : Single;
    FPort          : Byte;
    FBaudRate      : TBaudRate;
    FParityBits    : TParityBits;
    FDataBits      : TDataBits;
    FStopBits      : TStopBits;
    FFlowControl   : TFlowControl;
    FRxBufSize     : Word;
    FTxBufSize     : Word;
    FRxFull        : Word;
    FTxLow         : Word;
    FTxTimeout     : Word;
    FEvents        : TCommEvents;
    FOnCommEvent   : TNotifyCommEventEvent;
    FOnReceive     : TNotifyReceiveEvent;
    FOnTransmitLow : TNotifyTransmitLowEvent;
    FOnError       : TNotifyErrorEvent;
    FhWnd          : hWnd;
    cId            : Integer;                        { handle to comm port }
    Error          : String;
    procedure SetBaudRate(Value: TBaudRate);
    procedure SetParityBits(Value: TParityBits);
    procedure SetDataBits(Value: TDataBits);
    procedure SetStopBits(Value: TStopBits);
    procedure SetFlowControl(Value: TFlowControl);
    procedure SetRxFull(Value: Word);
    procedure SetTxLow(Value: Word);
    procedure SetEvents(Value: TCommEvents);
    procedure WndProc(var Msg: TMessage);
    procedure DoEvent;
    procedure DoReceive;
    procedure DoTransmit;
    function  parseOpenErr(Errcode: Integer): String;
    function  parseGenErr(var Stat : TComStat): String;
  public
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy; override;
    procedure   Write(Data: PChar; Len: Word);
    procedure   Read(Data: PChar; Len: Word);
    function    Open: Boolean;
    procedure   Close;
    procedure   FlushRX;
    procedure   FlushTX;
    function    GetError: String;
  published
    property Version       : Single                  read FVersion;
    property Port          : Byte                    read FPort          write FPort;
    property BaudRate      : TBaudRate               read FBaudRate      write SetBaudRate;
    property ParityBits    : TParityBits             read FParityBits    write SetParityBits;
    property DataBits      : TDataBits               read FDataBits      write SetDataBits;
    property StopBits      : TStopBits               read FStopBits      write SetStopBits;
    property FlowControl   : TFlowControl            read FFlowControl   write SetFlowControl;
    property TxBufSize     : Word                    read FTxBufSize     write FTxBufSize;
    property RxBufSize     : Word                    read FRxBufSize     write FRxBufSize;
    property RxFullCount   : Word                    read FRxFull        write SetRxFull;
    property TxLowCount    : Word                    read FTxLow         write SetTxLow;
    property TxTimeout     : Word                    read FTxTimeout     write FTxTimeout;
    property Events        : TCommEvents             read FEvents        write SetEvents;
    property OnCommEvent   : TNotifyCommEventEvent   read FOnCommEvent   write FOnCommEvent;
    property OnReceive     : TNotifyReceiveEvent     read FOnReceive     write FOnReceive;
    property OnTransmitLow : TNotifyTransmitLowEvent read FOnTransmitLow write FOnTransmitLow;
    property OnError       : TNotifyErrorEvent       read FOnError       write FOnError;
  end;

procedure Register;


implementation


{ Set baud rate: 110-256,000. Notice that this will change the baud rate of the port
 immediately-- if it is currently open! This goes for most of the other com port
 settings below as well.}
procedure TMSComm.SetBaudRate(Value: TBaudRate);
var DCB: TDCB;
begin
  FBaudRate := Value;
  if cId >= 0 then begin
    GetCommState(cId, DCB);
    case Value of
      br110:    DCB.BaudRate := CBR_110;
      br300:    DCB.BaudRate := CBR_300;
      br600:    DCB.BaudRate := CBR_600;
      br1200:   DCB.BaudRate := CBR_1200;
      br2400:   DCB.BaudRate := CBR_2400;
      br4800:   DCB.BaudRate := CBR_4800;
      br9600:   DCB.BaudRate := CBR_9600;
      br14400:  DCB.BaudRate := CBR_14400;
      br19200:  DCB.BaudRate := CBR_19200;
      br38400:  DCB.BaudRate := CBR_38400;
      br56000:  DCB.BaudRate := CBR_56000;
      br128000: DCB.BaudRate := CBR_128000;
      br256000: DCB.BaudRate := CBR_256000;
    end; {case}
    SetCommState(DCB);
  end; {if}
end; {SetBaudRate}


{ set parity: none, odd, even, mark, space }
procedure TMSComm.SetParityBits(Value: TParityBits);
var DCB: TDCB;
begin
  FParityBits := Value;
  if cId >= 0 then begin
    GetCommState(cId, DCB);
    DCB.Parity := ord( Value );
    SetCommState(DCB);
  end; {if}
end; {SetParityBits}


{ set # of data bits 4-8 }
procedure TMSComm.SetDataBits(Value: TDataBits);
var DCB: TDCB;
begin
  FDataBits := Value;
  if cId >= 0 then begin
    GetCommState(cId, DCB);
    DCB.ByteSize := ord( Value ) + 4;
    SetCommState(DCB);
  end; {if}
end; {SetDataBits}


{ set number of stop bits 1, 1.5 or 2 }
procedure TMSComm.SetStopBits(Value: TStopBits);
var DCB: TDCB;
begin
  FStopBits := Value;
  if cId >= 0 then begin
    GetCommState(cId, DCB);
    DCB.StopBits := ord( Value );
    SetCommState(DCB);
  end; {if}
end; {SetStopBits}


{ Set flow control: None, RTS/CTS, or Xon/Xoff. Flow control works in conjunction
with the read and write buffers to ensure that the flow of data *will* stop if
the buffers get critically full. If there is no flow control, it's possible
to lose data.. with flow control on, the chance for losing data is much lower.
Standard (16450 and 16550) UARTs do not implement flow control in hardware, so
if the processor doesn't service the interrupt and halt the flow, data will be
lost anyway.  (I've heard rumors of a 16650 UART that works correctly, but I
haven't seen one yet.) }
procedure TMSComm.SetFlowControl(Value: TFlowControl);
var DCB: TDCB;
begin
  FFlowControl := Value;
  if cId >= 0 then begin
    GetCommState(cId, DCB);
    DCB.Flags := DCB.Flags or dcb_Binary;  {Set Binary Mode (no EOF checking) }
    DCB.Flags := DCB.Flags and (not(dcb_OutxCtsFlow or dcb_Rtsflow or dcb_OutX or dcb_InX));
    case Value of
      fcNone: ;
      fcRTSCTS: DCB.Flags := DCB.Flags or dcb_OutxCtsFlow or dcb_Rtsflow;
      fcXONXOFF: DCB.Flags := DCB.Flags or dcb_OutX or dcb_InX;
    end;
    SetCommState(DCB);
  end; {if}
end; {SetFlowControl}


{ RxFull indicates the number of bytes the COM driver must write to the
application's input queue before sending a notification message. The message
signals the application to read information from the input queue. This "forces"
the driver to send notification during periods of data "streaming." It will
stop what it's doing and notify you when it gets at least this many chars.
This will only affect data streaming; normally data is sent during lulls in
the "stream." If there are no lulls, this setting comes into effect. The
event OnReceive fires when ANY amount of data is received. The maximum
chunk of data you will receive is set by the RxFull amount. }
procedure TMSComm.SetRxFull(Value: Word);
begin
  FRxFull := Value;
  if cId >= 0 then
    EnableCommNotification(cId, FhWnd, FRxFull, FTxLow);
end; {SetRxFull}


{ TxLow Indicates the minimum number of bytes in the output queue. When the
number of bytes in the output queue falls below this number, the COM driver
sends the application a notification message, signaling it to write information
to the output queue. This can be handy to avoid overflowing the (outgoing)
read buffer. The event OnTransmitLow fires when this happens.}
procedure TMSComm.SetTxLow(Value: Word);
begin
  FTxLow := Value;
  if cId >= 0 then
    EnableCommNotification(cId, FhWnd, FRxFull, FTxLow);
end; {SetTxLow}


{ Build the event mask. Indicates which misc events we want the comm control to
tell us about. }
procedure TMSComm.SetEvents(Value: TCommEvents);
begin
  FEvents := Value;
  if cId >= 0 then begin
    SetCommEventMask(cId, word(FEvents));
  end; {if}
end; {SetEvents}


{ This is the message handler for the invisible window; it handles comm msgs
that are handed to the invisible window. We hook into these messages using
EnableCommNotification and our invisible window handle. This routine hands
off to the "do(x)" routines below. }
procedure TMSComm.WndProc(var Msg: TMessage);
begin
  with Msg do begin
    if Msg = WM_COMMNOTIFY then begin
      case lParamLo of
        CN_EVENT    : DoEvent;
        CN_RECEIVE  : DoReceive;
        CN_TRANSMIT : DoTransmit;
      end; {case}
    end else
      Result := DefWindowProc(FhWnd, Msg, wParam, lParam);
  end; {with}
end; {WndProc}


{ some comm event occured. see if we need to report it as an event based
 on the FOnEvent flags set in the control. }
procedure TMSComm.DoEvent;
var Events: Word;
begin
  if (cId >= 0) and Assigned(FOnCommEvent) then begin
    Events := GetCommEventMask(cId, Integer(FEvents));
    FOnCommEvent(Self, TCommEvents( Events ));
  end; {if}
end; {DoEvent}


{ we rec'd some data, see if receive event is on and fire }
procedure TMSComm.DoReceive;
var Stat: TComStat;
begin
  if (cId >= 0) and Assigned(FOnReceive) then begin
    Error := ParseGenErr( Stat );
    repeat
      FOnReceive(Self, Stat.cbInQue);
      Error := ParseGenErr( Stat );
    until (Stat.cbInQue = 0);
    {If more characters have arrived, read them immediately.}
  end; {if}
end; {DoReceive}


{ This event will fire when the transmit buffer goes BELOW the point set
 in txLowCount. It will NOT fire when a transmission takes place. }
procedure TMSComm.DoTransmit;
var Stat: TComStat;
begin
  if (cId >= 0) and Assigned(FOnTransmitLow) then begin
    Error := ParseGenErr( Stat );
    FOnTransmitLow(Self, Stat.cbOutQue);
  end; {if}
end; {DoTransmit}


{ construct: create invisible message window, set default values }
constructor TMSComm.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FhWnd       := AllocateHWnd(WndProc);
  Error       := '';
  FVersion    := 1.10;
  FPort       := 2;
  FBaudRate   := br9600;
  FParityBits := pbNone;
  FDataBits   := dbEight;
  FStopBits   := sbOne;
  FTxBufSize  := 2048;
  FRxBufSize  := 4096;
  FRxFull     := 2048;
  FTxLow      := 512;
  FTxTimeout  := 30;  {default 30 sec wait for buffer to empty on TX}
  FEvents     := [];
  cId         := -1;
end; {Create}


{ destructor: close invisible message window, close comm port }
destructor TMSComm.Destroy;
begin
  DeallocatehWnd(FhWnd);
  if cId >= 0 then
    CloseComm(cId);
  inherited Destroy;
end; {Destroy}


{ Write data to comm port. }
procedure TMSComm.Write(Data: PChar; Len: Word);
var Stat      : TComStat;
    FirstTick : LongInt;
begin
  if cId >= 0 then begin
    if Len > FTxBufSize then begin
      {Write too large, split it in two and call ourself recursively.}
      Write( Data, (Len div 2) );
      Write( @(Data[Len div 2]), Len - (Len div 2) );
    end;

    FirstTick := GetTickCount;
    repeat
      Error := ParseGenErr( Stat );
    until ((FTxBufSize - stat.cbOutQue) >= Len) or
          ((GetTickCount - FirstTick) >= (FTxTimeout * 1000));
    { Wait 30 Seconds at most for TX buffer to clear }
    { This may fail if windows has been running for 49 days (yeah, right) }

    if WriteComm(cId, Data, Len) < 0 then
      Error := ParseGenErr( Stat );
    GetCommEventMask(cId, Integer($FFFF));
  end; {if}
end; {Write}


{ Read data from comm port. Should only do read when you've been notified you
 have data. Attempting to read when nothing is in read buffer results
 in spurious error. You can never read a larger chunk than the read buffer
 size. NOTE: theoretically, you should check the Error property for errors
 after every read. Any error during read or write can stop flow of data. }
procedure TMSComm.Read(Data: PChar; Len: Word);
var Stat : TComStat;
begin
  if cId >= 0 then begin
    if ReadComm(cId, Data, Len) < 0 then
      Error := ParseGenErr( Stat );
    GetCommEventMask(cId, Integer($FFFF));
  end; {if}
end; {Read}


{ failure to open results in a negative cId, this will translate the
  negative cId value into an explanation. }
function TMSComm.parseOpenErr(Errcode: Integer): String;
begin
  case errcode of
    IE_BADID    : result := 'Device identifier is invalid or unsupported';
    IE_OPEN     : result := 'Device is already open.';
    IE_NOPEN    : result := 'Device is not open.';
    IE_MEMORY   : result := 'Cannot allocate queues.';
    IE_DEFAULT  : result := 'Default parameters are in error.';
    IE_HARDWARE : result := 'Hardware not available (locked by another device).';
    IE_BYTESIZE : result := 'Specified byte size is invalid.';
    IE_BAUDRATE : result := 'Device baud rate is unsupported.';
  else
    result := 'Open error ' + IntToStr(Errcode);
  end; {case}

  if (cId >= 0) and Assigned(FOnError) then
    FOnError(Self, Errcode, result);       {Send Error Msg to Handler}
end; {parseOpenErr}


{ failure to read or write to comm port results in a negative returned
  value. This will translate the value into an explanation. }
function TMSComm.ParseGenErr(var Stat : TComStat): String;
var errCode: Word;
begin
  errCode := GetCommError(cId, stat);
  if (errCode <> 0) then begin
    case errcode of
      CE_BREAK    : result := 'Hardware detected a break condition.';
      CE_CTSTO    : result := 'CTS (clear-to-send) timeout.';
      CE_DNS      : result := 'Parallel device was not selected.';
      CE_DSRTO    : result := 'DSR (data-set-ready) timeout.';
      CE_FRAME    : result := 'Hardware detected a framing error.';
      CE_IOE      : result := 'I/O error during communication with parallel device.';
      CE_MODE     : result := 'Requested mode is not supported';
      CE_OOP      : result := 'Parallel device is out of paper.';
      CE_OVERRUN  : result := 'Character was overwritten before it could be retrieved.';
      CE_PTO      : result := 'Timeout during communication with parallel device.';
      CE_RLSDTO   : result := 'RLSD (receive-line-signal-detect) timeout.';
      CE_RXOVER   : result := 'Receive buffer overflow.';
      CE_RXPARITY : result := 'Hardware detected a parity error.';
      CE_TXFULL   : result := 'Transmit buffer overflow.';
    else
      result := 'General error ' + IntToStr(errcode);
    end; {case}

    if (cId >= 0) and Assigned(FOnError) then
      FOnError(Self, Errcode, result);

  end else
    result := Error;  {Return existing error string (no change) in status}
end; {ParseGenErr}


{ returns error text (if any) and clears it }
function TMSComm.GetError: String;
begin
  Result := Error;
  Error := '';
end; {GetError}


{ Explicitly open port. Returns success/failure, check error property for details.
 This routine also begins hooking the comm messages to our invisible window we
 created upon instantiation. Will close port (if open) before re-opening. }
function TMSComm.Open: Boolean;
var commName: PChar;
    tempStr: String;
begin
  if Fport <> 0 then begin
    close;
    tempStr := 'COM' + IntToStr(Fport) + ':';
    commName := StrAlloc(10);
    StrPCopy(commName, tempStr);
    cId := OpenComm(commName, RxBufSize, TxBufSize);
    StrDispose(commName);
    if cId < 0 then begin
      Error := parseOpenErr(cId);
      result := False;
      exit;
    end; {if}
    SetBaudRate(FBaudRate);
    SetParityBits(FParityBits);
    SetDataBits(FDataBits);
    SetStopBits(FStopBits);
    SetFlowControl(FFlowControl);
    SetEvents(FEvents);
    EnableCommNotification(cId, FhWnd, FRxFull, FTxLow);
    result := True;
  end else
    result := False;  {Can't open a non-existant port}
end; {Open}


{ closes the comm port, if it is open. }
procedure TMSComm.Close;
begin
  if cId >= 0 then
    CloseComm(cId);
end; {Close}


{ Empty the RX Queue }
procedure TMSComm.FlushRX;
begin
  FlushComm( cId, 1 ); {Empty RX Queue}
end; {FlushRX}


{ Empty the TX Queue }
procedure TMSComm.FlushTX;
begin
  FlushComm( cId, 0 ); {Empty TX Queue}
end; {FlushTX}


{ registers this VCL component and adds the icon to the palette }
procedure Register;
begin
  RegisterComponents('Custom', [TMSComm]);
end; {Register}


end.
