unit MainUnit;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls, ImgList, Menus, Digits, Registry;

Type
  TBestPlayer = Record
     PlayerName,          { Player Name }
     BoardSize: String;   { Board Size and Tiles}
     Time,                { Time Elapsed }
     Mines,               { Number of Mines }
     Score: Integer;      { Score }
  End;
  TTileState =
    (tsCover, tsUncover,  { Possible Tile states}
     tsMarked, tsQuestion);
  TBoardTile = Record
       MinesAround: Byte; { Number of Mines around the tile }
       IsMine: Boolean;   { Is Mine or not }
       State: TTileState; { Curren State of The Tile }
  End;

type
  TMainForm = class(TForm)
    CloseBrdImg: TImageList;
    OpenBrdImg: TImageList;
    Panel1: TPanel;
    Board: TPaintBox;
    MinesLeftLbl: TLabel;
    MainMenu: TMainMenu;
    File1: TMenuItem;
    NewGame1: TMenuItem;
    Exit1: TMenuItem;
    N1: TMenuItem;
    ScoreBoard1: TMenuItem;
    Timer: TTimer;
    Help1: TMenuItem;
    Abouty1: TMenuItem;
    TimeElapsedCntr: TDigits;
    Label1: TLabel;
    MinesLeftCntr: TDigits;
    Label2: TLabel;
    procedure BoardPaint(Sender: TObject);
    procedure BoardMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure FormCreate(Sender: TObject);
    procedure NewGame1Click(Sender: TObject);
    procedure Exit1Click(Sender: TObject);
    procedure TimerTimer(Sender: TObject);
    procedure Abouty1Click(Sender: TObject);
    procedure ScoreBoard1Click(Sender: TObject);
    procedure LoadHighScores;
    procedure SaveHighScores;
    procedure FormDestroy(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
    BestPlayers: Array[1..10] of TBestPlayer;
    MineBoard: Array[0..50,0..50] of TBoardTile;
  end;

var
  MainForm: TMainForm;

  PlayingGame: Boolean; { Just respond to events while playing }
  MinesLeft: Integer;   { Mines to Uncover Left }
  MaxX, MaxY,           { Board dimensions }
  MaxMines: Byte;       { Maximum number of mines }
  SecsElapsed : Integer;{ Time Elapsed }

implementation

uses NewGameUnit, AboutUnit, ScoreBoardUnit, WinnerUnit, LoserUnit,
  HighScoreUnit;

{$R *.DFM}

{ Load the Top Ten Table from the Registry to memory}
procedure TMainForm.LoadHighScores;
Var
  RegScores: TRegIniFile;
  i: Byte;
  ps: String[2];
Begin
  RegScores:= TRegIniFile.Create('MineKiller');
  For i:=1 to 10 do
  Begin
    ps:= IntToStr(i); If i<10 then ps:='0'+ps;
    With BestPlayers[i] do
    Begin
      PlayerName:= RegScores.ReadString( 'Player'+ps,'PlayerName','Anonymous Player');
      BoardSize:=  RegScores.ReadString( 'Player'+ps,'BoardSize','10x10 (100)');
      Time:=       RegScores.ReadInteger('Player'+ps,'Time',20);
      Mines:=      RegScores.ReadInteger('Player'+ps,'Mines',10);
      Score:=      RegScores.ReadInteger('Player'+ps,'Score',10);
    End;
  End;
  RegScores.Free;
End;

{ Save the High Scores table from memory to the Registry}
procedure TMainForm.SaveHighScores;
Var
  RegScores: TRegIniFile;
  i:Byte;
  ps: String[2];
Begin
  RegScores := TRegIniFile.Create('MineKiller');
  ps:= IntToStr(i); If i<10 then ps:='0'+ps;
  For i:=1 to 10 do
  Begin
    With BestPlayers[i] do
    Begin
      RegScores.WriteString( 'Player'+ps,'PlayerName',PlayerName);
      RegScores.WriteString( 'Player'+ps,'BoardSize',BoardSize);
      RegScores.WriteInteger('Player'+ps,'Time',Time);
      RegScores.WriteInteger('Player'+ps,'Mines',Mines);
      RegScores.WriteInteger('Player'+ps,'Score',Score);
    End;
  End;
  RegScores.Free;
End;

{ Main procedure to draw the Board State }
procedure TMainForm.BoardPaint(Sender: TObject);
var
  i,j:Byte;
  tmpGraphic: TBitmap;
begin
  tmpGraphic := TBitmap.Create;
  For i:=0 to (MaxX-1) do
    For j:=0 to (MaxY-1) do
  Begin
    Case MineBoard[i,j].State Of
      tsCover:    CloseBrdImg.GetBitmap(0,tmpGraphic); { The Covered State }
      tsMarked:   CloseBrdImg.GetBitmap(1,tmpGraphic); { The Marked State }
      tsQuestion: CloseBrdImg.GetBitmap(2,tmpGraphic); { The Question State }
      tsUncover:  Begin
                    If Not(MineBoard[i,j].IsMine)
                      Then OpenBrdImg.GetBitmap(MineBoard[i,j].MinesAround,tmpGraphic)
                      Else CloseBrdImg.GetBitmap(3,tmpGraphic);
                  End;
    End;
    { Draw the 16x16 tile }
    (Sender As TPaintBox).Canvas.Draw(i*16,j*16,tmpGraphic);
  End;
  tmpGraphic.FreeImage;
end;

{ Respond to user interaction }
procedure TMainForm.BoardMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);

  { This procedure tries to uncover as many tiles as possible }
  { depending of the number of mines the tile around has }
  Procedure DoUnCover(xx,yy:Byte);
  Begin
    If Not(MineBoard[xx,yy].State = tsUncover) Then
    Begin

      { Increase the mines marked odometer }
      If (MineBoard[xx,yy].State = tsMarked) Then
      Begin
        Inc(MinesLeft);
        MinesLeftCntr.Increase;
      End;

      { Uncover this mine }
      MineBoard[xx,yy].State := tsUncover;
      { If it has not any mine around, try to uncover the cells }
      { around this one in the eight possible directions }
      If MineBoard[xx,yy].MinesAround = 0 Then
      Begin
        If (xx>0)    and     (yy>00)       Then DoUnCover(xx-1,yy-1);
        If (xx>0)                          Then DoUnCover(xx-1,yy);
        If (xx>0)    and     (yy<(MaxY-1)) Then DoUnCover(xx-1,yy+1);
        If                   (yy>0)        Then DoUnCover(xx,yy-1);
        If                   (yy<(MaxY-1)) Then DoUnCover(xx,yy+1);
        If (xx<(MaxX-1)) and (yy>0)        Then DoUnCover(xx+1,yy-1);
        If (xx<(MaxX-1))                   Then DoUnCover(xx+1,yy);
        If (xx<(MaxX-1)) and (yy<(MaxY-1)) Then DoUnCover(xx+1,yy);
      End;
    End;
  End;

  { This procedure checks the current data for a Winner, and }
  { asks for it in case it is a high score }
  Procedure CheckForAWinner;
  Var
    xx,yy:Byte;
    tCount:Integer;

    { Check for a high score after the Winner dialog }
    Procedure CheckScores;
    Var
      PlyrScore: Integer;
      i,j:Byte;
    Begin
      { First, calculate the score }
      { Maybe I need a better score function. }
      PlyrScore:= Abs(MaxMines*20-TimeElapsedCntr.Value);
      i:=10;
      While (PlyrScore>BestPlayers[i].Score) and (i>0) do
        Dec(i);
      If i<10 Then { If it is a high score... }
      Begin
        HighScoreForm := THighScoreForm.Create(Self);
        HighScoreForm.ScoreLbl.Caption :=
          'You scored '+IntToStr(PlyrScore)+' points. '+
          'You are in position '+IntToStr(i+1)+' in the '+
          'Minekiller high score table!';
        HighScoreForm.ShowModal;

        { Move down the score table one position}
        For j:=9 downto (i+1) do
          BestPlayers[j+1]:= BestPlayers[j];

        With BestPlayers[i+1] do
        Begin
          PlayerName := HighScoreForm.PlayerName.Text;
          BoardSize :=  IntToStr(MaxX)+'x'+IntToStr(MaxY)+' ('+IntToStr(MaxX*MaxY)+')';
          Time :=       TimeElapsedCntr.Value;
          Mines :=      MaxMines;
          Score :=      PlyrScore;
        End;

        HighScoreForm.Free;

      End;
    End;

  Begin
    { Count all the uncovered tiles }
    { if No = MaxX*MaxY-MaxMines, then whe have a Winner }
    tCount := 0;
    For xx:=0 to (MaxX-1) do
      For yy:=0 to (MaxY-1) do
        If (MineBoard[xx,yy].State = tsUncover) Then
          Inc(tCount);

    If (tCount = (MaxX*MaxY-MaxMines)) Then
    Begin
      Timer.Enabled := False;
      PlayingGame:=False;
      { Show the Winner form and store the score }
      Board.Repaint;
      Beep;
      WinnerForm := TWinnerForm.Create(Self);
      WinnerForm.ShowModal;
      WinnerForm.Free;
      { Check for a High Score }
      CheckScores;
    End;
  End;

Var
  xtile, ytile: Byte;
  i,j: Byte;
begin
  If PlayingGame Then { Respond to events just when playing a game }
  Begin
    xtile := x div 16;
    ytile := y div 16;
    { Pressing the Right Mouse Button }
    { Cycle through informative states }
    If (Shift = [ssRight]) and
    (MineBoard[xtile,ytile].State in [tsCover, tsMarked, tsQuestion]) Then
    Begin
      Case MineBoard[xtile,ytile].State Of
        tsCover:    Begin
                      MineBoard[xtile,ytile].State := tsMarked;
                      Dec(MinesLeft);
                      MinesLeftCntr.Decrease;
                    End;
        tsMarked:   Begin
                      MineBoard[xtile,ytile].State := tsQuestion;
                      Inc(MinesLeft);
                      MinesLeftCntr.Increase;
                    End;
        tsQuestion: MineBoard[xtile,ytile].State := tsCover;
      End;
    End;

    { Uncover one tile }
    If (Shift = [ssLeft]) and
    Not(MineBoard[xtile,ytile].State = tsUncover) Then
    Begin
      If Not(MineBoard[xtile,ytile].IsMine) Then
      Begin
        { Ok Uncover JUST one mine if it has more than zero mines}
        { or as many as possible if it has zero and the ones around too }
        { -- Recursive Call }
        DoUnCover(xtile,ytile);
        { Now, check for a Winner }
        CheckForAWinner;
      End Else
      Begin
        { Ok. You've LOST! you clicked over a little black mine }
        { Now's time to show all the mines -- well uncover all of them }
        For i:=0 to (MaxX-1) do
          For j:=0 to (MaxY-1) do
            MineBoard[i,j].State := tsUncover;
        { Stop Game and Timer }
        PlayingGame := False;
        Timer.Enabled:=False;
        Beep;
        Board.Repaint;
        { You're a loser }
        LoserForm := TLoserForm.Create(Self);
        LoserForm.ShowModal;
        LoserForm.Free;
      End;
    End;
    Board.Repaint;
  End;
end;

procedure TMainForm.FormCreate(Sender: TObject);
begin
  { Set the Form Width and Height and Load Scores}
  PlayingGame := False;
  LoadHighScores;
end;

{ Start a new game }
procedure TMainForm.NewGame1Click(Sender: TObject);
Var
  i, j, tMinesAround, nLeft: Byte;
  mPos: Word;
begin
  NewGameForm := TNewGameForm.Create(Self);
  NewGameForm.ShowModal;
  If NewGameForm.ModalResult = mrOk Then
  Begin
    MaxX:= NewGameForm.HorSizeSlider.Position;
    MaxY:= NewGameForm.VertSizeSlider.Position;
    MaxMines := NewGameForm.NoMinesSlider.Position;

    { Initialize the Board -- All Tiles Covered}
    For i:=0 to (MaxX-1) do
      For j:=0 to (MaxY-1) do
        With MineBoard[i,j] do
        Begin
          State := tsCover;
          IsMine := False;
          MinesAround:= 0;
        End;

    { Create the Mines }
    nLeft := MaxMines;
    Randomize;
    Repeat
      mPos:= Random(MaxX*MaxY);
      If Not(MineBoard[(mPos div MaxY),(mPos mod MaxY)].IsMine) Then
      Begin
        MineBoard[(mPos div MaxY),(mPos mod MaxY)].IsMine:= True;
        Dec(nLeft);
      End;
    Until (nLeft = 0);

    { Initialize mines Around }
    For i:=0 to (MaxX-1) do
      For j:=0 to (MaxY-1) do
      Begin
        If Not(MineBoard[i,j].IsMine) Then
        Begin
          { Just to calculate the mines around each tile }
          tMinesAround:= 0;
          If (i>00)      and  (j>00)       Then If MineBoard[i-1,j-1].IsMine Then Inc(tMinesAround);
          If (i>00)                        Then If MineBoard[i-1,j].IsMine   Then Inc(tMinesAround);
          If (i>00)      and  (j<(MaxY-1)) Then If MineBoard[i-1,j+1].IsMine Then Inc(tMinesAround);
          If                  (j>00)       Then If MineBoard[i,j-1].IsMine   Then Inc(tMinesAround);
          If                  (j<(MaxY-1)) Then If MineBoard[i,j+1].IsMine   Then Inc(tMinesAround);
          If (i<(MaxX-1)) and (j>00)       Then If MineBoard[i+1,j-1].IsMine Then Inc(tMinesAround);
          If (i<(MaxX-1))                  Then If MineBoard[i+1,j].IsMine   Then Inc(tMinesAround);
          If (i<(MaxX-1)) and (j<(MaxY-1)) Then If MineBoard[i+1,j+1].IsMine Then Inc(tMinesAround);
          MineBoard[i,j].MinesAround := tMinesAround;
       End;
     End;

    { Paint the Board }
    MainForm.Width := 6+16*MaxX;
    MainForm.Height:= 94+16*MaxY;
    MainForm.Left := (Screen.Width - MainForm.Width) div 2;
    MainForm.Top :=  (Screen.Height - MainForm.Height) div 2;
    Board.Width := 16*MaxX;
    Board.Height:= 16*MaxY;
    PlayingGame:= True;
    MinesLeft := MaxMines;
    MinesLeftCntr.Value := MinesLeft;
    Timer.Enabled := True;
    SecsElapsed:=0;
    TimeElapsedCntr.Value :=0;
    Board.Visible := True;
    Board.Repaint;
  End;
  NewGameForm.Free;
end;

procedure TMainForm.Exit1Click(Sender: TObject);
begin
  { End of program }
  Close;
end;

procedure TMainForm.TimerTimer(Sender: TObject);
begin
  { Update the time odometer }
  TimeElapsedCntr.Increase;
end;

procedure TMainForm.Abouty1Click(Sender: TObject);
begin
  { Show the About Box}
  AboutForm := TAboutForm.Create(Self);
  AboutForm.ShowModal;
  AboutForm.Free;
end;

procedure TMainForm.ScoreBoard1Click(Sender: TObject);
begin
  { Show the High Scores table }
  ScoreBoardForm := TScoreBoardForm.Create(Self);
  ScoreBoardForm.ShowModal;
  ScoreBoardForm.Free;
end;

procedure TMainForm.FormDestroy(Sender: TObject);
begin
  { Save data to the registry }
  SaveHighScores;
end;

end.
