{******************************************************}
{                                              10.08.98}
{  MathCtrl unit ( version 1.8 )                       }
{  Copyright (c) 1996, 1998  Dimak                     }
{                                                      }
{******************************************************}

unit MathCtrl;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Forms, Graphics,
  Controls, StdCtrls, Menus, ClipBrd, DsgnIntf, Dialogs;

type
  TMathGridOption = ( mgBorder, mgInstantRepaint, mgMemoryDraw, mgWeb );
  TMathGridOptions = set of TMathGridOption;

type
  TMathGrid = class(TGraphicControl)
  private
    FDecimals: Word;            { Decimal part }
    FLegend: string;            { Legend       }
    FMaxX, FMaxY,
    FMinX, FMinY:  Extended;    { Field        }
    FOptions: TMathGridOptions; { Options      }
    FPartsX, FPartsY: Word;     { Parts        }
    FSizeX, FSizeY: Extended;   { Field        }
    FLogScaleX,
    FLogScaleY: Boolean;        { Logaritmic Scale }
    FBaseX, FBaseY: Extended;   { Base of log scale }
    FOnResize: TNotifyEvent;
    { Unaccessible ( internal use only ) }
    MashtX, MashtY : Extended;  { Mashtabs }
    Bitmap: TBitmap;            { Savings  }
    BitPos: TPoint;             { Position of Bitmap }
    NumSize: TPoint;            { Numbers dimensions }
    { Access & adjust methods }
    {$IFNDEF VER100}
    function  GetGridColor: TColor;
    procedure SetGridColor(NewValue: TColor);
    {$ENDIF}
    function  GetPixels: TPoint;
    procedure SetDecimals(NewValue: Word);
    procedure SetLegend(NewValue: string);
    procedure SetField(Index: Integer; NewValue: Extended);
    procedure SetLogScale(Index: Integer; NewValue: Boolean);
    procedure SetOptions(NewValue: TMathGridOptions);
    procedure SetParts(Index: Integer; NewValue: Word);
    procedure Adjust;
    procedure AdjustBitmap;
    function  CreateImage(Colored: Boolean): TBitmap;
    procedure PaintTo(ACanvas: TCanvas; Colored: Boolean);
    procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
    {$IFDEF VER100}
    procedure CMColorChanged(var Message: TMessage); message CM_COLORCHANGED;
    {$ENDIF}
    function  ToLog(Value, Base: Extended): Extended;
    function  FromLog(Value, Base: Extended): Extended;
  protected
    procedure Paint; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
    { New Methods }
    procedure Clear;
    procedure CopyToClipboard(Colored: Boolean);
    procedure Line(AX, AY, BX, BY: Extended; AC: TColor);
    procedure PointAtPos(APos: TPoint; var X, Y: Extended);
    function  PosAtPoint(AX, AY: Extended): TPoint;
    procedure PutPixel(AX, AY: Extended; AC: TColor);
    procedure SaveToFile(const FileName: String; Colored: Boolean);
    procedure SetLimits(AMinX, AMinY, AMaxX, AMaxY: Extended);
    property  Canvas;
    property  MapSize: TPoint read GetPixels;
  published
    { Properties }
    property Align;
    property Color;
    property Decimals: Word read FDecimals write SetDecimals default 2;
    property Font;
    {$IFNDEF VER100}
    property GridColor: TColor read GetGridColor write SetGridColor default clSilver;
    {$ENDIF}
    property Legend: string read FLegend write SetLegend;
    property MaxX: Extended index 1 read FMaxX write SetField stored False;
    property MaxY: Extended index 2 read FMaxY write SetField stored False;
    property MinX: Extended index 3 read FMinX write SetField;
    property MinY: Extended index 4 read FMinY write SetField;
    property Options: TMathGridOptions read FOptions write SetOptions default [];
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property PartsX: Word index 1 read FPartsX write SetParts;
    property PartsY: Word index 2 read FPartsY write SetParts;
    property PopupMenu;
    property SizeX: Extended index 5 read FSizeX write SetField;
    property SizeY: Extended index 6 read FSizeY write SetField;
    property LogScaleX: boolean index 1 read FLogScaleX write SetLogScale default false;
    property LogScaleY: boolean index 2 read FLogScaleY write SetLogScale default false;
    property LogBaseX: extended index 7 read FBaseX write SetField stored False;
    property LogBaseY: extended index 8 read FBaseY write SetField stored False;
    property ShowHint;
    property Visible;
    { Events }
    property OnResize: TNotifyEvent read FOnResize write FOnResize;
    property OnClick;
    property OnDblClick;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

  TMathCurve = class(TComponent)
  private
    { Accessible }
    FGrid: TMathGrid;
    FColor: TColor;
    FUsed: Boolean;
    { Unaccessible }
    PrevX, PrevY: Extended;
    procedure SetGrid(NewValue: TMathGrid);
  protected
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  public
    constructor Create(AOwner: TComponent); override;
    procedure PutPixel(AX, AY: Extended);
    procedure Reset;
    property Used: Boolean read FUsed write FUsed;
  published
    property Color: TColor read FColor write FColor default clRed;
    property Grid: TMathGrid read FGrid write SetGrid;
  end;

  TIntEdit = class(TCustomEdit)
  private
    FMinValue: LongInt;
    FMaxValue: LongInt;
    FDefaultValue: LongInt;
    FIncrement: LongInt;
    function GetValue: LongInt;
    function CheckValue (NewValue: LongInt): LongInt;
    procedure SetValue (NewValue: LongInt);
    procedure SetDefaultValue (NewValue: LongInt);
    procedure SetMaxValue(NewValue: LongInt);
    procedure SetMinValue(NewValue: LongInt);
    procedure CMExit(var Message: TCMExit);   message CM_EXIT;
  protected
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyPress(var Key: Char); override;
  public
    constructor Create(AOwner: TComponent); override;
  published
    property AutoSelect;
    property AutoSize;
    property Color;
    property Ctl3D;
    property DefaultValue: LongInt read FDefaultValue write SetDefaultValue;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property Increment: LongInt read FIncrement write FIncrement;
    property MaxLength;
    property MaxValue: LongInt read FMaxValue write SetMaxValue;
    property MinValue: LongInt read FMinValue write SetMinValue;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ReadOnly;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Value: LongInt read GetValue write SetValue;
    property Visible;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

  TFloatEdit = class(TCustomEdit)
  private
    FMinValue: Extended;
    FMaxValue: Extended;
    FDefaultValue: Extended;
    FIncrement: Extended;
    FIncludeBounds: Boolean;
    function GetValue: Extended;
    function CheckValue (NewValue: Extended): Extended;
    procedure SetValue (NewValue: Extended);
    procedure SetDefaultValue(NewValue: Extended);
    procedure SetMaxValue(NewValue: Extended);
    procedure SetMinValue(NewValue: Extended);
    procedure SetIncludeBounds(NewValue: Boolean);
    procedure CMExit(var Message: TCMExit);   message CM_EXIT;
  protected
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyPress(var Key: Char); override;
  public
    constructor Create(AOwner: TComponent); override;
  published
    property AutoSelect;
    property AutoSize;
    property Color;
    property Ctl3D;
    property DefaultValue: Extended read FDefaultValue write SetDefaultValue;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property IncludeBounds: Boolean read FIncludeBounds write SetIncludeBounds;
    property Increment: Extended read FIncrement write FIncrement;
    property MaxLength;
    property MaxValue: Extended read FMaxValue write SetMaxValue;
    property MinValue: Extended read FMinValue write SetMinValue;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ReadOnly;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Value: Extended read GetValue write SetValue;
    property Visible;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Math', [TMathGrid, TMathCurve, TIntEdit, TFloatEdit]);
end;

const
  Spacer = 4;
  MinMapSize = 3;
  MaxDec = 10;
  MaxParts = 20;
  sLogBaseError = 'Log Base must be greater than one';
  sLogMinError = 'Minimun must be greater than zero when LogScale is activated';
  sSizeError = 'Size must be greater than zero';
  sDefValueError = 'DefaultValue must be between MinValue and MaxValue';

{ TMathCurve }
{------------------------------------------------------}
constructor TMathCurve.Create;
{------------------------------------------------------}
begin
  inherited Create(AOwner);
  Used := False;
  FColor := clRed;
  FGrid := nil;
end;

{------------------------------------------------------}
procedure TMathCurve.Reset;
{------------------------------------------------------}
begin
  FUsed := False;
end;

{ Pixel }
{------------------------------------------------------}
procedure TMathCurve.PutPixel(AX, AY: Extended);
{------------------------------------------------------}
begin
  if Assigned(FGrid)
  then begin
    if Used
      then FGrid.Line(PrevX, PrevY, AX, AY, FColor)
      else FGrid.PutPixel(AX, AY, FColor);
    Used := True;
    PrevX := AX;
    PrevY := AY;
  end;
end;

{------------------------------------------------------}
procedure TMathCurve.SetGrid(NewValue: TMathGrid);
{------------------------------------------------------}
begin
  FGrid := NewValue;
  FUsed := False;
end;

{------------------------------------------------------}
procedure TMathCurve.Notification(AComponent: TComponent; Operation: TOperation);
{------------------------------------------------------}
begin
  if ( Operation = opRemove ) and ( AComponent = FGrid )
    then FGrid := nil;
end;

{ TMathGrid }
{------------------------------------------------------}
constructor TMathGrid.Create(AOwner: TComponent);
{------------------------------------------------------}
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle + [csOpaque];
  Bitmap := TBitmap.Create;
  {$IFDEF VER100}
  inherited Color := clSilver;
  Bitmap.Transparent := True;
  Bitmap.TransparentColor := clSilver;
  {$ELSE}
  Color := clSilver;
  {$ENDIF}
  Bitmap.Canvas.Brush.Color := clSilver;
  FDecimals := 2;
  FLegend := '';
  FMaxX := 1;
  FMaxY := 1;
  FMinX := 0;
  FMinY := 0;
  FLogScaleX := False;
  FLogScaleY := False;
  FBaseX := 10;
  FBaseY := 10;
  FOptions := [mgBorder];
  FPartsX := 0;
  FPartsY := 0;
  FSizeX := FMaxX - FMinX;
  FSizeY := FMaxY - FMinY;
  NumSize := Point(0, 0);
  BitPos := Point(Spacer, Spacer);
  SetBounds(0, 0, 200, 100);
end;

{------------------------------------------------------}
destructor TMathGrid.Destroy;
{------------------------------------------------------}
begin
  Bitmap.Free;
  inherited Destroy;
end;

{ Pixel }
{------------------------------------------------------}
procedure TMathGrid.PutPixel(AX, AY: Extended; AC: TColor);
{------------------------------------------------------}
var
  tX, tY: Integer;
begin
  if ( AX >= FMinX ) and ( AX <= FMaxX ) and
     ( AY >= FMinY ) and ( AY <= FMaxY )
  then begin
    if FLogScaleX then
     tX := Round(ToLog((AX - FMinX), FBaseX) * MashtX)
    else
     tX := Round((AX - FMinx) * MashtX);
    if FLogScaleY then
     tY := Bitmap.Height - 1 - Round(tolog((AY - FMinY),FBaseY) * MashtY)
    else
     tY := Bitmap.Height - 1 - Round((AY - FMinY) * MashtY);
    Bitmap.Canvas.Pixels[tX, tY] := AC;
    if not (mgMemoryDraw in FOptions)
      then Canvas.Pixels[tX + BitPos.X, tY + BitPos.Y] := AC;
  end;
end;

{ Line }
{------------------------------------------------------}
procedure TMathGrid.Line(AX, AY, BX, BY: Extended; AC: TColor);
{------------------------------------------------------}
var
  tX, tY, tX2, tY2: Integer;
  Rgn: HRgn;
begin
  if not ((AX < MinX) and (BX < MinX)) or ((AX > MaxX) and (BX > MaxX))
     or  ((AY < MinY) and (BY < MinY)) or ((AY > MaxY) and (BY > MaxY))
  then begin
    if not (mgMemoryDraw in FOptions)
      then begin
      { Set clipping region }
      Rgn := CreateRectRgn(Left + BitPos.X, Top + BitPos.Y,
                           Left + Bitmap.Width + BitPos.X, Top + Bitmap.Height + BitPos.Y);
      if Rgn <> 0
      then SelectClipRgn(Canvas.Handle, Rgn);
      DeleteObject(RGN);
    end;
    { Draw }
    if FLogScaleX then
     tX := Round(ToLog((AX - FMinX), FBaseX) * MashtX)
    else
     tX := Round((AX - FMinX) * MashtX);
    if FLogScaleY then
     tY := Bitmap.Height - 1 - Round(tolog((AY - FMinY), FBaseY) * MashtY)
    else
     tY := Bitmap.Height - 1 - Round((AY - FMinY) * MashtY);
    Bitmap.Canvas.MoveTo(tX, tY);
    if not (mgMemoryDraw in FOptions)
      then Canvas.MoveTo(tX + BitPos.X, tY + BitPos.Y);
    if FLogScaleX then
     tX2 := Round(tolog((BX - FMinX), FBaseX) * MashtX)
    else
     tX2 := Round((BX - FMinX) * MashtX);
    if FLogScaleY then
     tY2 := Bitmap.Height - 1 - Round(tolog((BY - FMinY), FBaseY) * MashtY)
    else
     tY2 := Bitmap.Height - 1 - Round((BY - FMinY) * MashtY);
    if (tX2 = tX) and (tY2 = tY)
      then begin
        Bitmap.Canvas.Pixels[tX2, tY2] := AC;
        if not (mgMemoryDraw in FOptions)
          then Canvas.Pixels[tX2 + BitPos.X, tY2 + BitPos.Y] := AC;
      end
      else begin
        Bitmap.Canvas.Pen.Color := AC;
        Bitmap.Canvas.LineTo(tX2, tY2);
        if not (mgMemoryDraw in FOptions)
          then begin
            Canvas.Pen.Color := AC;
            Canvas.LineTo(tX2 + BitPos.X, tY2 + BitPos.Y);
          end;
      end;
  end;
end;

{ Clearing }
{------------------------------------------------------}
procedure TMathGrid.Clear;
{------------------------------------------------------}
begin
  with Bitmap.Canvas do
    FillRect(Rect(0, 0, Width, Height));
  if mgInstantRepaint in Options
    then Refresh
    else Invalidate;
end;

{------------------------------------------------------}
function  TMathGrid.PosAtPoint(AX, AY: Extended): TPoint;
{------------------------------------------------------}
var
  tX, tY: Integer;
begin
  if ( AX >= FMinX ) and ( AX <= FMaxX ) and
     ( AY >= FMinY ) and ( AY <= FMaxY )
  then begin
    if FLogScaleX then
     Result.X := Round(ToLog((AX - FMinX), FBaseX) * MashtX)
    else
     Result.X := Round((AX - FMinx) * MashtX);
    if FLogScaleY then
     Result.Y := Bitmap.Height - 1 - Round(tolog((AY - FMinY),FBaseY) * MashtY)
    else
     Result.Y := Bitmap.Height - 1 - Round((AY - FMinY) * MashtY);
    Result.X := Result.X + BitPos.X;
    Result.Y := Result.Y + BitPos.Y;
  end
  else Result := Point(-1, -1);
end;

{------------------------------------------------------}
procedure TMathGrid.PointAtPos(APos: TPoint; var X, Y: Extended);
{------------------------------------------------------}
begin
  if APos.X < BitPos.X
    then APos.X := BitPos.X;
  if APos.Y < BitPos.Y
    then APos.Y := BitPos.Y;
  if APos.X > BitPos.X + Bitmap.Width - 1
    then APos.X := BitPos.X + Bitmap.Width - 1;
  if APos.Y > BitPos.Y + Bitmap.Height - 1
    then APos.Y := BitPos.Y + Bitmap.Height - 1;
  if MashtX = 0
    then X := FMinX
    else if FLogScaleX
           then X := FromLog(ToLog(FMinX, FBaseX) + (APos.X - BitPos.X)/MashtX, FBaseX)
           else X := FMinX + (APos.X - BitPos.X)/MashtX;
  if MashtY = 0
    then Y := FMinY
    else if FLogScaleY
           then Y := FromLog(ToLog(FMinY, FBaseY) - (APos.Y - BitPos.Y - Bitmap.Height + 1)/MashtY, FBaseY)
           else Y := FMinY - (APos.Y - BitPos.Y - Bitmap.Height + 1)/MashtY;
end;

{------------------------------------------------------}
function TMathGrid.CreateImage;
{------------------------------------------------------}
begin
  Result := TBitmap.Create;
  Result.Monochrome := not Colored;
  Result.Width := ClientRect.Right;
  Result.Height := ClientRect.Bottom;
  PaintTo(Result.Canvas, Colored);
end;

{ Copying to ClipBoard }
{------------------------------------------------------}
procedure TMathGrid.CopyToClipboard;
{------------------------------------------------------}
var
  TempCur: TCursor;
  TempBMP: TBitmap;
begin
  TempCur := Screen.Cursor;
  Screen.Cursor := crHourGlass;
  TempBMP := CreateImage(Colored);
  try
    ClipBoard.Assign(TempBMP);
  finally
    TempBMP.Free;
    Screen.Cursor := TempCur;
  end;
end;

{ Saving as Windows Bitmap file }
{------------------------------------------------------}
procedure TMathGrid.SaveToFile;
{------------------------------------------------------}
var
  TempCur: TCursor;
  TempBMP: TBitmap;
begin
  TempCur := Screen.Cursor;
  Screen.Cursor := crHourGlass;
  TempBMP := CreateImage(Colored);
  try
    TempBMP.SaveToFile(FileName);
  finally
    TempBMP.Free;
    Screen.Cursor := TempCur;
  end;
end;

{ Drawing }
{------------------------------------------------------}
procedure TMathGrid.PaintTo(ACanvas: TCanvas; Colored: Boolean);
{------------------------------------------------------}
var
  GridPos: TRect;
  i, w, h, SkipY: Integer;
  Fmt: string[12];
begin
  GridPos := Rect(BitPos.X, BitPos.Y, Bitmap.Width + BitPos.X - 1, Bitmap.Height + BitPos.Y - 1);
  with ACanvas do
  begin
    { Background }
    if Colored then Brush.Color := Color
      else Brush.Color := clWhite;
    FillRect(ClientRect);
    if mgBorder in FOptions
    then begin
      { Frames }
      Pen.Mode := pmCopy;
      Pen.Style := psSolid;
      if Colored
        then begin
          Pen.Color := clWhite;
          PolyLine([Point(1, Height-1), Point(Width-1, Height-1), Point(Width-1, 0)]);
          PolyLine([Point(GridPos.Left-1, GridPos.Bottom+1),
                    Point(GridPos.Left-1, GridPos.Top-1),
                    Point(GridPos.Right+1, GridPos.Top-1)]);
          Pen.Color := clGray;
          PolyLine([Point(0, Height-1), Point(0, 0), Point(Width-1, 0)]);
          PolyLine([Point(GridPos.Left, GridPos.Bottom+1),
                    Point(GridPos.Right+1, GridPos.Bottom+1),
                    Point(GridPos.Right+1, GridPos.Top)]);
        end
        else begin
          Pen.Color := clBlack;
          Rectangle(0, 0, Width - 1, Height - 1);
          Rectangle(GridPos.Left - 1, GridPos.Top - 1, GridPos.Right + 1, GridPos.Bottom + 1);
        end;
    end;
    { Digits }
    Font.Assign(Self.Font);
    if not Colored then
      Font.Color := clBlack;
    Fmt := '0';
    if FDecimals > 0 then Fmt := Fmt + '.';
    for i := 1 to FDecimals do Fmt := Fmt + '0';
    SkipY := Spacer;
    if PartsY > 0
    then begin
      SkipY := NumSize.Y div 2;
      for i := 0 to FPartsY do
        if FLogScaleY then
         TextOut(Spacer, Spacer + ((GridPos.Bottom - GridPos.Top) div FPartsY)*i,
           FormatFloat(Fmt,
             FromLog(ToLog(FMinY, FBaseY)+(FPartsY-i)*(ToLog(FMaxY, FBaseY)-ToLog(FMinY, FBaseY))/FPartsY, FBaseY)))
        else
         TextOut(Spacer, Spacer + ((GridPos.Bottom - GridPos.Top) div FPartsY)*i,
           FormatFloat(Fmt, FMinY + (FPartsY-i)*FSizeY/FPartsY));
    end;
    if PartsX > 0 then
      for i := 0 to FPartsX do
        if FLogScaleX then
         TextOut(GridPos.Left - NumSize.X div 2 + ((GridPos.Right - GridPos.Left) div FPartsX)*i,
                 GridPos.Bottom + SkipY,
           FormatFloat(Fmt,
             FromLog(ToLog(FMinX, FBaseX)+i*(ToLog(FMaxX, FBaseX)-ToLog(FMinX, FBaseX))/FPartsX, FBaseX)))
        else
         TextOut(GridPos.Left - NumSize.X div 2 + ((GridPos.Right - GridPos.Left) div FPartsX)*i,
                 GridPos.Bottom + SkipY,
           FormatFloat(Fmt, FMinX + i*FSizeX/FPartsX));
    { Legend }
    if FLegend <> ''
    then begin
      if Colored then Pen.Color := clWhite else Pen.Color := clBlack;
      Rectangle(Spacer, Height - NumSize.Y - 2*Spacer - 2,
                Width - Spacer, Height - NumSize.Y - 2*Spacer - 1);
      Pen.Color := clGray;
      Rectangle(Spacer, Height - NumSize.Y - 2*Spacer - 1,
                Width - Spacer, Height - NumSize.Y - 2*Spacer);
      TextOut(Spacer, Height - NumSize.Y - Spacer, FLegend);
    end;
    { Bitmap }
    if Colored
      then Draw(BitPos.X, BitPos.Y, Bitmap)
      else
        for h := 0 to Bitmap.Height - 1 do
          for w := 0 to Bitmap.Width - 1 do
            if Bitmap.Canvas.Pixels[w, h] <> {$IFDEF VER100}Color{$ELSE}GridColor{$ENDIF}
              then Pixels[BitPos.X + w, BitPos.Y + h] := clBlack;
    { Web }
    if mgWeb in FOptions
    then begin
      Pen.Style := psDot;
      if Colored then Pen.Color := clWhite else Pen.Color := clBlack;
      for i := 1 to FPartsY - 1 do
        PolyLine([Point(GridPos.Left + 1, GridPos.Top + ((GridPos.Bottom - GridPos.Top) div FPartsY)*i),
          Point(GridPos.Right - 1 , GridPos.Top + ((GridPos.Bottom - GridPos.Top) div FPartsY)*i)]);
      for i := 1 to FPartsX - 1 do
        PolyLine([Point(GridPos.Left + ((GridPos.Right - GridPos.Left) div FPartsX)*i, GridPos.Top + 1),
          Point(GridPos.Left + ((GridPos.Right - GridPos.Left) div FPartsX)*i, GridPos.Bottom - 1)]);
      Pen.Style := psSolid;
    end;
  end;
end;

{------------------------------------------------------}
procedure TMathGrid.Paint;
{------------------------------------------------------}
begin
  PaintTo(Canvas, True);
end;

{ Size changing }
{------------------------------------------------------}
procedure TMathGrid.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
{------------------------------------------------------}
begin
  if (Left <> ALeft) or (Top <> ATop) or (Width <> AWidth) or (Height <> AHeight)
  then begin
    inherited SetBounds (ALeft, ATop, AWidth, AHeight);
    AdjustBitmap;
  end;
end;

{ Field }
{------------------------------------------------------}
procedure TMathGrid.SetField;
{------------------------------------------------------}
begin
  case Index of
    { MaxX }
    1: if FMaxX <> NewValue
       then SetField(5, NewValue - FMinX);
    { MaxY }
    2: if FMaxY <> NewValue
       then SetField(6, NewValue - FMinY);
    { MinX }
    3: if FMinX <> NewValue
       then if not (LogScaleX and (NewValue <= 0)) then
        begin
         FMinX := NewValue;
         FMaxX := FMinX + FSizeX;
         Adjust;
        end
       else raise EPropertyError.Create(sLogMinError);
    { MinY }
    4: if FMinY <> NewValue
       then if not (LogScaleY and (NewValue <= 0)) then
        begin
         FMinY := NewValue;
         FMaxY := FMinY + FSizeY;
         Adjust;
        end
       else raise EPropertyError.Create(sLogMinError);
    { SizeX }
    5: if FSizeX <> NewValue
       then if NewValue > 0
            then begin
              FSizeX := NewValue;
              FMaxX := FMinX + FSizeX;
              Adjust;
            end
            else raise EPropertyError.Create(sSizeError);
    { SizeY }
    6: if FSizeY <> NewValue
       then if NewValue > 0
            then begin
              FSizeY := NewValue;
              FMaxY := FMinY + FSizeY;
              Adjust;
        end
        else raise EPropertyError.Create(sSizeError);
    { LogBaseX }
    7: if FBaseX <> NewValue
       then if NewValue > 1
            then begin
             FBaseX := NewValue;
             Adjust;
            end
            else raise EPropertyError.Create(sLogBaseError);
    { LogBaseY }
    8: if FBaseY <> NewValue
       then if (NewValue > 1)
            then begin
             FBaseY := NewValue;
             Adjust;
            end
            else raise EPropertyError.Create(sLogBaseError);
  end;
end;

{ LogScale }
{------------------------------------------------------}
procedure TMathGrid.SetLogScale;
{------------------------------------------------------}
begin
 case Index of
  { LogScaleX }
  1: if FLogScaleX <> NewValue then
      if not (NewValue and (FMinX <= 0)) then
       begin
        FLogScaleX := NewValue;
        Adjust;
       end
      else raise EPropertyError.create(sLogMinError);
  { LogScaleY }
  2: if FLogScaleY <> NewValue then
      if not (NewValue and (FMinY <= 0)) then
       begin
        FLogScaleY := NewValue;
        Adjust;
       end
      else raise EPropertyError.create(sLogMinError);
 end;
end;

{ Pixels }
{------------------------------------------------------}
function TMathGrid.GetPixels: TPoint;
{------------------------------------------------------}
begin
  Result.X := Bitmap.Width;
  Result.Y := Bitmap.Height;
end;

{ Parts }
{------------------------------------------------------}
procedure TMathGrid.SetParts(Index: Integer; NewValue: Word);
{------------------------------------------------------}
begin
  if NewValue > MaxParts then NewValue := MaxParts;
  if (Index = 1) and (FPartsX <> NewValue)
    then begin
      FPartsX := NewValue;
      Adjust;
    end;
  if (Index = 2) and (FPartsY <> NewValue)
    then begin
      FPartsY := NewValue;
      Adjust;
    end;
end;

{ Legend }
{------------------------------------------------------}
procedure TMathGrid.SetLegend(NewValue: string);
{------------------------------------------------------}
var
  Temp: Boolean;
begin
  if FLegend <> NewValue
  then begin
    Temp := (FLegend = '') or (NewValue = '');
    FLegend := NewValue;
    if Temp
      then Adjust
      else Invalidate;
  end;
end;

{ Options }
{------------------------------------------------------}
procedure TMathGrid.SetOptions(NewValue: TMathGridOptions);
{------------------------------------------------------}
begin
  if NewValue <> FOptions
  then begin
    FOptions := NewValue;
    if mgInstantRepaint in FOptions
      then Refresh
      else Invalidate;
  end;
end;

{ Decimals }
{------------------------------------------------------}
procedure TMathGrid.SetDecimals(NewValue: Word);
{------------------------------------------------------}
begin
  if NewValue <> FDecimals
  then begin
    FDecimals := NewValue;
    if FDecimals > MaxDec then FDecimals := MaxDec;
    Adjust;
  end;
end;

{------------------------------------------------------}
function TMathGrid.ToLog;
{------------------------------------------------------}
begin
  if Value > 0 then
   Result := ln(Value)/ln(Base)
  else
   Result := 0;
end;

{------------------------------------------------------}
function TMathGrid.FromLog;
{------------------------------------------------------}
begin
  Result := exp(ln(Base) * Value);
end;

{ Font }
{------------------------------------------------------}
procedure TMathGrid.CMFontChanged(var Message: TMessage);
{------------------------------------------------------}
begin
  Adjust;
end;

{$IFDEF VER100}
{------------------------------------------------------}
procedure TMathGrid.CMColorChanged(var Message: TMessage);
{------------------------------------------------------}
begin
  if Bitmap <> nil
  then begin
    Bitmap.Canvas.Brush.Color := Color;
    Bitmap.TransparentColor := Color;
    Clear;
  end;
end;
{$ELSE}
{ GridColor }
{------------------------------------------------------}
procedure TMathGrid.SetGridColor(NewValue: TColor);
{------------------------------------------------------}
begin
  if NewValue <> Bitmap.Canvas.Brush.Color
  then begin
    Bitmap.Canvas.Brush.Color := NewValue;
    Clear;
  end;
end;

{------------------------------------------------------}
function TMathGrid.GetGridColor: TColor;
{------------------------------------------------------}
begin
  GetGridColor := Bitmap.Canvas.Brush.Color;
end;
{$ENDIF}

{ Field }
{------------------------------------------------------}
procedure TMathGrid.SetLimits(AMinX, AMinY, AMaxX, AMaxY: Extended);
{------------------------------------------------------}
begin
 if (LogScaleX and (AMinX <= 0)) or
    (LogScaleY and (AMinY <= 0)) then
  raise EPropertyError.create(sLogMinError)
  else if (AMaxX - AMinX > 0) and (AMaxY - AMinY > 0)
    then begin
      if (FMinX <> AMinX) or (FMaxX <> AMaxX) or (FMinY <> AMinY) or (FMaxY <> AMaxY)
      then begin
        FMinX := AMinX;
        FMaxX := AMaxX;
        FMinY := AMinY;
        FMaxY := AMaxY;
        FSizeX := AMaxX - AMinX;
        FSizeY := AMaxY - AMinY;
        Adjust;
      end;
    end
    else raise EPropertyError.Create(sSizeError);
end;

{ Adjustment }
{------------------------------------------------------}
procedure TMathGrid.Adjust;
{------------------------------------------------------}
var
  Temp: Integer;
  Fmt: string[12];
  i: Byte;
begin
  Canvas.Font.Assign(Self.Font);
  Fmt := '0';
  if FDecimals > 0 then Fmt := Fmt + '.';
  for i := 1 to FDecimals do
    Fmt := Fmt + '0';
  Temp := Canvas.TextWidth(FormatFloat(Fmt, FMinY + FSizeY));
  NumSize.X := Canvas.TextWidth(FormatFloat(Fmt, FMinY));
  if Temp > NumSize.X then NumSize.X := Temp;
  Temp := Canvas.TextWidth(FormatFloat(Fmt, FMinX + FSizeX));
  if Temp > NumSize.X then NumSize.X := Temp;
  NumSize.Y := Canvas.TextHeight('0');
  AdjustBitmap;
end;

{------------------------------------------------------}
procedure TMathGrid.AdjustBitmap;
{------------------------------------------------------}
var
  NewBitPos: TRect;
begin
  if [csLoading, csDesigning] * ComponentState = []
    then try
      if Assigned(FOnResize) then FOnResize(Self);
    except
      exit;
    end;
  NewBitPos := Rect(Spacer, Spacer, Width - Spacer - 1, Height - Spacer - 1);
  with NewBitPos do
  begin
    if PartsY > 0
    then begin
      if PartsX > 0
      then begin
        Inc(Left, Spacer + NumSize.X);
        Inc(Top, NumSize.Y div 2);
        Dec(Bottom, NumSize.Y + NumSize.Y div 2);
        Dec(Right, NumSize.X div 2);
      end
      else begin
        Inc(Left, Spacer + NumSize.X);
        Inc(Top, NumSize.Y div 2);
        Dec(Bottom, NumSize.Y div 2);
      end
    end
    else
      if PartsX > 0
      then begin
        Inc(Left, NumSize.X div 2);
        Dec(Bottom, Spacer + NumSize.Y);
        Dec(Right, NumSize.X div 2);
      end;
    if FLegend <> ''
      then Dec(Bottom, NumSize.Y + 2*Spacer + 2);

    if Right - Left < MinMapSize
    then begin
      Bitmap.Width := MinMapSize;
      inherited Width := Width - (Right - Left) + MinMapSize;
    end
    else Bitmap.Width := Right - Left;
    BitPos.X := Left;

    if Bottom - Top < MinMapSize
    then begin
      Bitmap.Height := MinMapSize;
      inherited Height := Height - (Bottom - Top) + MinMapSize;
    end
    else Bitmap.Height := Bottom - Top;
    BitPos.Y := Top;
  end;
  if FLogScaleX then
   MashtX := ( Bitmap.Width - 1 ) / ToLog(FMaxX/FMinX, FBaseX)
  else
   MashtX := ( Bitmap.Width - 1 ) / FSizeX;
  if FLogScaleY then
   MashtY := ( Bitmap.Height - 1 ) / ToLog(FMaxY/FMinY, FBaseY)
  else
   MashtY := ( Bitmap.Height - 1 ) / FSizeY;
  Clear;
end;

{ TIntEdit }
{------------------------------------------------------}
constructor TIntEdit.Create(AOwner: TComponent);
{------------------------------------------------------}
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle - [csSetCaption];
  FMaxValue := 0;
  FMinValue := 0;
  FDefaultValue := 0;
  Text := '0';
end;

{------------------------------------------------------}
procedure TIntEdit.KeyDown(var Key: Word; Shift: TShiftState);
{------------------------------------------------------}
begin
  if Key = VK_UP then Value := Value + FIncrement
    else if Key = VK_DOWN then Value := Value - FIncrement
           else inherited KeyDown(Key, Shift);
end;

{------------------------------------------------------}
procedure TIntEdit.KeyPress(var Key: Char);
{------------------------------------------------------}
begin
  if not ( Key in ['+', '-', '0'..'9', #0..#31] ) then
  begin
    Key := #0;
    MessageBeep($FFFF);
  end
  else inherited KeyPress(Key);
end;

{------------------------------------------------------}
function TIntEdit.GetValue: LongInt;
{------------------------------------------------------}
begin
  if Text = ''
    then GetValue := FDefaultValue
    else try
      GetValue := StrToInt(Text);
    except
      on E: Exception do
      begin
        if not ( csDesigning in ComponentState ) then SetFocus;
        raise EPropertyError.Create(E.Message);
      end;
    end;
end;

{------------------------------------------------------}
procedure TIntEdit.SetValue(NewValue: LongInt);
{------------------------------------------------------}
begin
  Text := IntToStr(CheckValue(NewValue));
end;

{------------------------------------------------------}
procedure TIntEdit.SetDefaultValue (NewValue: LongInt);
{------------------------------------------------------}
begin
  if NewValue <> FDefaultValue
  then if NewValue = CheckValue(NewValue)
       then FDefaultValue := NewValue
       else raise EPropertyError.Create(sDefValueError);
end;

{------------------------------------------------------}
procedure TIntEdit.SetMaxValue (NewValue: LongInt);
{------------------------------------------------------}
begin
  if FMaxValue <> NewValue
  then begin
    FMaxValue := NewValue;
    Text := IntToStr(CheckValue(Value));
    FDefaultValue := CheckValue(FDefaultValue);
  end;
end;

{------------------------------------------------------}
procedure TIntEdit.SetMinValue (NewValue: LongInt);
{------------------------------------------------------}
begin
  if FMinValue <> NewValue
  then begin
    FMinValue := NewValue;
    Text := IntToStr(CheckValue(Value));
    FDefaultValue := CheckValue(FDefaultValue);
  end;
end;

{------------------------------------------------------}
function TIntEdit.CheckValue (NewValue: LongInt): LongInt;
{------------------------------------------------------}
begin
  Result := NewValue;
  if (FMaxValue <> FMinValue) then
  begin
    if NewValue < FMinValue then
      Result := FMinValue
    else if NewValue > FMaxValue then
      Result := FMaxValue;
  end;
end;

{------------------------------------------------------}
procedure TIntEdit.CMExit(var Message: TCMExit);
{------------------------------------------------------}
begin
  inherited;
  if Text <> ''
  then try
    SetValue(CheckValue(Value));
  except
    on E: Exception do
    begin
      SetFocus;
      raise EPropertyError.Create(E.Message);
    end;
  end;
end;

{ TFloatEdit }
{------------------------------------------------------}
constructor TFloatEdit.Create(AOwner: TComponent);
{------------------------------------------------------}
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle - [csSetCaption];
  FMaxValue := 0;
  FMinValue := 0;
  FDefaultValue := 0;
  FIncludeBounds := True;
  Text := '0';
end;

{------------------------------------------------------}
procedure TFloatEdit.KeyDown(var Key: Word; Shift: TShiftState);
{------------------------------------------------------}
begin
  if Key = VK_UP then Value := Value + FIncrement
    else if Key = VK_DOWN then Value := Value - FIncrement
           else inherited KeyDown(Key, Shift);
end;

{------------------------------------------------------}
procedure TFloatEdit.KeyPress(var Key: Char);
{------------------------------------------------------}
begin
  if not ( Key in [DecimalSeparator, '+', '-', '0'..'9', #0..#31] ) then
  begin
    Key := #0;
    MessageBeep($FFFF)
  end
  else inherited KeyPress(Key);
end;

{------------------------------------------------------}
function TFloatEdit.GetValue: Extended;
{------------------------------------------------------}
begin
  if Text = ''
    then GetValue := FDefaultValue
    else try
      GetValue := StrToFloat(Text);
    except
      on E: Exception do
      begin
        if not ( csDesigning in ComponentState ) then SetFocus;
        raise EPropertyError.Create(E.Message);
      end;
    end;
end;

{------------------------------------------------------}
procedure TFloatEdit.SetValue(NewValue: Extended);
{------------------------------------------------------}
begin
  Text := FloatToStr(CheckValue(NewValue));
end;

{------------------------------------------------------}
procedure TFloatEdit.SetDefaultValue(NewValue: Extended);
{------------------------------------------------------}
begin
  if NewValue <> FDefaultValue
  then if NewValue = CheckValue(NewValue)
       then FDefaultValue := NewValue
       else raise EPropertyError.Create(sDefValueError);
end;

{------------------------------------------------------}
procedure TFloatEdit.SetMaxValue (NewValue: Extended);
{------------------------------------------------------}
begin
  if FMaxValue <> NewValue
  then begin
    FMaxValue := NewValue;
    SetValue(CheckValue(Value));
    FDefaultValue := CheckValue(FDefaultValue);
  end;
end;

{------------------------------------------------------}
procedure TFloatEdit.SetMinValue (NewValue: Extended);
{------------------------------------------------------}
begin
  if FMinValue <> NewValue
  then begin
    FMinValue := NewValue;
    SetValue(CheckValue(Value));
    FDefaultValue := CheckValue(FDefaultValue);
  end;
end;

{------------------------------------------------------}
function TFloatEdit.CheckValue (NewValue: Extended): Extended;
{------------------------------------------------------}
begin
  Result := NewValue;
  if (FMaxValue <> FMinValue) then
    if FIncludeBounds
    then begin
      if NewValue < FMinValue then
        Result := FMinValue
      else if NewValue > FMaxValue then
        Result := FMaxValue;
    end
    else if ( NewValue <= FMinValue ) or ( NewValue >= FMaxValue )
         then Result := FMinValue + ( FMaxValue - FMinValue ) / 2;
end;

{------------------------------------------------------}
procedure TFloatEdit.SetIncludeBounds(NewValue: Boolean);
{------------------------------------------------------}
begin
  if NewValue <> FIncludeBounds
  then begin
    FIncludeBounds := NewValue;
    SetValue(CheckValue(Value));
  end;
end;

{------------------------------------------------------}
procedure TFloatEdit.CMExit(var Message: TCMExit);
{------------------------------------------------------}
begin
  inherited;
  if Text <> ''
  then try
    SetValue(CheckValue(Value));
  except
    on E: Exception do
    begin
    SetFocus;
      raise EPropertyError.Create(E.Message);
    end;
  end;
end;

end.
