{  Main form for Update

  Copyright (c) 1998 Gregory L. Bullock (bullock@mbay.net).
  Freeware: May be freely distributed and modified. Use at your own risk.
  This program draws heavily on

    - the aDiff unit, Copyright (c) 1997 S.Kurinny & S.Kostinsky

    - the Lh5Unit unit, which had various contributers, as noted in its
      source code

  History:
   28-10-98 Made Progress dialog get updated more frequently under 32-bits.

     3-7-98 Added a CRC for the *compressed* differences to better detect
            whether an .Upd is corrupted.

    16-5-98 Modified to compile under D3 (32-bits) as well as D1 (16-bits).

            Added a new compiler directive, MAKE16AND32COMPATIBLE, which
            will ensure that the *.Upd file is compatible independent
            of the platform (16- or 32-bit) which creates it or uses it.
            On the one test case I ran, I found that using MAKE16AND32COMPATIBLE
            increased the size of the *.Upd file.  The file sizes I got were
              29304  for the 32-bit .Upd file
              30746  for the 16-bit .Upd file
              30759  for the compatible .Upd file

            Improved one of the log messages and corrected the help file.

    23-4-98 Initial version.

  If you fix any bugs or make significant enhancements, I ask you to send
  me your modifications.

  For example, at present, these programs only allow you to make changes to
  existing files.  Update will not add a new file nor will it delete an
  existing file.  Perhaps a future version of these programs will enable
  Update to add new files or delete obsolete files.

  If you make any changes to the structure of the Update File,
  change the UpdateFileHeader in the Common Unit so older versions
  of Update won't try to read an Update File that they won't understand.
}

unit UpdateFm;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, Buttons, FileCtrl, LogForm, About,
  LH5Unit, Progress;

type
  TUpdateForm = class(TForm)
    GoBtn: TBitBtn;
    CancelBtn: TBitBtn;
    OpenDialog: TOpenDialog;
    GroupBox1: TGroupBox;
    TargetFile: TLabel;
    TargetDirectoryListBox: TDirectoryListBox;
    DriveComboBox1: TDriveComboBox;
    GroupBox2: TGroupBox;
    ChangesFile: TEdit;
    BrowseChangesFile: TButton;
    AboutBtn: TBitBtn;
    procedure ApplyUpdateFile(ChangesFileName: string);
    procedure BrowseChangesFileClick(Sender: TObject);
    procedure AboutBtnClick(Sender: TObject);
    procedure GoBtnClick(Sender: TObject);
    procedure CancelBtnClick(Sender: TObject);
    procedure FormShow(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
  end;

var
  UpdateForm: TUpdateForm;

implementation

uses
  Common, aDiff,
   {$IFDEF Win32} aCRC32; {$ELSE} aCRC3216; {$ENDIF}

{$R *.DFM}
{$IFDEF Win32}
{$R UpdRes32.RES}
{$ELSE}
{$R UpdRes16.RES}
{$ENDIF}

type
  EUpdateFileError = class(Exception);
  EDifferenceError = class(Exception);

{$I UpdateSt.Inc}

procedure TUpdateForm.BrowseChangesFileClick(Sender: TObject);
begin
  OpenDialog.FileName := ChangesFile.Text;
  if OpenDialog.Execute then
    ChangesFile.Text := AnsiLowerCase(OpenDialog.FileName);
end;

procedure TUpdateForm.AboutBtnClick(Sender: TObject);
begin
  AboutBox := TAboutBox.Create(Self);
  AboutBox.ShowModal;
  AboutBox.Free;
  AboutBox := nil;
end;

procedure TUpdateForm.GoBtnClick(Sender: TObject);
begin
  ApplyUpdateFile(ChangesFile.Text);
end;

procedure TUpdateForm.ApplyUpdateFile(ChangesFileName: string);
var
  FileHeader    : string[15];
  OldVersion,
  NewVersion    : string[31];
  NewFileName,
  OldFileName   : TFileName;
  FileInfo      : TFileUpdateInfo;
  NextUpdPosition,
  CRC,
  DiffSize      : LongInt;
{$IFDEF DebugLog}
  i : integer;
  LogStream     : TFileStream;
  LogMsg        : string;
{$ENDIF}
begin
{$IFDEF DebugLog}
    LogStream := TFileStream.Create('Update.Log',fmCreate);
    LogMsg := 'ApplyUpdateFile 1: ' + ChangesFileName + #13#10;
    LogStream.WriteBuffer(LogMsg[1], Length(LogMsg));
{$ENDIF}
  {Apply Update file}
  UpdateStream := nil;
  DiffStream := nil;
  FileInfo := nil;
  UpdAccumulateStream := nil;
  UpdLogForm.LogMessage.Lines.Clear;
  DiffSize := 0;
  NextUpdPosition := 0;
  try try
    if not FileExists(ChangesFileName) then
      raise EUpdateFileError.CreateResFmt(IDS_UpdFileNotFoundFmt,[ChangesFileName]);
    try
      UpdateStream  := TFileStream.Create(ChangesFileName,fmOpenRead);
      FileHeader := UpdateFileHeader; {This copies the length byte}
      UpdateStream.ReadBuffer(FileHeader[1],Length(UpdateFileHeader));
    except
      raise EUpdateFileError.CreateResFmt(IDS_CannotReadUpdateFileFmt,[ChangesFileName]);
    end;
    if FileHeader <> UpdateFileHeader then
      raise EUpdateFileError.CreateResFmt(IDS_InvalidUpdateFileFmt,[ChangesFileName,UpdateFileHeader,FileHeader]);
    try
      UpdateStream.ReadBuffer(OldVersion,SizeOf(OldVersion));
      UpdateStream.ReadBuffer(NewVersion,SizeOf(NewVersion));
{$IFDEF DebugLog}
      LogMsg := 'ApplyUpdateFile 2: ' + '"' + OldVersion + '", "' + NewVersion + '"'#13#10;
      LogStream.WriteBuffer(LogMsg[1], Length(LogMsg));
{$ENDIF}
    except
      raise EUpdateFileError.CreateResFmt(IDS_CannotReadUpdateFileFmt,[ChangesFileName]);
    end;

    NewFileName := GetFullPathTo('$UpdTmp$.~aa',TargetDirectoryListBox);
    while FileExists(NewFileName) do
      if NewFileName[Length(NewFileName)] <> 'z' then
        Inc(NewFileName[Length(NewFileName)])
      else begin
        Inc(NewFileName[Length(NewFileName)-1]);
        NewFileName[Length(NewFileName)] := 'a';
      end;

    UpdateStream.ReadBuffer(CRC,SizeOf(CRC));
    if CRC <> CalculateStreamCRC(UpdateStream) then
      raise EUpdateFileError.Create(LoadStr(IDS_InvalidChangesForFile));

{$IFDEF DebugLog}
    LogMsg := 'ApplyUpdateFile 3: ' + NewFileName + #13#10;
    LogStream.WriteBuffer(LogMsg[1], Length(LogMsg));
{$ENDIF}
    DiffStream := TMemoryStream.Create;
    UpdAccumulateStream := TMemoryStream.Create;
    LHAExpand(UpdateStream, UpdAccumulateStream);
    UpdAccumulateStream.Position := 0;
    FileInfo := TFileUpdateInfo.Create;
    ProgressDlg.Show;
    while UpdAccumulateStream.Read(FileInfo.Name, SaveFileUpdateInfoSize) = SaveFileUpdateInfoSize do
    begin
      try
        try
          OldStream := nil;
          NewStream := nil;
          OldFileName := GetFullPathTo(FileInfo.Name,TargetDirectoryListBox);
{$IFDEF DebugLog}
          LogMsg := 'ApplyUpdateFile 4: ' + OldFileName + #13#10;
          LogStream.WriteBuffer(LogMsg[1], Length(LogMsg));
{$ENDIF}
          if not FileExists(OldFileName) then
            raise EDifferenceError.Create(LoadStr(IDS_FileNotFound));
          try
            OldStream := TFileStream.Create(OldFileName,fmOpenRead);
            NewStream := TFileStream.Create(NewFileName,fmCreate);
          except
            raise EDifferenceError.Create(LoadStr(IDS_CannotOpenFile));
          end;
          CRC := CalculateStreamCRC(OldStream);
          if CRC = FileInfo.NewCRC then
            raise EDifferenceError.CreateResFmt(IDS_FileAlreadyUpdatedFmt,[NewVersion]);
          if CRC <> FileInfo.OldCRC then
            raise EDifferenceError.CreateResFmt(IDS_FileWrongVersionFmt,[OldVersion]);
          try
{$IFDEF DebugLog}
            LogMsg := 'ApplyUpdateFile 5: ' + IntToStr(CRC) + #13#10;
            LogStream.WriteBuffer(LogMsg[1], Length(LogMsg));
{$ENDIF}
            ProgressDlg.FileNameLabel.Caption := OldFileName;
            ProgressDlg.ProgressBar.Progress := 0;
            Application.ProcessMessages;
            Screen.Cursor := crHourglass;
            DiffStream.Clear;
            UpdAccumulateStream.ReadBuffer(DiffSize,SizeOf(DiffSize));
{$IFDEF DebugLog}
            LogMsg := 'ApplyUpdateFile 6: ' + IntToStr(DiffSize) + #13#10;
            LogStream.WriteBuffer(LogMsg[1], Length(LogMsg));
{$ENDIF}
            NextUpdPosition := UpdAccumulateStream.Position + DiffSize;
            DiffStream.CopyFrom(UpdAccumulateStream,DiffSize-SizeOf(CRC));
            UpdAccumulateStream.ReadBuffer(CRC,SizeOf(CRC));
          except
            raise EDifferenceError.Create(LoadStr(IDS_InvalidChangesForFile));
          end;
          DiffStream.Position := 0;
          if CRC <> CalculateStreamCRC(DiffStream) then
            raise EDifferenceError.Create(LoadStr(IDS_InvalidChangesForFile));
          try
            DiffStreamExtract(DiffStream,OldStream,NewStream,UpdateProgress);
          except
            raise EDifferenceError.Create(LoadStr(IDS_ErrorApplyingChanges));
          end;
          NewStream.Position := 0;
          if CalculateStreamCRC(NewStream) <> FileInfo.NewCRC then
            raise EDifferenceError.Create(LoadStr(IDS_UpdatedFileFailsCRC));
          OldStream.Free;
          NewStream.Free;
          OldStream := nil;
          NewStream := nil;
          if not SysUtils.DeleteFile(OldFileName) then
            raise EDifferenceError.Create(LoadStr(IDS_CannotDeleteOldFile));
{$IFDEF DebugLog}
          LogMsg := 'ApplyUpdateFile 7: ' + OldFileName + #13#10;
          LogStream.WriteBuffer(LogMsg[1], Length(LogMsg));
{$ENDIF}
          if not RenameFile(NewFileName,OldFileName) then
            raise EDifferenceError.Create(LoadStr(IDS_CannotRenameUpdatedFile));
          UpdLogForm.LogMessage.Lines.Add(FmtLoadStr(IDS_UpdatedFileFmt,[NewVersion,OldFileName]));
        finally
          OldStream.Free;
          NewStream.Free;
          SysUtils.DeleteFile(NewFileName);
          Screen.Cursor := crDefault;
        end;
      except
        on E: EDifferenceError do
{$IFDEF DebugLog}
          begin
            LogMsg := 'ApplyUpdateFile except: ' + E.Message + #13#10;
            LogStream.WriteBuffer(LogMsg[1], Length(LogMsg));
{$ENDIF}
          UpdLogForm.LogMessage.Lines.Add(FmtLoadStr(IDS_SkippingFileFmt,[OldFileName,E.Message]))
{$IFDEF DebugLog}
          end;
{$ENDIF}
      end;
      if NextUpdPosition = 0 then
      begin
        if DiffSize = 0 then
          UpdAccumulateStream.ReadBuffer(DiffSize,SizeOf(DiffSize));
        NextUpdPosition := UpdAccumulateStream.Position + DiffSize;
      end;
      UpdAccumulateStream.Position := NextUpdPosition;
      DiffSize := 0;
      NextUpdPosition := 0;
    end;
  except
    on E: EUpdateFileError do
{$IFDEF DebugLog}
    begin
      LogMsg := 'ApplyUpdateFile outer except: ' + E.Message + #13#10;
      LogStream.WriteBuffer(LogMsg[1], Length(LogMsg));
{$ENDIF}
      UpdLogForm.LogMessage.Lines.Add(FmtLoadStr(IDS_UpdatesNotAppliedFmt,[E.Message]));
  end;
  finally
{$IFDEF DebugLog}
    LogMsg := 'ApplyUpdateFile 8: '#13#10;
    LogStream.WriteBuffer(LogMsg[1], Length(LogMsg));
{$ENDIF}
    Screen.Cursor := crDefault;
    ProgressDlg.Hide;
    {Now show the log messages...}
    if UpdLogForm.LogMessage.Lines.Count = 0 then
      UpdLogForm.LogMessage.Lines.Add(LoadStr(IDS_NoMessages));
{$IFDEF DebugLog}
    for i := 0 to Pred(UpdLogForm.LogMessage.Lines.Count) do
    begin
      LogMsg := 'ApplyUpdateFile 9: ' + IntToStr(i) + '"' + UpdLogForm.LogMessage.Lines[i] + '"'#13#10;
      LogStream.WriteBuffer(LogMsg[1], Length(LogMsg));
    end;
{$ENDIF}
    UpdLogForm.ShowModal;
    UpdAccumulateStream.Free;
    UpdateStream.Free;
    FileInfo.Free;
    DiffStream.Free;
  end;
{$IFDEF DebugLog}
    end;
    LogStream.Free;
{$ENDIF}
end;

procedure TUpdateForm.CancelBtnClick(Sender: TObject);
begin
  Close;
end;

procedure TUpdateForm.FormShow(Sender: TObject);
begin
  Show;
  Application.ProcessMessages;
  if ParamStr(1) >'' then
   begin
    ApplyUpdateFile(ParamStr(1));
    Application.Terminate;
   end;
end;

end.

