{  Main form for Make Update

  Copyright (c) 1998 Gregory L. Bullock (bullock@mbay.net).
  Freeware: May be freely distributed and modified. Use at your own risk.
  This program draws heavily on

    - the aDiff unit, copyright (c) 1997 S.Kurinny & S.Kostinsky

    - the Lh5Unit unit, which had various contributers, as noted in its
      source code

  History:
   28-10-98 Changed Browse button's action to automatically add all selected
            files to the list. (Previously, it would add all but the last
            selected file to the list).  Made Progress dialog get updated
            more frequently under 32-bits. 

     3-7-98 Added a CRC for the *compressed* differences to better detect
            whether an .Upd is corrupted.

    16-5-98 Modified to compile under D3 (32-bits) as well as D1 (16-bits).

            Added a new compiler directive, MAKE16AND32COMPATIBLE, which
            will ensure that the *.Upd file is compatible independent
            of the platform (16- or 32-bit) which creates it or uses it.
            On the one test case I ran, I found that using MAKE16AND32COMPATIBLE
            increased the size of the *.Upd file.  The file sizes I got were
              29304  for the 32-bit .Upd file
              30746  for the 16-bit .Upd file
              30759  for the compatible .Upd file

            Improved one of the log messages and corrected the help file.

    23-4-98 Initial version.

  If you fix any bugs or make significant enhancements, I ask you to send
  me your modifications.

  For example, at present, these programs only allow you to make changes to
  existing files.  Update will not add a new file nor will it delete an
  existing file.  Perhaps a future version of these programs will enable
  Update to add new files or delete obsolete files.

  If you make any changes to the structure of the Update File,
  change the UpdateFileHeader in the Common Unit so older versions
  of Update won't try to read an Update File that they won't understand.
}

unit Updprops;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, TabNotBk, StdCtrls, Buttons, FileCtrl, ExtCtrls, IniFiles,
  Common, {$IFDEF WIN32} ComCtrls, {$ENDIF}
  LH5Unit, Progress;

type
  TTabbedNotebookDlg = class(TForm)
    TabbedNotebook1: TTabbedNotebook;
    MakeBtn: TBitBtn;
    CloseBtn: TBitBtn;
    HelpBtn: TBitBtn;
    GroupBox1: TGroupBox;
    OldDirectoryLabel: TLabel;
    OldDirectoryListBox: TDirectoryListBox;
    OldDriveComboBox: TDriveComboBox;
    OldVersionLabel: TLabel;
    OldVersion: TEdit;
    GroupBox2: TGroupBox;
    NewDirectoryLabel: TLabel;
    NewDirectoryListBox: TDirectoryListBox;
    NewDriveComboBox: TDriveComboBox;
    NewVersionLabel: TLabel;
    NewVersion: TEdit;
    FileListBox: TListBox;
    OpenDialog: TOpenDialog;
    FileNameLabel: TLabel;
    FileName: TEdit;
    BrowseButton: TButton;
    AddBtn: TBitBtn;
    DeleteBtn: TBitBtn;
    MoveUpBtn: TBitBtn;
    MoveDownBtn: TBitBtn;
    RefreshCRCBtn: TBitBtn;
    LoadBtn: TBitBtn;
    SaveBtn: TBitBtn;
    SaveDialog: TSaveDialog;
    AboutBtn: TBitBtn;
    Label1: TLabel;
    Label2: TLabel;
    procedure OldDirectoryListBoxChange(Sender: TObject);
    procedure FileNameChange(Sender: TObject);
    procedure FileListBoxClick(Sender: TObject);
    procedure BrowseButtonClick(Sender: TObject);
    procedure AddBtnClick(Sender: TObject);
    procedure DeleteBtnClick(Sender: TObject);
    procedure MoveUpBtnClick(Sender: TObject);
    procedure MoveDownBtnClick(Sender: TObject);
    procedure FileListBoxDrawItem(Control: TWinControl; Index: Integer;
      Rect: TRect; State: TOwnerDrawState);
    procedure RefreshCRCBtnClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure LoadBtnClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure SaveBtnClick(Sender: TObject);
    procedure CloseBtnClick(Sender: TObject);
    procedure FileNameExit(Sender: TObject);
    procedure InvalidateCRCs(Sender: TObject);
    procedure MakeBtnClick(Sender: TObject);
    procedure AboutBtnClick(Sender: TObject);
    procedure ContolEnter(Sender: TObject);
  private
    { Private declarations }
    procedure ClearFileList;
  public
    { Public declarations }
    function CalculateCRCs(FileInfo: TFileUpdateInfo; ErrMsgID: integer): Boolean;
    function CanAddFileNameToList : Boolean;
    function OldAndNewDirectoriesOK : Boolean;
  end;

var
  TabbedNotebookDlg: TTabbedNotebookDlg;

implementation

uses
  AboutM, aDiff,
   {$IFDEF Win32} aCRC32; {$ELSE} aCRC3216; {$ENDIF}

{$R *.DFM}
{$IFDEF Win32}
{$R MURes32.RES}
{$ELSE}
{$R MURes16.RES}
{$ENDIF}

{$I MakeUpSt.Inc}

var
  SessionFileName  : TFileName;

procedure TTabbedNotebookDlg.FormCreate(Sender: TObject);
begin
  SessionFileName  := 'Default.UpS';
end;

procedure TTabbedNotebookDlg.FormDestroy(Sender: TObject);
begin
  ClearFileList;
end;

procedure TTabbedNotebookDlg.ClearFileList;
var
  i : integer;
begin
  with FileListBox, Items do
  begin
    for i := 0 to Pred(Count) do
      TFileUpdateInfo(Objects[i]).Free;
    Clear;
  end;
  FileListBoxClick(Self);
end;

function TTabbedNotebookDlg.CalculateCRCs(FileInfo: TFileUpdateInfo; ErrMsgID: integer): Boolean;
var
  OldNewFileName : array[0..1] of TFileName;
  OldNewCRC : array[0..1] of LongInt;
  InStream  : TFileStream;
  i : integer;
begin
  {Compute the Old & New file CRCs }
  Result := False;
  InStream := nil;
  try
    OldNewFileName[0] := GetFullPathTo(FileInfo.Name,OldDirectoryListBox);
    OldNewFileName[1] := GetFullPathTo(FileInfo.Name,NewDirectoryListBox);
    for i := 0 to 1 do
    begin
      if not FileExists(OldNewFileName[i]) then
      begin
        MessageDlg(FmtLoadStr(IDS_FileNotFoundFmt,
          [OldNewFileName[i],LoadStr(ErrMsgID)]),mtError,[mbOk],0);
        Exit;
      end;
      InStream := TFileStream.Create(OldNewFileName[i],fmOpenRead);
      InStream.Position := 0;
      OldNewCRC[i] := CalculateStreamCRC(InStream);
      InStream.Free;
      {Set InStream to nil so we don't end up Free-ing it twice when there
      is no exception.}
      InStream := nil;
    end;
    {Add the FileName and CRCs to the FileInfo}
    with FileInfo do
    begin
      OldCRC  := OldNewCRC[0];
      NewCRC  := OldNewCRC[1];
      InvalidCRC := IINo;
    end;
    Result := True;
  finally
    InStream.Free; {Free InStream in case an exception got us here.}
  end;
end;

procedure TTabbedNotebookDlg.InvalidateCRCs(Sender: TObject);
var
  i : integer;
begin
  with FileListBox,Items do
  begin
    for i := 0 to Pred(Count) do
      TFileUpdateInfo(Objects[i]).InvalidCRC := IIYes;
    Invalidate;
  end;
end;

procedure TTabbedNotebookDlg.OldDirectoryListBoxChange(Sender: TObject);
begin
  OpenDialog.InitialDir := OldDirectoryListBox.Directory;
  InvalidateCRCs(Sender);
end;

function TTabbedNotebookDlg.CanAddFileNameToList : Boolean;
var
  i : integer;
begin
  Result := FileName.Text <> '';
  i := 0;
  with FileListBox,Items do
  while Result and (i < Count) do
  begin
    Result := CompareText(TFileUpdateInfo(Objects[i]).Name,FileName.Text) <> 0;
    Inc(i);
  end;
end;

procedure TTabbedNotebookDlg.FileNameChange(Sender: TObject);
begin
  AddBtn.Enabled := CanAddFileNameToList;
end;

procedure TTabbedNotebookDlg.FileNameExit(Sender: TObject);
begin
  with FileListBox do
    if CanAddFileNameToList and (ItemIndex > -1) then
    begin
      FillChar(TFileUpdateInfo(Items.Objects[ItemIndex]).Name,
        FileNameSize,0); {Fill with zeros to clear "random" bytes from the .Upd file.}
      TFileUpdateInfo(Items.Objects[ItemIndex]).Name := FileName.Text;
      TFileUpdateInfo(Items.Objects[ItemIndex]).InvalidCRC := IIYes;
      Invalidate;
    end
end;

procedure TTabbedNotebookDlg.FileListBoxClick(Sender: TObject);
begin
  with FileListBox,Items do
  begin
    DeleteBtn.Enabled := ItemIndex > -1;
    RefreshCRCBtn.Enabled := ItemIndex > -1;
    MoveUpBtn.Enabled := ItemIndex > 0;
    MoveDownBtn.Enabled := (ItemIndex > -1) and (ItemIndex < Pred(Items.Count));
    if ItemIndex > -1 then
      FileName.Text := TFileUpdateInfo(Objects[ItemIndex]).Name;
  end;
end;

procedure TTabbedNotebookDlg.BrowseButtonClick(Sender: TObject);
var
  DirNam,
  FilNam  : TFileName;
  i,
  PathPos : integer;
  CheckDirectories  : Boolean;
begin
  DirNam := AnsiLowerCase(OldDirectoryListBox.Directory);
  CheckDirectories := True;
  if OpenDialog.Execute then
  for i := 0 to Pred(OpenDialog.Files.Count) do
  begin
    FilNam := OpenDialog.Files[i];
    FilNam := AnsiLowerCase(FilNam);
    PathPos := Pos(DirNam,FilNam);
    if (DirNam <> '') and (PathPos = 1) then
    begin
      Delete(FilNam,1,Length(DirNam));
      if FilNam[1] = '\' then
        Delete(FilNam,1,1);
      if (CheckDirectories and not OldAndNewDirectoriesOK) then
        Exit;
      FileName.Text := FilNam;
      AddBtnClick(Sender); {Automatically add each file to the list}
      CheckDirectories := False;
    end
    else
      MessageDlg(FmtLoadStr(IDS_FileNotInDirectoryTreeFmt,[FilNam,DirNam]),mtError,[mbOk],0);
  end;
end;

function TTabbedNotebookDlg.OldAndNewDirectoriesOK : Boolean;
begin
  Result := (CompareText(OldDirectoryListBox.Directory,NewDirectoryListBox.Directory) <> 0)
    or (MessageDlg(LoadStr(IDS_IdenticalOldNewDirectories),mtConfirmation,[mbYes,mbNo],0) = mrYes);
end;

procedure TTabbedNotebookDlg.AddBtnClick(Sender: TObject);
var
  FileInfo  : TFileUpdateInfo;
begin
  if ((Sender = AddBtn) and not OldAndNewDirectoriesOK)
    or not CanAddFileNameToList then
      Exit;

  FileInfo := TFileUpdateInfo.Create;
  FileInfo.Name := FileName.Text;
  if CalculateCRCs(FileInfo, IDS_FileNotAdded) then
  begin
    {Add the FileName and CRCs to the List}
    FileListBox.Items.AddObject('',FileInfo);
    AddBtn.Enabled := False;
  end
  else
    FileInfo.Free;
  FileListBoxClick(Sender);
end;

procedure TTabbedNotebookDlg.DeleteBtnClick(Sender: TObject);
begin
  with FileListBox,Items do
    if ItemIndex > -1 then
    begin
      TFileUpdateInfo(Objects[ItemIndex]).Free;
      Delete(ItemIndex);
    end;
  FileNameChange(Sender); {To adjust the Add button enabling}
  FileListBoxClick(Sender);
end;

procedure TTabbedNotebookDlg.MoveUpBtnClick(Sender: TObject);
var
  SelectedItem : integer;
begin
  with FileListBox,Items do
    if ItemIndex > 0 then
    begin
      SelectedItem := ItemIndex;
      Exchange(ItemIndex,Pred(ItemIndex));
      ItemIndex := Pred(SelectedItem);
      FileListBoxClick(Sender);
    end;
end;

procedure TTabbedNotebookDlg.MoveDownBtnClick(Sender: TObject);
var
  SelectedItem : integer;
begin
  with FileListBox,Items do
    if (ItemIndex > -1) and (ItemIndex < Pred(Count)) then
    begin
      SelectedItem := ItemIndex;
      Exchange(ItemIndex,Succ(ItemIndex));
      ItemIndex := Succ(SelectedItem);
      FileListBoxClick(Sender);
    end;
end;

procedure TTabbedNotebookDlg.RefreshCRCBtnClick(Sender: TObject);
begin
  with FileListBox,Items do
    if ItemIndex > -1 then
    begin
      CalculateCRCs(TFileUpdateInfo(Objects[ItemIndex]), IDS_CRCNotCalculated);
      Invalidate;
    end;
end;

procedure TTabbedNotebookDlg.FileListBoxDrawItem(Control: TWinControl;
  Index: Integer; Rect: TRect; State: TOwnerDrawState);
var
  Alignment : word;
  Offset,
  TxtWidth  : integer;
  ValidChar : char;
begin
  with (Control as TListBox),Canvas,TFileUpdateInfo(Items.Objects[Index]) do
  begin
    FillRect(Rect);
    TextOut(Rect.Left, Rect.Top, Name);
    TxtWidth := TextWidth('m');
    Alignment := GetTextAlign(Canvas.Handle);
    SetTextAlign(Canvas.Handle,TA_TOP or TA_RIGHT);
    Case InvalidCRC of
      IINo    : ValidChar := '!';
      IIYes   : ValidChar := '*';
      IIMaybe : ValidChar := '?'
      else ValidChar := ' ';
    end;
    Offset := Rect.Right - 1;
    TextOut(Offset, Rect.Top, ValidChar);
    TextOut(Offset-TxtWidth, Rect.Top, Format('%8.8x',[NewCRC]));
    TextOut(Offset-9*TxtWidth, Rect.Top, Format('%8.8x',[OldCRC]));
    SetTextAlign(Canvas.Handle,Alignment);
  end;
end;

procedure TTabbedNotebookDlg.LoadBtnClick(Sender: TObject);
var
  SaveTitle,
  SaveFileName: string[127];
  SaveFilter  : string;
  SessionFile : TIniFile;
  SectionLabel: string[63];
  Directory   : TFileName;
  FileInfo    : TFileUpdateInfo;
  i, NumFiles : integer;
begin
  SaveFileName := OpenDialog.FileName;
  SaveFilter := OpenDialog.Filter;
  SaveTitle := OpenDialog.Title;
  OpenDialog.Filter := LoadStr(IDS_UpdateSessionFilter);
  OpenDialog.Title := LoadStr(IDS_OpenSessionTitle);
  OpenDialog.FileName := SessionFileName;
  if OpenDialog.Execute then
  begin
    SessionFileName := OpenDialog.FileName;
    SessionFile := TIniFile.Create(SessionFileName);

    try
      SectionLabel := LoadStr(IDS_NewFilesSection);
      Directory := SessionFile.ReadString(SectionLabel,LoadStr(IDS_DirectoryLabel),NewDirectoryListBox.Directory);
      if FileGetAttr(Directory) >= 0 then
        NewDirectoryListBox.Directory := Directory;
      NewVersion.Text := SessionFile.ReadString(SectionLabel,LoadStr(IDS_VersionInfoLabel),NewVersion.Text);

      SectionLabel := LoadStr(IDS_OldFilesSection);
      Directory := SessionFile.ReadString(SectionLabel,LoadStr(IDS_DirectoryLabel),OldDirectoryListBox.Directory);
      if FileGetAttr(Directory) >= 0 then
        OldDirectoryListBox.Directory := Directory;
      OldVersion.Text := SessionFile.ReadString(SectionLabel,LoadStr(IDS_VersionInfoLabel),OldVersion.Text);

      ClearFileList;
      SectionLabel := LoadStr(IDS_FilesListLabel);
      NumFiles := SessionFile.ReadInteger(SectionLabel,LoadStr(IDS_NumberFilesLabel),0);
      for i := 1 to NumFiles do
      begin
        FileInfo := TFileUpdateInfo.Create;
        FileInfo.Name := SessionFile.ReadString(SectionLabel,FmtLoadStr(IDS_FileNameLabelFmt,[i]),'');
        FileInfo.OldCRC := SessionFile.ReadInteger(SectionLabel,FmtLoadStr(IDS_OldCRCLabelFmt,[i]),0);
        FileInfo.NewCRC := SessionFile.ReadInteger(SectionLabel,FmtLoadStr(IDS_NewCRCLabelFmt,[i]),0);
        FileInfo.InvalidCRC := IIMaybe;
        FileListBox.Items.AddObject('',FileInfo);
      end;

    finally
      SessionFile.Free;
    end;

  end;
  OpenDialog.FileName := SaveFileName;
  OpenDialog.Filter := SaveFilter;
  OpenDialog.Title := SaveTitle;
end;

procedure TTabbedNotebookDlg.SaveBtnClick(Sender: TObject);
var
  SessionFile : TIniFile;
  SectionLabel: string[63];
  i           : integer;
begin
  SaveDialog.FileName := SessionFileName;
  if SaveDialog.Execute then
  begin
    SessionFileName := SaveDialog.FileName;
    SessionFile := TIniFile.Create(SessionFileName);

    try
      SectionLabel := LoadStr(IDS_OldFilesSection);
      SessionFile.WriteString(SectionLabel,LoadStr(IDS_DirectoryLabel),OldDirectoryListBox.Directory);
      SessionFile.WriteString(SectionLabel,LoadStr(IDS_VersionInfoLabel),OldVersion.Text);

      SectionLabel := LoadStr(IDS_NewFilesSection);
      SessionFile.WriteString(SectionLabel,LoadStr(IDS_DirectoryLabel),NewDirectoryListBox.Directory);
      SessionFile.WriteString(SectionLabel,LoadStr(IDS_VersionInfoLabel),NewVersion.Text);

      SectionLabel := LoadStr(IDS_FilesListLabel);
      SessionFile.EraseSection(SectionLabel);
      with FileListBox.Items do
      begin
        SessionFile.WriteInteger(SectionLabel,LoadStr(IDS_NumberFilesLabel),Count);
        for i := 1 to Count do
        with TFileUpdateInfo(Objects[Pred(i)]) do
        begin
          SessionFile.WriteString(SectionLabel,FmtLoadStr(IDS_FileNameLabelFmt,[i]),Name);
          SessionFile.WriteInteger(SectionLabel,FmtLoadStr(IDS_OldCRCLabelFmt,[i]),OldCRC);
          SessionFile.WriteInteger(SectionLabel,FmtLoadStr(IDS_NewCRCLabelFmt,[i]),NewCRC);
        end;
      end;

    finally
      SessionFile.Free;
    end;

  end;
end;

procedure TTabbedNotebookDlg.CloseBtnClick(Sender: TObject);
begin
  Close;
end;

function HasExtension(const Name : string; var DotPos : Word) : Boolean;
  {-Return whether and position of extension separator dot in a pathname}
var
  I : Word;
begin
  DotPos := 0;
  for I := Length(Name) downto 1 do
    if (Name[I] = '.') and (DotPos = 0) then
      DotPos := I;
  HasExtension := (DotPos > 0) and (Pos('\', Copy(Name, Succ(DotPos), 64)) = 0);
end;

function ForceExtension(const Name, Ext : string) : string;
  {-Return a pathname with the specified extension attached}
var
  DotPos : Word;
begin
  if HasExtension(Name, DotPos) then
    ForceExtension := Copy(Name, 1, DotPos)+Ext
  else
    ForceExtension := Name+'.'+Ext;
end;

function JustFilename(const PathName : string) : string;
  {-Return just the filename of a pathname}
var
  I : Word;
begin
  I := Succ(Word(Length(PathName)));
  repeat
    Dec(I);
  until (PathName[I] in ['\', ':', #0]) or (I = 0);
  JustFilename := Copy(PathName, Succ(I), 64);
end;

procedure TTabbedNotebookDlg.MakeBtnClick(Sender: TObject);
var
  i : integer;
  VersionInfo   : string[31];
  SavePosition,
  CRC,
  DiffSize      : LongInt;
  FileInfo      : TFileUpdateInfo;
begin
  {Create Update file}
  UpdateStream := nil;
  DiffStream := nil;
  UpdAccumulateStream := nil;
  if SessionFileName = '' then
    SaveDialog.FileName := 'Default.Upd'
  else
    SaveDialog.FileName := ForceExtension(JustFileName(SessionFileName),'Upd');
  SaveDialog.FilterIndex := 2;
  try
    if SaveDialog.Execute then
    begin
      UpdateStream  := TFileStream.Create(SaveDialog.FileName,fmCreate);
      UpdateStream.WriteBuffer(UpdateFileHeader[1],Length(UpdateFileHeader));
      FillChar(VersionInfo,SizeOf(VersionInfo),0); {Fill with zeros to clear "random" bytes from the .Upd.}
      VersionInfo := OldVersion.Text;
      UpdateStream.WriteBuffer(VersionInfo,SizeOf(VersionInfo));
      FillChar(VersionInfo,SizeOf(VersionInfo),0); {Fill with zeros to clear "random" bytes from the .Upd.}
      VersionInfo := NewVersion.Text;
      UpdateStream.WriteBuffer(VersionInfo,SizeOf(VersionInfo));

      DiffStream := TMemoryStream.Create;
      UpdAccumulateStream := TMemoryStream.Create;

      ProgressDlg.Show;
      with FileListBox.Items do
        for i := 0 to Pred(Count) do
        if CalculateCRCs(TFileUpdateInfo(Objects[i]), IDS_FileNotIncluded) then
        begin
          OldStream := nil;
          NewStream := nil;
          try
            FileInfo  := TFileUpdateInfo(Objects[i]);
            UpdAccumulateStream.WriteBuffer(FileInfo.Name, SaveFileUpdateInfoSize);
            OldStream := TFileStream.Create(GetFullPathTo(FileInfo.Name,OldDirectoryListBox),fmOpenRead);
            NewStream := TFileStream.Create(GetFullPathTo(FileInfo.Name,NewDirectoryListBox),fmOpenRead);
            ProgressDlg.FileNameLabel.Caption := GetFullPathTo(FileInfo.Name,NewDirectoryListBox);
            ProgressDlg.ProgressBar.Progress := 0;
            Application.ProcessMessages;
            DiffStream.Clear;
            Screen.Cursor := crHourglass;
            DiffStreamCompress(NewStream,OldStream,DiffStream,UpdateProgress,1000);
            DiffSize := DiffStream.Size + SizeOf(CRC);
            UpdAccumulateStream.WriteBuffer(DiffSize,SizeOf(DiffSize));
            DiffStream.Position := 0;
            UpdAccumulateStream.CopyFrom(DiffStream,DiffStream.Size);
            DiffStream.Position := 0;
            CRC := CalculateStreamCRC(DiffStream);
            UpdAccumulateStream.WriteBuffer(CRC,SizeOf(CRC));
          finally
            OldStream.Free;
            NewStream.Free;
            Screen.Cursor := crDefault;
          end;
        end;
      FileListBox.Invalidate; {To show the updated CRCs}
      UpdAccumulateStream.Position := 0;
      SavePosition := UpdateStream.Position;
      UpdateStream.WriteBuffer(CRC,SizeOf(CRC)); {Make a place holder for the compressed CRC.}

      LHACompress(UpdAccumulateStream, UpdateStream); {Compress the stream}

      UpdateStream.Position := SavePosition + SizeOf(CRC); {Calculate the compressed CRC.}
      CRC := CalculateStreamCRC(UpdateStream);
      UpdateStream.Position := SavePosition;
      UpdateStream.WriteBuffer(CRC,SizeOf(CRC)); {Now write the actual compressed CRC.}
    end;
  finally
    ProgressDlg.Hide;
    SaveDialog.FilterIndex := 1;
    UpdateStream.Free;
    DiffStream.Free;
    UpdAccumulateStream.Free;
  end;
end;

procedure TTabbedNotebookDlg.AboutBtnClick(Sender: TObject);
begin
  AboutBox := TAboutBox.Create(Self);
  AboutBox.ShowModal;
  AboutBox.Free;
  AboutBox := nil;
end;

procedure TTabbedNotebookDlg.ContolEnter(Sender: TObject);
var
  Control: TWinControl;
begin
  if Sender is TWinControl then
  begin
    Control := TWinControl(Sender);
    while (Control <> nil) and (Control.HelpContext = 0) do
      Control := Control.Parent;
    if Control <> nil then
      HelpBtn.HelpContext := Control.HelpContext;
  end;
end;

end.
