{  Main form for Update

  Copyright (c) 1998 Gregory L. Bullock (bullock@mbay.net).
  Freeware: May be freely distributed and modified. Use at your own risk.
  This program draws heavily on

    - the aDiff unit, Copyright (c) 1997 S.Kurinny & S.Kostinsky

    - the Lh5Unit unit, which had various contributers, as noted in its
      source code

  History:
    23-4-98 Initial version.

  If you fix any bugs or make significant enhancements, I ask you to send
  me your modifications.

  For example, at present, these programs only allow you to make changes to
  existing files.  Update will not add a new file nor will it delete an
  existing file.  Perhaps a future version of these programs will enable
  Update to add new files or delete obsolete files.

  If you make any changes to the structure of the Update File,
  change the UpdateFileHeader in the Common Unit so older versions
  of Update won't try to read an Update File that they won't understand.
}

unit UpdateFm;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, Buttons, FileCtrl, LogForm, About,
  LH5Unit, Progress;

type
  TUpdateForm = class(TForm)
    GoBtn: TBitBtn;
    CancelBtn: TBitBtn;
    OpenDialog: TOpenDialog;
    GroupBox1: TGroupBox;
    TargetFile: TLabel;
    DirectoryListBox1: TDirectoryListBox;
    DriveComboBox1: TDriveComboBox;
    GroupBox2: TGroupBox;
    ChangesFile: TEdit;
    BrowseChangesFile: TButton;
    AboutBtn: TBitBtn;
    procedure BrowseChangesFileClick(Sender: TObject);
    procedure AboutBtnClick(Sender: TObject);
    procedure GoBtnClick(Sender: TObject);
    procedure CancelBtnClick(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
  end;

var
  UpdateForm: TUpdateForm;

implementation

uses
  Common, aDiff,
   {$IFDEF Win32} aCRC32; {$ELSE} aCRC3216; {$ENDIF}

{$R *.DFM}
{$IFDEF Win32}
{$R UpdRes32.RES}
{$ELSE}
{$R UpdRes16.RES}
{$ENDIF}

type
  EUpdateFileError = class(Exception);
  EDifferenceError = class(Exception);

{$I UpdateSt.Inc}

var
  TotalProgressCalls : longint;

procedure TUpdateForm.BrowseChangesFileClick(Sender: TObject);
begin
  OpenDialog.FileName := ChangesFile.Text;
  if OpenDialog.Execute then
    ChangesFile.Text := LowerCase(OpenDialog.FileName);
end;

procedure TUpdateForm.AboutBtnClick(Sender: TObject);
begin
  AboutBox := TAboutBox.Create(Self);
  AboutBox.ShowModal;
  AboutBox.Free;
  AboutBox := nil;
end;

procedure UpdateProgress; far;
begin
  ProgressDlg.ProgressBar.Progress := ProgressDlg.ProgressBar.Progress + 100 div TotalProgressCalls;
  Application.ProcessMessages;
end;

procedure TUpdateForm.GoBtnClick(Sender: TObject);
var
  FileHeader    : string[15];
  OldVersion,
  NewVersion    : string[31];
  DiffFileName,
  NewFileName,
  OldFileName   : TFileName;
  FileInfo      : TFileUpdateInfo;
  FileInfoSize  : integer;
  NextUpdPosition,
  CRC,
  DiffSize      : LongInt;
begin
  {Apply Update file}
  UpdateStream := nil;
  DiffStream := nil;
  FileInfo := nil;
  UpdAccumulateStream := nil;
  UpdLogForm.LogMessage.Lines.Clear;
  DiffSize := 0;
  NextUpdPosition := 0;
  DiffFileName := ChangesFile.Text;
  try try
    if not FileExists(DiffFileName) then
      raise EUpdateFileError.CreateResFmt(IDS_UpdFileNotFoundFmt,[ChangesFile.Text]);
    try
      UpdateStream  := TFileStream.Create(ChangesFile.Text,fmOpenRead);
      FileHeader := UpdateFileHeader; {This copies the length byte}
      UpdateStream.ReadBuffer(FileHeader[1],Length(UpdateFileHeader));
    except
      raise EUpdateFileError.CreateResFmt(IDS_CannotReadUpdateFileFmt,[ChangesFile.Text]);
    end;
    if FileHeader <> UpdateFileHeader then
      raise EUpdateFileError.CreateResFmt(IDS_InvalidUpdateFileFmt,[ChangesFile.Text]);
    try
      UpdateStream.ReadBuffer(OldVersion,SizeOf(OldVersion));
      UpdateStream.ReadBuffer(NewVersion,SizeOf(NewVersion));
    except
      raise EUpdateFileError.CreateResFmt(IDS_CannotReadUpdateFileFmt,[ChangesFile.Text]);
    end;

    NewFileName := GetFullPathTo('$UpdTmp$.~aa',TargetFile);
    while FileExists(NewFileName) do
      if NewFileName[Length(NewFileName)] <> 'z' then
        Inc(NewFileName[Length(NewFileName)])
      else begin
        Inc(NewFileName[Length(NewFileName)-1]);
        NewFileName[Length(NewFileName)] := 'a';
      end;

    DiffStream := TMemoryStream.Create;
    UpdAccumulateStream := TMemoryStream.Create;
    LHAExpand(UpdateStream, UpdAccumulateStream);
    UpdAccumulateStream.Position := 0;
    FileInfo := TFileUpdateInfo.Create;
    FileInfoSize := Ofs(FileInfo.InvalidCRC) - Ofs(FileInfo.Name);
    ProgressDlg.Show;
    while UpdAccumulateStream.Read(FileInfo.Name, FileInfoSize) = FileInfoSize do
    begin
      try
        try
          OldStream := nil;
          NewStream := nil;
          OldFileName := GetFullPathTo(FileInfo.Name,TargetFile);
          if not FileExists(OldFileName) then
            raise EDifferenceError.Create(LoadStr(IDS_FileNotFound));
          try
            OldStream := TFileStream.Create(OldFileName,fmOpenRead);
            NewStream := TFileStream.Create(NewFileName,fmCreate);
          except
            raise EDifferenceError.Create(LoadStr(IDS_CannotOpenFile));
          end;
          CRC := CalculateStreamCRC(OldStream);
          if CRC = FileInfo.NewCRC then
            raise EDifferenceError.CreateResFmt(IDS_FileAlreadyUpdatedFmt,[NewVersion]);
          if CRC <> FileInfo.OldCRC then
            raise EDifferenceError.CreateResFmt(IDS_FileWrongVersionFmt,[OldVersion]);
          try
            ProgressDlg.FileNameLabel.Caption := OldFileName;
            ProgressDlg.ProgressBar.Progress := 0;
            TotalProgressCalls := (OldStream.Size + Pred(aDiff.BufSize)) div aDiff.BufSize;
            Application.ProcessMessages;
            Screen.Cursor := crHourglass;
            DiffStream.Clear;
            UpdAccumulateStream.ReadBuffer(DiffSize,SizeOf(DiffSize));
            NextUpdPosition := UpdAccumulateStream.Position + DiffSize;
            DiffStream.CopyFrom(UpdAccumulateStream,DiffSize-SizeOf(CRC));
            UpdAccumulateStream.ReadBuffer(CRC,SizeOf(CRC));
          except
            raise EDifferenceError.Create(LoadStr(IDS_InvalidChangesForFile));
          end;
          if CRC <> CalculateStreamCRC(DiffStream) then
            raise EDifferenceError.Create(LoadStr(IDS_InvalidChangesForFile));
          try
            DiffStreamExtract(DiffStream,OldStream,NewStream,UpdateProgress);
          except
            raise EDifferenceError.Create(LoadStr(IDS_ErrorApplyingChanges));
          end;
          if CalculateStreamCRC(NewStream) <> FileInfo.NewCRC then
            raise EDifferenceError.Create(LoadStr(IDS_UpdatedFileFailsCRC));
          OldStream.Free;
          NewStream.Free;
          OldStream := nil;
          NewStream := nil;
          DeleteFile(OldFileName);
          RenameFile(NewFileName,OldFileName);
          UpdLogForm.LogMessage.Lines.Add(FmtLoadStr(IDS_UpdatedFileFmt,[NewVersion,OldFileName]));
        finally
          OldStream.Free;
          NewStream.Free;
          DeleteFile(NewFileName);
          Screen.Cursor := crDefault;
        end;
      except
        on E: EDifferenceError do
          UpdLogForm.LogMessage.Lines.Add(FmtLoadStr(IDS_SkippingFileFmt,[OldFileName,E.Message]))
      end;
      if NextUpdPosition = 0 then
      begin
        if DiffSize = 0 then
          UpdAccumulateStream.ReadBuffer(DiffSize,SizeOf(DiffSize));
        NextUpdPosition := UpdAccumulateStream.Position + DiffSize;
      end;
      UpdAccumulateStream.Position := NextUpdPosition;
      DiffSize := 0;
      NextUpdPosition := 0;
    end;
  finally
    Screen.Cursor := crDefault;
    ProgressDlg.Hide;
    {Now show the log messages...}
    if UpdLogForm.LogMessage.Lines.Count = 0 then
      UpdLogForm.LogMessage.Lines.Add(LoadStr(IDS_NoMessages));
    UpdLogForm.ShowModal;
    UpdAccumulateStream.Free;
    UpdateStream.Free;
    FileInfo.Free;
    DiffStream.Free;
  end;
  except
    on E: EUpdateFileError do
      MessageDlg(FmtLoadStr(IDS_UpdatesNotAppliedFmt,[E.Message]),mtError,[mbOk],0);
  end;
end;

procedure TUpdateForm.CancelBtnClick(Sender: TObject);
begin
  Close;
end;

end.

