unit S_List;

{ Secrets of Delphi 2, by Ray Lischner. (1996, Waite Group Press).
  Chapter 5: Dynamic Arrays: the TList Class
  Copyright  1996 The Waite Group, Inc. }

{ Custom list class for Delphi 1.0 and Delphi 2.0 }

interface

uses Classes, WinTypes, WinProcs, SysUtils;

type
  TS_PackProc = procedure(Index: LongInt) of object;
  TS_CustomList = class
  private
    fList: Pointer;     { Pointer to the base of the list }
    fCount: LongInt;    { Number of items in the list }
    fCapacity: LongInt; { Number of available slots in the list }
  protected
    function GetPointer(Index: LongInt): Pointer; virtual;
    procedure ReAllocList(NewCapacity: LongInt); virtual;
    procedure CheckIndex(Index: LongInt); virtual;
    function ExpandSize: LongInt; virtual;
    function GetItem(Index: LongInt): Pointer; virtual;
    procedure SetItem(Index: LongInt; Item: Pointer); virtual;
    procedure SetCapacity(NewCapacity: LongInt); virtual;
    procedure SetCount(NewCount: LongInt); virtual;
    procedure DoPack(PackProc: TS_PackProc);
    property Items[Index: LongInt]: Pointer read GetItem write SetItem;
  public
    destructor Destroy; override;
    function Add(Item: Pointer): LongInt; virtual;
    property Capacity: LongInt read fCapacity write SetCapacity;
    procedure Clear; virtual;
    property Count: LongInt read fCount write SetCount;
    { This is different from TList, where Delete is a procedure.
      It seems more useful to return the deleted pointer, which
      can save the caller a step sometimes. }
    function Delete(Index: LongInt): Pointer; virtual;
    procedure Exchange(I1, I2: LongInt); virtual;
    function Expand: TS_CustomList;
    function First: Pointer;
    function IndexOf(Item: Pointer): LongInt; virtual;
    procedure Insert(Index: LongInt; Item: Pointer); virtual;
    function Last: Pointer;
    procedure Move(CurIndex, NewIndex: LongInt); virtual;
    procedure Pack; virtual;
    function Remove(Item: Pointer): LongInt; virtual;
  end;

  { A logical equivalent to TList, but with LongInt indexes. }
  TS_List = class(TS_CustomList)
  public
    property Items; default;
  end;

  { This class is slightly more convenient to use when creating a list
    of objects. Expect lots of warnings from the Delphi 2.0 compiler
    about hiding virtual methods. That's the idea: to hide the Pointer
    methods with TObject equivalents. }
  TS_ObjectList = class(TS_CustomList)
  protected
    function GetObject(Index: LongInt): TObject;
    procedure SetObject(Index: LongInt; Value: TObject);
    property Objects[Index: LongInt]: TObject read GetObject write SetObject;
  public
    function Add(Item: TObject): LongInt; virtual;
    function Delete(Index: LongInt): TObject; virtual;
    function First: TObject;
    function Last: TObject;
    function Remove(Item: TObject): LongInt; virtual;
  end;

  { TS_OwnerList owns the objects it contains. }
  TS_OwnerList = class(TS_ObjectList)
  protected
    procedure SetCapacity(NewCapacity: LongInt); override;
    procedure SetCount(NewCount: LongInt); override;
  public
    procedure Clear; override;
    function Delete(Index: LongInt): TObject; override;
  end;

implementation

uses Consts, S_Huge;

{$ifdef WIN32}
{ Move Size pointers within an array, from index Src to index Dst. }
procedure HugeMove(Base: Pointer; Dst, Src, Size: LongInt);
var
  SrcPtr, DstPtr: PChar;
begin
  SrcPtr := PChar(Base) + Src * SizeOf(Pointer);
  DstPtr := PChar(Base) + Dst * SizeOf(Pointer);
  Move(SrcPtr^, DstPtr^, Size*SizeOf(Pointer));
end;

{$else}

procedure HugeShift; far; external 'KERNEL' index 113;

{
  Copy memory from Src to Dst, copying Size units from
  Base+Src to Base+Dst.  The reason to separate the Base pointer
  from the Dst and Src indexes is to easily compare
  the indexes to see what direction to copy, without
  having to muck about with normalizing segments.
  Offsets and the size are in longwords, for convenience,
  but you need to convert them to short words, in case the code
  is running on a 286 machine, which does not support the movsd
  instruction, just movsw.

  I'm not a fan of writing assembly language, so most of the code
  is Pascal, and I only dip into assembly for the low-level
  copying code, in DoMove, plus setting the direction flag,
  which is done with inline assembler procedures.
}
procedure cld; inline ($fc);	{ clear direction flag }
procedure std; inline ($fd);	{ set direction flag }

{ When computing how many words to move, you need to see
  how far from a segment boundary SrcPtr and DstPtr are,
  since the MOVS instruction won't cross a segment boundary
  properly.  Since you know that the list is allocated on
  a segment boundary, and TCustomList stores long words,
  SrcPtr and DstPtr will always be on longword boundaries.

  It is actually a little tricky to determine how many words
  to copy before hitting a segment boundary.  It depends on
  whether the copying goes up or down.  When copying down, the
  difference between $10000 and the pointer offset determines
  how many bytes to copy in that segment.  If the segment
  offset is zero, then truncating $10000 to a word yields zero,
  in which case, copy the maximum number of words, $7FFF.
}
function ComputeDownMoveSize(SrcOffset, DstOffset: Word): Word;
begin
  { Determine the number of words remaining in the segment, to copy. }
  if SrcOffset > DstOffset then
    Result := Word($10000 - SrcOffset) div 2
  else
    Result := Word($10000 - DstOffset) div 2;
  { Copy the entire segment. }
  if Result = 0 then
    Result := $7FFF;
end;

function ComputeUpMoveSize(SrcOffset, DstOffset: Word): Word;
begin
  if SrcOffset = $FFFF then
    { Copy as many words as are available in the destination. }
    Result := DstOffset div 2
  else if DstOffset = $FFFF then
    { Copy as many words as are available in the source. }
    Result := SrcOffset div 2
  else if SrcOffset > DstOffset then
    { Copy as many words as are available in the destination. }
    Result := DstOffset div 2 + 1
  else
    { Copy as many words as are available in the source. }
    Result := SrcOffset div 2 + 1;
end;

procedure MoveWords(SrcPtr, DstPtr: Pointer; Size: Word); assembler;
asm
  push ds                 { Save DS, which is modified below. }
  lds si, SrcPtr          { Get the source pointer. }
  les di, DstPtr          { Get the destination pointer. }
  mov cx, Size.Word[0]    { Get the number of words to copy. }
  rep movsw               { Copy the memory. }
  pop ds                  { Restore DS. }
end;

procedure HugeMove(Base: Pointer; Dst, Src, Size: LongInt);
var
  SrcPtr, DstPtr: Pointer;
  MoveSize: Word;
begin
  SrcPtr := HugeOffset(Base, Src * SizeOf(Pointer));
  DstPtr := HugeOffset(Base, Dst * SizeOf(Pointer));
  { Convert longword size to words. }
  Size := Size * (SizeOf(LongInt) div SizeOf(Word));
  if Src < Dst then
  begin
    { Start from the far end and work toward the front. }
    std;
    HugeInc(SrcPtr, (Size-1) * SizeOf(Word));
    HugeInc(DstPtr, (Size-1) * SizeOf(Word));
    while Size > 0 do
    begin
      { Compute how many bytes to move in the current segment. }
      MoveSize := ComputeUpMoveSize(Word(SrcPtr), Word(DstPtr));
      if MoveSize > Size then
        MoveSize := Word(Size);

      { Move the bytes. }
      MoveWords(SrcPtr, DstPtr, MoveSize);

      { Update the number of bytes left to move. }
      Dec(Size, MoveSize);
      { Update the pointers. }
      HugeDec(SrcPtr, MoveSize * SizeOf(Word));
      HugeDec(DstPtr, MoveSize * SizeOf(Word));
    end;
    cld;     { reset the direction flag }
  end
  else
  begin
    { Start from the beginning and work toward the end. }
    cld;
    while Size > 0 do
    begin
      { Compute how many bytes to move in the current segment. }
      MoveSize := ComputeDownMoveSize(Word(SrcPtr), Word(DstPtr));
      if MoveSize > Size then
        MoveSize := Word(Size);

      { Move the bytes. }
      MoveWords(SrcPtr, DstPtr, MoveSize);
   
      { Update the number of bytes left to move. }
      Dec(Size, MoveSize);
      { Advance the pointers. }
      HugeInc(SrcPtr, MoveSize * SizeOf(Word));
      HugeInc(DstPtr, MoveSize * SizeOf(Word));
    end;
  end;
end;
{$endif}


{ Check a list index and raise an exception if it is not valid. }
procedure TS_CustomList.CheckIndex(Index: LongInt);
begin
  if (Index < 0) or (Index >= Count) then
    raise EListError.CreateRes(SListIndexError)
end;

{ Destroy a TS_CustomList by unlocking and freeing the list's memory }
destructor TS_CustomList.Destroy;
begin
  Clear;
  inherited Destroy;
end;

{ Reallocate the list, copying the old list values into
  the new list.  Reset all the fields for the new list.
}
procedure TS_CustomList.ReAllocList(NewCapacity: LongInt);
var
  NewList: Pointer;
  Ptr: Pointer;
begin
  { If the list is shrinking, then update Count for the smaller size. }
  if NewCapacity < Count then
    fCount := NewCapacity;

  { Try to reallocate the existing list, if there is one. }
  if fList = nil then
    NewList := nil
  else
    NewList := GlobalReallocPtr(fList, NewCapacity*SizeOf(Pointer), GMem_Moveable);

  if NewList = nil then
  begin
    { Either this is the first time, or the reallocation failed. }
    NewList := GlobalAllocPtr(GMem_Fixed, NewCapacity*SizeOf(Pointer));
    if NewList = nil then
      OutOfMemoryError;

    { Copy the old list into the new one. }
    if fList <> nil then
    begin
      HMemCpy(NewList, fList, Count*SizeOf(Pointer));
      GlobalFreePtr(fList);
    end;
  end;

  fList := NewList;

  { If Windows allocated even more memory, then see how much
    we really have.  The user might not have requested that
    much, but why waste memory?
  }
{$ifdef WIN32}
  fCapacity := GlobalSize(GlobalHandle(fList)) div SizeOf(Pointer);
{$else}
  { Get the selector of the list pointer, to retrieve the memory handle.
    Extract the handle part of the value returned by GlobalHandle to
    get the size in bytes. Then divide to get the size as a capacity. }
  fCapacity := GlobalSize(Word(GlobalHandle(Seg(fList^)))) div SizeOf(Pointer);
{$endif}

  { Zero out the unused slots. }
  if Capacity > Count then
  begin
    Ptr := HugeOffset(fList, Count * SizeOf(Pointer));
    FillChar(Ptr^, (Capacity-Count)*SizeOf(Pointer), 0);
  end;
end;

{ Return the new capacity of the list when we expand it.
  Here we emulate TList, which expands the list by 4, 8, or
  16 elements, depending on the number of items already in
  the list.
}
function TS_CustomList.ExpandSize: LongInt;
begin
  if Capacity > 8 then
    ExpandSize := Capacity + 16
  else if Capacity > 4 then
    ExpandSize := Capacity + 8
  else
    ExpandSize := Capacity + 4
end;

{ Return an item of the list, raising an exception for an invalid index }
function TS_CustomList.GetItem(Index: LongInt): Pointer;
var
  Ptr: ^Pointer;
begin
  CheckIndex(Index);
  Ptr := HugeOffset(fList, Index*SizeOf(Pointer));
  GetItem := Ptr^;
end;

{ Get an item without checking for range errors.  Return a pointer
  to the slot in the list, so the pointer value can be changed.
  This function is available for use by derived classes, but is
  not used by TS_CustomList itself.  Instead, TS_CustomList calls
  HugeOffset() directly, which improves performance slightly.
}
function TS_CustomList.GetPointer(Index: LongInt): Pointer;
begin
  GetPointer := HugeOffset(fList, Index * SizeOf(Pointer));
end;

{ Set a list item. }
procedure TS_CustomList.SetItem(Index: LongInt; Item: Pointer);
var
  Ptr: ^Pointer;
begin
  CheckIndex(Index);
  Ptr := HugeOffset(fList, Index * SizeOf(Pointer));
  Ptr^ := Item;
end;

{ Set the new capacity of the list.  If the list shrinks,
  then adjust Count.
}
procedure TS_CustomList.SetCapacity(NewCapacity: LongInt);
begin
  if NewCapacity < 0 then
    raise EListError.CreateRes(sListIndexError);
  if Capacity <> NewCapacity then
    ReAllocList(NewCapacity);
end;

{ Set the count. If the new count is larger, then set the new
  items to nil. Increase the capacity as needed. }
procedure TS_CustomList.SetCount(NewCount: LongInt);
var
  Ptr: ^Pointer;
  NZeroBytes: LongInt;
begin
  if NewCount < 0 then
    raise EListError.CreateRes(sListIndexError);
  if NewCount > Capacity then
    Capacity := NewCount;
  if Count < NewCount then
  begin
    { zero all the new items, which may cross segment boundaries }
    Ptr := HugeOffset(fList, Count * SizeOf(Pointer));
    NZeroBytes := (NewCount - Count) * SizeOf(Pointer);
    ZeroMemory(Ptr, NZeroBytes);
  end;
  fCount := NewCount;
end;

{ Add Item to the end of the list }
function TS_CustomList.Add(Item: Pointer): LongInt;
begin
  Insert(Count, Item);
  Add := Count-1;
end;

{ Clear all the items in the list. }
procedure TS_CustomList.Clear;
begin
  if fList <> nil then
  begin
    GlobalFreePtr(fList);
    fList := nil;
  end;
  fCount := 0;
  fCapacity := 0;
end;

function TS_CustomList.Delete(Index: LongInt): Pointer;
begin
  Result := Items[Index];
  Dec(fCount);
  HugeMove(fList, Index, Index+1, Count-Index);
end;

{ Exchange the items at indexes I1 and I2. }
procedure TS_CustomList.Exchange(I1, I2: LongInt);
var
  Tmp: Pointer;
  P, Q: ^Pointer;
begin
  CheckIndex(I1);
  CheckIndex(I2);
  P := HugeOffset(fList, I1 * SizeOf(Pointer));
  Q := HugeOffset(fList, I2 * SizeOf(Pointer));
  Tmp := P^;
  P^ := Q^;
  Q^ := Tmp;
end;

{ Return the first item in the list }
function TS_CustomList.First: Pointer;
begin
  First := Items[0];
end;

function TS_CustomList.IndexOf(Item: Pointer): LongInt;
var
  Ptr: ^Pointer;
begin
  Ptr := fList;
  for Result := 0 to Count-1 do
  begin
    if Ptr^ = Item then
      Exit;
    Ptr := HugeOffset(Ptr, SizeOf(Pointer));
  end;
  Result := -1;
end;

{ Insert Item at position, Index.  Slide all other items
  over to make room.  The user can insert to any valid index,
  or to one past the end of the list, thereby appending an
  item to the list.  In the latter case, adjust the capacity
  if needed. }
procedure TS_CustomList.Insert(Index: LongInt; Item: Pointer);
var
  Ptr: ^Pointer;
begin
  if (Index < 0) or (Index > Count) then
    raise EListError.CreateRes(SListIndexError);
  if Count >= Capacity then
    Expand;
  { Make room for the inserted item. }
  Ptr := HugeOffset(fList, Index * SizeOf(Pointer));
  HugeMove(Ptr, 1, 0, Count-Index);
  Ptr^ := Item;
  Inc(fCount);
end;

{ Return the last item in the list.  Raise an exception
  if the list is empty.
}
function TS_CustomList.Last: Pointer;
begin
  Last := Items[Count - 1];
end;

{ Move an item from CurIndex to NewIndex. Only move the
  items that lie between CurIndex and NewIndex, leaving
  the rest of the list alone.
}
procedure TS_CustomList.Move(CurIndex, NewIndex: LongInt);
var
  Tmp: Pointer;
  Ptr: ^Pointer;
begin
  CheckIndex(NewIndex);
  if NewIndex <> CurIndex then
  begin
    Tmp := Items[CurIndex];
    if NewIndex < CurIndex then
    begin
      Ptr := HugeOffset(fList, NewIndex * SizeOf(Pointer));
      HugeMove(Ptr, 1, 0, CurIndex-NewIndex)
    end
    else if CurIndex < NewIndex then
    begin
      Ptr := HugeOffset(fList, CurIndex * SizeOf(Pointer));
      HugeMove(Ptr, 0, 1, NewIndex-CurIndex);
      Ptr := HugeOffset(fList, NewIndex * SizeOf(Pointer));
    end;
    Ptr^ := Tmp;
  end;
end;

{ Pack the list by removing nil slots.  After packing
  Count might be smaller.  After each loop iteration,
  the following is invariant:
    Items[k] <> nil for all k <= i
  Thus, when at the end of the loop, the list is packed.

  The loop marches through the list, using the I index.
  Whenever Items[I] = nil, collect a maximal string of nil slots,
  and then shift down the remaining items, adjusting Count to match.
}
procedure TS_CustomList.DoPack(PackProc: TS_PackProc);
var
  I, J, K: LongInt;
  P, Q: ^Pointer;
begin
  { Instead of a for loop, use a while loop, and use the
    current value of Count for each iteration, since Count
    changes during the loop. }
  I := 0;
  P := fList;
  while I < Count do
  begin
    if P^ <> nil then
    begin
      Inc(I);
      P := HugeOffset(fList, I*SizeOf(Pointer));
    end
    else
    begin
      if Assigned(PackProc) then
        PackProc(I);
      { Collect a run of nil slots. }
      for J := I+1 to Count-1 do
      begin
        P := HugeOffset(fList, J * SizeOf(Pointer));
        if P^ <> nil then
          Break
        else if Assigned(PackProc) then
          PackProc(J)
        else
      end;
      { Shift slots if there is a non-nil value.
        If all the remaining slots are nil, then the loop is done. }
      if P^ = nil then
      begin
        fCount := I;
        Break;
      end;
      { Now shift the slots; setting the newly vacated slots to nil,
        as a safety measure. Stop at the next nil slot. }
      K := I;
      while J < Count do
      begin
        P := HugeOffset(fList, K * SizeOf(Pointer));
        Q := HugeOffset(fList, J * SizeOf(Pointer));
        P^ := Q^;
        { Check after assigning to P^, so the check for nil at
          the top of the loop is true.  A small inefficiency for
          greater programming ease and maintainability. }
        if Q^ = nil then
          Break;
        Q^ := nil;
        Inc(K);
        Inc(J);
      end;
      { Adjust Count by the number of nil slots removed. }
      Dec(fCount, J-K);
      { Set the loop counter to the next nil slot. }
      I := K;
    end;
  end;
end;

procedure TS_CustomList.Pack;
begin
  DoPack(nil)
end;

function TS_CustomList.Remove(Item: Pointer): LongInt;
begin
  Result := IndexOf(Item);
  if Result >= 0 then
    Delete(Result);
end;

function TS_CustomList.Expand: TS_CustomList;
begin
  Capacity := ExpandSize;
  Result := Self;
end;

{ TS_ObjectList }
{ This is just a list of TObjects, instead of Pointers. This makes the
  class slightly easier to use because a derived class can use the AS
  operator directly, e.g., Objects[I] as TFoo. }
function TS_ObjectList. GetObject(Index: LongInt): TObject;
begin
  Result := TObject(Items[Index])
end;

procedure TS_ObjectList.SetObject(Index: LongInt; Value: TObject);
begin
  Items[Index] := Pointer(Value)
end;

function TS_ObjectList.Add(Item: TObject): LongInt;
begin
  Result := inherited Add(Pointer(Item))
end;

function TS_ObjectList.Delete(Index: LongInt): TObject;
begin
  Result := TObject(inherited Delete(Index))
end;

function TS_ObjectList.First: TObject;
begin
  Result := TObject(inherited First)
end;

function TS_ObjectList.Last: TObject;
begin
  Result := TObject(inherited Last)
end;

function TS_ObjectList.Remove(Item: TObject): LongInt;
begin
  Result := inherited Remove(Pointer(Item))
end;

{ TS_OwnerList }
{ Just like TS_ObjectList, but this list owns the objects inserted in the list.
  When objects are deleted, or when the list is freed or cleared, the owned
  objects are freed. To remove an object without freeing it, set the object
  to nil, and then delete it. }

{ If the capacity is set to be smaller, free the objects that are implicitly deleted. }
procedure TS_OwnerList.SetCapacity(NewCapacity: LongInt);
var
  I: LongInt;
begin
  if NewCapacity < Capacity then
    for I := Capacity to NewCapacity-1 do
    begin
      Objects[I].Free;
      Objects[I] := nil;
    end;
  inherited SetCapacity(NewCapacity);
end;

{ If the Count is set to be smaller, free the objects that are implicitly deleted. }
procedure TS_OwnerList.SetCount(NewCount: LongInt);
var
  I: LongInt;
begin
  if NewCount < Count then
    for I := Count to NewCount-1 do
    begin
      Objects[I].Free;
      Objects[I] := nil;
    end;
  inherited SetCount(NewCount);
end;

{ Free all objects when clearing the list. }
procedure TS_OwnerList.Clear;
var
  I: LongInt;
begin
  for I := 0 to Count-1 do
  begin
    Objects[I].Free;
    Objects[I] := nil;
  end;
  inherited Clear;
end;

{ Since the object is freed when it is deleted, the only meaningful
  value to return is nil. }
function TS_OwnerList.Delete(Index: LongInt): TObject;
var
  Obj: TObject;
begin
  Obj := inherited Delete(Index);
  Obj.Free;
  Result := nil;
end;

end.
