unit S_StrLst;

{ Secrets of Delphi 2, by Ray Lischner. (1996, Waite Group Press).
  Chapter 6: The Elegant, Versatile TStrings Class
  Copyright  1996 The Waite Group, Inc. }

{ Huge strings lists for Delphi 1.0 }

interface

uses Classes, SysUtils, WinTypes, WinProcs, S_List;

type
{$ifdef WIN32}
  TS_StringList = class(TStringList)
  public
    { For compatibility with the Delphi 1.0 version of TS_StringList. }
    procedure AddTStrings(Strings: TStrings); virtual;
  end;
  TS_Strings = TStringList;
{$else}
  { List of PStrItem pointers. }
  TS_StrItemList = class(TS_CustomList)
  protected
    function GetStrItem(Index: LongInt): PStrItem;
    procedure SetStrItem(Index: LongInt; Item: PStrItem);
    procedure InsertItem(Index: LongInt; const Str: string; Obj: TObject);
    procedure SetItem(Index: LongInt; const Str: string);
  public
    procedure Clear; override;
    function Delete(Index: LongInt): Pointer; override;
    property StrItems[Index: LongInt]: PStrItem read GetStrItem write SetStrItem; default;
  end;

  { Implementing a huge string list introduces a dilemma. Do you
    derive the class from TStrings, thereby making it possible, in a limited
    way, to use the huge string list as a Delphi string list, and be
    able to assign it to other TStrings properties, etc., or do you create
    a completely independent class (like TS_CustomList)?  Let's do both.
    TS_StringList is an independent class, with an interface that looks
    and acts like TStringList, but it does not derive from TStrings.

    TS_Strings derives from TStrings, and implements the appropriate
    protocol, but adds a parallel protocol to allow access to the items
    with indexes past 16K. }
  TS_StringList = class(TPersistent)
  private
    fUpdateCount: Cardinal;
    fList: TS_StrItemList;
    fDuplicates: TDuplicates;
    fSorted: Boolean;
    fOnChange: TNotifyEvent;
    fOnChanging: TNotifyEvent;
  protected
    procedure Changed; virtual;
    procedure Changing; virtual;
    procedure DefineProperties(Filer: TFiler); override;
    function GetCount: LongInt; virtual;
    function GetName(Index: LongInt): string; virtual;
    function GetString(Index: LongInt): string; virtual;
    function GetObject(Index: LongInt): TObject; virtual;
    function GetValue(const Name: string): string;
    procedure SetValue(const Name, Value: string);
    function Partition(Lo, Hi: LongInt): LongInt; virtual;
    procedure ReadData(Reader: TReader);
    procedure WriteData(Writer: TWriter);
    procedure QuickSort(Lo, Hi: LongInt); virtual;
    procedure SlowSort(Lo, Hi: LongInt); virtual;
    procedure PutString(Index: LongInt; const Str: string); virtual;
    procedure PutObject(Index: LongInt; Obj: TObject); virtual;
    procedure SetSorted(Sorted: Boolean); virtual;
    procedure SetUpdateState(Updating: Boolean); virtual;
    property UpdateCount: Cardinal read fUpdateCount;
    property Items: TS_StrItemList read fList;
  public
    constructor Create; virtual;
    destructor Destroy; override;

    function Add(const S: string): LongInt; virtual;
    function AddObject(const S: string; Obj: TObject): LongInt; virtual;
    procedure AddStrings(Strings: TS_StringList); virtual;
    procedure AddTStrings(Strings: TStrings); virtual;
    procedure Assign(Source: TPersistent); virtual;
    procedure BeginUpdate;
    procedure Clear; virtual;
    procedure Delete(Index: LongInt); virtual;
    procedure EndUpdate;
    procedure Exchange(Index1, Index2: LongInt); virtual;
    function Find(const S: string; var Index: LongInt): Boolean; virtual;
    function GetText: PChar; virtual;
    function IndexOf(const S: string): LongInt; virtual;
    function IndexOfObject(Obj: TObject): LongInt;
    procedure Insert(Index: LongInt; const S: string); virtual;
    procedure InsertObject(Index: LongInt; const S: string; Obj: TObject);
    procedure LoadFromFile(const FileName: string);
    procedure LoadFromStream(Stream: TStream); virtual;
    procedure Move(CurIndex, NewIndex: LongInt); virtual;
    procedure SaveToFile(const FileName: string);
    procedure SaveToStream(Stream: TStream); virtual;
    procedure SetText(Text: PChar); virtual;
    procedure Sort; virtual;

    property Count: LongInt read GetCount;
    property Duplicates: TDuplicates read fDuplicates write fDuplicates;
    property Sorted: Boolean read FSorted write SetSorted;
    property Objects[Index: LongInt]: TObject read GetObject write PutObject;
    property Names[Index: LongInt]: string read GetName;
    property Strings[Index: LongInt]: string read GetString write PutString; default;
    property Values[const Name: string]: string read GetValue write SetValue;
    property OnChange: TNotifyEvent read fOnChange write fOnChange;
    property OnChanging: TNotifyEvent read fOnChanging write fOnChanging;
  end;

  TS_Strings = class(TStrings)
  private
    fList: TS_StringList;
    function GetDuplicates: TDuplicates;
    function GetSorted: Boolean;
    function GetOnChange: TNotifyEvent;
    function GetOnChanging: TNotifyEvent;
    procedure SetOnChange(OnChange: TNotifyEvent);
    procedure SetOnChanging(OnChanging: TNotifyEvent);
    procedure SetDuplicates(Duplicates: TDuplicates);
    procedure SetSorted(Sorted: Boolean);
  protected
    function Get(Index: Integer): string; override;
    function GetCount: Integer; override;
    function GetObject(Index: Integer): TObject; override;
    procedure Put(Index: Integer; const S: string); override;
    procedure PutObject(Index: Integer; Obj: TObject); override;
    procedure SetUpdateState(Updating: Boolean); override;
    procedure ReadData(Reader: TReader); virtual;
    procedure WriteData(Writer: TWriter); virtual;
    procedure DefineProperties(Filer: TFiler); override;
    function Trunc(Index: LongInt): Integer;
  public
    constructor Create;
    destructor Destroy; override;

    function Add(const S: string): Integer; override;
    function AddObject(const S: string; Obj: TObject): Integer; override;
    procedure Assign(Source: TPersistent); override;
    procedure Clear; override;
    procedure Delete(Index: Integer); override;
    procedure Exchange(Index1, Index2: Integer); override;
    function Find(const S: string; var Index: Integer): Boolean; virtual;
    function GetText: PChar; override;
    function IndexOf(const S: string): Integer; override;
    procedure Insert(Index: Integer; const S: string); override;
    procedure LoadFromStream(Stream: TStream); override;
    procedure Move(CurIndex, NewIndex: Integer); override;
    procedure SaveToStream(Stream: TStream); override;
    procedure SetText(Text: PChar); override;
    procedure Sort; virtual;

    property Count: Integer read GetCount;
    property Duplicates: TDuplicates read GetDuplicates write SetDuplicates;
    property Sorted: Boolean read GetSorted write SetSorted;
    property OnChange: TNotifyEvent read GetOnChange write SetOnChange;
    property OnChanging: TNotifyEvent read GetOnChanging write SetOnChanging;
  end;

{ The next two routines are useful if you must use the
  Delphi 1.0 implementation of TStrings, which does not have
  as much support for Names as Delphi 2.0. Use these routines
  to fill in the gaps. }

{ Look up a name in a string list. Return the index or -1.
  A name is that part of a string before an equal sign.
  Case is not significant. }
function IndexOfName(Strings: TStrings; const Name: string): Integer;

{ Look up name at position Index in Strings. A name is that part
  of a string before an equal sign. }
function GetName(Strings: TStrings; Index: Integer): string;

{ The SortThreshold is the division between quick sort and an iteration sort. }
const
  SortThreshold: Cardinal = 20;

{$endif}

{ The next two routines are most useful for Delphi 1.0, but there
  is no reason to be exclusive. Therefore, implement them for Delphi 2.0, too. }

{ CopyLine copies the string Str, to Text, and then appends
  a CR-LF (Carriage Return-Line Feed) for a line ending.
  It returns a pointer just past the LF, in preparation
  for copying the next line.  The memory pointed to by
  Text can be larger than 64K. }
function CopyLine(Text: PChar; const Str: string): PChar;

{ ExtractLine copies a line of text from Text to S. A line
  ends with a CR-LF, a plain CR or a plain LF, or the end
  of the string, which can be marked by a Ctrl+Z or a zero
  byte.  It returns a pointer to the beginning of the next
  line, or the Ctrl+Z or zero terminating character.
  The memory pointed to by Text can be larger than 64K. }
function ExtractLine(Text: PChar; var Str: string): PChar;


implementation

uses Consts, S_Huge;

const
  CR = #13;
  EOF = ^Z;  { Sometimes used to mark end-of-file in DOS text files. }
  LF = #10;
  NUL = #0;

{ Look up a name in a string list. Return the index or -1.
  A name is that part of a string before an equal sign.
  Case is not significant. }
function IndexOfName(Strings: TStrings; const Name: string): Integer;
var
  Equal: Integer;   { index of equal size }
begin
  for Result := 0 to Strings.Count-1 do
  begin
    { Extract the name part of a string. If there is no equal sign,
      then there is no name. }
    Equal := Pos('=', Strings[Result]);
    if (Equal <> 0) and
       (CompareText(Name, Copy(Strings[Result], 1, Equal-1)) = 0) then
    begin
      Exit;
    end;
  end;
  { No match. }
  Result := -1;
end;

{ Look up name at position Index in Strings. A name is that part
  of a string before an equal sign. }
function GetName(Strings: TStrings; Index: Integer): string;
var
  Equal: Integer;
  Str: string;
begin
  Str := Strings[Index];
  { Look for the equal sign. }
  Equal := Pos('=', Str);
  if Equal = 0 then
    { No equal sign, so return an empty string. }
    Result := ''
  else
    { Return the name, without the equal sign. }
    Result := Copy(Str, 1, Equal-1);
end;


{$ifdef WIN32}

{ For compatibility with the Delphi 1.0 version of TS_StringList. }
procedure TS_StringList.AddTStrings(Strings: TStrings);
begin
  AddStrings(Strings)
end;

{$else}

{ TS_StrItemList }
{ Custom list of PStrItem pointers. Make it easier to work with string items.
  The TStrItem record and related types and routines can be found in the Classes unit. }
{ Get a string item. }
function TS_StrItemList.GetStrItem(Index: LongInt): PStrItem;
begin
  Result := Items[Index]
end;

{ Set a string item. Use this only when you take care to dispose of
  the previous string item. (See SetItem, for example.) }
procedure TS_StrItemList.SetStrItem(Index: LongInt; Item: PStrItem);
begin
  Items[Index] := Item;
end;

{ Insert a new string item. }
procedure TS_StrItemList.InsertItem(Index: LongInt; const Str: string; Obj: TObject);
begin
  Insert(Index, NewStrItem(Str, Obj));
end;

{ Set an existing string item, preserving the associated object. }
procedure TS_StrItemList.SetItem(Index: LongInt; const Str: string);
var
  OldItem: PStrItem;
begin
  OldItem := StrItems[Index];
  StrItems[Index] := NewStrItem(Str, OldItem^.fObject);
  DisposeStrItem(OldItem);
end;

{ Clear the list by freeing all the string items. }
procedure TS_StrItemList.Clear;
var
  I: LongInt;
begin
  for I := 0 to Count-1 do
    DisposeStrItem(StrItems[I]);
  inherited Clear;
end;

{ Delete a string item. }
function TS_StrItemList.Delete(Index: LongInt): Pointer;
var
  Item: PStrItem;
begin
  Item := inherited Delete(Index);
  DisposeStrItem(Item);
  Result := nil;
end;


{ TS_StringList }
constructor TS_StringList.Create;
begin
  inherited Create;
  fList := TS_StrItemList.Create;
end;

destructor TS_StringList.Destroy;
begin
  Clear;
  fList.Free;
  inherited Destroy;
end;

function TS_StringList.GetCount: LongInt;
begin
  Result := fList.Count
end;

function TS_StringList.GetString(Index: LongInt): string;
begin
  Result := Items[Index]^.fString
end;

function TS_StringList.GetObject(Index: LongInt): TObject;
begin
  Result := Items[Index]^.fObject
end;

function TS_StringList.GetName(Index: LongInt): string;
var
  Equal: Integer;
  Str: string;
begin
  Str := GetString(Index);
  { Look for the equal sign. }
  Equal := Pos('=', Str);
  if Equal = 0 then
    { No equal sign, so return an empty string. }
    Result := ''
  else
    { Return the name, without the equal sign. }
    Result := Copy(Str, 1, Equal-1);
end;

procedure TS_StringList.PutString(Index: LongInt; const Str: string);
begin
  if Sorted then
    raise EListError.CreateRes(SSortedListError);
  fList.SetItem(Index, Str);
end;

procedure TS_StringList.PutObject(Index: LongInt; Obj: TObject);
begin
  Items[Index]^.fObject := Obj
end;

{ An unnested BeginUpdate sets the update state, and when
  the nesting returns to zero in EndUpdate, that means
  we are done making changes. }
procedure TS_StringList.SetUpdateState(Updating: Boolean);
begin
  if Updating then
    Changing
  else
    Changed
end;

{ The list has changed. }
procedure TS_StringList.Changed;
begin
  if Assigned(fOnChange) then
    fOnChange(Self)
end;

{ The list is about to change. }
procedure TS_StringList.Changing;
begin
  if Assigned(fOnChanging) then
    fOnChanging(Self)
end;

{ Clearing the Sorted flag leaves the order alone. Setting
  the Sorted flag immediately sorts the list. }
procedure TS_StringList.SetSorted(Sorted: Boolean);
begin
  if Sorted <> fSorted then
  begin
    if Sorted then
      Sort;
    fSorted := Sorted;
  end;
end;

{ Add a string, allocating room for it. If the list is sorted,
  and the string is already in the list, then what we do
  depends on the Duplicates property. }
function TS_StringList.Add(const S: string): LongInt;
begin
  if not Sorted then
    Result := FList.Count
  else
    if Find(S, Result) then
      case Duplicates of
        dupIgnore: Exit;
        dupError: raise EListError.CreateRes(SDuplicateString);
        dupAccept: ; { nothing, so we can insert the string below }
      end;
  Changing;
  fList.InsertItem(Result, S, nil);
  Changed;
end;

{ Add a string and an associated object. }
function TS_StringList.AddObject(const S: string; Obj: TObject): LongInt;
begin
  Result := Add(S);
  Objects[Result] := Obj;
end;

{ Add all the strings in the TS_StringList. }
procedure TS_StringList.AddStrings(Strings: TS_StringList);
var
  I: LongInt;
begin
 BeginUpdate;
 try
   for I := 0 to Strings.Count-1 do
     AddObject(Strings.Strings[I], Strings.Objects[I]);
 finally
   EndUpdate;
 end;
end;

{ Add all the strings in a TStrings. This makes it convenient
  to copy the strings from, say, a TMemo or TListBox into a
  TS_StringList. }
procedure TS_StringList.AddTStrings(Strings: TStrings);
var
  I: Integer;
begin
 BeginUpdate;
 try
   for I := 0 to Strings.Count-1 do
     AddObject(Strings.Strings[I], Strings.Objects[I]);
 finally
   EndUpdate;
 end;
end;

{ Accept assignments from a TStrings, for the user's convenience, but
  if the TStrings is a TS_Strings, then copy all the strings
  by accessing the TS_StringList field. }
procedure TS_StringList.Assign(Source: TPersistent);
begin
  if Source is TS_Strings then
    Assign(TS_Strings(Source).fList)
  else if Source is TStrings then
  begin
    BeginUpdate;
    try
      Clear;
      AddTStrings(TStrings(Source));
    finally
      EndUpdate;
    end;
  end
  else if Source is TS_StringList then
  begin
    BeginUpdate;
    try
      Clear;
      AddStrings(TS_StringList(Source));
    finally
      EndUpdate;
    end;
  end
  else
    inherited Assign(Source);
end;

{ Begin a series of changes to the list that are to be grouped
  as a single change. Do not signal the change if nested in other
  BeginUpdate/EndUpdate calls. }
procedure TS_StringList.BeginUpdate;
begin
  if fUpdateCount = 0 then
    SetUpdateState(True);
  Inc(fUpdateCount);
end;

procedure TS_StringList.Clear;
begin
  Changing;
  fList.Clear;
  Changed;
end;

{ Define a pseudo-property, so we can save a huge string list in a form file. }
procedure TS_StringList.DefineProperties(Filer: TFiler);
begin
  Filer.DefineProperty('Strings', ReadData, WriteData, Count>0);
end;

{ Read strings from a TReader, e.g., a form file. }
procedure TS_StringList.ReadData(Reader: TReader);
begin
  Reader.ReadListBegin;
  BeginUpdate;
  try
    Clear;
    while not Reader.EndOfList do
      Add(Reader.ReadString);
    Reader.ReadListEnd;
  finally
    EndUpdate;
  end;
end;

{ Write strings to a TWriter, e.g., a form file }
procedure TS_StringList.WriteData(Writer: TWriter);
var
  I: LongInt;
begin
  Writer.WriteListbegin;
  for I := 0 to Count-1 do
    Writer.WriteString(Items[I]^.fString);
  Writer.WriteListEnd;
end;

procedure TS_StringList.Delete(Index: LongInt);
begin
  Changing;
  fList.Delete(Index);
  Changed;
end;

{ End a series of changes to the list, which were grouped as a single
  change. Do not signal the change if nested in other BeginUpdate/
  EndUpdate calls. }
procedure TS_StringList.EndUpdate;
begin
  Dec(fUpdateCount);
  if fUpdateCount = 0 then
    SetUpdateState(False);
end;

{ Exchange the items at Index1 and Index2. Clearly, we can't
  do this if the list is sorted. }
procedure TS_StringList.Exchange(Index1, Index2: LongInt);
begin
  if Sorted then
    raise EListError.CreateRes(SSortedListError);
  Changing;
  fList.Exchange(Index1, Index2);
  Changed;
end;

{ Use binary search to find something in a sorted list. If the list
  is unsorted, then switch to IndexOf. }
function TS_StringList.Find(const S: string; var Index: LongInt): Boolean;
var
  Lo, Hi, Mid: LongInt;
  C: Integer;
begin
  if not Sorted then
  begin
    Index := IndexOf(S);
    Result := Index >= 0;
  end
  else
  begin
    Result := False;
    Lo := 0;
    Hi := Count-1;
    while Lo <= Hi do
    begin
      Mid := (Lo + Hi) div 2;
      C := CompareText(Items[Mid]^.fString, S);
      if C < 0 then
        Lo := Mid + 1
      else
      begin
        Hi := Mid - 1;
        if C = 0 then
        begin
          Result := True;
          if Duplicates <> dupAccept then
            Lo := Mid;
        end;
      end;
    end;
    Index := Lo;
  end;
end;

{ Copy all the strings into one, big string, ending each
  string with a CR-LF.  End the entire string with a zero byte.
  Since this is a Huge string list, allow the resulting string
  to be any size, so long as there is enough memory for it. }
function TS_StringList.GetText: PChar;
var
  I, Size: LongInt;
  P: PChar;
begin
  { First determine how much memory is needed for the string. }
  Size := 1;           { for the trailing #0 byte }
  for I := 0 to Count-1 do
    Size := Size + Length(Items[I]^.fString) + 2;  { CR-LF }

  Result := GlobalAllocPtr(GMem_Fixed or GMem_Share, Size);
  if Result = nil then
    OutOfMemoryError;

  { Now copy all the strings into the result. }
  P := Result;
  for I := 0 to Count-1 do
    P := CopyLine(P, Items[I]^.fString);
  P[0] := #0;
end;

{ Get a value from a string. String/value pairs are expressed
  like environment variables, e.g., Name=Value.  Look for the
  equal sign, and copy everything that comes after it. }
function TS_StringList.GetValue(const Name: string): string;
var
  Equal: Integer;
  I: LongInt;
begin
  for I := 0 to Count-1 do
  begin
    Equal := Pos('=', Items[I]^.fString);
    if (Equal <> 0) and (CompareText(Name, Copy(Items[I]^.fString, 1, Equal-1)) = 0) then
    begin
      Result := Copy(Items[I]^.fString, Equal+1, 255);
      Exit;
    end;
  end;
  Result := '';
end;

{ Set the value of a string/value pair. Look for the equal sign,
  and change what comes after it. An empty string deletes the item.
  If the name is not in the list, then add a new item. }
procedure TS_StringList.SetValue(const Name, Value: string);
var
  Equal: Integer;
  I: LongInt;
begin
  for I := 0 to Count-1 do
  begin
    Equal := Pos('=', Items[I]^.fString);
    if (Equal <> 0) and (CompareText(Name, Copy(Items[I]^.fString, 1, Equal-1)) = 0) then
    begin
      if Value = '' then
        Delete(I)
      else
        Strings[I] := Name + '=' + Value;
      Exit;
    end;
  end;
  if Value <> '' then
    Add(Name + '=' + Value);
end;

{ Return the index of a string. If the list is sorted, then
  call Find, to use a binary search. }
function TS_StringList.IndexOf(const S: string): LongInt;
begin
  if Sorted then
  begin
    if Find(S, Result) then
      Exit;
  end
  else
  begin
    for Result := 0 to Count-1 do
      if CompareText(S, Items[Result]^.fString) = 0 then
        Exit;
  end;
  Result := -1;
end;

{ Return the index of an object. }
function TS_StringList.IndexOfObject(Obj: TObject): LongInt;
begin
  for Result := 0 to Count-1 do
    if Objects[Result] = Obj then
      Exit;
  Result := -1;
end;

{ Insert at a specific index. Clearly, we cannot do this to a sorted list. }
procedure TS_StringList.Insert(Index: LongInt; const S: string);
begin
  Changing;
  if Sorted then
    raise EListError.CreateRes(SSortedListError);
  fList.InsertItem(Index, S, nil);
  Changed;
end;

{ Insert a string and an associated object. }
procedure TS_StringList.InsertObject(Index: LongInt; const S: string; Obj: TObject);
begin
  Insert(Index, S);
  Objects[Index] := Obj;
end;

{ Load a new string list from a file }
procedure TS_StringList.LoadFromFile(const FileName: string);
var
  Stream: TFileStream;
begin
  Stream := TFileStream.Create(FileName, fmOpenRead);
  try
    LoadFromStream(Stream);
  finally
    Stream.Free;
  end;
end;

{ Load a new string list from a stream. Buffer the stream,
  for improved I/O performance.  Line endings separate strings;
  use DOS (CR-LF), Mac (CR), or Unix (LF) style line endings.
  The file ends with a zero byte, a Ctrl+Z, or more commonly,
  reaching the end of the file. }
procedure TS_StringList.LoadFromStream(Stream: TStream);
const
  BufSize = 8192;
var
  Len: Integer;
  Buffer, BufPtr, BufEnd, BufLast: PChar;
  LineLast: PChar;
  S: string;
begin
  BeginUpdate;
  try
    Clear;
    GetMem(Buffer, BufSize);
    try
      BufEnd := Buffer + BufSize;
      BufLast := BufEnd;
      LineLast := BufLast;
      repeat
        { Copy the last partial line to the start of the buffer }
        Len := BufLast - LineLast;
        if Len <> 0 then
          System.Move(LineLast^, Buffer^, Len);
        BufLast := Buffer + Len;
        { fill the rest of the buffer }
        BufLast := BufLast + Stream.Read(BufLast^, BufEnd - BufLast);
        { find the last end-of-line in the buffer }
        LineLast := BufLast;
        while LineLast > Buffer do
        begin
          Dec(LineLast);
          if LineLast^ in [NUL, LF, CR, EOF] then
            Break;
        end;
        if LineLast^ = CR then
          Inc(LineLast);
        if LineLast^ = LF then
          Inc(LineLast);
        { Now split the buffer into lines }
        BufPtr := Buffer;
        while (BufPtr < LineLast) and (LineLast > Buffer) do
        begin
          BufPtr := ExtractLine(BufPtr, S);
          Add(S);
        end;
      until BufLast = Buffer;
    finally
      FreeMem(Buffer, BufSize);
    end;
  finally
    EndUpdate;
  end;
end;

{ Move an item from CurIndex to NewIndex. Don't do it if the list is sorted. }
procedure TS_StringList.Move(CurIndex, NewIndex: LongInt);
begin
  if Sorted then
    raise EListError.CreateRes(SSortedListError);
  Changing;
  fList.Move(CurIndex, NewIndex);
  Changed;
end;

{ Parition for quick sort. Override this if you want to use a more
  sophisticated algorithm for finding the pivot. }
function TS_StringList.Partition(Lo, Hi: LongInt): LongInt;
var
  Pivot: PStrItem;
begin
  { Choose a pivot.  We can do a better job, but this will suffice for now. }
  Pivot := Items[(Lo + Hi) div 2];
  while true do
  begin
    while CompareText(Items[Lo]^.fString, Pivot^.fString) < 0 do
      Inc(Lo);
    while CompareText(Items[Hi]^.fString, Pivot^.fString) > 0 do
      Dec(Hi);
    if Lo < Hi then
    begin
      fList.Exchange(Lo, Hi);
      Inc(Lo);
      Dec(Hi);
    end
    else if Lo >= Hi then
      Break;
  end;
  Result := Hi;
end;

{ Slow sort for small lists. }
procedure TS_StringList.SlowSort(Lo, Hi: LongInt);
var
  I, J: LongInt;
begin
  for I := Lo to Hi do
    for J := I+1 to Hi do
      if CompareText(Items[I]^.fString, Items[J]^.fString) > 0 then
        fList.Exchange(I, J);
end;

{ Quick sort. Use tail recursion, to minimize the stack growth.
  Use a slow sort for small lists. The name, "SlowSort," is merely an
  indicator that of its asymptotic performance. Typically, SlowSort is
  a bubble sort or insertion sort, but one can override it for a Shell sort,
  for example. Hence, the generic name. }
procedure TS_StringList.QuickSort(Lo, Hi: LongInt);
var
  Mid: LongInt;
  I, L, H: LongInt;
begin
  L := Lo;
  H := Hi;
  { Force tail recursion to sort the larger partition. This keeps the stack
    size from growing too much, and helps the speed a little. }
  while Lo < Hi do
    if Hi-Lo < SortThreshold then
    begin
      SlowSort(Lo, Hi);
      Break;
    end
    else
    begin
      Mid := Partition(Lo, Hi);
      if Mid-Lo > Hi-Mid then
      begin
        QuickSort(Mid+1, Hi);
        Hi := Mid;
      end
      else
      begin
        QuickSort(Lo, Mid);
        Lo := Mid+1;
      end;
    end;
end;

{ Save all the strings to a file. }
procedure TS_StringList.SaveToFile(const FileName: string);
var
  Stream: TFileStream;
begin
  Stream := TFileStream.Create(FileName, fmCreate);
  try
    SaveToStream(Stream);
  finally
    Stream.Free;
  end;
end;

{ Save all the strings to a stream, one per line. }
procedure TS_StringList.SaveToStream(Stream: TStream);
const
  BufSize = 8192;
var
  I: LongInt;
  Buffer, BufPtr: PChar;
begin
  GetMem(Buffer, BufSize);
  try
    BufPtr := Buffer;
    for I := 0 to Count-1 do
    begin
      { Is there room for a string + CRLF? }
      if BufPtr - Buffer >= BufSize - 256 then
      begin
        Stream.WriteBuffer(Buffer[0], BufPtr - Buffer);
        BufPtr := Buffer;
      end;
      StrPCopy(BufPtr, Items[I]^.fString);
      Inc(BufPtr, Length(Items[I]^.fString));
      BufPtr[0] := CR;
      BufPtr[1] := LF;
      Inc(BufPtr, 2);
    end;
    Stream.WriteBuffer(Buffer[0], BufPtr - Buffer);
  finally
    FreeMem(Buffer, BufSize);
  end;
end;

{ Set all the strings by reading lines from Text. }
procedure TS_StringList.SetText(Text: PChar);
var
  S: string;
begin
  BeginUpdate;
  try
    Clear;
    while not (Text^ in [NUL, EOF]) do
    begin
      Text := ExtractLine(Text, S);
      Add(S);
    end;
  finally
    EndUpdate;
  end;
end;

{ Sort the list, if it is not already sorted. }
procedure TS_StringList.Sort;
begin
  if not Sorted and (Count > 1) then
  begin
    Changing;
    QuickSort(0, Count-1);
    Changed;
  end;
end;


{ TS_Strings }
{ In many cases, we can hide the long indexes, but sometimes,
  we cannot, e.g., IndexOf. In these situations, we let Delphi
  catch an overflow error. }
{$Q+}
constructor TS_Strings.Create;
begin
  fList := TS_StringList.Create;
  inherited Create;
end;

destructor TS_Strings.Destroy;
begin
  fList.Free;
  inherited Destroy;
end;

function TS_Strings.GetDuplicates: TDuplicates;
begin
  Result := fList.Duplicates
end;

function TS_Strings.GetSorted: Boolean;
begin
  Result := fList.Sorted
end;

function TS_Strings.GetOnChange: TNotifyEvent;
begin
  Result := fList.OnChange
end;

function TS_Strings.GetOnChanging: TNotifyEvent;
begin
  Result := fList.OnChanging
end;

procedure TS_Strings.SetOnChange(OnChange: TNotifyEvent);
begin
  fList.OnChange := OnChange;
end;

procedure TS_Strings.SetOnChanging(OnChanging: TNotifyEvent);
begin
  fList.OnChanging := OnChanging
end;

procedure TS_Strings.SetDuplicates(Duplicates: TDuplicates);
begin
  fList.Duplicates := Duplicates
end;

procedure TS_Strings.SetSorted(Sorted: Boolean);
begin
  fList.Sorted := Sorted
end;

function TS_Strings.Get(Index: Integer): string;
begin
  Result := fList[Index]
end;

function TS_Strings.GetCount: Integer;
begin
  Result := Trunc(fList.Count);
end;

function TS_Strings.GetObject(Index: Integer): TObject;
begin
  Result := fList.Objects[Index]
end;

procedure TS_Strings.Put(Index: Integer; const S: string);
begin
  fList[Index] := S
end;

procedure TS_Strings.PutObject(Index: Integer; Obj: TObject);
begin
  fList.Objects[Index] := Obj
end;

procedure TS_Strings.SetUpdateState(Updating: Boolean);
begin
  if Updating then
    fList.BeginUpdate
  else
    fList.EndUpdate
end;

procedure TS_Strings.ReadData(Reader: TReader);
begin
  fList.ReadData(Reader)
end;

procedure TS_Strings.WriteData(Writer: TWriter);
begin
  fList.WriteData(Writer)
end;

procedure TS_Strings.DefineProperties(Filer: TFiler);
begin
  Filer.DefineProperty('Strings', ReadData, WriteData, Count>1)
end;

function TS_Strings.Add(const S: string): Integer;
begin
  Result := Trunc(fList.Add(S))
end;

function TS_Strings.AddObject(const S: string; Obj: TObject): Integer;
begin
  Result := Trunc(fList.AddObject(S, Obj));
end;

procedure TS_Strings.Assign(Source: TPersistent);
begin
  fList.Assign(Source)
end;

procedure TS_Strings.Clear;
begin
  fList.Clear
end;

procedure TS_Strings.Delete(Index: Integer);
begin
  fList.Delete(Index)
end;

procedure TS_Strings.Exchange(Index1, Index2: Integer);
begin
  fList.Exchange(Index1, Index2)
end;

function TS_Strings.Find(const S: string; var Index: Integer): Boolean;
var
  HugeIndex: LongInt;
begin
  Result := fList.Find(S, HugeIndex);
  Index := Trunc(HugeIndex);
end;

function TS_Strings.GetText: PChar;
begin
  Result := fList.GetText;
end;

function TS_Strings.IndexOf(const S: string): Integer;
begin
  Result := fList.IndexOf(S)
  { Let Delphi catch an overflow error if the Index > MaxInt }
end;

procedure TS_Strings.Insert(Index: Integer; const S: string);
begin
  fList.Insert(Index, S)
end;

procedure TS_Strings.LoadFromStream(Stream: TStream);
begin
  fList.LoadFromStream(Stream)
end;

procedure TS_Strings.Move(CurIndex, NewIndex: Integer);
begin
  fList.Move(CurIndex, NewIndex);
end;

procedure TS_Strings.SaveToStream(Stream: TStream);
begin
  fList.SaveToStream(Stream)
end;

procedure TS_Strings.SetText(Text: PChar);
begin
  fList.SetText(Text)
end;

procedure TS_Strings.Sort;
begin
  fList.Sort
end;

function TS_Strings.Trunc(Index: LongInt): Integer;
begin
  if Index > MaxInt then
    Result := MaxInt
  else
    Result := Integer(Index)
end;
{$endif}

{ Copy the string, Str, to Text, and append a CR-LF.
  Even though this function copies no more than 256 bytes, use
  HMemCopy, since Text might cross a segment boundary. }
function CopyLine(Text: PChar; const Str: string): PChar;
begin
  HMemCpy(Text, @Str[1], Length(Str));
  Result := HugeOffset(Text, Length(Str));
  { Now append the CR-LF }
  Result^ := CR;
  Result := HugeOffset(Result, 1);
  Result^ := LF;
  Result := HugeOffset(Result, 1);
end;

{ Extract the next line from Text, and copy it into Str.
  Advance Text past the end of the string. A string ends with
  a line ending (CRLF, CR, or LF), at the end of the string
  (Ctrl+Z or zero byte), or at 255 bytes in Delphi 1.0. A string
  can cross a segment boundary, so SetString cannot be used in
  Delphi 1.0. The line ending is not included in the string Str. }
function ExtractLine(Text: PChar; var Str: string): PChar;
var
  Len: Integer;
begin
  Len := 0;
  Result := Text;
  while not (Result^ in [NUL, LF, CR, EOF]) do
  begin
    Result := HugeOffset(Result, 1);
    Inc(Len);
{$ifndef WIN32}
    if Len = 255 then
      Break;
{$endif}
  end;
{$ifdef WIN32}
  SetString(Str, Text, Len);
{$else}
  { Cannot use SetString, since Text might cross a segment boundary. }
  Str[0] := Chr(Len);
  HMemCpy(@Str[1], Text, Len);
{$endif}
  if Result^ = CR then
    Result := HugeOffset(Result, 1);
  if Result^ = LF then
    Result := HugeOffset(Result, 1);
end;

end.
