//{$A+,B-,D+,F-,G+,I+,K+,L+,N+,P+,Q-,R-,S+,T-,V-,W-,X+,Y+}
unit UjetButton;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Buttons, DsgnIntf;

type
  TAboutJetButton = class(TPropertyEditor)
  public
    procedure Edit; override;
    function GetAttributes: TPropertyAttributes; override;
    function GetValue: string; override;
  end;

  TJetButton = class(TGraphicControl)
  private
    AboutBox: TAboutJetButton;
    FAutoSize: Boolean;
    FBitmapUp: TBitmap;
    FBitmapDown: TBitmap;
    FBitmapOver: TBitmap;
    FBitmapDisabled: TBitmap;
    TempBitmap : TBitmap;
    MouseOver : Boolean;
    FTransparentColor : TColor;
    procedure AdjustBounds;
    procedure BitmapUpChanged(Sender: TObject);
    procedure BitmapDownChanged(Sender: TObject);
    procedure BitmapOverChanged(Sender: TObject);
    procedure BitmapDisabledChanged(Sender: TObject);
    procedure SetBitmapDown(Value: TBitmap);
    procedure SetBitmapUp(Value: TBitmap);
    procedure SetBitmapOver(Value: TBitmap);
    procedure SetBitmapDisabled(Value: TBitmap);
    procedure CMDialogChar(var Message: TCMDialogChar); message CM_DIALOGCHAR;
    procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
    procedure CMTextChanged(var Message: TMessage); message CM_TEXTCHANGED;
    procedure CMSysColorChange(var Message: TMessage); message CM_SYSCOLORCHANGE;
    procedure CMMouseEnter(var Message: TMessage); message CM_MOUSEENTER;
    procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Click; override;
    procedure Invalidate; override;
    function PtInMask(const X, Y: Integer): Boolean;
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
  protected
    FState: TButtonState;
    FEnabled : boolean;
    function GetPalette: HPALETTE; override;
    procedure SetEnabled(Value : Boolean);
    procedure DrawButtonText(Canvas: TCanvas; const Caption: String; TextBounds: TRect; State: TButtonState);
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;  X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState;  X, Y: Integer); override;
    procedure Paint; override;
    procedure Loaded; override;
  published
    property About: TAboutJetButton read AboutBox write AboutBox;
    property TransparentColor : TColor read FTransparentColor write FTransparentColor;
    property BitmapUp: TBitmap read FBitmapUp write SetBitmapUp;
    property BitmapDown: TBitmap read FBitmapDown write SetBitmapDown;
    property BitmapOver: TBitmap read FBitmapOver write SetBitmapOver;
    property BitmapDisabled: TBitmap read FBitmapDisabled write SetBitmapDisabled;
    property Caption;
    property Enabled : Boolean read FEnabled write SetEnabled;
    property Font;
    property ShowHint;
    property Visible;
    property OnClick;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

procedure Register;

implementation

function MakeMask(const ColorBmp: TBitmap; TransparentColor: TColor): TBitmap;
var Temp: TRect;
    OldBkColor: TColorRef;
    TmpBitmap : Tbitmap;
begin
  Makemask := nil;
  TmpBitmap := TBitmap.Create;
  try
    TmpBitmap.Monochrome := True;
    TmpBitmap.Width := ColorBmp.Width;
    TmpBitmap.Height := ColorBmp.Height;
    OldBkColor := SetBkColor(ColorBmp.Canvas.Handle, ColorToRGB(TransparentColor));
    Temp := Rect(0, 0, ColorBmp.Width, ColorBmp.Height);
    TmpBitmap.Canvas.CopyMode := cmSrcCopy;
    TmpBitmap.Canvas.CopyRect(Temp, ColorBmp.Canvas, Temp);
    SetBkColor(ColorBmp.Canvas.Handle, OldBkColor);
    MakeMask := TmpBitmap;
  except
    TmpBitmap.Free;
  end;
end;
////////////////////////////////////////////////////////////////////////////////////////
procedure TAboutJetButton.Edit;
begin
  Application.MessageBox('TJetButton v1.00 for Delphi32. (C) 1999 Jimmy Theo Weku.'+#10#13+
                         'for more information about how to use this component please read README.TXT that included with this component',
                         'About TJetButton Component', MB_OK + MB_ICONINFORMATION);
end;

function TAboutJetButton.GetAttributes: TPropertyAttributes;
begin
  Result:= [paMultiSelect, paDialog, paReadOnly];
end;

function TAboutJetButton.GetValue: string;
begin
  Result:= '(About)';
end;

////////////////////////////////////////////////////////////////////////////////////////
constructor TJetButton.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  SetBounds(0, 0, 50, 50);
  ControlStyle := [csCaptureMouse, csOpaque];
  FAutoSize := True;
  FBitmapUp := TBitmap.Create;
  FBitmapUp.OnChange := BitmapUpChanged;
  FBitmapDown := TBitmap.Create;
  FBitmapDown.OnChange := BitmapDownChanged;
  FBitmapOver := TBitmap.Create;
  FBitmapOver.OnChange := BitmapOverChanged;
  FBitmapDisabled := TBitmap.Create;
  FBitmapDisabled.OnChange := BitmapDisabledChanged;
  FTransparentColor := clWhite;
  TempBitmap := nil;
  ParentFont := True;
  FEnabled := true;
  MouseOver := false;
  FState := bsUp;
end;

destructor TJetButton.Destroy;
begin
  FBitmapUp.Free;
  FBitmapDown.Free;
  FBitmapOver.Free;
  FBitmapDisabled.Free;
  TempBitmap.Free;
  inherited Destroy;
end;

procedure TJetButton.Paint;
var W, H: Integer;
    Composite, Mask, Overlay, CurrentBmp: TBitmap;
    R, NewR: TRect;
begin
  if csDesigning in ComponentState then
    with Canvas do
    begin
      Pen.Style := psSolid;
      Brush.Style := bsClear;
      Rectangle(0, 0, Width, Height);
    end;

  if (FState in [bsDisabled, bsExclusive]) then
  begin
     if Not FBitmapDisabled.Empty then CurrentBmp := FBitmapDisabled else CurrentBmp := FBitmapUp;
  end else
  if (FState = bsUp) and (not MouseOver) then CurrentBmp := FBitmapUp
  else
  if (FState = bsUp) and MouseOver then
  begin
    if Not FBitmapOver.Empty then CurrentBmp := FBitmapOver else CurrentBmp := FBitmapUp;
  end else
  begin
    if Not FBitmapDown.Empty then CurrentBmp := FBitmapDown else CurrentBmp := FBitmapUp;
  end;

  if not CurrentBmp.Empty then
  begin
    W := Width;
    H := Height;
    R := ClientRect;
    NewR := R;

    Composite := TBitmap.Create;
    Overlay := TBitmap.Create;

    try
      with Composite do
      begin
        Width := W;
        Height := H;
        Canvas.CopyMode := cmSrcCopy;
        Canvas.CopyRect(R, Self.Canvas, R);
      end;

      with Overlay do
      begin
        Width := W;
        Height := H;
        Canvas.CopyMode := cmSrcCopy;
        Canvas.Brush.Color := CurrentBmp.TransparentColor;
        Canvas.FillRect(R);
        Canvas.CopyRect(NewR, CurrentBmp.Canvas, R);
      end;

      Mask := MakeMask(Overlay, CurrentBmp.TransparentColor);
      try
        Composite.Canvas.CopyMode := cmSrcAnd;
        Composite.Canvas.CopyRect(R, Mask.Canvas, R);

        Overlay.Canvas.CopyMode := $00220326;
        Overlay.Canvas.CopyRect(R, Mask.Canvas, R);

        Composite.Canvas.CopyMode := cmSrcPaint;
        Composite.Canvas.CopyRect(R, Overlay.Canvas, R);

        Canvas.CopyMode := cmSrcCopy;
        Canvas.CopyRect(R, Composite.Canvas, R);

      finally
        Mask.Free;
      end;

    finally
      Composite.Free;
      Overlay.Free;
    end;

  end;

  if Length(Caption) > 0 then
  begin
    Canvas.Font := Self.Font;
    R := CLIENTRECT;
    DrawButtonText(Canvas, Caption, R, FState);
  end;

end;

function TJetButton.PtInMask(const X, Y: Integer): Boolean;
begin
  Result := True;
  if TempBitmap <> nil then
    Result := (TempBitmap.Canvas.Pixels[X, Y] = clBlack);
end;

procedure TJetButton.MouseMove(Shift: TShiftState; X, Y: Integer);
var Last : Boolean;
begin
  inherited MouseMove(Shift, X, Y);
  Last := MouseOver;
  MouseOver := PtInMask(X, Y);
  if (Last <> MouseOver) and (not (Fstate = bsDown)) then
  begin
    if FBitmapUp.Empty and Enabled then Invalidate else Repaint;
  end;
end;

procedure TJetButton.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
var Clicked: Boolean;
begin
  inherited MouseDown(Button, Shift, X, Y);
  if (Button = mbLeft) and Enabled then
  begin
    Clicked := PtInMask(X, Y);
    if Clicked then
    begin
      FState := bsDown;
//      MouseOver := true;
      Repaint;
    end;
  end;
end;

procedure TJetButton.MouseUp(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
var
  DoClick: Boolean;
begin
  inherited MouseUp(Button, Shift, X, Y);
  DoClick := PtInMask(X, Y);
  if (FState = bsDown) then
    begin
      FState := bsUp;
      Repaint;
      if FBitmapUp.Empty and Enabled then Invalidate else Repaint;
    end;
    if DoClick then
    begin
      MouseOver := true;
      Click;
    end;
end;

procedure TJetButton.Click;
begin
  inherited Click;
end;

function TJetButton.GetPalette: HPALETTE;
begin
  Result := FBitmapUp.Palette;
end;

procedure TJetButton.SetBitmapUp(Value: TBitmap);
begin
  FBitmapUp.Assign(Value);
end;

procedure TJetButton.SetBitmapDown(Value: TBitmap);
begin
  FBitmapDown.Assign(Value);
end;

procedure TJetButton.SetBitmapOver(Value: TBitmap);
begin
  FBitmapOver.Assign(Value);
end;

procedure TJetButton.SetBitmapDisabled(Value: TBitmap);
begin
  FBitmapDisabled.Assign(Value);
end;

procedure TJetButton.BitmapUpChanged(Sender: TObject);
Var Maskbmp : TBitmap;
    R : TRect;
begin
  AdjustBounds;
  MaskBmp := TBitmap.create;
  MaskBmp.Width := FBitmapUp.width;
  MaskBmp.Height := FBitmapUp.Height;
  R := Rect(0,0,FBitmapUp.Width,FBitmapUp.Height);
  MaskBmp.Canvas.CopyRect(R, FBitmapUp.Canvas, R);
  FTransparentColor := FBitmapUp.TransparentColor;
  TempBitmap.Free;
  TempBitmap := MakeMask(MaskBmp, FTransparentColor);
  MaskBmp.free;
  Invalidate;
end;

procedure TJetButton.BitmapDownChanged(Sender: TObject);
begin
  Invalidate;
end;

procedure TJetButton.BitmapOverChanged(Sender: TObject);
Var Maskbmp : TBitmap;
    R : TRect;
begin
  if FBitmapUp.Empty then
  begin
    SetBounds(Left, Top, FBitmapOver.Width, FBitmapOver.Height);
    MaskBmp := TBitmap.create;
    MaskBmp.Width := FBitmapOver.width;
    MaskBmp.Height := FBitmapOver.Height;
    R := Rect(0,0,FBitmapOver.Width,FBitmapOver.Height);
    FTransparentColor := FBitmapOver.TransparentColor;
    MaskBmp.Canvas.CopyRect(R, FBitmapOver.Canvas, R);
    TempBitmap.Free;
    TempBitmap := MakeMask(MaskBmp, FTransparentColor);
    MaskBmp.free;
  end;
  Invalidate;
end;

procedure TJetButton.BitmapDisabledChanged(Sender: TObject);
begin
  Invalidate;
end;

procedure TJetButton.CMDialogChar(var Message: TCMDialogChar);
begin
  with Message do
    if IsAccel(CharCode, Caption) and Enabled then
    begin
      Click;
      Result := 1;
    end else
      inherited;
end;

procedure TJetButton.CMFontChanged(var Message: TMessage);
begin
  Invalidate;
end;

procedure TJetButton.CMTextChanged(var Message: TMessage);
begin
  Invalidate;
end;

procedure TJetButton.CMSysColorChange(var Message: TMessage);
begin
  BitmapUpChanged(Self);
  BitmapDownChanged(Self);
  BitmapOverChanged(Self);
  BitmapDisabledChanged(Self);
end;

procedure TJetButton.CMMouseEnter(var Message: TMessage);
begin
  inherited;
end;

procedure TJetButton.CMMouseLeave(var Message: TMessage);
begin
  inherited;
  MouseOver := false;
  if (Fstate = bsDown) then exit;
  if FBitmapUp.Empty and Enabled then Invalidate else Repaint;
end;

procedure TJetButton.DrawButtonText(Canvas: TCanvas; const Caption: String;
  TextBounds: TRect; State: TButtonState);
var
  CString: array[0..255] of Char;
begin
  StrPCopy(CString, Caption);
  Canvas.Brush.Style := bsClear;
  if State = bsDown then OffsetRect(TextBounds, 1, 1);
  DrawText(Canvas.Handle, CString, -1, TextBounds,
      DT_CENTER or DT_VCENTER or DT_SINGLELINE);
end;

procedure TJetButton.Loaded;
begin
  inherited Loaded;
  if (FBitmapUp <> nil) and (FBitmapUp.Width > 0) and (FBitmapUp.Height > 0) then
  begin
    BitmapUpChanged(Self);
  end else if (FBitmapOver <> nil) and (FBitmapOver.Width > 0) and (FBitmapOver.Height > 0) then
  begin
    BitmapOverChanged(Self);
  end;
end;

procedure TJetButton.AdjustBounds;
begin
  SetBounds(Left, Top, Width, Height);
end;

procedure TJetButton.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
var W, H: Integer;
begin
  W := AWidth;
  H := AHeight;
  if not (csReading in ComponentState) and FAutoSize and not FBitmapUp.Empty then
  begin
    W := FBitmapUp.Width;
    H := FBitmapUp.Height;
  end;
  inherited SetBounds(ALeft, ATop, W, H);
end;

procedure TJetButton.Invalidate;
var R: TRect;
begin
  if (Visible or (csDesigning in ComponentState)) and
    (Parent <> nil) and Parent.HandleAllocated then
  begin
    R := BoundsRect;
    InvalidateRect(Parent.Handle, @R, True);
  end;
end;

procedure TJetButton.SetEnabled(Value : Boolean);
begin
 if Value <> FEnabled then
 begin
   FEnabled := Value;
   if FEnabled = false then FState := bsDisabled else FState := bsUp;
   Invalidate;
 end;
end;

procedure Register;
begin
  RegisterComponents('Jet', [TJetButton]);
  RegisterPropertyEditor(TypeInfo(TAboutJetButton), TJetButton, 'ABOUT', TAboutJetButton);
end;

end.
