unit Graphx;

{ Dancer's Graphic Extention Controls v1.0.6}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, ExtCtrls;

type
  TRGBColor = record
  case integer of
    1: (Color : TColor;);
    2: (R: byte;
        G: byte;
        B: byte)
  end;

  TByGradientSlider = class(TCustomControl)
  private
    FShowArrow: Boolean;
    FColorValue: TColor;
    FArrowActive: TColor;
    FArrowInactive: TColor;
    FGradientColor: TColor;
    FOnChanging: TNotifyEvent;
    FOnChanged: TNotifyEvent;

    FOldPosition: Integer;
    FArrowPosition: Integer;
    FRepeatTimer: TTimer;
    FIsKeyPlus: Boolean;

    procedure SetArrow(value: Boolean);
    procedure SetAActive(value: TColor);
    procedure SetAInActive(value: TColor);
    procedure SetAPosition(Value: integer);
    procedure SetGradientColor(Value: TColor);

    procedure RepeatKey(sender: TObject);
    procedure WMSize(var Msg: TMessage);  Message WM_SIZE;
    procedure WMSetFocus(var Msg: TMessage); message WM_SETFOCUS;
    procedure WMKillFocus(var Msg: TMessage); message WM_KILLFOCUS;
    procedure WMGetDlgCode(var Msg: TMessage); message WM_GETDLGCODE;
    procedure CMCtl3DChanged(var Msg: TMessage); message CM_CTL3DCHANGED;
  protected
    PaintRect: TRect;
    procedure AdjustRect;
    function GetColor: TColor;
    procedure MoveArrow(Shift: TShiftState; X, Y: Integer); virtual;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyUp(var Key: Word; Shift: TShiftState); override;
  public
    procedure DrawArrow;
{    procedure CreateParams(var Params: TCreateParams); override; }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Paint; override;
  published
    property ArrowActive: TColor read FArrowActive write SetAActive;
    property ArrowInactive: TColor  read FArrowInActive write SetAInActive;
    property ArrowPosition: Integer read FArrowPosition write SetAPosition;
    property Color;
    property ColorValue: TColor read FColorValue;
    property Ctl3D;
    property Enabled;
    property GradientColor: TColor read FGradientColor write SetGradientColor;
    property ParentColor;
    property ParentCtl3D;
    property ShowArrow: Boolean read FShowArrow write SetArrow default true;
    property TabOrder;
    property Visible;
    property OnChanging: TNotifyEvent read FOnChanging write FOnChanging;
    property OnChanged: TNotifyEvent read FOnChanged write FOnChanged;
    property OnKeyDown;
    property OnKeyUp;
    property OnKeyPress;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

const
  RepeatPause = 100;

procedure Register;

Implementation

Procedure Register;
begin
  RegisterComponents('Beyond 98', [TByGradientSlider]);
end;

Function ProcessValue(var value: byte; pos: integer): integer;
var v: longint;
begin
  v := round(value * (1 + pos /100));
  if v > 255 then
     v := 255 else
  if v < 0 then
     v := 0;
  value := v;
  ProcessValue := v;
end;

Procedure TByGradientSlider.DrawArrow;
var
  c : TColor;
  x : integer;
begin
  if Focused
    then c := FArrowActive
    else c := FArrowInactive;

  with Canvas do
  begin
    Brush.Color := Color;
    FillRect(Rect(0,PaintRect.bottom+1,width,height));

    for x := PaintRect.Left to PaintRect.Left + 6 do
      Pixels[x + FArrowPosition + 48, Height] := c;

    for x := PaintRect.Left + 1 to PaintRect.Left + 5 do
      Pixels[x + FArrowPosition + 48, PaintRect.bottom + 3] := c;

    for x := PaintRect.Left + 2 to PaintRect.Left + 4 do
      Pixels[x + FArrowPosition + 48, PaintRect.bottom + 2] := c;

    Pixels[PaintRect.Left + 3 + FArrowPosition + 48, PaintRect.Bottom + 1]:= c;
  end;
end;

Procedure TByGradientSlider.Paint;
var
  x : integer;
  c : TRGBColor;
  r : TRect;
begin
  if ShowArrow then DrawArrow;

  r := PaintRect;

  if Ctl3D then Frame3D(Canvas, r, clBtnShadow, clBtnHighLight,1);

  With Canvas do
    for x := -50 to 49 do
    begin
      c.color := GradientColor;
      ProcessValue(c.r, x);
      ProcessValue(c.g, x);
      ProcessValue(c.b, x);
      Pen.Color := rgb(c.r, c.g, c.b);
      MoveTo(r.Left + x + 49, r.Top);
      LineTo(r.Left + x + 49, r.Bottom);
    end;
end;

procedure TByGradientSlider.SetArrow(value: Boolean);
begin
  if FShowArrow <> value then
  begin
    FShowArrow := value;
    AdjustRect;
    Invalidate;
  end;
end;

procedure TByGradientSlider.SetAActive(value: TColor);
begin
  if FArrowActive <> value then
  begin
    FArrowActive := Value;
    if FShowArrow and Focused then DrawArrow;
  end;
end;

procedure TByGradientSlider.SetAInactive(value: TColor);
begin
  if FArrowInactive <> value then
  begin
    FArrowInactive := Value;
    if FShowArrow and not Focused then DrawArrow;
  end;
end;

function TByGradientSlider.GetColor: TColor;
var c: TRGBColor;
begin
  c.color := GradientColor;
  ProcessValue(c.r, FArrowPosition);
  ProcessValue(c.g, FArrowPosition);
  ProcessValue(c.b, FArrowPosition);
  GetColor := rgb(c.r, c.g, c.b);
end;

procedure TByGradientSlider.SetAPosition(Value: integer);
begin
  if value <> FArrowPosition then
  begin
    if value > 49 then value := 49 else
    if value < -50 then value := -50;
    FArrowPosition := value;

    FColorValue := GetColor;
    if FShowArrow then DrawArrow;
  end;
end;

Procedure TByGradientSlider.SetGradientColor(value: TColor);
begin
  if value <> FGradientColor then
  begin
    FGradientColor := value;
    FColorValue := GetColor;
    if Assigned(FOnChanged) then FOnChanged(self);
    Paint;
  end;
end;

Procedure TByGradientSlider.AdjustRect;
{ Adjust the PaintRect, where gradient color + 3D frame would be painted. }
begin
  if FShowArrow then
    if Ctl3D
      then PaintRect := Rect(2, 0, 104, Height - 4)
      else PaintRect := Rect(3, 0, 103, Height - 4) else
    if Ctl3D
      then PaintRect := Rect(0, 0, 102, Height)
      else PaintRect := Rect(0, 0, 100, Height);
end;

Procedure TByGradientSlider.CMCtl3DChanged(var Msg: TMessage);
begin
  if FShowArrow
    then Width := 106
    else
      if Ctl3D
        then Width := 102
        else Width := 100;

  inherited;
  AdjustRect;
  Invalidate;
end;

Procedure TByGradientSlider.WMSize(var Msg: TMessage);
begin
  if LOWORD(Msg.lParam) <> Width then
    if FShowArrow
      then Width := 106
      else
        if Ctl3D
          then Width := 102
          else Width := 100;

  Msg.Result := 0;
  AdjustRect;
  Invalidate;
end;

procedure TByGradientSlider.WMKillFocus(var Msg: TMessage);
begin
  inherited;
  if FRepeatTimer.Enabled then
  begin
    FRepeatTimer.Enabled := false;
    FArrowPosition := FOldPosition;
  end;
  if FShowArrow then DrawArrow;
end;

procedure TByGradientSlider.WMSetFocus(var Msg: TMessage);
begin
  inherited;
  if FShowArrow then DrawArrow;
end;

procedure TByGradientSlider.WMGetDlgCode(var Msg: TMessage);
begin
  Msg.Result := DLGC_WANTARROWS;
end;

procedure TByGradientSlider.MoveArrow(Shift: TShiftState; X, Y: Integer);
var r: TRect;
begin
  r := PaintRect;
  if Ctl3D then inflateRect(r, -1, -1);

  if x < r.left
     then x := -50
     else if x > r.right
       then x := 49
       else x := x - r.left - 50;

  if x <> FArrowPosition then SetAPosition(x);
end;

procedure TByGradientSlider.MouseMove(Shift: TShiftState; X, Y: Integer);
begin
  inherited MouseMove(Shift,x,y);

  if ssLeft in Shift then
  begin
    MoveArrow(Shift, x, y);
    if Assigned(FOnChanging) then FOnchanging(Self);
  end;
end;

procedure TByGradientSlider.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  inherited MouseUp(Button, Shift, x, y);

  if mbLeft = Button then
  begin
    MoveArrow(Shift, x, y);
    if FOldPosition <> FArrowPosition then
      if Assigned(FOnChanged) then FOnChanged(Self);
  end;
end;

procedure TByGradientSlider.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  inherited MouseDown(Button, Shift, x, y);
  SetFocus;

  if mbLeft = Button then
  begin
    FOldPosition := FArrowPosition;
    MoveArrow(Shift, x, y);
    if Assigned(FOnChanging) then FOnchanging(Self);
  end;
end;

procedure TByGradientSlider.KeyDown(var Key: Word; Shift: TShiftState);
begin
  if (Key = VK_Right) or (Key = VK_Left) then
  begin
    if Key = VK_Right then FIsKeyPlus := true else FIsKeyPlus := false;
    FRepeatTimer.Enabled := true;
    RepeatKey(Self);
  end;
end;

procedure TByGradientSlider.KeyUp(var Key: Word; Shift: TShiftState);
begin
  if FRepeatTimer.Enabled then
  begin
    FRepeatTimer.Enabled := false;
    if Assigned(FOnChanged) then FOnChanged(Self)
  end
end;

procedure TByGradientSlider.RepeatKey(Sender: TObject);
begin
  if FIsKeyPlus then
  begin
    SetAPosition(FArrowPosition + 1);
    if Assigned(FOnChanging) then FOnChanging(Self);
    if FArrowPosition = 49 then FRepeatTimer.Enabled := false;
  end else
  begin
    SetAPosition(FArrowPosition - 1);
    if Assigned(FOnChanging) then FOnChanging(Self);
    if FArrowPosition = -50 then FRepeatTimer.Enabled := false;
  end;
end;

constructor TByGradientSlider.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  Width := 104;
  Height := 10;

  FArrowActive := clNavy;
  FArrowInactive:= clGray;

  FGradientColor := clNavy;
  FOldPosition := 0;
  FArrowPosition := 0;
  FColorValue := Color;

  FShowArrow := true;

  AdjustRect;

  FIsKeyPlus := false;

  FRepeatTimer:= TTimer.Create(Self);
  FRepeatTimer.Interval := RepeatPause;
  FRepeatTimer.Enabled := false;
  FRepeatTimer.OnTimer := RepeatKey;
end;

Destructor TByGradientSlider.Destroy;
begin
  FRepeatTimer.free;
  inherited Destroy;
end;

{procedure TByGradientSlider.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  Params.ExStyle := Params.ExStyle or WS_EX_TRANSPARENT;
end;}

end.
