unit HugeForm;

{ This demo program demonstrates the use of the THugeList class,
  and the HugeInc, HugeDec, and HugeOffset routines.  There is also
  a simple timer, for crude performance measurements.

  Copyright  1995 Tempest Software.  All Rights Reserved.
}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, WinCrt, Huge, StdCtrls, ExtCtrls, Spin, Mask, TabNotBk;

type
  TTestList = THugeList;  { change to TList to compare with standard list type }

  TMainForm = class(TForm)
    TabbedNotebook: TTabbedNotebook;
    Label1: TLabel;
    Label5: TLabel;
    Label2: TLabel;
    Label3: TLabel;
    Label4: TLabel;
    IndexEdit: TEdit;
    IndexEdit2: TEdit;
    ValueEdit: TEdit;
    CountEdit: TEdit;
    CapacityEdit: TEdit;
    AddButton: TButton;
    DeleteButton: TButton;
    LookupButton: TButton;
    PackButton: TButton;
    ExchangeButton: TButton;
    FillButton: TButton;
    ClearButton: TButton;
    SetButton: TButton;
    Label6: TLabel;
    PointerEdit: TMaskEdit;
    Label7: TLabel;
    OffsetEdit: TMaskEdit;
    IncrButton: TButton;
    DecrButton: TButton;
    OffsetButton: TButton;
    SpinOffset: TSpinButton;
    StatusBar: TPanel;
    SpinButton1: TSpinButton;
    Label8: TLabel;
    Label9: TLabel;
    Iterations: TEdit;
    Label10: TLabel;
    Label11: TLabel;
    ElapsedTime: TEdit;
    StartButton: TButton;
    AbortButton: TButton;
    Label12: TLabel;
    IterCount: TEdit;
    MoveButton: TButton;
    IndexOfButton: TButton;
    procedure AddButtonClick(Sender: TObject);
    procedure IndexEditChange(Sender: TObject);
    procedure ValueEditChange(Sender: TObject);
    procedure DeleteButtonClick(Sender: TObject);
    procedure PackButtonClick(Sender: TObject);
    procedure LookupButtonClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure ClearButtonClick(Sender: TObject);
    procedure ExchangeButtonClick(Sender: TObject);
    procedure MoveButtonClick(Sender: TObject);
    procedure SetButtonClick(Sender: TObject);
    procedure FillButtonClick(Sender: TObject);
    procedure IncrButtonClick(Sender: TObject);
    procedure DecrButtonClick(Sender: TObject);
    procedure OffsetButtonClick(Sender: TObject);
    procedure SpinOffsetDownClick(Sender: TObject);
    procedure SpinOffsetUpClick(Sender: TObject);
    procedure SpinButton1DownClick(Sender: TObject);
    procedure SpinButton1UpClick(Sender: TObject);
    procedure StartButtonClick(Sender: TObject);
    procedure AbortButtonClick(Sender: TObject);
    procedure IterationsChange(Sender: TObject);
    procedure IndexOfButtonClick(Sender: TObject);
  private
    AbortFlag: Boolean;
    function GetPointer: Pointer;
    function GetOffset: LongInt;
    procedure SetPointer(Ptr: Pointer);
    procedure SetOffset(Offset: LongInt);
    procedure DoOneTest;
    procedure ShowHint(Sender: TObject);
  public
    List: TTestList;
    procedure Update;
    procedure EnableButtons;
    procedure ExceptionHandler(Sender: TObject; Ex: Exception);
    property PointerValue: Pointer read GetPointer write SetPointer;
    property OffsetValue: LongInt read GetOffset write SetOffset;
    procedure StartTiming;
  end;

var
  MainForm: TMainForm;

implementation

{$R *.DFM}

{ Trap exceptions and display the message in the status bar.  The
  messages are usually about indexes out of range, and so on.
  If you run this demo from Delphi, you probably want to turn off
  'Break on Exception' in the Options>Environment dialog.
}
procedure TMainForm.ExceptionHandler(Sender: TObject; Ex: Exception);
begin
  StatusBar.Caption := Ex.Message;
end;

procedure TMainForm.ShowHint(Sender: TObject);
begin
  StatusBar.Caption := Application.Hint;
end;

procedure TMainForm.FormCreate(Sender: TObject);
begin
  Application.OnException := ExceptionHandler;
  Application.OnHint := ShowHint;
  List := TTestList.Create;
end;

procedure TMainForm.FormDestroy(Sender: TObject);
begin
  List.Free;
end;

{ THugeList page }

{ After any change to the list, update the count and capacity fields }
procedure TMainForm.Update;
begin
  CountEdit.Text := IntToStr(List.Count);
  CapacityEdit.Text := IntToStr(List.Capacity);
end;

{ When the user types, enable or disable the buttons according to
  the availability of one or both indexes and the value.
}
procedure TMainForm.EnableButtons;
var
  HaveIndex, HaveIndex2, HaveValue: Boolean;
begin
  HaveIndex := IndexEdit.Text <> '';
  HaveIndex2:= IndexEdit2.Text <> '';
  HaveValue := ValueEdit.Text <> '';
  AddButton.Enabled := HaveValue;
  DeleteButton.Enabled := HaveIndex;
  LookupButton.Enabled := HaveIndex;
  ExchangeButton.Enabled := HaveIndex and HaveIndex2;
  MoveButton.Enabled := HaveIndex and HaveIndex2;
  IndexOfButton.Enabled := HaveValue;
end;

procedure TMainForm.IndexEditChange(Sender: TObject);
var
  Edit: TEdit;
begin
  StatusBar.Caption := '';
  Edit := Sender as TEdit;
  if Edit.Text <> '' then
  begin
    try
      StrToInt(Edit.Text);
      EnableButtons;
      if IndexEdit.Text <> '' then
        AddButton.Caption := '&Insert'
      else
        AddButton.Caption := '&Add';
    except
    on E: Exception do
      StatusBar.Caption := E.Message;
    end;
  end;
end;

procedure TMainForm.ValueEditChange(Sender: TObject);
begin
  StatusBar.Caption := '';
  EnableButtons;
end;

{ Add or Insert, depending on whether the user supplies an index }
procedure TMainForm.AddButtonClick(Sender: TObject);
var
  Index: LongInt;
  Value: PString;
begin
  if ValueEdit.Text = 'nil' then
    Value := nil
  else
    Value := NewStr(ValueEdit.Text);
  if IndexEdit.Text = '' then
    StatusBar.Caption := 'Added at index ' + IntToStr(List.Add(Value))
  else
  begin
    Index := StrToInt(IndexEdit.Text);
    List.Insert(Index, Value);
    StatusBar.Caption := 'Inserted at index ' + IntToStr(Index);
  end;
  Update;
end;

procedure TMainForm.DeleteButtonClick(Sender: TObject);
var
  Index: LongInt;
  Value: PString;
begin
  Index := StrToInt(IndexEdit.Text);
  Value := PString(List.Items[Index]);
  List.Delete(Index);
  if Value = nil then
    StatusBar.Caption := Format('Deleted nil from index %d', [Index])
  else
    StatusBar.Caption := Format('Deleted ''%s'' from index %d', [Value^, Index]);
  if Value <> nil then
    DisposeStr(Value);
  Update;
end;

procedure TMainForm.PackButtonClick(Sender: TObject);
begin
  List.Pack;
  StatusBar.Caption := 'Packed';
  Update;
end;

procedure TMainForm.LookupButtonClick(Sender: TObject);
var
  Index: LongInt;
  Value: PString;
  Found: LongInt;
begin
  if IndexEdit.Text <> '' then
  begin
    Index := StrToInt(IndexEdit.Text);
    Value := PString(List.Items[Index]);
    if Value = nil then
      StatusBar.Caption := Format('Items[%d]=nil', [Index])
    else
      StatusBar.Caption := Format('Items[%d]=''%s''', [Index, Value^]);
  end
  else
  begin
    Found := -1;
    for Index := 0 to List.Count-1 do
    begin
      Value := PString(List.Items[Index]);
      if (Value = nil) and (ValueEdit.Text = 'nil') then
      begin
        Found := Index;
        Break;
      end
      else if (Value <> nil) and (Value^ = ValueEdit.Text) then
      begin
        Found := Index;
        Break;
      end;
    end;
    if Found = -1 then
      StatusBar.Caption := 'Value ''' + ValueEdit.Text + ''' not found'
    else
      StatusBar.Caption := Format('Index of ''%s'' = %d', [ValueEdit.Text, Found]);
  end;
end;

procedure TMainForm.ClearButtonClick(Sender: TObject);
begin
  List.Clear;
  Update;
  StatusBar.Caption := 'Cleared';
end;

procedure TMainForm.ExchangeButtonClick(Sender: TObject);
var
  Index1, Index2: LongInt;
begin
  Index1 := StrToInt(IndexEdit.Text);
  Index2 := StrToInt(IndexEdit2.Text);
  List.Exchange(Index1, Index2);
  StatusBar.Caption := Format('Exchanged %d and %d', [Index1, Index2]);
  Update;
end;

procedure TMainForm.MoveButtonClick(Sender: TObject);
var
  Index1, Index2: LongInt;
begin
  Index1 := StrToInt(IndexEdit.Text);
  Index2 := StrToInt(IndexEdit2.Text);
  List.Move(Index1, Index2);
  StatusBar.Caption := Format('Moved %d to %d', [Index1, Index2]);
  Update;
end;

procedure TMainForm.SetButtonClick(Sender: TObject);
var
  Value: PString;
  Index: LongInt;
begin
  Index := StrToInt(IndexEdit.Text);

  if (Index >= 0) and (Index < List.Count) then
  begin
    Value := List[Index];
    if Value <> nil then
      DisposeStr(Value);
  end;

  if ValueEdit.Text = 'nil' then
    Value := nil
  else
    Value := NewStr(ValueEdit.Text);
  List[Index] := Value;
  if Value = nil then
    StatusBar.Caption := Format('Set Items[%d] = nil', [Index])
  else
    StatusBar.Caption := Format('Set Items[%d] = ''%s''', [Index, ValueEdit.Text]);
  Update;
end;

procedure TMainForm.FillButtonClick(Sender: TObject);
var
  i, Limit: LongInt;
  Str: String;
begin
  Str := IntToStr($10000);
  if InputQuery('Huge List', 'Fill the list up to index:', Str) then
  begin
    Limit := StrToInt(Str);
    if Limit + List.Count > List.Capacity then
      List.Capacity := Limit + List.Count;
    Screen.Cursor := crHourglass;
    try
      for i := 0 to Limit-1 do
        List.Add(NewStr(IntToStr(i)));
    finally
      Screen.Cursor := crDefault;
    end;
    StatusBar.Caption := Format('Added items from 0 to %d', [Limit-1]);
    Update;
  end;
end;

{ IndexOf looks for the specific value, and does not compare strings.
  Find the string manually, and then use IndexOf to find the pointer.
  Strange, but a good test. }
procedure TMainForm.IndexOfButtonClick(Sender: TObject);
const
  NotFound = -2;    { value that is never returned by IndexOf }
var
 I, Index: LongInt;
begin
  Index := NotFound;
  for I := 0 to List.Count-1 do
    if (List[I] <> nil) and (PString(List[I])^ = ValueEdit.Text) then
    begin
      Index := List.IndexOf(List[I]);
      Break;
    end;
  if Index = NotFound then
    Index := List.IndexOf(@Index);   { a value we know is not present }
  StatusBar.Caption := Format('Index = %d', [Index]);
end;


{ Pointer page }
function TMainForm.GetPointer: Pointer;
begin
  if PointerEdit.Text = '' then
    Result := nil
  else
    Result := Pointer(StrToInt('$' + PointerEdit.Text));
end;

function TMainForm.GetOffset: LongInt;
begin
  if OffsetEdit.Text = '' then
    Result := 0
  else
    Result := StrToInt(OffsetEdit.Text);
end;

procedure TMainForm.SetPointer(Ptr: Pointer);
begin
  PointerEdit.Text := IntToHex(LongInt(Ptr), 8);
end;

procedure TMainForm.SetOffset(Offset: LongInt);
begin
  OffsetEdit.Text := IntToStr(Offset);
end;

procedure TMainForm.IncrButtonClick(Sender: TObject);
var
  Ptr: Pointer;
begin
  Ptr := PointerValue;
  HugeInc(Ptr, OffsetValue);
  PointerValue := Ptr;
end;

procedure TMainForm.DecrButtonClick(Sender: TObject);
var
  Ptr: Pointer;
begin
  Ptr := PointerValue;
  HugeDec(Ptr, OffsetValue);
  PointerValue := Ptr;
end;

procedure TMainForm.OffsetButtonClick(Sender: TObject);
begin
  PointerValue := HugeOffset(PointerValue, OffsetValue);
end;

procedure TMainForm.SpinOffsetDownClick(Sender: TObject);
begin
  OffsetValue := OffsetValue - $10000;
end;

procedure TMainForm.SpinOffsetUpClick(Sender: TObject);
begin
  OffsetValue := OffsetValue + $10000;
end;

procedure TMainForm.SpinButton1DownClick(Sender: TObject);
begin
  OffsetValue := OffsetValue - 1024;
end;

procedure TMainForm.SpinButton1UpClick(Sender: TObject);
begin
  OffsetValue := OffsetValue + 1024;
end;

{ Timing page }
procedure TMainForm.DoOneTest;
var
  Index: LongInt;
begin
  Index := Trunc(Random * List.Count);
  List.Insert(Index, nil);
  List.Delete(Index);
end;

procedure TMainForm.StartTiming;
var
  Start, Ticks: LongInt;
  I, NIterations: LongInt;
begin
  NIterations := StrToInt(Iterations.Text);
  Ticks := 0;
  for I := 1 to NIterations do
  begin
    IterCount.Text := IntToStr(I);
    Start := GetTickCount;
    DoOneTest;
    Ticks := Ticks + (GetTickCount - Start);
    ElapsedTime.Text := IntToStr(Ticks);
    Application.ProcessMessages;
    if AbortFlag then
      Break;
  end;
  ElapsedTime.Text := IntToStr(Ticks);
end;

procedure TMainForm.StartButtonClick(Sender: TObject);
begin
  AbortFlag := False;
  AbortButton.Visible := True;
  ABortButton.Enabled := True;
  StartButton.Visible := False;
  StartButton.Enabled := False;
  try
    StatusBar.Caption := 'Timing...  Click Abort to interrupt.';
    StartTiming;
  finally
    StartButton.Visible := True;
    StartButton.Enabled := True;
    AbortButton.Visible := False;
    AbortButton.Enabled := False;
    if AbortFlag then
      StatusBar.Caption := 'Aborted.'
    else
      StatusBar.Caption := '';
  end;
end;

procedure TMainForm.AbortButtonClick(Sender: TObject);
begin
  AbortFlag := True;
end;

procedure TMainForm.IterationsChange(Sender: TObject);
begin
  StartButton.Enabled := (Iterations.Text <> '') and
                         (StrToInt(Iterations.Text) > 0);
end;

end.
{--=>revision-history<=--}
{1 HUGEFORM.PAS 7-Jun-95,15:12:10,`lisch' CHANGE=INITIAL                      }
{     Initial version.                                                        }
{2 HUGEFORM.PAS 12-Aug-95,18:34:40,`lisch' CHANGE=V14                         }
{     Fixed Move to delete & insert, instead of Copy.                         }
{3 HUGEFORM.PAS 3-Sep-95,15:14:56,`lisch' CHANGE=V14                          }
{     Version 1.4: Optimized Move.  Added Move/IndexOf to demo program.       }
{--=>revision-history<=--}

