{*
Enhanced HelpContext Property Editor
Version 1.0
Original developed as Hlplst.pas by Roger.Karas@fnb.sprint.com
Enhanced by John Ruysbroek (john.ruysbroek@tip.nl)
*}

unit HlpConEd;

interface

uses Classes, dsgnIntf, SysUtils, Dialogs, Controls;

type
  THelpContextEditor = Class(TPropertyEditor)
  public
    function  GetAttributes: TPropertyAttributes; override;
    function  GetValue: String; override;
    procedure GetValues(Proc: TGetStrProc); override;
    procedure SetValue(Const Value: string); override;
  end;

  procedure Register;
  function  FormatHelpOption(Str: string): string;
  function  GetFirstFileName(const Extension: string): string;
  procedure CreateHelpProjectFile(const FileName: string);

const
  Tab    = #9;
  Space  = #32;
  Remark = #59;
  Equal  = #61;

implementation

procedure Register;
begin
  RegisterPropertyEditor(TypeInfo(THelpContext),nil,'',THelpContextEditor);
end;

function THelpContextEditor.GetAttributes: TPropertyAttributes;
begin
  Result := [paValueList,paMultiSelect];
end;

function THelpContextEditor.GetValue: string;
begin
  Result := IntToStr(GetOrdValue);
end;

procedure THelpContextEditor.GetValues(Proc: TGetStrProc);
var
  i: Integer;
  f: TextFile;
  blnInHelpSection: Boolean;
  strLine: String;
  ProjectFile: String[12];
begin
  { Search for file with 'hpj' extension.
    If it can not be found, ask the user if he/she wants
    to create a default help project file with same name
    as the current Delphi projectname. }
  try
    { Get help project filename }
    ProjectFile := GetFirstFileName('hpj');

    { Try to open file }
    AssignFile(f, ConCat(ProjectFile,'.hpj'));
    Reset(f);

  except
    on e: Exception do
      begin
        { Get Delphi project name }
        ProjectFile := GetFirstFileName('dpr');

        { Get out if no project file exist }
        if ProjectFile = '' then Exit;

        { Ok, file exist, ask confirmation to create default file }
        if MessageDlg('File ' + ProjectFile + '.HPJ could not be found.' +
                       #13#13 +
                      'Do you want to create ' + ProjectFile + '.HPJ?',
                      mtError, [mbYes, mbNo], 0) = mrYes
        then
          begin
           { Create an default help project file }
           CreateHelpProjectFile(ProjectFile);

           { Try to open file }
           AssignFile(f, ConCat(ProjectFile,'.hpj'));
           Reset(f);
          end
        else
      	  exit;
      end;
  end;

  { Read through the file one line at a time, looking for help
    identifers between the header 'MAP' and the next header }
  blnInHelpSection := False;

  while (not Eof(f)) and (not blnInHelpSection) do
    begin
     ReadLn(f, strLine);

     if pos('[MAP]', Uppercase(strLine)) > 0 then
        { Found it! }
    	blnInHelpSection := True;
    end;

  while (blnInHelpSection) and (not Eof(f)) do
    begin
     Readln(f, strLine);

     if pos('[', strLine) > 0 then
        { Beginning of next section found }
    	blnInHelpSection := False
     else
       begin
        strLine := FormatHelpOption(strLine);

        if strLine <> '' then Proc(strLine);
       end;
    end;

  CloseFile(F);
end;

procedure THelpContextEditor.SetValue(const Value : string);
begin
  try
    { Convert first 5 characters of 'value' to integer }
    SetOrdValue(StrToInt(Copy(Value, 1, 5)));
  except
    { Something went wrong, so do nothing,
     leave current HelpContext value intact }
  end;
end;

function FormatHelpOption(Str: string): string;
var
  Description, Value: string;
  n: integer;
begin
  { Find 'remark' chararter in row }
  n := Pos(Remark, Str);

  { Remarks found? }
  if n > 0 then
    begin
     { Remove remarks in row }
     Str := Copy(Str, 1, n - 1);

     { Check length }
     if Str = '' then
       begin
        { String is empty, no further processing is needed }
        Result := '';
        Exit;
       end;
    end;

  { Init }
  Description := '';
  Value       := '0000';

  { Break display string into two seperate strings }
  for n := 1 to Length(Str) do
    if not (Str[n] in [Tab,Space,Equal,'0'..'9']) then
       { Store 'description' part of row }
       Description := ConCat(Description, Str[n])
    else if Str[n] in ['0'..'9'] then
       { Store 'value' part of row }
       Value := ConCat(Value, Str[n]);

  { Check length of value, correct if longer then 5 characters }
  if Length(Value) > 5 then
     Delete(Value, 1, Length(Value) - 5);

  if Description = '' then
     { Allow empty rows for marking sections! }
     Result := ' '
  else
     { Build the display string by storing 'value' first }
     Result := Concat(Value, '  ', Description);
end;

function GetFirstFileName(const Extension: string): string;
var
  CurrentDir: String;
  SearchRec: TSearchRec;
begin
  { Get current directory }
  GetDir(0, CurrentDir);

  { Check if a file with given extension exist }
  FindFirst(CurrentDir + '\*.' + Extension, faAnyFile, SearchRec);

  { Return filename without the extension }
  Result := Copy(SearchRec.Name, 0, Pos('.', SearchRec.Name) - 1);
end;

procedure CreateHelpProjectFile(const FileName: string);
var
  f: TextFile;
  Year, Month, Day: Word;
  CurrentDir: String;
begin
  { Decode date }
  DecodeDate(now, Year, Month, Day);

  { Get current directory }
  GetDir(0, CurrentDir);

  { Make the file }
  AssignFile(f, FileName + '.hpj');
  Rewrite(f);

  { Make default help project file }
  WriteLn(f, '; ' + FileName + '.HPJ');
  WriteLn(f, '; Author: J.C. Ruysbroek');
  WriteLn(f, ';');
  WriteLn(f, '; Revision history:');
  WriteLn(f, ';     ' + DateToStr(now) + ': Project file created');
  WriteLn(f, '');
  WriteLn(f, '');
  WriteLn(f, '[OPTIONS]');
  WriteLn(f, 'CONTENTS=HELP_Contents');
  WriteLn(f, 'TITLE=');
  WriteLn(f, 'COPYRIGHT=Copyright  ' + IntToStr(Year) + ' J.C. Ruysbroek.');
  WriteLn(f, 'COMPRESS=MEDIUM');
  WriteLn(f, 'WARNING=3');
  WriteLn(f, 'REPORT=ON');
  WriteLn(f, 'ErrorLog=HLPERROR.LOG');
  WriteLn(f, 'ROOT=' + CurrentDir);
  WriteLn(f, 'BMROOT=' + CurrentDir);
  WriteLn(f, '');
  WriteLn(f, '[WINDOWS]');
  WriteLn(f, 'Main= , (250,0,773,950),0,(255,255,255),(255,255,210)');
  WriteLn(f, '');
  WriteLn(f, '[FILES]');
  WriteLn(f, FileName + '.RTF');
  WriteLn(f, '');
  WriteLn(f, '[CONFIG]');
  WriteLn(f, 'BrowseButtons()');
  WriteLn(f, '');
  WriteLn(f, '[MAP]');
  WriteLn(f, 'HELP_Contents' + Tab + Tab + '1');
  WriteLn(f, 'HELP_Glossary' + Tab + Tab + '2');
  WriteLn(f, 'HELP_TopicSearch' + Tab + '3');
  WriteLn(f, 'HELP_HowToUseHelp' + Tab + '4');
  WriteLn(f, 'HELP_TechSupport' + Tab + '5');
  WriteLn(f, 'HELP_About' + Tab + Tab + '6');
  WriteLn(f, '');

  { Close the file }
  CloseFile(f);
end;

end.
