{******************************************************************************
  HEXMAP COMPONENT  version 1.0

  This component implements a Hexagonal Grid, similer to grids used in
  various wargames.

  All source copyright 1997  Patrick Kemp ( Cynewulf@qadas.com )

  Last modified - 1/21/97
  modified by Patrick Kemp

*******************************************************************************}


unit Hexmap;


interface

 uses
    SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,Menus;

 type TPointType = (ptRowCol,ptXY); { Used in the Convertcoords function }

 type
   THexMap = class(TGraphicControl)
   private
     FHexColumns:Integer;          { Number of Columns in the map }
     FHexRows:Integer;             { Number of rows in the map }
     FHexRadius:Integer;           { The radius of one hexagon }
     FHexShowLabels:Boolean;       { If True, lables are displayed in a hex }
     FHex3d:Boolean;               { If true, hexes are shaded like buttons }
     FHexColor: TColor;            { Color of a Hexagon }
     FBackColor: TColor;           { Backround color of bitmap }
     FLineColor: Tcolor;           { Color of lines used to draw a hexagon}
     TempMap:TBitMap;  {Used as a drawing surface, then copied to control }

     procedure SetHexColumns(Value: Integer);
     procedure SetHexRows(Value: Integer);
     procedure SetHexRadius(Value: Integer);
     procedure SetHexShowLabels(Value:Boolean);
     procedure SetHex3d(Value:Boolean);
     procedure SetHexColor(Value: TColor);
     procedure SetBackColor(Value: TColor);
     procedure SetLineColor(Value: TColor);
     procedure MakeMap;


   protected
     procedure Paint; Override;

   public
     constructor Create(AOwner: TComponent); Override;
     destructor destroy; OverRide;

     function ConvertCoords(point:Tpoint;        { pair to be converted }
                           pointtype:Tpointtype) { Type to be converted }
                           :Tpoint;              { result of conversion }

     procedure DrawHex(Target:TCanvas;         { Canvas to draw hex on }
                      Fillstyle : TBrushStyle; { How to fill the hex }
                      Fillcolor : TColor;      { Color to fill it }
                      Linestyle : TPenStyle;   { Wwhat kind of lines }
                      LineColor : Tcolor;      { What color for lines }
                      x,y,radius: integer;     { Position and size of Hex }
                      button    : boolean);    { Make hexes look like buttons? }


   published
     property HexColumns: Integer read FHexColumns write SetHexColumns;
     property HexRows: Integer read FHexRows write SetHexRows;
     property HexRadius: Integer read FHexRadius write SetHexRadius;
     property HexShowLabels: Boolean read FHexShowLabels write SetHexShowLabels;
     property Hex3d: Boolean read FHex3d write SetHex3d;
     property HexColor: TColor read FHexColor write SetHexColor;
     property BackColor: TColor read FBackColor write SetBackColor;
     property LineColor: TColor read FLineColor write SetLineColor;

     {Inherited properties}
     property Align;
     property Visible;
     property Enabled;
     property Font;
     property DragCursor;
     property DragMode;
     property OnDragDrop;
     property OnDragOver;
     Property OnEndDrag;
     property OnMouseDown;
     property OnMouseMove;
     property OnMouseUp;
     property OnClick;
     property OnDblClick;
     property PopupMenu;

 end;

 procedure Register;


implementation

 constructor THexMap.Create(AOwner: Tcomponent);
 begin
   inherited Create(AOwner);

   TempMap := TBitMap.create;  {prepare the offscreen (temp) bitmap}

   {Set intial property values for component}
   FHexColumns := 8;
   FHexRows := 8;
   FHexRadius := 30;
   FHexShowLabels := True;
   FHex3d := False;
   FHexColor := clGray;
   FBackColor := clTeal;
   FLineColor := clBlack;

   {Size component in order to show all rows / cols}
   width := (HexColumns div 2) * (3*HexRadius) + (4*HexRadius);
   height := ((2*HexRadius) * HexRows) + (4*HexRadius);

   {Size bitmap (Should match component size) }
   TempMap.width := width;
   TempMap.height := height;

   MakeMap; {creates the grid on temp bitmap}

 end;


{******************************************************************************}
{ Free any resources allocated to component                                    }
 destructor ThexMap.Destroy;
 begin
   TempMap.free;
   inherited Destroy;
 end;

{******************************************************************************}
{Hexagon drawing function}
 procedure THexMap.DrawHex(Target:TCanvas;
                          Fillstyle : TBrushStyle;
                          Fillcolor : TColor;
                          Linestyle : TPenStyle;
                          LineColor : Tcolor;
                          x,y,radius: integer;
                          button    : boolean);

 var
   p0,p1,p2,p3,p4,p5,p6:TPoint;
 Begin
   p0 := Point(x,y);

   {compute each point of hex based on center coordinate (p0) }
   p1.x := p0.x - Radius div 2;
   p1.y := p0.y - Radius;
   p2.x := p0.x + Radius div 2;
   p2.y := p1.y;
   p3.x := p0.x + Radius;
   p3.y := p0.y;
   p4.x := p2.x;
   p4.y := p0.y + Radius;
   p5.x := p1.x;
   p5.y := p4.y;
   p6.x := p0.x - Radius;
   p6.y := p0.y;

   {set color / Style of lines}
   target.pen.color := linecolor;
   target.pen.style := linestyle;

   {Set color and style of hex }
   target.brush.color := FillColor;
   target.brush.style := FillStyle;

   {draw the hex}
   target.polygon([p1,p2,p3,p4,p5,p6]);

   {If Desired, draw the borders for the hex}
   if button = true then
   begin
     with target do
     begin
       pen.mode:=pmcopy;
       pen.color:=clWhite;
       moveto(p5.x+1,p5.y-1);
       lineto(p6.x+1,p6.y);
       lineto(p1.x+1,p1.y+1);
       lineto(p2.x-1,p2.y+1);
       pen.color:=clBlack;
       lineto(p3.x-1,p3.y);
       lineto(p4.x-1,p4.y-1);
       lineto(p5.x+1,p5.y-1);
     end;
   end;
 end;

{******************************************************************************}
{ This Function Draws the map                                                  }
 procedure THexMap.paint;
 Begin
   {Copy everything to the Control}
   Canvas.CopyMode := cmSrcCopy;
   Canvas.Draw(0,0,TempMap);
 end;

{******************************************************************************}
{  This function constructs the grid                                           }
 procedure THexMap.makemap;
 var
   p0:tpoint;
   looprow,loopcol:integer;
   hex_id : string;
 begin
   with TempMap.canvas do
   begin
     {set backround color of bitmap}
     brush.color := BackColor;
     fillrect(rect(0,0,TempMap.width,TempMap.height));

     {draw hexes left to right / top to bottom}
     for looprow := 1 to HexRows do
     begin
       for loopcol := 1 to HexColumns do
       begin
         {compute center coordinates}
         p0 := convertcoords(point(Loopcol,Looprow),ptROWCOL);
         {draw the hex}
         drawhex(tempmap.canvas,bsSolid,hexcolor,psSolid,
                    linecolor,p0.x,p0.y,hexradius,hex3d);

         {If desired, draw label for hex}
         if HexShowLabels = true then
         begin
           hex_id := format('%.2d%.2d',[loopcol,looprow]);
           {font := self.font;}
           textout(p0.x - (trunc(textwidth(hex_id) / 2)),
                     p0.y - (textheight(hex_id)),hex_id);


         end;
       end;
     end;
   end;
 end;

{******************************************************************************}
{  This function will return the Row / Col pair based on a given X/Y           }
 function THexMap.ConvertCoords(point:Tpoint;pointtype:Tpointtype):Tpoint;
 var
   offset:integer; {number of pixels to adjust mouse hit }
   temp:TPoint;
 begin
   case PointType of
     ptXY: {Convert from x/y to Row/Col}
     Begin
       {offset is always 1/2 of the radius of a hex.  This number in pixels
       is added to the point.x coordinate }
       offset := HexRadius div 4;
       temp.x := trunc((point.x-offset) / 1.5) div HexRadius;

       if not odd(temp.x) then
         temp.y := (point.y div HexRadius) Div 2
       else
         temp.y := (point.y + HexRadius) div HexRadius div 2;

       { This section insures row / col is good}
       if (temp.x < 1) or (temp.y < 1) then
         begin
           temp.x := 0;
           temp.y := 0;
          end
       else if (temp.y > HexRows) or (temp.x > HexColumns) then
         begin
           temp.y := 0;
           temp.x := 0;
         end;

       ConvertCoords := temp;
     end;

     ptRowCol:  { Converts Row/Col to X/Y }
     begin
       temp.x := point.x * (round(1.5 * hexradius))+hexradius;
       if odd(point.x) then
         temp.y := point.y * (2 * hexRadius)
       else
         temp.y := (point.y * (2*hexRadius))+ hexRadius;

       ConvertCoords := temp;
     end;

   end;
 end;

{******************************************************************************}
{ Hexmap Field Implementations                                                 }
 Procedure THexMap.SetHexColumns(Value:Integer);
 begin
   if Value <> FHexColumns then
   begin
     FHexColumns := Value;
     {Size bitmap and component in order to show all rows / cols}
     tempmap.width := (HexColumns div 2) * (3*HexRadius) + (4*HexRadius);
     width := tempmap.width;
     MakeMap;
     Refresh;
   End;
 End;

 {***}
 Procedure THexMap.SetHexRows(Value:Integer);
 begin
   if Value <> FHexRows then
   begin
     FHexRows := Value;
     {Size bitmap and component in order to show all rows / cols}
     tempmap.height := ((2*HexRadius) * HexRows) + (4*HexRadius);
     height := tempmap.height;
     MakeMap;
     Refresh;
   End;
 End;

 {***}
 Procedure THexMap.SetHexRadius(Value:Integer);
 begin
   if Value <> FHexRadius then
   begin
     FHexRadius := Value;
     if Odd(FHexRadius) then
       inc(FHexRadius); { Even values tend to work better }

     {Size component in order to show all rows / cols}
     width := (HexColumns div 2) * (3*HexRadius) + (4*HexRadius);
     height := ((2*hexRadius) * HexRows) + (4*HexRadius);

     {Size bitmap (Should match component size) }
     TempMap.width := width;
     TempMap.height := height;
     MakeMap;
     Refresh;
   End;
 End;

 {***}
 Procedure THexMap.SetHexShowLabels(Value:Boolean);
 begin
   if Value <> FHexShowLabels then
   begin
     FHexShowLabels := Value;
     MakeMap;
     Refresh;
   End;
 End;

 {***}
 Procedure THexMap.SetHex3d(Value:Boolean);
 begin
   if Value <> FHex3d then
   begin
     FHex3d := Value;
     MakeMap;
     Refresh;
   End;
 End;

 {***}
 Procedure THexMap.SetHexColor(Value: TColor);
 begin
   if Value <> FHexColor then
   begin
     FHexColor := Value;
     MakeMap;
     Refresh;
   End;
 End;

 {***}
 Procedure THexMap.SetBackColor(Value: TColor);
 begin
   if Value <> FBackColor then
   begin
     FBackColor := Value;
     MakeMap;
     Refresh;
   End;
 End;

 {***}
 Procedure THexMap.SetLineColor(Value: TColor);
 begin
   if Value <> FLineColor then
   begin
     FLineColor := Value;
     MakeMap;
     Refresh;
   End;
 End;

{******************************************************************************}
{ Registor procedure to place component on Delphi Component Pallet             }
 procedure Register;
 begin
   RegisterComponents('CUSTOM', [THexMap]);
 end;

end.
