unit strTools;

interface

uses
  Classes,
  SysUtils;

const
  whitespaces: set of char = [#8, #10, #12, #13, ' '];

function  Str2Int(   s: string): longint;
function  Str2Float( s: String): extended;
function  StrLTrim(  s: string): string;
function  StrRTrim(  s: string): string;
function  StrTrim(   s: string): string;
procedure StrMerge( list : TStringList; var s: string; sep : char);
procedure StrParse( var list : TStringList; s: string; sep : char);
procedure StripQuoteChars( var destination: string; strip : char);
procedure StripRChar( var destination: string; strip : char);

implementation

(*
 *  Str2Int
 *
 *  Str2Int can be used as a substitute for the standard StrToInt function.
 *  The StrToInt function throws an exeption if the Text property contains
 *  a non-numeric character like spaces.
 *  This function returns 0 when this exception occurs.
 *)

function Str2Int(s: string): longint;
begin
  s := strTrim( s);
  try
    if s = '' then s := '0';
    Str2Int := StrToInt( s);  (* Try normal conversion *)
  except
    Str2Int := 0;             (* Return 0 when an exception occurs *)
  end;
end;

(*
 *  Str2Float
 *
 *  Str2Float can be used as a substitute for the standard StrToFloat function.
 *  The StrToFloat function throws an exeption if the string property contains
 *  a non-numeric character like spaces, or if the decimal separator is
 *  a comma where it should be a dot or vice versa.
 *  This function takes care of leading and trailing
 *  backspaces and of wrong kind of decinal separator.
 *  The function returns 0 when exceptions occur.
 *)


function Str2Float( s:String): extended;
var
  ct : integer;
begin
  Result:=0;
  s := strTrim( s);
  if Length( s) > 0 then begin
    try
      for ct := 1 to Length( s) do begin
        if s[ ct] in [ '.', ','] then begin
          s[ ct] := DecimalSeparator;
        end;
      end;                  
      Result:=StrtoFloat( s);
    except
      Result:=0;
    end;
  end;
end;



(*
 *  StrLTrim, StrRTrim, StrTrim
 *
 *  These FUNCTIONS will trim leading whitespaces, trailing
 *  whitespaces, or both.
 *)

function StrLTrim( s: string): string;
begin
  while( length( s) > 0) and
         ( s[1] in whitespaces) do delete( s, 1, 1);
  Result := s;
end;

function StrRTrim( s: string): string;
begin
  while( length( s) > 0) and
         ( s[ length( s)] in whitespaces) do
            delete( s, length( s), 1);
  Result := s;
end;

function StrTrim( s: string): string;
begin
  Result := strLTrim( s);
  Result := strRTrim( Result);
end;


(*
 *  StrParse
 *
 *  This procedure parses the given string s and stores it as
 *  separate strings in the given TStrings object. Each new
 *  string is inserted at it's sequential position.
 *  The input string is assumed to be made up out of sequences
 *  of any characters, separated by the char given in 'sep'.
 *  A 'sep' char following the last substring is optional.
 *)

procedure StrParse( var list : TStringList; s: string; sep : char);
var ct, oldct : integer;
    len       : integer;
    seq       : integer;
    t         : string;
begin
  if list <> nil then begin

    ct    := 1;
    seq   := 0;
    while ct <= Length( s) do begin
      oldct  := ct;
      len    := 0;
      while ( s[ ct] <> sep) and
            (ct <= Length( s)) do begin
        ct  := ct  + 1;
        len := len + 1;
      end;
      ct := ct + 1;
      try
        t:= Copy( s, oldct, len);
        list.Insert( seq, StrTrim( t));
      finally
        seq := seq + 1;
      end;
    end;
  end;
end;


(*
 *  StrMerge
 *
 *  StrMerge will take all strings in the list and place these
 *  in a single string, separated by the 'sep' character.
 *  There is no 'sep' char after the last subtring.
 *)

procedure StrMerge( list : TStringList; var s: string; sep : char);
var len       : integer;
    seq       : integer;
begin
  if (list <> nil) and (list.Count > 0) then begin

    len   := Length( list[ 0]) + 1;;
    seq   := 1;
    s     := list[ 0];

    while (len < 250) and (seq < list.Count) do begin
      s   := s   + sep + list[ seq];
      seq := seq + 1;
      if seq > list.Count-1 then begin
        break;
      end;
      len := len + Length( list[ seq]) + 1;
    end;
  end;
end;

(*
 *  StripQuoteChars
 *
 *  will strip the given char from the start and end of the string,
 *  but only if it is at both extremities.
 *)

procedure StripQuoteChars( var destination: string; strip : char);
var l : integer;
begin
  l := Length( destination);
  if l > 2 then begin
    if (destination[ 1] = strip) and
       (destination[ l] = strip) then begin
      destination := Copy( destination, 2, l-2);
    end;
  end;
end;

(*
 *  StripRChar
 *
 *  Will strip the given char from the end of the string if it is there.
 *)

procedure StripRChar( var destination: string; strip : char);
var l : integer;
begin
  l := Length( destination);
  if l > 1 then begin
    if (destination[ l] = strip) then begin
      destination := Copy( destination, 1, l-1);
    end;
  end;
end;


end.
