unit Gridutil;

interface

(*
  These conditionals are here because in my own
  production code I use the EasyINI component
  instead of the standard Delphi IniFile.
  Also, the title caption of the dialog is in Dutch.
*)

{$IFDEF SIMON2NCCW }
uses
  EasyIni,
  StrTools,
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, ExtCtrls, StdCtrls, Buttons, DB, DBTables;
type
  TIniFileType  = TIniEditor;

const
  TitleCaption = 'Zichtbare Kolommen';
{$ELSE }
uses
  IniFiles,
  StrTools,
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, ExtCtrls, StdCtrls, Buttons, DB, DBTables;
type
  TIniFileType  = TIniFile;

const
  TitleCaption = 'Visible grid columns';
{$ENDIF}

type
  TGridLayoutOption  = ( glWidths, glOrder, glVisible);
  TGridLayoutOptions = set of TGridLayoutOption;
  TDlgColumnVisible  = class(TForm)
    PanelButtons: TPanel;
    ButtonClose: TBitBtn;
    ButtonHelp: TBitBtn;
    PanelClient: TPanel;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure CheckBoxClick(Sender: TObject);
    procedure FormShow(Sender: TObject);
  private
    { Private declarations }
    x, y        : integer;
    AllSources  : TList;
    Fcols       : integer;
    FItemHeight : integer;
    FItemWidth  : integer;
    FieldNames  : TStringList;
    procedure SetCols(       Value: integer);
    procedure SetItemHeight( Value : integer);
    procedure SetItemWidth(  Value : integer);
    function  GetDataSource : TDataSource;
    property  DataSource  : TDataSource read GetDataSource;

  public
    { Public declarations }
    procedure AddCheck( descript, fieldname: string);
    function  Run( Sources: array of TDataSource)  : TModalResult;
    procedure Clear;

    property  Cols:       integer read FCols       write SetCols;
    property  ItemHeight: integer read FItemHeight write SetItemHeight;
    property  ItemWidth:  integer read FItemWidth  write SetItemWidth;
  end;

const
  glAll : TGridLayoutOptions = [ glWidths, glOrder, glVisible];

procedure ReadGridLayout(  var ds: TDataSource; opt: TGridLayoutOptions; var ini: TIniFileType; section, id: string);
procedure WriteGridLayout( var ds: TDataSource; opt: TGridLayoutOptions; var ini: TIniFileType; section, id: string);
procedure CopyGridLayout(  var i,o: TDataSource; opt: TGridLayoutOptions);
procedure GetsGridLayout(  var ds: TDataSource; opt: TGridLayoutOptions; var v, o, w: string);
procedure PutsGridLayout(  var ds: TDataSource; opt: TGridLayoutOptions; var v, o, w: string);

var
  DlgColumnVisible : TDlgColumnVisible;

implementation

const
  XGap = 7;
  YGap = 2;

{$R *.DFM}

{
  The datasource property of the column visibility editor
  form is equal to the first datasource that was passed
  in the argument list for the Run method.
  It is a run-time and read-only property.
}

function TDlgColumnVisible.GetDataSource: TDataSource;
begin
  Result := AllSources[ 0];
end;

{
  The SetCols property sets the number of columns
  to any value between 1 and 4 and adjusts the width
  of the editor form.
  The number of columns determines how many columns
  of checkboxes are shown in the editor form.
  Checkboxes are added to the form left-to-right.
  Like all other properties of the editor, it is
  run-time only.
}

procedure TDlgColumnVisible.SetCols( Value: integer);
begin
  if (Value > 0) and (Value < 4) then begin
    FCols       := Value;
    ClientWidth := ItemWidth * Cols + (Cols+2) * XGap + PanelButtons.Width;
  end else begin
    Raise Exception.Create( 'Incorrect number of columns for DlgColumnVisible');
  end;
end;

{
  The itemheight property can be set in the interval
  20 to 100 pixels. It represents the vertical distance
  between the tops of the checbox components.
  An additional gap of YGap pixels is taken into account.
  Like all other properties of the editor, it is
  run-time only.
}

procedure TDlgColumnVisible.SetItemHeight( Value: integer);
begin
  if (Value >= 20) and (Value <= 100) then begin
    FItemHeight := Value;
  end else begin
    Raise Exception.Create( 'Incorrect item width for DlgColumnVisible');
  end;
end;

{
  The ItemWidth property can be set in the interval
  5 to 200 pixels. It represents the width of the
  checkbox components.
  An additional gap of XGap pixels is taken into account
  when more than one column of checkboxes is shown.
  Like all other properties of the editor, it is
  run-time only.
}

procedure TDlgColumnVisible.SetItemWidth( Value: integer);
begin
  if (Value >= 50) and (Value <= 200) then begin
    FItemWidth  := Value;
    ClientWidth := ItemWidth * Cols + (Cols+2) * XGap + PanelButtons.Width;
  end else begin
    Raise Exception.Create( 'Incorrect item width for DlgColumnVisible');
  end;
end;

{
  The FormCreate handler sets some default values
  for the properties. It also creates the stringlist that
  is used to store the fieldname information added by
  repeated calls to the AddCheck method, and the list
  that is used to store all affected datasources passed
  in the open array argument to the Run method.
}

procedure TDlgColumnVisible.FormCreate(Sender: TObject);
begin
  FItemWidth  := 180;
  FItemHeight := 24;
  Cols        := 1;
  x           := XGap;
  y           := YGap;
  FieldNames  := TStringList.Create;
  AllSources  := TList.Create;
  Caption     := TitleCaption;
end;

{
  The FormDestroy handler cleans up the TList and
  TStringList objects that were created in the
  FromCreate event
}

procedure TDlgColumnVisible.FormDestroy(Sender: TObject);
begin
  FieldNames.Free;
  AllSources.Free;
end;

{
  The public AddCheck method adds a checkbox
  to the editor form and then updates the coordinates
  for the next checkbox that will be added.
  The idea is that before you call the Run method,
  you call AddCheck once for each column in the grid
  for which you want the visibility to be edited.
  The first argument becomes the caption of the checkbox,
  the second argument is the FieldName of the field for
  which you want the visibility to be edited.
  The checkboxes are added to the form starting in the
  upper-left corner. I you have more than one column of
  checkboxes, adding goes from left to right until it
  is time for a new line.
  The height of the editor dialog is adjusted
  automatically for each new line of checkboxes added.
}

procedure TDlgColumnVisible.AddCheck( descript, fieldname: string);
var
  check : TCheckBox;
begin

  (* If no more checkboxes fit on the current line,
     advance to the start of the next line. *)

  if (x+ItemWidth+XGap) > PanelClient.Width then begin
    x            := XGap;
    y            := y + ItemHeight + YGap;

    if (y+ItemHeight+YGap) > PanelClient.Height then begin
      (* adjust the height of the editor dialog as well *)
      ClientHeight := y + ItemHeight + YGap;
    end;
  end;

  (* Now create and add the checkbox.
     The OnClick event handler is redirected to
     CheckBoxClick ( a method of TDlgColumnVisible)
     so that visibility of columns is altered as soon
     as the checkbox is clicked.
     The Tag property of the newly created checkbox is
     set to the index value of the name of the associated
     datafield name string as it is stored in the fieldnames
     stringlist. *)

  check          := TCheckBox.Create( Self);
  check.OnClick  := CheckBoxClick;
  check.Top      := y;
  check.Left     := x;
  check.width    := ItemWidth;
  check.height   := ItemHeight;
  check.caption  := descript;
  check.Tag      := FieldNames.Add( fieldname);
  check.Parent   := Self;

  (* If first component on panel, make it active
     on startup *)

  if FieldNames.Count = 1 then begin
    ActiveControl := check;
  end;

  (* Update the Left value for the next checkbox *)
  x := x + ItemWidth + XGap;

  (* Place Help button at bottom of panel *)
  ButtonHelp.Top := PanelButtons.Height - ButtonHelp.Height - ButtonClose.Top;
end;

{
  The private CheckBoxClick method changes the visibility
  in the grid of the data field that is associated with
  the checkbox that has just been clicked by the user.
  The Tag value of the clicked checkbox is also the index
  of the associated fieldname in the FieldNames stringlist
  of the editor dialog.
  The visibility is switched in all of the datasources
  that have been passed in the open array argument to
  the Run method. This allows you to control visibility
  of columns in more that one TDBGrid at the same time
  (I needed this feature myself, so I built it in )
  You are not supposed to call this method from your
  own form. Instead, this method is installed as an
  event handler for all the checkbox components that are
  added with calls to the (public) AddCheck method.
}

procedure TDlgColumnVisible.CheckBoxClick(Sender: TObject);
var
  ct : integer;
begin
  with Sender as TCheckBox do begin
    for ct := 0 to AllSources.Count-1 do begin
      if TDataSource( AllSources[ ct]).DataSet.FindField( FieldNames[ Tag]) <> nil then begin
        TDataSource( AllSources[ ct]).DataSet.FieldByName( FieldNames[ Tag]).Visible := Checked;
      end;
    end;
  end;
end;

{
  The public Clear method resets the dialog for
  re-initialization by a new series of calls
  to the AddCheck method.
  All old checkboxes are removed from the dialog and
  the height of the dialog and the starting
  point for AddCheck are reset to default values.
}

procedure TDlgColumnVisible.Clear;
var
  ct : integer;
begin
  for ct := ControlCount-1 downto 0 do begin
    if Controls[ ct] is TCheckBox then begin
      Controls[ ct].Free;
    end;
  end;
  FieldNames.Clear;
  x    := XGap;
  y    := 2 * YGap;
  Cols := Cols; (* set client width *)
  ClientHeight := 2 * ItemHeight;
end;

{
  The Run function is the public method that actually
  shows the visibility editor dialog.
  You should first setup the dialog by calling its Clear
  method once and its AddCheck method repeatedly (for
  everyfield associated with a grid column for which
  you want to edit visibility).
  During a Run, all the datasources passed to the open
  array argument are stored in the AllSources TList.
  This list is used in the private CheckBoxClick method
  to change visibility in all affected grids as soon
  as a checkbox is clicked.
}

function TDlgColumnVisible.Run( Sources: array of TDataSource) : TModalResult;
var
  ct : integer;
begin

  (* Store the array of datasources that are passed.
     Ususally, there is only a single datasource. *)
  AllSources.Clear;
  for ct := Low( Sources) to High( Sources) do begin
    AllSources.Add( Sources[ ct]);
  end;

  (* Set checkmarks in the checkboxes according to
     the current visibility of the grid columns.
     For this purpose we only look at the first
     datasource that was passed in the open array.
     This datasource is accessed by the DataSource
     property of the dialog *)

  for ct := 0 to ComponentCount-1 do begin
    if Components[ ct] is TCheckBox then begin
      with Components[ ct] as TCheckBox do begin
        if DataSource.DataSet.FindField( FieldNames[ Tag]) <> nil then begin
          Checked := DataSource.DataSet.FieldByName( FieldNames[ Tag]).Visible;
        end;
      end;
    end;
  end;

  (* Finally, run the grid column visibility editor.
     The returned value is not really useful, more like
     a historic weight being schlepped around *)

  Result := ShowModal;
end;

(*****************************************************************)
(*****************************************************************)
(***************                                      ************)
(**************     READING & WRITING LAYOUT         *************)
(*************                                      **************)
(*****************************************************************)
(*****************************************************************)

{
  Reading and writing the grid layout is done by storing in
  and retrieving from a Windows INI file the visibility,
  display widths and order number of each grid column
  as a number.
  These values are stored and retrieved as three separate
  strings of numeric values separated by commas.
  The relative position of a number in one of these
  strings is also the index for the field grid column
  that it is applied to. The first set of numbers applies
  to field index zero, the second to field index 1 etcetera.
  Problem is thatwe cannot use the index property of the
  fields in the dataset Fields[] array to identify fields,
  as the index may change when the order of fields is changed.

  Instead, we identify the fields in the table by getting
  all field names and indexing them according to alfabetical
  order.
}

{
  These string constants are used as suffixes for the
  identifier strings in the windows INI file.
}

const
  WIDTHS   = ' widhts';
  VISIBLE  = ' visible';
  ORDER    = ' order';

{
  ReadGridLayout is an interface procedure to set a certain
  set of values for column order, width and visibility.
  The set of values is read from the given INI file, using
  the given section header and value string prefixes.
  The read values are applied to the dataset associated
  with the given datasource. When you want to do this for
  a subset of the visibility, display width and display order
  of a grid, use another set of option constants instead
  of glAll.
}

procedure ReadGridLayout(  var ds: TDataSource; opt: TGridLayoutOptions; var ini: TIniFileType; section, id: string);
var
  v, o, w: string;
begin
  v := ini.ReadString( section, id+VISIBLE,  '');
  o := ini.ReadString( section, id+ORDER,    '');
  w := ini.ReadString( section, id+WIDTHS,   '');
  GetsGridLayout( ds, opt, v, o, w);
end;

{
  GetsGridLayout is the actual work procedure for applying a
  grid layout to a grid. It receives three strings that
  hold comma-separated integer values that denote visibility,
  order and display width informationfor each column in the
  grid associated with the datasource.
  The string values are broken down into individual values
  using StrParse and then applied to the individual columns
  of the grid.
  The position in which a particular value appears in one of
  the three input strings is the same as the index of
  the associated field in the alphabetically sorted
  fieldnames array of the grid.

  The visibility, width and order are only set when the
  corresponding option value is set in the options array.
}

procedure GetsGridLayout( var ds: TDataSource; opt: TGridLayoutOptions; var v, o, w: string);
var
  flist : TStringList;
  vlist : TStringList;
  olist : TStringList;
  wlist : TStringList;
  ct    : integer;
begin
  try
    flist := TStringList.Create;   (* field list *)
    vlist := TStringList.Create;   (* visibility *)
    olist := TStringList.Create;   (* order      *)
    wlist := TStringList.Create;   (* width      *)

    with ds.Dataset do begin
      (* Get the sorted fieldnames list*)
      TStringList(flist).Sorted := True;
      GetFieldNames( flist);

      (* Breakdown the visibility, order and width data *)
      StrParse( vlist, v, ',');
      StrParse( olist, o, ',');
      StrParse( wlist, w, ',');

      (* Loop around setting the properties *)
      for ct := 0 to flist.Count-1 do begin
        if (glVisible in opt) and (ct < vlist.Count) then begin
          if Str2Int( vlist[ ct]) <> 0 then begin
            FieldByName( flist[ ct]).Visible := True;
          end else begin
            FieldByName( flist[ ct]).Visible := False;
          end;
        end;
        if (glOrder in opt) and (ct < olist.Count) then begin
          FieldByName( flist[ ct]).Index   := Str2Int( olist[ ct]);
        end;
        if (glWidths in opt) and (ct < wlist.Count) then begin
          FieldByName( flist[ ct]).DisplayWidth := Str2Int( wlist[ ct]);
        end;
      end;
    end;
  finally
    flist.Free;
    vlist.Free;
    olist.Free;
    wlist.Free;
  end;
end;

{
  WriteGridLayout is an interface procedure to save a certain
  set of values for column order, width and visibility.
  The set of values is read from the given ds.DataSet and
  written to the given INI file, using the given section
  header and value string prefixes.
  When you want to do this for a subset of the visibility,
  display width and display order of a grid, use another
  set of option constants instead of glAll.
}

procedure WriteGridLayout( var ds: TDataSource; opt: TGridLayoutOptions; var ini: TIniFileType; section, id: string);
var
  v, o, w : string;
begin
  PutsGridLayout( ds, opt, v, o, w);
  ini.WriteString( section, id+VISIBLE, v);
  ini.WriteString( section, id+ORDER,   o);
  ini.WriteString( section, id+WIDTHS,  w);
end;

{
  PutsGridLayout is the actual work procedure for retrieving a
  grid layout from a grid into three string variables.
  It produces three strings that on ooutput hold
  comma-separated integer values that denote visibility,
  order and display width informationfor each column in the
  grid associated with the datasource.
  The string values are merged into strings from then
  individual column values using StrMerge.
  The position in which a particular value appears in one of
  the three output strings is the same as the index of
  the associated field in the alphabetically sorted
  fieldnames array of the grid.

  The visibility, width and order are only gotten when the
  corresponding option value is set in the options array.
}

procedure PutsGridLayout( var ds: TDataSource; opt: TGridLayoutOptions; var v, o, w: string);
var
  flist : TStringList;
  vlist : TStringList;
  olist : TStringList;
  wlist : TStringList;
  ct    : integer;
begin
  try
    flist := TStringList.Create;   (* field list *)
    vlist := TStringList.Create;   (* visibility *)
    olist := TStringList.Create;   (* order      *)
    wlist := TStringList.Create;   (* width      *)

    (* Get the sorted fieldnames *)
    TStringList(flist).sorted := True;
    with ds.DataSet do begin
      GetFieldNames( flist);

      (* Loop around getting the properties *)
      for ct := 0 to flist.Count-1 do begin
        if FieldByName( flist[ ct]).Visible then begin
          vlist.Add( '1');
        end else begin
          vlist.Add( '0');
        end;
        olist.Add( IntToStr( FieldByName( flist[ ct]).Index));
        wlist.Add( IntToStr( FieldByName( flist[ ct]).DisplayWidth));
      end;

      (* Merge and save the visibility, order and width data *)
      if (glVisible in opt) then begin
        StrMerge( vlist, v, ',');
      end;
      if (glOrder in opt) then begin
        StrMerge( olist, o, ',');
      end;
      if (glWidths in opt) then begin
        StrMerge( wlist, w, ',');
      end;
    end;
  finally
    flist.Free;
    vlist.Free;
    olist.Free;
    wlist.Free;
  end;
end;

{
  CopyGridLayout copies the layout data for one grid to
  another. It is necessary that both grids hold
  exactly the same number of fields and name them in the
  same way.
  Again, is only applied to the layout data (order,
  visibility or width) that are indicated by the
  options argument.
}

procedure CopyGridLayout(  var i, o: TDataSource; opt: TGridLayoutOptions);
var
  ilist : TStrings;
  olist : TStrings;
  ct    : integer;
begin
  try
    ilist := TStringList.Create;   (* input  field list *)
    olist := TStringList.Create;   (* output field list *)

    (* Get the sorted fieldnames *)
    TStringList(ilist).Sorted := True;
    i.DataSet.GetFieldNames( ilist);

    TStringList(olist).Sorted := True;
    o.DataSet.GetFieldNames( olist);

    (* Now loop around copying the visibility etc.
       Assume table field layout is exactly the same for i & o
       If not this will yield invalid layout so we may as
       well raise the exception about field index out of range *)

    for ct := 0 to o.DataSet.FieldCount-1 do begin
      if (glVisible in opt) then begin
        o.DataSet.FieldByName( olist[ ct]).Visible := i.DataSet.FieldByName( ilist[ ct]).Visible;
      end;
      if (glOrder in opt) then begin
        o.DataSet.FieldByName( olist[ ct]).Index   := i.DataSet.FieldByName( ilist[ ct]).Index;
      end;
      if (glWidths in opt) then begin
        o.DataSet.FieldByName( olist[ ct]).DisplayWidth := i.DataSet.FieldByName( ilist[ ct]).DisplayWidth;
      end;
    end;

  finally
    ilist.Free;
    olist.Free;
  end;
end;

procedure TDlgColumnVisible.FormShow(Sender: TObject);
begin
  (* Change the display font to whatever is current *)
  Font.Assign( Application.MainForm.Font);
end;

end.

