unit GridLay;

interface

uses
  Classes,
  DB,
  DBTables,
  IniFiles,
  StrTools,
  SysUtils;

type
  TGridLayoutOption = ( glWidths, glOrder, glVisible);
  TGridLayoutOptions = set of TGridLayoutOption;

const
  glAll : TGridLayoutOptions = [ glWidths, glOrder, glVisible];

procedure ReadGridLayout(  var tbl: TTable; opt: TGridLayoutOptions; var ini: TIniFile; section, id: string);
procedure WriteGridLayout( var tbl: TTable; opt: TGridLayoutOptions; var ini: TIniFile; section, id: string);

implementation

const
  WIDTHS   = ' widhts';
  VISIBLE  = ' visible';
  ORDER    = ' order';

(*
  ReadGridLayout will get three strings from the given IniFile,
  identified by the given section and id, and by the ORDER,
  VISIBLE and WIDTHS suffixes.

  These strings are then parsed using StrParse to break them
  down into three lists of values for the visibility, order and
  display width of each fieldin the table.

  The alfabetical sequence of the field names is used as an
  index into these three lists.

*)

procedure ReadGridLayout(  var tbl: TTable; opt: TGridLayoutOptions; var ini: TIniFile; section, id: string);
var
  flist : TStrings;
  vlist : TStrings;
  olist : TStrings;
  wlist : TStrings;
  s     : string;
  ct    : integer;
begin
  try
    flist := TStringList.Create;   (* field list *)
    vlist := TStringList.Create;   (* visibility *)
    olist := TStringList.Create;   (* order      *)
    wlist := TStringList.Create;   (* width      *)

    (* Get the sorted fieldnames *)
    TStringList(flist).Sorted := True;
    tbl.GetFieldNames( flist);

    (* The index of a field in 'flist' now indexes the field
       in the subsequent vlist, olist and wlist string lists *)

    (* Read and breakdown the visibility, order and width data *)
    s := ini.ReadString( section, id+VISIBLE,  '');
    StrParse( vlist, s, ',');
    s := ini.ReadString( section, id+ORDER,    '');
    StrParse( olist, s, ',');
    s := ini.ReadString( section, id+WIDTHS,   '');
    StrParse( wlist, s, ',');

    (* Loop around setting the properties *)
    for ct := 0 to flist.Count-1 do begin
      if (glVisible in opt) and (ct < vlist.Count) then begin
        if StrToInt( vlist[ ct]) <> 0 then begin
          tbl.FieldByName( flist[ ct]).Visible := True;
        end else begin
          tbl.FieldByName( flist[ ct]).Visible := False;
        end;
      end;
      if (glOrder in opt) and (ct < olist.Count) then begin
        tbl.FieldByName( flist[ ct]).Index   := StrToInt( olist[ ct]);
      end;
      if (glWidths in opt) and (ct < wlist.Count) then begin
        tbl.FieldByName( flist[ ct]).DisplayWidth := StrToInt( wlist[ ct]);
      end;
    end;
  finally
    flist.Free;
    vlist.Free;
    olist.Free;
    wlist.Free;
  end;
end;

(*
  WriteGridLayout will put three strings to the given IniFile,
  identified by the given section and id, and by the ORDER,
  VISIBLE and WIDTHS suffixes.

  It is supposed to do the exact reverse of ReadGridLayout.
*)


procedure WriteGridLayout( var tbl: TTable; opt: TGridLayoutOptions; var ini: TIniFile; section, id: string);
var
  flist : TStrings;
  vlist : TStrings;
  olist : TStrings;
  wlist : TStrings;
  s     : string;
  ct    : integer;
begin
  try
    flist := TStringList.Create;   (* field list *)
    vlist := TStringList.Create;   (* visibility *)
    olist := TStringList.Create;   (* order      *)
    wlist := TStringList.Create;   (* width      *)

    (* Get the sorted fieldnames *)
    TStringList(flist).sorted := True;
    tbl.GetFieldNames( flist);

    (* Loop around getting the properties *)
    for ct := 0 to flist.Count-1 do begin
      if tbl.FieldByName( flist[ ct]).Visible then begin
        vlist.Add( '1');
      end else begin
        vlist.Add( '0');
      end;
      olist.Add( IntToStr( tbl.FieldByName( flist[ ct]).Index));
      wlist.Add( IntToStr( tbl.FieldByName( flist[ ct]).DisplayWidth));
    end;

    (* Merge and save the visibility, order and width data *)
    if (glVisible in opt) then begin
      StrMerge( vlist, s, ',');
      ini.WriteString( section, id+VISIBLE, s);
    end;
    if (glOrder in opt) then begin
      StrMerge( olist, s, ',');
      ini.WriteString( section, id+ORDER, s);
    end;
    if (glWidths in opt) then begin
      StrMerge( wlist, s, ',');
      ini.WriteString( section, id+WIDTHS, s);
    end;
  finally
    flist.Free;
    vlist.Free;
    olist.Free;
    wlist.Free;
  end;
end;

end.
