unit FxImage;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, ExtCtrls, Menus, FxRtns,FxForms, DsgnIntf;


(*										*)
(*			TFxGraphic Class			*)
(*										*)
type TFXGraphic = class(TGraphicControl)
	private
    FPicture: TPicture;
    srcCanvas, dstCanvas : TCanvas;
	FBoxWidth : byte;
    FEffect : byte;
    function GetCanvas: TCanvas;
    procedure SetCanvases;
    procedure SetBoxWidth(Value : Byte);
    procedure SetEffect(Value: Byte);
  protected
    procedure Paint; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property Canvas: TCanvas read GetCanvas;
  published
    property Effect:Byte read FEffect write SetEffect default 0;
  	property BoxWidth: Byte read FBoxWidth write SetBoxWidth default 5;
  end;

(*										*)
(*				TFxImage Class			*)
(*										*)
type
  TFxImage = class(TFxGraphic)
  private
    FAutoSize: Boolean;
    FReserved: Byte;
    procedure PictureChanged(Sender: TObject);
    procedure SetAutoSize(Value: Boolean);
    procedure SetPicture(Value: TPicture);
  protected
    function GetPalette: HPALETTE; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property Align;
    property AutoSize: Boolean read FAutoSize write SetAutoSize default False;
    property BoxWidth;
    property DragCursor;
    property DragMode;
    property Enabled;
    property ParentShowHint;
    property Picture: TPicture read FPicture write SetPicture;
    property PopupMenu;
    property ShowHint;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;


(*										*)
(*				TFxLabel Class			*)
(*										*)
type
	TTextStyle = ( tsNone, tsRaised, tsRecessed );

type
  TFxLabel = class(TFxGraphic)
  private
    FAlignment: TAlignment;
    FTextStyle : TTextStyle;
    FBevelInner: TPanelBevel;
    FBevelOuter: TPanelBevel;
    FBevelWidth: TBevelWidth;
    FBorderWidth: TBorderWidth;
    FBorderStyle: TBorderStyle;
	Bitmap : TBitmap;
    procedure SetBitmap;
    procedure AdjustBounds;
    procedure DoDrawText(var Rect: TRect; Flags: Word);
    procedure SetAlignment(Value: TAlignment);
    procedure SetTextStyle( Value : TTextStyle );
    procedure SetBevelInner(Value: TPanelBevel);
    procedure SetBevelOuter(Value: TPanelBevel);
    procedure SetBevelWidth(Value: TBevelWidth);
    procedure SetBorderWidth(Value: TBorderWidth);
    procedure SetBorderStyle(Value: TBorderStyle);
    procedure CMTextChanged(var Message: TMessage); message CM_TEXTCHANGED;
    procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
  protected
    procedure Paint; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property Align;
    property Alignment: TAlignment read FAlignment write SetAlignment default taLeftJustify;
    property BevelInner: TPanelBevel read FBevelInner write SetBevelInner default bvNone;
    property BevelOuter: TPanelBevel read FBevelOuter write SetBevelOuter default bvRaised;
    property BevelWidth: TBevelWidth read FBevelWidth write SetBevelWidth default 1;
    property BorderWidth: TBorderWidth read FBorderWidth write SetBorderWidth default 0;
    property BorderStyle: TBorderStyle read FBorderStyle write SetBorderStyle default bsNone;
    property Caption;
    property Color;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TextStyle : TTextStyle read FTextStyle write SetTextStyle default tsRecessed;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

procedure Register;

implementation

(*												*)
(*		TFxGraphic methods implementation	    *)
(*												*)
constructor TFxGraphic.Create(AOwner: TComponent);
begin
  	inherited Create(AOwner);
	FBoxWidth := 5;
end;

(*		*)
destructor TFxGraphic.Destroy;
begin
	inherited Destroy;
end;

(*		*)
function TFxGraphic.GetCanvas: TCanvas;
var Bitmap: TBitmap;
begin
	if FPicture.Graphic = nil then begin
    	Bitmap := TBitmap.Create;
	    try
      		Bitmap.Width := Width;
      		Bitmap.Height := Height;
      		FPicture.Graphic := Bitmap;
        finally
      		Bitmap.Free;
    	end;
  	end;
  	if FPicture.Graphic is TBitmap then
    	Result := TBitmap(FPicture.Graphic).Canvas
  	else raise EInvalidOperation.Create('SImageCanvasNeedsBitmap');
end;

(*		*)
procedure TFxGraphic.Paint;
begin
  	if csDesigning in ComponentState then
  	with inherited Canvas do begin
  		Pen.Style := psDash;
	    Brush.Style := bsClear;
    	Rectangle(0, 0, Width, Height);
    end;
	if srcCanvas=nil then exit;
    PaintEffect(FEffect,srcCanvas,dstCanvas,FBoxWidth);
end;

(*		*)
procedure TFxGraphic.SetBoxWidth(Value : Byte);
begin
	if Value<2 then Value:=2;
    FBoxWidth := Value;
end;

(*		*)
procedure TFxGraphic.SetEffect(Value:byte);
begin
	if Value<0 then Value:=0;
    if Value>41 then Value:=41;
    FEffect := Value;
end;


(*		*)
procedure TFxGraphic.SetCanvases;
begin
	if FPicture.Graphic=nil
    	then srcCanvas := nil
        else srcCanvas := TBitmap(FPicture.Graphic).Canvas;
  	dstCanvas := inherited Canvas;
end;


(*												*)
(*		TFxImage  methods implementation		*)
(*												*)
constructor TFxImage.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FPicture := TPicture.Create;
  FPicture.OnChange := PictureChanged;
  Height := 105;
  Width := 105;
  SetCanvases;
end;

destructor TFxImage.Destroy;
begin
  FPicture.Free;
  inherited Destroy;
end;


function TFxImage.GetPalette: HPALETTE;
begin
	if FPicture.Graphic=nil then Result := 0
    else Result := TBitmap(FPicture.Graphic).Palette;
end;


procedure TFxImage.SetAutoSize(Value: Boolean);
begin
  FAutoSize := Value;
  PictureChanged(Self);
end;


procedure TFxImage.SetPicture(Value: TPicture);
begin
	if (Value.Graphic=nil) or (Value.Graphic is TBitmap) then
    begin
        FPicture.Assign(Value);
        SetCanvases;
    end;
end;


procedure TFxImage.PictureChanged(Sender: TObject);
begin
  if AutoSize and (Picture.Width > 0) and (Picture.Height > 0) then begin
    SetBounds(Left, Top, Picture.Width, Picture.Height);
  end;
  if (Picture.Width = Width) and (Picture.Height = Height)
  	then ControlStyle := ControlStyle + [csOpaque]
    else ControlStyle := ControlStyle - [csOpaque];
  SetCanvases;
  if not(csFocusing in ControlState) then Invalidate;
end;


(**									**)
(**				TFxLabel			**)
(**									**)
constructor TFxLabel.Create(AOwner: TComponent);
begin
  	inherited Create(AOwner);
  	ControlStyle := ControlStyle + [csOpaque];
  	Width := 100;
  	Height := 30;
    FPicture := TPicture.Create;
    SetBitmap;
    FTextStyle := tsRaised;
  	FAlignment := taCenter;
    BevelOuter := bvRaised;
    BevelInner := bvNone;
  	BevelWidth := 1;
    BorderWidth := 0;
  	FBorderStyle := bsNone;
  	Color := clBtnFace;
end;

destructor TFxLabel.Destroy;
begin
	Bitmap.Free;
	FPicture.Free;
    inherited Destroy;
end;

procedure TFxLabel.SetBitmap;
begin
	if Bitmap<>nil then Bitmap.Free;
	Bitmap := TBitmap.Create;
    Bitmap.Width := Width;
    Bitmap.Height := Height;
    FPicture.Graphic := Bitmap;
    SetCanvases;
end;

procedure TFxLabel.DoDrawText(var Rect: TRect; Flags: Word);
var Text: array[0..255] of Char;
begin
  	GetTextBuf(Text, SizeOf(Text));
  	if (Flags and DT_CALCRECT <> 0) and (Text[0] = #0) then StrCopy(Text, ' ');
  	Flags := Flags or DT_NOPREFIX;
  	srcCanvas.Font := Font;
  	if not Enabled then srcCanvas.Font.Color := clGrayText;
  	DrawText(srcCanvas.Handle, Text, StrLen(Text), Rect, Flags);
end;


procedure TFxLabel.Paint;
const Alignments: array[TAlignment] of Word = (DT_LEFT, DT_RIGHT, DT_CENTER);
var Rect: TRect;
  	TopColor, BottomColor: TColor;
    FontHeight : integer;
    Text: array[0..255] of Char;
    TmpRect    : TRect;
    UpperColor : TColor;
    LowerColor : TColor;
    flags : integer;

  	procedure AdjustColors(Bevel: TPanelBevel);
  	begin
    	TopColor := clBtnHighlight;
    	if Bevel = bvLowered then TopColor := clBtnShadow;
    	BottomColor := clBtnShadow;
    	if Bevel = bvLowered then BottomColor := clBtnHighlight;
  	end;

begin
	AdjustBounds;
    flags := (DT_EXPANDTABS or DT_VCENTER) or Alignments[FAlignment];
    Rect := srcCanvas.ClipRect;
    if BorderStyle=bsSingle then begin
    	srcCanvas.Brush.Style := bsClear;
        srcCanvas.Brush.Color := clBlack;
        srcCanvas.FrameRect(rect);
        InflateRect(Rect,-1,-1);
    end;
    if BevelOuter <> bvNone then begin
    	AdjustColors(BevelOuter);
        Frame3D(srcCanvas, Rect, TopColor, BottomColor, BevelWidth);
    end;
    Frame3D(srcCanvas, Rect, Color, Color, BorderWidth);
    if BevelInner <> bvNone then begin
    	AdjustColors(BevelInner);
        Frame3D(srcCanvas, Rect, TopColor, BottomColor, BevelWidth);
    end;
    srcCanvas.Brush.Color := Self.Color;
    srcCanvas.Brush.Style := bsSolid;
    srcCanvas.FillRect(Rect);
    srcCanvas.Brush.Style := bsClear;
   	srcCanvas.Font := Self.Font;
    FontHeight := srcCanvas.TextHeight('W');
    with Rect do begin
    	Top := ((Bottom + Top) - FontHeight) shr 1;
        Bottom := Top + FontHeight;
    end;
    UpperColor := clBtnHighlight;
    LowerColor := clBtnShadow;
    if FTextStyle = tsRecessed then begin
    	UpperColor := clBtnShadow;
        LowerColor := clBtnHighlight;
    end;
    StrPCopy(Text, Caption);
    if FTextStyle in [ tsRecessed, tsRaised ] then begin
		TmpRect := Rect;
        OffsetRect( TmpRect, 1, 1 );
        srcCanvas.Font.Color := LowerColor;
        DrawText(srcCanvas.Handle, Text, StrLen(Text), TmpRect, Flags);
        TmpRect := Rect;
        OffsetRect( TmpRect, -1, -1 );
        srcCanvas.Font.Color := UpperColor;
        DrawText(srcCanvas.Handle, Text, StrLen(Text), TmpRect, Flags);
    end;
    srcCanvas.Font.Color := Font.Color;
    DrawText(srcCanvas.Handle, Text, StrLen(Text), Rect,Flags);
    inherited Paint;
end;

procedure TFxLabel.AdjustBounds;
var X: Integer;
  	Rect: TRect;
    ovh:integer;
begin
	if not (csReading in ComponentState) then begin
    	SetBounds(Left, Top, Width, Height);
        SetBitmap;
  	end;
end;

procedure TFxLabel.SetAlignment(Value: TAlignment);
begin
	if FAlignment <> Value then begin
    	FAlignment := Value;
    	Invalidate;
  	end;
end;

procedure TFxLabel.SetTextStyle( Value : TTextStyle );
begin
	if Value <> FTextStyle then begin
    	FTextStyle := Value;
      	Invalidate;
    end;
end;


procedure TFxLabel.SetBevelInner(Value: TPanelBevel);
begin
  	FBevelInner := Value;
  	Invalidate;
end;

procedure TFxLabel.SetBevelOuter(Value: TPanelBevel);
begin
  	FBevelOuter := Value;
  	Invalidate;
end;

procedure TFxLabel.SetBevelWidth(Value: TBevelWidth);
begin
  	FBevelWidth := Value;
  	Invalidate;
end;

procedure TFxLabel.SetBorderWidth(Value: TBorderWidth);
begin
  	FBorderWidth := Value;
  	Invalidate;
end;

procedure TFxLabel.SetBorderStyle(Value: TBorderStyle);
begin
	if FBorderStyle <> Value then begin
    	FBorderStyle := Value;
	    Invalidate;
  	end;
end;


procedure TFxLabel.CMTextChanged(var Message: TMessage);
begin
  Invalidate;
  AdjustBounds;
end;

procedure TFxLabel.CMFontChanged(var Message: TMessage);
begin
  inherited;
  AdjustBounds;
end;


procedure Register;
begin
  	RegisterComponents('ASD', [TFxImage,TFxLabel]);
    RegisterPropertyEditor(TypeInfo(byte), TFxGraphic, 'Effect', TEffectProperty);
end;

begin
	Randomize;
end.
