unit Fincalc;

interface

Function TFV( pv, int : real; per : integer): Real;
Function TPV( fv, int : real; per : integer): Real;
Function TPVa( pmt, int : real; per : integer): Real;
Function TFVa( pmt, int : real; per : integer): Real;
Function TPMT( prn, int : real; per : integer): Real;
function TExp(Number, Exponent: Double): Double;

implementation

Function TFV( pv, int : real; per : integer): Real;
{****************************************************************************
 Description:      Calculates the future value of an initial amount
 Parameters:       pv - present value of an amount
                   int - annual interest rate
                   per - number of months
 Modified:         02/03/96
 By:               Mark D. Ungerman
****************************************************************************}
begin
  {normalize int rate}
  if int > 1 then
     int := int / 100 / 12
  else
     int := int / 12;

  Result := pv * TExp(1 + int, per)

end;

Function TPV( fv, int : real; per : integer): Real;
{****************************************************************************
 Description:      Calculates the present value of a future amount
 Parameters:       amt - future value of an amount
                   int - annual interest rate
                   per - number of months
 Modified:         02/03/96
 By:               Mark D. Ungerman
****************************************************************************}
begin

  {normalize int rate}
  if int > 1 then
     int := int / 100 / 12
  else
     int := int / 12;

  Result := fv / TExp(1 + int, per)

end;

Function TPVa( pmt, int : real; per : integer): Real;
{****************************************************************************
 Description:      Calculates the present value of a future annuity
 Parameters:       pmt - annuity payment
                   int - discount rate
                   per - number of periods (in months)
 Modified:         02/03/96
 By:               Mark D. Ungerman
****************************************************************************}
begin

  {normalize int rate}
  if int > 1 then
     int := int / 100 / 12
  else
     int := int / 12;

  Result := pmt * ( (1 - TExp(1 + int, -per) ) / int);

end;

Function TFVa( pmt, int : real; per : integer): Real;
{****************************************************************************
 Description:      Calculates the future value of a annuity
 Parameters:       pmt - annuity payment
                   int - discount rate
                   per - number of periods (in months)
 Modified:         02/03/96
 By:               Mark D. Ungerman
****************************************************************************}
begin

  {normalize int rate}
  if int > 1 then
     int := int / 100 / 12
  else
     int := int / 12;

  Result := pmt * ( (TExp(1 + int, per) - 1) / int);

end;


Function TPMT( Prn, int : real; per : integer): Real;
{****************************************************************************
 Description:      Calculates the payment of an amount
 Parameters:       prn - Principle amount
                   int - discount rate
                   per - number of periods (in months)
 Modified:         02/03/96
 By:               Mark D. Ungerman
****************************************************************************}
begin

  {normalize int rate}
  if int > 1 then
     int := int / 100 / 12
  else
     int := int / 12;

  Result := prn / ( (1 - TExp(1 + int, -per) ) / int);

end;

function TExp(Number, Exponent: Double): Double;
{****************************************************************************
 Description:      Calculates the result of a base number raised to some power
 Parameters:       Number - base value
                   Exponent - Power base is raised to
 Modified:         02/03/96
 By:               Mark D. Ungerman  - Based on [ts]'s Turbo Pascal FAQ
****************************************************************************}
Var
    TResult : Real;

begin
   if (Exponent = 0.0) then
      TResult := 1.0
   else
      if Number = 0.0 then
         TResult := 0.0
      else
         if Abs(Exponent*Ln(Abs(Number))) > 87.498 then
            RunError(205)                 {Floating point overflow}
         else
            if Number > 0.0 then
               TResult := Exp(Exponent*Ln(Number))
            else
               if (Number < 0.0) and (Frac(Exponent) = 0.0) then
                  if Odd(Round(Exponent)) then
                     TResult := -TExp(-Number, Exponent)
                  else TResult :=  TExp(-Number, Exponent)
               else RunError(207);        {Invalid float-op}

   Result := TResult;
end;  { TExp }

end.
