unit Filecopy;

{-----------------------------------------------------------------------}
{                                                                       }
{                    by G.Burzomato and L.Menghini                      }
{                                                                       }
{                    e-mail : lume@tn.village.it                        }
{                                                                       }
{-----------------------------------------------------------------------}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs;

type
  TFileCopy = class(TComponent)
  private
    { Private declarations }
    FSourceFile : string;
    FTargetFile : string;
    FShowOverwrite : boolean;
    procedure SetSourceFile(value : string);
    procedure SetTargetFile(value : string);
    procedure SetShowOverwrite(value : boolean);
    function  Execute(source_path,target_path : string) : boolean;
  protected
    { Protected declarations }
  public
    { Public declarations }
    procedure ExecCopy;
    constructor Create(AOwner: TComponent); override;
  published
    { Published declarations }
    property SourceFile : string read FSourceFile write SetSourceFile;
    property TargetFile : string read FTargetFile write SetTargetFile;
    property ShowOverwrite : boolean read FShowOverwrite
                    write SetShowOverwrite default true;
  end;

procedure Register;

const
   wm_FileCopy = wm_user + 7777;

implementation



procedure Register;
begin
  RegisterComponents('System', [TFileCopy]);
end;

constructor TFileCopy.Create(AOwner: TComponent);
begin
   inherited Create(AOwner);
   FSourceFile := '';
   FTargetFile := '';
   FShowOverwrite := true;
end;

procedure TFileCopy.SetSourceFile(value : string);
begin
   if value <> FSourceFile then FSourceFile := Value;
end;

procedure TFileCopy.SetTargetFile(value : string);
begin
   if value <> FTargetFile then FTargetFile := Value;
end;

procedure TFileCopy.SetShowOverwrite(value : boolean);
begin
   if value <> FShowOverwrite then FShowOverwrite := value;
end;

procedure TFileCopy.ExecCopy;
begin
   if FSourceFile = '' then
   begin
      ShowMessage('I can''t copy' + chr(13) +
                  'Source File name missing');
      exit;
   end;
   if FTargetFile = '' then
   begin
      ShowMessage('I can''t copy' + chr(13) +
                  'Target File name missing');
      exit;
   end;
   if not Execute(SourceFile,TargetFile) then
     showMessage('Execution failed');
end;

function TFileCopy.Execute(source_path,target_path : string) : boolean;
const
   BlockSize = $4000;
type
    pBuf = ^tBuf;
    tBuf = array[1..BlockSize] of char;
var
  SourSize           : LongInt;
  nBlocks            : real;
  pCentRate          : real;
  rRead, rWrite      : real;
  pCentRead          : word;    { % read blocks to send in the message }
  pCentWrite         : LongInt; { % written blocks to send in the message }
  sour               : tFileRec;
  targ               : tFileRec;
  nReaded,nWritten   : integer;
  buf                : pBuf;
  oldCursor          : hCursor;
  Ow                 : TForm;

  begin

     Ow := Owner as TForm;
     Execute := false;

    { if source and target are identical exit showing an error }
    if source_path = target_path then
    begin
      MessageBeep(0);
      oldCursor := screen.cursor;
      Screen.cursor := crDefault;
      ShowMessage('Error !!!' + chr(13) +
                  'Can''t copy the file on itself');
      screen.cursor := oldCursor;
      SendMessage(Ow.handle, WM_PAINT, 0, 0);
      exit;
    end;
    new(buf);

    { tries to open the source file }
    Sour.handle := fileOpen(source_path, fmOpenRead);

    { counts the number of blocks needed to copy }
    SourSize := fileSeek(Sour.handle, 0, 2);
    if SourSize < 0 then
    begin
       MessageBeep(0);
       oldCursor := screen.cursor;
       Screen.cursor := crDefault;
       ShowMessage(source_path + ' missing or damaged');
       screen.cursor := oldCursor;
       dispose(buf);
       SendMessage(Ow.handle, WM_PAINT, 0, 0);
       exit;
    end;

    { seeks the cursor at the beginning of the file }
    fileSeek(Sour.handle, 0, 0);

    nBlocks := SourSize*1.0 / BlockSize;
    { counts the percent rate for each block }
    pCentRate := 100.0 / nBlocks;
    { brings to zero all the percents }
    rRead      := 0;
    rWrite     := 0;
    pCentRead  := 0;
    pCentWrite := 0;

    if FileExists(target_path) then
    begin
        oldCursor := screen.cursor;
        Screen.cursor := crDefault;
        if MessageDlg('The file ' + target_path + chr(13) +
                       ' already exists. Overwrite ?',
          mtInformation, [mbYes, mbNo], 0) = mrNo then
        begin
           FileClose(Sour.Handle);
           dispose(buf);
           screen.cursor := oldCursor;
           SendMessage(Ow.handle, WM_PAINT, 0, 0);
           exit;
        end;
        screen.cursor := oldCursor;
        SendMessage(Ow.handle, WM_PAINT, 0, 0);
    end;

    { tries to create the target file }
    Targ.handle := FileCreate(target_path);
    if targ.handle < 0 then
    begin
       FileClose(Sour.Handle);
       dispose(buf);
       MessageBeep(0);
       oldCursor := screen.cursor;
       Screen.cursor := crDefault;
       ShowMessage('Error on creating ' + target_path);
       screen.cursor := oldCursor;
       SendMessage(Ow.handle, WM_PAINT, 0, 0);
       exit;
    end;

    { starts to copy }
    repeat
      { reads }
      nReaded := FileRead(Sour.Handle,Buf^, sizeOf(buf^));
      if nReaded < 0 then
      begin
         FileClose(Sour.Handle);
         FileClose(Targ.handle);
         dispose(buf);
         MessageBeep(0);
         oldCursor := screen.cursor;
         Screen.cursor := crDefault;
         ShowMessage('Error reading ' + chr(13) + source_path
           + chr(13) + 'Copy failed.');
         screen.cursor := oldCursor;
         SendMessage(Ow.handle, WM_PAINT, 0, 0);
         exit;
      end;
      { sends mess. with read percent }
      rRead := rRead + pCentRate;
      pCentRead := trunc(rRead);
      SendMessage(Ow.handle, wm_FileCopy, pCentRead, pCentWrite);

      { writes }
      nWritten := FileWrite(Targ.Handle, Buf^, nReaded);
      if nWritten < 0 then
      begin
         FileClose(Sour.Handle);
         FileClose(Targ.handle);
         dispose(buf);
         MessageBeep(0);
         oldCursor := screen.cursor;
         Screen.cursor := crDefault;
         ShowMessage('Error writing ' + chr(13) + target_path
           + chr(13) + 'Copy failed.');
         screen.cursor := oldCursor;
         SendMessage(Ow.handle, WM_PAINT, 0, 0);
         exit;
      end;
      { sends mess. with written percent }
      rWrite := rWrite + pCentRate;
      pCentWrite := trunc(rWrite);
      SendMessage(Ow.handle, wm_FileCopy, pCentRead, pCentWrite);
    until (nReaded = 0) or (nWritten <> nReaded);

    FileClose(Sour.Handle);
    FileClose(Targ.handle);
    dispose(buf);
    SendMessage(Ow.handle, wm_FileCopy, 100, 100);
    Execute := True;
  end;




end.
