(* 
 * Enhanced 32-bit FileListBox (Freeware)
 * 
 * 
 * Author: Paul K.F. Leung
 * Date:   16th Dec, 95
 * Email:  cs_paul@ug.cs.ust.hk
 *)

unit File32;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, FileCtrl, Call32NT, LongName;

const
  DefaultDir = 'c:\';
  DefaultMask = '*.*';

type
  ttime = array[0..1] of longint;
  fd = record
    dwFileAttributes:longint;
    ftCreationTime,
    ftLastAccessTime,
    ftLastWriteTime:ttime;
    nFileSizeHigh,
    nFileSizeLow,
    dwReserved0,
    dwReserved1:longint;
    cFileName:array[0..259] of char;
    cAlternateFileName:array[0..13] of char;
  end;
  pfd = ^fd;

  TFileType = ( ftHidden , ftSystem, ftArchive, ftReadOnly );
  TFileTypeSet = set of TFileType;

  TFile32ListBox = class(TListBox)
  private
    { Private declarations }
    FDirectory : string;
    FHandle:longint;
    FFileType : TFileTypeSet;
    FMask: string;
    procedure SetDirectory(Value: string);
  protected
    { Protected declarations }
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    procedure Update; override;
  published
    { Published declarations }
    property Directory : string read FDirectory write SetDirectory;
    property FileType : TFileTypeSet read FFileType write FFileType;
    property Mask : string read FMask write FMask;
  end;

procedure Register;

{Declaration of the 32 bit functions}
var
  W32FindFirstFile:
    function(lpszSearchFile:pchar;var lpffd:fd;id:longint):longint;
  W32FindNextFile:
    function(hFindFile:longint;var lpffd:fd;id:longint):longbool;
  W32FindClose:
    function(hFindFile:longint;id:longint):Longbool;

  {Declaration of a unique identifier for each 32 bit function}
  id_W32FindFirstFile,
  id_W32FindNextFile,
  id_W32FindClose : LongInt;
  lr : fd;
  localtime:ttime;
  ok:longbool;
  i:integer;


implementation

constructor TFile32ListBox.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);              {call inherited constructor}
  FDirectory := DefaultDir;
  FMask := DefaultMask;
end;


procedure TFile32ListBox.SetDirectory(Value: string);
begin
  if Value[Length(Value)] <> '\' then
    FDirectory := Value + '\'
  else
    FDirectory := Value;
  Update;
end;


procedure TFile32ListBox.Update;
var
  DirPtr : PChar;
  tmp : string;
  FAttr: LongInt;
begin
  tmp := FDirectory + Mask + #0;
  FAttr := 0;
  if ftArchive in FFileType then Inc(FAttr, 32);
  if ftReadOnly in FFileType then Inc(FAttr, 1);
  if ftHidden in FFileType then Inc(FAttr, 2);
  if ftSystem in FFileType then Inc(FAttr, 4);
  DirPtr := @tmp[1];
  FHandle := W32FindFirstFile(DirPtr, lr, id_W32FindFirstFile);
  Items.Clear;
  if FHandle <> -1 then
  repeat
    if (StrComp(lr.cfilename, '.') <> 0) and
       (StrComp(lr.cfilename, '..') <> 0) and
       (lr.dwFileAttributes >= FAttr) then
      Items.Add(StrPas(lr.cfilename));
    ok:=W32FindNextFile(FHandle, lr, id_W32FindNextFile);
  until not ok;
  W32FindClose(FHandle, id_W32FindClose);
end;


procedure Register;
begin
  RegisterComponents('Samples', [TFile32ListBox]);
end;


initialization
  {Initialization of the 32 bit functions}
  @W32FindFirstFile:=@Call32;
  @W32FindNextFile:=@Call32;
  @W32FindClose:=@Call32;

  id_W32FindFirstFile:=Declare32('FindFirstFile', 'kernel32', 'pp');
  id_W32FindNextFile:=Declare32('FindNextFile', 'kernel32', 'ip');
  id_W32FindClose:=Declare32('FindClose', 'kernel32', 'i');

  {Check if everything went well. If there was only a single error,
   Call32NTError=false}
  if Call32NTError then begin
    ShowMessage('FileListBox: Cannot load the desired 32 bit functions!');
    halt(1);
  end;
end.


