(*
 *  TExplorerButton and TOfficeButton version 2.6
 *  (c)1996-1997 Fabrice Deville
 *  fdev@tornado.be
 *  http://www.tornado.be/~fdev/
 *
 *  Accelerator support thanks to Fred Schetterer
 *
 *  Freeware - compiles under Delphi 1.0 & 2.0
 *
 *  Source code provided under some conditions:
 *
 *  - if you use significant parts of the code, please credit me
 *  - do not release a new version of this component without
 *    significant enhancements
 *
 *)

unit explbtn;

interface

uses
  WinTypes, WinProcs, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Menus, dsgnIntf, ExtCtrls;

type
	TAboutExpButtonProperty = class(TPropertyEditor)
	public
		procedure Edit; override;
		function GetAttributes: TPropertyAttributes; override;
		function GetValue: string; override;
	end;

	TExplorerButtonLayout = (blBitmapTop, blBitmapBottom, blBitmapLeft, blBitmapRight);
	TExplorerButtonOption = (boMonoDisplay, boIconOnly, boPopupMark, boShowBevel);
	TExplorerButtonOptions = set of TExplorerButtonOption;
	TExplorerButtonShadingType = (stLight, stMedium, stDark);

	TExplorerButton = class(TGraphicControl)
	private
  	(* properties storage *)
		FAbout: TAboutExpButtonProperty;
		FAlignment: TAlignment;
		FAllowAllUp: Boolean;
		FBevelStyle: TBevelStyle;
		FBitmap, FNoFocusBitmap, FDisabledBitmap, IBitmap: TBitmap;
		FCaption: TCaption;
		FDown: Boolean;
		FDropDown: TPopupMenu;
		FEnabled: Boolean;
		FGroupIndex: Integer;
		FLayout: TExplorerButtonLayout;
		FOptions: TExplorerButtonOptions;
		FShadingType: TExplorerButtonShadingType;
		FShowDownPattern: Boolean;
		FUnselectedFontColor: TColor;
		FWordWrap: Boolean;
                FOnEnter, FOnExit: TNotifyEvent;
		(* state flags *)
		Pushed, MouseIn: Boolean;
	protected
		procedure DefineProperties(Filer: TFiler); override;
		procedure ReadIBitmap(Stream: TStream);
		procedure WriteIBitmap(Stream: TStream);
		procedure CMDialogChar(var Message: TCMDialogChar); message CM_DIALOGCHAR;
		procedure ComputeExtent(TempCaption: PChar; var TempRect: TRect);
		procedure CreateGrayscaleBitmap(outputbmp, bmp: TBitmap);
		procedure DrawTheText(theCanvas: TCanvas; TempRect: TRect; TempCaption: PChar);
		procedure DrawDisabledText(theCanvas: TCanvas; TempRect: TRect; TempCaption: PChar);
		procedure DrawPopupMark(theCanvas: TCanvas; x, y: Integer);
		procedure DrawOutline(theCanvas: TCanvas; pushed: Boolean);
		procedure GetLost;
		procedure Loaded; override;
		procedure Notification(AComponent: TComponent; Operation: TOperation); override ;
		procedure BitmapChange(Sender: TObject);
		procedure NoFocusBitmapChange(Sender: TObject);
		procedure DisabledBitmapChange(Sender: TObject);
		procedure Paint; override;
		procedure SetAlignment(value: TAlignment);
		procedure SetAllowAllUp(value: Boolean);
		procedure SetBevelStyle(value: TBevelStyle);
		procedure SetBitmap(value: TBitmap);
		procedure SetCaption(value: TCaption);
		procedure SetDisabledBitmap(value: TBitmap);
		procedure SetDown(value: Boolean);
		procedure SetDropDown(value: TPopupMenu);
		procedure SetEnabled(value: Boolean);
		procedure SetGroupIndex(value: Integer);
		procedure SetLayout(value: TExplorerButtonLayout);
		procedure SetNoFocusBitmap(value: TBitmap);
		procedure SetOptions(value: TExplorerButtonOptions);
		procedure SetShadingType(value: TExplorerButtonShadingType);
		procedure SetShowDownPattern(value: Boolean);
		procedure SetUnselectedFontColor(value: TColor);
		procedure SetWordWrap(value: Boolean);
		procedure WMRButtonUp(var msg: TWMRButtonUp); message WM_RBUTTONUP;
		procedure WMLButtonDown(var msg: TWMLButtonDown); message WM_LBUTTONDOWN;
		procedure WMLButtonUp(var msg: TWMLButtonUp); message WM_LBUTTONUP;
		procedure CMMouseEnter(var msg: TMessage); message CM_MOUSEENTER;
		procedure CMMouseLeave(var msg: TMessage); message CM_MOUSELEAVE;
	public
		constructor Create(aOwner: TComponent); override;
		destructor Destroy; override;
		procedure Click; override;
	published
		(* new properties *)
		property About: TAboutExpButtonProperty read FAbout write FAbout;
		property Alignment: TAlignment read FAlignment write SetAlignment stored True default taCenter;
		property AllowAllUp: Boolean read FAllowAllUp write SetAllowAllUp stored True default False;
		property BevelStyle: TBevelStyle read FBevelStyle write SetBevelStyle stored True default bsRaised;
		property NoFocusBitmap: TBitmap read FNoFocusBitmap write SetNoFocusBitmap stored True;
		property Bitmap: TBitmap read FBitmap write SetBitmap stored True;
		property Caption: TCaption read FCaption write SetCaption stored True;
		property GroupIndex: Integer read FGroupIndex write SetGroupIndex stored True default 0;
		property DisabledBitmap: TBitmap read FDisabledBitmap write SetDisabledBitmap stored True;
		property Down: Boolean read FDown write SetDown stored True default False;
		property DropDownMenu: TPopupMenu read FDropDown write SetDropDown stored True;
		property Enabled: Boolean read FEnabled write SetEnabled default True;
		property Layout: TExplorerButtonLayout read FLayout write SetLayout stored True default blBitmapTop;
		property Options: TExplorerButtonOptions read FOptions write SetOptions stored True;
		property ShadingType: TExplorerButtonShadingType read FShadingType write SetShadingType stored True default stMedium;
		property ShowDownPattern: Boolean read FShowDownPattern write SetShowDownPattern stored True default True;
		property UnselectedFontColor: TColor read FUnselectedFontColor write SetUnselectedFontColor stored True
			default clWindowText;
		property WordWrap: Boolean read FWordWrap write SetWordWrap stored True default False;
                property OnEnter: TNotifyEvent read FOnEnter write FOnEnter;
                property OnExit: TNotifyEvent read FOnExit write FOnExit;
		(* republished properties *)
		property Font;
		property OnClick;
		property ParentFont;
		property ParentShowHint;
		property ShowHint;
		property Visible;
		property OnDragDrop;
		property OnDragOver;
		property PopupMenu;
		property OnMouseMove;
 end;


	TAboutOffButtonProperty = class(TPropertyEditor)
	public
		procedure Edit; override;
		function GetAttributes: TPropertyAttributes; override;
		function GetValue: string; override;
	end;

	TOfficeButton = class(TCustomControl)
	private
		FAbout: TAboutOffButtonProperty;
		FAlignment: TAlignment;
		FAllowAllUp: Boolean;
		FBevelStyle: TBevelStyle;
		FBitmap, FNoFocusBitmap, FDisabledBitmap, IBitmap: TBitmap;
		FCaption: TCaption;
		FDown: Boolean;
		FDropDown: TPopupMenu;
		FEnabled: Boolean;
		FGroupIndex: Integer;
		FLayout: TExplorerButtonLayout;
		FOptions: TExplorerButtonOptions;
		FShadingType: TExplorerButtonShadingType;
		FShowDownPattern: Boolean;
		FUnselectedFontColor: TColor;
		FWordWrap: Boolean;
                FOnEnter, FOnExit: TNotifyEvent;
		(* state flags *)
		Pushed, MouseIn: Boolean;
	protected
		procedure CMDialogChar(var Message: TCMDialogChar); message CM_DIALOGCHAR;
		procedure ComputeExtent(TempCaption: PChar; var TempRect: TRect; theCanvas: TCanvas);
		procedure CreateGrayscaleBitmap(outputbmp, bmp: TBitmap);
		procedure DefineProperties(Filer: TFiler); override;
		procedure ReadIBitmap(Stream: TStream);
		procedure WriteIBitmap(Stream: TStream);
		procedure DrawTheText(theCanvas: TCanvas; TempRect: TRect; TempCaption: PChar);
		procedure DrawDisabledText(theCanvas: TCanvas; TempRect: TRect; TempCaption: PChar);
		procedure DrawPopupMark(theCanvas: TCanvas; x, y: Integer);
		procedure DrawOutline(theCanvas: TCanvas; pushed: Boolean);
		procedure GetLost;
		procedure Loaded; override;
		procedure Notification(AComponent: TComponent; Operation: TOperation); override ;
		procedure BitmapChange(Sender: TObject);
		procedure NoFocusBitmapChange(Sender: TObject);
		procedure DisabledBitmapChange(Sender: TObject);
		procedure Paint; override;
		procedure SetAlignment(value: TAlignment);
		procedure SetAllowAllUp(value: Boolean);
		procedure SetBevelStyle(value: TBevelStyle);
		procedure SetBitmap(value: TBitmap);
		procedure SetCaption(value: TCaption);
		procedure SetDisabledBitmap(value: TBitmap);
		procedure SetDown(value: Boolean);
		procedure SetDropDown(value: TPopupMenu);
		procedure SetEnabled(value: Boolean);
		procedure SetGroupIndex(value: Integer);
		procedure SetLayout(value: TExplorerButtonLayout);
		procedure SetNoFocusBitmap(value: TBitmap);
		procedure SetOptions(value: TExplorerButtonOptions);
		procedure SetShadingType(value: TExplorerButtonShadingType);
		procedure SetShowDownPattern(value: Boolean);
		procedure SetUnselectedFontColor(value: TColor);
		procedure SetWordWrap(value: Boolean);
		procedure WMRButtonUp(var msg: TWMRButtonUp); message WM_RBUTTONUP;
		procedure WMLButtonDown(var msg: TWMLButtonDown); message WM_LBUTTONDOWN;
		procedure WMLButtonUp(var msg: TWMLButtonUp); message WM_LBUTTONUP;
		procedure CMMouseEnter(var msg: TMessage); message CM_MOUSEENTER;
		procedure CMMouseLeave(var msg: TMessage); message CM_MOUSELEAVE;
	public
		constructor Create(aOwner: TComponent); override;
		destructor Destroy; override;
		procedure Click; override;
	published
		(* new properties *)
		property About: TAboutOffButtonProperty read FAbout write FAbout;
		property Alignment: TAlignment read FAlignment write SetAlignment stored True default taCenter;
		property AllowAllUp: Boolean read FAllowAllUp write SetAllowAllUp stored True default False;
		property BevelStyle: TBevelStyle read FBevelStyle write SetBevelStyle stored True default bsRaised;
		property NoFocusBitmap: TBitmap read FNoFocusBitmap write SetNoFocusBitmap stored True;
		property Bitmap: TBitmap read FBitmap write SetBitmap stored True;
		property Caption: TCaption read FCaption write SetCaption stored True;
		property GroupIndex: Integer read FGroupIndex write SetGroupIndex stored True default 0;
		property DisabledBitmap: TBitmap read FDisabledBitmap write SetDisabledBitmap stored True;
		property Down: Boolean read FDown write SetDown stored True default False;
		property DropDownMenu: TPopupMenu read FDropDown write SetDropDown stored True;
		property Enabled: Boolean read FEnabled write SetEnabled default True;
		property Layout: TExplorerButtonLayout read FLayout write SetLayout stored True default blBitmapTop;
		property Options: TExplorerButtonOptions read FOptions write SetOptions stored True;
		property ShadingType: TExplorerButtonShadingType read FShadingType write SetShadingType stored True default stMedium;
		property ShowDownPattern: Boolean read FShowDownPattern write SetShowDownPattern stored True default True;
		property UnselectedFontColor: TColor read FUnselectedFontColor write SetUnselectedFontColor stored True
			default clWindowText;
		property WordWrap: Boolean read FWordWrap write SetWordWrap stored True default False;
                property OnEnter: TNotifyEvent read FOnEnter write FOnEnter;
                property OnExit: TNotifyEvent read FOnExit write FOnExit;
		(* republished properties *)
		property Font;
		property OnClick;
		property ParentFont;
		property ParentShowHint;
		property ShowHint;
		property Visible;
		property OnDragDrop;
		property OnDragOver;
		property PopupMenu;
     property HelpContext;
		property OnMouseMove;
 end;


procedure Register;

var
   (* Version Flag *)
   bool_Version95 : Boolean;
   pattern: TBitmap;

implementation

(*
 * About property
 *)

procedure TAboutExpButtonProperty.Edit;
begin
	Application.MessageBox('This component is freeware. (c)1996-1997 Fabrice Deville',
   						'TExplorerButton component version 2.6', MB_OK+ MB_ICONINFORMATION);
end;

function TAboutExpButtonProperty.GetAttributes: TPropertyAttributes;
begin
	Result := [paMultiSelect, paDialog, paReadOnly];
end;

function TAboutExpButtonProperty.GetValue: string;
begin
	Result := '(about)';
end;

(* Get the System Version - thanks to Valentino Magri *)
procedure GetSystemVersion;
{$IFDEF WIN32}
var
   VersionInfo : TOsVersionInfo;
   Value       : Boolean;
begin
	bool_Version95 := True;
	try
		VersionInfo.dwOSVersionInfoSize := sizeof (VersionInfo);
		Value := GetVersionEx (VersionInfo);
		if (VersionInfo.dwPlatformId = VER_PLATFORM_WIN32_WINDOWS) or
        ((VersionInfo.dwPlatformId = VER_PLATFORM_WIN32_NT)
        and (VersionInfo.dwMajorVersion >= 4)) then
			bool_Version95 := True
		else
			bool_Version95 := False;
		except
			bool_Version95 := True;
	end;
{$ELSE}
begin
	bool_Version95 := False;
{$ENDIF}
end;

(*
 * Drawing of a disabled bitmap (Win95 style)
 *)

procedure DrawDisabledBitmap(Canvas: TCanvas; x, y: Integer; bmp: TBitmap);
var MonoBmp: TBitmap;
begin
	MonoBmp := TBitmap.Create;
	try
		MonoBmp.Assign(bmp);
		MonoBmp.Canvas.Brush.Color := clBlack;
		MonoBmp.Monochrome := True;
		Canvas.Brush.Color := clBtnHighlight;
		SetTextColor(Canvas.Handle, clBlack);
		SetBkColor(Canvas.Handle, clWhite);
		BitBlt(Canvas.Handle, x+1, y+1, bmp.Width, bmp.Height,
			MonoBmp.Canvas.Handle, 0, 0, $00E20746);
		Canvas.Brush.Color := clBtnShadow;
		SetTextColor(Canvas.Handle, clBlack);
		SetBkColor(Canvas.Handle, clWhite);
		BitBlt(Canvas.Handle, x, y, bmp.Width, bmp.Height,
			MonoBmp.Canvas.Handle, 0, 0, $00E20746);
	finally
		MonoBmp.Free;
	end
end;

procedure CreatePattern;
var
  X, Y: Integer;
begin
	Pattern := TBitmap.Create;
	Pattern.Width := 8;
	Pattern.Height := 8;
	with Pattern.Canvas do
	begin
		Brush.Style := bsSolid;
		Brush.Color := clBtnFace;
		FillRect(Rect(0, 0, Pattern.Width, Pattern.Height));
		for Y := 0 to 7 do
			for X := 0 to 7 do
			begin
				if (X mod 2) = (Y mod 2) then
					Pixels[X, Y] := clBtnHighlight;
			end;
	end;
end;

(*
 * Drawing of a bitmap on a canvas, with transparency
 * (adapted from code found in the Microsoft Knowledge Base)
 *)

procedure DrawTransparentBitmap(canvas: TCanvas; bmp: TBitmap; xStart,
                           yStart: Integer; cTransparentColor: LongInt);
var bm: WinTypes.TBitmap;
	cColor: TColorRef;
	bmAndBack, bmAndObject, bmAndMem, bmSave, oldBmp: HBITMAP;
	bmBackOld, bmObjectOld, bmMemOld, bmSaveOld, hBmp: HBITMAP;
	hdcMem, hdcBack, hdcObject, hdcTemp, hdcSave, dc: HDC;
	ptSize: WinTypes.TPoint;
	temp_bitmap: TBitmap;
begin
	temp_bitmap := TBitmap.Create;
	temp_bitmap.Assign(bmp);
	try
		dc := canvas.Handle;
		hBmp := temp_bitmap.Handle;
		hdcTemp := CreateCompatibleDC(dc);
		oldBmp := SelectObject(hdcTemp, hBmp);

		GetObject(hBmp, SizeOf(bm), @bm);
		ptSize.x := bm.bmWidth;
		ptSize.y := bm.bmHeight;

		hdcBack   := CreateCompatibleDC(dc);
		hdcObject := CreateCompatibleDC(dc);
		hdcMem    := CreateCompatibleDC(dc);
		hdcSave   := CreateCompatibleDC(dc);

		bmAndBack   := CreateBitmap(ptSize.x, ptSize.y, 1, 1, nil);

		bmAndObject := CreateBitmap(ptSize.x, ptSize.y, 1, 1, nil);

		bmAndMem    := CreateCompatibleBitmap(dc, ptSize.x, ptSize.y);
		bmSave      := CreateCompatibleBitmap(dc, ptSize.x, ptSize.y);

		bmBackOld   := SelectObject(hdcBack, bmAndBack);
		bmObjectOld := SelectObject(hdcObject, bmAndObject);
		bmMemOld    := SelectObject(hdcMem, bmAndMem);
		bmSaveOld   := SelectObject(hdcSave, bmSave);

		SetMapMode(hdcTemp, GetMapMode(dc));

		BitBlt(hdcSave, 0, 0, ptSize.x, ptSize.y, hdcTemp, 0, 0, SRCCOPY);

		cColor := SetBkColor(hdcTemp, cTransparentColor);

		BitBlt(hdcObject, 0, 0, ptSize.x, ptSize.y, hdcTemp, 0, 0, SRCCOPY);

		SetBkColor(hdcTemp, cColor);

		BitBlt(hdcBack, 0, 0, ptSize.x, ptSize.y, hdcObject, 0, 0, NOTSRCCOPY);
		BitBlt(hdcMem, 0, 0, ptSize.x, ptSize.y, dc, xStart, yStart, SRCCOPY);
		BitBlt(hdcMem, 0, 0, ptSize.x, ptSize.y, hdcObject, 0, 0, SRCAND);
		BitBlt(hdcTemp, 0, 0, ptSize.x, ptSize.y, hdcBack, 0, 0, SRCAND);
		BitBlt(hdcMem, 0, 0, ptSize.x, ptSize.y, hdcTemp, 0, 0, SRCPAINT);
		BitBlt(dc, xStart, yStart, ptSize.x, ptSize.y, hdcMem, 0, 0, SRCCOPY);
		BitBlt(hdcTemp, 0, 0, ptSize.x, ptSize.y, hdcSave, 0, 0, SRCCOPY);

		DeleteObject(SelectObject(hdcBack, bmBackOld));
		DeleteObject(SelectObject(hdcObject, bmObjectOld));
		DeleteObject(SelectObject(hdcMem, bmMemOld));
		DeleteObject(SelectObject(hdcSave, bmSaveOld));

		SelectObject(hdcTemp, oldBmp);

		DeleteDC(hdcMem);
		DeleteDC(hdcBack);
		DeleteDC(hdcObject);
		DeleteDC(hdcSave);
		DeleteDC(hdcTemp);
		finally
			temp_bitmap.Free;
	end;
end;

{$B-} (* Quick evaluation *)
(* Won't work properly in the case of 'text&' caption, but the one
   who defines such a caption has a really sick mind :)*)
function RemoveAmpersand(input: String): String;
var i: Integer;
begin
	Result := input;
	i := 1;
	while i < Length(Result) do
	begin
		if (Result[i] = '&') then
		begin
			if (Result[i+1] = '&') then
				Delete(Result, i + 1, 1)
			else
				Delete(Result, i, 1);
		end
		else
			Inc(i);
	end;
end;

(*
 * TExplorerButton implementation
 *)

constructor TExplorerButton.Create(aOwner: TComponent);
begin
	inherited Create(aOwner);
	GetSystemVersion;

	FBitmap := TBitmap.Create;
	FBitmap.OnChange := BitmapChange;
	FNoFocusBitmap := TBitmap.Create;
	FNoFocusBitmap.OnChange := NoFocusBitmapChange;
	FDisabledBitmap := TBitmap.Create;
	FDisabledBitmap.OnChange := DisabledBitmapChange;
	IBitmap := TBitmap.Create;

	ControlStyle := [csClickEvents, csCaptureMouse, csSetCaption];
	Pushed := False;
	MouseIn := False;
	Alignment := taCenter;
	Width := 50;
	Height := 40;
	if (csDesigning in ComponentState) and not (csLoading in TControl(Owner).ComponentState) then
		Caption := 'ExplorerButton';
	FDropDown := nil;
	FOptions := [boMonoDisplay, boPopupMark, boShowBevel];
	FLayout := blBitmapTop;
	FAllowAllUp := False;
	FDown := False;
	FEnabled := True;
	FGroupIndex := 0;
	FBevelStyle := bsRaised;
	FShadingType := stMedium;
	FShowDownPattern := True;
	FUnselectedFontColor := clWindowText;
end;

destructor TExplorerButton.Destroy;
begin
	FBitmap.Free;
	FNoFocusBitmap.Free;
	FDisabledBitmap.Free;
	IBitmap.Free;
	if pattern <> nil then
	begin
		pattern.Free;
		pattern := nil;
	end;
	inherited Destroy;
end;

procedure TExplorerButton.BitmapChange(Sender: TObject);
begin
	if not FBitmap.Empty and FNoFocusBitmap.Empty and (csDesigning in ComponentState)
  	and not (csLoading in ComponentState) then
		CreateGrayscaleBitmap(IBitmap, FBitmap);

  if not FBitmap.Empty then
  	FBitmap.Dormant;
	Repaint;
end;

procedure TExplorerButton.NoFocusBitmapChange(Sender: TObject);
begin
	if not FBitmap.Empty and FNoFocusBitmap.Empty and (csDesigning in ComponentState) then
  begin
		CreateGrayscaleBitmap(IBitmap, FBitmap);
  end;

  if not FNoFocusBitmap.Empty then
  	FNoFocusBitmap.Dormant;
	Repaint;
end;

procedure TExplorerButton.DefineProperties(Filer: TFiler);
begin
	Filer.DefineBinaryProperty('IBitmap', ReadIBitmap, WriteIBitmap, True);
end;

procedure TExplorerButton.ReadIBitmap(Stream: TStream);
begin
	IBitmap.LoadFromStream(Stream);
end;

procedure TExplorerButton.WriteIBitmap(Stream: TStream);
begin
	if not IBitmap.Empty then
		IBitmap.SaveToStream(Stream)
end;

procedure TExplorerButton.DisabledBitmapChange(Sender: TObject);
begin
  if not FDisabledBitmap.Empty then
  	FDisabledBitmap.Dormant;
	if not FEnabled then
		Repaint;
end;

procedure TExplorerButton.CMDialogChar(var Message: TCMDialogChar);
var lpPoint : TPoint;
begin
	with Message do
		if IsAccel(CharCode, Caption)and Enabled and Visible then
	begin
		MouseIn := True;
		Pushed := False;
		Repaint;
		Application.ProcessMessages;
		WMLButtonDown( TWMLBUTTONDOWN(Message));
		Application.ProcessMessages;
		WMLButtonUp( TWMLBUTTONUP(Message));
		Application.ProcessMessages;
		GetCursorPos(lpPoint);
		lpPoint := GetParentForm(self).ScreenToClient(lpPoint);
		if not  ((lpPoint.y > top) and (lpPoint.y < top + height)
			and (lpPoint.x > left) and (lpPoint.x < left + width)) then
		begin
			MouseIn := False;
			Repaint;
		end;
		Result := 1;
	end;
end;

procedure TExplorerButton.ComputeExtent(TempCaption: PChar; var TempRect: TRect);
var Flags: Integer;
begin
	if Alignment = taLeftJustify then
		Flags := DT_LEFT
	else if Alignment = taCenter then
		Flags := DT_CENTER
	else Flags := DT_RIGHT;

	if WordWrap then
	begin
		Flags := Flags or DT_WORDBREAK;
		(* Sometimes DrawText looses the last word, except when there's a space character. Uh ? *)
		StrCat(TempCaption, ' ');
	end;

	DrawText(Canvas.handle, TempCaption, StrLen(TempCaption), TempRect, DT_CALCRECT or Flags);
end;

(*
 * These thresholds are used for the grayscaling and were experimentaly
 * determined
 *)
const THRESHOLD1_LIGHT = 205;
      THRESHOLD2_LIGHT = 127;
      THRESHOLD3_LIGHT = 68;
      THRESHOLD1_MEDIUM = 553;
      THRESHOLD2_MEDIUM = 231;
      THRESHOLD3_MEDIUM = 57;
      THRESHOLD1_DARK = 335;
      THRESHOLD2_DARK = 274;
      THRESHOLD3_DARK = 175;
(*
 * Creation of the grayscale bitmap (pretty dumb routine, isn't it?)
 *)
procedure TExplorerButton.CreateGrayscaleBitmap(outputbmp, bmp: TBitmap);
var x, y: Integer;
    col, TransparentColor: LongInt;
    r, g, b, sum, threshold1, threshold2, threshold3: Integer;
begin
	outputbmp.Assign(bmp);
	TransparentColor := ColorToRGB(bmp.Canvas.Pixels[0,0]);
	if FShadingType = stLight then
	begin
		threshold1 := THRESHOLD1_LIGHT;
		threshold2 := THRESHOLD2_LIGHT;
		threshold3 := THRESHOLD3_LIGHT;
	end
	else
	if FShadingType = stMedium then
	begin
		threshold1 := THRESHOLD1_MEDIUM;
		threshold2 := THRESHOLD2_MEDIUM;
		threshold3 := THRESHOLD3_MEDIUM;
	end
	else
	if FShadingType = stDark then
	begin
		threshold1 := THRESHOLD1_DARK;
		threshold2 := THRESHOLD2_DARK;
		threshold3 := THRESHOLD3_DARK;
	end;
	for x := 0 to bmp.Width do
		for y := 0 to bmp.Height do
		begin
			col := ColorToRGB(bmp.Canvas.Pixels[x, y]);
			if col <> TransparentColor then
			begin
				r := col shr 16;
				g := (col shr 8) and $00FF;
				b := col and $0000FF;
				sum := r + g + b;
				if sum > THRESHOLD1 then
					outputbmp.Canvas.Pixels[x, y] := clWhite
				else if sum > THRESHOLD2 then
					outputbmp.Canvas.Pixels[x, y] := clBtnHighlight
				else if sum > THRESHOLD3 then
					outputbmp.Canvas.Pixels[x, y] := clBtnShadow
				else
					outputbmp.Canvas.Pixels[x, y] := clBlack;
			end;
		end;
 		if not bmp.Empty then
     	bmp.Dormant;
 		if not outputbmp.Empty then
     	outputbmp.Dormant;
end;

procedure TExplorerButton.DrawTheText(theCanvas: TCanvas; TempRect: TRect; TempCaption: PChar);
var Flags: Integer;
begin
	if Alignment = taLeftJustify then
		Flags := DT_LEFT
	else if Alignment = taCenter then
		Flags := DT_CENTER
	else Flags := DT_RIGHT;

	if WordWrap then
		Flags := Flags or DT_WORDBREAK;

	if bool_Version95 then
	begin
		{$IFDEF WIN32}
		DrawTextEx(theCanvas.handle, TempCaption, StrLen(TempCaption), TempRect,
     			 DT_END_ELLIPSIS or Flags, nil);
		{$ELSE}
		DrawText(theCanvas.handle, TempCaption, StrLen(TempCaption), TempRect, Flags);
		{$ENDIF}
	end
	else
		(* NT 3.51 users *)
		DrawText(theCanvas.handle, TempCaption, StrLen(TempCaption), TempRect, Flags);
end;

(*
 * Drawing of a disabled text (Win95 style)
 *)
procedure TExplorerButton.DrawDisabledText(theCanvas: TCanvas; TempRect: TRect; TempCaption: PChar);
begin
	theCanvas.Brush.Style := bsClear;
	theCanvas.Font.Color := clBtnHighlight;
	with TempRect do
	begin
		left := left + 1;
		top  := top + 1;
		right:= right + 1;
		bottom:= bottom + 1;
	end;
	DrawTheText(theCanvas, TempRect, TempCaption);
	theCanvas.Font.Color := clBtnShadow;
	with TempRect do
	begin
		left := left - 1;
		top  := top - 1;
		right:= right - 1;
		bottom:= bottom - 1;
	end;
	DrawTheText(theCanvas, TempRect, TempCaption);
end;

procedure TExplorerButton.DrawOutline(theCanvas: TCanvas; pushed: Boolean);
begin
	if boShowBevel in FOptions then
  begin
	if pushed then
	begin
		if (BevelStyle = bsLowered) then
			theCanvas.Pen.Color := clBtnHighlight
		else
			theCanvas.Pen.Color := clBtnShadow;
	end
	else
	begin
		if (BevelStyle = bsLowered) then
			theCanvas.Pen.Color := clBtnShadow
		else
			theCanvas.Pen.Color := clBtnHighlight;
	end;
	theCanvas.MoveTo(0, Height);
	theCanvas.LineTo(0, 0);
	theCanvas.LineTo(Width-1, 0);
	if pushed then
	begin
		if (BevelStyle = bsLowered) then
			theCanvas.Pen.Color := clBtnShadow
		else
			theCanvas.Pen.Color := clBtnHighlight;
	end
	else
	begin
		if (BevelStyle = bsLowered) then
			theCanvas.Pen.Color := clBtnHighlight
		else
			theCanvas.Pen.Color := clBtnShadow;
	end;
	theCanvas.LineTo(Width-1, Height-1);
	theCanvas.LineTo(0, Height-1);
  end;
end;

procedure TExplorerButton.DrawPopupMark(theCanvas: TCanvas; x, y: Integer);
var theColor: TColor;
begin
	(* loops were unrolled *)
	theColor := theCanvas.Font.Color;
	theCanvas.Pixels[x    , y - 1] := theColor;
	theCanvas.Pixels[x + 1, y - 1] := theColor;
	theCanvas.Pixels[x + 2, y - 1] := theColor;
	theCanvas.Pixels[x + 3, y - 1] := theColor;
	theCanvas.Pixels[x + 4, y - 1] := theColor;
	theCanvas.Pixels[x + 5, y - 1] := theColor;
	theCanvas.Pixels[x + 6, y - 1] := theColor;

	theCanvas.Pixels[x + 1, y    ] := theColor;
	theCanvas.Pixels[x + 2, y    ] := theColor;
	theCanvas.Pixels[x + 3, y    ] := theColor;
	theCanvas.Pixels[x + 4, y    ] := theColor;
	theCanvas.Pixels[x + 5, y    ] := theColor;

	theCanvas.Pixels[x + 2, y + 1] := theColor;
	theCanvas.Pixels[x + 3, y + 1] := theColor;
	theCanvas.Pixels[x + 4, y + 1] := theColor;

	theCanvas.Pixels[x + 3, y + 2] := theColor;
end;

procedure TExplorerButton.GetLost;
begin
	if FGroupIndex = 0 then
		FDown := False;
	Pushed := False;
	MouseIn := False;
	Repaint;
end;

(*
 * After the loading of the properties, the grayscale version can be
 * safely created
 *)
procedure TExplorerButton.Loaded;
begin
	inherited Loaded;
	if (not FBitmap.Empty) and FNoFocusBitmap.Empty and (csDesigning in ComponentState) then
		CreateGrayscaleBitmap(IBitmap, FBitmap);
end;

procedure TExplorerButton.Notification(AComponent: TComponent; Operation:
TOperation);
begin
	if (Operation = opRemove) and (AComponent = FDropDown) then
		FDropDown := nil ;
end ;

const PADDING = 2; (* Space between the bitmap and the caption *)

(*
 * Painting of the button
 *)
procedure TExplorerButton.Paint;
var xt, yt, wt, _wt, ht, xb, yb, wb, hb, thePadding, theLeft, theTop: Integer;
	TempCaption : Array[0..256] of char;
	TempRect    : TRect;
	bitmp: TBitmap;
	ctrl: TControl;
begin
	if visible or (csDesigning in ComponentState) then
	begin
		(* Just to stop compiler warnings *)
		xt := 0;
		yb := 0;
		xb := 0;

		if FDown and FShowDownPattern then
		begin
			if pattern = nil then
				CreatePattern;
			Canvas.Brush.Bitmap := pattern;
			Canvas.FillRect(Rect(0, 0, Width, Height));
		end;

		Canvas.Brush.Style := bsClear;
		Canvas.Font := Self.Font;

		if not (boIconOnly in FOptions) and (Length(FCaption) > 0) then
		begin
			thePadding := PADDING;
			TempRect.Top := 0;
			TempRect.Left := 0;
			TempRect.Right := Width - 4;
			TempRect.Bottom := 0;
			StrPCopy(TempCaption, Caption);
			ComputeExtent(TempCaption, TempRect);
			wt := TempRect.Right;
			ht := TempRect.Bottom;
		end
		else
		begin
			thePadding := 0;
			wt := 0;
			ht := 0;
		end;

		if wt > Width - 4 then
			wt := Width - 4;

		wb := FBitmap.Width;
		(* Reserve place for the drawing of the popup mark *)
		if (boPopupMark in FOptions) and Assigned(FDropDown) then
			wb := wb + 10;
		hb := FBitmap.Height;

		if FBitmap.Empty then
		begin
			if Assigned(FDropDown) and (boPopupMark in FOptions) then
			begin
				_wt := wt + 10;
				if wt > Width - 14 then
					wt := Width - 14;
			end
			else
				_wt := wt;

			yt := (Height - ht) div 2;
			case Alignment of
				taLeftJustify:		xt := 3;
				taRightJustify:	xt := Width - _wt - 3;
				taCenter:			xt := (Width - _wt) div 2
			end;
		end
		else if (Layout = blBitmapTop) or (Layout = blBitmapBottom) then
		begin
			if Layout = blBitmapTop then
			begin
				yb := (Height - (ht + hb + thePadding)) div 2;
				yt := yb + hb + thePadding
			end
			else
			begin
				yt := (Height - (ht + hb + thePadding)) div 2;
				yb := yt + ht + thePadding
			end;
			case Alignment of
				taLeftJustify:
					begin
						xt := 3;
						xb := 3
					end;
				taRightJustify:
					begin
						xt := Width - wt - 3;
						xb := Width - wb - 3
					end;
				taCenter:
					begin
						xb := (Width - wb) div 2;
						xt := (Width - wt) div 2
					end;
			end;
		end
		else
		if Layout = blBitmapLeft then
		begin
			if wt + wb + thePadding > Width - 4 then
				wt := Width - 4 - thePadding - wb;
			yb := (Height - hb) div 2;
			yt := (Height - ht) div 2;

			case Alignment of
			taLeftJustify:
				begin
					xb := 3;
					xt := xb + wb + thePadding
				end;
			taRightJustify:
				begin
					xt := Width - wt - 3;
					xb := xt - wb - thePadding
				end;
			taCenter:
				begin
					xb := (Width - (wb + wt + thePadding)) div 2;
					xt := xb + wb + thePadding
				end;
			end;
		end
		else (* blBitmapRight *)
		begin
			if wt + wb + thePadding > Width - 4 then
				wt := Width - 4 - thePadding - wb;
			yb := (Height - hb) div 2;
			yt := (Height - ht) div 2;
			case Alignment of
			taLeftJustify:
				begin
					xt := 3;
					xb := xt + wt + thePadding
				end;
			taRightJustify:
				begin
					xb := Width - wb - 3;
					xt := xb - wt - thePadding
				end;
			taCenter:
				begin
					xt := (Width - (wb + wt + thePadding)) div 2;
					xb := xt + wt + thePadding
				end;
			end;
		end;

		if csDesigning in ComponentState then
		begin
			Canvas.Pen.Color := clBlack;
			Canvas.Pen.Style := psSolid;
			Canvas.Brush.Style := bsClear;
			Canvas.Rectangle(0, 0, Width, Height);
		end;

		if WordWrap and (xt + wt > width - 5) then
			wt := width - xt - 5;

		with TempRect do
		begin
			left := xt;
			top  := yt;
			right:= xt + wt;
			bottom:= yt + ht;
		end;

		if Enabled then
		begin
			if not (Pushed and MouseIn) and not Down then
			begin
				(* Unpushed state - Mouse in or out *)
				if MouseIn then
					DrawOutline(Canvas, False)
				else
					Canvas.Font.Color := UnselectedFontColor;
				if BevelStyle = bsLowered then
				begin
					Inc(TempRect.Left);
					Inc(TempRect.Top);
					Inc(TempRect.Right);
					Inc(TempRect.Bottom);
				end;
				if not (boIconOnly in FOptions) and (Length(Caption) > 0) then
					DrawTheText(Canvas, TempRect, TempCaption);
				if not FBitmap.Empty then
				begin
					(* Draw the normal or shaded bitmap.
					 * Transparency color is at (0,0)
					 *)
					if MouseIn or not (boMonoDisplay in FOptions) then
						DrawTransparentBitmap(Canvas, FBitmap, xb, yb, ColorToRGB(FBitmap.Canvas.Pixels[0, 0]))
					else if FNoFocusBitmap.Empty then
						DrawTransparentBitmap(Canvas, IBitmap, xb, yb, ColorToRGB(IBitmap.Canvas.Pixels[0, 0]))
					else
						DrawTransparentBitmap(Canvas, FNoFocusBitmap, xb, yb, ColorToRGB(FNoFocusBitmap.Canvas.Pixels[0, 0]));
					if (boPopupMark in FOptions) and Assigned(FDropDown) then
						DrawPopupMark(Canvas, xb + FBitmap.Width + 3, yb + (hb div 2));
				end
				else if (boPopupMark in FOptions) and Assigned(FDropDown) then
           				DrawPopupMark(Canvas, xt + wt + 3, yt + (ht div 2));
			end
			else
			begin
         	(* Pushed state *)
				DrawOutline(Canvas, True);
				if BevelStyle = bsRaised then
				begin
					Inc(TempRect.Left);
					Inc(TempRect.Top);
					Inc(TempRect.Right);
					Inc(TempRect.Bottom);
				end;
				if not (boIconOnly in FOptions) and (Length(Caption) > 0) then
					DrawTheText(Canvas, TempRect, TempCaption);
				if not FBitmap.Empty then
				begin
					DrawTransparentBitmap(Canvas, FBitmap, xb+1, yb+1, ColorToRGB(FBitmap.Canvas.Pixels[0, 0]));
					if (boPopupMark in FOptions) and Assigned(FDropDown) then
						DrawPopupMark(Canvas, xb + FBitmap.Width + 4, yb + (hb div 2) + 1);
				end
				else if (boPopupMark in FOptions) and Assigned(FDropDown) then
					DrawPopupMark(Canvas, xt + wt + 4, yt + (ht div 2) + 1);
			end
		end
		else
		begin
			(* Disabled state *)
			if Down then
				DrawOutline(Canvas, True);
			if not FDisabledBitmap.Empty then
				DrawTransparentBitmap(Canvas, FDisabledBitmap, xb, yb, ColorToRGB(FDisabledBitmap.Canvas.Pixels[0, 0]))
			else
				DrawDisabledBitmap(Canvas, xb, yb, FBitmap);
			if not (boIconOnly in FOptions) and (Length(Caption) > 0) then
			DrawDisabledText(Canvas, TempRect, TempCaption);
		end;
     if not FBitmap.Empty then
     	FBitmap.Dormant;
     if not FDisabledBitmap.Empty then
     	FDisabledBitmap.Dormant;
     if not FNoFocusBitmap.Empty then
        FNoFocusBitmap.Dormant;
     if not IBitmap.Empty then
        IBitmap.Dormant;
	end
end;

procedure TExplorerButton.SetAlignment(value: TAlignment);
begin
	FAlignment := value;
	Repaint;
end;

procedure TExplorerButton.SetAllowAllUp(value: Boolean);
var i: Integer;
	otherbutton: TExplorerButton;
begin
	FAllowAllUp := value;
	if FGroupIndex <> 0 then
	begin
		for i := 0 to Parent.ControlCount - 1 do
		begin
			if Parent.Controls[i] is TExplorerButton then
			begin
				otherbutton := (Parent.Controls[i] as TExplorerButton);
				if (otherbutton <> Self) and (otherbutton.GroupIndex = GroupIndex) then
					otherbutton.FAllowAllUp := value;
			end
		end
	end
end;

procedure TExplorerButton.SetBevelStyle(value: TBevelStyle);
begin
	FBevelStyle := value;
	Repaint;
end;

procedure TExplorerButton.SetBitmap(value: TBitmap);
begin
	FBitmap.Assign(value);
  if not FBitmap.Empty then
  	FBitmap.Dormant;
end;

procedure TExplorerButton.SetCaption(value: TCaption);
begin
	FCaption := value;
	if boIconOnly in FOptions then
		Hint := RemoveAmpersand(FCaption)
	else
		Repaint;
end;

procedure TExplorerButton.SetDisabledBitmap(value: TBitmap);
begin
	FDisabledBitmap.Assign(value);
  if not FDisabledBitmap.Empty then
  	FDisabledBitmap.Dormant;
end;

procedure TExplorerButton.SetDown(value: Boolean);
var i: Integer;
	otherbutton: TExplorerButton;
begin
	if FDown = value then
   	Exit;
	if FGroupIndex <> 0 then
	begin
		FDown := value;
		Repaint;
	   	if FDown = True then
		begin
			for i := 0 to Parent.ControlCount - 1 do
   			begin
				if Parent.Controls[i] is TExplorerButton then
				begin
					otherbutton := (Parent.Controls[i] as TExplorerButton);
					if (otherbutton <> Self) and (otherbutton.GroupIndex = GroupIndex)
						and (otherbutton.FDown = True) then
					begin
						otherbutton.FDown := False;
						otherbutton.Repaint;
					end
				end
			end
		end
	end
end;

procedure TExplorerButton.SetDropDown(value: TPopupMenu);
begin
	FDropDown := value;
	if boPopupMark in FOptions then
		Repaint;
end;

procedure TExplorerButton.SetEnabled(value: Boolean);
begin
	if FEnabled = value then
   	Exit;
	FEnabled := value;
	GetLost;
end;

procedure TExplorerButton.SetGroupIndex(value: Integer);
begin
	if FGroupIndex = value then
		Exit;
	FGroupIndex := value;
	if (FGroupIndex = 0) and FDown then
	begin
		FDown := False;
		Repaint;
	end;
end;

procedure TExplorerButton.SetLayout(value: TExplorerButtonLayout);
begin
	if FLayout = value then
   	Exit;
	FLayout := value;
	Repaint;
end;

procedure TExplorerButton.SetNoFocusBitmap(value: TBitmap);
begin
	FNoFocusBitmap.Assign(value);
  if not FNoFocusBitmap.Empty then
  	FNoFocusBitmap.Dormant;
end;

procedure TExplorerButton.SetOptions(value: TExplorerButtonOptions);
begin
	FOptions := value;
	if (boIconOnly in FOptions) and (Hint = '') then
	begin
		Hint := RemoveAmpersand(FCaption);
		ShowHint := True;
	end
	else if (not(boIconOnly in FOptions)) and (Hint = RemoveAmpersand(FCaption)) then
	begin
		Hint := '';
		ShowHint := False;
	end;
	Repaint;
end;

procedure TExplorerButton.SetShadingType(value: TExplorerButtonShadingType);
begin
	if value <> FShadingType then
	begin
		FShadingType := value;
		if not FBitmap.Empty and FNoFocusBitmap.Empty and (csDesigning in ComponentState)
  		and not (csLoading in ComponentState) then
		begin
			CreateGrayscaleBitmap(IBitmap, FBitmap);
			Repaint;
		end;
	end;
end;

procedure TExplorerButton.SetShowDownPattern(value: Boolean);
begin
	if value <> FShowDownPattern then
	begin
		FShowDownPattern := value;
		Repaint;
	end;
end;

procedure TExplorerButton.SetUnselectedFontColor(value: TColor);
begin
	if FUnselectedFontColor <> value then
	begin
		FUnselectedFontColor := value;
		Repaint;
	end;
end;

procedure TExplorerButton.SetWordWrap(value: Boolean);
begin
	if FWordWrap <> value then
	begin
		FWordWrap := value;
		Repaint;
	end;
end;

procedure TExplorerButton.WMLButtonDown(var msg: TWMLButtonDown);
var p: TPoint;
	theMsg: TMsg;
	lpPoint: TPoint;
begin
	if not GetParentForm(Self).Active then
		GetParentForm(Self).BringToFront;

	if Enabled and Visible then
	begin
		Pushed := True;
		MouseIn := True;
		Repaint;
		(* If the popup is defined, display it under the button *)
		if FDropDown <> nil then
		begin
			p := ClientToScreen(Point(0, Height));
			FDropDown.Popup(p.x, p.y);
			while PeekMessage(theMsg, HWND(0), WM_MOUSEFIRST, WM_MOUSELAST, PM_REMOVE) do
				;
			if GetCapture <> 0 then
				SendMessage(GetCapture, WM_CANCELMODE, 0, 0);
			GetCursorPos(lpPoint);
			lpPoint := Parent.ScreenToClient(lpPoint);
			if not  ((lpPoint.y > top) and (lpPoint.y < top + height)
				and (lpPoint.x > left) and (lpPoint.x < left + width)) then
				MouseIn := False;

			Pushed := False;
			Repaint;
		end
     else
     	SetCaptureControl(Self);
	end;
end;

procedure TExplorerButton.WMLButtonUp(var msg: TWMLButtonUp);
var i: Integer;
	otherbutton: TExplorerButton;
	lpPoint: TPoint;
begin
	if Enabled and Visible and Pushed then
	begin
		Pushed := False;
		if MouseIn and (GroupIndex <> 0) then
		begin
			(* Was Down -> allow up state if AllowAllUp is True *)
			if FDown then
			begin
				if FAllowAllUp then
					FDown := False
			end
			else
			(* Was Up *)
			begin
				(* Set 'up' all buttons having the same parent
					and GroupIndex *)
				for i := 0 to Parent.ControlCount - 1 do
				begin
					if Parent.Controls[i] is TExplorerButton then
					begin
						otherbutton := (Parent.Controls[i] as TExplorerButton);
						if (otherbutton <> Self) and (otherbutton.GroupIndex = GroupIndex)
							and (otherbutton.FDown = True) then
						begin
							otherbutton.Down := False;
							otherbutton.Repaint;
						end
					end
				end;
				FDown := True;
			end;
		end;
		SetCaptureControl(nil);
		if MouseIn then
		begin
			GetCursorPos(lpPoint);
			lpPoint := Parent.ScreenToClient(lpPoint);
			if not  ((lpPoint.y > top) and (lpPoint.y < top + height)
				and (lpPoint.x > left) and (lpPoint.x < left + width)) then
				MouseIn := False;
			Invalidate;
			(* If the popup is not defined, activate the click event *)
			if (FDropDown = nil) then
				Click;
		end
	end
end;

procedure TExplorerButton.CMMouseEnter(var msg: TMessage);
begin
	MouseIn := True;
	if not FDown and FEnabled then
		Repaint;
        if Assigned(FOnEnter) then
                FOnEnter(Self);
end;

procedure TExplorerButton.CMMouseLeave(var msg: TMessage);
begin
	MouseIn := False;
	if not FDown and FEnabled then
		Repaint;
        if Assigned(FOnExit) then
                FOnExit(Self);
end;

procedure TExplorerButton.WMRButtonUp(var msg: TWMRButtonUp);
var p: TPoint;
	theMsg: TMsg;
	lpPoint: TPoint;
begin
	inherited;
	if (PopupMenu <> nil) or (GetParentForm(Self).PopupMenu <> nil) then
	begin
		while PeekMessage(theMsg, HWND(0), WM_MOUSEFIRST, WM_MOUSELAST, PM_REMOVE) do
			;
		if GetCapture <> 0 then
			SendMessage(GetCapture, WM_CANCELMODE, 0, 0);
		GetCursorPos(lpPoint);
		lpPoint := Parent.ScreenToClient(lpPoint);
		if not  ((lpPoint.y > top) and (lpPoint.y < top + height)
			and (lpPoint.x > left) and (lpPoint.x < left + width)) then
			MouseIn := False;
		Pushed := False;
		Repaint;
	end;
end;

procedure TExplorerButton.Click;
begin
	inherited Click;
end;

(* OFFICE BUTTON: SAME AS ABOVE, BUT DERIVED FROM TCUSTOMCONTROL, NO FLICKERING GUARANTEED *)

procedure TAboutOffButtonProperty.Edit;
begin
	Application.MessageBox('This component is freeware. (c)1996-1997 Fabrice Deville',
   						'TOfficeButton component version 2.6', MB_OK+ MB_ICONINFORMATION);
end;

function TAboutOffButtonProperty.GetAttributes: TPropertyAttributes;
begin
	Result := [paMultiSelect, paDialog, paReadOnly];
end;

function TAboutOffButtonProperty.GetValue: string;
begin
	Result := '(about)';
end;

(*
 * TOfficeButton implementation
 *)

constructor TOfficeButton.Create(aOwner: TComponent);
begin
	inherited Create(aOwner);
	GetSystemVersion;

	FBitmap := TBitmap.Create;
	FBitmap.OnChange := BitmapChange;
	FNoFocusBitmap := TBitmap.Create;
	FNoFocusBitmap.OnChange := NoFocusBitmapChange;
	FDisabledBitmap := TBitmap.Create;
	FDisabledBitmap.OnChange := DisabledBitmapChange;
	IBitmap := TBitmap.Create;

	ControlStyle := [csClickEvents, csCaptureMouse, csSetCaption, csOpaque];
	Pushed := False;
	MouseIn := False;
	Alignment := taCenter;
	Width := 50;
	Height := 40;
	if (csDesigning in ComponentState) and not (csLoading in TControl(Owner).ComponentState) then
		Caption := 'OfficeButton';
	FDropDown := nil;
	FOptions := [boMonoDisplay, boPopupMark, boShowBevel];
	FLayout := blBitmapTop;
	FAllowAllUp := False;
	FDown := False;
	FEnabled := True;
	FGroupIndex := 0;
	FBevelStyle := bsRaised;
	FShadingType := stMedium;
	FShowDownPattern := True;
	FUnselectedFontColor := clWindowText;
end;

destructor TOfficeButton.Destroy;
begin
	FBitmap.Free;
	FNoFocusBitmap.Free;
	FDisabledBitmap.Free;
	IBitmap.Free;
	if pattern <> nil then
	begin
		pattern.Free;
		pattern := nil;
	end;
	inherited Destroy;
end;

procedure TOfficeButton.BitmapChange(Sender: TObject);
begin
	if not FBitmap.Empty and FNoFocusBitmap.Empty and (csDesigning in ComponentState)
  	and not (csLoading in ComponentState) then
		CreateGrayscaleBitmap(IBitmap, FBitmap);

  if not FBitmap.Empty then
  	FBitmap.Dormant;
	Repaint;
end;

procedure TOfficeButton.NoFocusBitmapChange(Sender: TObject);
begin
	if not FBitmap.Empty and FNoFocusBitmap.Empty and (csDesigning in ComponentState) then
  begin
		CreateGrayscaleBitmap(IBitmap, FBitmap);
  end;

  if not FNoFocusBitmap.Empty then
  	FNoFocusBitmap.Dormant;
	Repaint;
end;

procedure TOfficeButton.DefineProperties(Filer: TFiler);
begin
	Filer.DefineBinaryProperty('IBitmap', ReadIBitmap, WriteIBitmap, True);
end;

procedure TOfficeButton.ReadIBitmap(Stream: TStream);
begin
	IBitmap.LoadFromStream(Stream);
end;

procedure TOfficeButton.WriteIBitmap(Stream: TStream);
begin
	if not IBitmap.Empty then
		IBitmap.SaveToStream(Stream)
end;

procedure TOfficeButton.DisabledBitmapChange(Sender: TObject);
begin
  if not FDisabledBitmap.Empty then
  	FDisabledBitmap.Dormant;
	if not FEnabled then
		Repaint;
end;

procedure TOfficeButton.CMDialogChar(var Message: TCMDialogChar);
var lpPoint : TPoint;
begin
	with Message do
		if IsAccel(CharCode, Caption)and Enabled and Visible then
	begin
		MouseIn := True;
		Pushed := False;
		Repaint;
		Application.ProcessMessages;
		WMLButtonDown( TWMLBUTTONDOWN(Message));
		Application.ProcessMessages;
		WMLButtonUp( TWMLBUTTONUP(Message));
		Application.ProcessMessages;
		GetCursorPos(lpPoint);
		lpPoint := GetParentForm(self).ScreenToClient(lpPoint);
		if not  ((lpPoint.y > top) and (lpPoint.y < top + height)
			and (lpPoint.x > left) and (lpPoint.x < left + width)) then
		begin
			MouseIn := False;
			Repaint;
		end;
		Result := 1;
	end;
end;

procedure TOfficeButton.ComputeExtent(TempCaption: PChar; var TempRect: TRect; theCanvas: TCanvas);
var Flags: Integer;
begin
	if Alignment = taLeftJustify then
		Flags := DT_LEFT
	else if Alignment = taCenter then
		Flags := DT_CENTER
	else Flags := DT_RIGHT;

	if WordWrap then
	begin
		Flags := Flags or DT_WORDBREAK;
		(* Sometimes DrawText looses the last word, except when there's a space character. Uh ? *)
		StrCat(TempCaption, ' ');
	end;

	DrawText(theCanvas.handle, TempCaption, StrLen(TempCaption), TempRect, DT_CALCRECT or Flags);
end;

procedure TOfficeButton.CreateGrayscaleBitmap(outputbmp, bmp: TBitmap);
var x, y: Integer;
    col, TransparentColor: LongInt;
    r, g, b, sum, threshold1, threshold2, threshold3: Integer;
begin
	outputbmp.Assign(bmp);
	TransparentColor := ColorToRGB(bmp.Canvas.Pixels[0,0]);

	if FShadingType = stLight then
	begin
		threshold1 := THRESHOLD1_LIGHT;
		threshold2 := THRESHOLD2_LIGHT;
		threshold3 := THRESHOLD3_LIGHT;
	end
	else
	if FShadingType = stMedium then
	begin
		threshold1 := THRESHOLD1_MEDIUM;
		threshold2 := THRESHOLD2_MEDIUM;
		threshold3 := THRESHOLD3_MEDIUM;
	end
	else
	if FShadingType = stDark then
	begin
		threshold1 := THRESHOLD1_DARK;
		threshold2 := THRESHOLD2_DARK;
		threshold3 := THRESHOLD3_DARK;
	end;
	for x := 0 to bmp.Width do
		for y := 0 to bmp.Height do
		begin
			col := ColorToRGB(bmp.Canvas.Pixels[x, y]);
			if col <> TransparentColor then
			begin
				r := col shr 16;
				g := (col shr 8) and $00FF;
				b := col and $0000FF;
				sum := r + g + b;
				if sum > THRESHOLD1 then
					outputbmp.Canvas.Pixels[x, y] := clWhite
				else if sum > THRESHOLD2 then
					outputbmp.Canvas.Pixels[x, y] := clBtnHighlight
				else if sum > THRESHOLD3 then
					outputbmp.Canvas.Pixels[x, y] := clBtnShadow
				else
					outputbmp.Canvas.Pixels[x, y] := clBlack;
			end;
		end;
 		if not bmp.Empty then
     	bmp.Dormant;
 		if not outputbmp.Empty then
     	outputbmp.Dormant;
end;

procedure TOfficeButton.DrawTheText(theCanvas: TCanvas; TempRect: TRect; TempCaption: PChar);
var Flags: Integer;
begin
	if Alignment = taLeftJustify then
		Flags := DT_LEFT
	else if Alignment = taCenter then
		Flags := DT_CENTER
	else Flags := DT_RIGHT;

	if WordWrap then
		Flags := Flags or DT_WORDBREAK;

	if bool_Version95 then
	begin
		{$IFDEF WIN32}
		DrawTextEx(theCanvas.handle, TempCaption, StrLen(TempCaption), TempRect,
     			 DT_END_ELLIPSIS or Flags, nil);
		{$ELSE}
		DrawText(theCanvas.handle, TempCaption, StrLen(TempCaption), TempRect, Flags);
		{$ENDIF}
	end
	else
		(* NT 3.51 users *)
		DrawText(theCanvas.handle, TempCaption, StrLen(TempCaption), TempRect, Flags);
end;

(*
 * Drawing of a disabled text (Win95 style)
 *)
procedure TOfficeButton.DrawDisabledText(theCanvas: TCanvas; TempRect: TRect; TempCaption: PChar);
begin
	theCanvas.Brush.Style := bsClear;
	theCanvas.Font.Color := clBtnHighlight;
	with TempRect do
	begin
		left := left + 1;
		top  := top + 1;
		right:= right + 1;
		bottom:= bottom + 1;
	end;
	DrawTheText(theCanvas, TempRect, TempCaption);
	theCanvas.Font.Color := clBtnShadow;
	with TempRect do
	begin
		left := left - 1;
		top  := top - 1;
		right:= right - 1;
		bottom:= bottom - 1;
	end;
	DrawTheText(theCanvas, TempRect, TempCaption);
end;

procedure TOfficeButton.DrawOutline(theCanvas: TCanvas; pushed: Boolean);
begin
	if boShowBevel in FOptions then
  begin
	if pushed then
	begin
		if (BevelStyle = bsLowered) then
			theCanvas.Pen.Color := clBtnHighlight
		else
			theCanvas.Pen.Color := clBtnShadow;
	end
	else
	begin
		if (BevelStyle = bsLowered) then
			theCanvas.Pen.Color := clBtnShadow
		else
			theCanvas.Pen.Color := clBtnHighlight;
	end;
	theCanvas.MoveTo(0, Height-1);
	theCanvas.LineTo(0, 0);
	theCanvas.LineTo(Width-1, 0);
	if pushed then
	begin
		if (BevelStyle = bsLowered) then
			theCanvas.Pen.Color := clBtnShadow
		else
			theCanvas.Pen.Color := clBtnHighlight;
	end
	else
	begin
		if (BevelStyle = bsLowered) then
			theCanvas.Pen.Color := clBtnHighlight
		else
			theCanvas.Pen.Color := clBtnShadow;
	end;
	theCanvas.LineTo(Width-1, Height-1);
	theCanvas.LineTo(0, Height-1);
  end;
end;

procedure TOfficeButton.DrawPopupMark(theCanvas: TCanvas; x, y: Integer);
var theColor: TColor;
begin
	(* loops were unrolled *)
	theColor := theCanvas.Font.Color;
	theCanvas.Pixels[x    , y - 1] := theColor;
	theCanvas.Pixels[x + 1, y - 1] := theColor;
	theCanvas.Pixels[x + 2, y - 1] := theColor;
	theCanvas.Pixels[x + 3, y - 1] := theColor;
	theCanvas.Pixels[x + 4, y - 1] := theColor;
	theCanvas.Pixels[x + 5, y - 1] := theColor;
	theCanvas.Pixels[x + 6, y - 1] := theColor;

	theCanvas.Pixels[x + 1, y    ] := theColor;
	theCanvas.Pixels[x + 2, y    ] := theColor;
	theCanvas.Pixels[x + 3, y    ] := theColor;
	theCanvas.Pixels[x + 4, y    ] := theColor;
	theCanvas.Pixels[x + 5, y    ] := theColor;

	theCanvas.Pixels[x + 2, y + 1] := theColor;
	theCanvas.Pixels[x + 3, y + 1] := theColor;
	theCanvas.Pixels[x + 4, y + 1] := theColor;

	theCanvas.Pixels[x + 3, y + 2] := theColor;
end;

procedure TOfficeButton.GetLost;
begin
	if FGroupIndex = 0 then
		FDown := False;
	Pushed := False;
	MouseIn := False;
	Repaint;
end;

(*
 * After the loading of the properties, the grayscale version can be
 * safely created
 *)
procedure TOfficeButton.Loaded;
begin
	inherited Loaded;
	if (not FBitmap.Empty) and FNoFocusBitmap.Empty and (csDesigning in ComponentState) then
  begin
		CreateGrayscaleBitmap(IBitmap, FBitmap);
     Repaint;
  end;
end;

procedure TOfficeButton.Notification(AComponent: TComponent; Operation:
TOperation);
begin
	if (Operation = opRemove) and (AComponent = FDropDown) then
		FDropDown := nil ;
end ;

(*
 * Painting of the button
 *)
procedure TOfficeButton.Paint;
var xt, yt, wt, _wt, ht, xb, yb, wb, hb, thePadding, theLeft, theTop: Integer;
	TempCaption : Array[0..256] of char;
	TempRect    : TRect;
	bitmp, OffScreen: TBitmap;
	ctrl: TControl;
begin
	if visible or (csDesigning in ComponentState) then
	begin
  	OffScreen := TBitmap.Create;
     OffScreen.Width := Width;
     OffScreen.Height := Height;
     OffScreen.Canvas.Brush.Color := clBtnFace;
     OffScreen.Canvas.FillRect(Rect(0,0,Width,Height));

		(* Just to stop compiler warnings *)
		xt := 0;
		yb := 0;
		xb := 0;

		if FDown and FShowDownPattern then
		begin
			if pattern = nil then
				CreatePattern;
			OffScreen.Canvas.Brush.Bitmap := pattern;
			OffScreen.Canvas.FillRect(Rect(0, 0, Width, Height));
		end;

		OffScreen.Canvas.Brush.Style := bsClear;
		OffScreen.Canvas.Font := Self.Font;

		if not (boIconOnly in FOptions) and (Length(FCaption) > 0) then
		begin
			thePadding := PADDING;
			TempRect.Top := 0;
			TempRect.Left := 0;
			TempRect.Right := Width - 4;
			TempRect.Bottom := 0;
			StrPCopy(TempCaption, Caption);
			ComputeExtent(TempCaption, TempRect, OffScreen.Canvas);
			wt := TempRect.Right;
			ht := TempRect.Bottom;
		end
		else
		begin
			thePadding := 0;
			wt := 0;
			ht := 0;
		end;

		if wt > Width - 4 then
			wt := Width - 4;

		wb := FBitmap.Width;
		(* Reserve place for the drawing of the popup mark *)
		if (boPopupMark in FOptions) and Assigned(FDropDown) then
			wb := wb + 10;
		hb := FBitmap.Height;

		if FBitmap.Empty then
		begin
			if Assigned(FDropDown) and (boPopupMark in FOptions) then
			begin
				_wt := wt + 10;
				if wt > Width - 14 then
					wt := Width - 14;
			end
			else
				_wt := wt;

			yt := (Height - ht) div 2;
			case Alignment of
				taLeftJustify:		xt := 3;
				taRightJustify:	xt := Width - _wt - 3;
				taCenter:			xt := (Width - _wt) div 2
			end;
		end
		else if (Layout = blBitmapTop) or (Layout = blBitmapBottom) then
		begin
			if Layout = blBitmapTop then
			begin
				yb := (Height - (ht + hb + thePadding)) div 2;
				yt := yb + hb + thePadding
			end
			else
			begin
				yt := (Height - (ht + hb + thePadding)) div 2;
				yb := yt + ht + thePadding
			end;
			case Alignment of
				taLeftJustify:
					begin
						xt := 3;
						xb := 3
					end;
				taRightJustify:
					begin
						xt := Width - wt - 3;
						xb := Width - wb - 3
					end;
				taCenter:
					begin
						xb := (Width - wb) div 2;
						xt := (Width - wt) div 2
					end;
			end;
		end
		else
		if Layout = blBitmapLeft then
		begin
			if wt + wb + thePadding > Width - 4 then
				wt := Width - 4 - thePadding - wb;
			yb := (Height - hb) div 2;
			yt := (Height - ht) div 2;

			case Alignment of
			taLeftJustify:
				begin
					xb := 3;
					xt := xb + wb + thePadding
				end;
			taRightJustify:
				begin
					xt := Width - wt - 3;
					xb := xt - wb - thePadding
				end;
			taCenter:
				begin
					xb := (Width - (wb + wt + thePadding)) div 2;
					xt := xb + wb + thePadding
				end;
			end;
		end
		else (* blBitmapRight *)
		begin
			if wt + wb + thePadding > Width - 4 then
				wt := Width - 4 - thePadding - wb;
			yb := (Height - hb) div 2;
			yt := (Height - ht) div 2;
			case Alignment of
			taLeftJustify:
				begin
					xt := 3;
					xb := xt + wt + thePadding
				end;
			taRightJustify:
				begin
					xb := Width - wb - 3;
					xt := xb - wt - thePadding
				end;
			taCenter:
				begin
					xt := (Width - (wb + wt + thePadding)) div 2;
					xb := xt + wt + thePadding
				end;
			end;
		end;

		if csDesigning in ComponentState then
		begin
			OffScreen.Canvas.Pen.Color := clBlack;
			OffScreen.Canvas.Pen.Style := psSolid;
			OffScreen.Canvas.Brush.Style := bsClear;
			OffScreen.Canvas.Rectangle(0, 0, Width, Height);
		end;

		if WordWrap and (xt + wt > width - 5) then
			wt := width - xt - 5;

		with TempRect do
		begin
			left := xt;
			top  := yt;
			right:= xt + wt;
			bottom:= yt + ht;
		end;

		if Enabled then
		begin
			if not (Pushed and MouseIn) and not Down then
			begin
				(* Unpushed state - Mouse in or out *)
				if MouseIn then
					DrawOutline(OffScreen.Canvas, False)
				else
					OffScreen.Canvas.Font.Color := UnselectedFontColor;
				if BevelStyle = bsLowered then
				begin
					Inc(TempRect.Left);
					Inc(TempRect.Top);
					Inc(TempRect.Right);
					Inc(TempRect.Bottom);
				end;
				if not (boIconOnly in FOptions) and (Length(Caption) > 0) then
					DrawTheText(OffScreen.Canvas, TempRect, TempCaption);
				if not FBitmap.Empty then
				begin
					(* Draw the normal or shaded bitmap.
					 * Transparency color is at (0,0)
					 *)
					if MouseIn or not (boMonoDisplay in FOptions) then
						DrawTransparentBitmap(OffScreen.Canvas, FBitmap, xb, yb, ColorToRGB(FBitmap.Canvas.Pixels[0, 0]))
					else if FNoFocusBitmap.Empty then
						DrawTransparentBitmap(OffScreen.Canvas, IBitmap, xb, yb, ColorToRGB(IBitmap.Canvas.Pixels[0, 0]))
					else
						DrawTransparentBitmap(OffScreen.Canvas, FNoFocusBitmap, xb, yb, ColorToRGB(FNoFocusBitmap.Canvas.Pixels[0, 0]));
					if (boPopupMark in FOptions) and Assigned(FDropDown) then
						DrawPopupMark(OffScreen.Canvas, xb + FBitmap.Width + 3, yb + (hb div 2));
				end
				else if (boPopupMark in FOptions) and Assigned(FDropDown) then
           				DrawPopupMark(OffScreen.Canvas, xt + wt + 3, yt + (ht div 2));
			end
			else
			begin
         	(* Pushed state *)
				DrawOutline(OffScreen.Canvas, True);
				if BevelStyle = bsRaised then
				begin
					Inc(TempRect.Left);
					Inc(TempRect.Top);
					Inc(TempRect.Right);
					Inc(TempRect.Bottom);
				end;
				if not (boIconOnly in FOptions) and (Length(Caption) > 0) then
					DrawTheText(OffScreen.Canvas, TempRect, TempCaption);
				if not FBitmap.Empty then
				begin
					DrawTransparentBitmap(OffScreen.Canvas, FBitmap, xb+1, yb+1, ColorToRGB(FBitmap.Canvas.Pixels[0, 0]));
					if (boPopupMark in FOptions) and Assigned(FDropDown) then
						DrawPopupMark(OffScreen.Canvas, xb + FBitmap.Width + 4, yb + (hb div 2) + 1);
				end
				else if (boPopupMark in FOptions) and Assigned(FDropDown) then
					DrawPopupMark(OffScreen.Canvas, xt + wt + 4, yt + (ht div 2) + 1);
			end
		end
		else
		begin
			(* Disabled state *)
			if Down then
				DrawOutline(OffScreen.Canvas, True);
			if not FDisabledBitmap.Empty then
				DrawTransparentBitmap(OffScreen.Canvas, FDisabledBitmap, xb, yb, ColorToRGB(FDisabledBitmap.Canvas.Pixels[0, 0]))
			else
				DrawDisabledBitmap(OffScreen.Canvas, xb, yb, FBitmap);
			if not (boIconOnly in FOptions) and (Length(Caption) > 0) then
			DrawDisabledText(OffScreen.Canvas, TempRect, TempCaption);
		end;

     Canvas.CopyRect(Rect(0,0,Width,Height), OffScreen.Canvas, Rect(0,0,Width,Height));

     OffScreen.Free;

     if not FBitmap.Empty then
     	FBitmap.Dormant;
     if not FDisabledBitmap.Empty then
     	FDisabledBitmap.Dormant;
     if not FNoFocusBitmap.Empty then
        FNoFocusBitmap.Dormant;
     if not IBitmap.Empty then
        IBitmap.Dormant;
	end
end;

procedure TOfficeButton.SetAlignment(value: TAlignment);
begin
	FAlignment := value;
	Repaint;
end;

procedure TOfficeButton.SetAllowAllUp(value: Boolean);
var i: Integer;
	otherbutton: TOfficeButton;
begin
	FAllowAllUp := value;
	if FGroupIndex <> 0 then
	begin
		for i := 0 to Parent.ControlCount - 1 do
		begin
			if Parent.Controls[i] is TOfficeButton then
			begin
				otherbutton := (Parent.Controls[i] as TOfficeButton);
				if (otherbutton <> Self) and (otherbutton.GroupIndex = GroupIndex) then
					otherbutton.FAllowAllUp := value;
			end
		end
	end
end;

procedure TOfficeButton.SetBevelStyle(value: TBevelStyle);
begin
	FBevelStyle := value;
	Repaint;
end;

procedure TOfficeButton.SetBitmap(value: TBitmap);
begin
	FBitmap.Assign(value);
  if not FBitmap.Empty then
  	FBitmap.Dormant;
end;

procedure TOfficeButton.SetCaption(value: TCaption);
begin
	FCaption := value;
	if boIconOnly in FOptions then
		Hint := RemoveAmpersand(FCaption)
	else
		Repaint;
end;

procedure TOfficeButton.SetDisabledBitmap(value: TBitmap);
begin
	FDisabledBitmap.Assign(value);
  if not FDisabledBitmap.Empty then
  	FDisabledBitmap.Dormant;
end;

procedure TOfficeButton.SetDown(value: Boolean);
var i: Integer;
	otherbutton: TOfficeButton;
begin
	if FDown = value then
   	Exit;
	if FGroupIndex <> 0 then
	begin
		FDown := value;
		Repaint;
	   	if FDown = True then
		begin
			for i := 0 to Parent.ControlCount - 1 do
   			begin
				if Parent.Controls[i] is TOfficeButton then
				begin
					otherbutton := (Parent.Controls[i] as TOfficeButton);
					if (otherbutton <> Self) and (otherbutton.GroupIndex = GroupIndex)
						and (otherbutton.FDown = True) then
					begin
						otherbutton.FDown := False;
						otherbutton.Repaint;
					end
				end
			end
		end
	end
end;

procedure TOfficeButton.SetDropDown(value: TPopupMenu);
begin
	FDropDown := value;
	if boPopupMark in FOptions then
		Repaint;
end;

procedure TOfficeButton.SetEnabled(value: Boolean);
begin
	if FEnabled = value then
   	Exit;
	FEnabled := value;
	GetLost;
end;

procedure TOfficeButton.SetGroupIndex(value: Integer);
begin
	if FGroupIndex = value then
		Exit;
	FGroupIndex := value;
	if (FGroupIndex = 0) and FDown then
	begin
		FDown := False;
		Repaint;
	end;
end;

procedure TOfficeButton.SetLayout(value: TExplorerButtonLayout);
begin
	if FLayout = value then
   	Exit;
	FLayout := value;
	Repaint;
end;

procedure TOfficeButton.SetNoFocusBitmap(value: TBitmap);
begin
	FNoFocusBitmap.Assign(value);
  if not FNoFocusBitmap.Empty then
  	FNoFocusBitmap.Dormant;
end;

procedure TOfficeButton.SetOptions(value: TExplorerButtonOptions);
begin
	FOptions := value;
	if (boIconOnly in FOptions) and (Hint = '') then
	begin
		Hint := RemoveAmpersand(FCaption);
		ShowHint := True;
	end
	else if (not(boIconOnly in FOptions)) and (Hint = RemoveAmpersand(FCaption)) then
	begin
		Hint := '';
		ShowHint := False;
	end;
	Repaint;
end;

procedure TOfficeButton.SetShadingType(value: TExplorerButtonShadingType);
begin
	if value <> FShadingType then
	begin
		FShadingType := value;
		if not FBitmap.Empty and FNoFocusBitmap.Empty and (csDesigning in ComponentState)
  		and not (csLoading in ComponentState) then
		begin
			CreateGrayscaleBitmap(IBitmap, FBitmap);
			Repaint;
		end;
	end;
end;

procedure TOfficeButton.SetShowDownPattern(value: Boolean);
begin
	if value <> FShowDownPattern then
	begin
		FShowDownPattern := value;
		Repaint;
	end;
end;

procedure TOfficeButton.SetUnselectedFontColor(value: TColor);
begin
	if FUnselectedFontColor <> value then
	begin
		FUnselectedFontColor := value;
		Repaint;
	end;
end;

procedure TOfficeButton.SetWordWrap(value: Boolean);
begin
	if FWordWrap <> value then
	begin
		FWordWrap := value;
		Repaint;
	end;
end;

procedure TOfficeButton.WMLButtonDown(var msg: TWMLButtonDown);
var p: TPoint;
	theMsg: TMsg;
	lpPoint: TPoint;
begin
	if not GetParentForm(Self).Active then
		GetParentForm(Self).BringToFront;

	if Enabled and Visible then
	begin
		Pushed := True;
		MouseIn := True;
		Repaint;
		(* If the popup is defined, display it under the button *)
		if FDropDown <> nil then
		begin
			p := ClientToScreen(Point(0, Height));
			FDropDown.Popup(p.x, p.y);
			while PeekMessage(theMsg, HWND(0), WM_MOUSEFIRST, WM_MOUSELAST, PM_REMOVE) do
				;
			if GetCapture <> 0 then
				SendMessage(GetCapture, WM_CANCELMODE, 0, 0);
			GetCursorPos(lpPoint);
			lpPoint := Parent.ScreenToClient(lpPoint);
			if not  ((lpPoint.y > top) and (lpPoint.y < top + height)
				and (lpPoint.x > left) and (lpPoint.x < left + width)) then
				MouseIn := False;

			Pushed := False;
			Repaint;
		end
     else
     	SetCaptureControl(Self);
	end;
end;

procedure TOfficeButton.WMLButtonUp(var msg: TWMLButtonUp);
var i: Integer;
	otherbutton: TOfficeButton;
	lpPoint: TPoint;
begin
	if Enabled and Visible and Pushed then
	begin
		Pushed := False;
		if MouseIn and (GroupIndex <> 0) then
		begin
			(* Was Down -> allow up state if AllowAllUp is True *)
			if FDown then
			begin
				if FAllowAllUp then
					FDown := False
			end
			else
			(* Was Up *)
			begin
				(* Set 'up' all buttons having the same parent
					and GroupIndex *)
				for i := 0 to Parent.ControlCount - 1 do
				begin
					if Parent.Controls[i] is TOfficeButton then
					begin
						otherbutton := (Parent.Controls[i] as TOfficeButton);
						if (otherbutton <> Self) and (otherbutton.GroupIndex = GroupIndex)
							and (otherbutton.FDown = True) then
						begin
							otherbutton.Down := False;
							otherbutton.Repaint;
						end
					end
				end;
				FDown := True;
			end;
		end;
		SetCaptureControl(nil);
		if MouseIn then
		begin
			GetCursorPos(lpPoint);
			lpPoint := Parent.ScreenToClient(lpPoint);
			if not  ((lpPoint.y > top) and (lpPoint.y < top + height)
				and (lpPoint.x > left) and (lpPoint.x < left + width)) then
				MouseIn := False;
			Invalidate;
			(* If the popup is not defined, activate the click event *)
			if (FDropDown = nil) then
				Click;
		end
	end
end;

procedure TOfficeButton.CMMouseEnter(var msg: TMessage);
begin
	MouseIn := True;
	if not FDown and FEnabled then
		Repaint;
        if Assigned(FOnEnter) then
                FOnEnter(Self);
end;

procedure TOfficeButton.CMMouseLeave(var msg: TMessage);
begin
	MouseIn := False;
	if not FDown and FEnabled then
		Repaint;
        if Assigned(FOnExit) then
                FOnExit(Self);
end;

procedure TOfficeButton.WMRButtonUp(var msg: TWMRButtonUp);
var p: TPoint;
	theMsg: TMsg;
	lpPoint: TPoint;
begin
	inherited;
	if (PopupMenu <> nil) or (GetParentForm(Self).PopupMenu <> nil) then
	begin
		while PeekMessage(theMsg, HWND(0), WM_MOUSEFIRST, WM_MOUSELAST, PM_REMOVE) do
			;
		if GetCapture <> 0 then
			SendMessage(GetCapture, WM_CANCELMODE, 0, 0);
		GetCursorPos(lpPoint);
		lpPoint := Parent.ScreenToClient(lpPoint);
		if not  ((lpPoint.y > top) and (lpPoint.y < top + height)
			and (lpPoint.x > left) and (lpPoint.x < left + width)) then
			MouseIn := False;
		Pushed := False;
		Repaint;
	end;
end;

procedure TOfficeButton.Click;
begin
	inherited Click;
end;

procedure Register;
begin
	RegisterComponents('Freeware', [TExplorerButton]);
	RegisterPropertyEditor(TypeInfo(TAboutExpButtonProperty), TExplorerButton, 'ABOUT', TAboutExpButtonProperty);
	RegisterComponents('Freeware', [TOfficeButton]);
	RegisterPropertyEditor(TypeInfo(TAboutOffButtonProperty), TOfficeButton, 'ABOUT', TAboutOffButtonProperty);
end;

initialization
	pattern := nil;

end.
