{--------------------------------------------------------------
  WinSock component for Borland Delphi.

  (C) 1995 by Ulf Sderberg, ulfs@sysinno.se
              Marc Palmer,   marc@cablenet.net
              Keith Hawes,   khawes@ccmail.com

  -- History --
  1.0 - 1.4   Moved to dWinSock.TXT
  1.41  30-Oct-95
        * Fixed bug in DoWrite. (Was calling the OnRead event)
---------------------------------------------------------------}
unit DWinSock;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, Buttons, DWSError, wsAPI;

const
  CM_SOCKMSG      = WM_USER+1;
  CM_SOCKCLOSE    = CM_SOCKMSG+1;
  { KH - Added for "false blocking" and non-blocking }
  CM_AsyncDone    = CM_SockClose+1;
  CM_AsyncBlock   = CM_AsyncDone+1;
  CM_ReleaseBlock = CM_AsyncBlock+1;

type
  { DWinSock exception type }
  ESockError = class(Exception);

  TAsyncOptionsType = ( csoRead, csoWrite, csoOOB );
  TAsyncOptions = set of TAsyncOptionsType;

  { Socket info codes }

  { MP 20/04/95 added siInactive }
  {    25/04/95 added siConnected, siClosed, siTimedOut }
  { KH 950719 added siAccept }
  { KH 950821 added siNoSuchSocket & Documentation}
  TSockInfo = (  siInactive,      { not used yet - obsolete?                  }
                 siLookUp,        { A Lookup is being performed               }
                 siConnect,       { A Connection is taking place              }
                 siConnected,     { Connection was sucessful                  }
                 siListen,        { Listining for connecitons                 }
                 siRecv,          { Recieving Data                            }
                 siSend,          { Sending Data                              }
                 siClosed,        { Connection Closed                         }
                 siTimedOut,      { Timout Occured                            }
                 siAccept,        { Connection Accepted by server             }
                 siNoSuchSocket); { Message arrived from a socket that has    }
                                  {   not been accepted by the server.        }
                                  {   *Socket will be NIL*                    }


  { KH 950905 }
  { KH - Socket message buffering records }
  PSockMsg = ^TSockMsg;
  TSockMsg = record
    handle: THandle;
    Msg,
    wParam: Word;
    lParam: Longint;
    next: PSockMsg;
  end;

  { KH - Last Async Task -- Used during synchronous Lookup functions }
  TLastTask = record
    handle: Thandle;
    err,
    buflen: word;
  end;

  { KH 950905 Result objects for AsyncLookup*By* }

  tAsyncInfo = class(TObject)
  private
    FTaskHandle: Thandle;
    FAsyncBuf: TAsyncBuf;
    FError: word;
  public
    property TaskHandle: Thandle read FTaskHandle;
    property Error: word read FError;
  end;

  tAsyncInfoClass = class of tAsyncInfo;

  tAsyncHostInfo = class(TAsyncInfo)
  private
    function GetName: string;
    function GetAddress: string;
    function GetAddr: in_addr;
  public
    property Name: string read GetName;
    property Address: String read GetAddress;
    property Addr: in_addr read GetAddr;
    property Info: HostEnt read FAsyncBuf.Host;
  end;

  tAsyncServiceInfo = class(TAsyncInfo)
  private
    function GetName: string;
    function GetPort: Word;
    function GetPortStr: string;
    function GetProtocol: string;
  public
    property Name: string read GetName;
    property Port: Word read GetPort;
    property PortStr: string read GetPortStr;
    property Protocol: String read GetProtocol;
    property Info: ServEnt read FAsyncBuf.Service;
  end;

  tAsyncProtocolInfo = class(TAsyncInfo)
  private
    function getName: string;
    function GetNumber: Word;
    function GetNumberStr: string;
  public
    property name: string read GetName;
    property Number: Word read GetNumber;
    property NumberStr: string read GetNumberStr;
    property Info: ProtoEnt read FAsyncBuf.Protocol;
  end;

  TSockCtrl = class;  { Forward declaration }
  TSocket   = class;  { Forward declaration }

  { Define notification events for socket controls. }
  TSockInfoEvent = procedure (Sender : TObject; icode : TSockInfo;
                              Socket: TSocket) of object;
  TSocketEvent = procedure (Sender : TObject; Socket : TSocket) of object;
  TSocketAsyncEvent = procedure (Sender : TObject; AsyncInfo: TAsyncInfo)
                                of object;


  { TSocket -- socket api wrapper class. }
  TSocket = class(TObject)
  private
    function RecvText : string;
    procedure SendText(const s : string);
    function SError(err, IgnoreErr, MsgId: Integer): integer;
  protected
    FLastError  : integer;      { Last return code form a winsock.dll call }
    FSocket     : TSock;        { socket id }
    FParent     : TSockCtrl;    { socket owner }
    FAddr       : sockaddr_in;  { host address }
    FConnected  : boolean;
    FBytesSent  : integer;      { bytes sent by last SendBuf call }

  public

    constructor Create(AParent : TSockCtrl); virtual;
    destructor Destroy; override; {KH 950718}

    procedure FillSocket(var name, addr, service : String; var port : Word);

    function LocalAddress : string;
    function LocalPort : Word;

    function RemoteHost : string;
    function RemoteAddress : string;
    function RemotePort : Word;

    function Send(var buf; cnt : integer) : integer;
    function Recv(var buf; cnt : integer) : integer;

    procedure SetOptions; virtual;
    procedure Listen(var name, addr, service : String; port : Word;
              nqlen : integer);
    procedure Open(var name, addr, service : String; port : Word;
              opts : TAsyncOptions);
    procedure Close; virtual;


    procedure ReadFlush;

    function InCount : integer;

    property BytesSent : integer read FBytesSent;
    property Text : string read RecvText write SendText;
    property LastError: integer read FLastError;
    property SocketID: TSock read FSocket;
    property Connected: boolean read FConnected;
    property Parent: TSockCtrl read FParent;
    property Addr: sockaddr_in read Faddr;
  end;

  TSocketClass = class of TSocket;

  TSocketList = class (TList)
  protected
    function GetSocket( Index : Integer ) : TSocket;
  public
    property Sockets[ Index : Integer ] : TSocket read GetSocket;
  end;

  TAsyncDataList = class (Tlist)
  protected
    function GetTask( TaskHandle : tHandle ) : TAsyncInfo;
  public
    property Tasks[ TaskHandle : tHandle ] : TAsyncInfo read GetTask;
  end;


  { TSockCtrl -- socket control component base class. }
  TSockCtrl = class(TComponent)
  private
    { US 950509 }
    FHWnd         : HWnd;

    { Event handler references }
    FOnInfo       : TSockInfoEvent;
    { MP 21/4/95 Moved from TClient+TSocket and 2 new properties added }
    FOnDisconnect    : TSocketEvent;
    FOnRead          : TSocketEvent;
    FOnWrite         : TSocketEvent;
    FOnTimeOut       : TSocketEvent;
    { KH 950905 Added FOnAsyncComplete }
    FOnAsyncComplete : TSocketAsyncEvent;

    fTimedOut     : Boolean;
    fAsyncData    : TAsyncDataList;

    { MP 25/4/95 New fields to handle timeout + timer event chains }
    FTimerChainParent, FTimerChainChild : TSockCtrl;
    FTimeOutRemaining   :  Integer;
    FTimeOutActive      :  Boolean;

    { Design time connection info }
    FHost         : String;
    FAddress      : String;
    FService      : String;
    FPort         : Word;

    FConn         : TSocket;        { Run time connection info }
    FClass        : TSocketClass;   { class of socket beeing used }
    FTimeOut      : integer;        { timeout length in seconds }

    { Access functions }
    procedure SetService(const s : String);
    procedure SetHost(const n : String);
    procedure SetAddress(const a : String);
    procedure SetPort(p : Word);
{ MP 25/4/95 }
    procedure SetTimeOut( p : Integer);

    { Returns the WinSock.DLL description }
    function GetDescription : string;
    function SAsyncError(TaskHandle: THandle; MsgId: integer): THandle;

  protected
    { Protected declarations }
    { KH 950905 }
    LastTask: TLastTask;

    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
    procedure CBSockClose(ASocket : TSocket); virtual;

    { US 950509 }
    procedure WndProc(var Message : TMessage);
    procedure OnSockMsg(var Message : TMessage); virtual; abstract;
    procedure OnAsyncDoneMsg( Var Message: TMessage); virtual;

    { KH 950905 }
    function WaitFor(ATask: THandle): integer; virtual;

    { MP 25/4/95 }
    procedure TimerEvent( Sender : TObject);
    procedure UseTimer;
    procedure ReleaseTimer;

    { MP 25/4/95  New properties }
    property OnTimeOut : TSocketEvent read FOnTimeOut write FOnTimeOut;
    property TimeOut : Integer read FTimeOut write SetTimeOut;

    { MP 21/4/95  Moved these props from client+server to TSockctrl
      KH 950825 Moved to protected Section }
    property OnDisconnect : TSocketEvent read FOnDisconnect write FOnDisconnect;
    property OnRead : TSocketEvent read FOnRead write FOnRead;
    property OnWrite : TSocketEvent read FOnWrite write FOnWrite;
    property OnAsyncComplete : TSocketAsyncEvent read FOnAsyncComplete
                                                 write FOnAsyncComplete;

    function TrackTask( InfoType: TAsyncInfoClass ): TAsyncInfo;
    procedure AsyncStartError( Info: TAsyncInfo );

  public
    { Public declarations }
    function LocalHost : string;

    { KH 950905 anme changes for Lookup* Functions }
    { US 950826 changed var to const }

    { DoXXX procedures }
    procedure DoDisconnect(var Socket: TSocket); virtual;
    procedure DoInfo(Socket: TSocket; icode : TSockInfo); virtual;
    procedure DoRead(var Socket: TSocket); virtual;
    procedure DoWrite(var Socket: TSocket); virtual;
    procedure DoTimeOut(var Socket: TSocket); virtual;

    { Lookup Functions }
    function LookupHostByAddr(const a : string) : String;
    function LookupAddrByName(const name : string) : in_addr;
    function LookupAddrByNameStr(const name : string) : string;
    function LookupPortByService(const service : string) : Word;
    function LookupServiceByPort(port : Word) : string;

    { Async Lookup Funcitons.
      Returns a TaskHandle that can be used to match the function call with
      the correct OnAsyncComplete event. }
    function AsyncLookupHostByAddr(const a: string): tHandle;
    function AsyncLookupHostByName(const name: string): tHandle;
    function AsyncLookupServiceByService(const service: string): tHandle;
    function AsyncLookupServicebyPort(const port: Word): thandle;


    property Handle : HWND read FHWnd;  { US 950509 }
    property Conn : TSocket read FConn;
    property Description : string read GetDescription;

  published
    { Published declarations }
    property Address : String read FAddress write SetAddress;
    property Port : Word read FPort write SetPort;
    property Service : String read FService write SetService;
    property OnInfo : TSockInfoEvent read FOnInfo write FOnInfo;

  end;

  { Definition of the TClientSocket component class }
  TClientSocket = class(TSockCtrl)
  private
    FOnConnect    : TSocketEvent;
    FOptions      : TAsyncOptions;

  protected
    { Protected declarations }
    procedure OnSockMsg(var Message : TMessage); override;

  public
    { Public declarations }
    procedure Open(ASocketClass : TSocketClass);
    procedure Close;
    function Connected : boolean;

    { KH 950825 DoXXX }
    procedure DoConnect(var Socket: TSocket); virtual;
    procedure DoDisconnect(var Socket: TSocket); override;

  published
    { Published declarations }
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;

    property Host : String read FHost write SetHost;
    property Options : TAsyncOptions read FOptions write FOptions
                                            default [csoRead, csoWrite];
    property OnConnect : TSocketEvent read FOnConnect write FOnConnect;
    property OnTimeOut;
    property TimeOut;
    { KH 950825 Added }
    property OnDisconnect;
    property OnRead;
    property OnWrite;
    property OnAsyncComplete;
  end;

  { Definition of the TServerSocket component class }
  TServerSocket = class(TSockCtrl)
  private
    { Event handler references }
    FOnAccept       : TSocketEvent;

    FMaxConns       : Integer;
    FConns          : TSocketList;
    FSocketClass    : TSocketClass;

    { MP 20/4/95 }
    FOptions        : TAsyncOptions;

    function GetClient(cid : integer) : TSocket;
    function GetClientCount : integer;


  protected
    { Protected declarations }
    procedure OnSockMsg(var Message : TMessage); override;
    procedure CBSockClose(ASocket : TSocket); override;

  public
    { Public declarations }
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;

    procedure Listen(nqlen : integer; ASocketClass : TSocketClass);
    procedure Close;

    { MP 20/04/95 added CloseDown declaration. Used CloseDown to avoid
    confusion with winsock's Shutdown }
    procedure CloseDown; { close server and all connections }
    { Return client socket }

    property Client[cid : integer] : TSocket read GetClient; default;
    property ClientCount : Integer read GetClientCount;

    procedure DoAccept(var Socket: TSocket);virtual;
    procedure DoDisconnect(var Socket: TSocket); override;

  published
    { Published declarations }
    property OnAccept : TSocketEvent read FOnAccept write FOnAccept;
    { MP 25/4/95 New property }
    property MaxConnections : Integer read FMaxConns write FMaxConns default 16;
    property ClientOptions : TAsyncOptions read FOptions write FOptions
                              default [csoRead, csoWrite];
    { KH 950825 Added }
    property OnDisconnect;
    property OnRead;
    property OnWrite;
    property OnAsyncComplete;
  end;

procedure Register;
function  ErrorStr(e: integer): string;
procedure StartUp;

implementation

uses ExtCtrls;

function  ErrorStr(e: integer): string;
begin
  if e = 0 then
    Result := ''
  else
    Result := LoadStr(e);
  if (e <> 0) and (Result = '') then
    Result := format('Unknown Error %d', [e]);
end;

{ KH extract message forming code to use for both TSocket.sError and
  TSockCtrl.sAsyncError                                                }

procedure SockException(Err, Msgid: integer);
var
  NPos: integer;
  SPos: integer;
  Msg,
  temp: string;
begin
  Msg := ErrorStr(MsgId);
  temp := Uppercase(Msg);
  NPos := pos('%D', temp);
  SPos := pos('%S', temp);
  if Msg = '' then
    temp := ErrorStr(err)
  else if (NPos = 0) and (SPos = 0) then
    temp := msg
  else if (NPos < SPos) and (NPos <> 0) then
    temp := format(msg, [err, ErrorStr(err)])
  else
    temp := format(msg, [ErrorStr(err), err]);
  raise ESockError.Create(temp);
end;

const
  { MP 20/04/95 Constant used for drawing component at design time
  dwsBtnBorderWidth = 2; }
  TimerUserCount : Integer = 0;
  TimerChainRoot : TSockCtrl = nil;

var
  bStarted      : boolean;
  nUsers        : integer;
  nWSErr        : integer;
  myVerReqd     : word;
  myWSAData     : WSADATA;
  Timer         : TTimer;
  BlockingTasks : integer;
  MsgQueue      : PSockMsg;
  ExitSave      : Pointer;

function MakeAsyncMask( Options : TAsyncOptions) : Longint;
begin
  Result := 0;

  if csoRead in Options then
    Result := FD_READ;

  if csoWrite in Options then
    Result := Result or FD_WRITE;

  if csoOOB in Options then
    Result := Result or FD_OOB;
end;

{ StartUp -- See if a Windows Socket DLL is present on the system. }
procedure StartUp;
begin
  if bStarted then exit;
  BlockingTasks:= 0;
  MsgQueue := nil;
  nUsers := 0;
  myVerReqd:=$0101;
  myVerReqd:=$0707;
  nWSErr := WSAStartup(myVerReqd, @myWSAData);
  if nWSErr = 0 then
    bStarted := True
  else
    SockException(nWSErr, DWSErrStart);
end;

{ CleanUp -- Tell Windows Socket DLL we don't need its services any longer. }
procedure CleanUp; far;
var
  temp: PSockMsg;
begin
  ExitProc := ExitSave;
{ MP 25/4/95 Free timer }
  Timer.Free;
  if bStarted then
  begin
     nWSErr := WSACleanup;
     bStarted := false;
     while MsgQueue <> nil do
     begin
       Dec(BlockingTasks);
       temp := MsgQueue;
       MsgQueue := MsgQueue^.Next;
       Dispose(temp);
     end;
     BlockingTasks := 0;
  end;
end;

{--------------------------------------------------------------
  TSocketList implementation
 --------------------------------------------------------------}
function TSocketList.GetSocket( Index : Integer ) : TSocket;
begin
  Result := Items[Index];
end;

{--------------------------------------------------------------
  TASsyncDataList implementation
 --------------------------------------------------------------}
function TAsyncDataList.GetTask( TaskHandle : tHandle ) : TAsyncInfo;
var
  I: integer;
begin
  Result := nil;
  for I := 0 to Count-1 do
  begin
    if Items[I] <> nil then
      if TAsyncInfo(Items[I]).TaskHandle = TaskHandle then
      begin
        Result := Items[I];
        break;
      end;
  end;
end;

{--------------------------------------------------------------
  TAsync*Info implementation
 --------------------------------------------------------------}
 function SafeStrPas( S: pChar): string;
 begin
   if S = nil then
     Result := ''
   else
     Result := StrPas(S);
 end;

 function TAsyncHostInfo.GetName: string;
 begin
   Result := SafeStrPas(FAsyncBuf.Host.h_name);
 end;

 function TAsyncHostInfo.GetAddress: string;
 var
   sa: in_addr;
 begin
   sa := GetAddr;
   Result := SafeStrPas(inet_ntoa(sa));
 end;

 function TAsyncHostInfo.GetAddr: in_addr;
 var
   pa: Pchar;
 begin
   pa := FAsyncBuf.Host.h_addr_list^;
   Result.S_un_b.s_b1 := byte(pa[0]);
   Result.S_un_b.s_b2 := byte(pa[1]);
   Result.S_un_b.s_b3 := byte(pa[2]);
   Result.S_un_b.s_b4 := byte(pa[3]);
 end;

function tAsyncServiceInfo.GetName: string;
begin
  Result := SafeStrPas(FAsyncBuf.Service.s_name);
end;

function tAsyncServiceInfo.GetPort: Word;
begin
  Result := ntohs(FAsyncBuf.Service.s_Port);
end;

function tAsyncServiceInfo.GetPortStr: string;
begin
  Result := IntToStr(getPort);
end;

function tAsyncServiceInfo.GetProtocol: string;
begin
  Result := SafeStrPas(FAsyncBuf.Service.s_Proto);
end;

function tAsyncProtocolInfo.getName: string;
begin
  Result := SafeStrPas(FAsyncBuf.Protocol.p_name);
end;

function tAsyncProtocolInfo.GetNumber: Word;
begin
  Result := ntohs(FAsyncBuf.Protocol.p_proto);
end;

function tAsyncProtocolInfo.GetNumberStr: string;
begin
  Result := InttoStr(GetNumber);
end;


{--------------------------------------------------------------
  TSocket implementation
 --------------------------------------------------------------}

constructor TSocket.Create(AParent : TSockCtrl);
begin
  inherited Create;
  FParent := AParent;
  FSocket := INVALID_SOCKET;
  FAddr.sin_family := PF_INET;
  FAddr.sin_addr.s_addr := INADDR_ANY;
  FAddr.sin_port := 0;
  FConnected := false;
  FBytesSent := 0;
end;

destructor TSocket.Destroy;
begin
  if FSocket <> INVALID_SOCKET  then
    CloseSocket(FSocket);
  inherited Destroy;
end;

{ LocalAddress -- get local address }
function TSocket.LocalAddress : string;
var
  sa : sockaddr_in;
  nl : integer;
begin
  Result := '';
  if FSocket = INVALID_SOCKET then exit;
  nl := SizeOf(sa);
  try { KH 950821 Change in error handling }
    sError(getsockname(FSocket, PSockaddr(@sa), @nl), 0, DWSErrGetSockName);
    Result := StrPas(inet_ntoa(sa.sin_addr));
  except
    on ESockError do ;
  end;
end;

{ LocalPort -- get local port number }
function TSocket.LocalPort : Word;
var
  sa : sockaddr_in;
  nl : integer;
begin
  Result := 0;
  if FSocket = INVALID_SOCKET then exit;
  nl := SizeOf(sa);
  try { KH 950821 Change in error handling }
    sError(getsockname(FSocket, PSockaddr(@sa), @nl), 0, DWSErrGetSockName);
    Result := ntohs(sa.sin_port);
  except
    on ESockError do ;
  end;
end;

{ RemoteHost -- get name of connected remote host }
function TSocket.RemoteHost : string;
var
  sa  : sockaddr_in;
  nl  : integer;
  buf : pAsyncBuf;
  ATask: Thandle;
begin
  Result := '';
  if not FConnected then exit;
  nl := sizeof(sa);
  { Get connection address info }
  sError(getpeername(FSocket, PSockaddr(@sa), @nl), 0, DWSErrGetPeerName);
  FAddr := sa;
  { Do a reverse lookup to get the host name }
  new(buf);
  try
    FParent.UseTimer;
    ATask := FParent.SAsyncError(WSAAsyncGetHostByAddr( FParent.Handle,
                     CM_AsyncBlock, PChar(@FAddr.sin_addr.s_addr), 4, PF_INET,
                     pchar(buf), SizeOf(Buf^)), DWSErrAsyncGetHostByAddr);
    if FParent.WaitFor(ATask) = 0 then
      Result := StrPas(Buf^.Host.h_name);
  finally
    dispose(buf);
    FParent.ReleaseTimer;
  end;
end;

{ RemoteAddress -- get address of connected remote host }
function TSocket.RemoteAddress : string;
var
  sa : sockaddr_in;
  nl : integer;
begin
  Result := '?';
  if (FSocket = INVALID_SOCKET) or (not FConnected) then exit;
  nl := SizeOf(sa);
  sError(getpeername(FSocket, PSockaddr(@sa), @nl), 0, DWSErrGetPeerName);
  Result := StrPas(inet_ntoa(sa.sin_addr));
end;

{ RemotePort -- get remote port number }
function TSocket.RemotePort : word;
var
  sa : sockaddr_in;
  nl : integer;
begin
  Result := 0;
  if (FSocket = INVALID_SOCKET) or (not FConnected) then exit;
  nl := SizeOf(sa);
  try { KH 950821 Change in error handling }
    sError(getpeername(FSocket, PSockaddr(@sa), @nl), 0, DWSErrGetPeerName);
    Result := ntohs(sa.sin_port)
  except
    on ESockError do Result := 0;
  end;
end;


{ FillSocket -- fill in address and port fields in socket struct }
procedure TSocket.FillSocket(var name, addr, service : String;
          var port : Word);
var
  s : array [1..32] of char;
begin
  { Fill in address field }
  if name <> '' then            { Host name given }
    begin
      FAddr.sin_addr := FParent.LookupAddrbyName(name);        {KH 950712}
      addr := StrPas(inet_ntoa(FAddr.sin_addr));
    end
  else if addr <> '' then       { IP address given }
    begin
      FAddr.sin_addr.s_addr := 0;
      if addr <> '0.0.0.0' then { beware of Trumpet bug! }
        begin
          StrPCopy(@s, addr);
          FAddr.sin_addr.s_addr := inet_addr(@s);
        end;
    end
  else                          { Neither name or address given }
    raise ESockError.Create(ErrorStr(DWSErrNoAddress));

  { Fill in port number field }
  if service <> '' then
    port := FParent.LookupPortByService(service); {KH 950712}
  FAddr.sin_port := htons(port);
end;

{ SetOptions -- set socket options }
procedure TSocket.SetOptions;
begin
  {Place Holder}
end;

{ Listen -- wait for incoming connection. }
procedure TSocket.Listen(var name, addr, service : String; port : Word;
                         nqlen : integer);
begin
  if (not bStarted) then
    SockException(WSANOTINITIALISED, 0);

  FSocket := wsAPI.socket(PF_INET, SOCK_STREAM, IPPROTO_IP);
  if FSocket = INVALID_SOCKET then
    sError(Socket_Error, 0, DWSErrCreate);

  FillSocket(name, addr, service, port);

  SetOptions;

  try { KH 950821 Change in error handling }
    sError(bind(FSocket, Psockaddr(@FAddr), SizeOf(FAddr)), 0, DWSErrBind);
  except
    on ESockError do
    begin
      Close;
      raise;
    end;
  end;

  WSAAsyncSelect(FSocket, FParent.Handle, CM_SOCKMSG, FD_ACCEPT or FD_CLOSE);

  { 950816 MP replaced q with nqlen}
  try { KH 950821 Change in error handling }
    sError(wsAPI.listen(FSocket, nqlen), 0, DWSErrListen);
    FParent.DoInfo(Self, siListen);
  except
    on ESockError do
    begin
      if FSocket <> INVALID_SOCKET then
        Close;
      raise;
    end;
  end;
end;

{ Open a connection. }
procedure TSocket.Open(var name, addr, service : String; port : Word;
                          opts : TAsyncOptions);
begin
  if (not bStarted) then
    SockException(WSANotInitialised, 0);

  if FConnected then
    SockException(WSAEIsConn, 0);

  FSocket := wsAPI.socket(PF_INET, SOCK_STREAM, IPPROTO_IP);
  if FSocket = INVALID_SOCKET then
    sError(Socket_Error, 0, DWSErrCreate);

  FParent.DoInfo(Self, siLookUp);
  { MP 25/4/95 }
  {FParent.UseTimer; { start timeout check } { KH Moved to Lookup*}

  FillSocket(name, addr, service, port);
  { MP 25/4/95 }
  {FParent.ReleaseTimer;}

  SetOptions;

  WSAAsyncSelect(FSocket, FParent.Handle, CM_SOCKMSG, MakeAsyncMask(opts) or
                FD_CONNECT or FD_CLOSE);

  { MP 25/4/95 }
  try
    FParent.UseTimer; { start timeout check }
    FParent.DoInfo(Self, siConnect);
    try { KH 950821 Change in error handling }
      sError(connect(FSocket, Psockaddr(@FAddr), SizeOf(FAddr)), WsaeWouldBlock,
             DWSErrConnect);
    except
      on ESockError do
      begin
        if FSocket <> INVALID_SOCKET then
          Close;
        raise;
      end;
    end;
  finally
    FParent.ReleaseTimer;
  end;
end;

procedure TSocket.Close;
begin
  if (not bStarted) or (FSocket = INVALID_SOCKET) then exit;
  ReadFlush;
  FConnected := false;
  closesocket(FSocket);
  FParent.DoInfo(Self, siClosed);
  FSocket := INVALID_SOCKET;
  FBytesSent := 0;
  PostMessage(FParent.Handle, CM_SOCKCLOSE, 0, Longint(self));
end;

function TSocket.SError(err, IgnoreErr, Msgid: integer): integer;
begin
  result := err;
  if err = SOCKET_ERROR then
  begin
    err := WSAGetLastError;
    FLastError := err;
    if err = IgnoreErr then { Ignore a specific error for both speed and  }
    begin;                  { leaving break on exceptions on for debuging }
      Result := 0;          { really helps for ignoring WsaeWouldBlock    }
      exit;
    end;
    SockException(err, MsgId);
  end
  else
    FLastError := 0;
end;

function TSocket.RecvText : string;
var
  n : integer;
begin
  n := Recv(PChar(@Result[1])^, 255);
  Result[0] := char(n);
end;

procedure TSocket.SendText(const s : string);
begin
  FBytesSent := Send(PChar(@s[1])^, Length(s));
end;

{ Send contents of passed buffer. }
function TSocket.Send(var buf; cnt : integer) : integer;
begin
  Result := 0;
  FParent.DoInfo(Self, siSend); {KH 950719}
  if (FSocket = INVALID_SOCKET) or (not FConnected) then exit;
  try { KH 950821 Change in error handling }
    result := SError(wsAPI.send(FSocket, @buf, cnt, 0), WsaeWouldBlock,
                     DWSErrSend);
  except
    on ESockError do
    begin
      close;
      raise;
    end;
  end;
end;

{ Request that passed buffer be filled with received data. }
function TSocket.Recv(var buf; cnt : integer) : integer;
begin
  Result := 0;
  FParent.DoInfo(Self, siRecv); {KH 950719}
  if (FSocket = INVALID_SOCKET) or (not FConnected) then
    raise ESockError.Create(ErrorStr(DWSErrSockNotOpen));

  try { KH 950821 Change in error handling }
    Result := sError(wsAPI.recv(FSocket, @buf, cnt, 0), WsaeWouldBlock,
                     DWSErrRecv);
  except
    on ESockError do
    begin
      Close;
      raise;
    end;
  end;
end;

{ ReadFlush -- Forces all data to be sent }
procedure TSocket.ReadFlush;
var
  temp: TSocket;
begin
  while InCount > 0 do
  begin
    temp := self;
    parent.DoRead(temp);
  end;
end;

{ InCount -- Get # of bytes in receive buffer }
function TSocket.InCount : integer;
const
  FIONREAD = $40000000 or ((longint(4)) shl 16) or (ord('f') shl 8) or 127;
var
  n   : longint;
begin
  sError(ioctlsocket(FSocket, FIONREAD, n), 0, DWSErrInCount);
  Result := n and $ffff;
end;

{--------------------------------------------------------------
  TSockCtrl implementation
 --------------------------------------------------------------}

{ Create -- initalization }
constructor TSockCtrl.Create(AOwner : TComponent);
begin
  inherited Create(AOwner);

  { The control should be visible at design time only.
    At run time, check if the WINSOCK has been started. }
  if not (csDesigning in ComponentState) then
  begin
    { US 950509 }
    FHWnd := AllocateHWnd(WndProc);
    StartUp;
  end;

  FHost := '';
  FAddress := '0.0.0.0';

  FService := '';
  FPort := 0;

  FillChar(LastTask, SizeOf(LastTask), 0);
  fTimedOut := False;

  inc(nUsers);
end;

{ Destroy -- destruction }
destructor TSockCtrl.Destroy;
begin
  ReleaseTimer;
  FConn.Free;
  fAsyncData.Free;
  Dec(nUsers);
  if nUsers <= 0 then
    CleanUp;
  { US 950509 }
  if not (csDesigning in ComponentState) then
    DeallocateHWnd(FHwnd);
  inherited Destroy;
end;

procedure TSockCtrl.CBSockClose(ASocket : TSocket);
begin

end;

{ US 950509: WndProc -- trap socket messages. }
procedure TSockCtrl.WndProc(var Message : TMessage);
{ KH 950905 }
var
  temp: PSockMsg;

  procedure NewMsg(Var Message: TMessage);
  begin
    if MsgQueue = nil then
    begin
      MsgQueue := new(PSockMsg);
      temp := MsgQueue;
    end
    else
    begin
      temp := MsgQueue;
      while temp^.Next <> nil do
        temp := temp^.Next;
      temp^.Next := new(PSockMsg);
      temp := temp^.Next;
    end;
    temp^.handle := Handle;
    temp^.msg := Message.Msg;
    temp^.wParam := Message.wParam;
    temp^.lParam := Message.lParam;
    temp^.Next := nil;
  end;

begin
  with Message do
  begin
    case Msg of
      CM_SOCKMSG   : if (BlockingTasks = 0) and (MsgQueue = nil) then
                       OnSockMsg(Message)                          { KH 950905 }
                     else
                       NewMsg(Message);                            { KH 950905 }
      CM_SOCKCLOSE : CBSockClose(TSocket(Message.lParam));
      CM_AsyncDone : OnAsyncDoneMsg(Message);                      { KH 950905 }
      CM_AsyncBlock: with LastTask do                              { KH 950905 }
                       begin
                         handle :=  Thandle(wParam);
                         err    := WSAGetAsyncError( lParam );
                         buflen := WSAGetAsyncBuflen( lParam);
                       end;
      CM_ReleaseBLock: {}; { Stop call to DefWindowProc } { KH 950905 }
      else
        { KH 9580814: Set Result field in Message record }
        Result := DefWindowProc(FHWnd, Msg, wParam, lParam);
      end;
  end;
  if (BlockingTasks = 0) and (MsgQueue <> nil) then { KH 950905 }
  begin
    temp := MsgQueue;
    MsgQueue := MsgQueue^.Next;
    PostMessage(temp^.Handle, temp^.Msg, temp^.wParam, Temp^.lParam);
    Dispose(temp);
  end;
end;

procedure TSockCtrl.OnAsyncDoneMsg( Var Message: TMessage);
var
  I: TAsyncInfo;
begin
  if fAsyncData <> nil then
  begin
    I := fAsyncData.Tasks[Message.wParam];
    if I = nil then
      exit; { Can't find Task so lets ignore it }
    I.FError := WSAGetAsyncError(Message.lParam);
    if Assigned(FOnAsyncComplete) then
    begin
      FOnAsyncComplete(Self, I);
    end;
    fAsyncData.Delete(fAsyncData.IndexOf(I));
    if fAsyncData.Count = 0 then
    begin
      fAsyncData.Free;
      fAsyncData := nil;
    end;
  end;
end;

function TSockCtrl.WaitFor(ATask: THandle): integer;
begin
  Inc(BlockingTasks);
  while (LastTask.Handle <> ATask) and (Not FTimedOut) do
    Application.ProcessMessages;
  Dec(BlockingTasks);
  if BlockingTasks >= 0 then
  begin
    BlockingTasks := 0;
    PostMessage(Handle, CM_ReleaseBlock, 0, 0);
  end;
  if (LastTask.Handle <> ATask) and (FTimedOut) then
    lastTask.err := WSAETimedOut;
  Result := LastTask.err;
end;

{ MP 25/4/95  Handle the time out timer events
              This gets a bit tricky, because we don't want to keep
              wasting CPU time if we have already timed out, so we release
              the timer if we time out. This can only be done once the
              other components in the chain have been called.
}
procedure TSockCtrl.TimerEvent( Sender : TObject );
begin
  if Assigned(FTimerChainChild) then
    FTimerChainChild.TimerEvent(Sender);
  if FTimeOutRemaining > 0 then
    Dec(FTimeOutRemaining);
  if FTimeOutRemaining = 0 then
    begin
      ReleaseTimer; { do this NOW in case event handler takes too long! }
      DoInfo(Conn, siTimedOut);
      { MP This should actually pass the actual socket in the case of a server }
      DoTimeOut(FConn);
    end;
end;

{ DoDisconnect -- Call the OnDisconnect event handler if any }
procedure TSockCtrl.DoDisconnect(var Socket: TSocket);
begin
  if Assigned(FOnDisconnect) then
    FOnDisconnect(Self, Socket);
end;

{ DoInfo -- call the OnInfo event handler if any. }
procedure TSockCtrl.DoInfo(Socket: TSocket; icode : TSockInfo);
begin
  if Assigned(FOnInfo) then
    FOnInfo(Self, icode, Socket);
end;

{ DoRead -- Call the OnRead event handler if any }
procedure TSockCtrl.DoRead(var Socket: TSocket);
begin
  if Assigned(FOnRead) then
    FOnRead(Self, Socket);
end;

{ DoWrite -- Call the OnWrite event handler if any }
procedure TSockCtrl.DoWrite(var Socket: TSocket);
begin
  if Assigned(FOnWrite) then
    FOnWrite(Self, Socket);
end;

{ DoTimeOut -- Call the OnTimeOut event handler if any }
procedure TSockCtrl.DoTimeOut(var Socket: TSocket);
begin
  if Assigned(FOnTimeOut) then
    FOnTimeOut(Self, Socket);
  FTimedOut := True;
end;

{ GetDescription -- return description of WinSock implementation }
function TSockCtrl.GetDescription : string;
begin
  Result := StrPas(myWSAdata.szDescription);
end;

function TSockCtrl.SAsyncError(TaskHandle: tHandle; MsgId: integer): THandle;
begin
  Result := TaskHandle;
  if TaskHandle = 0 then
    SockException(WSAGetLastError, MsgId);
end;

{ LocalHost -- return name of local host }
function TSockCtrl.LocalHost : string;
var
  sh : array [0..255] of char;
begin
  if not bStarted then
    begin
      Result := '';
      Exit;
    end;
  if gethostname(sh, 255) = 0 then
    Result := StrPas(sh)
  else
    Result := '';
end;

{ Set host name }
procedure TSockCtrl.SetHost(const n : String);
begin
  FHost := n;
  FAddress := '';
end;

{ Set host address }
procedure TSockCtrl.SetAddress(const a : String);
begin
  FAddress := a;
  FHost := '';
end;

{ Set service name }
procedure TSockCtrl.SetService(const s : String);
begin
  FService := s;
  if Not bStarted then
    FPort := 0
  else
  begin
    try
      FPort := LookupPortByService(S);
    except
      on ESockError do FPort := 0;
    end;
  end;
end;

{ Set port number }
procedure TSockCtrl.SetPort(p : Word);
begin
  FPort := p;
  if Not bStarted then
    FService := ''
  else
  begin
    try
      FService := LookupServiceByPort(p);
    except
      on ESockError do FService := '';
    end;
  end;
end;

{ MP 25/4/95 }
{ Set time out delay }
procedure TSockCtrl.SetTimeOut( p : Integer);
begin
  if p < 0 then p := 0; { trap negatives }
  FTimeOut := p;
end;

{ there is one global timer, and the different controls chain the calls
to the OnTimer event. }
procedure TSockCtrl.UseTimer;
begin
  if (csDesigning in ComponentState) then
    Exit;

  if (FTimeOut = 0) or (not Assigned(FOnTimeOut)) then exit;

  if not Assigned(Timer) then
    begin
      Timer := TTimer.Create(Self);
      Timer.Interval := 1000;
      Timer.Enabled := True;
    end;

  { Add ourselves to the top of the chain }
  FTimerChainChild := TimerChainRoot;
  FTimerChainParent := nil;
  TimerChainRoot := Self;
  Timer.OnTimer := TimerEvent;
  FTimeOutActive := True;
  FTimeOutRemaining := FTimeOut;
  Inc(TimerUserCount);
  FTimedOut := False;
end;

procedure TSockCtrl.ReleaseTimer;
begin
  if (csDesigning in ComponentState) then Exit;

  { US 950502 + removed lots of if FTimeOutActive from other places }
  if not FTimeOutActive then Exit;

  { remove ourselves from the chain }
  if Assigned(FTimerChainParent) then
    { reinstate previous handler }
    FTimerChainParent.FTimerChainChild := FTimerChainChild
  else
    begin
      if Assigned( FTimerChainChild) then
        Timer.OnTimer := FTimerChainChild.TimerEvent
      else
        Timer.OnTimer := nil;
      TimerChainRoot := FTimerChainChild;
    end;

  if Assigned(FTimerChainChild) then
    FTimerChainChild.FTimerChainParent := FTimerChainParent;

  Dec(TimerUserCount);
  FTimeOutActive := False;
  if TimerUserCount = 0 then
    begin
      Timer.Enabled := False;
      Timer.Free;
      Timer := nil;
    end;
end;

{ Reverse -- try to do a reverse lookup }
{ US 950826 changed 'var a' to 'const a' }
function TSockCtrl.LookupHostByAddr(const a : string) : string;
var
  Buf: pAsyncBuf;
  s    : array[0..31] of char;
  sa  : in_addr;
  ATask: tHandle;
begin
  Result := '';
  StrPCopy(@s, a);  { US 950826 added @ in front of 1st arg }
  sa.s_addr := inet_addr(s);
  if sa.s_addr = 0 then
    raise ESockError.Create(ErrorStr(DWSErrBasAddress));

  new(buf);
  try
    UseTimer;
    ATask := SAsyncError(WSAAsyncGetHostByAddr(handle, CM_AsyncBlock,
                         PChar(@sa.s_addr), 4, PF_INET,
                         Pchar(buf), SizeOf(Buf^)),
                         DWSErrAsyncGetHostbyAddr);
    if WaitFor(ATask) = 0 then
      Result := Strpas(buf^.Host.h_name)
    else
      SockException(LastTask.err, DWSErrAsyncHostToAddr);
  finally
    Dispose(buf);
    ReleaseTimer;
  end;
end;


{ LookupName -- try to look up host name }
function TSockCtrl.LookupAddrByName(const name : string) : in_addr;
var
  buf   : pAsyncBuf;
  pa    : PChar;
  sz    : array [1..64] of char;
  sa    : in_addr;
  ATask : THandle;
begin
  StrPCopy(@sz, name);
  New(buf);
  try
    UseTimer;
    ATask := sAsyncError( WSAAsyncGetHostByName( handle, CM_AsyncBlock, @SZ,
                                                pChar(buf), SizeOf(buf^) ),
                          DWSErrAsyncGetHostByName);
    if WaitFor(ATask) = 0 then
    begin
     { US 950518 fixed h_addr bug }
      pa := buf^.Host.h_addr_list^;
      sa.S_un_b.s_b1:=byte(pa[0]);
      sa.S_un_b.s_b2:=byte(pa[1]);
      sa.S_un_b.s_b3:=byte(pa[2]);
      sa.S_un_b.s_b4:=byte(pa[3]);
      Result := sa;
    end
    else
      SockException(LastTask.err,  DWSErrAsyncAddrToHost);
  finally
    Dispose(buf);
    ReleaseTimer;
  end;
end;

function TSockCtrl.LookupAddrByNameStr(const name : string): string;
var
  sa: in_addr;
begin
  sa := LookupAddrByName(name);
  Result := StrPas(inet_ntoa(sa));
end;

{ LookupService -- try to lookup service name }
{ KH 950905 }
function TSockCtrl.LookupPortByService(const service : string) : Word;
var
  buf : pAsyncBuf;
  proto : array [0..32] of char;
  name : array [0..64] of char;
  ATask : THandle;
begin
  Result := 0;
  StrPCopy(proto, 'tcp');
  StrPCopy(name, service);
  New(buf);
  try
    UseTimer;
    ATask := sAsyncError( WSAAsyncGetServByName( handle, CM_AsyncBlock, name, proto,
                                                 pChar(buf), SizeOf(buf^)),
                          DWSErrAsyncGetServbyName );
    if WaitFor(ATask) = 0 then
      Result := ntohs(buf^.service.s_port);
  (* KH  -- no exceptions, Port & Service properties cross reference each other
    else
      SockException(LastTask.err, 'Error %d "%s" finding port for service ' +
                                  name);
  *)
  finally
    Dispose(buf);
    ReleaseTimer;
  end;
end;

{ MP 18/7/95 LookupPort -- try to lookup port and return service name }
function TSockCtrl.LookupServiceByPort( port : Word) : string;
var
  Buf: pAsyncBuf;
  proto  : array [1..32] of char;
  ATask: THandle;
begin
  Result := '';
  StrPCopy(@proto, 'tcp');
  New(buf);
  try
    UseTimer;
    ATask := SAsyncError(WSAAsyncGetServByPort(Handle, CM_AsyncBlock,
                         htons(port), @proto, pChar(buf), SizeOF(Buf^)),
                         DWSErrAsyncGetServByPort);
    if WaitFor(ATask) = 0 then
      Result := Strpas(buf^.Service.s_name);
  (* KH  -- no exceptions, Port & Service properties cross reference each other
    else
      SockException(LastTask.err, 'Error %d "%s" finding service for port' +
                                  name);
  *)
  finally
    Dispose(buf);
    ReleaseTimer;
  end;
end;

function TSockCtrl.TrackTask(InfoType: TAsyncInfoClass ):
                    TAsyncInfo;
begin
  if fAsyncData = nil then
    fAsyncData := TAsyncDataList.Create;
  with fAsyncData do
    Result := Items[Add(InfoType.Create)];
end;

procedure TSockCtrl.AsyncStartError( Info: TAsyncInfo );
var
  Message: TMessage;
begin
  PostMessage( handle, CM_AsyncDone, Info.Taskhandle,
               WSAMakeAsyncReply( 0, WSAGetlastError))
end;

function TSockCtrl.AsyncLookupHostByAddr(const a: string): tHandle;
var
  Info: TAsyncInfo;
  S: array[0..31] of char;
  sa: in_Addr;
begin
  StrPcopy(s, a);
  sa.s_addr := inet_addr(s);
  if sa.s_addr = 0 then
    raise ESockError.Create(ErrorStr(DWSErrBasAddress));
  Info := TrackTask( TAsyncHostInfo );
  with Info do
  begin
    FTaskHandle := WSAAsyncGetHostByAddr( handle, CM_AsyncDone,
                   pChar(@sa.s_addr), 4, PF_INET,
                   @FAsyncBuf, SizeOf(FAsyncBuf));
    Result := FTaskHandle;
    if TaskHandle = 0 then
      AsyncStartError(Info);
  end;
end;

function TSockCtrl.AsyncLookupHostByName(const name: string): tHandle;
var
  Info: TAsyncInfo;
  S: array[0..31] of char;
begin
  StrPcopy(s, name);
  Info := TrackTask( TAsyncHostInfo );
  with Info do
  begin
    FTaskHandle := WSAAsyncGetHostByName( handle, CM_AsyncDone,
                   @S, @FAsyncBuf, SizeOf(FAsyncBuf));
    Result := FTaskHandle;
    if TaskHandle = 0 then
      AsyncStartError(Info);
  end;
end;

function TSockCtrl.AsyncLookupServiceByService(const service: string): tHandle;
var
  Info: TAsyncInfo;
  Proto: array[0..32] of char;
  name: array[0..64] of char;
begin
  StrPcopy(name, service);
  StrPCopy(Proto, 'tcp');
  Info := TrackTask( TAsyncServiceInfo );
  with Info do
  begin
    FTaskHandle := WSAAsyncGetServByName( handle, CM_AsyncDone,
                   @Name, @Proto, @FAsyncBuf, SizeOf(FAsyncBuf));
    Result := FTaskHandle;
    if TaskHandle = 0 then
      AsyncStartError(Info);
  end;
end;

function TSockCtrl.AsyncLookupServicebyPort(const port: Word): thandle;
var
  Info: TAsyncInfo;
  Proto: array[0..32] of char;
begin
  StrPCopy(Proto, 'tcp');
  Info := TrackTask( TAsyncServiceInfo );
  with Info do
  begin
    FTaskHandle := WSAAsyncGetServByPort( handle, CM_AsyncDone,
                   htons(port), @Proto, @FAsyncBuf, SizeOf(FAsyncBuf));
    Result := FTaskHandle;
    if TaskHandle = 0 then
      AsyncStartError(Info);
  end;
end;

{--------------------------------------------------------------
  TClientSocket implementation.
 --------------------------------------------------------------}

constructor TClientSocket.Create(AOwner : TComponent);
begin
  inherited Create(AOwner);
  FOptions := [ csoRead, csoWrite ];
end;

destructor TClientSocket.Destroy;
begin
  inherited Destroy;
end;

procedure TClientSocket.Open(ASocketClass : TSocketClass);
begin
  if Connected then
    raise ESockError.Create(ErrorStr(DWSErrSockAlreadyOpen));
  FConn.Free;
  FConn := ASocketClass.Create(self);
  FConn.Open(FHost, FAddress, FService, FPort, FOptions);
end;

procedure TClientSocket.Close;
begin
  { US 950502 }
  if FConn = nil then
    raise ESockError.Create(ErrorStr(DWSErrSockNotOpen));
  ReleaseTimer;
  FConn.Close;
end;

function TClientSocket.Connected : boolean;
begin
  Result := false;
  if FConn <> nil then
    Result := FConn.FConnected;
end;

procedure TClientSocket.DoConnect(var Socket: TSocket);
begin
  Socket.FConnected := true;
  { MP 25/4/95 }
  ReleaseTimer;
  { MP 950425 Let app know connection is made }
  DoInfo(Socket, siConnected);
  if Assigned(FOnConnect) then
    FOnConnect(Self, Socket);
end;

procedure TClientSocket.DoDisconnect(var Socket: TSocket);
begin
  if FConn.FConnected then
  begin
    { MP 20/4/95 }
    ReleaseTimer;
    {Info(siClosed);  KH 950719}
  end;
  inherited doDisconnect(socket);
 { US 950502 user must call xxx.Close method in OnDisconnect event }
end;

{ OnSockMsg -- handle CM_SOCKMSG }
procedure TClientSocket.OnSockMsg(var Message : TMessage);
var
  sock : TSock;
  evt, err : word;
begin
  sock := TSock(Message.wParam);
  evt := WSAGetSelectEvent(Message.lParam);
  err := WSAGetSelectError(Message.lParam);

  { US 950826 added following line }
  FConn.FLastError := err;

  case evt of
    FD_CONNECT: DoConnect(FConn);
    FD_CLOSE:   DoDisconnect(FConn);
    FD_OOB: ;
    FD_READ:    DoRead(FConn);
    FD_WRITE:   DoWrite(FConn);
  end;
end;
{--------------------------------------------------------------
  TServerSocket functions
 --------------------------------------------------------------}

constructor TServerSocket.Create(AOwner : TComponent);
begin
  inherited Create( AOwner );
  FConn := TSocket.Create( Self );
  FConns := TSocketList.Create;
  FMaxConns := 16;
  FOptions := [ csoRead, csoWrite ];
end;

destructor TServerSocket.Destroy;
var
  i : integer;
begin
  for i := 0 to FConns.Count-1 do
    FConns.Sockets[i].Free;
  FConns.Free;
  inherited Destroy;
end;

function TServerSocket.GetClient(cid : integer) : TSocket;
begin
  Result := FConns[cid];
end;

function TServerSocket.GetClientCount : integer;
begin
  Result := FConns.Count;
end;

procedure TServerSocket.Close;
begin
  { US 950502 }
  ReleaseTimer;
  FConn.Close;
end;

{ MP 20/04/95 CloseDown added. Closes all connection sockets and then closes
  the server socket. Useful for shutting down entire server without destroying
  the actual server object }
procedure TServerSocket.CloseDown;
var
  i : Integer;
begin
  for i := 0 to FConns.Count-1 do
    FConns.Sockets[i].Close;
  { MP 20/4/95 }
  FConn.Close;

  { US 950502 }
  ReleaseTimer;
end;

{ US 950427: CBSockClose }
procedure TServerSocket.CBSockClose(ASocket : TSocket);
var
  i   : integer;

begin
  if ASocket = FConn then Exit;  { Server's socket will NOT be in the list }
  for i := 0 to FConns.Count-1 do
    if FConns.Sockets[i].FSocket = ASocket.FSocket then
      begin
        FConns.Sockets[i].Free;
        FConns.Delete(i);
        FConns.Pack;  { ok, not particularly efficient }
        Break; { KH 950713 Why Keep going we just removed it }
      end;
end;

{ KH 950821 Change To Boolean and TSocket based}
procedure TServerSocket.DoAccept(var Socket: TSocket);
var
  nl     : integer;

  function NewConn : integer;
  begin
     Result := FConns.Add( FSocketClass.Create(Self) );
  end;

begin
{ MP 25/4/95 - Do not accept any more than FMaxConns connections.
  Should we do something to let the client know? Like accept and then
  close straight away ? }
  if FConns.Count >= FMaxConns then
    Exit;

  Socket := FConns[NewConn];
  nl := sizeof(sockaddr_in);
  Socket.FSocket := accept(FConn.FSocket, PSockaddr(@Socket.FAddr), @nl);
  if Socket.FSocket <> INVALID_SOCKET then
    begin
      WSAAsyncSelect(Socket.FSocket, Handle, CM_SOCKMSG,
                     MakeAsyncMask(FOptions) or FD_CLOSE);
      Socket.FConnected := True;
      {KH 950828 Moved DoInfo to after we have a valid Socket}
      DoInfo(Socket, siAccept);
      if Assigned(FOnAccept) then
        FOnAccept( Self, Socket );
    end;
end;

procedure TServerSocket.DoDisconnect(var Socket: TSocket);
begin
  { MP 18/4/95 changed this from NOT FConns[ to FConns[
    I think the logic was slightly erroneous }
  if Socket.FConnected then
  begin
   { MP 25/4/95 }
   ReleaseTimer;
   inherited DoDisconnect(Socket);
   {Info(siClosed); KH 950719}
 end;
 { US 950502 user must call xxx.Close method }
end;


{ OnSockMsg -- handle CM_SOCKMSG from WINSOCK }
{ KH 950821 Remove use of cid and switch to Socket and check on existance of
{           Socket and issue an info(siNoSuchSocket) if it does no exist. }
{           Cleaned up processing to avoide extra calls                   }
procedure TServerSocket.OnSockMsg(var Message : TMessage);
  function FindConn(var Socket: TSocket; sock: TSock): boolean;
  var
    i : integer;
  begin
    Result := False;
    for i := 0 to FConns.Count-1 do
      if FConns.Sockets[i].FSocket = sock then
        begin
          Socket := FConns.Sockets[i];
          Result := True;
          Exit;
        end;
  end;

var
  sock  : TSock;
  evt : word;
  err : word;
  Socket: TSocket;

begin
  sock := TSock(Message.wParam);
  evt := WSAGetSelectEvent(Message.lParam);
  err := WSAGetSelectError(Message.lParam);

  if evt = FD_Accept then
    DoAccept(Socket)
  else
  begin
    if not FindConn(Socket, Sock) then
      DoInfo(nil, siNoSuchSocket)
    else
    begin
      { US 950826 added following line }
      Socket.FLastError := err;
      case evt of
        FD_CLOSE: DoDisconnect(Socket);
        FD_READ:  DoRead(Socket);
        FD_WRITE: DoWrite(Socket);
        FD_OOB: ;
      end;
    end;
  end;
end;

procedure TServerSocket.Listen(nqlen : integer; ASocketClass : TSocketClass);
begin
  FSocketClass := ASocketClass;
  FConn.Listen(FHost, FAddress, FService, FPort, nqlen);
end;

{ Register our components. }
procedure Register;
begin
  RegisterComponents('Samples', [TClientSocket]);
  RegisterComponents('Samples', [TServerSocket]);
end;

{--------------------------------------------------------------
  Unit initialization code.
 --------------------------------------------------------------}

initialization
  bStarted := False;
  Timer := nil;
  ExitSave := ExitProc;
  ExitProc := @CleanUp;
end.



