unit main;

//
//  Demo 3
//  ------
//  Demonstration of the properties of the TDLDBDateEdit, TDLDateText,
//  TDLCalendar and TDLLED components
//  Copyright  1997-2002 by David Leeding
//  www.ozemail.com.au/~davidml/dlcomp
//
//  Requires that the DLDBDateEdit, TDLDateText, TDLCalendar and TDLLED
//  components be installed before opening!
//
//  This demo shows how the components might be used in a real database
//  application. In this case, it is a mock (and simplified) video library
//  database application. The left-hand area of the screen serves as an
//  "Info Panel" that can show information that is hidden from view (ie on
//  other tabs).
//
//  All date fields are edited using a TDBDateField. The
//  TDLCalendar is available when editing the Notes field (right-click)
//  at the point where you would like to insert a date. The TDLDateText
//  is implemented in the in the "Info Panel", as are the TDLEDs.
//
//  The purpose of the LEDs would be to alert the an operator that:
//
//  * The customer was having a birthday within the previous or coming
//    week (perhaps they qualify for a free video?!). Try changing the
//    customer birthdate on one record so that their birthday is either
//    today or within a week of today.
//
//  * The customer is a VIP due to the fact that they have spent a certain
//    amount their already (maybe they qualify for a special discount?!).
//    For this example, the limits are $74 for highest level VIP status
//    and $47 for secondary VIP status.
//
//  * The customer is a current member, have not been suspended nor have they
//    cancelled their membership. This could be used to alert the operator
//    that the customer should not be permitted to borrow videos!
//
//  * The customer has overdue items; perhaps they should not be allowed to
//    borrow more videos until they have returned the other ones?!
//
//  As you can see, the operator can quickly receive a great deal of pertinent
//  information without the need to wade through all the other information.
//
//  Note that this example uses the Enabled property of the LEDs rather than
//  "off". In some cases "Off" is quite acceptable, but when using different
//  colours meaningfully as in this demo, I prefer to use the Enabled property
//  for those times when the LED is not required. You could, alternatively, use
//  something like this:
//
//        ledOverdue.LEDState := lsOff;
//        ledOverdue.LEDColor := lcLBlue;
//
//  I like to give the end-user some control over how the application looks
//  (for example, some users prefer the small LEDs while others with poorer
//  eyesight find the larger ones easier to see). The Options function
//  demonstrates one way of doing this.
//
//

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  DBCtrls, DLLED, dlde, StdCtrls, Db, DBTables, ExtCtrls, ComCtrls, Mask,
  Menus, Grids, DBGrids;

type
  TfrmMain = class(TForm)
    pnlStatus: TPanel;
    Panel1: TPanel;
    nav: TDBNavigator;
    tblCustomer: TTable;
    dsCustomer: TDataSource;
    pnlInfo: TPanel;
    Label8: TLabel;
    DLDBDateText3: TDLDBDateText;
    Label10: TLabel;
    ledBirthday: TDLLED;
    ledVIP: TDLLED;
    ledStatus: TDLLED;
    lblBirthday: TLabel;
    lblVIP: TLabel;
    lblStatus: TLabel;
    DBText1: TDBText;
    Label1: TLabel;
    ledOverdue: TDLLED;
    lblOverdue: TLabel;
    Label11: TLabel;
    PageControl1: TPageControl;
    TabSheet1: TTabSheet;
    TabSheet2: TTabSheet;
    TabSheet3: TTabSheet;
    Label3: TLabel;
    DLDBDateText2: TDLDBDateText;
    Label2: TLabel;
    DBEdit1: TDBEdit;
    Label7: TLabel;
    DBEdit2: TDBEdit;
    Label4: TLabel;
    DBEdit3: TDBEdit;
    Label5: TLabel;
    DBEdit4: TDBEdit;
    Label6: TLabel;
    DBEdit5: TDBEdit;
    Label12: TLabel;
    dbdeDOB: TDLDBDateEdit;
    Label9: TLabel;
    dbdeJoined: TDLDBDateEdit;
    Label13: TLabel;
    dbdeSuspended: TDLDBDateEdit;
    Label15: TLabel;
    dbdeCancelled: TDLDBDateEdit;
    MainMenu1: TMainMenu;
    File1: TMenuItem;
    Options1: TMenuItem;
    Exit1: TMenuItem;
    Options2: TMenuItem;
    DBGrid1: TDBGrid;
    tblLoans: TTable;
    dsLoans: TDataSource;
    tblLibrary: TTable;
    dsLibrary: TDataSource;
    tblLoansItemName: TStringField;
    DBMemo1: TDBMemo;
    DBEdit6: TDBEdit;
    Label14: TLabel;
    Label16: TLabel;
    cal: TDLCalendar;
    PopupMenu1: TPopupMenu;
    InsertDate1: TMenuItem;
    InsertDateinLongFormat1: TMenuItem;
    Edit1: TEdit;
    tblLoansItem: TIntegerField;
    tblLoansId: TIntegerField;
    tblLoansLoan: TIntegerField;
    tblLoansBorrowed: TDateField;
    tblLoansDueBack: TDateField;
    tblLoansReturned: TDateField;
    tblLoansFeePaid: TCurrencyField;
    tblCustomerId: TIntegerField;
    tblCustomerName: TStringField;
    tblCustomerStreet: TStringField;
    tblCustomerCity: TStringField;
    tblCustomerPostcode: TStringField;
    tblCustomerPhone: TStringField;
    tblCustomerDateOfBirth: TDateField;
    tblCustomerJoinedOn: TDateField;
    tblCustomerSuspendedOn: TDateField;
    tblCustomerCancelledOn: TDateField;
    tblCustomerNotes: TStringField;
    Label17: TLabel;
    procedure FormCreate(Sender: TObject);
    procedure DBMemo1MouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure Exit1Click(Sender: TObject);
    procedure InsertDate1Click(Sender: TObject);
    procedure InsertDateinLongFormat1Click(Sender: TObject);
    procedure dsCustomerDataChange(Sender: TObject; Field: TField);
    procedure FormDestroy(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure Options2Click(Sender: TObject);
    procedure dbdeDOBOpenCalendar(Sender: TObject);
    procedure dbdeDOBCloseCalendar(Sender: TObject);
  private
    { Private declarations }
    procedure UpdateBirthdayLED;
    procedure UpdateVIPLED;
    procedure UpdateStatusLED;
    procedure UpdateOverdueLED;
  public
    { Public declarations }
  end;

const
  MAGIC_NUMBER_VIP_LVL1 = 74;  // Used in relation to the VIP LED
  MAGIC_NUMBER_VIP_LVL2 = 47;

var
  frmMain: TfrmMain;
  sCurrentDirectory: String;
  sDays: Array[1..7] of String;

implementation

uses dlg;

{$R *.DFM}

procedure TfrmMain.FormCreate(Sender: TObject);
begin
  GetDir(0, sCurrentDirectory);

  sDays[1] := 'Sunday';
  sDays[2] := 'Monday';
  sDays[3] := 'Tuesday';
  sDays[4] := 'Wednesday';
  sDays[5] := 'Thursday';
  sDays[6] := 'Friday';
  sDays[7] := 'Saturday';
end;

procedure TfrmMain.FormShow(Sender: TObject);
begin
  // This approach does away with the need for a BDE Alias
  tblCustomer.DatabaseName := sCurrentDirectory;
  tblLoans.DatabaseName := sCurrentDirectory;
  tblLibrary.DatabaseName := sCurrentDirectory;

  tblCustomer.Open;
  tblLoans.Open;
  tblLibrary.Open;

  PageControl1.ActivePage := TabSheet1;
end;

procedure TfrmMain.FormDestroy(Sender: TObject);
begin
  tblCustomer.Close;
  tblLoans.Close;
  tblLibrary.Close;
end;

procedure TfrmMain.Exit1Click(Sender: TObject);
begin
  Close;
end;

procedure TfrmMain.dsCustomerDataChange(Sender: TObject; Field: TField);
begin
  if Field = nil then
  begin
    UpdateBirthdayLED;
    UpdateVIPLED;
    UpdateStatusLED;
    UpdateOverdueLED;
  end;
end;

procedure TfrmMain.UpdateBirthdayLED;
var
  yDate, mDate, dDate: Word;
  yDOB, mDOB, dDOB: Word;
  dtBirthday: TDateTime;
begin

  with tblCustomer do
  begin
    // Birthday in the current year
    // ----------------------------
    if not FieldByName('DateOfBirth').IsNull then
    begin
      // Get the current year
      DecodeDate(Date, yDate, mDate, dDate);
      // and the month and date of birth
      DecodeDate(FieldByName('DateOfBirth').Value,
                 yDOB, mDOB, dDOB);

      // For leap year birthdays use 1/Mar instead
      if not IsLeapYear(yDate) and
         (mDOB = 2) and (dDOB = 29) then
      begin
        dDOB := 1; mDOB := 3;
      end;

      // Determine the birthday in the current year
      dtBirthday := EncodeDate(yDate, mDOB, dDOB);
    end else begin
      dtBirthday := Null;
    end;

    // Birthday in Data
    // ----------------
    if FieldByName('DateOfBirth').IsNull then
    begin
      ledBirthday.Enabled := False;
      lblBirthday.Caption := 'Birthday';
    end else begin
      if dtBirthday = Date then
      begin
        ledBirthday.LEDColor := lcGreen;
        ledBirthday.Enabled := True;
        lblBirthday.Caption := 'Birthday Today!';
      end else begin
        if (dtBirthday < (Date + 7)) and
           (dtBirthday > (Date - 7)) then
        begin
          ledBirthday.LEDColor := lcAmber;
          ledBirthday.Enabled := True;
          if dtBirthday < Date then
            lblBirthday.Caption := 'Birthday last ' + sDays[DayOfWeek(dtBirthday)]
          else
            lblBirthday.Caption := 'Birthday this ' + sDays[DayOfWeek(dtBirthday)];
        end else begin
          ledBirthday.Enabled := False;
          lblBirthday.Caption := 'Birthday Status';
        end; // if it's within three days of the birthday
      end;  // if it's the birthday
    end; // if birthdate is Null
  end;
end;

procedure TfrmMain.UpdateVIPLED;
var
  qry: TQuery;
begin

  if tblCustomer.FieldByName('CancelledOn').IsNull and
     tblCustomer.FieldByName('SuspendedOn').IsNull then
  begin
    // Only current members qualify VIP status
    qry := TQuery.Create(Self);
    try
      with qry do
      begin
        DatabaseName := sCurrentDirectory;
        SQL.Add('SELECT SUM(FeePaid) AS Total FROM "loans.db" WHERE (Id = ' +
                IntToStr(tblCustomer.FieldByName('Id').Value) + ')');
        Open;
        if not Fields[0].IsNull then
        begin
          if Fields[0].Value > MAGIC_NUMBER_VIP_LVL1 then
          begin
            // Big Spender!
            ledVIP.LEDColor := lcGreen;
            ledVIP.Enabled := True;
          end else begin
            if Fields[0].Value > MAGIC_NUMBER_VIP_LVL2 then
            begin
              // Regular Customer!
              ledVIP.LEDColor := lcAmber;
              ledVIP.Enabled := True;
            end else begin
              // Everyone else
              ledVIP.Enabled := False;
           end;
          end;
        end else begin
          ledVIP.Enabled := False;
        end;
        Close;
      end;
    finally
      qry.Free;
    end;

  end else begin
    ledVIP.Enabled := False;
  end;
end;

procedure TfrmMain.UpdateStatusLED;
begin
  if not tblCustomer.FieldByName('CancelledOn').IsNull then
  begin
    ledStatus.LEDColor := lcRed;
    ledStatus.Enabled := True;
    lblStatus.Caption := 'Membership Cancelled';
  end else begin
    if not tblCustomer.FieldByName('SuspendedOn').IsNull then
    begin
      ledStatus.LEDColor := lcAmber;
      ledStatus.Enabled := True;
      lblStatus.Caption := 'Membership Suspended';
    end else begin
      if not tblCustomer.FieldByName('JoinedOn').IsNull then
      begin
        ledStatus.LEDColor := lcGreen;
        ledStatus.Enabled := True;
        lblStatus.Caption := 'Current Member';
      end else begin
        ledStatus.Enabled := False;
        lblStatus.Caption := 'Not a member yet';
      end;
    end;
  end;
end;

procedure TfrmMain.UpdateOverdueLED;
var
  qry: TQuery;
begin
  qry := TQuery.Create(Self);
  try
    with qry do
    begin
      DatabaseName := sCurrentDirectory;
      SQL.Add('SELECT * FROM "loans.db" WHERE ' +
              '(Id = ' + IntToStr(tblCustomerId.Value) + ') ' +
              'AND (Returned IS NULL) ' +
              'AND (DueBack < "' + FormatDateTime('MM/DD/YYYY', Date ) + '")');
      Open;
      if RecordCount > 0 then
      begin
        ledOverdue.LEDColor := lcRed;
        ledOverdue.Enabled := True;
        lblOverdue.Caption := IntToStr(RecordCount) + ' Overdue Items';
      end else begin
        ledOverdue.Enabled := False;
        lblOverdue.Caption := 'No Overdue Items';
      end;
      Close;
    end;
  finally
    qry.Free;
  end;
end;

procedure TfrmMain.DBMemo1MouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  pt: TPoint;
begin
  if Button in [mbRight] then
  begin
    with cal do
    begin
      pt := DBMemo1.ClientToScreen(Point(X, Y));
      CalendarLeft := DBMemo1.Left + X;
      CalendarTop := DBMemo1.Top + Y;
    end;
    PopupMenu1.Popup(pt.X, pt.Y);
  end;
end;

procedure TfrmMain.InsertDate1Click(Sender: TObject);
begin
  cal.Date := Date;
  if cal.Opencalendar then
  begin
    with Edit1 do
    begin
      Text := DateToStr(cal.Date) + ' ';
      SelectAll;
      CopyToClipboard;
    end;
    DBMemo1.PasteFromClipboard;
  end;
end;

procedure TfrmMain.InsertDateinLongFormat1Click(Sender: TObject);
begin
  cal.Date := Date;
  if cal.Opencalendar then
  begin
    with Edit1 do
    begin
      Text := FormatDateTime(LongDateFormat, cal.Date) + ' ';
      SelectAll;
      CopyToClipboard;
    end;
    DBMemo1.PasteFromClipboard;
  end;
end;

// There are plenty of other (more graceful) ways of doing this in an application
// but this will suffice to show how the components can be configured by the
// end-user. (eg some users may like larger LEDs and others the smaller ones)
procedure TfrmMain.Options2Click(Sender: TObject);
var
  frm: TfrmDlg;
  lz: TLEDSize;
  ls: TLEDState;
  cl: TColor;
  ws: TWeekStart;
  i: Integer;
begin
  frm := TfrmDlg.Create(Self);
  try
    with frm do
    begin
      // Calendar options...get existing properties to comfigure dialog
      cbWeekStart.ItemIndex := Ord(dbdeDOB.CalendarWeekStart);
      case dbdeDOB.CalendarWeekendColor of
        clBtnText: cbWeekendColor.ItemIndex := 0;
        clBtnHighlight: cbWeekendColor.ItemIndex := 1;
        clBtnShadow: cbWeekendColor.ItemIndex := 2;
        clRed: cbWeekendColor.ItemIndex := 3;
        clWhite: cbWeekendColor.ItemIndex := 4;
        clBlack: cbWeekendColor.ItemIndex := 5;
        clGreen: cbWeekendColor.ItemIndex := 6;
        clMaroon: cbWeekendColor.ItemIndex := 7;
        clTeal: cbWeekendColor.ItemIndex := 8;
        clNavy: cbWeekendColor.ItemIndex := 9;
      else cbWeekendColor.ItemIndex := 0;
      end;
      chkFillPage.Checked := dbdeDOB.CalendarFillPage;

      // LED Options...get existing properties to comfigure dialog
      cbLEDSize.ItemIndex := Ord(ledBirthday.LEDSize);
      if ledBirthday.LEDState = lsOn then
        cbLEDState.ItemIndex := 0
      else
        cbLEDState.ItemIndex := 1;

      if ShowModal = idOk then
      begin
        // Get Calendar Options
        case cbWeekendColor.ItemIndex of
          0: cl := clBtnText;
          1: cl := clBtnHighlight;
          2: cl := clBtnShadow;
          3: cl := clRed;
          4: cl := clWhite;
          5: cl := clBlack;
          6: cl := clGreen;
          7: cl := clMaroon;
          8: cl := clTeal;
          9: cl := clNavy;
        else cl := clBtnText;
        end;
        case cbWeekStart.ItemIndex of
          0: ws := wsSunday;
          1: ws := wsMonday;
          2: ws := wsTuesday;
          3: ws := wsWednesday;
          4: ws := wsThursday;
          5: ws := wsFriday;
          6: ws := wsSaturday;
          else ws := wsSunday;
        end;
        // Apply calendar options
        for i := 0 to frmMain.ComponentCount - 1 do
          if frmMain.Components[i] is TDLDBDateEdit then
          begin
            TDLDBDateEdit(frmMain.Components[i]).CalendarWeekendColor := cl;
            TDLDBDateEdit(frmMain.Components[i]).CalendarWeekStart := ws;
            TDLDBDateEdit(frmMain.Components[i]).CalendarFillPage := chkFillPage.Checked;
          end;
        cal.CalendarWeekendColor := cl;
        cal.CalendarWeekStart := ws;

        // Get LED Options
        case cbLEDSize.ItemIndex of
          0: lz := lzSmall;
          1: lz := lzLarge;
        else
          lz := lzSmall;
        end;
        case cbLEDState.ItemIndex of
          0: ls := lsOn;
          1: ls := lsBright;
        else
          ls := lsOn;
        end;
        // Apply calendar options
        ledBirthday.LEDSize := lz;
        ledVIP.LEDSize := lz;
        ledStatus.LEDSize := lz;
        ledOverdue.LEDSize := lz;
        ledBirthday.LEDState := ls;
        ledVIP.LEDState := ls;
        ledStatus.LEDState := ls;
        ledOverdue.LEDState := ls;

      end;
    end;
  finally
    frm.Free;
  end;
end;

procedure TfrmMain.dbdeDOBOpenCalendar(Sender: TObject);
begin
  if TDLDBDateEdit(Sender).Name = 'dbdeDOB' then
    pnlStatus.Caption := 'Select the members Date of Birth. Press Esc to close the calendar without selecting a date'
  else
    if TDLDBDateEdit(Sender).Name = 'dbdeJoined' then
      pnlStatus.Caption := 'Select the date the customer joined. Press Esc to close the calendar without selecting a date'
    else
      if TDLDBDateEdit(Sender).Name = 'dbdeSuspended' then
        pnlStatus.Caption := 'Select the date the customer was suspended. Press Esc to close the calendar without selecting a date'
      else
        // Must be dbdeCancelled
        pnlStatus.Caption := 'Select the date the customer cancelled their membership. Press Esc to close the calendar without selecting a date';
end;

procedure TfrmMain.dbdeDOBCloseCalendar(Sender: TObject);
begin
  pnlStatus.Caption  := '';
end;

end.
