unit Idler;
{
  Idler.pas

  (c) 1996 Dwayne Mercredi

  declares Class TIdleAction

  class TIdleAction:
    uses application idle time; use the OnTrigger event
    to do any polling or low priority code and it will be called whenever
    the App is idle.

  IdleEventHandler
    points to the idle event handler that is associated with Application.OnIdle
}
interface

uses
  Classes, Controls, Forms, SysUtils;

type
  TIdleAction = class(TComponent)
  private
    FOnTrigger: TNotifyEvent;

    FEnabled: Boolean;

  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    {
      procedure DoTrigger
        always calls OnTrigger if it exists.  Called whenever CanTrigger
        returns True while the Application is idle
    }
    procedure DoTrigger; virtual;

    {
      function CanTrigger: Boolean
        controls whether this idle action is called when the application is
        idle.
        Returns Enabled for TIdleAction
    }
    function CanTrigger: Boolean; virtual;

  published

    {
      event OnTrigger
        idler event; called whenever the application is idle and this
        idle action is enabled
    }
    property OnTrigger: TNotifyEvent read FOnTrigger write FOnTrigger;

    {
      property Enabled
        determines if this idler is enabled
    }
    property Enabled: Boolean read FEnabled write FEnabled default True;
  end;

  {
    Idler Error thrown when there is an internal consistancy error in the
    idling list
  }
  EIdlerError = class(Exception);

const
  IdleEventHandler: TIdleEvent = Nil;

implementation

uses
  DBMUtils;

type
  TIdlerList = class(TObject)
  private
    IdlerList: TList;

  public
    constructor Create;
    destructor Destroy; override;

    {
      AddIdler
        Adds the given idler to the list.
    }
    procedure AddIdler(Idler: TIdleAction);

    {
      RemoveIdler
        Removes the given idler from the list
    }
    procedure RemoveIdler(Idler: TIdleAction);

    {
      DoIdle
        iterates through the idler list and calls DoIdle for all idlers
        that return true for CanTrigger
    }
    procedure DoIdle(Sender: TObject; var Done: Boolean);
  end;

var
  { old exit procedure -- for chaining into the exit code }
  OldExitProc: Pointer;

  { idler list -- handles all idle actions }
  IdlerList: TIdlerList;

{---------------------------------------------------------------------
                        class TIdlerList
 ---------------------------------------------------------------------}
constructor TIdlerList.Create;
begin
  inherited Create;

  IdlerList := TList.Create;
end;

destructor TIdlerList.Destroy;
begin
  IdlerList.Free;

  inherited Destroy;
end;

procedure TIdlerList.AddIdler(Idler: TIdleAction);
begin
  IdlerList.Add(Idler);
end;

procedure TIdlerList.RemoveIdler(Idler: TIdleAction);
begin
  IdlerList.Remove(Idler);
  IdlerList.Pack;
end;

procedure TIdlerList.DoIdle(Sender: TObject; var Done: Boolean);
var
  i: Integer;
  Triggered: Boolean;
begin
  { default to having this event handler not continually called }
  Done := True;

  { for each idler }
  for i := 0 to IdlerList.Count - 1 do begin
    Assert(IdlerList[i] <> NIL, 'Consistency Error in Idler List', EIdlerError);

    { do the idle action }
    with (TIdleAction(IdlerList[i])) do
      if (CanTrigger) then begin
        Done := False;
        DoTrigger;
      end;
  end;
end;

{---------------------------------------------------------------------
                        end class TIdlerList
 ---------------------------------------------------------------------}
{---------------------------------------------------------------------
                        class TIdleAction
 ---------------------------------------------------------------------}
constructor TIdleAction.Create(AOwner: TComponent);
begin
  { create }
  inherited Create(AOwner);

  { set up properties }
  FEnabled   := True;
  FOnTrigger := Nil;

  { add self to idler list }
  IdlerList.AddIdler(Self);
end;

destructor TIdleAction.Destroy;
begin
  { remove self from idler list }
  IdlerList.RemoveIdler(Self);

  { destroy }
  inherited Destroy;
end;

procedure TIdleAction.DoTrigger;
begin
  { call associated event }
  if (Assigned(FOnTrigger)) then
    FOnTrigger(Self);
end;

function TIdleAction.CanTrigger: Boolean;
begin
  { must be enabled and have event handler available to trigger }
  Result := Enabled and Assigned(FOnTrigger);
end;

{---------------------------------------------------------------------
                        end class TIdleAction
 ---------------------------------------------------------------------}

{
  procedure FreeIdler
    frees the idler and re-links the exit sub system
}
procedure FreeIdler; far;
begin
    ExitProc := OldExitProc;
    IdlerList.Free;
end;

initialization

{ create the idler list }
  IdlerList := TIdlerList.Create;

{ set up the application's OnIdle event to point to the Idler List }
  IdleEventHandler   := IdlerList.DoIdle;
  Application.OnIdle := IdleEventHandler;

{ set up the exit chain to free the idler }
  OldExitProc := ExitProc;
  ExitProc := @FreeIdler;
end.
